/*
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 *

 * PMU: intel_icl (Intel Icelake)
 */

static const intel_x86_umask_t intel_icl_ocr[]={
  { .uname   = "OTHER_LOCAL_DRAM",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that DRAM supplied the request.",
    .ucode   = 0x18400800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_LOCAL_DRAM",
    .udesc   = "Counts streaming stores that DRAM supplied the request.",
    .ucode   = 0x18400080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_LOCAL_DRAM",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that DRAM supplied the request.",
    .ucode   = 0x18400040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_LOCAL_DRAM",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that DRAM supplied the request.",
    .ucode   = 0x18400002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_LOCAL_DRAM",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that DRAM supplied the request.",
    .ucode   = 0x18400001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_LOCAL_DRAM",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that DRAM supplied the request.",
    .ucode   = 0x18400000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_LOCAL_DRAM",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that DRAM supplied the request.",
    .ucode   = 0x18400000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_LOCAL_DRAM",
    .udesc   = "Counts demand data reads that DRAM supplied the request.",
    .ucode   = 0x18400000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_L3_MISS",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_L3_MISS",
    .udesc   = "Counts streaming stores that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_L3_MISS",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_MISS",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_MISS",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_MISS",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_MISS",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_MISS",
    .udesc   = "Counts demand data reads that was not supplied by the L3 cache.",
    .ucode   = 0x3fffc0000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_DRAM",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that DRAM supplied the request.",
    .ucode   = 0x18400800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_DRAM",
    .udesc   = "Counts streaming stores that DRAM supplied the request.",
    .ucode   = 0x18400080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_DRAM",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that DRAM supplied the request.",
    .ucode   = 0x18400040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_DRAM",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that DRAM supplied the request.",
    .ucode   = 0x18400002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_DRAM",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that DRAM supplied the request.",
    .ucode   = 0x18400001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_DRAM",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that DRAM supplied the request.",
    .ucode   = 0x18400000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_DRAM",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that DRAM supplied the request.",
    .ucode   = 0x18400000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_DRAM",
    .udesc   = "Counts demand data reads that DRAM supplied the request.",
    .ucode   = 0x18400000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_L3_HIT_SNOOP_SENT",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that hit a cacheline in the L3 where a snoop was sent.",
    .ucode   = 0x1e003c800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_HIT_SNOOP_SENT",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that hit a cacheline in the L3 where a snoop was sent.",
    .ucode   = 0x1e003c002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_HIT_SNOOP_SENT",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that hit a cacheline in the L3 where a snoop was sent.",
    .ucode   = 0x1e003c001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_HIT_SNOOP_SENT",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that hit a cacheline in the L3 where a snoop was sent.",
    .ucode   = 0x1e003c000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_SENT",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that hit a cacheline in the L3 where a snoop was sent.",
    .ucode   = 0x1e003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_SENT",
    .udesc   = "Counts demand data reads that hit a cacheline in the L3 where a snoop was sent.",
    .ucode   = 0x1e003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_ANY_RESPONSE",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that have any type of response.",
    .ucode   = 0x1800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_ANY_RESPONSE",
    .udesc   = "Counts streaming stores that have any type of response.",
    .ucode   = 0x1080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_ANY_RESPONSE",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that have any type of response.",
    .ucode   = 0x1040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_ANY_RESPONSE",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that have any type of response.",
    .ucode   = 0x1002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_ANY_RESPONSE",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that have any type of response.",
    .ucode   = 0x1001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_ANY_RESPONSE",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that have any type of response.",
    .ucode   = 0x1000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_ANY_RESPONSE",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that have any type of response.",
    .ucode   = 0x1000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_ANY_RESPONSE",
    .udesc   = "Counts demand data reads that have any type of response.",
    .ucode   = 0x1000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L3_L3_HIT_ANY",
    .udesc   = "Counts hardware prefetches to the L3 only that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c238000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that hit a cacheline in the L3 where a snoop hit in another core, data forwarding is not required.",
    .ucode   = 0x4003c800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts miscellaneous requests, such as I/O and un-cacheable accesses that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c800000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_L3_HIT_ANY",
    .udesc   = "Counts streaming stores that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_L3_HIT_ANY",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L1D_AND_SWPF_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts L1 data cache prefetch requests and software prefetches (except PREFETCHW) that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c040000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_HIT_ANY",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that hit a cacheline in the L3 where a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that hit a cacheline in the L3 where a snoop hit in another core, data forwarding is not required.",
    .ucode   = 0x4003c002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_RFO_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts hardware prefetch RFOs (which bring data to L2) that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c002000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_HIT_ANY",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that hit a cacheline in the L3 where a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that hit a cacheline in the L3 where a snoop hit in another core, data forwarding is not required.",
    .ucode   = 0x4003c001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_L2_DATA_RD_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts hardware prefetch data reads (which bring data to L2)  that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c001000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_HIT_ANY",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that hit a cacheline in the L3 where a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that hit a cacheline in the L3 where a snoop hit in another core, data forwarding is not required.",
    .ucode   = 0x4003c000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_CODE_RD_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts demand instruction fetches and L1 instruction cache prefetches that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c000400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_ANY",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that hit a cacheline in the L3 where a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that hit a cacheline in the L3 where a snoop hit in another core, data forwarding is not required.",
    .ucode   = 0x4003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts demand reads for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_ANY",
    .udesc   = "Counts demand data reads that hit a cacheline in the L3 where a snoop was sent or not.",
    .ucode   = 0x3fc03c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand data reads that hit a cacheline in the L3 where a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts demand data reads that hit a cacheline in the L3 where a snoop hit in another core, data forwarding is not required.",
    .ucode   = 0x4003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_MISS",
    .udesc   = "Counts demand data reads that hit a cacheline in the L3 where a snoop was sent but no other cores had the data.",
    .ucode   = 0x2003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_NOT_NEEDED",
    .udesc   = "Counts demand data reads that hit a cacheline in the L3 where a snoop was not needed to satisfy the request.",
    .ucode   = 0x1003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_sq_misc[]={
  { .uname   = "SQ_FULL",
    .udesc   = "Cycles the thread is active and superQ cannot take any more entries.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_l2_lines_out[]={
  { .uname   = "USELESS_HWPF",
    .udesc   = "Cache lines that have been L2 hardware prefetched but not used by demand accesses",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NON_SILENT",
    .udesc   = "Modified cache lines that are evicted by L2 cache when triggered by an L2 cache fill.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SILENT",
    .udesc   = "Non-modified cache lines that are silently dropped by L2 cache when triggered by an L2 cache fill.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_l2_lines_in[]={
  { .uname   = "ALL",
    .udesc   = "L2 cache lines filling L2",
    .ucode   = 0x1f00ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_l2_trans[]={
  { .uname   = "L2_WB",
    .udesc   = "L2 writebacks that access L2 cache",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_baclears[]={
  { .uname   = "ANY",
    .udesc   = "Counts the total number when the front end is resteered, mainly when the BPU cannot provide a correct prediction and this is corrected by other branch handling mechanisms at the front end.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_mem_load_l3_miss_retired[]={
  { .uname   = "REMOTE_PMM",
    .udesc   = "Retired demand load instructions which missed L3 but serviced from remote IXP memory as data sources",
    .ucode   = 0x1000ull,
    .umodel  = PFM_PMU_INTEL_ICX,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "REMOTE_FWD",
    .udesc   = "Retired load instructions whose data sources was forwarded from a remote cache",
    .ucode   = 0x0800ull,
    .umodel  = PFM_PMU_INTEL_ICX,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "REMOTE_HITM",
    .udesc   = "Retired load instructions whose data sources was remote HITM",
    .ucode   = 0x0400ull,
    .umodel  = PFM_PMU_INTEL_ICX,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "REMOTE_DRAM",
    .udesc   = "Retired load instructions which data sources missed L3 but serviced from remote dram",
    .ucode   = 0x0200ull,
    .umodel  = PFM_PMU_INTEL_ICX,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOCAL_DRAM",
    .udesc   = "Retired load instructions which data sources missed L3 but serviced from local dram",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};


static const intel_x86_umask_t intel_icl_mem_load_l3_hit_retired[]={
  { .uname   = "XSNP_NONE",
    .udesc   = "Retired load instructions whose data sources were hits in L3 without snoops required",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_HITM",
    .udesc   = "Retired load instructions whose data sources were HitM responses from shared L3",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_HIT",
    .udesc   = "Retired load instructions whose data sources were L3 and cross-core snoop hits in on-pkg core cache",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_MISS",
    .udesc   = "Retired load instructions whose data sources were L3 hit and cross-core snoop missed in on-pkg core cache.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t intel_icl_mem_load_retired[]={
  { .uname   = "FB_HIT",
    .udesc   = "Number of completed demand load requests that missed the L1, but hit the FB(fill buffer), because a preceding miss to the same cacheline initiated the line to be brought into L1, but data is not yet ready in L1.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L3_MISS",
    .udesc   = "Retired load instructions missed L3 cache as data sources",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_MISS",
    .udesc   = "Retired load instructions missed L2 cache as data sources",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L1_MISS",
    .udesc   = "Retired load instructions missed L1 cache as data sources",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L3_HIT",
    .udesc   = "Retired load instructions with L3 cache hits as data sources",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_HIT",
    .udesc   = "Retired load instructions with L2 cache hits as data sources",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L1_HIT",
    .udesc   = "Retired load instructions with L1 cache hits as data sources",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOCAL_PMM",
    .udesc   = "Retired demand load instructions which missed L3 but serviced from local IXP memory as data sources",
    .ucode   = 0x8000ull,
    .umodel  = PFM_PMU_INTEL_ICX,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t intel_icl_mem_inst_retired[]={
  { .uname   = "ALL_STORES",
    .udesc   = "All retired store instructions.",
    .ucode   = 0x8200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ALL_LOADS",
    .udesc   = "All retired load instructions.",
    .ucode   = 0x8100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "SPLIT_STORES",
    .udesc   = "Retired store instructions that split across a cacheline boundary.",
    .ucode   = 0x4200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "SPLIT_LOADS",
    .udesc   = "Retired load instructions that split across a cacheline boundary.",
    .ucode   = 0x4100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOCK_LOADS",
    .udesc   = "Retired load instructions with locked access.",
    .ucode   = 0x2100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STLB_MISS_STORES",
    .udesc   = "Retired store instructions that miss the STLB.",
    .ucode   = 0x1200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STLB_MISS_LOADS",
    .udesc   = "Retired load instructions that miss the STLB.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t intel_icl_mem_trans_retired[]={
  { .uname  = "LOAD_LATENCY",
    .udesc  = "Memory load instructions retired above programmed clocks, minimum threshold value is 3 (Precise Event and ldlat required)",
    .ucode  = 0x100,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_misc_retired[]={
  { .uname   = "PAUSE_INST",
    .udesc   = "Number of retired PAUSE instructions.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LBR_INSERTS",
    .udesc   = "Increments whenever there is an update to the LBR array.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_rtm_retired[]={
  { .uname   = "ABORTED_EVENTS",
    .udesc   = "Number of times an RTM execution aborted due to none of the previous 4 categories (e.g. interrupt)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED_MEMTYPE",
    .udesc   = "Number of times an RTM execution aborted due to incompatible memory type",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED_UNFRIENDLY",
    .udesc   = "Number of times an RTM execution aborted due to HLE-unfriendly instructions",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED_MEM",
    .udesc   = "Number of times an RTM execution aborted due to various memory events (e.g. read/write capacity and conflicts)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED",
    .udesc   = "Number of times an RTM execution aborted.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "COMMIT",
    .udesc   = "Number of times an RTM execution successfully committed",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "START",
    .udesc   = "Number of times an RTM execution started.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_hle_retired[]={
  { .uname   = "ABORTED_EVENTS",
    .udesc   = "Number of times an HLE execution aborted due to unfriendly events (such as interrupts).",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED_UNFRIENDLY",
    .udesc   = "Number of times an HLE execution aborted due to HLE-unfriendly instructions and certain unfriendly events (such as AD assists etc.).",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED_MEM",
    .udesc   = "Number of times an HLE execution aborted due to various memory events (e.g., read/write capacity and conflicts).",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORTED",
    .udesc   = "Number of times an HLE execution aborted due to any reasons (multiple categories may count as one).",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "COMMIT",
    .udesc   = "Number of times an HLE execution successfully committed",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "START",
    .udesc   = "Number of times an HLE execution started.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_fp_arith_inst_retired[]={
  { .uname   = "512B_PACKED_SINGLE",
    .udesc   = "Counts number of SSE/AVX computational 512-bit packed double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 16 computation operations, one for each element.  Applies to SSE* and AVX* packed double precision floating-point instructions: ADD SUB MUL DIV MIN MAX SQRT RSQRT14 RCP14 FM(N)ADD/SUB. FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "512B_PACKED_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational 512-bit packed double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 8 computation operations, one for each element.  Applies to SSE* and AVX* packed double precision floating-point instructions: ADD SUB MUL DIV MIN MAX SQRT RSQRT14 RCP14 FM(N)ADD/SUB. FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "256B_PACKED_SINGLE",
    .udesc   = "Counts number of SSE/AVX computational 256-bit packed single precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 8 computation operations, one for each element.  Applies to SSE* and AVX* packed single precision floating-point instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX SQRT RSQRT RCP DPP FM(N)ADD/SUB.  DPP and FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "256B_PACKED_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational 256-bit packed double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 4 computation operations, one for each element.  Applies to SSE* and AVX* packed double precision floating-point instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX SQRT FM(N)ADD/SUB.  FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "128B_PACKED_SINGLE",
    .udesc   = "Number of SSE/AVX computational 128-bit packed single precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 4 computation operations, one for each element.  Applies to SSE* and AVX* packed single precision floating-point instructions: ADD SUB MUL DIV MIN MAX RCP14 RSQRT14 SQRT DPP FM(N)ADD/SUB.  DPP and FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "128B_PACKED_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational 128-bit packed double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 2 computation operations, one for each element.  Applies to SSE* and AVX* packed double precision floating-point instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX SQRT DPP FM(N)ADD/SUB.  DPP and FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCALAR_SINGLE",
    .udesc   = "Counts number of SSE/AVX computational scalar single precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 1 computational operation. Applies to SSE* and AVX* scalar single precision floating-point instructions: ADD SUB MUL DIV MIN MAX SQRT RSQRT RCP FM(N)ADD/SUB.  FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCALAR_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational scalar double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 1 computational operation. Applies to SSE* and AVX* scalar double precision floating-point instructions: ADD SUB MUL DIV MIN MAX SQRT FM(N)ADD/SUB.  FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_frontend_retired[]={
  { .uname   = "LATENCY_GE_1",
    .udesc   = "Retired instructions after front-end starvation of at least 1 cycle",
    .ucode   = 0x50010600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_2_BUBBLES_GE_1",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end had at least 1 bubble-slot for a period of 2 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x10020600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_512",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 512 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x52000600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_256",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 256 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x51000600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_128",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 128 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x50800600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_64",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 64 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x50400600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_32",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 32 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x50200600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_16",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 16 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x50100600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_8",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 8 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x50080600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_4",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 4 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x50040600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "LATENCY_GE_2",
    .udesc   = "Retired instructions after front-end starvation of at least 2 cycles",
    .ucode   = 0x50020600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
    .modhw   = _INTEL_X86_ATTR_FETHR,
  },
  { .uname   = "STLB_MISS",
    .udesc   = "Retired Instructions who experienced STLB (2nd level TLB) true miss.",
    .ucode   = 0x1500ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ITLB_MISS",
    .udesc   = "Retired Instructions who experienced iTLB true miss.",
    .ucode   = 0x1400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_MISS",
    .udesc   = "Retired Instructions who experienced Instruction L2 Cache true miss.",
    .ucode   = 0x1300ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L1I_MISS",
    .udesc   = "Retired Instructions who experienced Instruction L1 Cache true miss.",
    .ucode   = 0x1200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "DSB_MISS",
    .udesc   = "Retired Instructions who experienced DSB miss.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname  = "IDQ_4_BUBBLES",
    .udesc  = "Retired instructions after an interval where the front-end did not deliver any uops (4 bubbles) for a period determined by the fe_thres modifier (set to 1 cycle by default) and which was not interrupted by a back-end stall",
    .ucode  = (4 << 20 | 0x6) << 8,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname  = "IDQ_3_BUBBLES",
    .udesc  = "Counts instructions retired after an interval where the front-end did not deliver more than 1 uop (3 bubbles) for a period determined by the fe_thres modifier (set to 1 cycle by default) and which was not interrupted by a back-end stall",
    .ucode  = (3 << 20 | 0x6) << 8,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname  = "IDQ_2_BUBBLES",
    .udesc  = "Counts instructions retired after an interval where the front-end did not deliver more than 2 uops (2 bubbles) for a period determined by the fe_thres modifier (set to 1 cycle by default) and which was not interrupted by a back-end stall",
    .ucode  = (2 << 20 | 0x6) << 8,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname  = "IDQ_1_BUBBLE",
    .udesc  = "Counts instructions retired after an interval where the front-end did not deliver more than 3 uops (1 bubble) for a period determined by the fe_thres modifier (set to 1 cycle by default) and which was not interrupted by a back-end stall",
    .ucode  = (1 << 20 | 0x6) << 8,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t intel_icl_br_misp_retired[]={
  { .uname   = "INDIRECT",
    .udesc   = "All miss-predicted indirect branch instructions retired (excluding RETs. TSX aborts is considered indirect branch).",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_TAKEN",
    .udesc   = "Number of near branch instructions retired that were mispredicted and taken.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND",
    .udesc   = "Mispredicted conditional branch instructions retired.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_NTAKEN",
    .udesc   = "Mispredicted non-taken conditional branch instructions retired.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT_CALL",
    .udesc   = "Mispredicted indirect CALL instructions retired.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_TAKEN",
    .udesc   = "number of branch instructions retired that were mispredicted and taken. Non PEBS",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ALL_BRANCHES",
    .udesc   = "All mispredicted branch instructions retired.",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t intel_icl_br_inst_retired[]={
  { .uname   = "INDIRECT",
    .udesc   = "All indirect branch instructions retired (excluding RETs. TSX aborts are considered indirect branch).",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "FAR_BRANCH",
    .udesc   = "Far branch instructions retired.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_TAKEN",
    .udesc   = "Taken branch instructions retired.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND",
    .udesc   = "Conditional branch instructions retired.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_NTAKEN",
    .udesc   = "Not taken branch instructions retired.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_RETURN",
    .udesc   = "Return instructions retired.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_CALL",
    .udesc   = "Direct and indirect near call instructions retired.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_TAKEN",
    .udesc   = "Taken conditional branch instructions retired.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ALL_BRANCHES",
    .udesc   = "All branch instructions retired.",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t intel_icl_machine_clears[]={
  { .uname   = "SMC",
    .udesc   = "Self-modifying code (SMC) detected.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MEMORY_ORDERING",
    .udesc   = "Number of machine clears due to memory ordering conflicts.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "COUNT",
    .udesc   = "Number of machine clears (nukes) of any type.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
};

static const intel_x86_umask_t intel_icl_uops_retired[]={
  { .uname   = "SLOTS",
    .udesc   = "Retirement slots used.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "TOTAL_CYCLES",
    .udesc   = "Cycles with less than 10 actually retired uops.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_INV_BIT) | (0xa << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALL_CYCLES",
    .udesc   = "Cycles without actually retired uops.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t intel_icl_assists[]={
  { .uname   = "ANY",
    .udesc   = "Number of occurrences where a microcode assist is invoked by hardware.",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FP",
    .udesc   = "Counts all microcode FP assists.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_tlb_flush[]={
  { .uname   = "STLB_ANY",
    .udesc   = "STLB flush attempts",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DTLB_THREAD",
    .udesc   = "DTLB flush attempts of the thread-specific entries",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_uops_executed[]={
  { .uname   = "X87",
    .udesc   = "Counts the number of x87 uops dispatched.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CORE_CYCLES_GE_4",
    .udesc   = "Cycles at least 4 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x4 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE_CYCLES_GE_3",
    .udesc   = "Cycles at least 3 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x3 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE_CYCLES_GE_2",
    .udesc   = "Cycles at least 2 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE_CYCLES_GE_1",
    .udesc   = "Cycles at least 1 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE",
    .udesc   = "Number of uops executed on the core.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CYCLES_GE_4",
    .udesc   = "Cycles where at least 4 uops were executed per-thread",
    .ucode   = 0x0100ull | (0x4 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_3",
    .udesc   = "Cycles where at least 3 uops were executed per-thread",
    .ucode   = 0x0100ull | (0x3 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_2",
    .udesc   = "Cycles where at least 2 uops were executed per-thread",
    .ucode   = 0x0100ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_1",
    .udesc   = "Cycles where at least 1 uop was executed per-thread",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALL_CYCLES",
    .udesc   = "Counts number of cycles no uops were dispatched to be executed on this thread.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "THREAD",
    .udesc   = "Counts the number of uops to be executed per-thread each cycle.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_offcore_requests[]={
  { .uname   = "ALL_REQUESTS",
    .udesc   = "Any memory transaction that reached the SQ.",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "L3_MISS_DEMAND_DATA_RD",
    .udesc   = "Demand Data Read requests who miss L3 cache",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_DATA_RD",
    .udesc   = "Demand and prefetch data reads",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO",
    .udesc   = "Demand RFO requests including regular RFOs, locks, ItoM",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD",
    .udesc   = "Demand Data Read requests sent to uncore",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_dsb2mite_switches[]={
  { .uname   = "COUNT",
    .udesc   = "DSB-to-MITE transitions count.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "PENALTY_CYCLES",
    .udesc   = "DSB-to-MITE switch true penalty cycles.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_lsd[]={
  { .uname   = "CYCLES_OK",
    .udesc   = "Cycles optimal number of Uops delivered by the LSD, but did not come from the decoder.",
    .ucode   = 0x0100ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_ACTIVE",
    .udesc   = "Cycles Uops delivered by the LSD, but didn't come from the decoder.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "UOPS",
    .udesc   = "Number of Uops delivered by the LSD.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_exe_activity[]={
  { .uname   = "EXE_BOUND_0_PORTS",
    .udesc   = "Cycles where no uops were executed, the Reservation Station was not empty, the Store Buffer was full and there was no outstanding load.",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BOUND_ON_STORES",
    .udesc   = "Cycles where the Store Buffer was full and no loads caused an execution stall.",
    .ucode   = 0x4000ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "4_PORTS_UTIL",
    .udesc   = "Cycles total of 4 uops are executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "3_PORTS_UTIL",
    .udesc   = "Cycles total of 3 uops are executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "2_PORTS_UTIL",
    .udesc   = "Cycles total of 2 uops are executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "1_PORTS_UTIL",
    .udesc   = "Cycles total of 1 uop is executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_cycle_activity[]={
  { .uname   = "STALLS_MEM_ANY",
    .udesc   = "Execution stalls while memory subsystem has an outstanding load.",
    .ucode   = 0x1400ull | (0x14 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_MEM_ANY",
    .udesc   = "Cycles while memory subsystem has an outstanding load.",
    .ucode   = 0x1000ull | (0x10 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L1D_MISS",
    .udesc   = "Execution stalls while L1 cache miss demand load is outstanding.",
    .ucode   = 0x0c00ull | (0xc << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_L1D_MISS",
    .udesc   = "Cycles while L1 cache miss demand load is outstanding.",
    .ucode   = 0x0800ull | (0x8 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L3_MISS",
    .udesc   = "Execution stalls while L3 cache miss demand load is outstanding.",
    .ucode   = 0x0600ull | (0x6 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L2_MISS",
    .udesc   = "Execution stalls while L2 cache miss demand load is outstanding.",
    .ucode   = 0x0500ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_TOTAL",
    .udesc   = "Total execution stalls.",
    .ucode   = 0x0400ull | (0x4 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_L3_MISS",
    .udesc   = "Cycles while L3 cache miss demand load is outstanding.",
    .ucode   = 0x0200ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_L2_MISS",
    .udesc   = "Cycles while L2 cache miss demand load is outstanding.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t intel_icl_resource_stalls[]={
  { .uname   = "SB",
    .udesc   = "Cycles stalled due to no store buffers available. (not including draining form sync).",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCOREBOARD",
    .udesc   = "Counts cycles where the pipeline is stalled due to serializing operations.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_uops_dispatched[]={
  { .uname   = "PORT_7_8",
    .udesc   = "Number of uops executed on port 7 and 8",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_6",
    .udesc   = "Number of uops executed on port 6",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_5",
    .udesc   = "Number of uops executed on port 5",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_4_9",
    .udesc   = "Number of uops executed on port 4 and 9",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_2_3",
    .udesc   = "Number of uops executed on port 2 and 3",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_1",
    .udesc   = "Number of uops executed on port 1",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_0",
    .udesc   = "Number of uops executed on port 0",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_idq_uops_not_delivered[]={
  { .uname   = "CYCLES_FE_WAS_OK",
    .udesc   = "Cycles when optimal number of uops was delivered to the back-end when the back-end is not stalled",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_0_UOPS_DELIV_CORE",
    .udesc   = "Cycles when no uops are not delivered by the IDQ when backend of the machine is not stalled",
    .ucode   = 0x0100ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE",
    .udesc   = "Uops not delivered by IDQ when backend of the machine is not stalled",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_ild_stall[]={
  { .uname   = "LCP",
    .udesc   = "Stalls caused by changing prefix length of the instruction.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_itlb_misses[]={
  { .uname   = "STLB_HIT",
    .udesc   = "Instruction fetch requests that miss the ITLB and hit the STLB.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_ACTIVE",
    .udesc   = "Cycles when at least one PMH is busy with a page walk for code (instruction fetch) request.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "WALK_PENDING",
    .udesc   = "Number of page walks outstanding for an outstanding code request in the PMH each cycle.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Code miss in all TLB levels causes a page walk that completes. (All page sizes)",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_2M_4M",
    .udesc   = "Code miss in all TLB levels causes a page walk that completes. (2M/4M)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_4K",
    .udesc   = "Code miss in all TLB levels causes a page walk that completes. (4K)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_icache_64b[]={
  { .uname   = "IFTAG_STALL",
    .udesc   = "Cycles where a code fetch is stalled due to L1 instruction cache tag miss.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "IFTAG_MISS",
    .udesc   = "Instruction fetch tag lookups that miss in the instruction cache (L1I). Counts at 64-byte cache-line granularity.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "IFTAG_HIT",
    .udesc   = "Instruction fetch tag lookups that hit in the instruction cache (L1I). Counts at 64-byte cache-line granularity.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_icache_16b[]={
  { .uname   = "IFDATA_STALL",
    .udesc   = "Cycles where a code fetch is stalled due to L1 instruction cache miss.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_idq[]={
  { .uname   = "MS_CYCLES_ANY",
    .udesc   = "Cycles when uops are being delivered to IDQ while MS is busy",
    .ucode   = 0x3000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "MS_UOPS",
    .udesc   = "Uops delivered to IDQ while MS is busy",
    .ucode   = 0x3000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MS_SWITCHES",
    .udesc   = "Number of switches from DSB or MITE to the MS",
    .ucode   = 0x3000ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "DSB_CYCLES_ANY",
    .udesc   = "Cycles Decode Stream Buffer (DSB) is delivering any Uop",
    .ucode   = 0x0800ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DSB_CYCLES_OK",
    .udesc   = "Cycles DSB is delivering optimal number of Uops",
    .ucode   = 0x0800ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DSB_UOPS",
    .udesc   = "Uops delivered to Instruction Decode Queue (IDQ) from the Decode Stream Buffer (DSB) path",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MITE_CYCLES_ANY",
    .udesc   = "Cycles MITE is delivering any Uop",
    .ucode   = 0x0400ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "MITE_CYCLES_OK",
    .udesc   = "Cycles MITE is delivering optimal number of Uops",
    .ucode   = 0x0400ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "MITE_UOPS",
    .udesc   = "Uops delivered to Instruction Decode Queue (IDQ) from MITE path",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_offcore_requests_outstanding[]={
  { .uname   = "CYCLES_WITH_L3_MISS_DEMAND_DATA_RD",
    .udesc   = "Cycles where data return is pending for a Demand Data Read request who miss L3 cache.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_WITH_DATA_RD",
    .udesc   = "Cycles when offcore outstanding cacheable Core Data Read transactions are present in SuperQueue (SQ), queue to uncore.",
    .ucode   = 0x0800ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "ALL_DATA_RD",
    .udesc   = "Offcore outstanding cacheable Core Data Read transactions in SuperQueue (SQ), queue to uncore",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CYCLES_WITH_DEMAND_RFO",
    .udesc   = "Cycles with offcore outstanding demand rfo reads transactions in SuperQueue (SQ), queue to uncore.",
    .ucode   = 0x0400ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DEMAND_RFO",
    .udesc   = "Store Read transactions pending for off-core. Highly correlated.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD",
    .udesc   = "Demand Data Read transactions pending for off-core. Highly correlated.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_rs_events[]={
  { .uname   = "EMPTY_END",
    .udesc   = "Counts end of periods where the Reservation Station (RS) was empty.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "EMPTY_CYCLES",
    .udesc   = "Cycles when Reservation Station (RS) is empty for the thread",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_tx_exec[]={
  { .uname   = "MISC3",
    .udesc   = "Number of times an instruction execution caused the transactional nest count supported to be exceeded",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISC2",
    .udesc   = "Counts the number of times a class of instructions that may cause a transactional abort was executed inside a transactional region",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_tx_mem[]={
  { .uname   = "ABORT_CAPACITY_READ",
    .udesc   = "Speculatively counts the number of TSX aborts due to a data capacity limitation for transactional reads",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HLE_ELISION_BUFFER_FULL",
    .udesc   = "Number of times HLE lock could not be elided due to ElisionBufferAvailable being zero.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORT_HLE_ELISION_BUFFER_UNSUPPORTED_ALIGNMENT",
    .udesc   = "Number of times an HLE transactional execution aborted due to an unsupported read alignment from the elision buffer.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORT_HLE_ELISION_BUFFER_MISMATCH",
    .udesc   = "Number of times an HLE transactional execution aborted due to XRELEASE lock not satisfying the address and value requirements in the elision buffer",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORT_HLE_ELISION_BUFFER_NOT_EMPTY",
    .udesc   = "Number of times an HLE transactional execution aborted due to NoAllocatedElisionBuffer being non-zero.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORT_HLE_STORE_TO_ELIDED_LOCK",
    .udesc   = "Number of times a HLE transactional region aborted due to a non XRELEASE prefixed instruction writing to an elided lock in the elision buffer",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORT_CAPACITY_WRITE",
    .udesc   = "Speculatively counts the number of TSX aborts due to a data capacity limitation for transactional writes.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ABORT_CONFLICT",
    .udesc   = "Number of times a transactional abort was signaled due to a data conflict on a transactionally accessed address",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_l1d[]={
  { .uname   = "REPLACEMENT",
    .udesc   = "Counts the number of cache lines replaced in L1 data cache.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_load_hit_prefetch[]={
  { .uname   = "SWPF",
    .udesc   = "Counts the number of demand load dispatches that hit L1D fill buffer (FB) allocated for software prefetch.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_dtlb_store_misses[]={
  { .uname   = "STLB_HIT",
    .udesc   = "Stores that miss the DTLB and hit the STLB.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_ACTIVE",
    .udesc   = "Cycles when at least one PMH is busy with a page walk for a store.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "WALK_PENDING",
    .udesc   = "Number of page walks outstanding for a store in the PMH each cycle.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Store misses in all TLB levels causes a page walk that completes. (All page sizes)",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_2M_4M",
    .udesc   = "Page walks completed due to a demand data store to a 2M/4M page.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_4K",
    .udesc   = "Page walks completed due to a demand data store to a 4K page.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_l1d_pend_miss[]={
  { .uname   = "L2_STALL",
    .udesc   = "Number of cycles a demand request has waited due to L1D due to lack of L2 resources.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FB_FULL_PERIODS",
    .udesc   = "Number of phases a demand request has waited due to L1D Fill Buffer (FB) unavailablability.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "FB_FULL",
    .udesc   = "Number of cycles a demand request has waited due to L1D Fill Buffer (FB) unavailability.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PENDING_CYCLES",
    .udesc   = "Cycles with L1D load Misses outstanding.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "PENDING",
    .udesc   = "Number of L1D misses that are outstanding",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_sw_prefetch_access[]={
  { .uname   = "PREFETCHW",
    .udesc   = "Number of PREFETCHW instructions executed.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "T1_T2",
    .udesc   = "Number of PREFETCHT1 or PREFETCHT2 instructions executed.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "T0",
    .udesc   = "Number of PREFETCHT0 instructions executed.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NTA",
    .udesc   = "Number of PREFETCHNTA instructions executed.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_longest_lat_cache[]={
  { .uname   = "MISS",
    .udesc   = "Core-originated cacheable demand requests missed L3",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_core_power[]={
  { .uname   = "LVL2_TURBO_LICENSE",
    .udesc   = "Core cycles where the core was running in a manner where Turbo may be clipped to the AVX512 turbo schedule.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LVL1_TURBO_LICENSE",
    .udesc   = "Core cycles where the core was running in a manner where Turbo may be clipped to the AVX2 turbo schedule.",
    .ucode   = 0x1800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LVL0_TURBO_LICENSE",
    .udesc   = "Core cycles where the core was running in a manner where Turbo may be clipped to the Non-AVX turbo schedule.",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_l2_rqsts[]={
  { .uname   = "ALL_DEMAND_REFERENCES",
    .udesc   = "Demand requests to L2 cache",
    .ucode   = 0xe700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_CODE_RD",
    .udesc   = "L2 code requests",
    .ucode   = 0xe400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_RFO",
    .udesc   = "RFO requests to L2 cache",
    .ucode   = 0xe200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_DEMAND_DATA_RD",
    .udesc   = "Demand Data Read requests",
    .ucode   = 0xe100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SWPF_HIT",
    .udesc   = "SW prefetch requests that hit L2 cache.",
    .ucode   = 0xc800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CODE_RD_HIT",
    .udesc   = "L2 cache hits when fetching instructions, code reads.",
    .ucode   = 0xc400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RFO_HIT",
    .udesc   = "RFO requests that hit L2 cache",
    .ucode   = 0xc200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_HIT",
    .udesc   = "Demand Data Read requests that hit L2 cache",
    .ucode   = 0xc100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SWPF_MISS",
    .udesc   = "SW prefetch requests that miss L2 cache.",
    .ucode   = 0x2800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_DEMAND_MISS",
    .udesc   = "Demand requests that miss L2 cache",
    .ucode   = 0x2700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CODE_RD_MISS",
    .udesc   = "L2 cache misses when fetching instructions",
    .ucode   = 0x2400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RFO_MISS",
    .udesc   = "RFO requests that miss L2 cache",
    .ucode   = 0x2200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_MISS",
    .udesc   = "Demand Data Read miss L2, no rejects",
    .ucode   = 0x2100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_arith[]={
  { .uname   = "DIVIDER_ACTIVE",
    .udesc   = "Cycles when divide unit is busy executing divide or square root operations.",
    .ucode   = 0x0900ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_DFL,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t intel_icl_uops_issued[]={
  { .uname   = "STALL_CYCLES",
    .udesc   = "Cycles when RAT does not issue Uops to RS for the thread",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "VECTOR_WIDTH_MISMATCH",
    .udesc   = "Uops inserted at issue-stage in order to preserve upper bits of vector registers.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ANY",
    .udesc   = "Uops that RAT issues to RS",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_int_misc[]={
  { .uname   = "CLEAR_RESTEER_CYCLES",
    .udesc   = "Counts cycles after recovery from a branch misprediction or machine clear till the first uop is issued from the resteered path.",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "UOP_DROPPING",
    .udesc   = "TMA slots where uops got dropped",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_RECOVERY_CYCLES",
    .udesc   = "Cycles the Backend cluster is recovering after a miss-speculation or a Store Buffer or Load Buffer drain stall.",
    .ucode   = 0x0300ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "RECOVERY_CYCLES",
    .udesc   = "Core cycles the allocator was stalled due to recovery from earlier clear event for this thread",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_dtlb_load_misses[]={
  { .uname   = "STLB_HIT",
    .udesc   = "Loads that miss the DTLB and hit the STLB.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_ACTIVE",
    .udesc   = "Cycles when at least one PMH is busy with a page walk for a demand load.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "WALK_PENDING",
    .udesc   = "Number of page walks outstanding for a demand load in the PMH each cycle.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Load miss in all TLB levels causes a page walk that completes. (All page sizes)",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_2M_4M",
    .udesc   = "Page walks completed due to a demand data load to a 2M/4M page.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_4K",
    .udesc   = "Page walks completed due to a demand data load to a 4K page.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_ld_blocks_partial[]={
  { .uname   = "ADDRESS_ALIAS",
    .udesc   = "False dependencies in MOB due to partial compare on address.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t intel_icl_ld_blocks[]={
  { .uname   = "NO_SR",
    .udesc   = "The number of times that split load operations are temporarily blocked because all resources for handling the split accesses are in use.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STORE_FORWARD",
    .udesc   = "Loads blocked due to overlapping with a preceding store that cannot be forwarded.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_topdown[]={
  { .uname   = "BR_MISPREDICT_SLOTS",
    .udesc   = "TMA slots wasted due to incorrect speculation by branch mispredictions",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BACKEND_BOUND_SLOTS",
    .udesc   = "TMA slots where no uops were being issued due to lack of back-end resources.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOTS_P",
    .udesc   = "TMA slots available for an unhalted logical processor. General counter - architectural event",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOTS",
    .udesc   = "TMA slots available for an unhalted logical processor. Fixed counter - architectural event",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_cpu_clk_unhalted[]={
  { .uname   = "DISTRIBUTED",
    .udesc   = "Cycle counts are evenly distributed between active threads in the Core.",
    .ucode   = 0x02ecull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE,
  },
  { .uname   = "REF_DISTRIBUTED",
    .udesc   = "Core crystal clock cycles. Cycle counts are evenly distributed between active threads in the Core.",
    .ucode   = 0x08ecull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE,
  },
  { .uname   = "ONE_THREAD_ACTIVE",
    .udesc   = "Core crystal clock cycles when this thread is unhalted and the other thread is halted.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REF_XCLK",
    .udesc   = "Core crystal clock cycles when the thread is unhalted.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "THREAD_P",
    .udesc   = "Thread cycles when thread is not in halt state",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REF_TSC",
    .udesc   = "Reference cycles when the core is not in halt state.",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t intel_icl_inst_retired[]={
  { .uname   = "STALL_CYCLES",
    .udesc   = "Cycles without actually retired instructions.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "ANY_P",
    .udesc   = "Number of instructions retired. General Counter - architectural event",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PREC_DIST",
    .udesc   = "Precise instruction retired event with a reduced effect of PEBS shadow in IP distribution (Fixed counter 1 only. c, e, i, intx, intxcp modifiers not available)",
    .ucode   = 0x0100ull,
    .ucntmsk = 0x100000000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE | INTEL_X86_FIXED | INTEL_X86_PEBS,
    /*
     * because this encoding is for a fixed counter, not all modifiers are available. Given that we do not have per umask modmsk, we use
     * the hardcoded modifiers field instead. We mark all unavailable modifiers as set (to 0) so the user cannot modify them
     */
    .modhw   = _INTEL_X86_ATTR_INTX | _INTEL_X86_ATTR_INTXCP | _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_I,
  },
  { .uname   = "ANY",
    .udesc   = "Number of instructions retired. Fixed Counter - architectural event (c, e, i, intx, intxcp modifiers not available)",
    .ucode   = 0x0100ull,
    .ucntmsk = 0x100000000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE | INTEL_X86_FIXED,
    /*
     * because this encoding is for a fixed counter, not all modifiers are available. Given that we do not have per umask modmsk, we use
     * the hardcoded modifiers field instead. We mark all unavailable modifiers as set (to 0) so the user cannot modify them
     */
    .modhw   = _INTEL_X86_ATTR_INTX | _INTEL_X86_ATTR_INTXCP | _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_E,
  },
};

static const intel_x86_entry_t intel_icl_pe[]={
  { .name   = "UNHALTED_CORE_CYCLES",
    .desc   = "Count core clock cycles whenever the clock signal on the specific core is running (not halted)",
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x20000000full,
    .code = 0x3c,
  },
  { .name   = "UNHALTED_REFERENCE_CYCLES",
    .desc   = "Unhalted reference cycles",
    .modmsk = INTEL_FIXED3_ATTRS,
    .cntmsk = 0x400000000ull,
    .code = 0x0300, /* pseudo encoding */
    .flags = INTEL_X86_FIXED,
  },
  { .name   = "INSTRUCTION_RETIRED",
    .desc   = "Number of instructions at retirement",
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x1000000ffull,
    .code = 0xc0,
  },
  { .name   = "INSTRUCTIONS_RETIRED",
    .desc   = "Number of instructions at retirement",
    .modmsk = INTEL_V2_ATTRS,
    .equiv = "INSTRUCTION_RETIRED",
    .cntmsk = 0x1000000ffull,
    .code = 0xc0,
  },
  { .name   = "SQ_MISC",
    .desc   = "SuperQueue miscellaneous.",
    .code   = 0x00f4,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_sq_misc),
    .umasks = intel_icl_sq_misc,
  },
  { .name   = "L2_LINES_OUT",
    .desc   = "L2 lines evicted.",
    .code   = 0x00f2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_l2_lines_out),
    .umasks = intel_icl_l2_lines_out,
  },
  { .name   = "L2_LINES_IN",
    .desc   = "L2 lines allocated.",
    .code   = 0x00f1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_l2_lines_in),
    .umasks = intel_icl_l2_lines_in,
  },
  { .name   = "L2_TRANS",
    .desc   = "L2 transactions.",
    .code   = 0x00f0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_l2_trans),
    .umasks = intel_icl_l2_trans,
  },
  { .name   = "BACLEARS",
    .desc   = "Branch re-steers.",
    .code   = 0x00e6,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_baclears),
    .umasks = intel_icl_baclears,
  },
  { .name   = "MEM_LOAD_L3_HIT_RETIRED",
    .desc   = "L3 hit load uops retired.",
    .code   = 0x00d2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_mem_load_l3_hit_retired),
    .umasks = intel_icl_mem_load_l3_hit_retired,
  },
  { .name   = "MEM_LOAD_RETIRED",
    .desc   = "Retired load uops.",
    .code   = 0x00d1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_mem_load_retired),
    .umasks = intel_icl_mem_load_retired,
  },
  { .name   = "MEM_INST_RETIRED",
    .desc   = "Memory instructions retired.",
    .code   = 0x00d0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_mem_inst_retired),
    .umasks = intel_icl_mem_inst_retired,
  },
  { .name   = "MEM_LOAD_L3_MISS_RETIRED",
    .desc   = "Retired load instructions which data sources missed L3 but serviced from local dram",
    .code   = 0x00d3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_mem_load_l3_miss_retired),
    .umasks = intel_icl_mem_load_l3_miss_retired,
  },

  { .name   = "MEM_TRANS_RETIRED",
    .desc   = "Memory transactions retired,",
    .code   = 0x00cd,
    .modmsk = INTEL_V5_ATTRS | _INTEL_X86_ATTR_LDLAT,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_mem_trans_retired),
    .umasks = intel_icl_mem_trans_retired,
  },
  { .name   = "MISC_RETIRED",
    .desc   = "Miscellaneous retired events.",
    .code   = 0x00cc,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_misc_retired),
    .umasks = intel_icl_misc_retired,
  },
  { .name   = "RTM_RETIRED",
    .desc   = "RTM (Restricted Transaction Memory) execution.",
    .code   = 0x00c9,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_rtm_retired),
    .umasks = intel_icl_rtm_retired,
  },
  { .name   = "HLE_RETIRED",
    .desc   = "HLE (Hardware Lock Elision) execution.",
    .code   = 0x00c8,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_hle_retired),
    .umasks = intel_icl_hle_retired,
  },
  { .name   = "FP_ARITH_INST_RETIRED",
    .desc   = "Floating-point instructions retired.",
    .code   = 0x00c7,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_fp_arith_inst_retired),
    .umasks = intel_icl_fp_arith_inst_retired,
  },
  { .name   = "FP_ARITH",
    .desc   = "Floating-point instructions retired.",
    .equiv  = "FP_ARITH_INST_RETIRED",
    .code   = 0x00c7,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_fp_arith_inst_retired),
    .umasks = intel_icl_fp_arith_inst_retired,
  },
  { .name   = "FRONTEND_RETIRED",
    .desc   = "Precise frontend retired events.",
    .code   = 0x01c6,
    .modmsk = INTEL_SKL_FE_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_FRONTEND | INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_frontend_retired),
    .umasks = intel_icl_frontend_retired,
  },
  { .name   = "BR_MISP_RETIRED",
    .desc   = "Mispredicted branch instructions retired.",
    .code   = 0x00c5,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_br_misp_retired),
    .umasks = intel_icl_br_misp_retired,
  },
  { .name   = "BR_INST_RETIRED",
    .desc   = "Branch instructions retired.",
    .code   = 0x00c4,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_br_inst_retired),
    .umasks = intel_icl_br_inst_retired,
  },
  { .name   = "MACHINE_CLEARS",
    .desc   = "Machine clear asserted.",
    .code   = 0x00c3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_machine_clears),
    .umasks = intel_icl_machine_clears,
  },
  { .name   = "UOPS_RETIRED",
    .desc   = "Retired uops.",
    .code   = 0x00c2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_uops_retired),
    .umasks = intel_icl_uops_retired,
  },
  { .name   = "ASSISTS",
    .desc   = "Software assist.",
    .code   = 0x00c1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_assists),
    .umasks = intel_icl_assists,
  },
  { .name   = "TLB_FLUSH",
    .desc   = "Data TLB flushes.",
    .code   = 0x00bd,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_tlb_flush),
    .umasks = intel_icl_tlb_flush,
  },
  { .name   = "UOPS_EXECUTED",
    .desc   = "Uops executed.",
    .code   = 0x00b1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_uops_executed),
    .umasks = intel_icl_uops_executed,
  },
  { .name   = "OFFCORE_REQUESTS",
    .desc   = "Requests sent to uncore.",
    .code   = 0x00b0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_offcore_requests),
    .umasks = intel_icl_offcore_requests,
  },
  { .name   = "DSB2MITE_SWITCHES",
    .desc   = "Number of DSB to MITE switches.",
    .code   = 0x00ab,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_dsb2mite_switches),
    .umasks = intel_icl_dsb2mite_switches,
  },
  { .name   = "LSD",
    .desc   = "LSD (Loop stream detector) operations.",
    .code   = 0x00a8,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_lsd),
    .umasks = intel_icl_lsd,
  },
  { .name   = "EXE_ACTIVITY",
    .desc   = "Execution activity,",
    .code   = 0x00a6,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_exe_activity),
    .umasks = intel_icl_exe_activity,
  },
  { .name   = "CYCLE_ACTIVITY",
    .desc   = "Stalled cycles.",
    .code   = 0x00a3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_cycle_activity),
    .umasks = intel_icl_cycle_activity,
  },
  { .name   = "RESOURCE_STALLS",
    .desc   = "Cycles where Allocation is stalled due to Resource Related reasons.",
    .code   = 0x00a2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_resource_stalls),
    .umasks = intel_icl_resource_stalls,
  },
  { .name   = "UOPS_DISPATCHED",
    .desc   = "Uops dispatched to specific ports",
    .code   = 0x00a1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_uops_dispatched),
    .umasks = intel_icl_uops_dispatched,
  },
  { .name   = "UOPS_DISPATCHED_PORT",
    .desc   = "Uops dispatched to specific ports",
    .equiv  = "UOPS_DISPATCHED",
    .code   = 0x00a1,
    .cntmsk = 0xff,
    .ngrp   = 1,
    .modmsk = INTEL_V5_ATTRS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_uops_dispatched),
    .umasks = intel_icl_uops_dispatched,
    .flags  = INTEL_X86_SPEC,
  },

  { .name   = "IDQ_UOPS_NOT_DELIVERED",
    .desc   = "Uops not delivered.",
    .code   = 0x009c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_idq_uops_not_delivered),
    .umasks = intel_icl_idq_uops_not_delivered,
  },
  { .name   = "ILD_STALL",
    .desc   = "ILD (Instruction Length Decoder) stalls.",
    .code   = 0x0087,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_ild_stall),
    .umasks = intel_icl_ild_stall,
  },
  { .name   = "ITLB_MISSES",
    .desc = "Instruction TLB misses.",
    .code   = 0x0085,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_itlb_misses),
    .umasks = intel_icl_itlb_misses,
  },
  { .name   = "ICACHE_64B",
    .desc = "Instruction Cache.",
    .code   = 0x0083,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_icache_64b),
    .umasks = intel_icl_icache_64b,
  },
  { .name   = "ICACHE_16B",
    .desc = "Instruction Cache.",
    .code   = 0x0080,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_icache_16b),
    .umasks = intel_icl_icache_16b,
  },
  { .name   = "IDQ",
    .desc   = "IDQ (Instruction Decoded Queue) operations",
    .code   = 0x0079,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_idq),
    .umasks = intel_icl_idq,
  },
  { .name   = "OFFCORE_REQUESTS_OUTSTANDING",
    .desc   = "Outstanding offcore requests.",
    .code   = 0x0060,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_offcore_requests_outstanding),
    .umasks = intel_icl_offcore_requests_outstanding,
  },
  { .name   = "RS_EVENTS",
    .desc   = "Reservation Station.",
    .code   = 0x005e,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_rs_events),
    .umasks = intel_icl_rs_events,
  },
  { .name   = "TX_EXEC",
    .desc = "Transactional execution.",
    .code   = 0x005d,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_tx_exec),
    .umasks = intel_icl_tx_exec,
  },
  { .name   = "TX_MEM",
    .desc   = "Transactional memory.",
    .code   = 0x0054,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_tx_mem),
    .umasks = intel_icl_tx_mem,
  },
  { .name   = "L1D",
    .desc   = "L1D cache.",
    .code   = 0x0051,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_l1d),
    .umasks = intel_icl_l1d,
  },
  { .name   = "LOAD_HIT_PREFETCH",
    .desc   = "Load dispatches.",
    .code   = 0x004c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_load_hit_prefetch),
    .umasks = intel_icl_load_hit_prefetch,
  },
  { .name   = "LOAD_HIT_PRE",
    .desc   = "Load dispatches.",
    .equiv  = "LOAD_HIT_PREFETCH",
    .code   = 0x004c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_load_hit_prefetch),
    .umasks = intel_icl_load_hit_prefetch,
  },
  { .name   = "DTLB_STORE_MISSES",
    .desc = "Data TLB store misses.",
    .code   = 0x0049,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_dtlb_store_misses),
    .umasks = intel_icl_dtlb_store_misses,
  },
  { .name   = "L1D_PEND_MISS",
    .desc   = "L1D pending misses.",
    .code   = 0x0048,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_l1d_pend_miss),
    .umasks = intel_icl_l1d_pend_miss,
  },
  { .name   = "SW_PREFETCH_ACCESS",
    .desc   = "Software prefetches.",
    .code   = 0x0032,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = 0,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_sw_prefetch_access),
    .umasks = intel_icl_sw_prefetch_access,
  },
  { .name   = "SW_PREFETCH",
    .desc   = "Software prefetches.",
    .equiv  = "SW_PREFETCH_ACCESS",
    .code   = 0x0032,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_sw_prefetch_access),
    .umasks = intel_icl_sw_prefetch_access,
  },
  { .name   = "LONGEST_LAT_CACHE",
    .desc   = "L3 cache.",
    .code   = 0x002e,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_longest_lat_cache),
    .umasks = intel_icl_longest_lat_cache,
  },
  { .name   = "CORE_POWER",
    .desc   = "Power power cycles.",
    .code   = 0x0028,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_core_power),
    .umasks = intel_icl_core_power,
  },
  { .name   = "L2_RQSTS",
    .desc   = "L2 requests.",
    .code   = 0x0024,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_l2_rqsts),
    .umasks = intel_icl_l2_rqsts,
  },
  { .name   = "ARITH",
    .desc   = "Arithmetic uops.",
    .code   = 0x0014,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_arith),
    .umasks = intel_icl_arith,
  },
  { .name   = "UOPS_ISSUED",
    .desc   = "Uops issued.",
    .code   = 0x000e,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_uops_issued),
    .umasks = intel_icl_uops_issued,
  },
  { .name   = "INT_MISC",
    .desc   = "Miscellaneous interruptions.",
    .code   = 0x000d,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_int_misc),
    .umasks = intel_icl_int_misc,
  },
  { .name   = "DTLB_LOAD_MISSES",
    .desc   = "Data TLB load misses.",
    .code   = 0x0008,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_dtlb_load_misses),
    .umasks = intel_icl_dtlb_load_misses,
  },
  { .name   = "LD_BLOCKS_PARTIAL",
    .desc   = "Partial load blocks.",
    .code   = 0x0007,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_ld_blocks_partial),
    .umasks = intel_icl_ld_blocks_partial,
  },
  { .name   = "LD_BLOCKS",
    .desc   = "Blocking loads.",
    .code   = 0x0003,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_ld_blocks),
    .umasks = intel_icl_ld_blocks,
  },
  { .name   = "TOPDOWN",
    .desc   = "TMA slots available for an unhalted logical processor.",
    .code   = 0x0000,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0x800000000ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_topdown),
    .umasks = intel_icl_topdown,
  },
  { .name   = "CPU_CLK_UNHALTED",
    .desc   = "Count core clock cycles whenever the clock signal on the specific core is running (not halted).",
    .code   = 0x003c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0x200000000ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_cpu_clk_unhalted),
    .umasks = intel_icl_cpu_clk_unhalted,
  },
  { .name   = "INST_RETIRED",
    .desc   = "Number of instructions retired",
    .code   = 0xc0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_inst_retired),
    .umasks = intel_icl_inst_retired,
  },
  { .name   = "OFFCORE_RESPONSE_0",
    .desc   = "Offcore response event",
    .code   = 0x01b7,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC | INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_ocr),
    .umasks = intel_icl_ocr,
  },
  { .name   = "OFFCORE_RESPONSE_1",
    .desc   = "Offcore response event",
    .code   = 0x01bb,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC | INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_ocr),
    .umasks = intel_icl_ocr,
  },
  { .name   = "OCR",
    .desc   = "Offcore response event",
    .equiv  = "OFFCORE_RESPONSE_0",
    .code   = 0x01b7,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC | INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(intel_icl_ocr),
    .umasks = intel_icl_ocr,
  },
};
/* 56 events available */
