/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* libwpd
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002, 2005 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2002, 2004 Marc Maurer (uwog@uwog.net)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

/* "This product is not manufactured, approved, or supported by
 * Corel Corporation or Corel Corporation Limited."
 */
#include "libwpd_internal.h"
#include <librevenge-stream/librevenge-stream.h>
#include <cstdarg>
#include <cstdio>
#include <ctype.h>
#include <locale.h>
#include <string>
#include <vector>

#include <boost/spirit/include/qi.hpp>

#ifdef DEBUG
namespace libwpd
{

void debugPrint(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	std::vfprintf(stderr, format, args);
	va_end(args);
}

}
#endif

namespace
{

static int libwpd_unichar_to_utf8(unsigned c, char *outbuf)
{
	unsigned char len = 1;
	unsigned char first = 0;

	if (c < 0x80)
	{
		first = 0;
		len = 1;
	}
	else if (c < 0x800)
	{
		first = 0xc0;
		len = 2;
	}
	else if (c < 0x10000)
	{
		first = 0xe0;
		len = 3;
	}
	else if (c < 0x200000)
	{
		first = 0xf0;
		len = 4;
	}
	else if (c < 0x4000000)
	{
		first = 0xf8;
		len = 5;
	}
	else
	{
		first = 0xfc;
		len = 6;
	}

	if (outbuf)
	{
		for (auto i = (unsigned char)(len - 1); i > 0; --i)
		{
			outbuf[i] = (char)((c & 0x3f) | 0x80);
			c >>= 6;
		}
		outbuf[0] = (char)(c | first);
	}

	return len;
}

} // anonymous namespace

unsigned char readU8(librevenge::RVNGInputStream *input, WPXEncryption *encryption)
{
	unsigned long numBytesRead = 0;
	unsigned char const *p = (encryption ?
	                          encryption->readAndDecrypt(input, sizeof(unsigned char), numBytesRead) :
	                          input->read(sizeof(unsigned char), numBytesRead));

	if (!p || numBytesRead != sizeof(unsigned char))
		throw FileException();

	return p[0];
}

unsigned short readU16(librevenge::RVNGInputStream *input, WPXEncryption *encryption, bool bigendian)
{
	unsigned long numBytesRead = 0;
	unsigned char const *p = (encryption ?
	                          encryption->readAndDecrypt(input, sizeof(unsigned short), numBytesRead) :
	                          input->read(sizeof(unsigned short), numBytesRead));

	if (!p || numBytesRead != sizeof(unsigned short))
		throw FileException();

	if (bigendian)
		return (unsigned short)(p[1]|((unsigned short)p[0]<<8));
	return (unsigned short)(p[0]|((unsigned short)p[1]<<8));
}

signed short readS16(librevenge::RVNGInputStream *input, WPXEncryption *encryption, bool bigendian)
{
	return (signed short)readU16(input, encryption, bigendian);
}

unsigned readU32(librevenge::RVNGInputStream *input, WPXEncryption *encryption, bool bigendian)
{
	unsigned long numBytesRead = 0;
	unsigned char const *p = (encryption ?
	                          encryption->readAndDecrypt(input, sizeof(unsigned), numBytesRead) :
	                          input->read(sizeof(unsigned), numBytesRead));

	if (!p || numBytesRead != sizeof(unsigned))
		throw FileException();

	if (bigendian)
		return (unsigned)p[3]|((unsigned)p[2]<<8)|((unsigned)p[1]<<16)|((unsigned)p[0]<<24);
	return (unsigned)p[0]|((unsigned)p[1]<<8)|((unsigned)p[2]<<16)|((unsigned)p[3]<<24);
}

void appendUCS4(librevenge::RVNGString &str, unsigned ucs4)
{
	int charLength = libwpd_unichar_to_utf8(ucs4, nullptr);
	std::vector<char> utf8(charLength+1);
	utf8[charLength] = '\0';
	libwpd_unichar_to_utf8(ucs4, utf8.data());
	str.append(utf8.data());
}

librevenge::RVNGString readPascalString(librevenge::RVNGInputStream *input, WPXEncryption *encryption)
{
	int pascalStringLength = readU8(input, encryption);
	librevenge::RVNGString tmpString;
	for (int i=0; i<pascalStringLength; i++)
	{
		unsigned short tmpChar = readU8(input, encryption);
		if (tmpChar <= 0x7f)
			tmpString.append((char)tmpChar);
		else if (pascalStringLength > i++)
		{
			tmpChar = (unsigned short)((tmpChar << 8) | readU8(input, encryption));
			const unsigned *chars;
			int len = appleWorldScriptToUCS4(tmpChar, &chars);
			for (int j = 0; j < len; j++)
				appendUCS4(tmpString, chars[j]);
		}
	}
	return tmpString;
}

librevenge::RVNGString readCString(librevenge::RVNGInputStream *input, WPXEncryption *encryption)
{
	librevenge::RVNGString tmpString;
	char character;
	while ((character = (char)readU8(input, encryption)) != '\0')
		tmpString.append(character);
	return tmpString;
}

struct WPXComplexMap
{
	unsigned short charToMap;
	unsigned unicodeChars[6];
};


// the ascii map appears stupid, but we need the const 32-bit data for now
static const unsigned asciiMap[] =
{
	32,  33,  34,  35,  36,  37,  38,  39,
	40,  41,  42,  43,  44,  45,  46,  47,
	48,  49,  50,  51,  52,  53,  54,  55,
	56,  57,  58,  59,  60,  61,  62,  63,
	64,  65,  66,  67,  68,  69,  70,  71,
	72,  73,  74,  75,  76,  77,  78,  79,
	80,  81,  82,  83,  84,  85,  86,  87,
	88,  89,  90,  91,  92,  93,  94,  95,
	96,  97,  98,  99, 100, 101, 102, 103,
	104, 105, 106, 107, 108, 109, 110, 111,
	112, 113, 114, 115, 116, 117, 118, 119,
	120, 121, 122, 123, 124, 125, 126
};


static int findSimpleMap(unsigned short character, const unsigned **chars, const unsigned *simpleMap, const size_t simpleMapSize)
{
	if ((character < simpleMapSize) && simpleMap[character])
	{
		*chars = &simpleMap[character];
		return 1;
	}

	return 0;
}

static int findComplexMap(unsigned short character, const unsigned **chars, const WPXComplexMap *complexMap)
{
	if (!complexMap)
		return 0;

	unsigned i = 0;
	while (complexMap[i].charToMap)
	{
		if (complexMap[i].charToMap == character)
			break;
		i++;
	}
	if (!(complexMap[i].charToMap) || !(complexMap[i].unicodeChars[0]))
		return 0;

	*chars = complexMap[i].unicodeChars;

	for (unsigned j = 0; j<WPD_NUM_ELEMENTS(complexMap[i].unicodeChars); j++)
	{
		if (!(complexMap[i].unicodeChars[j]))
			return (int)j;
	}

	return 0;
}

/* WP6 Extended Character -> Unicode (UCS4) Mappings by Ariya Hidayat <ariyahidayat@yahoo.de> for the
   KWord project, licensed under the LGPL */

/* WP6 multinational characters (charset 1) */
static const unsigned multinationalWP6[] =
{
	0x0300, 0x00b7, 0x0303, 0x0302, 0x0335, 0x0338, 0x0301, 0x0308, // 0 - 7
	0x0304, 0x0313, 0x0315, 0x02bc, 0x0326, 0x0315, 0x00b0, 0x0307, // 8 - 15
	0x030b, 0x0327, 0x0328, 0x030c, 0x0337, 0x0305, 0x0306, 0x00df, // 16
	0x0138, 0x006a, 0x00c1, 0x00e1, 0x00c2, 0x00e2, 0x00c4, 0x00e4, // 24
	0x00c0, 0x00e0, 0x00c5, 0x00e5, 0x00c6, 0x00e6, 0x00c7, 0x00e7, // 32
	0x00c9, 0x00e9, 0x00ca, 0x00ea, 0x00cb, 0x00eb, 0x00c8, 0x00e8, // 40
	0x00cd, 0x00ed, 0x00ce, 0x00ee, 0x00cf, 0x00ef, 0x00cc, 0x00ec, // 48
	0x00d1, 0x00f1, 0x00d3, 0x00f3, 0x00d4, 0x00f4, 0x00d6, 0x00f6, // 56
	0x00d2, 0x00f2, 0x00da, 0x00fa, 0x00db, 0x00fb, 0x00dc, 0x00fc, // 64
	0x00d9, 0x00f9, 0x0178, 0x00ff, 0x00c3, 0x00e3, 0x0110, 0x0111, // 72
	0x00d8, 0x00f8, 0x00d5, 0x00f5, 0x00dd, 0x00fd, 0x00d0, 0x00f0, // 80
	0x00de, 0x00fe, 0x0102, 0x0103, 0x0100, 0x0101, 0x0104, 0x0105, // 88
	0x0106, 0x0107, 0x010c, 0x010d, 0x0108, 0x0109, 0x010a, 0x010b, // 96
	0x010e, 0x010f, 0x011a, 0x011b, 0x0116, 0x0117, 0x0112, 0x0113, // 104
	0x0118, 0x0119, 0x01f4, 0x01f5, 0x011e, 0x011f, 0x01e6, 0x01e7, // 112
	0x0122, 0x0123, 0x011c, 0x011d, 0x0120, 0x0121, 0x0124, 0x0125, // 120
	0x0126, 0x0127, 0x0130, 0x0069, 0x012a, 0x012b, 0x012e, 0x012f, // 128
	0x0128, 0x0129, 0x0132, 0x0133, 0x0134, 0x0135, 0x0136, 0x0137, // 136
	0x0139, 0x013a, 0x013d, 0x013e, 0x013b, 0x013c, 0x013f, 0x0140, // 144
	0x0141, 0x0142, 0x0143, 0x0144, 0x0000, 0x0149, 0x0147, 0x0148, // 152
	0x0145, 0x0146, 0x0150, 0x0151, 0x014c, 0x014d, 0x0152, 0x0153, // 160
	0x0154, 0x0155, 0x0158, 0x0159, 0x0156, 0x0157, 0x015a, 0x015b, // 168
	0x0160, 0x0161, 0x015e, 0x015f, 0x015c, 0x015d, 0x0164, 0x0165, // 176
	0x0162, 0x0163, 0x0166, 0x0167, 0x016c, 0x016d, 0x0170, 0x0171, // 184
	0x016a, 0x016b, 0x0172, 0x0173, 0x016e, 0x016f, 0x0168, 0x0169, // 192
	0x0174, 0x0175, 0x0176, 0x0177, 0x0179, 0x017a, 0x017d, 0x017e, // 200
	0x017b, 0x017c, 0x014a, 0x014b, 0x0000, 0x0000, 0x0000, 0x0000, // 208
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 216
	0x0000, 0x0000, 0x1ef2, 0x1ef3, 0x010e, 0x010f, 0x01a0, 0x01a1, // 224
	0x01af, 0x01b0, 0x0114, 0x0115, 0x012c, 0x012d, 0x0049, 0x0131, // 232
	0x014e, 0x014f                                                  // 240 - 241
};

/* WP6 multinational characters (charset 1) - multicharacter mapping */
static const WPXComplexMap multinationalWP6Complex[] =
{
	{ 156, { 0x02bc, 0x004e, 0x0000 } },

	{ 212, { 0x0304, 0x0044, 0x0000 } },
	{ 213, { 0x0304, 0x0064, 0x0000 } },
	{ 214, { 0x0304, 0x004c, 0x0000 } },
	{ 215, { 0x0304, 0x006c, 0x0000 } },

	{ 216, { 0x0304, 0x004e, 0x0000 } },
	{ 217, { 0x0304, 0x006e, 0x0000 } },
	{ 218, { 0x0300, 0x0052, 0x0000 } },
	{ 219, { 0x0300, 0x0072, 0x0000 } },
	{ 220, { 0x0304, 0x0053, 0x0000 } },
	{ 221, { 0x0304, 0x0073, 0x0000 } },
	{ 222, { 0x0304, 0x0054, 0x0000 } },
	{ 223, { 0x0304, 0x0074, 0x0000 } },

	{ 224, { 0x0306, 0x0059, 0x0000 } },
	{ 225, { 0x0306, 0x0079, 0x0000 } },

	{ 0, { 0x0000 } }
};


/* WP6 phonetic symbol (charset 2) */
static const unsigned phoneticWP6[] =
{
	0x02b9, 0x02ba, 0x02bb, 0x0020, 0x02bd, 0x02bc, 0x0020, 0x02be,
	0x02bf, 0x0310, 0x02d0, 0x02d1, 0x0306, 0x032e, 0x0329, 0x02c8,
	0x02cc, 0x02c9, 0x02ca, 0x02cb, 0x02cd, 0x02ce, 0x02cf, 0x02c6,
	0x02c7, 0x02dc, 0x0325, 0x02da, 0x032d, 0x032c, 0x0323, 0x0308,
	0x0324, 0x031c, 0x031d, 0x031e, 0x031f, 0x0320, 0x0321, 0x0322,
	0x032a, 0x032b, 0x02d2, 0x02d3, 0x0361, 0x0356, 0x005f, 0x2017,
	0x033e, 0x02db, 0x0327, 0x0233, 0x030d, 0x02b0, 0x02b6, 0x0250,
	0x0251, 0x0252, 0x0253, 0x0299, 0x0254, 0x0255, 0x0297, 0x0256,
	0x0257, 0x0258, 0x0259, 0x025a, 0x025b, 0x025c, 0x025d, 0x029a,
	0x025e, 0x025f, 0x0278, 0x0261, 0x0260, 0x0262, 0x029b, 0x0263,
	0x0264, 0x0265, 0x0266, 0x0267, 0x029c, 0x0268, 0x026a, 0x0269,
	0x029d, 0x029e, 0x026b, 0x026c, 0x026d, 0x029f, 0x026e, 0x028e,
	0x026f, 0x0270, 0x0271, 0x0272, 0x0273, 0x0274, 0x0276, 0x0277,
	0x02a0, 0x0279, 0x027a, 0x027b, 0x027c, 0x027d, 0x027e, 0x027f,
	0x0280, 0x0281, 0x0282, 0x0283, 0x0284, 0x0285, 0x0286, 0x0287,
	0x0288, 0x0275, 0x0289, 0x028a, 0x028c, 0x028b, 0x028d, 0x03c7,
	0x028f, 0x0290, 0x0291, 0x0292, 0x0293, 0x0294, 0x0295, 0x0296,
	0x02a1, 0x02a2, 0x0298, 0x02a3, 0x02a4, 0x02a5, 0x02a6, 0x02a7,
	0x02a8
};

/* WP6 box drawing symbol (charset 3) */
static const unsigned boxdrawingWP6[] =
{
	0x2591, 0x2592, 0x2593, 0x2588, 0x258c, 0x2580, 0x2590, 0x2584,
	0x2500, 0x2502, 0x250c, 0x2510, 0x2518, 0x2514, 0x251c, 0x252c,
	0x2524, 0x2534, 0x253c, 0x2550, 0x2551, 0x2554, 0x2557, 0x255d,
	0x255a, 0x2560, 0x2566, 0x2563, 0x2569, 0x256c, 0x2552, 0x2555,
	0x255b, 0x2558, 0x2553, 0x2556, 0x255c, 0x2559, 0x255e, 0x2565,
	0x2561, 0x2568, 0x255f, 0x2564, 0x2562, 0x2567, 0x256b, 0x256a,
	0x2574, 0x2575, 0x2576, 0x2577, 0x2578, 0x2579, 0x257a, 0x257b,
	0x257c, 0x257e, 0x257d, 0x257f, 0x251f, 0x2522, 0x251e, 0x2521,
	0x252e, 0x2532, 0x252d, 0x2531, 0x2527, 0x2526, 0x252a, 0x2529,
	0x2536, 0x253a, 0x2535, 0x2539, 0x2541, 0x2546, 0x253e, 0x2540,
	0x2544, 0x254a, 0x253d, 0x2545, 0x2548, 0x2543, 0x2549, 0x2547
};

/* WP6 typographic symbol (charset 4) */
static const unsigned typographicWP6[] =
{
	0x25cf, 0x25cb, 0x25a0, 0x2022, 0x002a, 0x00b6, 0x00a7, 0x00a1,
	0x00bf, 0x00ab, 0x00bb, 0x00a3, 0x00a5, 0x20a7, 0x0192, 0x00aa,
	0x00ba, 0x00bd, 0x00bc, 0x00a2, 0x00b2, 0x207f, 0x00ae, 0x00a9,
	0x00a4, 0x00be, 0x00b3, 0x201b, 0x2019, 0x2018, 0x201f, 0x201d,
	0x201c, 0x2013, 0x2014, 0x2039, 0x203a, 0x25cb, 0x25a1, 0x2020,
	0x2021, 0x2122, 0x2120, 0x211e, 0x25cf, 0x25e6, 0x25a0, 0x25aa,
	0x25a1, 0x25ab, 0x2012, 0xfb00, 0xfb03, 0xfb04, 0xfb01, 0xfb02,
	0x2026, 0x0024, 0x20a3, 0x20a2, 0x20a0, 0x20a4, 0x201a, 0x201e,
	0x2153, 0x2154, 0x215b, 0x215c, 0x215d, 0x215e, 0x24c2, 0x24c5,
	0x20ac, 0x2105, 0x2106, 0x2030, 0x2116, 0x2014, 0x00b9, 0x2409,
	0x240c, 0x240d, 0x240a, 0x2424, 0x240b, 0x267c, 0x20a9, 0x20a6,
	0x20a8, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x1d11e, 0x1d122
};

/* WP6 iconic symbol (charset 5) */
static const unsigned iconicWP6[] =
{
	0x2661, 0x2662, 0x2667, 0x2664, 0x2642, 0x2640, 0x263c, 0x263a,
	0x263b, 0x266a, 0x266c, 0x25ac, 0x2302, 0x203c, 0x221a, 0x21a8,
	0x2310, 0x2319, 0x25d8, 0x25d9, 0x21b5, 0x2104, 0x261c, 0x23b5,
	0x2610, 0x2612, 0x2639, 0x266f, 0x266d, 0x266e, 0x260e, 0x231a,
	0x231b, 0x2701, 0x2702, 0x2703, 0x2704, 0x260e, 0x2706, 0x2707,
	0x2708, 0x2709, 0x261b, 0x261e, 0x270c, 0x270d, 0x270e, 0x270f,
	0x2710, 0x2711, 0x2712, 0x2713, 0x2714, 0x2715, 0x2716, 0x2717,
	0x2718, 0x2719, 0x271a, 0x271b, 0x271c, 0x271d, 0x271e, 0x271f,
	0x2720, 0x2721, 0x2722, 0x2723, 0x2724, 0x2725, 0x2726, 0x2727,
	0x2605, 0x2606, 0x272a, 0x272b, 0x272c, 0x272d, 0x272e, 0x272f,
	0x2730, 0x2731, 0x2732, 0x2733, 0x2734, 0x2735, 0x2736, 0x2737,
	0x2738, 0x2739, 0x273a, 0x273b, 0x273c, 0x273d, 0x273e, 0x273f,
	0x2740, 0x2741, 0x2742, 0x2743, 0x2744, 0x2745, 0x2746, 0x2747,
	0x2748, 0x2749, 0x274a, 0x274b, 0x25cf, 0x274d, 0x25a0, 0x274f,
	0x2750, 0x2751, 0x2752, 0x25b2, 0x25bc, 0x25c6, 0x2756, 0x25d7,
	0x2758, 0x2759, 0x275a, 0x275b, 0x275c, 0x275d, 0x275e, 0x2036,
	0x2033, 0x0020, 0x0020, 0x0020, 0x0020, 0x2329, 0x232a, 0x005b,
	0x005d, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x2190, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x21e8, 0x21e6,
	0x2794, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x25d6, 0x0020,
	0x0020, 0x2761, 0x2762, 0x2763, 0x2764, 0x2765, 0x2766, 0x2767,
	0x2663, 0x2666, 0x2665, 0x2660, 0x2780, 0x2781, 0x2782, 0x2783,
	0x2784, 0x2785, 0x2786, 0x2787, 0x2788, 0x2789, 0x2776, 0x2777,
	0x2778, 0x2779, 0x277a, 0x277b, 0x277c, 0x277d, 0x277e, 0x277f,
	0x2780, 0x2781, 0x2782, 0x2783, 0x2784, 0x2785, 0x2786, 0x2787,
	0x2788, 0x2789, 0x278a, 0x278b, 0x278c, 0x278d, 0x278e, 0x278f,
	0x2790, 0x2791, 0x2792, 0x2793, 0x2794, 0x2192, 0x2194, 0x2195,
	0x2798, 0x2799, 0x279a, 0x279b, 0x279c, 0x279d, 0x279e, 0x279f,
	0x27a0, 0x27a1, 0x27a2, 0x27a3, 0x27a4, 0x27a5, 0x27a6, 0x27a7,
	0x27a8, 0x27a9, 0x27aa, 0x27ab, 0x27ac, 0x27ad, 0x27ae, 0x27af,
	0x0020, 0x27b1, 0x27b2, 0x27b3, 0x27b4, 0x27b5, 0x27b6, 0x27b7,
	0x27b8, 0x27b9, 0x27ba, 0x27bb, 0x27bc, 0x27bd, 0x27be
};

/* WP6 math/scientific (charset 6) */
static const unsigned mathWP6[] =
{
	0x2212, 0x00b1, 0x2264, 0x2265, 0x221d, 0x002f, 0x2215, 0x2216, // 0 - 7
	0x00f7, 0x2223, 0x27e8, 0x27e9, 0x223c, 0x2248, 0x2261, 0x2208, // 8 - 15
	0x2229, 0x2225, 0x2211, 0x221e, 0x00ac, 0x2192, 0x2190, 0x2191, // 16 -23
	0x2193, 0x2194, 0x2195, 0x25b8, 0x25c2, 0x25b4, 0x25be, 0x22c5, // 24 - 31
	0x00b7, 0x2218, 0x2219, 0x212b, 0x00b0, 0x00b5, 0x203e, 0x00d7, // 32 - 39
	0x222b, 0x220f, 0x2213, 0x2207, 0x2202, 0x2032, 0x2033, 0x2192, // 40 - 47
	0x212f, 0x2113, 0x210f, 0x2111, 0x211c, 0x2118, 0x21c4, 0x21c6, // 48 - 55
	0x21d2, 0x21d0, 0x21d1, 0x21d3, 0x21d4, 0x21d5, 0x2197, 0x2198, // 56 - 63
	0x2196, 0x2199, 0x222a, 0x2282, 0x2283, 0x2286, 0x2287, 0x220d, // 64 - 71
	0x2205, 0x2308, 0x2309, 0x230a, 0x230b, 0x226a, 0x226b, 0x2220, // 72 - 79
	0x2297, 0x2295, 0x2296, 0x2a38, 0x2299, 0x2227, 0x2228, 0x22bb, // 80 - 87
	0x22a4, 0x22a5, 0x2312, 0x22a2, 0x22a3, 0x25a1, 0x25a0, 0x25ca, // 88 - 95
	0x25c6, 0x27e6, 0x27e7, 0x2260, 0x2262, 0x2235, 0x2234, 0x2237, // 96 - 103
	0x222e, 0x2112, 0x212d, 0x2128, 0x2118, 0x20dd, 0x29cb, 0x25c7, // 104 - 111
	0x22c6, 0x2034, 0x2210, 0x2243, 0x2245, 0x227a, 0x227c, 0x227b, // 112 - 119
	0x227d, 0x2203, 0x2200, 0x22d8, 0x22d9, 0x228e, 0x228a, 0x228b, // 120 - 127
	0x2293, 0x2294, 0x228f, 0x2291, 0x22e4, 0x2290, 0x2292, 0x22e5, // 128 - 135
	0x25b3, 0x25bd, 0x25c3, 0x25b9, 0x22c8, 0x2323, 0x2322, 0x25ef, // 136 - 143
	0x219d, 0x21a9, 0x21aa, 0x21a3, 0x21bc, 0x21bd, 0x21c0, 0x21c1, // 144 - 151
	0x21cc, 0x21cb, 0x21bf, 0x21be, 0x21c3, 0x21c2, 0x21c9, 0x21c7, // 152 - 159
	0x22d3, 0x22d2, 0x22d0, 0x22d1, 0x229a, 0x229b, 0x229d, 0x2127, // 160 - 167
	0x2221, 0x2222, 0x25c3, 0x25b9, 0x25b5, 0x25bf, 0x2214, 0x2250, // 168 - 175
	0x2252, 0x2253, 0x224e, 0x224d, 0x22a8, 0x2258, 0x226c, 0x0285, // 176 - 183
	0x2605, 0x226e, 0x2270, 0x226f, 0x2271, 0x2241, 0x2244, 0x2247, // 184 - 191
	0x2249, 0x2280, 0x22e0, 0x2281, 0x22e1, 0x2284, 0x2285, 0x2288, // 192 - 199
	0x2289, 0x0020, 0x0020, 0x22e2, 0x22e3, 0x2226, 0x2224, 0x226d, // 200 - 207
	0x2204, 0x2209, 0x2247, 0x2130, 0x2131, 0x2102, 0x0020, 0x2115, // 208 - 215
	0x211d, 0x225f, 0x22be, 0x220b, 0x22ef, 0x2026, 0x22ee, 0x22f1, // 216 - 223
	0x0020, 0x20e1, 0x002b, 0x002d, 0x003d, 0x002a, 0x2032, 0x2033, // 224 - 231
	0x2034, 0x210b, 0x2118, 0x2272, 0x2273, 0x0020                  // 232 - 237
};

/* WP6 math/scientific extended (charset 7) */
static const unsigned mathextWP6[] =
{
	0x2320, 0x2321, 0x23a5, 0x23bd, 0x221a, 0x0020, 0x2211, 0x220f,
	0x2210, 0x222b, 0x222e, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x23a7, 0x23a8, 0x23a9, 0x23aa, 0x0020, 0x0020, 0x0020,
	0x0020, 0x23ab, 0x23ac, 0x23ad, 0x23aa, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x222a, 0x222b, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x239b, 0x239d, 0x239c, 0x0020, 0x0020, 0x0020, 0x0020, 0x239e,
	0x23a8, 0x239f, 0x0020, 0x0020, 0x0020, 0x0020, 0x23a1, 0x23a3,
	0x23a2, 0x0020, 0x20aa, 0x0020, 0x0020, 0x23a4, 0x23a6, 0x23a5,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x22c3, 0x22c2, 0x228e, 0x2a04, 0x2294, 0x2a06, 0x2227, 0x22c0,
	0x2228, 0x22c1, 0x2297, 0x2a02, 0x2295, 0x2a01, 0x2299, 0x2a00,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x0020, 0x0020, 0x229d, 0x0020, 0x2238, 0x0020,
	0x27e6, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x27e7,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x21bc, 0x21bd, 0x0020, 0x296c, 0x296d, 0x296a, 0x296b, 0x0020,
	0x21c9, 0x21c7, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x21be, 0x21bf, 0x21c3, 0x21c2, 0x0020, 0x2293, 0x2a05, 0x23a1,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020
};

/* WP6 greek (charset 8) */
static const unsigned greekWP6[] =
{
	0x0391, 0x03b1, 0x0392, 0x03b2, 0x0392, 0x03d0, 0x0393, 0x03b3,
	0x0394, 0x03b4, 0x0395, 0x03b5, 0x0396, 0x03b6, 0x0397, 0x03b7,
	0x0398, 0x03b8, 0x0399, 0x03b9, 0x039a, 0x03ba, 0x039b, 0x03bb,
	0x039c, 0x03bc, 0x039d, 0x03bd, 0x039e, 0x03be, 0x039f, 0x03bf,
	0x03a0, 0x03c0, 0x03a1, 0x03c1, 0x03a3, 0x03c3, 0x03a3, 0x03c2,
	0x03a4, 0x03c4, 0x03a5, 0x03c5, 0x03a6, 0x03c6, 0x03a7, 0x03c7,
	0x03a8, 0x03c8, 0x03a9, 0x03c9, 0x0386, 0x03ac, 0x0388, 0x03ad,
	0x0389, 0x03ae, 0x038a, 0x03af, 0x03aa, 0x03ca, 0x038c, 0x03cc,
	0x038e, 0x03cd, 0x03ab, 0x03cb, 0x038f, 0x03ce, 0x03b5, 0x03d1,
	0x03f0, 0x03d6, 0x03f1, 0x03c2, 0x03d2, 0x03d5, 0x03c9, 0x037e,
	0x0387, 0x0374, 0x0375, 0x0384, 0x00a8, 0x0385, 0x1fed, 0x1fef,
	0x1fc0, 0x1fbd, 0x1ffe, 0x037a, 0x1fce, 0x1fde, 0x1fcd, 0x1fdd,
	0x1fcf, 0x1fdf, 0x0384, 0x1fef, 0x1fc0, 0x1fbd, 0x1ffe, 0x1fce,
	0x1fde, 0x1fcd, 0x1fdd, 0x1fcf, 0x1fdf, 0x1f70, 0x1fb6, 0x1fb3,
	0x1fb4, 0x1fb2, 0x1fb7, 0x1f00, 0x1f04, 0x1f02, 0x1f06, 0x1f80,
	0x1f84, 0x1f82, 0x1f86, 0x1f01, 0x1f05, 0x1f03, 0x1f07, 0x1f81,
	0x1f85, 0x1f83, 0x1f87, 0x1f72, 0x1f10, 0x1f14, 0x1f12, 0x1f11,
	0x1f15, 0x1f13, 0x1f74, 0x1fc6, 0x1fc3, 0x1fc4, 0x1fc2, 0x1fc7,
	0x1f20, 0x1f24, 0x1f22, 0x1f26, 0x1f90, 0x1f94, 0x1f92, 0x1f96,
	0x1f21, 0x1f25, 0x1f23, 0x1f27, 0x1f91, 0x1f95, 0x1f93, 0x1f97,
	0x1f76, 0x1fd6, 0x0390, 0x1fd2, 0x1f30, 0x1f34, 0x1f32, 0x1f36,
	0x1f31, 0x1f35, 0x1f33, 0x1f37, 0x1f78, 0x1f40, 0x1f44, 0x1f42,
	0x1f41, 0x1f45, 0x1f43, 0x1fe5, 0x1fe4, 0x1f7a, 0x1fe6, 0x03b0,
	0x1fe2, 0x1f50, 0x1f54, 0x1f52, 0x1f56, 0x1f51, 0x1f55, 0x1f53,
	0x1f57, 0x1f7c, 0x1ff6, 0x1ff3, 0x1ff4, 0x1ff2, 0x1ff7, 0x1f60,
	0x1f64, 0x1f62, 0x1f66, 0x1fa0, 0x1fa4, 0x1fa2, 0x1fa6, 0x1f61,
	0x1f65, 0x1f63, 0x1f67, 0x1fa1, 0x1fa5, 0x1fa3, 0x1fa7, 0x03da,
	0x03dc, 0x03de, 0x03e0
};

/* WP6 hebrew (charset 9) */
static const unsigned hebrewWP6[] =
{
	0x05d0, 0x05d1, 0x05d2, 0x05d3, 0x05d4, 0x05d5, 0x05d6, 0x05d7,
	0x05d8, 0x05d9, 0x05da, 0x05db, 0x05dc, 0x05dd, 0x05de, 0x05df,
	0x05e0, 0x05e1, 0x05e2, 0x05e3, 0x05e4, 0x05e5, 0x05e6, 0x05e7,
	0x05e8, 0x05e9, 0x05ea, 0x05be, 0x05c0, 0x05c3, 0x05f3, 0x05f4,
	0x05b0, 0x05b1, 0x05b2, 0x05b3, 0x05b4, 0x05b5, 0x05b6, 0x05b7,
	0x05b8, 0x05b9, 0x05b9, 0x05bb, 0x05bc, 0x05bd, 0x05bf, 0x05b7,
	0xfb1e, 0x05f0, 0x05f1, 0x05f2, 0xfb1f, 0x0591, 0x0596, 0x0020,
	0x05a4, 0x059a, 0x059b, 0x05a3, 0x05a5, 0x05a6, 0x05a7, 0x05a2,
	0x0592, 0x0593, 0x0594, 0x0595, 0x0597, 0x0598, 0x0599, 0x05a8,
	0x05f3, 0x05f3, 0x05f4, 0x0020, 0x05a9, 0x05a0, 0x059f, 0x05ab,
	0x05ac, 0x05af, 0x05c4, 0x05aa, 0xfb30, 0xfb31, 0x05d1, 0xfb32,
	0xfb33, 0xfb34, 0xfb35, 0xfb4b, 0xfb36, 0x05d7, 0xfb38, 0xfb39,
	0xfb1d, 0xfb3b, 0xfb3a, 0x05da, 0x05da, 0x05da, 0x05da, 0x05da,
	0x05da, 0xfb3c, 0xfb3e, 0xfb40, 0x05d5, 0xfb41, 0xfb44, 0x05e4,
	0xfb46, 0xfb47, 0xfb2b, 0xfb2d, 0xfb2a, 0xfb2c, 0xfb4a, 0x05dc,
	0xfb3c, 0xfb49, 0x20aa
};

/* WP6 cyrillic (charset 10) */
static const unsigned cyrillicWP6[] =
{
	0x0410, 0x0430, 0x0411, 0x0431, 0x0412, 0x0432, 0x0413, 0x0433, // 0 - 7
	0x0414, 0x0434, 0x0415, 0x0435, 0x0401, 0x0451, 0x0416, 0x0436, // 8
	0x0417, 0x0437, 0x0418, 0x0438, 0x0419, 0x0439, 0x041a, 0x043a, // 16
	0x041b, 0x043b, 0x041c, 0x043c, 0x041d, 0x043d, 0x041e, 0x043e, // 24
	0x041f, 0x043f, 0x0420, 0x0440, 0x0421, 0x0441, 0x0422, 0x0442, // 32
	0x0423, 0x0443, 0x0424, 0x0444, 0x0425, 0x0445, 0x0426, 0x0446, // 40
	0x0427, 0x0447, 0x0428, 0x0448, 0x0429, 0x0449, 0x042a, 0x044a, // 48
	0x042b, 0x044b, 0x042c, 0x044c, 0x042d, 0x044d, 0x042e, 0x044e, // 56
	0x042f, 0x044f, 0x04d8, 0x04d9, 0x0403, 0x0453, 0x0490, 0x0491, // 64
	0x0492, 0x0493, 0x0402, 0x0452, 0x0404, 0x0454, 0x0404, 0x0454, // 72
	0x0496, 0x0497, 0x0405, 0x0455, 0x0020, 0x0020, 0x0418, 0x0438, // 80
	0x0406, 0x0456, 0x0407, 0x0457, 0x0020, 0x0020, 0x0408, 0x0458, // 88
	0x040c, 0x045c, 0x049a, 0x049b, 0x04c3, 0x04c4, 0x049c, 0x049d, // 96
	0x0409, 0x0459, 0x04a2, 0x04a3, 0x040a, 0x045a, 0x047a, 0x047b, // 104 - 111
	0x0460, 0x0461, 0x040b, 0x045b, 0x040e, 0x045e, 0x04ee, 0x04ef, // 112
	0x04ae, 0x04af, 0x04b0, 0x04b1, 0x0194, 0x0263, 0x04b2, 0x04b3, // 120
	0x0425, 0x0445, 0x04ba, 0x04bb, 0x047e, 0x047f, 0x040f, 0x045f, // 128
	0x04b6, 0x04b7, 0x04b8, 0x04b9, 0x0428, 0x0448, 0x0462, 0x0463, // 136
	0x0466, 0x0467, 0x046a, 0x046b, 0x046e, 0x046f, 0x0470, 0x0471, // 144
	0x0472, 0x0473, 0x0474, 0x0475, 0x0410, 0x0430, 0x0415, 0x0435, // 152
	0x0404, 0x0454, 0x0418, 0x0438, 0x0406, 0x0456, 0x0407, 0x0457, // 160
	0x041e, 0x043e, 0x0423, 0x0443, 0x042b, 0x044b, 0x042d, 0x044d, // 168
	0x042e, 0x044e, 0x042f, 0x044f, 0x0410, 0x0430, 0x0400, 0x0450, // 176
	0x0401, 0x0451, 0x040d, 0x045d, 0x041e, 0x043e, 0x0423, 0x0443, // 184
	0x042b, 0x044b, 0x042d, 0x044d, 0x042e, 0x044e, 0x042f, 0x044f, // 192
	0x0301, 0x0300, 0x0308, 0x0306, 0x0326, 0x0328, 0x0304, 0x0020, // 200
	0x201e, 0x201c, 0x10d0, 0x10d1, 0x10d2, 0x10d3, 0x10d4, 0x10d5, // 208
	0x10d6, 0x10f1, 0x10d7, 0x10d8, 0x10d9, 0x10da, 0x10db, 0x10dc, // 216
	0x10f2, 0x10dd, 0x10de, 0x10df, 0x10e0, 0x10e1, 0x10e2, 0x10e3, // 224
	0x10f3, 0x10e4, 0x10e5, 0x10e6, 0x10e7, 0x10e8, 0x10e9, 0x10ea, // 232
	0x10eb, 0x10ec, 0x10ed, 0x10ee, 0x10f4, 0x10ef, 0x10f0, 0x10f5, // 240 - 247
	0x10f6, 0x10e3
};

/* WP6 japanese (charset 11) */
static const unsigned japaneseWP6[] =
{
	0xff61, 0xff62, 0xff63, 0xff64, 0xff65, 0xff66, 0xff67, 0xff68,
	0xff69, 0xff6a, 0xff6b, 0xff6c, 0xff6d, 0xff6e, 0xff6f, 0xff70,
	0xff71, 0xff72, 0xff73, 0xff74, 0xff75, 0xff76, 0xff77, 0xff78,
	0xff79, 0xff7a, 0xff7b, 0xff7c, 0xff7d, 0xff7e, 0xff7f, 0xff80,
	0xff81, 0xff82, 0xff83, 0xff84, 0xff85, 0xff86, 0xff87, 0xff88,
	0xff89, 0xff8a, 0xff8b, 0xff8c, 0xff8d, 0xff8e, 0xff8f, 0xff90,
	0xff91, 0xff92, 0xff93, 0xff94, 0xff95, 0xff96, 0xff97, 0xff98,
	0xff99, 0xff9a, 0xff9b, 0xff9c, 0xff9d, 0xff9e, 0xff9f
};

/* WP6 Tibetan (charset 12)
 *
 * WP6TibetanMap.h is generated automatically, so it doesn't make sense
 * to copy it in here every time it's regenerated.
 */

/* WP arabic (charset 13) */
static const unsigned arabicWP6[] =
{
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0020, 0x064e, 0xfe77, 0x064f, 0xfe79, 0x0650, 0xfe7b,
	0x064b, 0x064c, 0x064c, 0x064d, 0x0652, 0xfe7f, 0x0651, 0xfe7d,
	0xfc60, 0xfcf2, 0xfc61, 0xfcf3, 0xfc62, 0xfcf4, 0x064b, 0xfc5e,
	0xfc5e, 0xfc5f, 0x0653, 0x0670, 0x0654, 0x0020, 0x060c, 0x061b,
	0x061f, 0x066d, 0x066a, 0x00bb, 0x00ab, 0x0029, 0x0028, 0x0661,
	0x0662, 0x0663, 0x0664, 0x0665, 0x0666, 0x0667, 0x0668, 0x0669,
	0x0660, 0x0662, 0x0627, 0xfe8e, 0x0628, 0xfe91, 0xfe92, 0xfe90,
	0x062a, 0xfe97, 0xfe98, 0xfe96, 0x062b, 0xfe9b, 0xfe9c, 0xfe9a,
	0x062c, 0xfe9f, 0xfea0, 0xfe9e, 0x062d, 0xfea3, 0xfea4, 0xfea2,
	0x062e, 0xfea7, 0xfea8, 0xfea6, 0x062f, 0xfeaa, 0x0630, 0xfeac,
	0x0631, 0xfeae, 0x0632, 0xfeaf, 0x0633, 0xfeb3, 0xfeb4, 0xfeb2,
	0x0634, 0xfeb7, 0xfeb8, 0xfeb6, 0x0635, 0xfebb, 0xfebc, 0xfeba,
	0x0636, 0xfebf, 0xfec0, 0xfebe, 0x0637, 0xfec3, 0xfec4, 0xfec2,
	0x0638, 0xfec7, 0xfec8, 0xfec6, 0x0639, 0xfecb, 0xfecc, 0xfeca,
	0x063a, 0xfecf, 0xfed0, 0xfece, 0x0641, 0xfed3, 0xfed4, 0xfed2,
	0x0642, 0xfed7, 0xfed8, 0xfed6, 0x0643, 0xfedb, 0xfedc, 0xfeda,
	0x0644, 0xfedf, 0xfee0, 0xfede, 0x0645, 0xfee3, 0xfee4, 0xfee2,
	0x0646, 0xfee7, 0xfee8, 0xfee6, 0x0647, 0xfeeb, 0xfeec, 0xfeea,
	0x0629, 0xfe94, 0x0648, 0xfeee, 0x064a, 0xfef3, 0xfef4, 0xfef2,
	0x0649, 0xfef3, 0xfef4, 0xfef0, 0x0621, 0x0623, 0xfe84, 0x0625,
	0xfe88, 0x0624, 0xfe86, 0x0626, 0xfe8b, 0xfe8c, 0xfe8a, 0xfd3d,
	0xfd3c, 0x0622, 0xfe82, 0x0671, 0xfb51, 0xfefb, 0xfefc, 0xfef7,
	0xfef8, 0xfef9, 0xfefa, 0x0020, 0xfefc, 0xfef5, 0xfef6, 0x0020,
	0x0020, 0xfdf2, 0x0640, 0x0640
};

/* WP arabic script (charset 14) */
static const unsigned arabicScriptWP6[] =
{
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0615,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0615, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x06d4, 0x0020, 0x0020, 0x00b0, 0x0020, 0x065a, 0x0020,
	0x065a, 0x0659, 0x0020, 0x0020, 0x0654, 0x064c, 0xfc5e, 0x065a,
	0x065a, 0x06f4, 0x06f4, 0x06f5, 0x06f6, 0x06f6, 0x06f7, 0x06f8,
	0x067b, 0xfb54, 0xfb55, 0xfb53, 0x0680, 0xfb5c, 0xfb5d, 0xfb5b,
	0x067e, 0xfb58, 0xfb59, 0xfb57, 0x0679, 0xfb68, 0xfb69, 0xfb67,
	0x067c, 0x067c, 0x067c, 0x067c, 0x067f, 0xfb64, 0xfb65, 0xfb63,
	0x067d, 0x067d, 0x067d, 0x067d, 0x067a, 0xfb60, 0xfb61, 0xfb5f,
	0x0684, 0xfb74, 0xfb75, 0xfb73, 0x0683, 0xfb78, 0xfb79, 0xfb77,
	0x0686, 0xfb7c, 0xfb7d, 0xfb7b, 0x0687, 0xfb80, 0xfb81, 0xfb7f,
	0x0685, 0x0685, 0x0685, 0x0685, 0x0681, 0x0681, 0x0681, 0x0681,
	0x0688, 0xfb89, 0x0689, 0x0689, 0x068c, 0xfb85, 0x068e, 0xfb87,
	0x068a, 0x068a, 0x068d, 0xfb83, 0x0693, 0x0693, 0x0691, 0xfb8d,
	0x0699, 0x0699, 0x0695, 0x0695, 0x0692, 0x0692, 0x0698, 0xfb8b,
	0x0696, 0x0696, 0x075b, 0x075b, 0x069a, 0x069a, 0x069a, 0x069a,
	0x06a0, 0x06a0, 0x06a0, 0x06a0, 0x06a4, 0xfb6c, 0xfb6d, 0xfb6b,
	0x06a6, 0xfb70, 0xfb71, 0xfb6f, 0x06a9, 0xfb90, 0xfb91, 0xfb8f,
	0x0643, 0xfedb, 0xfedc, 0xfeda, 0x06aa, 0x06aa, 0x06aa, 0x06aa,
	0x06af, 0xfb94, 0xfb95, 0xfb93, 0x06af, 0xfb94, 0xfb95, 0xfb93,
	0x06ab, 0x06ab, 0x06ab, 0x06ab, 0x06b1, 0xfb9c, 0xfb9d, 0xfb9b,
	0x06b3, 0xfb98, 0xfb99, 0xfb97, 0x06b5, 0x06b5, 0x06b5, 0x06b5,
	0x0020, 0x0020, 0x06ba, 0x0020, 0x0020, 0xfb9f, 0x06bc, 0x06bc,
	0x06bc, 0x06bc, 0x06bb, 0xfba2, 0xfba3, 0xfba1, 0x06c6, 0xfbda,
	0x0020, 0x0020, 0x06ca, 0x06ca, 0x06c1, 0xfba8, 0xfba9, 0xfba7,
	0x06ce, 0x06ce, 0x06ce, 0x06ce, 0x06d2, 0xfbaf, 0x06d1, 0x06d1,
	0x06d1, 0x06d1, 0x06c0, 0xfba5
};

#include "WP6TibetanMap.h"
#include "WP6FileStructure.h"

int extendedCharacterWP6ToUCS4(unsigned char character,
                               unsigned char characterSet, const unsigned **chars)
{
	int i;
	int retVal = 0;

	if (characterSet == 0)
	{
		// if characterset == 0, we have ascii. note that this is different from the doc. body
		// this is not documented in the file format specifications
		if (character >= 0x20 && character < 0x7F)
			*chars = &asciiMap[character - 0x20];
		else
			*chars = &asciiMap[0x00];
		return 1;
	}

	switch (characterSet)
	{
	case WP6_MULTINATIONAL_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, multinationalWP6, WPD_NUM_ELEMENTS(multinationalWP6))))
			return retVal;
		if ((retVal = findComplexMap(character, chars, multinationalWP6Complex)))
			return retVal;
		break;

	case WP6_PHONETIC_SYMBOL_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, phoneticWP6, WPD_NUM_ELEMENTS(phoneticWP6))))
			return retVal;
		break;

	case WP6_BOX_DRAWING_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, boxdrawingWP6, WPD_NUM_ELEMENTS(boxdrawingWP6))))
			return retVal;
		break;

	case WP6_TYPOGRAPHIC_SYMBOL_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, typographicWP6, WPD_NUM_ELEMENTS(typographicWP6))))
			return retVal;
		break;

	case WP6_ICONIC_SYMBOL_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, iconicWP6, WPD_NUM_ELEMENTS(iconicWP6))))
			return retVal;
		break;

	case WP6_MATH_SCIENTIFIC_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, mathWP6, WPD_NUM_ELEMENTS(mathWP6))))
			return retVal;
		break;

	case WP6_MATH_SCIENTIFIC_EXTENDED_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, mathextWP6, WPD_NUM_ELEMENTS(mathextWP6))))
			return retVal;
		break;

	case WP6_GREEK_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, greekWP6, WPD_NUM_ELEMENTS(greekWP6))))
			return retVal;
		break;

	case WP6_HEBREW_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, hebrewWP6, WPD_NUM_ELEMENTS(hebrewWP6))))
			return retVal;
		break;

	case WP6_CYRILLIC_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, cyrillicWP6, WPD_NUM_ELEMENTS(cyrillicWP6))))
			return retVal;
		break;

	case WP6_JAPANESE_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, japaneseWP6, WPD_NUM_ELEMENTS(japaneseWP6))))
			return retVal;
		break;

	case WP6_ARABIC_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, arabicWP6, WPD_NUM_ELEMENTS(arabicWP6))))
			return retVal;
		break;

	case WP6_ARABIC_SCRIPT_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, arabicScriptWP6, WPD_NUM_ELEMENTS(arabicScriptWP6))))
			return retVal;
		break;

	case WP6_TIBETAN_CHARACTER_SET:
		if (tibetanMap1[character])
		{
			for (i = 0; tibetanMap1[character][i]; i++)
				;
			*chars = tibetanMap1[character];
			return i;
		}
		break;
	default:
		break;
	}

	// last resort: return whitespace
	*chars = &asciiMap[0x00];
	return 1;
}

/* WP5 International 1 (charset 1) */
/* is identical to WP6 Multinational charset */

/* WP5 International 2 (charset 2) */
static const unsigned international2WP5[] =
{
	0x0323, 0x0324, 0x02da, 0x0325, 0x02bc, 0x032d, 0x2017, 0x005f,
	0x0138, 0x032e, 0x033e, 0x2018, 0x0020, 0x02bd, 0x02db, 0x0327,
	0x0321, 0x0322, 0x030d, 0x2019, 0x0329, 0x0020, 0x0621, 0x02be,
	0x0306, 0x0310, 0x2032, 0x2034
};

/* WP5 Box drawing (charset 3) */
/* identical to WP6 Box drawing symbol charset */


/* WP5 Typographic symbols (charset 4) */
/* identical to WP6 Typographic symbols charset */

/* WP5 Iconic symbols (charset 5) */
static const unsigned iconicWP5[] =
{
	0x2665, 0x2666, 0x2663, 0x2660, 0x2642, 0x2640, 0x263c, 0x263a,
	0x263b, 0x266a, 0x266c, 0x25ac, 0x2302, 0x203c, 0x221a, 0x21a8,
	0x2310, 0x2319, 0x25d8, 0x25d9, 0x21b5, 0x261e, 0x261c, 0x2713,
	0x2610, 0x2612, 0x2639, 0x266f, 0x266d, 0x266e, 0x260e, 0x231a,
	0x231b, 0x2104, 0x23b5
};

/* WP5 math/scientific (charset 6) */
/* is identical to the WP6 math/scientific charset */

/* WP5 math/scientific extended (charset 7) */
/* is identical to the WP6 math/scientific extended charset */

/* WP5 greek (charset 8) */
static const unsigned greekWP5[] =
{
	0x0391, 0x03b1, 0x0392, 0x03b2, 0x0392, 0x03d0, 0x0393, 0x03b3,
	0x0394, 0x03b4, 0x0395, 0x03b5, 0x0396, 0x03b6, 0x0397, 0x03b7,
	0x0398, 0x03b8, 0x0399, 0x03b9, 0x039a, 0x03ba, 0x039b, 0x03bb,
	0x039c, 0x03bc, 0x039d, 0x03bd, 0x039e, 0x03be, 0x039f, 0x03bf,
	0x03a0, 0x03c0, 0x03a1, 0x03c1, 0x03a3, 0x03c3, 0x03f9, 0x03db,
	0x03a4, 0x03c4, 0x03a5, 0x03c5, 0x03a6, 0x03d5, 0x03a7, 0x03c7,
	0x03a8, 0x03c8, 0x03a9, 0x03c9, 0x03ac, 0x03ad, 0x03ae, 0x03af,
	0x03ca, 0x03cc, 0x03cd, 0x03cb, 0x03ce, 0x03b5, 0x03d1, 0x03f0,
	0x03d6, 0x1fe5, 0x03d2, 0x03c6, 0x03c9, 0x037e, 0x0387, 0x0384,
	0x00a8, 0x0385, 0x1fed, 0x1fef, 0x1fc0, 0x1fbd, 0x1fbf, 0x1fbe,
	0x1fce, 0x1fde, 0x1fcd, 0x1fdd, 0x1fcf, 0x1fdf, 0x0384, 0x1fef,
	0x1fc0, 0x1fbd, 0x1fbf, 0x1fce, 0x1fde, 0x1fcd, 0x1fdd, 0x1fcf,
	0x1fdf, 0x1f70, 0x1fb6, 0x1fb3, 0x1fb4, 0x1fb7, 0x1f00, 0x1f04,
	0x1f02, 0x1f06, 0x1f80, 0x1f84, 0x1f86, 0x1f01, 0x1f05, 0x1f03,
	0x1f07, 0x1f81, 0x1f85, 0x1f87, 0x1f72, 0x1f10, 0x1f14, 0x1f13,
	0x1f11, 0x1f15, 0x1f13, 0x1f74, 0x1fc6, 0x1fc3, 0x1fc4, 0x1fc2,
	0x1fc7, 0x1f20, 0x1f24, 0x1f22, 0x1f26, 0x1f90, 0x1f94, 0x1f96,
	0x1f21, 0x1f25, 0x1f23, 0x1f27, 0x1f91, 0x1f95, 0x1f97, 0x1f76,
	0x1fd6, 0x0390, 0x1fd2, 0x1f30, 0x1f34, 0x1f32, 0x1f36, 0x1f31,
	0x1f35, 0x1f33, 0x1f37, 0x1f78, 0x1f40, 0x1f44, 0x1f42, 0x1f41,
	0x1f45, 0x1f43, 0x1f7a, 0x1fe6, 0x03b0, 0x1fe3, 0x1f50, 0x1f54,
	0x1f52, 0x1f56, 0x1f51, 0x1f55, 0x1f53, 0x1f57, 0x1f7c, 0x1ff6,
	0x1ff3, 0x1ff4, 0x1ff2, 0x1ff7, 0x1f60, 0x1f64, 0x1f62, 0x1f66,
	0x1fa0, 0x1fa4, 0x1fa6, 0x1f61, 0x1f65, 0x1f63, 0x1f67, 0x1fa1,
	0x1fa5, 0x1fa7, 0x0374, 0x0375, 0x03db, 0x03dd, 0x03d9, 0x03e1,
	0x0386, 0x0388, 0x0389, 0x038a, 0x038c, 0x038e, 0x038f, 0x03aa,
	0x03ab, 0x1fe5
};

/* WP5 Hebrew (charset 9) */
static const unsigned hebrewWP5[] =
{
	0x05d0, 0x05d1, 0x05d2, 0x05d3, 0x05d4, 0x05d5, 0x05d6, 0x05d7,
	0x05d8, 0x05d9, 0x05da, 0x05db, 0x05dc, 0x05dd, 0x05de, 0x05df,
	0x05e0, 0x05e1, 0x05e2, 0x05e3, 0x05e4, 0x05e5, 0x05e6, 0x05e7,
	0x05e8, 0x05e9, 0x05ea, 0x05be, 0x05c0, 0x05c3, 0x05f3, 0x05f4,
	0x05b0, 0x05b1, 0x05b2, 0x05b3, 0x05b4, 0x05b5, 0x05b6, 0x05b7,
	0x05b8, 0x05b9, 0x05ba, 0x05bb, 0x05bc, 0x05bd, 0x05bf, 0x05b7,
	0xfbe1, 0x05f0, 0x05f1, 0x05f2, 0x0591, 0x0596, 0x05ad, 0x05a4,
	0x059a, 0x059b, 0x05a3, 0x05a5, 0x05a6, 0x05a7, 0x09aa, 0x0592,
	0x0593, 0x0594, 0x0595, 0x0597, 0x0598, 0x0599, 0x05a8, 0x059c,
	0x059d, 0x059e, 0x05a1, 0x05a9, 0x05a0, 0x059f, 0x05ab, 0x05ac,
	0x05af, 0x05c4, 0x0544, 0x05d0, 0xfb31, 0xfb32, 0xfb33, 0xfb34,
	0xfb35, 0xfb4b, 0xfb36, 0x05d7, 0xfb38, 0xfb39, 0xfb3b, 0xfb3a,
	0x05da, 0x05da, 0x05da, 0x05da, 0x05da, 0x05da, 0xfb3c, 0xfb3e,
	0xfb40, 0x05df, 0xfb41, 0xfb44, 0xfb46, 0xfb47, 0xfb2b, 0xfb2d,
	0xfb2a, 0xfb2c, 0xfb4a, 0xfb4c, 0xfb4e, 0xfb1f, 0xfb1d
};

/* WP5 cyrillic (charset 10) */
static const unsigned cyrillicWP5[] =
{
	0x0410, 0x0430, 0x0411, 0x0431, 0x0412, 0x0432, 0x0413, 0x0433, // 0 - 7
	0x0414, 0x0434, 0x0415, 0x0435, 0x0401, 0x0451, 0x0416, 0x0436, // 8 - 15
	0x0417, 0x0437, 0x0418, 0x0438, 0x0419, 0x0439, 0x041a, 0x043a, // 16 - 23
	0x041b, 0x043b, 0x041c, 0x043c, 0x041d, 0x043d, 0x041e, 0x043e, // 24 - 31
	0x041f, 0x043f, 0x0420, 0x0440, 0x0421, 0x0441, 0x0422, 0x0442, // 32 - 39
	0x0423, 0x0443, 0x0424, 0x0444, 0x0425, 0x0445, 0x0426, 0x0446, // 40 - 47
	0x0427, 0x0447, 0x0428, 0x0448, 0x0429, 0x0449, 0x042a, 0x044a, // 48 - 55
	0x042b, 0x044b, 0x042c, 0x044c, 0x042d, 0x044d, 0x042e, 0x044e, // 56 - 63
	0x042f, 0x044f, 0x0490, 0x0491, 0x0402, 0x0452, 0x0403, 0x0453, // 64 - 71
	0x0404, 0x0454, 0x0405, 0x0455, 0x0406, 0x0456, 0x0407, 0x0457, // 72 - 79
	0x0408, 0x0458, 0x0409, 0x0459, 0x040a, 0x045a, 0x040b, 0x045b, // 80 - 87
	0x040c, 0x045c, 0x040e, 0x045e, 0x040f, 0x045f, 0x0462, 0x0463, // 88 - 95
	0x0472, 0x0473, 0x0474, 0x0475, 0x046a, 0x046b, 0xa640, 0xa641, // 96 - 103
	0x0429, 0x0449, 0x04c0, 0x04cf, 0x0466, 0x0467, 0x0000, 0x0000, // 104 - 111
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 112 - 119
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 120 - 127
	0x0000, 0x0000, 0x0400, 0x0450, 0x0000, 0x0000, 0x040d, 0x045d, // 128 - 135
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 136 - 143
	0x0000, 0x0000, 0x0000, 0x0000, 0x0301, 0x0300                  // 144 - 149
};

/* WP5 cyrillic (charset 10) - multicharacter mapping */
static const WPXComplexMap cyrillicWP5Complex[] =
{
	{ 110, { 0x0410, 0x0301, 0x0000 } }, { 111, { 0x0430, 0x0301, 0x0000 } },
	{ 112, { 0x0415, 0x0301, 0x0000 } }, { 113, { 0x0435, 0x0301, 0x0000 } },
	{ 114, { 0x0418, 0x0301, 0x0000 } }, { 115, { 0x0458, 0x0301, 0x0000 } },
	{ 116, { 0x041e, 0x0301, 0x0000 } }, { 117, { 0x043e, 0x0301, 0x0000 } },
	{ 118, { 0x0423, 0x0301, 0x0000 } }, { 119, { 0x0443, 0x0301, 0x0000 } },
	{ 120, { 0x042b, 0x0301, 0x0000 } }, { 121, { 0x044b, 0x0301, 0x0000 } },
	{ 122, { 0x042d, 0x0301, 0x0000 } }, { 123, { 0x044d, 0x0301, 0x0000 } },
	{ 124, { 0x042e, 0x0301, 0x0000 } }, { 125, { 0x044e, 0x0301, 0x0000 } },
	{ 126, { 0x042f, 0x0301, 0x0000 } }, { 127, { 0x044f, 0x0301, 0x0000 } },
	{ 128, { 0x0410, 0x0300, 0x0000 } }, { 129, { 0x0430, 0x0300, 0x0000 } },

	{ 132, { 0x0401, 0x0300, 0x0000 } }, { 133, { 0x0451, 0x0300, 0x0000 } },

	{ 136, { 0x041e, 0x0300, 0x0000 } }, { 137, { 0x043e, 0x0300, 0x0000 } },
	{ 138, { 0x0423, 0x0300, 0x0000 } }, { 139, { 0x0443, 0x0300, 0x0000 } },
	{ 140, { 0x042b, 0x0300, 0x0000 } }, { 141, { 0x044b, 0x0300, 0x0000 } },
	{ 142, { 0x042d, 0x0300, 0x0000 } }, { 143, { 0x044d, 0x0300, 0x0000 } },
	{ 144, { 0x042e, 0x0300, 0x0000 } }, { 145, { 0x044e, 0x0300, 0x0000 } },
	{ 146, { 0x042f, 0x0300, 0x0000 } }, { 147, { 0x044f, 0x0300, 0x0000 } },

	{ 0, { 0x0000 } }
};


/* WP5 Japanese (charset 11) */
static const unsigned japaneseWP5[] =
{
	0x3041, 0x3043, 0x3045, 0x3047, 0x3049, 0x3053, 0x3083, 0x3085,
	0x3087, 0x3094, 0x3095, 0x3096, 0x3042, 0x3044, 0x3046, 0x3048,
	0x304a, 0x304b, 0x304d, 0x3047, 0x3051, 0x3053, 0x304c, 0x304e,
	0x3050, 0x3052, 0x3054, 0x3055, 0x3057, 0x3059, 0x305b, 0x305d,
	0x3056, 0x3058, 0x305a, 0x305c, 0x305e, 0x305f, 0x3051, 0x3064,
	0x3066, 0x3068, 0x3060, 0x3062, 0x3065, 0x3067, 0x3069, 0x306a,
	0x306b, 0x306c, 0x306d, 0x306e, 0x306f, 0x3072, 0x3075, 0x3078,
	0x307b, 0x3070, 0x3073, 0x3076, 0x3079, 0x307c, 0x3071, 0x3074,
	0x3077, 0x307a, 0x307d, 0x307e, 0x307f, 0x3080, 0x3081, 0x3082,
	0x3084, 0x3086, 0x3088, 0x3089, 0x308a, 0x308b, 0x308c, 0x308d,
	0x308e, 0x3092, 0x3093, 0x3014, 0x3015, 0xff3b, 0xff3d, 0x300c,
	0x300d, 0x300c, 0x300d, 0x302a, 0x3002, 0x3001, 0x309d, 0x309e,
	0x3003, 0x30fc, 0x309b, 0x309c, 0x30a1, 0x30a3, 0x30a5, 0x30a7,
	0x30a9, 0x30c3, 0x30e3, 0x30e5, 0x3057, 0x30f4, 0x30f5, 0x30f6,
	0x30a2, 0x30a4, 0x30a6, 0x30a8, 0x30aa, 0x30ab, 0x30ad, 0x30af,
	0x30b1, 0x30b3, 0x30ac, 0x30ae, 0x30b0, 0x30b2, 0x30b4, 0x30b5,
	0x30c4, 0x30b9, 0x30bb, 0x30bd, 0x30b6, 0x30b8, 0x30ba, 0x30bc,
	0x30be, 0x30bf, 0x30c1, 0x30c4, 0x30c6, 0x30c8, 0x30c0, 0x30c2,
	0x30c5, 0x30c7, 0x30c9, 0x30ca, 0x30cb, 0x30cc, 0x30cd, 0x30ce,
	0x30cf, 0x30d2, 0x30d5, 0x30d8, 0x03d0, 0x30db, 0x30d3, 0x30d6,
	0x30d9, 0x30dc, 0x30d1, 0x30d4, 0x30d7, 0x30da, 0x30dd, 0x30de,
	0x30df, 0x30e0, 0x30e1, 0x30e2, 0x30e4, 0x30e6, 0x30e8, 0x30e9,
	0x30ea, 0x30ab, 0x30ec, 0x30ed, 0x30ef, 0x30f2, 0x30f3, 0x30fd,
	0x30fe
};

/* WP5 arabic (charset 13) */
static const unsigned arabicWP5[] =
{
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, // 0 - 7
	0x0020, 0x0020, 0x064e, 0xfe77, 0x064f, 0xfe79, 0x0650, 0xfe7b, // 8 - 15
	0x064b, 0x064c, 0x064c, 0x064d, 0x0652, 0xfe7f, 0x0651, 0xfe7d, // 16 - 23
	0xfc60, 0x0000, 0xfc61, 0x0000, 0xfc62, 0x0000, 0x0000, 0xfc5e, // 24 - 31
	0xfc5e, 0xfc5f, 0x0653, 0x0000, 0x0654, 0x060c, 0x061b, 0x061f, // 32 - 39
	0x066d, 0x0661, 0x0662, 0x0663, 0x0664, 0x0665, 0x0666, 0x0667, // 40 - 47
	0x0668, 0x0669, 0x0660, 0x06f2, 0x0627, 0xfe8e, 0x0628, 0xfe91, // 48 - 55
	0xfe92, 0xfe90, 0x062a, 0xfe97, 0xfe98, 0xfe96, 0x062b, 0xfe9b, // 56 - 63
	0xfe9c, 0xfe9a, 0x062c, 0xfe9f, 0xfea0, 0xfe9e, 0x062d, 0xfea3, // 64 - 71
	0xfea4, 0xfea2, 0x062e, 0xfea7, 0xfea8, 0xfea6, 0x062f, 0xfeaa, // 72 - 79
	0x0630, 0xfeac, 0x0631, 0xfeae, 0x0632, 0xfeb0, 0x0633, 0xfeb3, // 80 - 87
	0xfeb4, 0xfeb2, 0x0634, 0xfeb7, 0xfeb8, 0xfeb6, 0x0635, 0xfebb, // 88 - 95
	0xfebc, 0xfeba, 0x0636, 0xfebf, 0xfec0, 0xfebe, 0x0637, 0xfec3, // 96 - 103
	0xfec4, 0xfec2, 0x0638, 0xfec7, 0xfec8, 0xfec6, 0x0639, 0xfecb, // 104 - 111
	0xfecc, 0xfeca, 0x063a, 0xfecf, 0xfed0, 0xfece, 0x0641, 0xfed3, // 112 - 119
	0xfed4, 0xfed2, 0x0642, 0xfed6, 0xfed8, 0xfed6, 0x0643, 0xfedb, // 120 - 127
	0xfedc, 0xfeda, 0x0644, 0xfedf, 0xfee0, 0xfede, 0x0645, 0xfee3, // 128 - 135
	0xfee4, 0xfee2, 0x0646, 0xfee7, 0xfee8, 0xfee6, 0x0647, 0xfeeb, // 136 - 143
	0xfeec, 0xfeea, 0x0629, 0xfe94, 0x0648, 0xfeee, 0x064a, 0xfef3, // 144 - 151
	0xfef4, 0xfef2, 0x0649, 0xfef0, 0x0621, 0x0623, 0xfe84, 0x0625, // 152 - 159
	0xfe88, 0x0624, 0xfe86, 0x0626, 0xfe8b, 0xfe8c, 0xfe8a, 0xfd3d, // 160 - 167
	0xfd3c, 0x0622, 0xfe82, 0x02c6, 0xfefb, 0xfefc, 0xfef7, 0xfef8, // 168 - 175
	0xfef9, 0xfefa, 0x0000, 0x0000, 0xfef5, 0xfef6, 0x0000, 0x0000, // 176 - 183
	0xfdf2, 0x0640, 0x0640, 0x0671, 0xfb51, 0x066a, 0x00bb, 0x00ab, // 184 - 191
	0xfd3f, 0xfd3e                                                  // 192 - 193
};

/* WP5 arabic (charset 13) - multicharacter mapping */
static const WPXComplexMap arabicWP5Complex[] =
{
	{ 25, { 0x0640, 0xfc60, 0x0000 } },

	{ 27, { 0x0640, 0xfc61, 0x0000 } },

	{ 29, { 0x0640, 0xfc62, 0x0000 } },
	{ 30, { 0x064b, 0x0651, 0x0000 } },

	{ 35, { 0x0640, 0x0670, 0x0000 } },

	{ 178, { 0xfefb, 0x064b, 0x0000 } },
	{ 179, { 0xfefc, 0x064b, 0x0000 } },

	{ 182, { 0xfefb, 0x02c6, 0x0000 } },
	{ 183, { 0xfefc, 0x02c6, 0x0000 } },

	{ 0, { 0x0000 } }
};


/* WP5 arabic script (charset 14) */
static const unsigned arabicScriptWP5[] =
{
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0615,
	0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0615, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020,
	0x0020, 0x06d4, 0x0020, 0x0020, 0x00b0, 0x0020, 0x065a, 0x0020,
	0x065a, 0x0659, 0x0020, 0x0020, 0x0654, 0x064c, 0xfc5e, 0x065a,
	0x065a, 0x06f4, 0x06f4, 0x06f5, 0x06f6, 0x06f6, 0x06f7, 0x06f8,
	0x067b, 0xfb54, 0xfb55, 0xfb53, 0x0680, 0xfb5c, 0xfb5d, 0xfb5b,
	0x067e, 0xfb58, 0xfb59, 0xfb57, 0x0679, 0xfb68, 0xfb69, 0xfb67,
	0x067c, 0x067c, 0x067c, 0x067c, 0x067f, 0xfb64, 0xfb65, 0xfb63,
	0x067d, 0x067d, 0x067d, 0x067d, 0x067a, 0xfb60, 0xfb61, 0xfb5f,
	0x0684, 0xfb74, 0xfb75, 0xfb73, 0x0683, 0xfb78, 0xfb79, 0xfb77,
	0x0686, 0xfb7c, 0xfb7d, 0xfb7b, 0x0687, 0xfb80, 0xfb81, 0xfb7f,
	0x0685, 0x0685, 0x0685, 0x0685, 0x0681, 0x0681, 0x0681, 0x0681,
	0x0688, 0xfb89, 0x0689, 0x0689, 0x068c, 0xfb85, 0x068e, 0xfb87,
	0x068a, 0x068a, 0x068d, 0xfb83, 0x0693, 0x0693, 0x0691, 0xfb8d,
	0x0699, 0x0699, 0x0695, 0x0695, 0x0692, 0x0692, 0x0698, 0xfb8b,
	0x0696, 0x0696, 0x075b, 0x075b, 0x069a, 0x069a, 0x069a, 0x069a,
	0x06a0, 0x06a0, 0x06a0, 0x06a0, 0x06a4, 0xfb6c, 0xfb6d, 0xfb6b,
	0x06a6, 0xfb70, 0xfb71, 0xfb6f, 0x06a9, 0xfb90, 0xfb91, 0xfb8f,
	0x0643, 0xfedb, 0xfedc, 0xfeda, 0x06aa, 0x06aa, 0x06aa, 0x06aa,
	0x06af, 0xfb94, 0xfb95, 0xfb93, 0x06af, 0xfb94, 0xfb95, 0xfb93,
	0x06ab, 0x06ab, 0x06ab, 0x06ab, 0x06b1, 0xfb9c, 0xfb9d, 0xfb9b,
	0x06b3, 0xfb98, 0xfb99, 0xfb97, 0x06b5, 0x06b5, 0x06b5, 0x06b5,
	0x0020, 0x0020, 0x06ba, 0x0020, 0x0020, 0xfb9f, 0x06bc, 0x06bc,
	0x06bc, 0x06bc, 0x06bb, 0xfba2, 0xfba3, 0xfba1, 0x06c6, 0xfbda,
	0x0020, 0x0020, 0x06ca, 0x06ca, 0x06c1, 0xfba8, 0xfba9, 0xfba7,
	0x06ce, 0x06ce, 0x06ce, 0x06ce, 0x06d2, 0xfbaf, 0x06d1, 0x06d1,
	0x06d1, 0x06d1, 0x06c0, 0xfba5
};

#include "WP5FileStructure.h"

int extendedCharacterWP5ToUCS4(unsigned char character,
                               unsigned char characterSet, const unsigned **chars)
{
	int retVal = 0;

	if (characterSet == 0)
	{
		// if characterset == 0, we have ascii. note that this is different from the doc. body
		// this is not documented in the file format specifications
		if (character >= 0x20 && character < 0x7F)
			*chars = &asciiMap[character - 0x20];
		else
			*chars = &asciiMap[0x00];
		return 1;
	}

	switch (characterSet)
	{
	case WP5_INTERNATIONAL_1_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, multinationalWP6, WPD_NUM_ELEMENTS(multinationalWP6))))
			return retVal;
		if ((retVal = findComplexMap(character, chars, multinationalWP6Complex)))
			return retVal;
		break;

	case WP5_INTERNATIONAL_2_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, international2WP5, WPD_NUM_ELEMENTS(international2WP5))))
			return retVal;
		break;

	case WP5_BOX_DRAWING_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, boxdrawingWP6, WPD_NUM_ELEMENTS(boxdrawingWP6))))
			return retVal;
		break;

	case WP5_TYPOGRAPHIC_SYMBOL_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, typographicWP6, WPD_NUM_ELEMENTS(typographicWP6))))
			return retVal;
		break;

	case WP5_ICONIC_SYMBOL_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, iconicWP5, WPD_NUM_ELEMENTS(iconicWP5))))
			return retVal;
		break;

	case WP5_MATH_SCIENTIFIC_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, mathWP6, WPD_NUM_ELEMENTS(mathWP6))))
			return retVal;
		break;

	case WP5_MATH_SCIENTIFIC_EXTENDED_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, mathextWP6, WPD_NUM_ELEMENTS(mathextWP6))))
			return retVal;
		break;

	case WP5_GREEK_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, greekWP5, WPD_NUM_ELEMENTS(greekWP5))))
			return retVal;
		break;

	case WP5_HEBREW_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, hebrewWP5, WPD_NUM_ELEMENTS(hebrewWP5))))
			return retVal;
		break;

	case WP5_CYRILLIC_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, cyrillicWP5, WPD_NUM_ELEMENTS(cyrillicWP5))))
			return retVal;
		if ((retVal = findComplexMap(character, chars, cyrillicWP5Complex)))
			return retVal;
		break;

	case WP5_JAPANESE_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, japaneseWP5, WPD_NUM_ELEMENTS(japaneseWP5))))
			return retVal;
		break;

	case WP5_ARABIC_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, arabicWP5, WPD_NUM_ELEMENTS(arabicWP5))))
			return retVal;
		if ((retVal = findComplexMap(character, chars, arabicWP5Complex)))
			return retVal;
		break;

	case WP5_ARABIC_SCRIPT_CHARACTER_SET:
		if ((retVal = findSimpleMap(character, chars, arabicScriptWP5, WPD_NUM_ELEMENTS(arabicScriptWP5))))
			return retVal;
		break;
	default:
		break;
	}

	// last resort: return whitespace
	*chars = &asciiMap[0x00];
	return 1;
}

static const unsigned extendedCharactersWP42[] =
{
	/*   0 */	0x0020, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
	/*   8 */	0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266c, 0x263c,
	/*  16 */	0x25b8, 0x25c2, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
	/*  24 */	0x2191, 0x2193, 0x2192, 0x2190, 0x2319, 0x2194, 0x25b4, 0x25be,
	/*  32 */	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	/*  40 */	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	/*  48 */	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	/*  56 */	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	/*  64 */	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	/*  72 */	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	/*  80 */	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	/*  88 */	0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	/*  96 */	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	/* 104 */	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	/* 112 */	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	/* 120 */	0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2302,
	/* 128 */	0x00c7, 0x00fc, 0x00eb, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
	/* 136 */	0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
	/* 144 */	0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
	/* 152 */	0x00ff, 0x00d6, 0x00dc, 0x00a2, 0x00a3, 0x00a5, 0x20a7, 0x0192,
	/* 160 */	0x00e1, 0x00ed, 0x00f3, 0x00f4, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
	/* 168 */	0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
	/* 176 */	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	/* 184 */	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	/* 192 */	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	/* 200 */	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	/* 208 */	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	/* 216 */	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	/* 224 */	0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x03bc, 0x03c4,
	/* 232 */	0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0x22c2,
	/* 240 */	0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
	/* 248 */	0x00b0, 0x2022, 0x22c5, 0x221a, 0x207f, 0x00b2, 0x25fc, 0x0020
};

int extendedCharacterWP42ToUCS4(unsigned char character, const unsigned **chars)
{
	int retVal = 0;

	if ((retVal = findSimpleMap(character, chars, extendedCharactersWP42, WPD_NUM_ELEMENTS(extendedCharactersWP42))))
		return retVal;
	*chars = &asciiMap[0x00];
	return 1;
}

#include "WPXFileStructure.h"
#include "libwpd_math.h"

unsigned short fixedPointToWPUs(const unsigned fixedPointNumber)
{
	auto fixedPointNumberIntegerPart = (signed short)((fixedPointNumber & 0xFFFF0000) >> 16);
	auto fixedPointNumberFractionalPart = (double)((double)(fixedPointNumber & 0x0000FFFF)/(double)0xFFFF);
	auto numberWPU = (unsigned short)rint((((double)fixedPointNumberIntegerPart + fixedPointNumberFractionalPart)*50)/3);
	return numberWPU;
}

double fixedPointToDouble(const unsigned fixedPointNumber)
{
	auto fixedPointNumberIntegerPart = (signed short)((fixedPointNumber & 0xFFFF0000) >> 16);
	auto fixedPointNumberFractionalPart = (double)((double)(fixedPointNumber & 0x0000FFFF)/(double)0xFFFF);
	return ((double)fixedPointNumberIntegerPart + fixedPointNumberFractionalPart);
}

double wpuToFontPointSize(const unsigned short wpuNumber)
{
	return (double)rint((double)((((double)wpuNumber)/100.0)*2.0));
}

RGBSColor::RGBSColor(unsigned char r, unsigned char g, unsigned char b, unsigned char s)
	:	m_r(r),
	  m_g(g),
	  m_b(b),
	  m_s(s)
{
}

RGBSColor::RGBSColor()
	:	m_r(0),
	  m_g(0),
	  m_b(0),
	  m_s(0)
{
}

RGBSColor::RGBSColor(unsigned short red, unsigned short green, unsigned short blue)
	:	m_r((unsigned char)((red >> 8) & 0xFF)),
	  m_g((unsigned char)((green >> 8) & 0xFF)),
	  m_b((unsigned char)((blue >> 8) & 0xFF)),
	  m_s(100)
{
}

WPXTabStop::WPXTabStop()
	:	m_position(0.0),
	  m_alignment(LEFT),
	  m_leaderCharacter('\0'),
	  m_leaderNumSpaces(0)
{
}

WPXColumnDefinition::WPXColumnDefinition()
	:	m_width(0.0),
	  m_leftGutter(0.0),
	  m_rightGutter(0.0)
{
}

WPXColumnProperties::WPXColumnProperties()
	:	m_attributes(0x00000000),
	  m_alignment(0x00)
{
}

// HACK: this function is really cheesey
int _extractNumericValueFromRoman(const char romanChar)
{
	int retValue = 0;
	switch (romanChar)
	{
	case 'I':
	case 'i':
		retValue = 1;
		break;
	case 'V':
	case 'v':
		retValue = 5;
		break;
	case 'X':
	case 'x':
		retValue = 10;
		break;
	default:
		throw ParseException();
	}
	return retValue;
}

// _extractDisplayReferenceNumberFromBuf: given a nuWP6_DEFAULT_FONT_SIZEmber string in UCS4 represented
// as letters, numbers, or roman numerals.. return an integer value representing its number
// HACK: this function is really cheesey
// NOTE: if the input is not valid, the output is unspecified
int _extractDisplayReferenceNumberFromBuf(const librevenge::RVNGString &buf, const WPXNumberingType listType)
{
	if (listType == LOWERCASE_ROMAN || listType == UPPERCASE_ROMAN)
	{
		int currentSum = 0;
		int lastMark = 0;
		librevenge::RVNGString::Iter i(buf);
		for (i.rewind(); i.next();)
		{
			int currentMark = _extractNumericValueFromRoman(*(i()));
			if (lastMark < currentMark)
			{
				currentSum = currentMark - lastMark;
			}
			else
				currentSum+=currentMark;
			lastMark = currentMark;
		}
		return currentSum;
	}
	else if (listType == LOWERCASE || listType == UPPERCASE)
	{
		// FIXME: what happens to a lettered list that goes past z? ah
		// the sweet mysteries of life
		if (buf.len()==0)
			throw ParseException();
		char c = buf.cstr()[0];
		if (!isalpha(c))
			throw ParseException();
		if (listType==LOWERCASE)
			c = (char)toupper(c);
		return (c - 64);
	}
	else if (listType == ARABIC)
	{
		int currentSum = 0;
		const char *first = buf.cstr();
		const char *const last = first + buf.size();
		namespace qi = boost::spirit::qi;
		const bool succeed = qi::parse(first, last, qi::int_, currentSum);
		return succeed ? currentSum : 0;
	}

	return 1;
}

WPXNumberingType _extractWPXNumberingTypeFromBuf(const librevenge::RVNGString &buf, const WPXNumberingType putativeWPXNumberingType)
{
	librevenge::RVNGString::Iter i(buf);
	for (i.rewind(); i.next();)
	{
		if ((*(i()) == 'I' || *(i()) == 'V' || *(i()) == 'X') &&
		        (putativeWPXNumberingType == LOWERCASE_ROMAN || putativeWPXNumberingType == UPPERCASE_ROMAN))
			return UPPERCASE_ROMAN;
		else if ((*(i()) == 'i' || *(i()) == 'v' || *(i()) == 'x') &&
		         (putativeWPXNumberingType == LOWERCASE_ROMAN || putativeWPXNumberingType == UPPERCASE_ROMAN))
			return LOWERCASE_ROMAN;
		else if (*(i()) >= 'A' && *(i()) <= 'Z')
			return UPPERCASE;
		else if (*(i()) >= 'a' && *(i()) <= 'z')
			return LOWERCASE;
	}

	return ARABIC;
}

librevenge::RVNGString _numberingTypeToString(WPXNumberingType t)
{
	librevenge::RVNGString sListTypeSymbol("1");
	switch (t)
	{
	case ARABIC:
		sListTypeSymbol = "1";
		break;
	case LOWERCASE:
		sListTypeSymbol = "a";
		break;
	case UPPERCASE:
		sListTypeSymbol = "A";
		break;
	case LOWERCASE_ROMAN:
		sListTypeSymbol = "i";
		break;
	case UPPERCASE_ROMAN:
		sListTypeSymbol = "I";
		break;
	default:
		break;
	}

	return sListTypeSymbol;
}

/* Mapping of Apple's MacRoman character set in Unicode (UCS4)
 * used in the WordPerfect Macintosh file format */

const unsigned macRomanCharacterMap[] =
{
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x0020,
	0x00c4, 0x00c5, 0x00c7, 0x00c9, 0x00d1, 0x00d6, 0x00dc, 0x00e1,
	0x00e0, 0x00e2, 0x00e4, 0x00e3, 0x00e5, 0x00e7, 0x00e9, 0x00e8,
	0x00ea, 0x00eb, 0x00ed, 0x00ec, 0x00ee, 0x00ef, 0x00f1, 0x00f3,
	0x00f2, 0x00f4, 0x00f6, 0x00f5, 0x00fa, 0x00f9, 0x00fb, 0x00fc,
	0x2020, 0x00b0, 0x00a2, 0x00a3, 0x00a7, 0x2022, 0x00b6, 0x00df,
	0x00ae, 0x00a9, 0x2122, 0x00b4, 0x00a8, 0x2260, 0x00c6, 0x00d8,
	0x221e, 0x00b1, 0x2264, 0x2265, 0x00a5, 0x00b5, 0x2202, 0x2211,
	0x220f, 0x03c0, 0x222b, 0x00aa, 0x00ba, 0x03a9, 0x00e6, 0x00f8,
	0x00bf, 0x00a1, 0x00ac, 0x221a, 0x0192, 0x2248, 0x2206, 0x00ab,
	0x00bb, 0x2026, 0x00a0, 0x00c0, 0x00c3, 0x00d5, 0x0152, 0x0153,
	0x2013, 0x2014, 0x201c, 0x201d, 0x2018, 0x2019, 0x00f7, 0x25ca,
	0x00ff, 0x0178, 0x2044, 0x20ac, 0x2039, 0x203a, 0xfb01, 0xfb02,
	0x2021, 0x00b7, 0x201a, 0x201e, 0x2030, 0x00c2, 0x00ca, 0x00c1,
	0x00cb, 0x00c8, 0x00cd, 0x00ce, 0x00cf, 0x00cc, 0x00d3, 0x00d4,
	0xf8ff, 0x00d2, 0x00da, 0x00db, 0x00d9, 0x0131, 0x02c6, 0x02dc,
	0x00af, 0x02d8, 0x02d9, 0x02da, 0x00b8, 0x02dd, 0x02db, 0x02c7
};

librevenge::RVNGString doubleToString(const double value)
{
	librevenge::RVNGString tempString;
	if (value < 0.0001 && value > -0.0001)
		tempString.sprintf("0.0000");
	else
		tempString.sprintf("%.4f", value);
#ifndef __ANDROID__
	std::string decimalPoint(localeconv()->decimal_point);
#else
	std::string decimalPoint(".");
#endif
	if ((decimalPoint.size() == 0) || (decimalPoint == "."))
		return tempString;
	std::string stringValue(tempString.cstr());
	if (!stringValue.empty())
	{
		std::string::size_type pos;
		while ((pos = stringValue.find(decimalPoint)) != std::string::npos)
			stringValue.replace(pos,decimalPoint.size(),".");
	}
	return librevenge::RVNGString(stringValue.c_str());
}

int appleWorldScriptToUCS4(unsigned short character, const unsigned **chars)
{
	static const unsigned charSimpleMap[] =
	{
		0x3000, 0x3001, 0x3002, 0xff0c, 0xff0e, 0x30fb, 0xff1a, 0xff1b, // 0x8140 - 0x8147
		0xff1f, 0xff01, 0x309b, 0x309c, 0x00b4, 0xff40, 0x00a8, 0xff3e, // 0x8148 - 0x814f
		0xffe3, 0xff3f, 0x30fd, 0x30fe, 0x309d, 0x309e, 0x3003, 0x4edd, // 0x8150 - 0x8157
		0x3005, 0x3006, 0x3007, 0x30fc, 0x2014, 0x2010, 0xff0f, 0xff3c, // 0x8158 - 0x815f
		0x301c, 0x2016, 0xff5c, 0x2026, 0x2025, 0x2018, 0x2019, 0x201c, // 0x8160 - 0x8167
		0x201d, 0xff08, 0xff09, 0x3014, 0x3015, 0xff3b, 0xff3d, 0xff5b, // 0x8168 - 0x816f
		0xff5d, 0x3008, 0x3009, 0x300a, 0x300b, 0x300c, 0x300d, 0x300e, // 0x8170 - 0x8177
		0x300f, 0x3010, 0x3011, 0xff0b, 0x2212, 0x00b1, 0x00d7, 0x0000, // 0x8178 - 0x817f
		0x00f7, 0xff1d, 0x2260, 0xff1c, 0xff1e, 0x2266, 0x2267, 0x221e, // 0x8180 - 0x8187
		0x2234, 0x2642, 0x2640, 0x00b0, 0x2032, 0x2033, 0x2103, 0xffe5, // 0x8188 - 0x818f
		0xff04, 0x00a2, 0x00a3, 0xff05, 0xff03, 0xff06, 0xff0a, 0xff20, // 0x8190 - 0x8197
		0x00a7, 0x2606, 0x2605, 0x25cb, 0x25cf, 0x25ce, 0x25c7, 0x25c6, // 0x8198 - 0x819f
		0x25a1, 0x25a0, 0x25b3, 0x25b2, 0x25bd, 0x25bc, 0x203b, 0x3012, // 0x81a0 - 0x81a7
		0x2192, 0x2190, 0x2191, 0x2193, 0x3013, 0x0000, 0x0000, 0x0000, // 0x81a8 - 0x81af
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x81b0 - 0x81b7
		0x2208, 0x220b, 0x2286, 0x2287, 0x2282, 0x2283, 0x222a, 0x2229, // 0x81b8 - 0x81bf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x81c0 - 0x81c7
		0x2227, 0x2228, 0x00ac, 0x21d2, 0x21d4, 0x2200, 0x2203, 0x0000, // 0x81c8 - 0x81cf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x81d0 - 0x81d7
		0x0000, 0x0000, 0x2220, 0x22a5, 0x2312, 0x2202, 0x2207, 0x2261, // 0x81d8 - 0x81df
		0x2252, 0x226a, 0x226b, 0x221a, 0x223d, 0x221d, 0x2235, 0x222b, // 0x81e0 - 0x81e7
		0x222c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x81e8 - 0x81ef
		0x212b, 0x2030, 0x266f, 0x266d, 0x266a, 0x2020, 0x2021, 0x00b6, // 0x81f0 - 0x81f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x25ef, 0x0000, 0x0000, 0x0000, // 0x81f8 - 0x81ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8200 - 0x8207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8208 - 0x820f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8210 - 0x8217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8218 - 0x821f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8220 - 0x8227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8228 - 0x822f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8230 - 0x8237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8238 - 0x823f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8240 - 0x8247
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xff10, // 0x8248 - 0x824f
		0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, 0xff18, // 0x8250 - 0x8257
		0xff19, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8258 - 0x825f
		0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27, 0xff28, // 0x8260 - 0x8267
		0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f, 0xff30, // 0x8268 - 0x826f
		0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37, 0xff38, // 0x8270 - 0x8277
		0xff39, 0xff3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8278 - 0x827f
		0x0000, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47, // 0x8280 - 0x8287
		0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f, // 0x8288 - 0x828f
		0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57, // 0x8290 - 0x8297
		0xff58, 0xff59, 0xff5a, 0x0000, 0x0000, 0x0000, 0x0000, 0x3041, // 0x8298 - 0x829f
		0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047, 0x3048, 0x3049, // 0x82a0 - 0x82a7
		0x304a, 0x304b, 0x304c, 0x304d, 0x304e, 0x304f, 0x3050, 0x3051, // 0x82a8 - 0x82af
		0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057, 0x3058, 0x3059, // 0x82b0 - 0x82b7
		0x305a, 0x305b, 0x305c, 0x305d, 0x305e, 0x305f, 0x3060, 0x3061, // 0x82b8 - 0x82bf
		0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 0x3067, 0x3068, 0x3069, // 0x82c0 - 0x82c7
		0x306a, 0x306b, 0x306c, 0x306d, 0x306e, 0x306f, 0x3070, 0x3071, // 0x82c8 - 0x82cf
		0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077, 0x3078, 0x3079, // 0x82d0 - 0x82d7
		0x307a, 0x307b, 0x307c, 0x307d, 0x307e, 0x307f, 0x3080, 0x3081, // 0x82d8 - 0x82df
		0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087, 0x3088, 0x3089, // 0x82e0 - 0x82e7
		0x308a, 0x308b, 0x308c, 0x308d, 0x308e, 0x308f, 0x3090, 0x3091, // 0x82e8 - 0x82ef
		0x3092, 0x3093, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x82f0 - 0x82f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x82f8 - 0x82ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8300 - 0x8307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8308 - 0x830f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8310 - 0x8317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8318 - 0x831f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8320 - 0x8327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8328 - 0x832f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8330 - 0x8337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8338 - 0x833f
		0x30a1, 0x30a2, 0x30a3, 0x30a4, 0x30a5, 0x30a6, 0x30a7, 0x30a8, // 0x8340 - 0x8347
		0x30a9, 0x30aa, 0x30ab, 0x30ac, 0x30ad, 0x30ae, 0x30af, 0x30b0, // 0x8348 - 0x834f
		0x30b1, 0x30b2, 0x30b3, 0x30b4, 0x30b5, 0x30b6, 0x30b7, 0x30b8, // 0x8350 - 0x8357
		0x30b9, 0x30ba, 0x30bb, 0x30bc, 0x30bd, 0x30be, 0x30bf, 0x30c0, // 0x8358 - 0x835f
		0x30c1, 0x30c2, 0x30c3, 0x30c4, 0x30c5, 0x30c6, 0x30c7, 0x30c8, // 0x8360 - 0x8367
		0x30c9, 0x30ca, 0x30cb, 0x30cc, 0x30cd, 0x30ce, 0x30cf, 0x30d0, // 0x8368 - 0x836f
		0x30d1, 0x30d2, 0x30d3, 0x30d4, 0x30d5, 0x30d6, 0x30d7, 0x30d8, // 0x8370 - 0x8377
		0x30d9, 0x30da, 0x30db, 0x30dc, 0x30dd, 0x30de, 0x30df, 0x0000, // 0x8378 - 0x837f
		0x30e0, 0x30e1, 0x30e2, 0x30e3, 0x30e4, 0x30e5, 0x30e6, 0x30e7, // 0x8380 - 0x8387
		0x30e8, 0x30e9, 0x30ea, 0x30eb, 0x30ec, 0x30ed, 0x30ee, 0x30ef, // 0x8388 - 0x838f
		0x30f0, 0x30f1, 0x30f2, 0x30f3, 0x30f4, 0x30f5, 0x30f6, 0x0000, // 0x8390 - 0x8397
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0391, // 0x8398 - 0x839f
		0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 0x0398, 0x0399, // 0x83a0 - 0x83a7
		0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f, 0x03a0, 0x03a1, // 0x83a8 - 0x83af
		0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8, 0x03a9, 0x0000, // 0x83b0 - 0x83b7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x03b1, // 0x83b8 - 0x83bf
		0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7, 0x03b8, 0x03b9, // 0x83c0 - 0x83c7
		0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1, // 0x83c8 - 0x83cf
		0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8, 0x03c9, 0x0000, // 0x83d0 - 0x83d7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x83d8 - 0x83df
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x83e0 - 0x83e7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x83e8 - 0x83ef
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x83f0 - 0x83f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x83f8 - 0x83ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8400 - 0x8407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8408 - 0x840f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8410 - 0x8417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8418 - 0x841f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8420 - 0x8427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8428 - 0x842f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8430 - 0x8437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8438 - 0x843f
		0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401, 0x0416, // 0x8440 - 0x8447
		0x0417, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, // 0x8448 - 0x844f
		0x041f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, // 0x8450 - 0x8457
		0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, // 0x8458 - 0x845f
		0x042f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8460 - 0x8467
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8468 - 0x846f
		0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451, 0x0436, // 0x8470 - 0x8477
		0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x0000, // 0x8478 - 0x847f
		0x043e, 0x043f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, // 0x8480 - 0x8487
		0x0446, 0x0447, 0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, // 0x8488 - 0x848f
		0x044e, 0x044f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8490 - 0x8497
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2500, // 0x8498 - 0x849f
		0x2502, 0x250c, 0x2510, 0x2518, 0x2514, 0x251c, 0x252c, 0x2524, // 0x84a0 - 0x84a7
		0x2534, 0x253c, 0x2501, 0x2503, 0x250f, 0x2513, 0x251b, 0x2517, // 0x84a8 - 0x84af
		0x2523, 0x2533, 0x252b, 0x253b, 0x254b, 0x2520, 0x252f, 0x2528, // 0x84b0 - 0x84b7
		0x2537, 0x253f, 0x251d, 0x2530, 0x2525, 0x2538, 0x2542, 0x0000, // 0x84b8 - 0x84bf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84c0 - 0x84c7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84c8 - 0x84cf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84d0 - 0x84d7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84d8 - 0x84df
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84e0 - 0x84e7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84e8 - 0x84ef
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84f0 - 0x84f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x84f8 - 0x84ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8500 - 0x8507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8508 - 0x850f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8510 - 0x8517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8518 - 0x851f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8520 - 0x8527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8528 - 0x852f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8530 - 0x8537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8538 - 0x853f
		0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466, 0x2467, // 0x8540 - 0x8547
		0x2468, 0x2469, 0x246a, 0x246b, 0x246c, 0x246d, 0x246e, 0x246f, // 0x8548 - 0x854f
		0x2470, 0x2471, 0x2472, 0x2473, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8550 - 0x8557
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2474, 0x2475, // 0x8558 - 0x855f
		0x2476, 0x2477, 0x2478, 0x2479, 0x247a, 0x247b, 0x247c, 0x247d, // 0x8560 - 0x8567
		0x247e, 0x247f, 0x2480, 0x2481, 0x2482, 0x2483, 0x2484, 0x2485, // 0x8568 - 0x856f
		0x2486, 0x2487, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8570 - 0x8577
		0x0000, 0x0000, 0x0000, 0x0000, 0x2776, 0x2777, 0x2778, 0x0000, // 0x8578 - 0x857f
		0x2779, 0x277a, 0x277b, 0x277c, 0x277d, 0x277e, 0x0000, 0x0000, // 0x8580 - 0x8587
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8588 - 0x858f
		0x0000, 0x0000, 0x2488, 0x2489, 0x248a, 0x248b, 0x248c, 0x248d, // 0x8590 - 0x8597
		0x248e, 0x248f, 0x2490, 0x0000, 0x0000, 0x0000, 0x0000, 0x2160, // 0x8598 - 0x859f
		0x2161, 0x2162, 0x2163, 0x2164, 0x2165, 0x2166, 0x2167, 0x2168, // 0x85a0 - 0x85a7
		0x2169, 0x216a, 0x216b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x85a8 - 0x85af
		0x0000, 0x0000, 0x0000, 0x2170, 0x2171, 0x2172, 0x2173, 0x2174, // 0x85b0 - 0x85b7
		0x2175, 0x2176, 0x2177, 0x2178, 0x2179, 0x217a, 0x217b, 0x0000, // 0x85b8 - 0x85bf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x85c0 - 0x85c7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x85c8 - 0x85cf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x85d0 - 0x85d7
		0x0000, 0x0000, 0x0000, 0x249c, 0x249d, 0x249e, 0x249f, 0x24a0, // 0x85d8 - 0x85df
		0x24a1, 0x24a2, 0x24a3, 0x24a4, 0x24a5, 0x24a6, 0x24a7, 0x24a8, // 0x85e0 - 0x85e7
		0x24a9, 0x24aa, 0x24ab, 0x24ac, 0x24ad, 0x24ae, 0x24af, 0x24b0, // 0x85e8 - 0x85ef
		0x24b1, 0x24b2, 0x24b3, 0x24b4, 0x24b5, 0x0000, 0x0000, 0x0000, // 0x85f0 - 0x85f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x85f8 - 0x85ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8600 - 0x8607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8608 - 0x860f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8610 - 0x8617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8618 - 0x861f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8620 - 0x8627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8628 - 0x862f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8630 - 0x8637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8638 - 0x863f
		0x339c, 0x339f, 0x339d, 0x33a0, 0x33a4, 0x0000, 0x33a1, 0x33a5, // 0x8640 - 0x8647
		0x339e, 0x33a2, 0x338e, 0x0000, 0x338f, 0x33c4, 0x3396, 0x3397, // 0x8648 - 0x864f
		0x2113, 0x3398, 0x33b3, 0x33b2, 0x33b1, 0x33b0, 0x2109, 0x33d4, // 0x8650 - 0x8657
		0x33cb, 0x3390, 0x3385, 0x3386, 0x3387, 0x0000, 0x0000, 0x0000, // 0x8658 - 0x865f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8660 - 0x8667
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8668 - 0x866f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8670 - 0x8677
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8678 - 0x867f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8680 - 0x8687
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8688 - 0x868f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8690 - 0x8697
		0x0000, 0x0000, 0x0000, 0x2116, 0x33cd, 0x2121, 0x0000, 0x2664, // 0x8698 - 0x869f
		0x2667, 0x2661, 0x2662, 0x2660, 0x2663, 0x2665, 0x2666, 0x0000, // 0x86a0 - 0x86a7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86a8 - 0x86af
		0x0000, 0x0000, 0x0000, 0x3020, 0x260e, 0x3004, 0x0000, 0x0000, // 0x86b0 - 0x86b7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86b8 - 0x86bf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x261e, // 0x86c0 - 0x86c7
		0x261c, 0x261d, 0x261f, 0x21c6, 0x21c4, 0x21c5, 0x0000, 0x21e8, // 0x86c8 - 0x86cf
		0x21e6, 0x21e7, 0x21e9, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86d0 - 0x86d7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86d8 - 0x86df
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86e0 - 0x86e7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86e8 - 0x86ef
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86f0 - 0x86f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x86f8 - 0x86ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8700 - 0x8707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8708 - 0x870f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8710 - 0x8717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8718 - 0x871f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8720 - 0x8727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8728 - 0x872f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8730 - 0x8737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8738 - 0x873f
		0x3230, 0x322a, 0x322b, 0x322c, 0x322d, 0x322e, 0x322f, 0x3240, // 0x8740 - 0x8747
		0x3237, 0x3242, 0x3243, 0x3239, 0x323a, 0x3231, 0x323e, 0x3234, // 0x8748 - 0x874f
		0x3232, 0x323b, 0x3236, 0x3233, 0x3235, 0x323c, 0x323d, 0x323f, // 0x8750 - 0x8757
		0x3238, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8758 - 0x875f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8760 - 0x8767
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8768 - 0x876f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8770 - 0x8777
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8778 - 0x877f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8780 - 0x8787
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8788 - 0x878f
		0x0000, 0x0000, 0x0000, 0x32a4, 0x32a5, 0x32a6, 0x32a7, 0x32a8, // 0x8790 - 0x8797
		0x32a9, 0x3296, 0x329d, 0x3298, 0x329e, 0x0000, 0x3299, 0x3349, // 0x8798 - 0x879f
		0x3322, 0x334d, 0x3314, 0x3316, 0x3305, 0x3333, 0x334e, 0x3303, // 0x87a0 - 0x87a7
		0x3336, 0x3318, 0x3315, 0x3327, 0x3351, 0x334a, 0x3339, 0x3357, // 0x87a8 - 0x87af
		0x330d, 0x3342, 0x3323, 0x3326, 0x333b, 0x332b, 0x0000, 0x0000, // 0x87b0 - 0x87b7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3300, 0x331e, 0x332a, // 0x87b8 - 0x87bf
		0x3331, 0x3347, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87c0 - 0x87c7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87c8 - 0x87cf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87d0 - 0x87d7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87d8 - 0x87df
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x337e, 0x337d, 0x337c, // 0x87e0 - 0x87e7
		0x337b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87e8 - 0x87ef
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87f0 - 0x87f7
		0x0000, 0x0000, 0x337f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x87f8 - 0x87ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8800 - 0x8807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8808 - 0x880f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8810 - 0x8817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8818 - 0x881f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8820 - 0x8827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8828 - 0x882f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8830 - 0x8837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8838 - 0x883f
		0x222e, 0x221f, 0x22bf, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8840 - 0x8847
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8848 - 0x884f
		0x0000, 0x0000, 0x0000, 0x0000, 0x301d, 0x301f, 0x0000, 0x0000, // 0x8850 - 0x8857
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8858 - 0x885f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8860 - 0x8867
		0x3094, 0x0000, 0x30f7, 0x30f8, 0x30f9, 0x30fa, 0x0000, 0x0000, // 0x8868 - 0x886f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8870 - 0x8877
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8878 - 0x887f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8880 - 0x8887
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8888 - 0x888f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8890 - 0x8897
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e9c, // 0x8898 - 0x889f
		0x5516, 0x5a03, 0x963f, 0x54c0, 0x611b, 0x6328, 0x59f6, 0x9022, // 0x88a0 - 0x88a7
		0x8475, 0x831c, 0x7a50, 0x60aa, 0x63e1, 0x6e25, 0x65ed, 0x8466, // 0x88a8 - 0x88af
		0x82a6, 0x9bf5, 0x6893, 0x5727, 0x65a1, 0x6271, 0x5b9b, 0x59d0, // 0x88b0 - 0x88b7
		0x867b, 0x98f4, 0x7d62, 0x7dbe, 0x9b8e, 0x6216, 0x7c9f, 0x88b7, // 0x88b8 - 0x88bf
		0x5b89, 0x5eb5, 0x6309, 0x6697, 0x6848, 0x95c7, 0x978d, 0x674f, // 0x88c0 - 0x88c7
		0x4ee5, 0x4f0a, 0x4f4d, 0x4f9d, 0x5049, 0x56f2, 0x5937, 0x59d4, // 0x88c8 - 0x88cf
		0x5a01, 0x5c09, 0x60df, 0x610f, 0x6170, 0x6613, 0x6905, 0x70ba, // 0x88d0 - 0x88d7
		0x754f, 0x7570, 0x79fb, 0x7dad, 0x7def, 0x80c3, 0x840e, 0x8863, // 0x88d8 - 0x88df
		0x8b02, 0x9055, 0x907a, 0x533b, 0x4e95, 0x4ea5, 0x57df, 0x80b2, // 0x88e0 - 0x88e7
		0x90c1, 0x78ef, 0x4e00, 0x58f1, 0x6ea2, 0x9038, 0x7a32, 0x8328, // 0x88e8 - 0x88ef
		0x828b, 0x9c2f, 0x5141, 0x5370, 0x54bd, 0x54e1, 0x56e0, 0x59fb, // 0x88f0 - 0x88f7
		0x5f15, 0x98f2, 0x6deb, 0x80e4, 0x852d, 0x0000, 0x0000, 0x0000, // 0x88f8 - 0x88ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8900 - 0x8907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8908 - 0x890f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8910 - 0x8917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8918 - 0x891f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8920 - 0x8927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8928 - 0x892f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8930 - 0x8937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8938 - 0x893f
		0x9662, 0x9670, 0x96a0, 0x97fb, 0x540b, 0x53f3, 0x5b87, 0x70cf, // 0x8940 - 0x8947
		0x7fbd, 0x8fc2, 0x96e8, 0x536f, 0x9d5c, 0x7aba, 0x4e11, 0x7893, // 0x8948 - 0x894f
		0x81fc, 0x6e26, 0x5618, 0x5504, 0x6b1d, 0x851a, 0x9c3b, 0x59e5, // 0x8950 - 0x8957
		0x53a9, 0x6d66, 0x74dc, 0x958f, 0x5642, 0x4e91, 0x904b, 0x96f2, // 0x8958 - 0x895f
		0x834f, 0x990c, 0x53e1, 0x55b6, 0x5b30, 0x5f71, 0x6620, 0x66f3, // 0x8960 - 0x8967
		0x6804, 0x6c38, 0x6cf3, 0x6d29, 0x745b, 0x76c8, 0x7a4e, 0x9834, // 0x8968 - 0x896f
		0x82f1, 0x885b, 0x8a60, 0x92ed, 0x6db2, 0x75ab, 0x76ca, 0x99c5, // 0x8970 - 0x8977
		0x60a6, 0x8b01, 0x8d8a, 0x95b2, 0x698e, 0x53ad, 0x5186, 0x0000, // 0x8978 - 0x897f
		0x5712, 0x5830, 0x5944, 0x5bb4, 0x5ef6, 0x6028, 0x63a9, 0x63f4, // 0x8980 - 0x8987
		0x6cbf, 0x6f14, 0x708e, 0x7114, 0x7159, 0x71d5, 0x733f, 0x7e01, // 0x8988 - 0x898f
		0x8276, 0x82d1, 0x8597, 0x9060, 0x925b, 0x9d1b, 0x5869, 0x65bc, // 0x8990 - 0x8997
		0x6c5a, 0x7525, 0x51f9, 0x592e, 0x5965, 0x5f80, 0x5fdc, 0x62bc, // 0x8998 - 0x899f
		0x65fa, 0x6a2a, 0x6b27, 0x6bb4, 0x738b, 0x7fc1, 0x8956, 0x9d2c, // 0x89a0 - 0x89a7
		0x9d0e, 0x9ec4, 0x5ca1, 0x6c96, 0x837b, 0x5104, 0x5c4b, 0x61b6, // 0x89a8 - 0x89af
		0x81c6, 0x6876, 0x7261, 0x4e59, 0x4ffa, 0x5378, 0x6069, 0x6e29, // 0x89b0 - 0x89b7
		0x7a4f, 0x97f3, 0x4e0b, 0x5316, 0x4eee, 0x4f55, 0x4f3d, 0x4fa1, // 0x89b8 - 0x89bf
		0x4f73, 0x52a0, 0x53ef, 0x5609, 0x590f, 0x5ac1, 0x5bb6, 0x5be1, // 0x89c0 - 0x89c7
		0x79d1, 0x6687, 0x679c, 0x67b6, 0x6b4c, 0x6cb3, 0x706b, 0x73c2, // 0x89c8 - 0x89cf
		0x798d, 0x79be, 0x7a3c, 0x7b87, 0x82b1, 0x82db, 0x8304, 0x8377, // 0x89d0 - 0x89d7
		0x83ef, 0x83d3, 0x8766, 0x8ab2, 0x5629, 0x8ca8, 0x8fe6, 0x904e, // 0x89d8 - 0x89df
		0x971e, 0x868a, 0x4fc4, 0x5ce8, 0x6211, 0x7259, 0x753b, 0x81e5, // 0x89e0 - 0x89e7
		0x82bd, 0x86fe, 0x8cc0, 0x96c5, 0x9913, 0x99d5, 0x4ecb, 0x4f1a, // 0x89e8 - 0x89ef
		0x89e3, 0x56de, 0x584a, 0x58ca, 0x5efb, 0x5feb, 0x602a, 0x6094, // 0x89f0 - 0x89f7
		0x6062, 0x61d0, 0x6212, 0x62d0, 0x6539, 0x0000, 0x0000, 0x0000, // 0x89f8 - 0x89ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a00 - 0x8a07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a08 - 0x8a0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a10 - 0x8a17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a18 - 0x8a1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a20 - 0x8a27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a28 - 0x8a2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a30 - 0x8a37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8a38 - 0x8a3f
		0x9b41, 0x6666, 0x68b0, 0x6d77, 0x7070, 0x754c, 0x7686, 0x7d75, // 0x8a40 - 0x8a47
		0x82a5, 0x87f9, 0x958b, 0x968e, 0x8c9d, 0x51f1, 0x52be, 0x5916, // 0x8a48 - 0x8a4f
		0x54b3, 0x5bb3, 0x5d16, 0x6168, 0x6982, 0x6daf, 0x788d, 0x84cb, // 0x8a50 - 0x8a57
		0x8857, 0x8a72, 0x93a7, 0x9ab8, 0x6d6c, 0x99a8, 0x86d9, 0x57a3, // 0x8a58 - 0x8a5f
		0x67ff, 0x86ce, 0x920e, 0x5283, 0x5687, 0x5404, 0x5ed3, 0x62e1, // 0x8a60 - 0x8a67
		0x64b9, 0x683c, 0x6838, 0x6bbb, 0x7372, 0x78ba, 0x7a6b, 0x899a, // 0x8a68 - 0x8a6f
		0x89d2, 0x8d6b, 0x8f03, 0x90ed, 0x95a3, 0x9694, 0x9769, 0x5b66, // 0x8a70 - 0x8a77
		0x5cb3, 0x697d, 0x984d, 0x984e, 0x639b, 0x7b20, 0x6a2b, 0x0000, // 0x8a78 - 0x8a7f
		0x6a7f, 0x68b6, 0x9c0d, 0x6f5f, 0x5272, 0x559d, 0x6070, 0x62ec, // 0x8a80 - 0x8a87
		0x6d3b, 0x6e07, 0x6ed1, 0x845b, 0x8910, 0x8f44, 0x4e14, 0x9c39, // 0x8a88 - 0x8a8f
		0x53f6, 0x691b, 0x6a3a, 0x9784, 0x682a, 0x515c, 0x7ac3, 0x84b2, // 0x8a90 - 0x8a97
		0x91dc, 0x938c, 0x565b, 0x9d28, 0x6822, 0x8305, 0x8431, 0x7ca5, // 0x8a98 - 0x8a9f
		0x5208, 0x82c5, 0x74e6, 0x4e7e, 0x4f83, 0x51a0, 0x5bd2, 0x520a, // 0x8aa0 - 0x8aa7
		0x52d8, 0x52e7, 0x5dfb, 0x559a, 0x582a, 0x59e6, 0x5b8c, 0x5b98, // 0x8aa8 - 0x8aaf
		0x5bdb, 0x5e72, 0x5e79, 0x60a3, 0x611f, 0x6163, 0x61be, 0x63db, // 0x8ab0 - 0x8ab7
		0x6562, 0x67d1, 0x6853, 0x68fa, 0x6b3e, 0x6b53, 0x6c57, 0x6f22, // 0x8ab8 - 0x8abf
		0x6f97, 0x6f45, 0x74b0, 0x7518, 0x76e3, 0x770b, 0x7aff, 0x7ba1, // 0x8ac0 - 0x8ac7
		0x7c21, 0x7de9, 0x7f36, 0x7ff0, 0x809d, 0x8266, 0x839e, 0x89b3, // 0x8ac8 - 0x8acf
		0x8acc, 0x8cab, 0x9084, 0x9451, 0x9593, 0x9591, 0x95a2, 0x9665, // 0x8ad0 - 0x8ad7
		0x97d3, 0x9928, 0x8218, 0x4e38, 0x542b, 0x5cb8, 0x5dcc, 0x73a9, // 0x8ad8 - 0x8adf
		0x764c, 0x773c, 0x5ca9, 0x7feb, 0x8d0b, 0x96c1, 0x9811, 0x9854, // 0x8ae0 - 0x8ae7
		0x9858, 0x4f01, 0x4f0e, 0x5371, 0x559c, 0x5668, 0x57fa, 0x5947, // 0x8ae8 - 0x8aef
		0x5b09, 0x5bc4, 0x5c90, 0x5e0c, 0x5e7e, 0x5fcc, 0x63ee, 0x673a, // 0x8af0 - 0x8af7
		0x65d7, 0x65e2, 0x671f, 0x68cb, 0x68c4, 0x0000, 0x0000, 0x0000, // 0x8af8 - 0x8aff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b00 - 0x8b07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b08 - 0x8b0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b10 - 0x8b17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b18 - 0x8b1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b20 - 0x8b27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b28 - 0x8b2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b30 - 0x8b37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8b38 - 0x8b3f
		0x6a5f, 0x5e30, 0x6bc5, 0x6c17, 0x6c7d, 0x757f, 0x7948, 0x5b63, // 0x8b40 - 0x8b47
		0x7a00, 0x7d00, 0x5fbd, 0x898f, 0x8a18, 0x8cb4, 0x8d77, 0x8ecc, // 0x8b48 - 0x8b4f
		0x8f1d, 0x98e2, 0x9a0e, 0x9b3c, 0x4e80, 0x507d, 0x5100, 0x5993, // 0x8b50 - 0x8b57
		0x5b9c, 0x622f, 0x6280, 0x64ec, 0x6b3a, 0x72a0, 0x7591, 0x7947, // 0x8b58 - 0x8b5f
		0x7fa9, 0x87fb, 0x8abc, 0x8b70, 0x63ac, 0x83ca, 0x97a0, 0x5409, // 0x8b60 - 0x8b67
		0x5403, 0x55ab, 0x6854, 0x6a58, 0x8a70, 0x7827, 0x6775, 0x9ecd, // 0x8b68 - 0x8b6f
		0x5374, 0x5ba2, 0x811a, 0x8650, 0x9006, 0x4e18, 0x4e45, 0x4ec7, // 0x8b70 - 0x8b77
		0x4f11, 0x53ca, 0x5438, 0x5bae, 0x5f13, 0x6025, 0x6551, 0x0000, // 0x8b78 - 0x8b7f
		0x673d, 0x6c42, 0x6c72, 0x6ce3, 0x7078, 0x7403, 0x7a76, 0x7aae, // 0x8b80 - 0x8b87
		0x7b08, 0x7d1a, 0x7cfe, 0x7d66, 0x65e7, 0x725b, 0x53bb, 0x5c45, // 0x8b88 - 0x8b8f
		0x5de8, 0x62d2, 0x62e0, 0x6319, 0x6e20, 0x865a, 0x8a31, 0x8ddd, // 0x8b90 - 0x8b97
		0x92f8, 0x6f01, 0x79a6, 0x9b5a, 0x4ea8, 0x4eab, 0x4eac, 0x4f9b, // 0x8b98 - 0x8b9f
		0x4fa0, 0x50d1, 0x5147, 0x7af6, 0x5171, 0x51f6, 0x5354, 0x5321, // 0x8ba0 - 0x8ba7
		0x537f, 0x53eb, 0x55ac, 0x5883, 0x5ce1, 0x5f37, 0x5f4a, 0x602f, // 0x8ba8 - 0x8baf
		0x6050, 0x606d, 0x631f, 0x6559, 0x6a4b, 0x6cc1, 0x72c2, 0x72ed, // 0x8bb0 - 0x8bb7
		0x77ef, 0x80f8, 0x8105, 0x8208, 0x854e, 0x90f7, 0x93e1, 0x97ff, // 0x8bb8 - 0x8bbf
		0x9957, 0x9a5a, 0x4ef0, 0x51dd, 0x5c2d, 0x6681, 0x696d, 0x5c40, // 0x8bc0 - 0x8bc7
		0x66f2, 0x6975, 0x7389, 0x6850, 0x7c81, 0x50c5, 0x52e4, 0x5747, // 0x8bc8 - 0x8bcf
		0x5dfe, 0x9326, 0x65a4, 0x6b23, 0x6b3d, 0x7434, 0x7981, 0x79bd, // 0x8bd0 - 0x8bd7
		0x7b4b, 0x7dca, 0x82b9, 0x83cc, 0x887f, 0x895f, 0x8b39, 0x8fd1, // 0x8bd8 - 0x8bdf
		0x91d1, 0x541f, 0x9280, 0x4e5d, 0x5036, 0x53e5, 0x533a, 0x72d7, // 0x8be0 - 0x8be7
		0x7396, 0x77e9, 0x82e6, 0x8eaf, 0x99c6, 0x99c8, 0x99d2, 0x5177, // 0x8be8 - 0x8bef
		0x611a, 0x865e, 0x55b0, 0x7a7a, 0x5076, 0x5bd3, 0x9047, 0x9685, // 0x8bf0 - 0x8bf7
		0x4e32, 0x6adb, 0x91e7, 0x5c51, 0x5c48, 0x0000, 0x0000, 0x0000, // 0x8bf8 - 0x8bff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c00 - 0x8c07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c08 - 0x8c0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c10 - 0x8c17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c18 - 0x8c1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c20 - 0x8c27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c28 - 0x8c2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c30 - 0x8c37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8c38 - 0x8c3f
		0x6398, 0x7a9f, 0x6c93, 0x9774, 0x8f61, 0x7aaa, 0x718a, 0x9688, // 0x8c40 - 0x8c47
		0x7c82, 0x6817, 0x7e70, 0x6851, 0x936c, 0x52f2, 0x541b, 0x85ab, // 0x8c48 - 0x8c4f
		0x8a13, 0x7fa4, 0x8ecd, 0x90e1, 0x5366, 0x8888, 0x7941, 0x4fc2, // 0x8c50 - 0x8c57
		0x50be, 0x5211, 0x5144, 0x5553, 0x572d, 0x73ea, 0x578b, 0x5951, // 0x8c58 - 0x8c5f
		0x5f62, 0x5f84, 0x6075, 0x6176, 0x6167, 0x61a9, 0x63b2, 0x643a, // 0x8c60 - 0x8c67
		0x656c, 0x666f, 0x6842, 0x6e13, 0x7566, 0x7a3d, 0x7cfb, 0x7d4c, // 0x8c68 - 0x8c6f
		0x7d99, 0x7e4b, 0x7f6b, 0x830e, 0x834a, 0x86cd, 0x8a08, 0x8a63, // 0x8c70 - 0x8c77
		0x8b66, 0x8efd, 0x981a, 0x9d8f, 0x82b8, 0x8fce, 0x9be8, 0x0000, // 0x8c78 - 0x8c7f
		0x5287, 0x621f, 0x6483, 0x6fc0, 0x9699, 0x6841, 0x5091, 0x6b20, // 0x8c80 - 0x8c87
		0x6c7a, 0x6f54, 0x7a74, 0x7d50, 0x8840, 0x8a23, 0x6708, 0x4ef6, // 0x8c88 - 0x8c8f
		0x5039, 0x5026, 0x5065, 0x517c, 0x5238, 0x5263, 0x55a7, 0x570f, // 0x8c90 - 0x8c97
		0x5805, 0x5acc, 0x5efa, 0x61b2, 0x61f8, 0x62f3, 0x6372, 0x691c, // 0x8c98 - 0x8c9f
		0x6a29, 0x727d, 0x72ac, 0x732e, 0x7814, 0x786f, 0x7d79, 0x770c, // 0x8ca0 - 0x8ca7
		0x80a9, 0x898b, 0x8b19, 0x8ce2, 0x8ed2, 0x9063, 0x9375, 0x967a, // 0x8ca8 - 0x8caf
		0x9855, 0x9a13, 0x9e78, 0x5143, 0x539f, 0x53b3, 0x5e7b, 0x5f26, // 0x8cb0 - 0x8cb7
		0x6e1b, 0x6e90, 0x7384, 0x73fe, 0x7d43, 0x8237, 0x8a00, 0x8afa, // 0x8cb8 - 0x8cbf
		0x9650, 0x4e4e, 0x500b, 0x53e4, 0x547c, 0x56fa, 0x59d1, 0x5b64, // 0x8cc0 - 0x8cc7
		0x5df1, 0x5eab, 0x5f27, 0x6238, 0x6545, 0x67af, 0x6e56, 0x72d0, // 0x8cc8 - 0x8ccf
		0x7cca, 0x88b4, 0x80a1, 0x80e1, 0x83f0, 0x864e, 0x8a87, 0x8de8, // 0x8cd0 - 0x8cd7
		0x9237, 0x96c7, 0x9867, 0x9f13, 0x4e94, 0x4e92, 0x4f0d, 0x5348, // 0x8cd8 - 0x8cdf
		0x5449, 0x543e, 0x5a2f, 0x5f8c, 0x5fa1, 0x609f, 0x68a7, 0x6a8e, // 0x8ce0 - 0x8ce7
		0x745a, 0x7881, 0x8a9e, 0x8aa4, 0x8b77, 0x9190, 0x4e5e, 0x9bc9, // 0x8ce8 - 0x8cef
		0x4ea4, 0x4f7c, 0x4faf, 0x5019, 0x5016, 0x5149, 0x516c, 0x529f, // 0x8cf0 - 0x8cf7
		0x52b9, 0x52fe, 0x539a, 0x53e3, 0x5411, 0x0000, 0x0000, 0x0000, // 0x8cf8 - 0x8cff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d00 - 0x8d07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d08 - 0x8d0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d10 - 0x8d17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d18 - 0x8d1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d20 - 0x8d27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d28 - 0x8d2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d30 - 0x8d37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8d38 - 0x8d3f
		0x540e, 0x5589, 0x5751, 0x57a2, 0x597d, 0x5b54, 0x5b5d, 0x5b8f, // 0x8d40 - 0x8d47
		0x5de5, 0x5de7, 0x5df7, 0x5e78, 0x5e83, 0x5e9a, 0x5eb7, 0x5f18, // 0x8d48 - 0x8d4f
		0x6052, 0x614c, 0x6297, 0x62d8, 0x63a7, 0x653b, 0x6602, 0x6643, // 0x8d50 - 0x8d57
		0x66f4, 0x676d, 0x6821, 0x6897, 0x69cb, 0x6c5f, 0x6d2a, 0x6d69, // 0x8d58 - 0x8d5f
		0x6e2f, 0x6e9d, 0x7532, 0x7687, 0x786c, 0x7a3f, 0x7ce0, 0x7d05, // 0x8d60 - 0x8d67
		0x7d18, 0x7d5e, 0x7db1, 0x8015, 0x8003, 0x80af, 0x80b1, 0x8154, // 0x8d68 - 0x8d6f
		0x818f, 0x822a, 0x8352, 0x884c, 0x8861, 0x8b1b, 0x8ca2, 0x8cfc, // 0x8d70 - 0x8d77
		0x90ca, 0x9175, 0x9271, 0x783f, 0x92fc, 0x95a4, 0x964d, 0x0000, // 0x8d78 - 0x8d7f
		0x9805, 0x9999, 0x9ad8, 0x9d3b, 0x525b, 0x52ab, 0x53f7, 0x5408, // 0x8d80 - 0x8d87
		0x58d5, 0x62f7, 0x6fe0, 0x8c6a, 0x8f5f, 0x9eb9, 0x514b, 0x523b, // 0x8d88 - 0x8d8f
		0x544a, 0x56fd, 0x7a40, 0x9177, 0x9d60, 0x9ed2, 0x7344, 0x6f09, // 0x8d90 - 0x8d97
		0x8170, 0x7511, 0x5ffd, 0x60da, 0x9aa8, 0x72db, 0x8fbc, 0x6b64, // 0x8d98 - 0x8d9f
		0x9803, 0x4eca, 0x56f0, 0x5764, 0x58be, 0x5a5a, 0x6068, 0x61c7, // 0x8da0 - 0x8da7
		0x660f, 0x6606, 0x6839, 0x68b1, 0x6df7, 0x75d5, 0x7d3a, 0x826e, // 0x8da8 - 0x8daf
		0x9b42, 0x4e9b, 0x4f50, 0x53c9, 0x5506, 0x5d6f, 0x5de6, 0x5dee, // 0x8db0 - 0x8db7
		0x67fb, 0x6c99, 0x7473, 0x7802, 0x8a50, 0x9396, 0x88df, 0x5750, // 0x8db8 - 0x8dbf
		0x5ea7, 0x632b, 0x50b5, 0x50ac, 0x518d, 0x6700, 0x54c9, 0x585e, // 0x8dc0 - 0x8dc7
		0x59bb, 0x5bb0, 0x5f69, 0x624d, 0x63a1, 0x683d, 0x6b73, 0x6e08, // 0x8dc8 - 0x8dcf
		0x707d, 0x91c7, 0x7280, 0x7815, 0x7826, 0x796d, 0x658e, 0x7d30, // 0x8dd0 - 0x8dd7
		0x83dc, 0x88c1, 0x8f09, 0x969b, 0x5264, 0x5728, 0x6750, 0x7f6a, // 0x8dd8 - 0x8ddf
		0x8ca1, 0x51b4, 0x5742, 0x962a, 0x583a, 0x698a, 0x80b4, 0x54b2, // 0x8de0 - 0x8de7
		0x5d0e, 0x57fc, 0x7895, 0x9dfa, 0x4f5c, 0x524a, 0x548b, 0x643e, // 0x8de8 - 0x8def
		0x6628, 0x6714, 0x67f5, 0x7a84, 0x7b56, 0x7d22, 0x932f, 0x685c, // 0x8df0 - 0x8df7
		0x9bad, 0x7b39, 0x5319, 0x518a, 0x5237, 0x0000, 0x0000, 0x0000, // 0x8df8 - 0x8dff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e00 - 0x8e07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e08 - 0x8e0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e10 - 0x8e17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e18 - 0x8e1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e20 - 0x8e27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e28 - 0x8e2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e30 - 0x8e37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8e38 - 0x8e3f
		0x5bdf, 0x62f6, 0x64ae, 0x64e6, 0x672d, 0x6bba, 0x85a9, 0x96d1, // 0x8e40 - 0x8e47
		0x7690, 0x9bd6, 0x634c, 0x9306, 0x9bab, 0x76bf, 0x6652, 0x4e09, // 0x8e48 - 0x8e4f
		0x5098, 0x53c2, 0x5c71, 0x60e8, 0x6492, 0x6563, 0x685f, 0x71e6, // 0x8e50 - 0x8e57
		0x73ca, 0x7523, 0x7b97, 0x7e82, 0x8695, 0x8b83, 0x8cdb, 0x9178, // 0x8e58 - 0x8e5f
		0x9910, 0x65ac, 0x66ab, 0x6b8b, 0x4ed5, 0x4ed4, 0x4f3a, 0x4f7f, // 0x8e60 - 0x8e67
		0x523a, 0x53f8, 0x53f2, 0x55e3, 0x56db, 0x58eb, 0x59cb, 0x59c9, // 0x8e68 - 0x8e6f
		0x59ff, 0x5b50, 0x5c4d, 0x5e02, 0x5e2b, 0x5fd7, 0x601d, 0x6307, // 0x8e70 - 0x8e77
		0x652f, 0x5b5c, 0x65af, 0x65bd, 0x65e8, 0x679d, 0x6b62, 0x0000, // 0x8e78 - 0x8e7f
		0x6b7b, 0x6c0f, 0x7345, 0x7949, 0x79c1, 0x7cf8, 0x7d19, 0x7d2b, // 0x8e80 - 0x8e87
		0x80a2, 0x8102, 0x81f3, 0x8996, 0x8a5e, 0x8a69, 0x8a66, 0x8a8c, // 0x8e88 - 0x8e8f
		0x8aee, 0x8cc7, 0x8cdc, 0x96cc, 0x98fc, 0x6b6f, 0x4e8b, 0x4f3c, // 0x8e90 - 0x8e97
		0x4f8d, 0x5150, 0x5b57, 0x5bfa, 0x6148, 0x6301, 0x6642, 0x6b21, // 0x8e98 - 0x8e9f
		0x6ecb, 0x6cbb, 0x723e, 0x74bd, 0x75d4, 0x78c1, 0x793a, 0x800c, // 0x8ea0 - 0x8ea7
		0x8033, 0x81ea, 0x8494, 0x8f9e, 0x6c50, 0x9e7f, 0x5f0f, 0x8b58, // 0x8ea8 - 0x8eaf
		0x9d2b, 0x7afa, 0x8ef8, 0x5b8d, 0x96eb, 0x4e03, 0x53f1, 0x57f7, // 0x8eb0 - 0x8eb7
		0x5931, 0x5ac9, 0x5ba4, 0x6089, 0x6e7f, 0x6f06, 0x75be, 0x8cea, // 0x8eb8 - 0x8ebf
		0x5b9f, 0x8500, 0x7be0, 0x5072, 0x67f4, 0x829d, 0x5c61, 0x854a, // 0x8ec0 - 0x8ec7
		0x7e1e, 0x820e, 0x5199, 0x5c04, 0x6368, 0x8d66, 0x659c, 0x716e, // 0x8ec8 - 0x8ecf
		0x793e, 0x7d17, 0x8005, 0x8b1d, 0x8eca, 0x906e, 0x86c7, 0x90aa, // 0x8ed0 - 0x8ed7
		0x501f, 0x52fa, 0x5c3a, 0x6753, 0x707c, 0x7235, 0x914c, 0x91c8, // 0x8ed8 - 0x8edf
		0x932b, 0x82e5, 0x5bc2, 0x5f31, 0x60f9, 0x4e3b, 0x53d6, 0x5b88, // 0x8ee0 - 0x8ee7
		0x624b, 0x6731, 0x6b8a, 0x72e9, 0x73e0, 0x7a2e, 0x816b, 0x8da3, // 0x8ee8 - 0x8eef
		0x9152, 0x9996, 0x5112, 0x53d7, 0x546a, 0x5bff, 0x6388, 0x6a39, // 0x8ef0 - 0x8ef7
		0x7dac, 0x9700, 0x56da, 0x53ce, 0x5468, 0x0000, 0x0000, 0x0000, // 0x8ef8 - 0x8eff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f00 - 0x8f07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f08 - 0x8f0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f10 - 0x8f17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f18 - 0x8f1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f20 - 0x8f27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f28 - 0x8f2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f30 - 0x8f37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x8f38 - 0x8f3f
		0x5b97, 0x5c31, 0x5dde, 0x4fee, 0x6101, 0x62fe, 0x6d32, 0x79c0, // 0x8f40 - 0x8f47
		0x79cb, 0x7d42, 0x7e4d, 0x7fd2, 0x81ed, 0x821f, 0x8490, 0x8846, // 0x8f48 - 0x8f4f
		0x8972, 0x8b90, 0x8e74, 0x8f2f, 0x9031, 0x914b, 0x916c, 0x96c6, // 0x8f50 - 0x8f57
		0x919c, 0x4ec0, 0x4f4f, 0x5145, 0x5341, 0x5f93, 0x620e, 0x67d4, // 0x8f58 - 0x8f5f
		0x6c41, 0x6e0b, 0x7363, 0x7e26, 0x91cd, 0x9283, 0x53d4, 0x5919, // 0x8f60 - 0x8f67
		0x5bbf, 0x6dd1, 0x795d, 0x7e2e, 0x7c9b, 0x587e, 0x719f, 0x51fa, // 0x8f68 - 0x8f6f
		0x8853, 0x8ff0, 0x4fca, 0x5cfb, 0x6625, 0x77ac, 0x7ae3, 0x821c, // 0x8f70 - 0x8f77
		0x99ff, 0x51c6, 0x5faa, 0x65ec, 0x696f, 0x6b89, 0x6df3, 0x0000, // 0x8f78 - 0x8f7f
		0x6e96, 0x6f64, 0x76fe, 0x7d14, 0x5de1, 0x9075, 0x9187, 0x9806, // 0x8f80 - 0x8f87
		0x51e6, 0x521d, 0x6240, 0x6691, 0x66d9, 0x6e1a, 0x5eb6, 0x7dd2, // 0x8f88 - 0x8f8f
		0x7f72, 0x66f8, 0x85af, 0x85f7, 0x8af8, 0x52a9, 0x53d9, 0x5973, // 0x8f90 - 0x8f97
		0x5e8f, 0x5f90, 0x6055, 0x92e4, 0x9664, 0x50b7, 0x511f, 0x52dd, // 0x8f98 - 0x8f9f
		0x5320, 0x5347, 0x53ec, 0x54e8, 0x5546, 0x5531, 0x5617, 0x5968, // 0x8fa0 - 0x8fa7
		0x59be, 0x5a3c, 0x5bb5, 0x5c06, 0x5c0f, 0x5c11, 0x5c1a, 0x5e84, // 0x8fa8 - 0x8faf
		0x5e8a, 0x5ee0, 0x5f70, 0x627f, 0x6284, 0x62db, 0x638c, 0x6377, // 0x8fb0 - 0x8fb7
		0x6607, 0x660c, 0x662d, 0x6676, 0x677e, 0x68a2, 0x6a1f, 0x6a35, // 0x8fb8 - 0x8fbf
		0x6cbc, 0x6d88, 0x6e09, 0x6e58, 0x713c, 0x7126, 0x7167, 0x75c7, // 0x8fc0 - 0x8fc7
		0x7701, 0x785d, 0x7901, 0x7965, 0x79f0, 0x7ae0, 0x7b11, 0x7ca7, // 0x8fc8 - 0x8fcf
		0x7d39, 0x8096, 0x83d6, 0x848b, 0x8549, 0x885d, 0x88f3, 0x8a1f, // 0x8fd0 - 0x8fd7
		0x8a3c, 0x8a54, 0x8a73, 0x8c61, 0x8cde, 0x91a4, 0x9266, 0x937e, // 0x8fd8 - 0x8fdf
		0x9418, 0x969c, 0x9798, 0x4e0a, 0x4e08, 0x4e1e, 0x4e57, 0x5197, // 0x8fe0 - 0x8fe7
		0x5270, 0x57ce, 0x5834, 0x58cc, 0x5b22, 0x5e38, 0x60c5, 0x64fe, // 0x8fe8 - 0x8fef
		0x6761, 0x6756, 0x6d44, 0x72b6, 0x7573, 0x7a63, 0x84b8, 0x8b72, // 0x8ff0 - 0x8ff7
		0x91b8, 0x9320, 0x5631, 0x57f4, 0x98fe, 0x0000, 0x0000, 0x0000, // 0x8ff8 - 0x8fff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9000 - 0x9007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9008 - 0x900f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9010 - 0x9017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9018 - 0x901f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9020 - 0x9027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9028 - 0x902f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9030 - 0x9037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9038 - 0x903f
		0x62ed, 0x690d, 0x6b96, 0x71ed, 0x7e54, 0x8077, 0x8272, 0x89e6, // 0x9040 - 0x9047
		0x98df, 0x8755, 0x8fb1, 0x5c3b, 0x4f38, 0x4fe1, 0x4fb5, 0x5507, // 0x9048 - 0x904f
		0x5a20, 0x5bdd, 0x5be9, 0x5fc3, 0x614e, 0x632f, 0x65b0, 0x664b, // 0x9050 - 0x9057
		0x68ee, 0x699b, 0x6d78, 0x6df1, 0x7533, 0x75b9, 0x771f, 0x795e, // 0x9058 - 0x905f
		0x79e6, 0x7d33, 0x81e3, 0x82af, 0x85aa, 0x89aa, 0x8a3a, 0x8eab, // 0x9060 - 0x9067
		0x8f9b, 0x9032, 0x91dd, 0x9707, 0x4eba, 0x4ec1, 0x5203, 0x5875, // 0x9068 - 0x906f
		0x58ec, 0x5c0b, 0x751a, 0x5c3d, 0x814e, 0x8a0a, 0x8fc5, 0x9663, // 0x9070 - 0x9077
		0x976d, 0x7b25, 0x8acf, 0x9808, 0x9162, 0x56f3, 0x53a8, 0x0000, // 0x9078 - 0x907f
		0x9017, 0x5439, 0x5782, 0x5e25, 0x63a8, 0x6c34, 0x708a, 0x7761, // 0x9080 - 0x9087
		0x7c8b, 0x7fe0, 0x8870, 0x9042, 0x9154, 0x9310, 0x9318, 0x968f, // 0x9088 - 0x908f
		0x745e, 0x9ac4, 0x5d07, 0x5d69, 0x6570, 0x67a2, 0x8da8, 0x96db, // 0x9090 - 0x9097
		0x636e, 0x6749, 0x6919, 0x83c5, 0x9817, 0x96c0, 0x88fe, 0x6f84, // 0x9098 - 0x909f
		0x647a, 0x5bf8, 0x4e16, 0x702c, 0x755d, 0x662f, 0x51c4, 0x5236, // 0x90a0 - 0x90a7
		0x52e2, 0x59d3, 0x5f81, 0x6027, 0x6210, 0x653f, 0x6574, 0x661f, // 0x90a8 - 0x90af
		0x6674, 0x68f2, 0x6816, 0x6b63, 0x6e05, 0x7272, 0x751f, 0x76db, // 0x90b0 - 0x90b7
		0x7cbe, 0x8056, 0x58f0, 0x88fd, 0x897f, 0x8aa0, 0x8a93, 0x8acb, // 0x90b8 - 0x90bf
		0x901d, 0x9192, 0x9752, 0x9759, 0x6589, 0x7a0e, 0x8106, 0x96bb, // 0x90c0 - 0x90c7
		0x5e2d, 0x60dc, 0x621a, 0x65a5, 0x6614, 0x6790, 0x77f3, 0x7a4d, // 0x90c8 - 0x90cf
		0x7c4d, 0x7e3e, 0x810a, 0x8cac, 0x8d64, 0x8de1, 0x8e5f, 0x78a9, // 0x90d0 - 0x90d7
		0x5207, 0x62d9, 0x63a5, 0x6442, 0x6298, 0x8a2d, 0x7a83, 0x7bc0, // 0x90d8 - 0x90df
		0x8aac, 0x96ea, 0x7d76, 0x820c, 0x8749, 0x4ed9, 0x5148, 0x5343, // 0x90e0 - 0x90e7
		0x5360, 0x5ba3, 0x5c02, 0x5c16, 0x5ddd, 0x6226, 0x6247, 0x64b0, // 0x90e8 - 0x90ef
		0x6813, 0x6834, 0x6cc9, 0x6d45, 0x6d17, 0x67d3, 0x6f5c, 0x714e, // 0x90f0 - 0x90f7
		0x717d, 0x65cb, 0x7a7f, 0x7bad, 0x7dda, 0x0000, 0x0000, 0x0000, // 0x90f8 - 0x90ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9100 - 0x9107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9108 - 0x910f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9110 - 0x9117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9118 - 0x911f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9120 - 0x9127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9128 - 0x912f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9130 - 0x9137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9138 - 0x913f
		0x7e4a, 0x7fa8, 0x817a, 0x821b, 0x8239, 0x85a6, 0x8a6e, 0x8cce, // 0x9140 - 0x9147
		0x8df5, 0x9078, 0x9077, 0x92ad, 0x9291, 0x9583, 0x9bae, 0x524d, // 0x9148 - 0x914f
		0x5584, 0x6f38, 0x7136, 0x5168, 0x7985, 0x7e55, 0x81b3, 0x7cce, // 0x9150 - 0x9157
		0x564c, 0x5851, 0x5ca8, 0x63aa, 0x66fe, 0x66fd, 0x695a, 0x72d9, // 0x9158 - 0x915f
		0x758f, 0x758e, 0x790e, 0x7956, 0x79df, 0x7c97, 0x7d20, 0x7d44, // 0x9160 - 0x9167
		0x8607, 0x8a34, 0x963b, 0x9061, 0x9f20, 0x50e7, 0x5275, 0x53cc, // 0x9168 - 0x916f
		0x53e2, 0x5009, 0x55aa, 0x58ee, 0x594f, 0x723d, 0x5b8b, 0x5c64, // 0x9170 - 0x9177
		0x531d, 0x60e3, 0x60f3, 0x635c, 0x6383, 0x633f, 0x63bb, 0x0000, // 0x9178 - 0x917f
		0x64cd, 0x65e9, 0x66f9, 0x5de3, 0x69cd, 0x69fd, 0x6f15, 0x71e5, // 0x9180 - 0x9187
		0x4e89, 0x75e9, 0x76f8, 0x7a93, 0x7cdf, 0x7dcf, 0x7d9c, 0x8061, // 0x9188 - 0x918f
		0x8349, 0x8358, 0x846c, 0x84bc, 0x85fb, 0x88c5, 0x8d70, 0x9001, // 0x9190 - 0x9197
		0x906d, 0x9397, 0x971c, 0x9a12, 0x50cf, 0x5897, 0x618e, 0x81d3, // 0x9198 - 0x919f
		0x8535, 0x8d08, 0x9020, 0x4fc3, 0x5074, 0x5247, 0x5373, 0x606f, // 0x91a0 - 0x91a7
		0x6349, 0x675f, 0x6e2c, 0x8db3, 0x901f, 0x4fd7, 0x5c5e, 0x8cca, // 0x91a8 - 0x91af
		0x65cf, 0x7d9a, 0x5352, 0x8896, 0x5176, 0x63c3, 0x5b58, 0x5b6b, // 0x91b0 - 0x91b7
		0x5c0a, 0x640d, 0x6751, 0x905c, 0x4ed6, 0x591a, 0x592a, 0x6c70, // 0x91b8 - 0x91bf
		0x8a51, 0x553e, 0x5815, 0x59a5, 0x60f0, 0x6253, 0x67c1, 0x8235, // 0x91c0 - 0x91c7
		0x6955, 0x9640, 0x99c4, 0x9a28, 0x4f53, 0x5806, 0x5bfe, 0x8010, // 0x91c8 - 0x91cf
		0x5cb1, 0x5e2f, 0x5f85, 0x6020, 0x614b, 0x6234, 0x66ff, 0x6cf0, // 0x91d0 - 0x91d7
		0x6ede, 0x80ce, 0x817f, 0x82d4, 0x888b, 0x8cb8, 0x9000, 0x902e, // 0x91d8 - 0x91df
		0x968a, 0x9edb, 0x9bdb, 0x4ee3, 0x53f0, 0x5927, 0x7b2c, 0x918d, // 0x91e0 - 0x91e7
		0x984c, 0x9df9, 0x6edd, 0x7027, 0x5353, 0x5544, 0x5b85, 0x6258, // 0x91e8 - 0x91ef
		0x629e, 0x62d3, 0x6ca2, 0x6fef, 0x7422, 0x8a17, 0x9438, 0x6fc1, // 0x91f0 - 0x91f7
		0x8afe, 0x8338, 0x51e7, 0x86f8, 0x53ea, 0x0000, 0x0000, 0x0000, // 0x91f8 - 0x91ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9200 - 0x9207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9208 - 0x920f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9210 - 0x9217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9218 - 0x921f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9220 - 0x9227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9228 - 0x922f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9230 - 0x9237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9238 - 0x923f
		0x53e9, 0x4f46, 0x9054, 0x8fb0, 0x596a, 0x8131, 0x5dfd, 0x7aea, // 0x9240 - 0x9247
		0x8fbf, 0x68da, 0x8c37, 0x72f8, 0x9c48, 0x6a3d, 0x8ab0, 0x4e39, // 0x9248 - 0x924f
		0x5358, 0x5606, 0x5766, 0x62c5, 0x63a2, 0x65e6, 0x6b4e, 0x6de1, // 0x9250 - 0x9257
		0x6e5b, 0x70ad, 0x77ed, 0x7aef, 0x7baa, 0x7dbb, 0x803d, 0x80c6, // 0x9258 - 0x925f
		0x86cb, 0x8a95, 0x935b, 0x56e3, 0x58c7, 0x5f3e, 0x65ad, 0x6696, // 0x9260 - 0x9267
		0x6a80, 0x6bb5, 0x7537, 0x8ac7, 0x5024, 0x77e5, 0x5730, 0x5f1b, // 0x9268 - 0x926f
		0x6065, 0x667a, 0x6c60, 0x75f4, 0x7a1a, 0x7f6e, 0x81f4, 0x8718, // 0x9270 - 0x9277
		0x9045, 0x99b3, 0x7bc9, 0x755c, 0x7af9, 0x7b51, 0x84c4, 0x0000, // 0x9278 - 0x927f
		0x9010, 0x79e9, 0x7a92, 0x8336, 0x5ae1, 0x7740, 0x4e2d, 0x4ef2, // 0x9280 - 0x9287
		0x5b99, 0x5fe0, 0x62bd, 0x663c, 0x67f1, 0x6ce8, 0x866b, 0x8877, // 0x9288 - 0x928f
		0x8a3b, 0x914e, 0x92f3, 0x99d0, 0x6a17, 0x7026, 0x732a, 0x82e7, // 0x9290 - 0x9297
		0x8457, 0x8caf, 0x4e01, 0x5146, 0x51cb, 0x558b, 0x5bf5, 0x5e16, // 0x9298 - 0x929f
		0x5e33, 0x5e81, 0x5f14, 0x5f35, 0x5f6b, 0x5fb4, 0x61f2, 0x6311, // 0x92a0 - 0x92a7
		0x66a2, 0x671d, 0x6f6e, 0x7252, 0x753a, 0x773a, 0x8074, 0x8139, // 0x92a8 - 0x92af
		0x8178, 0x8776, 0x8abf, 0x8adc, 0x8d85, 0x8df3, 0x929a, 0x9577, // 0x92b0 - 0x92b7
		0x9802, 0x9ce5, 0x52c5, 0x6357, 0x76f4, 0x6715, 0x6c88, 0x73cd, // 0x92b8 - 0x92bf
		0x8cc3, 0x93ae, 0x9673, 0x6d25, 0x589c, 0x690e, 0x69cc, 0x8ffd, // 0x92c0 - 0x92c7
		0x939a, 0x75db, 0x901a, 0x585a, 0x6802, 0x63b4, 0x69fb, 0x4f43, // 0x92c8 - 0x92cf
		0x6f2c, 0x67d8, 0x8fbb, 0x8526, 0x7db4, 0x9354, 0x693f, 0x6f70, // 0x92d0 - 0x92d7
		0x576a, 0x58f7, 0x5b2c, 0x7d2c, 0x722a, 0x540a, 0x91e3, 0x9db4, // 0x92d8 - 0x92df
		0x4ead, 0x4f4e, 0x505c, 0x5075, 0x5243, 0x8c9e, 0x5448, 0x5824, // 0x92e0 - 0x92e7
		0x5b9a, 0x5e1d, 0x5e95, 0x5ead, 0x5ef7, 0x5f1f, 0x608c, 0x62b5, // 0x92e8 - 0x92ef
		0x633a, 0x63d0, 0x68af, 0x6c40, 0x7887, 0x798e, 0x7a0b, 0x7de0, // 0x92f0 - 0x92f7
		0x8247, 0x8a02, 0x8ae6, 0x8e44, 0x9013, 0x0000, 0x0000, 0x0000, // 0x92f8 - 0x92ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9300 - 0x9307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9308 - 0x930f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9310 - 0x9317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9318 - 0x931f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9320 - 0x9327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9328 - 0x932f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9330 - 0x9337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9338 - 0x933f
		0x90b8, 0x912d, 0x91d8, 0x9f0e, 0x6ce5, 0x6458, 0x64e2, 0x6575, // 0x9340 - 0x9347
		0x6ef4, 0x7684, 0x7b1b, 0x9069, 0x93d1, 0x6eba, 0x54f2, 0x5fb9, // 0x9348 - 0x934f
		0x64a4, 0x8f4d, 0x8fed, 0x9244, 0x5178, 0x586b, 0x5929, 0x5c55, // 0x9350 - 0x9357
		0x5e97, 0x6dfb, 0x7e8f, 0x751c, 0x8cbc, 0x8ee2, 0x985b, 0x70b9, // 0x9358 - 0x935f
		0x4f1d, 0x6bbf, 0x6fb1, 0x7530, 0x96fb, 0x514e, 0x5410, 0x5835, // 0x9360 - 0x9367
		0x5857, 0x59ac, 0x5c60, 0x5f92, 0x6597, 0x675c, 0x6e21, 0x767b, // 0x9368 - 0x936f
		0x83df, 0x8ced, 0x9014, 0x90fd, 0x934d, 0x7825, 0x783a, 0x52aa, // 0x9370 - 0x9377
		0x5ea6, 0x571f, 0x5974, 0x6012, 0x5012, 0x515a, 0x51ac, 0x0000, // 0x9378 - 0x937f
		0x51cd, 0x5200, 0x5510, 0x5854, 0x5858, 0x5957, 0x5b95, 0x5cf6, // 0x9380 - 0x9387
		0x5d8b, 0x60bc, 0x6295, 0x642d, 0x6771, 0x6843, 0x68bc, 0x68df, // 0x9388 - 0x938f
		0x76d7, 0x6dd8, 0x6e6f, 0x6d9b, 0x706f, 0x71c8, 0x5f53, 0x75d8, // 0x9390 - 0x9397
		0x7977, 0x7b49, 0x7b54, 0x7b52, 0x7cd6, 0x7d71, 0x5230, 0x8463, // 0x9398 - 0x939f
		0x8569, 0x85e4, 0x8a0e, 0x8b04, 0x8c46, 0x8e0f, 0x9003, 0x900f, // 0x93a0 - 0x93a7
		0x9419, 0x9676, 0x982d, 0x9a30, 0x95d8, 0x50cd, 0x52d5, 0x540c, // 0x93a8 - 0x93af
		0x5802, 0x5c0e, 0x61a7, 0x649e, 0x6d1e, 0x77b3, 0x7ae5, 0x80f4, // 0x93b0 - 0x93b7
		0x8404, 0x9053, 0x9285, 0x5ce0, 0x9d07, 0x533f, 0x5f97, 0x5fb3, // 0x93b8 - 0x93bf
		0x6d9c, 0x7279, 0x7763, 0x79bf, 0x7be4, 0x6bd2, 0x72ec, 0x8aad, // 0x93c0 - 0x93c7
		0x6803, 0x6a61, 0x51f8, 0x7a81, 0x6934, 0x5c4a, 0x9cf6, 0x82eb, // 0x93c8 - 0x93cf
		0x5bc5, 0x9149, 0x701e, 0x5678, 0x5c6f, 0x60c7, 0x6566, 0x6c8c, // 0x93d0 - 0x93d7
		0x8c5a, 0x9041, 0x9813, 0x5451, 0x66c7, 0x920d, 0x5948, 0x90a3, // 0x93d8 - 0x93df
		0x5185, 0x4e4d, 0x51ea, 0x8599, 0x8b0e, 0x7058, 0x637a, 0x934b, // 0x93e0 - 0x93e7
		0x6962, 0x99b4, 0x7e04, 0x7577, 0x5357, 0x6960, 0x8edf, 0x96e3, // 0x93e8 - 0x93ef
		0x6c5d, 0x4e8c, 0x5c3c, 0x5f10, 0x8fe9, 0x5302, 0x8cd1, 0x8089, // 0x93f0 - 0x93f7
		0x8679, 0x5eff, 0x65e5, 0x4e73, 0x5165, 0x0000, 0x0000, 0x0000, // 0x93f8 - 0x93ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9400 - 0x9407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9408 - 0x940f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9410 - 0x9417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9418 - 0x941f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9420 - 0x9427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9428 - 0x942f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9430 - 0x9437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9438 - 0x943f
		0x5982, 0x5c3f, 0x97ee, 0x4efb, 0x598a, 0x5fcd, 0x8a8d, 0x6fe1, // 0x9440 - 0x9447
		0x79b0, 0x7962, 0x5be7, 0x8471, 0x732b, 0x71b1, 0x5e74, 0x5ff5, // 0x9448 - 0x944f
		0x637b, 0x649a, 0x71c3, 0x7c98, 0x4e43, 0x5efc, 0x4e4b, 0x57dc, // 0x9450 - 0x9457
		0x56a2, 0x60a9, 0x6fc3, 0x7d0d, 0x80fd, 0x8133, 0x81bf, 0x8fb2, // 0x9458 - 0x945f
		0x8997, 0x86a4, 0x5df4, 0x628a, 0x64ad, 0x8987, 0x6777, 0x6ce2, // 0x9460 - 0x9467
		0x6d3e, 0x7436, 0x7834, 0x5a46, 0x7f75, 0x82ad, 0x99ac, 0x4ff3, // 0x9468 - 0x946f
		0x5ec3, 0x62dd, 0x6392, 0x6557, 0x676f, 0x76c3, 0x724c, 0x80cc, // 0x9470 - 0x9477
		0x80ba, 0x8f29, 0x914d, 0x500d, 0x57f9, 0x5a92, 0x6885, 0x0000, // 0x9478 - 0x947f
		0x6973, 0x7164, 0x72fd, 0x8cb7, 0x58f2, 0x8ce0, 0x966a, 0x9019, // 0x9480 - 0x9487
		0x877f, 0x79e4, 0x77e7, 0x8429, 0x4f2f, 0x5265, 0x535a, 0x62cd, // 0x9488 - 0x948f
		0x67cf, 0x6cca, 0x767d, 0x7b94, 0x7c95, 0x8236, 0x8584, 0x8feb, // 0x9490 - 0x9497
		0x66dd, 0x6f20, 0x7206, 0x7e1b, 0x83ab, 0x99c1, 0x9ea6, 0x51fd, // 0x9498 - 0x949f
		0x7bb1, 0x7872, 0x7bb8, 0x8087, 0x7b48, 0x6ae8, 0x5e61, 0x808c, // 0x94a0 - 0x94a7
		0x7551, 0x7560, 0x516b, 0x9262, 0x6e8c, 0x767a, 0x9197, 0x9aea, // 0x94a8 - 0x94af
		0x4f10, 0x7f70, 0x629c, 0x7b4f, 0x95a5, 0x9ce9, 0x567a, 0x5859, // 0x94b0 - 0x94b7
		0x86e4, 0x96bc, 0x4f34, 0x5224, 0x534a, 0x53cd, 0x53db, 0x5e06, // 0x94b8 - 0x94bf
		0x642c, 0x6591, 0x677f, 0x6c3e, 0x6c4e, 0x7248, 0x72af, 0x73ed, // 0x94c0 - 0x94c7
		0x7554, 0x7e41, 0x822c, 0x85e9, 0x8ca9, 0x7bc4, 0x91c6, 0x7169, // 0x94c8 - 0x94cf
		0x9812, 0x98ef, 0x633d, 0x6669, 0x756a, 0x76e4, 0x78d0, 0x8543, // 0x94d0 - 0x94d7
		0x86ee, 0x532a, 0x5351, 0x5426, 0x5983, 0x5e87, 0x5f7c, 0x60b2, // 0x94d8 - 0x94df
		0x6249, 0x6279, 0x62ab, 0x6590, 0x6bd4, 0x6ccc, 0x75b2, 0x76ae, // 0x94e0 - 0x94e7
		0x7891, 0x79d8, 0x7dcb, 0x7f77, 0x80a5, 0x88ab, 0x8ab9, 0x8cbb, // 0x94e8 - 0x94ef
		0x907f, 0x975e, 0x98db, 0x6a0b, 0x7c38, 0x5099, 0x5c3e, 0x5fae, // 0x94f0 - 0x94f7
		0x6787, 0x6bd8, 0x7435, 0x7709, 0x7f8e, 0x0000, 0x0000, 0x0000, // 0x94f8 - 0x94ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9500 - 0x9507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9508 - 0x950f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9510 - 0x9517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9518 - 0x951f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9520 - 0x9527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9528 - 0x952f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9530 - 0x9537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9538 - 0x953f
		0x9f3b, 0x67ca, 0x7a17, 0x5339, 0x758b, 0x9aed, 0x5f66, 0x819d, // 0x9540 - 0x9547
		0x83f1, 0x8098, 0x5f3c, 0x5fc5, 0x7562, 0x7b46, 0x903c, 0x6867, // 0x9548 - 0x954f
		0x59eb, 0x5a9b, 0x7d10, 0x767e, 0x8b2c, 0x4ff5, 0x5f6a, 0x6a19, // 0x9550 - 0x9557
		0x6c37, 0x6f02, 0x74e2, 0x7968, 0x8868, 0x8a55, 0x8c79, 0x5edf, // 0x9558 - 0x955f
		0x63cf, 0x75c5, 0x79d2, 0x82d7, 0x9328, 0x92f2, 0x849c, 0x86ed, // 0x9560 - 0x9567
		0x9c2d, 0x54c1, 0x5f6c, 0x658c, 0x6d5c, 0x7015, 0x8ca7, 0x8cd3, // 0x9568 - 0x956f
		0x983b, 0x654f, 0x74f6, 0x4e0d, 0x4ed8, 0x57e0, 0x592b, 0x5a66, // 0x9570 - 0x9577
		0x5bcc, 0x51a8, 0x5e03, 0x5e9c, 0x6016, 0x6276, 0x6577, 0x0000, // 0x9578 - 0x957f
		0x65a7, 0x666e, 0x6d6e, 0x7236, 0x7b26, 0x8150, 0x819a, 0x8299, // 0x9580 - 0x9587
		0x8b5c, 0x8ca0, 0x8ce6, 0x8d74, 0x961c, 0x9644, 0x4fae, 0x64ab, // 0x9588 - 0x958f
		0x6b66, 0x821e, 0x8461, 0x856a, 0x90e8, 0x5c01, 0x6953, 0x98a8, // 0x9590 - 0x9597
		0x847a, 0x8557, 0x4f0f, 0x526f, 0x5fa9, 0x5e45, 0x670d, 0x798f, // 0x9598 - 0x959f
		0x8179, 0x8907, 0x8986, 0x6df5, 0x5f17, 0x6255, 0x6cb8, 0x4ecf, // 0x95a0 - 0x95a7
		0x7269, 0x9b92, 0x5206, 0x543b, 0x5674, 0x58b3, 0x61a4, 0x626e, // 0x95a8 - 0x95af
		0x711a, 0x596e, 0x7c89, 0x7cde, 0x7d1b, 0x96f0, 0x6587, 0x805e, // 0x95b0 - 0x95b7
		0x4e19, 0x4f75, 0x5175, 0x5840, 0x5e63, 0x5e73, 0x5f0a, 0x67c4, // 0x95b8 - 0x95bf
		0x4e26, 0x853d, 0x9589, 0x965b, 0x7c73, 0x9801, 0x50fb, 0x58c1, // 0x95c0 - 0x95c7
		0x7656, 0x78a7, 0x5225, 0x77a5, 0x8511, 0x7b86, 0x504f, 0x5909, // 0x95c8 - 0x95cf
		0x7247, 0x7bc7, 0x7de8, 0x8fba, 0x8fd4, 0x904d, 0x4fbf, 0x52c9, // 0x95d0 - 0x95d7
		0x5a29, 0x5f01, 0x97ad, 0x4fdd, 0x8217, 0x92ea, 0x5703, 0x6355, // 0x95d8 - 0x95df
		0x6b69, 0x752b, 0x88dc, 0x8f14, 0x7a42, 0x52df, 0x5893, 0x6155, // 0x95e0 - 0x95e7
		0x620a, 0x66ae, 0x6bcd, 0x7c3f, 0x83e9, 0x5023, 0x4ff8, 0x5305, // 0x95e8 - 0x95ef
		0x5446, 0x5831, 0x5949, 0x5b9d, 0x5cf0, 0x5cef, 0x5d29, 0x5e96, // 0x95f0 - 0x95f7
		0x62b1, 0x6367, 0x653e, 0x65b9, 0x670b, 0x0000, 0x0000, 0x0000, // 0x95f8 - 0x95ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9600 - 0x9607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9608 - 0x960f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9610 - 0x9617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9618 - 0x961f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9620 - 0x9627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9628 - 0x962f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9630 - 0x9637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9638 - 0x963f
		0x6cd5, 0x6ce1, 0x70f9, 0x7832, 0x7e2b, 0x80de, 0x82b3, 0x840c, // 0x9640 - 0x9647
		0x84ec, 0x8702, 0x8912, 0x8a2a, 0x8c4a, 0x90a6, 0x92d2, 0x98fd, // 0x9648 - 0x964f
		0x9cf3, 0x9d6c, 0x4e4f, 0x4ea1, 0x508d, 0x5256, 0x574a, 0x59a8, // 0x9650 - 0x9657
		0x5e3d, 0x5fd8, 0x5fd9, 0x623f, 0x66b4, 0x671b, 0x67d0, 0x68d2, // 0x9658 - 0x965f
		0x5192, 0x7d21, 0x80aa, 0x81a8, 0x8b00, 0x8c8c, 0x8cbf, 0x927e, // 0x9660 - 0x9667
		0x9632, 0x5420, 0x982c, 0x5317, 0x50d5, 0x535c, 0x58a8, 0x64b2, // 0x9668 - 0x966f
		0x6734, 0x7267, 0x7766, 0x7a46, 0x91e6, 0x52c3, 0x6ca1, 0x6b86, // 0x9670 - 0x9677
		0x5800, 0x5e4c, 0x5954, 0x672c, 0x7ffb, 0x51e1, 0x76c6, 0x0000, // 0x9678 - 0x967f
		0x6469, 0x78e8, 0x9b54, 0x9ebb, 0x57cb, 0x59b9, 0x6627, 0x679a, // 0x9680 - 0x9687
		0x6bce, 0x54e9, 0x69d9, 0x5e55, 0x819c, 0x6795, 0x9baa, 0x67fe, // 0x9688 - 0x968f
		0x9c52, 0x685d, 0x4ea6, 0x4fe3, 0x53c8, 0x62b9, 0x672b, 0x6cab, // 0x9690 - 0x9697
		0x8fc4, 0x4fad, 0x7e6d, 0x9ebf, 0x4e07, 0x6162, 0x6e80, 0x6f2b, // 0x9698 - 0x969f
		0x8513, 0x5473, 0x672a, 0x9b45, 0x5df3, 0x7b95, 0x5cac, 0x5bc6, // 0x96a0 - 0x96a7
		0x871c, 0x6e4a, 0x84d1, 0x7a14, 0x8108, 0x5999, 0x7c8d, 0x6c11, // 0x96a8 - 0x96af
		0x7720, 0x52d9, 0x5922, 0x7121, 0x725f, 0x77db, 0x9727, 0x9d61, // 0x96b0 - 0x96b7
		0x690b, 0x5a7f, 0x5a18, 0x51a5, 0x540d, 0x547d, 0x660e, 0x76df, // 0x96b8 - 0x96bf
		0x8ff7, 0x9298, 0x9cf4, 0x59ea, 0x725d, 0x6ec5, 0x514d, 0x68c9, // 0x96c0 - 0x96c7
		0x7dbf, 0x7dec, 0x9762, 0x9eba, 0x6478, 0x6a21, 0x8302, 0x5984, // 0x96c8 - 0x96cf
		0x5b5f, 0x6bdb, 0x731b, 0x76f2, 0x7db2, 0x8017, 0x8499, 0x5132, // 0x96d0 - 0x96d7
		0x6728, 0x9ed9, 0x76ee, 0x6762, 0x52ff, 0x9905, 0x5c24, 0x623b, // 0x96d8 - 0x96df
		0x7c7e, 0x8cb0, 0x554f, 0x60b6, 0x7d0b, 0x9580, 0x5301, 0x4e5f, // 0x96e0 - 0x96e7
		0x51b6, 0x591c, 0x723a, 0x8036, 0x91ce, 0x5f25, 0x77e2, 0x5384, // 0x96e8 - 0x96ef
		0x5f79, 0x7d04, 0x85ac, 0x8a33, 0x8e8d, 0x9756, 0x67f3, 0x85ae, // 0x96f0 - 0x96f7
		0x9453, 0x6109, 0x6108, 0x6cb9, 0x7652, 0x0000, 0x0000, 0x0000, // 0x96f8 - 0x96ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9700 - 0x9707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9708 - 0x970f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9710 - 0x9717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9718 - 0x971f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9720 - 0x9727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9728 - 0x972f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9730 - 0x9737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9738 - 0x973f
		0x8aed, 0x8f38, 0x552f, 0x4f51, 0x512a, 0x52c7, 0x53cb, 0x5ba5, // 0x9740 - 0x9747
		0x5e7d, 0x60a0, 0x6182, 0x63d6, 0x6709, 0x67da, 0x6e67, 0x6d8c, // 0x9748 - 0x974f
		0x7336, 0x7337, 0x7531, 0x7950, 0x88d5, 0x8a98, 0x904a, 0x9091, // 0x9750 - 0x9757
		0x90f5, 0x96c4, 0x878d, 0x5915, 0x4e88, 0x4f59, 0x4e0e, 0x8a89, // 0x9758 - 0x975f
		0x8f3f, 0x9810, 0x50ad, 0x5e7c, 0x5996, 0x5bb9, 0x5eb8, 0x63da, // 0x9760 - 0x9767
		0x63fa, 0x64c1, 0x66dc, 0x694a, 0x69d8, 0x6d0b, 0x6eb6, 0x7194, // 0x9768 - 0x976f
		0x7528, 0x7aaf, 0x7f8a, 0x8000, 0x8449, 0x84c9, 0x8981, 0x8b21, // 0x9770 - 0x9777
		0x8e0a, 0x9065, 0x967d, 0x990a, 0x617e, 0x6291, 0x6b32, 0x0000, // 0x9778 - 0x977f
		0x6c83, 0x6d74, 0x7fcc, 0x7ffc, 0x6dc0, 0x7f85, 0x87ba, 0x88f8, // 0x9780 - 0x9787
		0x6765, 0x83b1, 0x983c, 0x96f7, 0x6d1b, 0x7d61, 0x843d, 0x916a, // 0x9788 - 0x978f
		0x4e71, 0x5375, 0x5d50, 0x6b04, 0x6feb, 0x85cd, 0x862d, 0x89a7, // 0x9790 - 0x9797
		0x5229, 0x540f, 0x5c65, 0x674e, 0x68a8, 0x7406, 0x7483, 0x75e2, // 0x9798 - 0x979f
		0x88cf, 0x88e1, 0x91cc, 0x96e2, 0x9678, 0x5f8b, 0x7387, 0x7acb, // 0x97a0 - 0x97a7
		0x844e, 0x63a0, 0x7565, 0x5289, 0x6d41, 0x6e9c, 0x7409, 0x7559, // 0x97a8 - 0x97af
		0x786b, 0x7c92, 0x9686, 0x7adc, 0x9f8d, 0x4fb6, 0x616e, 0x65c5, // 0x97b0 - 0x97b7
		0x865c, 0x4e86, 0x4eae, 0x50da, 0x4e21, 0x51cc, 0x5bee, 0x6599, // 0x97b8 - 0x97bf
		0x6881, 0x6dbc, 0x731f, 0x7642, 0x77ad, 0x7a1c, 0x7ce7, 0x826f, // 0x97c0 - 0x97c7
		0x8ad2, 0x907c, 0x91cf, 0x9675, 0x9818, 0x529b, 0x7dd1, 0x502b, // 0x97c8 - 0x97cf
		0x5398, 0x6797, 0x6dcb, 0x71d0, 0x7433, 0x81e8, 0x8f2a, 0x96a3, // 0x97d0 - 0x97d7
		0x9c57, 0x9e9f, 0x7460, 0x5841, 0x6d99, 0x7d2f, 0x985e, 0x4ee4, // 0x97d8 - 0x97df
		0x4f36, 0x4f8b, 0x51b7, 0x52b1, 0x5dba, 0x601c, 0x73b2, 0x793c, // 0x97e0 - 0x97e7
		0x82d3, 0x9234, 0x96b7, 0x96f6, 0x970a, 0x9e97, 0x9f62, 0x66a6, // 0x97e8 - 0x97ef
		0x6b74, 0x5217, 0x52a3, 0x70c8, 0x88c2, 0x5ec9, 0x604b, 0x6190, // 0x97f0 - 0x97f7
		0x6f23, 0x7149, 0x7c3e, 0x7df4, 0x806f, 0x0000, 0x0000, 0x0000, // 0x97f8 - 0x97ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9800 - 0x9807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9808 - 0x980f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9810 - 0x9817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9818 - 0x981f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9820 - 0x9827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9828 - 0x982f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9830 - 0x9837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9838 - 0x983f
		0x84ee, 0x9023, 0x932c, 0x5442, 0x9b6f, 0x6ad3, 0x7089, 0x8cc2, // 0x9840 - 0x9847
		0x8def, 0x9732, 0x52b4, 0x5a41, 0x5eca, 0x5f04, 0x6717, 0x697c, // 0x9848 - 0x984f
		0x6994, 0x6d6a, 0x6f0f, 0x7262, 0x72fc, 0x7bed, 0x8001, 0x807e, // 0x9850 - 0x9857
		0x874b, 0x90ce, 0x516d, 0x9e93, 0x7984, 0x808b, 0x9332, 0x8ad6, // 0x9858 - 0x985f
		0x502d, 0x548c, 0x8a71, 0x6b6a, 0x8cc4, 0x8107, 0x60d1, 0x67a0, // 0x9860 - 0x9867
		0x9df2, 0x4e99, 0x4e98, 0x9c10, 0x8a6b, 0x85c1, 0x8568, 0x6900, // 0x9868 - 0x986f
		0x6e7e, 0x7897, 0x8155, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9870 - 0x9877
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9878 - 0x987f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9880 - 0x9887
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9888 - 0x988f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9890 - 0x9897
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f0c, // 0x9898 - 0x989f
		0x4e10, 0x4e15, 0x4e2a, 0x4e31, 0x4e36, 0x4e3c, 0x4e3f, 0x4e42, // 0x98a0 - 0x98a7
		0x4e56, 0x4e58, 0x4e82, 0x4e85, 0x8c6b, 0x4e8a, 0x8212, 0x5f0d, // 0x98a8 - 0x98af
		0x4e8e, 0x4e9e, 0x4e9f, 0x4ea0, 0x4ea2, 0x4eb0, 0x4eb3, 0x4eb6, // 0x98b0 - 0x98b7
		0x4ece, 0x4ecd, 0x4ec4, 0x4ec6, 0x4ec2, 0x4ed7, 0x4ede, 0x4eed, // 0x98b8 - 0x98bf
		0x4edf, 0x4ef7, 0x4f09, 0x4f5a, 0x4f30, 0x4f5b, 0x4f5d, 0x4f57, // 0x98c0 - 0x98c7
		0x4f47, 0x4f76, 0x4f88, 0x4f8f, 0x4f98, 0x4f7b, 0x4f69, 0x4f70, // 0x98c8 - 0x98cf
		0x4f91, 0x4f6f, 0x4f86, 0x4f96, 0x5118, 0x4fd4, 0x4fdf, 0x4fce, // 0x98d0 - 0x98d7
		0x4fd8, 0x4fdb, 0x4fd1, 0x4fda, 0x4fd0, 0x4fe4, 0x4fe5, 0x501a, // 0x98d8 - 0x98df
		0x5028, 0x5014, 0x502a, 0x5025, 0x5005, 0x4f1c, 0x4ff6, 0x5021, // 0x98e0 - 0x98e7
		0x5029, 0x502c, 0x4ffe, 0x4fef, 0x5011, 0x5006, 0x5043, 0x5047, // 0x98e8 - 0x98ef
		0x6703, 0x5055, 0x5050, 0x5048, 0x505a, 0x5056, 0x506c, 0x5078, // 0x98f0 - 0x98f7
		0x5080, 0x509a, 0x5085, 0x50b4, 0x50b2, 0x0000, 0x0000, 0x0000, // 0x98f8 - 0x98ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9900 - 0x9907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9908 - 0x990f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9910 - 0x9917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9918 - 0x991f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9920 - 0x9927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9928 - 0x992f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9930 - 0x9937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9938 - 0x993f
		0x50c9, 0x50ca, 0x50b3, 0x50c2, 0x50d6, 0x50de, 0x50e5, 0x50ed, // 0x9940 - 0x9947
		0x50e3, 0x50ee, 0x50f9, 0x50f5, 0x5109, 0x5101, 0x5102, 0x5116, // 0x9948 - 0x994f
		0x5115, 0x5114, 0x511a, 0x5121, 0x513a, 0x5137, 0x513c, 0x513b, // 0x9950 - 0x9957
		0x513f, 0x5140, 0x5152, 0x514c, 0x5154, 0x5162, 0x7af8, 0x5169, // 0x9958 - 0x995f
		0x516a, 0x516e, 0x5180, 0x5182, 0x56d8, 0x518c, 0x5189, 0x518f, // 0x9960 - 0x9967
		0x5191, 0x5193, 0x5195, 0x5196, 0x51a4, 0x51a6, 0x51a2, 0x51a9, // 0x9968 - 0x996f
		0x51aa, 0x51ab, 0x51b3, 0x51b1, 0x51b2, 0x51b0, 0x51b5, 0x51bd, // 0x9970 - 0x9977
		0x51c5, 0x51c9, 0x51db, 0x51e0, 0x8655, 0x51e9, 0x51ed, 0x0000, // 0x9978 - 0x997f
		0x51f0, 0x51f5, 0x51fe, 0x5204, 0x520b, 0x5214, 0x520e, 0x5227, // 0x9980 - 0x9987
		0x522a, 0x522e, 0x5233, 0x5239, 0x524f, 0x5244, 0x524b, 0x524c, // 0x9988 - 0x998f
		0x525e, 0x5254, 0x526a, 0x5274, 0x5269, 0x5273, 0x527f, 0x527d, // 0x9990 - 0x9997
		0x528d, 0x5294, 0x5292, 0x5271, 0x5288, 0x5291, 0x8fa8, 0x8fa7, // 0x9998 - 0x999f
		0x52ac, 0x52ad, 0x52bc, 0x52b5, 0x52c1, 0x52cd, 0x52d7, 0x52de, // 0x99a0 - 0x99a7
		0x52e3, 0x52e6, 0x98ed, 0x52e0, 0x52f3, 0x52f5, 0x52f8, 0x52f9, // 0x99a8 - 0x99af
		0x5306, 0x5308, 0x7538, 0x530d, 0x5310, 0x530f, 0x5315, 0x531a, // 0x99b0 - 0x99b7
		0x5323, 0x532f, 0x5331, 0x5333, 0x5338, 0x5340, 0x5346, 0x5345, // 0x99b8 - 0x99bf
		0x4e17, 0x5349, 0x534d, 0x51d6, 0x535e, 0x5369, 0x536e, 0x5918, // 0x99c0 - 0x99c7
		0x537b, 0x5377, 0x5382, 0x5396, 0x53a0, 0x53a6, 0x53a5, 0x53ae, // 0x99c8 - 0x99cf
		0x53b0, 0x53b6, 0x53c3, 0x7c12, 0x96d9, 0x53df, 0x66fc, 0x71ee, // 0x99d0 - 0x99d7
		0x53ee, 0x53e8, 0x53ed, 0x53fa, 0x5401, 0x543d, 0x5440, 0x542c, // 0x99d8 - 0x99df
		0x542d, 0x543c, 0x542e, 0x5436, 0x5429, 0x541d, 0x544e, 0x548f, // 0x99e0 - 0x99e7
		0x5475, 0x548e, 0x545f, 0x5471, 0x5477, 0x5470, 0x5492, 0x547b, // 0x99e8 - 0x99ef
		0x5480, 0x5476, 0x5484, 0x5490, 0x5486, 0x54c7, 0x54a2, 0x54b8, // 0x99f0 - 0x99f7
		0x54a5, 0x54ac, 0x54c4, 0x54c8, 0x54a8, 0x0000, 0x0000, 0x0000, // 0x99f8 - 0x99ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a00 - 0x9a07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a08 - 0x9a0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a10 - 0x9a17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a18 - 0x9a1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a20 - 0x9a27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a28 - 0x9a2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a30 - 0x9a37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9a38 - 0x9a3f
		0x54ab, 0x54c2, 0x54a4, 0x54be, 0x54bc, 0x54d8, 0x54e5, 0x54e6, // 0x9a40 - 0x9a47
		0x550f, 0x5514, 0x54fd, 0x54ee, 0x54ed, 0x54fa, 0x54e2, 0x5539, // 0x9a48 - 0x9a4f
		0x5540, 0x5563, 0x554c, 0x552e, 0x555c, 0x5545, 0x5556, 0x5557, // 0x9a50 - 0x9a57
		0x5538, 0x5533, 0x555d, 0x5599, 0x5580, 0x54af, 0x558a, 0x559f, // 0x9a58 - 0x9a5f
		0x557b, 0x557e, 0x5598, 0x559e, 0x55ae, 0x557c, 0x5583, 0x55a9, // 0x9a60 - 0x9a67
		0x5587, 0x55a8, 0x55da, 0x55c5, 0x55df, 0x55c4, 0x55dc, 0x55e4, // 0x9a68 - 0x9a6f
		0x55d4, 0x5614, 0x55f7, 0x5616, 0x55fe, 0x55fd, 0x561b, 0x55f9, // 0x9a70 - 0x9a77
		0x564e, 0x5650, 0x71df, 0x5634, 0x5636, 0x5632, 0x5638, 0x0000, // 0x9a78 - 0x9a7f
		0x566b, 0x5664, 0x562f, 0x566c, 0x566a, 0x5686, 0x5680, 0x568a, // 0x9a80 - 0x9a87
		0x56a0, 0x5694, 0x568f, 0x56a5, 0x56ae, 0x56b6, 0x56b4, 0x56c2, // 0x9a88 - 0x9a8f
		0x56bc, 0x56c1, 0x56c3, 0x56c0, 0x56c8, 0x56ce, 0x56d1, 0x56d3, // 0x9a90 - 0x9a97
		0x56d7, 0x56ee, 0x56f9, 0x5700, 0x56ff, 0x5704, 0x5709, 0x5708, // 0x9a98 - 0x9a9f
		0x570b, 0x570d, 0x5713, 0x5718, 0x5716, 0x55c7, 0x571c, 0x5726, // 0x9aa0 - 0x9aa7
		0x5737, 0x5738, 0x574e, 0x573b, 0x5740, 0x574f, 0x5769, 0x57c0, // 0x9aa8 - 0x9aaf
		0x5788, 0x5761, 0x577f, 0x5789, 0x5793, 0x57a0, 0x57b3, 0x57a4, // 0x9ab0 - 0x9ab7
		0x57aa, 0x57b0, 0x57c3, 0x57c6, 0x57d4, 0x57d2, 0x57d3, 0x580a, // 0x9ab8 - 0x9abf
		0x57d6, 0x57e3, 0x580b, 0x5819, 0x581d, 0x5872, 0x5821, 0x5862, // 0x9ac0 - 0x9ac7
		0x584b, 0x5870, 0x6bc0, 0x5852, 0x583d, 0x5879, 0x5885, 0x58b9, // 0x9ac8 - 0x9acf
		0x589f, 0x58ab, 0x58ba, 0x58de, 0x58bb, 0x58b8, 0x58ae, 0x58c5, // 0x9ad0 - 0x9ad7
		0x58d3, 0x58d1, 0x58d7, 0x58d9, 0x58d8, 0x58e5, 0x58dc, 0x58e4, // 0x9ad8 - 0x9adf
		0x58df, 0x58ef, 0x58fa, 0x58f9, 0x58fb, 0x58fc, 0x58fd, 0x5902, // 0x9ae0 - 0x9ae7
		0x590a, 0x5910, 0x591b, 0x68a6, 0x5925, 0x592c, 0x592d, 0x5932, // 0x9ae8 - 0x9aef
		0x5938, 0x593e, 0x7ad2, 0x5955, 0x5950, 0x594e, 0x595a, 0x5958, // 0x9af0 - 0x9af7
		0x5962, 0x5960, 0x5967, 0x596c, 0x5969, 0x0000, 0x0000, 0x0000, // 0x9af8 - 0x9aff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b00 - 0x9b07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b08 - 0x9b0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b10 - 0x9b17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b18 - 0x9b1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b20 - 0x9b27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b28 - 0x9b2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b30 - 0x9b37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9b38 - 0x9b3f
		0x5978, 0x5981, 0x599d, 0x4f5e, 0x4fab, 0x59a3, 0x59b2, 0x59c6, // 0x9b40 - 0x9b47
		0x59e8, 0x59dc, 0x598d, 0x59d9, 0x59da, 0x5a25, 0x5a1f, 0x5a11, // 0x9b48 - 0x9b4f
		0x5a1c, 0x5a09, 0x5a1a, 0x5a40, 0x5a6c, 0x5a49, 0x5a35, 0x5a36, // 0x9b50 - 0x9b57
		0x5a62, 0x5a6a, 0x5a9a, 0x5abc, 0x5abe, 0x5acb, 0x5ac2, 0x5abd, // 0x9b58 - 0x9b5f
		0x5ae3, 0x5ad7, 0x5ae6, 0x5ae9, 0x5ad6, 0x5afa, 0x5afb, 0x5b0c, // 0x9b60 - 0x9b67
		0x5b0b, 0x5b16, 0x5b32, 0x5ad0, 0x5b2a, 0x5b36, 0x5b3e, 0x5b43, // 0x9b68 - 0x9b6f
		0x5b45, 0x5b40, 0x5b51, 0x5b55, 0x5b5a, 0x5b5b, 0x5b65, 0x5b69, // 0x9b70 - 0x9b77
		0x5b70, 0x5b73, 0x5b75, 0x5b78, 0x6588, 0x5b7a, 0x5b80, 0x0000, // 0x9b78 - 0x9b7f
		0x5b83, 0x5ba6, 0x5bb8, 0x5bc3, 0x5bc7, 0x5bc9, 0x5bd4, 0x5bd0, // 0x9b80 - 0x9b87
		0x5be4, 0x5be6, 0x5be2, 0x5bde, 0x5be5, 0x5beb, 0x5bf0, 0x5bf6, // 0x9b88 - 0x9b8f
		0x5bf3, 0x5c05, 0x5c07, 0x5c08, 0x5c0d, 0x5c13, 0x5c20, 0x5c22, // 0x9b90 - 0x9b97
		0x5c28, 0x5c38, 0x5c39, 0x5c41, 0x5c46, 0x5c4e, 0x5c53, 0x5c50, // 0x9b98 - 0x9b9f
		0x5c4f, 0x5b71, 0x5c6c, 0x5c6e, 0x4e62, 0x5c76, 0x5c79, 0x5c8c, // 0x9ba0 - 0x9ba7
		0x5c91, 0x5c94, 0x599b, 0x5cab, 0x5cbb, 0x5cb6, 0x5cbc, 0x5cb7, // 0x9ba8 - 0x9baf
		0x5cc5, 0x5cbe, 0x5cc7, 0x5cd9, 0x5ce9, 0x5cfd, 0x5cfa, 0x5ced, // 0x9bb0 - 0x9bb7
		0x5d8c, 0x5cea, 0x5d0b, 0x5d15, 0x5d17, 0x5d5c, 0x5d1f, 0x5d1b, // 0x9bb8 - 0x9bbf
		0x5d11, 0x5d14, 0x5d22, 0x5d1a, 0x5d19, 0x5d18, 0x5d4c, 0x5d52, // 0x9bc0 - 0x9bc7
		0x5d4e, 0x5d4b, 0x5d6c, 0x5d73, 0x5d76, 0x5d87, 0x5d84, 0x5d82, // 0x9bc8 - 0x9bcf
		0x5da2, 0x5d9d, 0x5dac, 0x5dae, 0x5dbd, 0x5d90, 0x5db7, 0x5dbc, // 0x9bd0 - 0x9bd7
		0x5dc9, 0x5dcd, 0x5dd3, 0x5dd2, 0x5dd6, 0x5ddb, 0x5deb, 0x5df2, // 0x9bd8 - 0x9bdf
		0x5df5, 0x5e0b, 0x5e1a, 0x5e19, 0x5e11, 0x5e1b, 0x5e36, 0x5e37, // 0x9be0 - 0x9be7
		0x5e44, 0x5e43, 0x5e40, 0x5e4e, 0x5e57, 0x5e54, 0x5e5f, 0x5e62, // 0x9be8 - 0x9bef
		0x5e64, 0x5e47, 0x5e75, 0x5e76, 0x5e7a, 0x9ebc, 0x5e7f, 0x5ea0, // 0x9bf0 - 0x9bf7
		0x5ec1, 0x5ec2, 0x5ec8, 0x5ed0, 0x5ecf, 0x0000, 0x0000, 0x0000, // 0x9bf8 - 0x9bff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c00 - 0x9c07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c08 - 0x9c0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c10 - 0x9c17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c18 - 0x9c1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c20 - 0x9c27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c28 - 0x9c2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c30 - 0x9c37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9c38 - 0x9c3f
		0x5ed6, 0x5ee3, 0x5edd, 0x5eda, 0x5edb, 0x5ee2, 0x5ee1, 0x5ee8, // 0x9c40 - 0x9c47
		0x5ee9, 0x5eec, 0x5ef1, 0x5ef3, 0x5ef0, 0x5ef4, 0x5ef8, 0x5efe, // 0x9c48 - 0x9c4f
		0x5f03, 0x5f09, 0x5f5d, 0x5f5c, 0x5f0b, 0x5f11, 0x5f16, 0x5f29, // 0x9c50 - 0x9c57
		0x5f2d, 0x5f38, 0x5f41, 0x5f48, 0x5f4c, 0x5f4e, 0x5f2f, 0x5f51, // 0x9c58 - 0x9c5f
		0x5f56, 0x5f57, 0x5f59, 0x5f61, 0x5f6d, 0x5f73, 0x5f77, 0x5f83, // 0x9c60 - 0x9c67
		0x5f82, 0x5f7f, 0x5f8a, 0x5f88, 0x5f91, 0x5f87, 0x5f9e, 0x5f99, // 0x9c68 - 0x9c6f
		0x5f98, 0x5fa0, 0x5fa8, 0x5fad, 0x5fbc, 0x5fd6, 0x5ffb, 0x5fe4, // 0x9c70 - 0x9c77
		0x5ff8, 0x5ff1, 0x5fdd, 0x60b3, 0x5fff, 0x6021, 0x6060, 0x0000, // 0x9c78 - 0x9c7f
		0x6019, 0x6010, 0x6029, 0x600e, 0x6031, 0x601b, 0x6015, 0x602b, // 0x9c80 - 0x9c87
		0x6026, 0x600f, 0x603a, 0x605a, 0x6041, 0x606a, 0x6077, 0x605f, // 0x9c88 - 0x9c8f
		0x604a, 0x6046, 0x604d, 0x6063, 0x6043, 0x6064, 0x6042, 0x606c, // 0x9c90 - 0x9c97
		0x606b, 0x6059, 0x6081, 0x608d, 0x60e7, 0x6083, 0x609a, 0x6084, // 0x9c98 - 0x9c9f
		0x609b, 0x6096, 0x6097, 0x6092, 0x60a7, 0x608b, 0x60e1, 0x60b8, // 0x9ca0 - 0x9ca7
		0x60e0, 0x60d3, 0x60b4, 0x5ff0, 0x60bd, 0x60c6, 0x60b5, 0x60d8, // 0x9ca8 - 0x9caf
		0x614d, 0x6115, 0x6106, 0x60f6, 0x60f7, 0x6100, 0x60f4, 0x60fa, // 0x9cb0 - 0x9cb7
		0x6103, 0x6121, 0x60fb, 0x60f1, 0x610d, 0x610e, 0x6147, 0x613e, // 0x9cb8 - 0x9cbf
		0x6128, 0x6127, 0x614a, 0x613f, 0x613c, 0x612c, 0x6134, 0x613d, // 0x9cc0 - 0x9cc7
		0x6142, 0x6144, 0x6173, 0x6177, 0x6158, 0x6159, 0x615a, 0x616b, // 0x9cc8 - 0x9ccf
		0x6174, 0x616f, 0x6165, 0x6171, 0x615f, 0x615d, 0x6153, 0x6175, // 0x9cd0 - 0x9cd7
		0x6199, 0x6196, 0x6187, 0x61ac, 0x6194, 0x619a, 0x618a, 0x6191, // 0x9cd8 - 0x9cdf
		0x61ab, 0x61ae, 0x61cc, 0x61ca, 0x61c9, 0x61f7, 0x61c8, 0x61c3, // 0x9ce0 - 0x9ce7
		0x61c6, 0x61ba, 0x61cb, 0x7f79, 0x61cd, 0x61e6, 0x61e3, 0x61f6, // 0x9ce8 - 0x9cef
		0x61fa, 0x61f4, 0x61ff, 0x61fd, 0x61fc, 0x61fe, 0x6200, 0x6208, // 0x9cf0 - 0x9cf7
		0x6209, 0x620d, 0x620c, 0x6214, 0x621b, 0x0000, 0x0000, 0x0000, // 0x9cf8 - 0x9cff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d00 - 0x9d07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d08 - 0x9d0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d10 - 0x9d17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d18 - 0x9d1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d20 - 0x9d27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d28 - 0x9d2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d30 - 0x9d37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9d38 - 0x9d3f
		0x621e, 0x6221, 0x622a, 0x622e, 0x6230, 0x6232, 0x6233, 0x6241, // 0x9d40 - 0x9d47
		0x624e, 0x625e, 0x6263, 0x625b, 0x6260, 0x6268, 0x627c, 0x6282, // 0x9d48 - 0x9d4f
		0x6289, 0x627e, 0x6292, 0x6293, 0x6296, 0x62d4, 0x6283, 0x6294, // 0x9d50 - 0x9d57
		0x62d7, 0x62d1, 0x62bb, 0x62cf, 0x62ff, 0x62c6, 0x64d4, 0x62c8, // 0x9d58 - 0x9d5f
		0x62dc, 0x62cc, 0x62ca, 0x62c2, 0x62c7, 0x629b, 0x62c9, 0x630c, // 0x9d60 - 0x9d67
		0x62ee, 0x62f1, 0x6327, 0x6302, 0x6308, 0x62ef, 0x62f5, 0x6350, // 0x9d68 - 0x9d6f
		0x633e, 0x634d, 0x641c, 0x634f, 0x6396, 0x638e, 0x6380, 0x63ab, // 0x9d70 - 0x9d77
		0x6376, 0x63a3, 0x638f, 0x6389, 0x639f, 0x63b5, 0x636b, 0x0000, // 0x9d78 - 0x9d7f
		0x6369, 0x63be, 0x63e9, 0x63c0, 0x63c6, 0x63e3, 0x63c9, 0x63d2, // 0x9d80 - 0x9d87
		0x63f6, 0x63c4, 0x6416, 0x6434, 0x6406, 0x6413, 0x6426, 0x6436, // 0x9d88 - 0x9d8f
		0x651d, 0x6417, 0x6428, 0x640f, 0x6467, 0x646f, 0x6476, 0x644e, // 0x9d90 - 0x9d97
		0x652a, 0x6495, 0x6493, 0x64a5, 0x64a9, 0x6488, 0x64bc, 0x64da, // 0x9d98 - 0x9d9f
		0x64d2, 0x64c5, 0x64c7, 0x64bb, 0x64d8, 0x64c2, 0x64f1, 0x64e7, // 0x9da0 - 0x9da7
		0x8209, 0x64e0, 0x64e1, 0x62ac, 0x64e3, 0x64ef, 0x652c, 0x64f6, // 0x9da8 - 0x9daf
		0x64f4, 0x64f2, 0x64fa, 0x6500, 0x64fd, 0x6518, 0x651c, 0x6505, // 0x9db0 - 0x9db7
		0x6524, 0x6523, 0x652b, 0x6534, 0x6535, 0x6537, 0x6536, 0x6538, // 0x9db8 - 0x9dbf
		0x754b, 0x6548, 0x6556, 0x6555, 0x654d, 0x6558, 0x655e, 0x655d, // 0x9dc0 - 0x9dc7
		0x6572, 0x6578, 0x6582, 0x6583, 0x8b8a, 0x659b, 0x659f, 0x65ab, // 0x9dc8 - 0x9dcf
		0x65b7, 0x65c3, 0x65c6, 0x65c1, 0x65c4, 0x65cc, 0x65d2, 0x65db, // 0x9dd0 - 0x9dd7
		0x65d9, 0x65e0, 0x65e1, 0x65f1, 0x6772, 0x660a, 0x6603, 0x65fb, // 0x9dd8 - 0x9ddf
		0x6773, 0x6635, 0x6636, 0x6634, 0x661c, 0x664f, 0x6644, 0x6649, // 0x9de0 - 0x9de7
		0x6641, 0x665e, 0x665d, 0x6664, 0x6667, 0x6668, 0x665f, 0x6662, // 0x9de8 - 0x9def
		0x6670, 0x6683, 0x6688, 0x668e, 0x6689, 0x6684, 0x6698, 0x669d, // 0x9df0 - 0x9df7
		0x66c1, 0x66b9, 0x66c9, 0x66be, 0x66bc, 0x0000, 0x0000, 0x0000, // 0x9df8 - 0x9dff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e00 - 0x9e07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e08 - 0x9e0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e10 - 0x9e17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e18 - 0x9e1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e20 - 0x9e27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e28 - 0x9e2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e30 - 0x9e37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9e38 - 0x9e3f
		0x66c4, 0x66b8, 0x66d6, 0x66da, 0x66e0, 0x663f, 0x66e6, 0x66e9, // 0x9e40 - 0x9e47
		0x66f0, 0x66f5, 0x66f7, 0x670f, 0x6716, 0x671e, 0x6726, 0x6727, // 0x9e48 - 0x9e4f
		0x9738, 0x672e, 0x673f, 0x6736, 0x6741, 0x6738, 0x6737, 0x6746, // 0x9e50 - 0x9e57
		0x675e, 0x6760, 0x6759, 0x6763, 0x6764, 0x6789, 0x6770, 0x67a9, // 0x9e58 - 0x9e5f
		0x677c, 0x676a, 0x678c, 0x678b, 0x67a6, 0x67a1, 0x6785, 0x67b7, // 0x9e60 - 0x9e67
		0x67ef, 0x67b4, 0x67ec, 0x67b3, 0x67e9, 0x67b8, 0x67e4, 0x67de, // 0x9e68 - 0x9e6f
		0x67dd, 0x67e2, 0x67ee, 0x67b9, 0x67ce, 0x67c6, 0x67e7, 0x6a9c, // 0x9e70 - 0x9e77
		0x681e, 0x6846, 0x6829, 0x6840, 0x684d, 0x6832, 0x684e, 0x0000, // 0x9e78 - 0x9e7f
		0x68b3, 0x682b, 0x6859, 0x6863, 0x6877, 0x687f, 0x689f, 0x688f, // 0x9e80 - 0x9e87
		0x68ad, 0x6894, 0x689d, 0x689b, 0x6883, 0x6aae, 0x68b9, 0x6874, // 0x9e88 - 0x9e8f
		0x68b5, 0x68a0, 0x68ba, 0x690f, 0x688d, 0x687e, 0x6901, 0x68ca, // 0x9e90 - 0x9e97
		0x6908, 0x68d8, 0x6922, 0x6926, 0x68e1, 0x690c, 0x68cd, 0x68d4, // 0x9e98 - 0x9e9f
		0x68e7, 0x68d5, 0x6936, 0x6912, 0x6904, 0x68d7, 0x68e3, 0x6925, // 0x9ea0 - 0x9ea7
		0x68f9, 0x68e0, 0x68ef, 0x6928, 0x692a, 0x691a, 0x6923, 0x6921, // 0x9ea8 - 0x9eaf
		0x68c6, 0x6979, 0x6977, 0x695c, 0x6978, 0x696b, 0x6954, 0x697e, // 0x9eb0 - 0x9eb7
		0x696e, 0x6939, 0x6974, 0x693d, 0x6959, 0x6930, 0x6961, 0x695e, // 0x9eb8 - 0x9ebf
		0x695d, 0x6981, 0x696a, 0x69b2, 0x69ae, 0x69d0, 0x69bf, 0x69c1, // 0x9ec0 - 0x9ec7
		0x69d3, 0x69be, 0x69ce, 0x5be8, 0x69ca, 0x69dd, 0x69bb, 0x69c3, // 0x9ec8 - 0x9ecf
		0x69a7, 0x6a2e, 0x6991, 0x69a0, 0x699c, 0x6995, 0x69b4, 0x69de, // 0x9ed0 - 0x9ed7
		0x69e8, 0x6a02, 0x6a1b, 0x69ff, 0x6b0a, 0x69f9, 0x69f2, 0x69e7, // 0x9ed8 - 0x9edf
		0x6a05, 0x69b1, 0x6a1e, 0x69ed, 0x6a14, 0x69eb, 0x6a0a, 0x6a12, // 0x9ee0 - 0x9ee7
		0x6ac1, 0x6a23, 0x6a13, 0x6a44, 0x6a0c, 0x6a72, 0x6a36, 0x6a78, // 0x9ee8 - 0x9eef
		0x6a47, 0x6a62, 0x6a59, 0x6a66, 0x6a48, 0x6a38, 0x6a22, 0x6a90, // 0x9ef0 - 0x9ef7
		0x6a8d, 0x6aa0, 0x6a84, 0x6aa2, 0x6aa3, 0x0000, 0x0000, 0x0000, // 0x9ef8 - 0x9eff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f00 - 0x9f07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f08 - 0x9f0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f10 - 0x9f17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f18 - 0x9f1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f20 - 0x9f27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f28 - 0x9f2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f30 - 0x9f37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0x9f38 - 0x9f3f
		0x6a97, 0x8617, 0x6abb, 0x6ac3, 0x6ac2, 0x6ab8, 0x6ab3, 0x6aac, // 0x9f40 - 0x9f47
		0x6ade, 0x6ad1, 0x6adf, 0x6aaa, 0x6ada, 0x6aea, 0x6afb, 0x6b05, // 0x9f48 - 0x9f4f
		0x8616, 0x6afa, 0x6b12, 0x6b16, 0x9b31, 0x6b1f, 0x6b38, 0x6b37, // 0x9f50 - 0x9f57
		0x76dc, 0x6b39, 0x98ee, 0x6b47, 0x6b43, 0x6b49, 0x6b50, 0x6b59, // 0x9f58 - 0x9f5f
		0x6b54, 0x6b5b, 0x6b5f, 0x6b61, 0x6b78, 0x6b79, 0x6b7f, 0x6b80, // 0x9f60 - 0x9f67
		0x6b84, 0x6b83, 0x6b8d, 0x6b98, 0x6b95, 0x6b9e, 0x6ba4, 0x6baa, // 0x9f68 - 0x9f6f
		0x6bab, 0x6baf, 0x6bb2, 0x6bb1, 0x6bb3, 0x6bb7, 0x6bbc, 0x6bc6, // 0x9f70 - 0x9f77
		0x6bcb, 0x6bd3, 0x6bdf, 0x6bec, 0x6beb, 0x6bf3, 0x6bef, 0x0000, // 0x9f78 - 0x9f7f
		0x9ebe, 0x6c08, 0x6c13, 0x6c14, 0x6c1b, 0x6c24, 0x6c23, 0x6c5e, // 0x9f80 - 0x9f87
		0x6c55, 0x6c62, 0x6c6a, 0x6c82, 0x6c8d, 0x6c9a, 0x6c81, 0x6c9b, // 0x9f88 - 0x9f8f
		0x6c7e, 0x6c68, 0x6c73, 0x6c92, 0x6c90, 0x6cc4, 0x6cf1, 0x6cd3, // 0x9f90 - 0x9f97
		0x6cbd, 0x6cd7, 0x6cc5, 0x6cdd, 0x6cae, 0x6cb1, 0x6cbe, 0x6cba, // 0x9f98 - 0x9f9f
		0x6cdb, 0x6cef, 0x6cd9, 0x6cea, 0x6d1f, 0x884d, 0x6d36, 0x6d2b, // 0x9fa0 - 0x9fa7
		0x6d3d, 0x6d38, 0x6d19, 0x6d35, 0x6d33, 0x6d12, 0x6d0c, 0x6d63, // 0x9fa8 - 0x9faf
		0x6d93, 0x6d64, 0x6d5a, 0x6d79, 0x6d59, 0x6d8e, 0x6d95, 0x6fe4, // 0x9fb0 - 0x9fb7
		0x6d85, 0x6df9, 0x6e15, 0x6e0a, 0x6db5, 0x6dc7, 0x6de6, 0x6db8, // 0x9fb8 - 0x9fbf
		0x6dc6, 0x6dec, 0x6dde, 0x6dcc, 0x6de8, 0x6dd2, 0x6dc5, 0x6dfa, // 0x9fc0 - 0x9fc7
		0x6dd9, 0x6de4, 0x6dd5, 0x6dea, 0x6dee, 0x6e2d, 0x6e6e, 0x6e2e, // 0x9fc8 - 0x9fcf
		0x6e19, 0x6e72, 0x6e5f, 0x6e3e, 0x6e23, 0x6e6b, 0x6e2b, 0x6e76, // 0x9fd0 - 0x9fd7
		0x6e4d, 0x6e1f, 0x6e43, 0x6e3a, 0x6e4e, 0x6e24, 0x6eff, 0x6e1d, // 0x9fd8 - 0x9fdf
		0x6e38, 0x6e82, 0x6eaa, 0x6e98, 0x6ec9, 0x6eb7, 0x6ed3, 0x6ebd, // 0x9fe0 - 0x9fe7
		0x6eaf, 0x6ec4, 0x6eb2, 0x6ed4, 0x6ed5, 0x6e8f, 0x6ea5, 0x6ec2, // 0x9fe8 - 0x9fef
		0x6e9f, 0x6f41, 0x6f11, 0x704c, 0x6eec, 0x6ef8, 0x6efe, 0x6f3f, // 0x9ff0 - 0x9ff7
		0x6ef2, 0x6f31, 0x6eef, 0x6f32, 0x6ecc, 0x0000, 0x0000, 0x0000, // 0x9ff8 - 0x9fff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa000 - 0xa007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa008 - 0xa00f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa010 - 0xa017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa018 - 0xa01f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa020 - 0xa027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa028 - 0xa02f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa030 - 0xa037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa038 - 0xa03f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa040 - 0xa047
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa048 - 0xa04f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa050 - 0xa057
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa058 - 0xa05f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa060 - 0xa067
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa068 - 0xa06f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa070 - 0xa077
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa078 - 0xa07f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa080 - 0xa087
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa088 - 0xa08f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa090 - 0xa097
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa098 - 0xa09f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0a0 - 0xa0a7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0a8 - 0xa0af
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0b0 - 0xa0b7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0b8 - 0xa0bf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0c0 - 0xa0c7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0c8 - 0xa0cf
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0d0 - 0xa0d7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0d8 - 0xa0df
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0e0 - 0xa0e7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0e8 - 0xa0ef
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0f0 - 0xa0f7
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa0f8 - 0xa0ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa100 - 0xa107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa108 - 0xa10f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa110 - 0xa117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa118 - 0xa11f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa120 - 0xa127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa128 - 0xa12f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa130 - 0xa137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa138 - 0xa13f
		0x3000, 0xff0c, 0x3001, 0x3002, 0xff0e, 0x00b7, 0xff1b, 0xff1a, // 0xa140 - 0xa147
		0xff1f, 0xff01, 0xfe30, 0x22ef, 0x2025, 0xfe59, 0xfe5a, 0x0000, // 0xa148 - 0xa14f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xff5c, 0x2013, 0xfe31, // 0xa150 - 0xa157
		0x2014, 0xfe33, 0x2986, 0xfe34, 0xfe4f, 0xff08, 0xff09, 0xfe35, // 0xa158 - 0xa15f
		0xfe36, 0xff5b, 0xff5d, 0xfe37, 0xfe38, 0x3014, 0x3015, 0xfe39, // 0xa160 - 0xa167
		0xfe3a, 0x3010, 0x3011, 0xfe3b, 0xfe3c, 0x300a, 0x300b, 0xfe3d, // 0xa168 - 0xa16f
		0xfe3e, 0x3008, 0x3009, 0xfe3f, 0xfe40, 0x300c, 0x300d, 0xfe41, // 0xa170 - 0xa177
		0xfe42, 0x300e, 0x300f, 0xfe43, 0xfe44, 0x207e, 0x0000, 0x0000, // 0xa178 - 0xa17f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa180 - 0xa187
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa188 - 0xa18f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x204c, 0x204d, // 0xa190 - 0xa197
		0x02bc, 0x2997, 0x2998, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa198 - 0xa19f
		0x0000, 0x3000, 0x3001, 0x3002, 0x00b7, 0x02c9, 0x02c7, 0x00a8, // 0xa1a0 - 0xa1a7
		0x3003, 0x3005, 0x2014, 0x301c, 0x2016, 0x22ef, 0x2018, 0x2019, // 0xa1a8 - 0xa1af
		0x201c, 0x201d, 0x3014, 0x3015, 0x3008, 0x3009, 0x300a, 0x300b, // 0xa1b0 - 0xa1b7
		0x300c, 0x300d, 0x300e, 0x300f, 0x3016, 0x3017, 0x3010, 0x3011, // 0xa1b8 - 0xa1bf
		0x00b1, 0x00d7, 0x00f7, 0x2236, 0x2227, 0x2228, 0x2211, 0x220f, // 0xa1c0 - 0xa1c7
		0x222a, 0x2229, 0x2208, 0x2237, 0x221a, 0x22a5, 0x2225, 0x2220, // 0xa1c8 - 0xa1cf
		0x2312, 0x2299, 0x222b, 0x222e, 0x2261, 0x224c, 0x2248, 0x223d, // 0xa1d0 - 0xa1d7
		0x221d, 0x2260, 0x226e, 0x226f, 0x2264, 0x2265, 0x221e, 0x2235, // 0xa1d8 - 0xa1df
		0x2234, 0x2642, 0x2640, 0x00b0, 0x2032, 0x2033, 0x2103, 0xff04, // 0xa1e0 - 0xa1e7
		0x00a4, 0x00a2, 0x00a3, 0x2030, 0x00a7, 0x2116, 0x2606, 0x2605, // 0xa1e8 - 0xa1ef
		0x25cb, 0x25cf, 0x25ce, 0x25c7, 0x25c6, 0x25a1, 0x25a0, 0x25b3, // 0xa1f0 - 0xa1f7
		0x25b2, 0x203b, 0x2192, 0x2190, 0x2191, 0x2193, 0x3013, 0x0000, // 0xa1f8 - 0xa1ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa200 - 0xa207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa208 - 0xa20f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa210 - 0xa217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa218 - 0xa21f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa220 - 0xa227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa228 - 0xa22f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa230 - 0xa237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa238 - 0xa23f
		0x0000, 0xff0f, 0xff3c, 0xff04, 0x00a5, 0x3012, 0x00a2, 0x00a3, // 0xa240 - 0xa247
		0xff05, 0xff20, 0x2103, 0x2109, 0xfe69, 0xfe6a, 0xfe6b, 0x33d5, // 0xa248 - 0xa24f
		0x339c, 0x339d, 0x339e, 0x33ce, 0x33a1, 0x338e, 0x338f, 0x33c4, // 0xa250 - 0xa257
		0x00b0, 0x5159, 0x515b, 0x515e, 0x515d, 0x5161, 0x5163, 0x55e7, // 0xa258 - 0xa25f
		0x74e9, 0x7cce, 0x2581, 0x2582, 0x2583, 0x2584, 0x2585, 0x2586, // 0xa260 - 0xa267
		0x2587, 0x2588, 0x258f, 0x258e, 0x258d, 0x258c, 0x258b, 0x258a, // 0xa268 - 0xa26f
		0x2589, 0x253c, 0x2534, 0x252c, 0x2524, 0x251c, 0x2594, 0x2500, // 0xa270 - 0xa277
		0x2502, 0x0000, 0x250c, 0x2510, 0x2514, 0x2518, 0x256d, 0x0000, // 0xa278 - 0xa27f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa280 - 0xa287
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa288 - 0xa28f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa290 - 0xa297
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa298 - 0xa29f
		0x0000, 0x256e, 0x2570, 0x256f, 0x2550, 0x255e, 0x256a, 0x2561, // 0xa2a0 - 0xa2a7
		0x25e2, 0x25e3, 0x25e5, 0x25e4, 0x2571, 0x2572, 0x2573, 0xff10, // 0xa2a8 - 0xa2af
		0xff11, 0x2488, 0x2489, 0x248a, 0x248b, 0x248c, 0x248d, 0x248e, // 0xa2b0 - 0xa2b7
		0x248f, 0x2490, 0x2491, 0x2492, 0x2493, 0x2494, 0x2495, 0x2496, // 0xa2b8 - 0xa2bf
		0x2497, 0x2498, 0x2499, 0x249a, 0x249b, 0x2474, 0x2475, 0x2476, // 0xa2c0 - 0xa2c7
		0x2477, 0x2478, 0x2479, 0x247a, 0x247b, 0x247c, 0x247d, 0x247e, // 0xa2c8 - 0xa2cf
		0x247f, 0x2480, 0x2481, 0x2482, 0x2483, 0x2484, 0x2485, 0x2486, // 0xa2d0 - 0xa2d7
		0x2487, 0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466, // 0xa2d8 - 0xa2df
		0x2467, 0x2468, 0x2469, 0xff35, 0xff36, 0x3220, 0x3221, 0x3222, // 0xa2e0 - 0xa2e7
		0x3223, 0x3224, 0x3225, 0x3226, 0x3227, 0x3228, 0x3229, 0xff47, // 0xa2e8 - 0xa2ef
		0xff48, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 0x2165, 0x2166, // 0xa2f0 - 0xa2f7
		0x2167, 0x2168, 0x2169, 0x216a, 0x216b, 0xff55, 0xff56, 0x0000, // 0xa2f8 - 0xa2ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa300 - 0xa307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa308 - 0xa30f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa310 - 0xa317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa318 - 0xa31f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa320 - 0xa327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa328 - 0xa32f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa330 - 0xa337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa338 - 0xa33f
		0xff57, 0xff58, 0xff59, 0xff5a, 0x0391, 0x0392, 0x0393, 0x0394, // 0xa340 - 0xa347
		0x0395, 0x0396, 0x0397, 0x0398, 0x0399, 0x039a, 0x039b, 0x039c, // 0xa348 - 0xa34f
		0x039d, 0x039e, 0x039f, 0x03a0, 0x03a1, 0x03a3, 0x03a4, 0x03a5, // 0xa350 - 0xa357
		0x03a6, 0x03a7, 0x03a8, 0x03a9, 0x03b1, 0x03b2, 0x03b3, 0x03b4, // 0xa358 - 0xa35f
		0x03b5, 0x03b6, 0x03b7, 0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, // 0xa360 - 0xa367
		0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1, 0x03c3, 0x03c4, 0x03c5, // 0xa368 - 0xa36f
		0x03c6, 0x03c7, 0x03c8, 0x03c9, 0x3105, 0x3106, 0x3107, 0x3108, // 0xa370 - 0xa377
		0x3109, 0x310a, 0x310b, 0x310c, 0x310d, 0x310e, 0x310f, 0x0000, // 0xa378 - 0xa37f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x24b6, 0x24b7, // 0xa380 - 0xa387
		0x24b8, 0x24b9, 0x24ba, 0x24bb, 0x24bc, 0x24bd, 0x24be, 0x24bf, // 0xa388 - 0xa38f
		0x24c0, 0x24c1, 0x24c2, 0x24c3, 0x24c4, 0x24c5, 0x24c6, 0x24c7, // 0xa390 - 0xa397
		0x24c8, 0x24c9, 0x24ca, 0x24cb, 0x24cc, 0x24cd, 0x24ce, 0x24cf, // 0xa398 - 0xa39f
		0x0000, 0xff01, 0xff02, 0xff03, 0x00a5, 0xff05, 0xff06, 0xff07, // 0xa3a0 - 0xa3a7
		0xff08, 0xff09, 0xff0a, 0xff0b, 0xff0c, 0xff0d, 0xff0e, 0xff0f, // 0xa3a8 - 0xa3af
		0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, // 0xa3b0 - 0xa3b7
		0xff18, 0xff19, 0xff1a, 0xff1b, 0xff1c, 0xff1d, 0xff1e, 0xff1f, // 0xa3b8 - 0xa3bf
		0xff20, 0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27, // 0xa3c0 - 0xa3c7
		0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f, // 0xa3c8 - 0xa3cf
		0xff30, 0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37, // 0xa3d0 - 0xa3d7
		0xff38, 0xff39, 0xff3a, 0xff3b, 0xff3c, 0xff3d, 0xff3e, 0xff3f, // 0xa3d8 - 0xa3df
		0xff40, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47, // 0xa3e0 - 0xa3e7
		0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f, // 0xa3e8 - 0xa3ef
		0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57, // 0xa3f0 - 0xa3f7
		0xff58, 0xff59, 0xff5a, 0xff5b, 0xff5c, 0xff5d, 0x203e, 0x0000, // 0xa3f8 - 0xa3ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa400 - 0xa407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa408 - 0xa40f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa410 - 0xa417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa418 - 0xa41f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa420 - 0xa427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa428 - 0xa42f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa430 - 0xa437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa438 - 0xa43f
		0x4e00, 0x4e59, 0x4e01, 0x4e03, 0x4e43, 0x4e5d, 0x4e86, 0x4e8c, // 0xa440 - 0xa447
		0x4eba, 0x513f, 0x5165, 0x516b, 0x51e0, 0x5200, 0x5201, 0x529b, // 0xa448 - 0xa44f
		0x5315, 0x5341, 0x535c, 0x53c8, 0x4e09, 0x4e0b, 0x4e08, 0x4e0a, // 0xa450 - 0xa457
		0x4e2b, 0x4e38, 0x51e1, 0x4e45, 0x4e48, 0x4e5f, 0x4e5e, 0x4e8e, // 0xa458 - 0xa45f
		0x4ea1, 0x5140, 0x5203, 0x52fa, 0x5343, 0x53c9, 0x53e3, 0x571f, // 0xa460 - 0xa467
		0x58eb, 0x5915, 0x5927, 0x5973, 0x5b50, 0x5b51, 0x5b53, 0x5bf8, // 0xa468 - 0xa46f
		0x5c0f, 0x5c22, 0x5c38, 0x5c71, 0x5ddd, 0x5de5, 0x5df1, 0x5df2, // 0xa470 - 0xa477
		0x5df3, 0x5dfe, 0x5e72, 0x5efe, 0x5f0b, 0x5f13, 0x624d, 0x0000, // 0xa478 - 0xa47f
		0x0000, 0x227a, 0x227b, 0x22ce, 0x22cf, 0x2280, 0x2281, 0x2270, // 0xa480 - 0xa487
		0x2271, 0x2272, 0x2273, 0x2ac5, 0x2acb, 0x2ac6, 0x2acc, 0x2276, // 0xa488 - 0xa48f
		0x2277, 0x2279, 0x22da, 0x22db, 0x2a8b, 0x2a8c, 0x2a91, 0x2a92, // 0xa490 - 0xa497
		0x0000, 0x2245, 0x2243, 0x2248, 0x29a3, 0x22a4, 0x0000, 0x0000, // 0xa498 - 0xa49f
		0x0000, 0x3041, 0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047, // 0xa4a0 - 0xa4a7
		0x3048, 0x3049, 0x304a, 0x304b, 0x304c, 0x304d, 0x304e, 0x304f, // 0xa4a8 - 0xa4af
		0x3050, 0x3051, 0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057, // 0xa4b0 - 0xa4b7
		0x3058, 0x3059, 0x305a, 0x305b, 0x305c, 0x305d, 0x305e, 0x305f, // 0xa4b8 - 0xa4bf
		0x3060, 0x3061, 0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 0x3067, // 0xa4c0 - 0xa4c7
		0x3068, 0x3069, 0x306a, 0x306b, 0x306c, 0x306d, 0x306e, 0x306f, // 0xa4c8 - 0xa4cf
		0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077, // 0xa4d0 - 0xa4d7
		0x3078, 0x3079, 0x307a, 0x307b, 0x307c, 0x307d, 0x307e, 0x307f, // 0xa4d8 - 0xa4df
		0x3080, 0x3081, 0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087, // 0xa4e0 - 0xa4e7
		0x3088, 0x3089, 0x308a, 0x308b, 0x308c, 0x308d, 0x308e, 0x308f, // 0xa4e8 - 0xa4ef
		0x3090, 0x3091, 0x3092, 0x3093, 0x6c34, 0x706b, 0x722a, 0x7236, // 0xa4f0 - 0xa4f7
		0x723b, 0x7247, 0x7259, 0x725b, 0x72ac, 0x738b, 0x4e19, 0x0000, // 0xa4f8 - 0xa4ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa500 - 0xa507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa508 - 0xa50f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa510 - 0xa517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa518 - 0xa51f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa520 - 0xa527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa528 - 0xa52f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa530 - 0xa537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa538 - 0xa53f
		0x4e16, 0x4e15, 0x4e14, 0x4e18, 0x4e3b, 0x4e4d, 0x4e4f, 0x4e4e, // 0xa540 - 0xa547
		0x4ee5, 0x4ed8, 0x4ed4, 0x4ed5, 0x4ed6, 0x4ed7, 0x4ee3, 0x4ee4, // 0xa548 - 0xa54f
		0x4ed9, 0x4ede, 0x5145, 0x5144, 0x5189, 0x518a, 0x51ac, 0x51f9, // 0xa550 - 0xa557
		0x51fa, 0x51f8, 0x520a, 0x52a0, 0x529f, 0x5305, 0x5306, 0x5317, // 0xa558 - 0xa55f
		0x531d, 0x4edf, 0x534a, 0x5349, 0x5361, 0x5360, 0x536f, 0x536e, // 0xa560 - 0xa567
		0x53bb, 0x53ef, 0x53e4, 0x53f3, 0x53ec, 0x53ee, 0x53e9, 0x53e8, // 0xa568 - 0xa56f
		0x53fc, 0x53f8, 0x53f5, 0x53eb, 0x53e6, 0x53ea, 0x53f2, 0x53f1, // 0xa570 - 0xa577
		0x53f0, 0x53e5, 0x53ed, 0x53fb, 0x56db, 0x56da, 0x5916, 0x0000, // 0xa578 - 0xa57f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa580 - 0xa587
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa588 - 0xa58f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa590 - 0xa597
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa598 - 0xa59f
		0x0000, 0x30a1, 0x30a2, 0x30a3, 0x30a4, 0x30a5, 0x30a6, 0x30a7, // 0xa5a0 - 0xa5a7
		0x30a8, 0x30a9, 0x30aa, 0x30ab, 0x30ac, 0x30ad, 0x30ae, 0x30af, // 0xa5a8 - 0xa5af
		0x30b0, 0x30b1, 0x30b2, 0x30b3, 0x30b4, 0x30b5, 0x30b6, 0x30b7, // 0xa5b0 - 0xa5b7
		0x30b8, 0x30b9, 0x30ba, 0x30bb, 0x30bc, 0x30bd, 0x30be, 0x30bf, // 0xa5b8 - 0xa5bf
		0x30c0, 0x30c1, 0x30c2, 0x30c3, 0x30c4, 0x30c5, 0x30c6, 0x30c7, // 0xa5c0 - 0xa5c7
		0x30c8, 0x30c9, 0x30ca, 0x30cb, 0x30cc, 0x30cd, 0x30ce, 0x30cf, // 0xa5c8 - 0xa5cf
		0x30d0, 0x30d1, 0x30d2, 0x30d3, 0x30d4, 0x30d5, 0x30d6, 0x30d7, // 0xa5d0 - 0xa5d7
		0x30d8, 0x30d9, 0x30da, 0x30db, 0x30dc, 0x30dd, 0x30de, 0x30df, // 0xa5d8 - 0xa5df
		0x30e0, 0x30e1, 0x30e2, 0x30e3, 0x30e4, 0x30e5, 0x30e6, 0x30e7, // 0xa5e0 - 0xa5e7
		0x30e8, 0x30e9, 0x30ea, 0x30eb, 0x30ec, 0x30ed, 0x30ee, 0x30ef, // 0xa5e8 - 0xa5ef
		0x30f0, 0x30f1, 0x30f2, 0x30f3, 0x30f4, 0x30f5, 0x30f6, 0x4efd, // 0xa5f0 - 0xa5f7
		0x4f01, 0x4f0b, 0x5149, 0x5147, 0x5146, 0x5148, 0x5168, 0x0000, // 0xa5f8 - 0xa5ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa600 - 0xa607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa608 - 0xa60f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa610 - 0xa617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa618 - 0xa61f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa620 - 0xa627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa628 - 0xa62f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa630 - 0xa637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa638 - 0xa63f
		0x5171, 0x518d, 0x51b0, 0x5217, 0x5211, 0x5212, 0x520e, 0x5216, // 0xa640 - 0xa647
		0x52a3, 0x5308, 0x5321, 0x5320, 0x5370, 0x5371, 0x5409, 0x540f, // 0xa648 - 0xa64f
		0x540c, 0x540a, 0x5410, 0x5401, 0x540b, 0x5404, 0x5411, 0x540d, // 0xa650 - 0xa657
		0x5408, 0x5403, 0x540e, 0x5406, 0x5412, 0x56e0, 0x56de, 0x56dd, // 0xa658 - 0xa65f
		0x5733, 0x5730, 0x5728, 0x572d, 0x572c, 0x572f, 0x5729, 0x5919, // 0xa660 - 0xa667
		0x591a, 0x5937, 0x5938, 0x5984, 0x5978, 0x5983, 0x597d, 0x5979, // 0xa668 - 0xa66f
		0x5982, 0x5981, 0x5b57, 0x5b58, 0x5b87, 0x5b88, 0x5b85, 0x5b89, // 0xa670 - 0xa677
		0x5bfa, 0x5c16, 0x5c79, 0x5dde, 0x5e06, 0x5e76, 0x5e74, 0x0000, // 0xa678 - 0xa67f
		0x0000, 0xf809, 0x0000, 0x274d, 0x25cd, 0x0000, 0x0000, 0x0000, // 0xa680 - 0xa687
		0x2720, 0x0000, 0x0000, 0x0000, 0x0000, 0x2741, 0x0000, 0xf808, // 0xa688 - 0xa68f
		0x0000, 0x0000, 0x0000, 0x262f, 0x0000, 0x0000, 0xf80b, 0x0000, // 0xa690 - 0xa697
		0x0000, 0x2740, 0xf80c, 0x0000, 0x0000, 0x0000, 0x3020, 0xf807, // 0xa698 - 0xa69f
		0x0000, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, // 0xa6a0 - 0xa6a7
		0x0398, 0x0399, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f, // 0xa6a8 - 0xa6af
		0x03a0, 0x03a1, 0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8, // 0xa6b0 - 0xa6b7
		0x03a9, 0x6b64, 0x6b7b, 0x6c16, 0x6c5d, 0x6c57, 0x6c59, 0x6c5f, // 0xa6b8 - 0xa6bf
		0x6c60, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7, // 0xa6c0 - 0xa6c7
		0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf, // 0xa6c8 - 0xa6cf
		0x03c0, 0x03c1, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8, // 0xa6d0 - 0xa6d7
		0x03c9, 0x808c, 0x81e3, 0x81ea, 0x81f3, 0x81fc, 0x820c, 0x821b, // 0xa6d8 - 0xa6df
		0xfe35, 0xfe36, 0xfe39, 0xfe3a, 0xfe3f, 0xfe40, 0xfe3d, 0xfe3e, // 0xa6e0 - 0xa6e7
		0xfe41, 0xfe42, 0xfe43, 0xfe44, 0x4f4d, 0x4f4f, 0xfe3b, 0xfe3c, // 0xa6e8 - 0xa6ef
		0xfe37, 0xfe38, 0xfe31, 0x4f55, 0xfe33, 0xfe34, 0x4f51, 0x4f3d, // 0xa6f0 - 0xa6f7
		0x4f3a, 0x4f38, 0x4f43, 0x4f54, 0x4f3c, 0x4f46, 0x4f63, 0x0000, // 0xa6f8 - 0xa6ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa700 - 0xa707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa708 - 0xa70f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa710 - 0xa717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa718 - 0xa71f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa720 - 0xa727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa728 - 0xa72f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa730 - 0xa737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa738 - 0xa73f
		0x4f5c, 0x4f60, 0x4f2f, 0x4f4e, 0x4f36, 0x4f59, 0x4f5d, 0x4f48, // 0xa740 - 0xa747
		0x4f5a, 0x514c, 0x514b, 0x514d, 0x5175, 0x51b6, 0x51b7, 0x5225, // 0xa748 - 0xa74f
		0x5224, 0x5229, 0x522a, 0x5228, 0x52ab, 0x52a9, 0x52aa, 0x52ac, // 0xa750 - 0xa757
		0x5323, 0x5373, 0x5375, 0x541d, 0x542d, 0x541e, 0x543e, 0x5426, // 0xa758 - 0xa75f
		0x544e, 0x5427, 0x5446, 0x5443, 0x5433, 0x5448, 0x5442, 0x541b, // 0xa760 - 0xa767
		0x5429, 0x544a, 0x5439, 0x543b, 0x5438, 0x542e, 0x5435, 0x5436, // 0xa768 - 0xa76f
		0x5420, 0x543c, 0x5440, 0x5431, 0x542b, 0x541f, 0x542c, 0x56ea, // 0xa770 - 0xa777
		0x56f0, 0x56e4, 0x56eb, 0x574a, 0x5751, 0x5740, 0x574d, 0x0000, // 0xa778 - 0xa77f
		0x0000, 0x0000, 0x329e, 0x0000, 0x203c, 0x2049, 0x0000, 0x2047, // 0xa780 - 0xa787
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2588, // 0xa788 - 0xa78f
		0x25e6, 0x0000, 0x0000, 0x0000, 0x25bf, 0x25b5, 0x25b9, 0x25c3, // 0xa790 - 0xa797
		0x2666, 0x2981, 0x25fc, 0x0000, 0x25ca, 0x3231, 0x3239, 0x33cb, // 0xa798 - 0xa79f
		0x0000, 0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401, // 0xa7a0 - 0xa7a7
		0x0416, 0x0417, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, // 0xa7a8 - 0xa7af
		0x041e, 0x041f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, // 0xa7b0 - 0xa7b7
		0x0426, 0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, // 0xa7b8 - 0xa7bf
		0x042e, 0x042f, 0x5c91, 0x5c94, 0x5c8c, 0x5deb, 0x5e0c, 0x5e8f, // 0xa7c0 - 0xa7c7
		0x5e87, 0x5e8a, 0x5ef7, 0x5f04, 0x5f1f, 0x5f64, 0x5f62, 0x5f77, // 0xa7c8 - 0xa7cf
		0x5f79, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451, // 0xa7d0 - 0xa7d7
		0x0436, 0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, // 0xa7d8 - 0xa7df
		0x043e, 0x043f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, // 0xa7e0 - 0xa7e7
		0x0446, 0x0447, 0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, // 0xa7e8 - 0xa7ef
		0x044e, 0x044f, 0x65f1, 0x66f4, 0x675f, 0x674e, 0x674f, 0x6750, // 0xa7f0 - 0xa7f7
		0x6751, 0x675c, 0x6756, 0x675e, 0x6749, 0x6746, 0x6760, 0x0000, // 0xa7f8 - 0xa7ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa800 - 0xa807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa808 - 0xa80f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa810 - 0xa817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa818 - 0xa81f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa820 - 0xa827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa828 - 0xa82f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa830 - 0xa837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa838 - 0xa83f
		0x6753, 0x6757, 0x6b65, 0x6bcf, 0x6c42, 0x6c5e, 0x6c99, 0x6c81, // 0xa840 - 0xa847
		0x6c88, 0x6c89, 0x6c85, 0x6c9b, 0x6c6a, 0x6c7a, 0x6c90, 0x6c70, // 0xa848 - 0xa84f
		0x6c8c, 0x6c68, 0x6c96, 0x6c92, 0x6c7d, 0x6c83, 0x6c72, 0x6c7e, // 0xa850 - 0xa857
		0x6c74, 0x6c86, 0x6c76, 0x6c8d, 0x6c94, 0x6c98, 0x6c82, 0x7076, // 0xa858 - 0xa85f
		0x707c, 0x707d, 0x7078, 0x7262, 0x7261, 0x7260, 0x72c4, 0x72c2, // 0xa860 - 0xa867
		0x7396, 0x752c, 0x752b, 0x7537, 0x7538, 0x7682, 0x76ef, 0x77e3, // 0xa868 - 0xa86f
		0x79c1, 0x79c0, 0x79bf, 0x7a76, 0x7cfb, 0x7f55, 0x8096, 0x8093, // 0xa870 - 0xa877
		0x809d, 0x8098, 0x809b, 0x809a, 0x80b2, 0x826f, 0x8292, 0x0000, // 0xa878 - 0xa87f
		0x0000, 0x21b4, 0x21b0, 0x21b3, 0x0000, 0x0000, 0x2936, 0x0000, // 0xa880 - 0xa887
		0x0000, 0x2935, 0x0000, 0x2937, 0x2939, 0x2934, 0x0000, 0x0000, // 0xa888 - 0xa88f
		0x0000, 0x0000, 0x21bc, 0x21c0, 0xf841, 0x0000, 0x0000, 0x0000, // 0xa890 - 0xa897
		0x0000, 0x0000, 0x0000, 0x0000, 0xf849, 0xf848, 0x21c4, 0x21c5, // 0xa898 - 0xa89f
		0x0000, 0x0101, 0x00e1, 0x01ce, 0x00e0, 0x0113, 0x00e9, 0x011b, // 0xa8a0 - 0xa8a7
		0x00e8, 0x012b, 0x00ed, 0x01d0, 0x00ec, 0x014d, 0x00f3, 0x01d2, // 0xa8a8 - 0xa8af
		0x00f2, 0x016b, 0x00fa, 0x01d4, 0x00f9, 0x01d6, 0x01d8, 0x01da, // 0xa8b0 - 0xa8b7
		0x01dc, 0x00fc, 0x00ea, 0x0251, 0x1e3f, 0x0144, 0x0148, 0x01f9, // 0xa8b8 - 0xa8bf
		0x0261, 0x962a, 0x962c, 0x4e26, 0x4e56, 0x3105, 0x3106, 0x3107, // 0xa8c0 - 0xa8c7
		0x3108, 0x3109, 0x310a, 0x310b, 0x310c, 0x310d, 0x310e, 0x310f, // 0xa8c8 - 0xa8cf
		0x3110, 0x3111, 0x3112, 0x3113, 0x3114, 0x3115, 0x3116, 0x3117, // 0xa8d0 - 0xa8d7
		0x3118, 0x3119, 0x311a, 0x311b, 0x311c, 0x311d, 0x311e, 0x311f, // 0xa8d8 - 0xa8df
		0x3120, 0x3121, 0x3122, 0x3123, 0x3124, 0x3125, 0x3126, 0x3127, // 0xa8e0 - 0xa8e7
		0x3128, 0x3129, 0x5237, 0x523a, 0x5230, 0x522e, 0x5236, 0x5241, // 0xa8e8 - 0xa8ef
		0x52be, 0x52bb, 0x5352, 0x5354, 0x5353, 0x5351, 0x5366, 0x5377, // 0xa8f0 - 0xa8f7
		0x5378, 0x5379, 0x53d6, 0x53d4, 0x53d7, 0x5473, 0x5475, 0x0000, // 0xa8f8 - 0xa8ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa900 - 0xa907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa908 - 0xa90f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa910 - 0xa917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa918 - 0xa91f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa920 - 0xa927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa928 - 0xa92f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa930 - 0xa937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa938 - 0xa93f
		0x5496, 0x5478, 0x5495, 0x5480, 0x547b, 0x5477, 0x5484, 0x5492, // 0xa940 - 0xa947
		0x5486, 0x547c, 0x5490, 0x5471, 0x5476, 0x548c, 0x549a, 0x5462, // 0xa948 - 0xa94f
		0x5468, 0x548b, 0x547d, 0x548e, 0x56fa, 0x5783, 0x5777, 0x576a, // 0xa950 - 0xa957
		0x5769, 0x5761, 0x5766, 0x5764, 0x577c, 0x591c, 0x5949, 0x5947, // 0xa958 - 0xa95f
		0x5948, 0x5944, 0x5954, 0x59be, 0x59bb, 0x59d4, 0x59b9, 0x59ae, // 0xa960 - 0xa967
		0x59d1, 0x59c6, 0x59d0, 0x59cd, 0x59cb, 0x59d3, 0x59ca, 0x59af, // 0xa968 - 0xa96f
		0x59b3, 0x59d2, 0x59c5, 0x5b5f, 0x5b64, 0x5b63, 0x5b97, 0x5b9a, // 0xa970 - 0xa977
		0x5b98, 0x5b9c, 0x5b99, 0x5b9b, 0x5c1a, 0x5c48, 0x5c45, 0x0000, // 0xa978 - 0xa97f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa980 - 0xa987
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa988 - 0xa98f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa990 - 0xa997
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xa998 - 0xa99f
		0x0000, 0x5c46, 0x5cb7, 0x5ca1, 0x2500, 0x2501, 0x2502, 0x2503, // 0xa9a0 - 0xa9a7
		0x2504, 0x2505, 0x2506, 0x2507, 0x2508, 0x2509, 0x250a, 0x250b, // 0xa9a8 - 0xa9af
		0x250c, 0x250d, 0x250e, 0x250f, 0x2510, 0x2511, 0x2512, 0x2513, // 0xa9b0 - 0xa9b7
		0x2514, 0x2515, 0x2516, 0x2517, 0x2518, 0x2519, 0x251a, 0x251b, // 0xa9b8 - 0xa9bf
		0x251c, 0x251d, 0x251e, 0x251f, 0x2520, 0x2521, 0x2522, 0x2523, // 0xa9c0 - 0xa9c7
		0x2524, 0x2525, 0x2526, 0x2527, 0x2528, 0x2529, 0x252a, 0x252b, // 0xa9c8 - 0xa9cf
		0x252c, 0x252d, 0x252e, 0x252f, 0x2530, 0x2531, 0x2532, 0x2533, // 0xa9d0 - 0xa9d7
		0x2534, 0x2535, 0x2536, 0x2537, 0x2538, 0x2539, 0x253a, 0x253b, // 0xa9d8 - 0xa9df
		0x253c, 0x253d, 0x253e, 0x253f, 0x2540, 0x2541, 0x2542, 0x2543, // 0xa9e0 - 0xa9e7
		0x2544, 0x2545, 0x2546, 0x2547, 0x2548, 0x2549, 0x254a, 0x254b, // 0xa9e8 - 0xa9ef
		0x62ce, 0x653e, 0x65a7, 0x65bc, 0x65fa, 0x6614, 0x6613, 0x660c, // 0xa9f0 - 0xa9f7
		0x6606, 0x6602, 0x660e, 0x6600, 0x660f, 0x6615, 0x660a, 0x0000, // 0xa9f8 - 0xa9ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa00 - 0xaa07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa08 - 0xaa0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa10 - 0xaa17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa18 - 0xaa1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa20 - 0xaa27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa28 - 0xaa2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa30 - 0xaa37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa38 - 0xaa3f
		0x6607, 0x670d, 0x670b, 0x676d, 0x678b, 0x6795, 0x6771, 0x679c, // 0xaa40 - 0xaa47
		0x6773, 0x6777, 0x6787, 0x679d, 0x6797, 0x676f, 0x6770, 0x677f, // 0xaa48 - 0xaa4f
		0x6789, 0x677e, 0x6790, 0x6775, 0x679a, 0x6793, 0x677c, 0x676a, // 0xaa50 - 0xaa57
		0x6772, 0x6b23, 0x6b66, 0x6b67, 0x6b7f, 0x6c13, 0x6c1b, 0x6ce3, // 0xaa58 - 0xaa5f
		0x6ce8, 0x6cf3, 0x6cb1, 0x6ccc, 0x6ce5, 0x6cb3, 0x6cbd, 0x6cbe, // 0xaa60 - 0xaa67
		0x6cbc, 0x6ce2, 0x6cab, 0x6cd5, 0x6cd3, 0x6cb8, 0x6cc4, 0x6cb9, // 0xaa68 - 0xaa6f
		0x6cc1, 0x6cae, 0x6cd7, 0x6cc5, 0x6cf1, 0x6cbf, 0x6cbb, 0x6ce1, // 0xaa70 - 0xaa77
		0x6cdb, 0x6cca, 0x6cac, 0x6cef, 0x6cdc, 0x6cd6, 0x6ce0, 0x0000, // 0xaa78 - 0xaa7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa80 - 0xaa87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa88 - 0xaa8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa90 - 0xaa97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaa98 - 0xaa9f
		0x0000, 0x7095, 0x708e, 0x7092, 0x708a, 0x7099, 0x722c, 0x722d, // 0xaaa0 - 0xaaa7
		0x7238, 0x7248, 0x7267, 0x7269, 0x72c0, 0x72ce, 0x72d9, 0x72d7, // 0xaaa8 - 0xaaaf
		0x72d0, 0x73a9, 0x73a8, 0x739f, 0x73ab, 0x73a5, 0x753d, 0x759d, // 0xaab0 - 0xaab7
		0x7599, 0x759a, 0x7684, 0x76c2, 0x76f2, 0x76f4, 0x77e5, 0x77fd, // 0xaab8 - 0xaabf
		0x793e, 0x7940, 0x7941, 0x79c9, 0x79c8, 0x7a7a, 0x7a79, 0x7afa, // 0xaac0 - 0xaac7
		0x7cfe, 0x7f54, 0x7f8c, 0x7f8b, 0x8005, 0x80ba, 0x80a5, 0x80a2, // 0xaac8 - 0xaacf
		0x80b1, 0x80a1, 0x80ab, 0x80a9, 0x80b4, 0x80aa, 0x80af, 0x81e5, // 0xaad0 - 0xaad7
		0x81fe, 0x820d, 0x82b3, 0x829d, 0x8299, 0x82ad, 0x82bd, 0x829f, // 0xaad8 - 0xaadf
		0x82b9, 0x82b1, 0x82ac, 0x82a5, 0x82af, 0x82b8, 0x82a3, 0x82b0, // 0xaae0 - 0xaae7
		0x82be, 0x82b7, 0x864e, 0x8671, 0x521d, 0x8868, 0x8ecb, 0x8fce, // 0xaae8 - 0xaaef
		0x8fd4, 0x8fd1, 0x90b5, 0x90b8, 0x90b1, 0x90b6, 0x91c7, 0x91d1, // 0xaaf0 - 0xaaf7
		0x9577, 0x9580, 0x961c, 0x9640, 0x963f, 0x963b, 0x9644, 0x0000, // 0xaaf8 - 0xaaff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab00 - 0xab07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab08 - 0xab0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab10 - 0xab17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab18 - 0xab1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab20 - 0xab27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab28 - 0xab2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab30 - 0xab37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab38 - 0xab3f
		0x9642, 0x96b9, 0x96e8, 0x9752, 0x975e, 0x4e9f, 0x4ead, 0x4eae, // 0xab40 - 0xab47
		0x4fe1, 0x4fb5, 0x4faf, 0x4fbf, 0x4fe0, 0x4fd1, 0x4fcf, 0x4fdd, // 0xab48 - 0xab4f
		0x4fc3, 0x4fb6, 0x4fd8, 0x4fdf, 0x4fca, 0x4fd7, 0x4fae, 0x4fd0, // 0xab50 - 0xab57
		0x4fc4, 0x4fc2, 0x4fda, 0x4fce, 0x4fde, 0x4fb7, 0x5157, 0x5192, // 0xab58 - 0xab5f
		0x5191, 0x51a0, 0x524e, 0x5243, 0x524a, 0x524d, 0x524c, 0x524b, // 0xab60 - 0xab67
		0x5247, 0x52c7, 0x52c9, 0x52c3, 0x52c1, 0x530d, 0x5357, 0x537b, // 0xab68 - 0xab6f
		0x539a, 0x53db, 0x54ac, 0x54c0, 0x54a8, 0x54ce, 0x54c9, 0x54b8, // 0xab70 - 0xab77
		0x54a6, 0x54b3, 0x54c7, 0x54c2, 0x54bd, 0x54aa, 0x54c1, 0x0000, // 0xab78 - 0xab7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab80 - 0xab87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab88 - 0xab8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab90 - 0xab97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xab98 - 0xab9f
		0x0000, 0x54c4, 0x54c8, 0x54af, 0x54ab, 0x54b1, 0x54bb, 0x54a9, // 0xaba0 - 0xaba7
		0x54a7, 0x54bf, 0x56ff, 0x5782, 0x578b, 0x57a0, 0x57a3, 0x57a2, // 0xaba8 - 0xabaf
		0x57ce, 0x57ae, 0x5793, 0x5955, 0x5951, 0x594f, 0x594e, 0x5950, // 0xabb0 - 0xabb7
		0x59dc, 0x59d8, 0x59ff, 0x59e3, 0x59e8, 0x5a03, 0x59e5, 0x59ea, // 0xabb8 - 0xabbf
		0x59da, 0x59e6, 0x5a01, 0x59fb, 0x5b69, 0x5ba3, 0x5ba6, 0x5ba4, // 0xabc0 - 0xabc7
		0x5ba2, 0x5ba5, 0x5c01, 0x5c4e, 0x5c4f, 0x5c4d, 0x5c4b, 0x5cd9, // 0xabc8 - 0xabcf
		0x5cd2, 0x5df7, 0x5e1d, 0x5e25, 0x5e1f, 0x5e7d, 0x5ea0, 0x5ea6, // 0xabd0 - 0xabd7
		0x5efa, 0x5f08, 0x5f2d, 0x5f65, 0x5f88, 0x5f85, 0x5f8a, 0x5f8b, // 0xabd8 - 0xabdf
		0x5f87, 0x5f8c, 0x5f89, 0x6012, 0x601d, 0x6020, 0x6025, 0x600e, // 0xabe0 - 0xabe7
		0x6028, 0x604d, 0x6070, 0x6068, 0x6062, 0x6046, 0x6043, 0x606c, // 0xabe8 - 0xabef
		0x606b, 0x606a, 0x6064, 0x6241, 0x62dc, 0x6316, 0x6309, 0x62fc, // 0xabf0 - 0xabf7
		0x62ed, 0x6301, 0x62ee, 0x62fd, 0x6307, 0x62f1, 0x62f7, 0x0000, // 0xabf8 - 0xabff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac00 - 0xac07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac08 - 0xac0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac10 - 0xac17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac18 - 0xac1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac20 - 0xac27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac28 - 0xac2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac30 - 0xac37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac38 - 0xac3f
		0x62ef, 0x62ec, 0x62fe, 0x62f4, 0x6311, 0x6302, 0x653f, 0x6545, // 0xac40 - 0xac47
		0x65ab, 0x65bd, 0x65e2, 0x6625, 0x662d, 0x6620, 0x6627, 0x662f, // 0xac48 - 0xac4f
		0x661f, 0x6628, 0x6631, 0x6624, 0x66f7, 0x67ff, 0x67d3, 0x67f1, // 0xac50 - 0xac57
		0x67d4, 0x67d0, 0x67ec, 0x67b6, 0x67af, 0x67f5, 0x67e9, 0x67ef, // 0xac58 - 0xac5f
		0x67c4, 0x67d1, 0x67b4, 0x67da, 0x67e5, 0x67b8, 0x67cf, 0x67de, // 0xac60 - 0xac67
		0x67f3, 0x67b0, 0x67d9, 0x67e2, 0x67dd, 0x67d2, 0x6b6a, 0x6b83, // 0xac68 - 0xac6f
		0x6b86, 0x6bb5, 0x6bd2, 0x6bd7, 0x6c1f, 0x6cc9, 0x6d0b, 0x6d32, // 0xac70 - 0xac77
		0x6d2a, 0x6d41, 0x6d25, 0x6d0c, 0x6d31, 0x6d1e, 0x6d17, 0x0000, // 0xac78 - 0xac7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac80 - 0xac87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x261d, 0x261f, 0x0000, // 0xac88 - 0xac8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac90 - 0xac97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xac98 - 0xac9f
		0x0000, 0x6d3b, 0x6d3d, 0x6d3e, 0x6d36, 0x6d1b, 0x6cf5, 0x6d39, // 0xaca0 - 0xaca7
		0x6d27, 0x6d38, 0x6d29, 0x6d2e, 0x6d35, 0x6d0e, 0x6d2b, 0x70ab, // 0xaca8 - 0xacaf
		0x70ba, 0x70b3, 0x70ac, 0x70af, 0x70ad, 0x70b8, 0x70ae, 0x70a4, // 0xacb0 - 0xacb7
		0x7230, 0x7272, 0x726f, 0x7274, 0x72e9, 0x72e0, 0x72e1, 0x73b7, // 0xacb8 - 0xacbf
		0x73ca, 0x73bb, 0x73b2, 0x73cd, 0x73c0, 0x73b3, 0x751a, 0x752d, // 0xacc0 - 0xacc7
		0x754f, 0x754c, 0x754e, 0x754b, 0x75ab, 0x75a4, 0x75a5, 0x75a2, // 0xacc8 - 0xaccf
		0x75a3, 0x7678, 0x7686, 0x7687, 0x7688, 0x76c8, 0x76c6, 0x76c3, // 0xacd0 - 0xacd7
		0x76c5, 0x7701, 0x76f9, 0x76f8, 0x7709, 0x770b, 0x76fe, 0x76fc, // 0xacd8 - 0xacdf
		0x7707, 0x77dc, 0x7802, 0x7814, 0x780c, 0x780d, 0x7946, 0x7949, // 0xace0 - 0xace7
		0x7948, 0x7947, 0x79b9, 0x79ba, 0x79d1, 0x79d2, 0x79cb, 0x7a7f, // 0xace8 - 0xacef
		0x7a81, 0x7aff, 0x7afd, 0x7c7d, 0x7d02, 0x7d05, 0x7d00, 0x7d09, // 0xacf0 - 0xacf7
		0x7d07, 0x7d04, 0x7d06, 0x7f38, 0x7f8e, 0x7fbf, 0x8004, 0x0000, // 0xacf8 - 0xacff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad00 - 0xad07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad08 - 0xad0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad10 - 0xad17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad18 - 0xad1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad20 - 0xad27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad28 - 0xad2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad30 - 0xad37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad38 - 0xad3f
		0x8010, 0x800d, 0x8011, 0x8036, 0x80d6, 0x80e5, 0x80da, 0x80c3, // 0xad40 - 0xad47
		0x80c4, 0x80cc, 0x80e1, 0x80db, 0x80ce, 0x80de, 0x80e4, 0x80dd, // 0xad48 - 0xad4f
		0x81f4, 0x8222, 0x82e7, 0x8303, 0x8305, 0x82e3, 0x82db, 0x82e6, // 0xad50 - 0xad57
		0x8304, 0x82e5, 0x8302, 0x8309, 0x82d2, 0x82d7, 0x82f1, 0x8301, // 0xad58 - 0xad5f
		0x82dc, 0x82d4, 0x82d1, 0x82de, 0x82d3, 0x82df, 0x82ef, 0x8306, // 0xad60 - 0xad67
		0x8650, 0x8679, 0x867b, 0x867a, 0x884d, 0x886b, 0x8981, 0x89d4, // 0xad68 - 0xad6f
		0x8a08, 0x8a02, 0x8a03, 0x8c9e, 0x8ca0, 0x8d74, 0x8d73, 0x8db4, // 0xad70 - 0xad77
		0x8ecd, 0x8ecc, 0x8ff0, 0x8fe6, 0x8fe2, 0x8fea, 0x8fe5, 0x0000, // 0xad78 - 0xad7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad80 - 0xad87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad88 - 0xad8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad90 - 0xad97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xad98 - 0xad9f
		0x0000, 0x8fed, 0x8feb, 0x8fe4, 0x8fe8, 0x90ca, 0x90ce, 0x90c1, // 0xada0 - 0xada7
		0x90c3, 0x914b, 0x914a, 0x91cd, 0x9582, 0x9650, 0x964b, 0x964c, // 0xada8 - 0xadaf
		0x964d, 0x9762, 0x9769, 0x97cb, 0x97ed, 0x97f3, 0x9801, 0x98a8, // 0xadb0 - 0xadb7
		0x98db, 0x98df, 0x9996, 0x9999, 0x4e58, 0x4eb3, 0x500c, 0x500d, // 0xadb8 - 0xadbf
		0x5023, 0x4fef, 0x5026, 0x5025, 0x4ff8, 0x5029, 0x5016, 0x5006, // 0xadc0 - 0xadc7
		0x503c, 0x501f, 0x501a, 0x5012, 0x5011, 0x4ffa, 0x5000, 0x5014, // 0xadc8 - 0xadcf
		0x5028, 0x4ff1, 0x5021, 0x500b, 0x5019, 0x5018, 0x4ff3, 0x4fee, // 0xadd0 - 0xadd7
		0x502d, 0x502a, 0x4ffe, 0x502b, 0x5009, 0x517c, 0x51a4, 0x51a5, // 0xadd8 - 0xaddf
		0x51a2, 0x51cd, 0x51cc, 0x51c6, 0x51cb, 0x5256, 0x525c, 0x5254, // 0xade0 - 0xade7
		0x525b, 0x525d, 0x532a, 0x537f, 0x539f, 0x539d, 0x53df, 0x54e8, // 0xade8 - 0xadef
		0x5510, 0x5501, 0x5537, 0x54fc, 0x54e5, 0x54f2, 0x5506, 0x54fa, // 0xadf0 - 0xadf7
		0x5514, 0x54e9, 0x54ed, 0x54e1, 0x5509, 0x54ee, 0x54ea, 0x0000, // 0xadf8 - 0xadff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae00 - 0xae07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae08 - 0xae0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae10 - 0xae17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae18 - 0xae1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae20 - 0xae27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae28 - 0xae2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae30 - 0xae37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae38 - 0xae3f
		0x54e6, 0x5527, 0x5507, 0x54fd, 0x550f, 0x5703, 0x5704, 0x57c2, // 0xae40 - 0xae47
		0x57d4, 0x57cb, 0x57c3, 0x5809, 0x590f, 0x5957, 0x5958, 0x595a, // 0xae48 - 0xae4f
		0x5a11, 0x5a18, 0x5a1c, 0x5a1f, 0x5a1b, 0x5a13, 0x59ec, 0x5a20, // 0xae50 - 0xae57
		0x5a23, 0x5a29, 0x5a25, 0x5a0c, 0x5a09, 0x5b6b, 0x5c58, 0x5bb0, // 0xae58 - 0xae5f
		0x5bb3, 0x5bb6, 0x5bb4, 0x5bae, 0x5bb5, 0x5bb9, 0x5bb8, 0x5c04, // 0xae60 - 0xae67
		0x5c51, 0x5c55, 0x5c50, 0x5ced, 0x5cfd, 0x5cfb, 0x5cea, 0x5ce8, // 0xae68 - 0xae6f
		0x5cf0, 0x5cf6, 0x5d01, 0x5cf4, 0x5dee, 0x5e2d, 0x5e2b, 0x5eab, // 0xae70 - 0xae77
		0x5ead, 0x5ea7, 0x5f31, 0x5f92, 0x5f91, 0x5f90, 0x6059, 0x0000, // 0xae78 - 0xae7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae80 - 0xae87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae88 - 0xae8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae90 - 0xae97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xae98 - 0xae9f
		0x0000, 0x6063, 0x6065, 0x6050, 0x6055, 0x606d, 0x6069, 0x606f, // 0xaea0 - 0xaea7
		0x6084, 0x609f, 0x609a, 0x608d, 0x6094, 0x608c, 0x6085, 0x6096, // 0xaea8 - 0xaeaf
		0x6247, 0x62f3, 0x6308, 0x62ff, 0x634e, 0x633e, 0x632f, 0x6355, // 0xaeb0 - 0xaeb7
		0x6342, 0x6346, 0x634f, 0x6349, 0x633a, 0x6350, 0x633d, 0x632a, // 0xaeb8 - 0xaebf
		0x632b, 0x6328, 0x634d, 0x634c, 0x6548, 0x6549, 0x6599, 0x65c1, // 0xaec0 - 0xaec7
		0x65c5, 0x6642, 0x6649, 0x664f, 0x6643, 0x6652, 0x664c, 0x6645, // 0xaec8 - 0xaecf
		0x6641, 0x66f8, 0x6714, 0x6715, 0x6717, 0x6821, 0x6838, 0x6848, // 0xaed0 - 0xaed7
		0x6846, 0x6853, 0x6839, 0x6842, 0x6854, 0x6829, 0x68b3, 0x6817, // 0xaed8 - 0xaedf
		0x684c, 0x6851, 0x683d, 0x67f4, 0x6850, 0x6840, 0x683c, 0x6843, // 0xaee0 - 0xaee7
		0x682a, 0x6845, 0x6813, 0x6818, 0x6841, 0x6b8a, 0x6b89, 0x6bb7, // 0xaee8 - 0xaeef
		0x6c23, 0x6c27, 0x6c28, 0x6c26, 0x6c24, 0x6cf0, 0x6d6a, 0x6d95, // 0xaef0 - 0xaef7
		0x6d88, 0x6d87, 0x6d66, 0x6d78, 0x6d77, 0x6d59, 0x6d93, 0x0000, // 0xaef8 - 0xaeff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf00 - 0xaf07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf08 - 0xaf0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf10 - 0xaf17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf18 - 0xaf1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf20 - 0xaf27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf28 - 0xaf2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf30 - 0xaf37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf38 - 0xaf3f
		0x6d6c, 0x6d89, 0x6d6e, 0x6d5a, 0x6d74, 0x6d69, 0x6d8c, 0x6d8a, // 0xaf40 - 0xaf47
		0x6d79, 0x6d85, 0x6d65, 0x6d94, 0x70ca, 0x70d8, 0x70e4, 0x70d9, // 0xaf48 - 0xaf4f
		0x70c8, 0x70cf, 0x7239, 0x7279, 0x72fc, 0x72f9, 0x72fd, 0x72f8, // 0xaf50 - 0xaf57
		0x72f7, 0x7386, 0x73ed, 0x7409, 0x73ee, 0x73e0, 0x73ea, 0x73de, // 0xaf58 - 0xaf5f
		0x7554, 0x755d, 0x755c, 0x755a, 0x7559, 0x75be, 0x75c5, 0x75c7, // 0xaf60 - 0xaf67
		0x75b2, 0x75b3, 0x75bd, 0x75bc, 0x75b9, 0x75c2, 0x75b8, 0x768b, // 0xaf68 - 0xaf6f
		0x76b0, 0x76ca, 0x76cd, 0x76ce, 0x7729, 0x771f, 0x7720, 0x7728, // 0xaf70 - 0xaf77
		0x77e9, 0x7830, 0x7827, 0x7838, 0x781d, 0x7834, 0x7837, 0x0000, // 0xaf78 - 0xaf7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf80 - 0xaf87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf88 - 0xaf8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf90 - 0xaf97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xaf98 - 0xaf9f
		0x0000, 0x7825, 0x782d, 0x7820, 0x781f, 0x7832, 0x7955, 0x7950, // 0xafa0 - 0xafa7
		0x7960, 0x795f, 0x7956, 0x795e, 0x795d, 0x7957, 0x795a, 0x79e4, // 0xafa8 - 0xafaf
		0x79e3, 0x79e7, 0x79df, 0x79e6, 0x79e9, 0x79d8, 0x7a84, 0x7a88, // 0xafb0 - 0xafb7
		0x7ad9, 0x7b06, 0x7b11, 0x7c89, 0x7d21, 0x7d17, 0x7d0b, 0x7d0a, // 0xafb8 - 0xafbf
		0x7d20, 0x7d22, 0x7d14, 0x7d10, 0x7d15, 0x7d1a, 0x7d1c, 0x7d0d, // 0xafc0 - 0xafc7
		0x7d19, 0x7d1b, 0x7f3a, 0x7f5f, 0x7f94, 0x7fc5, 0x7fc1, 0x8006, // 0xafc8 - 0xafcf
		0x8018, 0x8015, 0x8019, 0x8017, 0x803d, 0x803f, 0x80f1, 0x8102, // 0xafd0 - 0xafd7
		0x80f0, 0x8105, 0x80ed, 0x80f4, 0x8106, 0x80f8, 0x80f3, 0x8108, // 0xafd8 - 0xafdf
		0x80fd, 0x810a, 0x80fc, 0x80ef, 0x81ed, 0x81ec, 0x8200, 0x8210, // 0xafe0 - 0xafe7
		0x822a, 0x822b, 0x8228, 0x822c, 0x82bb, 0x832b, 0x8352, 0x8354, // 0xafe8 - 0xafef
		0x834a, 0x8338, 0x8350, 0x8349, 0x8335, 0x8334, 0x834f, 0x8332, // 0xaff0 - 0xaff7
		0x8339, 0x8336, 0x8317, 0x8340, 0x8331, 0x8328, 0x8343, 0x0000, // 0xaff8 - 0xafff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb000 - 0xb007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb008 - 0xb00f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb010 - 0xb017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb018 - 0xb01f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb020 - 0xb027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb028 - 0xb02f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb030 - 0xb037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb038 - 0xb03f
		0x8654, 0x868a, 0x86aa, 0x8693, 0x86a4, 0x86a9, 0x868c, 0x86a3, // 0xb040 - 0xb047
		0x869c, 0x8870, 0x8877, 0x8881, 0x8882, 0x887d, 0x8879, 0x8a18, // 0xb048 - 0xb04f
		0x8a10, 0x8a0e, 0x8a0c, 0x8a15, 0x8a0a, 0x8a17, 0x8a13, 0x8a16, // 0xb050 - 0xb057
		0x8a0f, 0x8a11, 0x8c48, 0x8c7a, 0x8c79, 0x8ca1, 0x8ca2, 0x8d77, // 0xb058 - 0xb05f
		0x8eac, 0x8ed2, 0x8ed4, 0x8ecf, 0x8fb1, 0x9001, 0x9006, 0x8ff7, // 0xb060 - 0xb067
		0x9000, 0x8ffa, 0x8ff4, 0x9003, 0x8ffd, 0x9005, 0x8ff8, 0x9095, // 0xb068 - 0xb06f
		0x90e1, 0x90dd, 0x90e2, 0x9152, 0x914d, 0x914c, 0x91d8, 0x91dd, // 0xb070 - 0xb077
		0x91d7, 0x91dc, 0x91d9, 0x9583, 0x9662, 0x9663, 0x9661, 0x0000, // 0xb078 - 0xb07f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb080 - 0xb087
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb088 - 0xb08f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb090 - 0xb097
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb098 - 0xb09f
		0x0000, 0x554a, 0x963f, 0x57c3, 0x6328, 0x54ce, 0x5509, 0x54c0, // 0xb0a0 - 0xb0a7
		0x7691, 0x764c, 0x853c, 0x77ee, 0x827e, 0x788d, 0x7231, 0x9698, // 0xb0a8 - 0xb0af
		0x978d, 0x6c28, 0x5b89, 0x4ffa, 0x6309, 0x6697, 0x5cb8, 0x80fa, // 0xb0b0 - 0xb0b7
		0x6848, 0x80ae, 0x6602, 0x76ce, 0x51f9, 0x6556, 0x71ac, 0x7ff1, // 0xb0b8 - 0xb0bf
		0x8884, 0x50b2, 0x5965, 0x61ca, 0x6fb3, 0x82ad, 0x634c, 0x6252, // 0xb0c0 - 0xb0c7
		0x53ed, 0x5427, 0x7b06, 0x516b, 0x75a4, 0x5df4, 0x62d4, 0x8dcb, // 0xb0c8 - 0xb0cf
		0x9776, 0x628a, 0x8019, 0x575d, 0x9738, 0x7f62, 0x7238, 0x767d, // 0xb0d0 - 0xb0d7
		0x67cf, 0x767e, 0x6446, 0x4f70, 0x8d25, 0x62dc, 0x7a17, 0x6591, // 0xb0d8 - 0xb0df
		0x73ed, 0x642c, 0x6273, 0x822c, 0x9881, 0x677f, 0x7248, 0x626e, // 0xb0e0 - 0xb0e7
		0x62cc, 0x4f34, 0x74e3, 0x534a, 0x529e, 0x7eca, 0x90a6, 0x5e2e, // 0xb0e8 - 0xb0ef
		0x6886, 0x699c, 0x8180, 0x7ed1, 0x68d2, 0x78c5, 0x868c, 0x9551, // 0xb0f0 - 0xb0f7
		0x508d, 0x8c24, 0x82de, 0x80de, 0x5305, 0x8912, 0x5265, 0x0000, // 0xb0f8 - 0xb0ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb100 - 0xb107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb108 - 0xb10f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb110 - 0xb117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb118 - 0xb11f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb120 - 0xb127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb128 - 0xb12f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb130 - 0xb137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb138 - 0xb13f
		0x5a3c, 0x5a62, 0x5a5a, 0x5a46, 0x5a4a, 0x5b70, 0x5bc7, 0x5bc5, // 0xb140 - 0xb147
		0x5bc4, 0x5bc2, 0x5bbf, 0x5bc6, 0x5c09, 0x5c08, 0x5c07, 0x5c60, // 0xb148 - 0xb14f
		0x5c5c, 0x5c5d, 0x5d07, 0x5d06, 0x5d0e, 0x5d1b, 0x5d16, 0x5d22, // 0xb150 - 0xb157
		0x5d11, 0x5d29, 0x5d14, 0x5d19, 0x5d24, 0x5d27, 0x5d17, 0x5de2, // 0xb158 - 0xb15f
		0x5e38, 0x5e36, 0x5e33, 0x5e37, 0x5eb7, 0x5eb8, 0x5eb6, 0x5eb5, // 0xb160 - 0xb167
		0x5ebe, 0x5f35, 0x5f37, 0x5f57, 0x5f6c, 0x5f69, 0x5f6b, 0x5f97, // 0xb168 - 0xb16f
		0x5f99, 0x5f9e, 0x5f98, 0x5fa1, 0x5fa0, 0x5f9c, 0x607f, 0x60a3, // 0xb170 - 0xb177
		0x6089, 0x60a0, 0x60a8, 0x60cb, 0x60b4, 0x60e6, 0x60bd, 0x0000, // 0xb178 - 0xb17f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb180 - 0xb187
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb188 - 0xb18f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb190 - 0xb197
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb198 - 0xb19f
		0x0000, 0x8584, 0x96f9, 0x4fdd, 0x5821, 0x9971, 0x5b9d, 0x62b1, // 0xb1a0 - 0xb1a7
		0x62a5, 0x66b4, 0x8c79, 0x9c8d, 0x7206, 0x676f, 0x7891, 0x60b2, // 0xb1a8 - 0xb1af
		0x5351, 0x5317, 0x8f88, 0x80cc, 0x8d1d, 0x94a1, 0x500d, 0x72c8, // 0xb1b0 - 0xb1b7
		0x5907, 0x60eb, 0x7119, 0x88ab, 0x5954, 0x82ef, 0x672c, 0x7b28, // 0xb1b8 - 0xb1bf
		0x5d29, 0x7ef7, 0x752d, 0x6cf5, 0x8e66, 0x8ff8, 0x903c, 0x9f3b, // 0xb1c0 - 0xb1c7
		0x6bd4, 0x9119, 0x7b14, 0x5f7c, 0x78a7, 0x84d6, 0x853d, 0x6bd5, // 0xb1c8 - 0xb1cf
		0x6bd9, 0x6bd6, 0x5e01, 0x5e87, 0x75f9, 0x95ed, 0x655d, 0x5f0a, // 0xb1d0 - 0xb1d7
		0x5fc5, 0x8f9f, 0x58c1, 0x81c2, 0x907f, 0x965b, 0x97ad, 0x8fb9, // 0xb1d8 - 0xb1df
		0x7f16, 0x8d2c, 0x6241, 0x4fbf, 0x53d8, 0x535e, 0x8fa8, 0x8fa9, // 0xb1e0 - 0xb1e7
		0x8fab, 0x904d, 0x6807, 0x5f6a, 0x8198, 0x8868, 0x9cd6, 0x618b, // 0xb1e8 - 0xb1ef
		0x522b, 0x762a, 0x5f6c, 0x658c, 0x6fd2, 0x6ee8, 0x5bbe, 0x6448, // 0xb1f0 - 0xb1f7
		0x5175, 0x51b0, 0x67c4, 0x4e19, 0x79c9, 0x997c, 0x70b3, 0x0000, // 0xb1f8 - 0xb1ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb200 - 0xb207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb208 - 0xb20f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb210 - 0xb217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb218 - 0xb21f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb220 - 0xb227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb228 - 0xb22f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb230 - 0xb237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb238 - 0xb23f
		0x6beb, 0x6bec, 0x6c2b, 0x6d8e, 0x6dbc, 0x6df3, 0x6dd9, 0x6db2, // 0xb240 - 0xb247
		0x6de1, 0x6dcc, 0x6de4, 0x6dfb, 0x6dfa, 0x6e05, 0x6dc7, 0x6dcb, // 0xb248 - 0xb24f
		0x6daf, 0x6dd1, 0x6dae, 0x6dde, 0x6df9, 0x6db8, 0x6df7, 0x6df5, // 0xb250 - 0xb257
		0x6dc5, 0x6dd2, 0x6e1a, 0x6db5, 0x6dda, 0x6deb, 0x6dd8, 0x6dea, // 0xb258 - 0xb25f
		0x6df1, 0x6dee, 0x6de8, 0x6dc6, 0x6dc4, 0x6daa, 0x6dec, 0x6dbf, // 0xb260 - 0xb267
		0x6de6, 0x70f9, 0x7109, 0x710a, 0x70fd, 0x70ef, 0x723d, 0x727d, // 0xb268 - 0xb26f
		0x7281, 0x731c, 0x731b, 0x7316, 0x7313, 0x7319, 0x7387, 0x7405, // 0xb270 - 0xb277
		0x740a, 0x7403, 0x7406, 0x73fe, 0x740d, 0x74e0, 0x74f6, 0x0000, // 0xb278 - 0xb27f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb280 - 0xb287
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb288 - 0xb28f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb290 - 0xb297
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb298 - 0xb29f
		0x0000, 0x75c5, 0x5e76, 0x73bb, 0x83e0, 0x64ad, 0x62e8, 0x94b5, // 0xb2a0 - 0xb2a7
		0x6ce2, 0x535a, 0x52c3, 0x640f, 0x94c2, 0x7b94, 0x4f2f, 0x5e1b, // 0xb2a8 - 0xb2af
		0x8236, 0x8116, 0x818a, 0x6e24, 0x6cca, 0x9a73, 0x6355, 0x535c, // 0xb2b0 - 0xb2b7
		0x54fa, 0x8865, 0x57e0, 0x4e0d, 0x5e03, 0x6b65, 0x7c3f, 0x90e8, // 0xb2b8 - 0xb2bf
		0x6016, 0x64e6, 0x731c, 0x88c1, 0x6750, 0x624d, 0x8d22, 0x776c, // 0xb2c0 - 0xb2c7
		0x8e29, 0x91c7, 0x5f69, 0x83dc, 0x8521, 0x9910, 0x53c2, 0x8695, // 0xb2c8 - 0xb2cf
		0x6b8b, 0x60ed, 0x60e8, 0x707f, 0x82cd, 0x8231, 0x4ed3, 0x6ca7, // 0xb2d0 - 0xb2d7
		0x85cf, 0x64cd, 0x7cd9, 0x69fd, 0x66f9, 0x8349, 0x5395, 0x7b56, // 0xb2d8 - 0xb2df
		0x4fa7, 0x518c, 0x6d4b, 0x5c42, 0x8e6d, 0x63d2, 0x53c9, 0x832c, // 0xb2e0 - 0xb2e7
		0x8336, 0x67e5, 0x78b4, 0x643d, 0x5bdf, 0x5c94, 0x5dee, 0x8be7, // 0xb2e8 - 0xb2ef
		0x62c6, 0x67f4, 0x8c7a, 0x6400, 0x63ba, 0x8749, 0x998b, 0x8c17, // 0xb2f0 - 0xb2f7
		0x7f20, 0x94f2, 0x4ea7, 0x9610, 0x98a4, 0x660c, 0x7316, 0x0000, // 0xb2f8 - 0xb2ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb300 - 0xb307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb308 - 0xb30f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb310 - 0xb317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb318 - 0xb31f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb320 - 0xb327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb328 - 0xb32f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb330 - 0xb337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb338 - 0xb33f
		0x8386, 0x83a7, 0x8655, 0x5f6a, 0x86c7, 0x86c0, 0x86b6, 0x86c4, // 0xb340 - 0xb347
		0x86b5, 0x86c6, 0x86cb, 0x86b1, 0x86af, 0x86c9, 0x8853, 0x889e, // 0xb348 - 0xb34f
		0x8888, 0x88ab, 0x8892, 0x8896, 0x888d, 0x888b, 0x8993, 0x898f, // 0xb350 - 0xb357
		0x8a2a, 0x8a1d, 0x8a23, 0x8a25, 0x8a31, 0x8a2d, 0x8a1f, 0x8a1b, // 0xb358 - 0xb35f
		0x8a22, 0x8c49, 0x8c5a, 0x8ca9, 0x8cac, 0x8cab, 0x8ca8, 0x8caa, // 0xb360 - 0xb367
		0x8ca7, 0x8d67, 0x8d66, 0x8dbe, 0x8dba, 0x8edb, 0x8edf, 0x9019, // 0xb368 - 0xb36f
		0x900d, 0x901a, 0x9017, 0x9023, 0x901f, 0x901d, 0x9010, 0x9015, // 0xb370 - 0xb377
		0x901e, 0x9020, 0x900f, 0x9022, 0x9016, 0x901b, 0x9014, 0x0000, // 0xb378 - 0xb37f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb380 - 0xb387
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb388 - 0xb38f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb390 - 0xb397
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb398 - 0xb39f
		0x0000, 0x573a, 0x5c1d, 0x5e38, 0x957f, 0x507f, 0x80a0, 0x5382, // 0xb3a0 - 0xb3a7
		0x655e, 0x7545, 0x5531, 0x5021, 0x8d85, 0x6284, 0x949e, 0x671d, // 0xb3a8 - 0xb3af
		0x5632, 0x6f6e, 0x5de2, 0x5435, 0x7092, 0x8f66, 0x626f, 0x64a4, // 0xb3b0 - 0xb3b7
		0x63a3, 0x5f7b, 0x6f88, 0x90f4, 0x81e3, 0x8fb0, 0x5c18, 0x6668, // 0xb3b8 - 0xb3bf
		0x5ff1, 0x6c89, 0x9648, 0x8d81, 0x886c, 0x6491, 0x79f0, 0x57ce, // 0xb3c0 - 0xb3c7
		0x6a59, 0x6210, 0x5448, 0x4e58, 0x7a0b, 0x60e9, 0x6f84, 0x8bda, // 0xb3c8 - 0xb3cf
		0x627f, 0x901e, 0x9a8b, 0x79e4, 0x5403, 0x75f4, 0x6301, 0x5319, // 0xb3d0 - 0xb3d7
		0x6c60, 0x8fdf, 0x5f1b, 0x9a70, 0x803b, 0x9f7f, 0x4f88, 0x5c3a, // 0xb3d8 - 0xb3df
		0x8d64, 0x7fc5, 0x65a5, 0x70bd, 0x5145, 0x51b2, 0x866b, 0x5d07, // 0xb3e0 - 0xb3e7
		0x5ba0, 0x62bd, 0x916c, 0x7574, 0x8e0c, 0x7a20, 0x6101, 0x7b79, // 0xb3e8 - 0xb3ef
		0x4ec7, 0x7ef8, 0x7785, 0x4e11, 0x81ed, 0x521d, 0x51fa, 0x6a71, // 0xb3f0 - 0xb3f7
		0x53a8, 0x8e87, 0x9504, 0x96cf, 0x6ec1, 0x9664, 0x695a, 0x0000, // 0xb3f8 - 0xb3ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb400 - 0xb407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb408 - 0xb40f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb410 - 0xb417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb418 - 0xb41f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb420 - 0xb427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb428 - 0xb42f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb430 - 0xb437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb438 - 0xb43f
		0x5a77, 0x5a9a, 0x5a7f, 0x5a92, 0x5a9b, 0x5aa7, 0x5b73, 0x5b71, // 0xb440 - 0xb447
		0x5bd2, 0x5bcc, 0x5bd3, 0x5bd0, 0x5c0a, 0x5c0b, 0x5c31, 0x5d4c, // 0xb448 - 0xb44f
		0x5d50, 0x5d34, 0x5d47, 0x5dfd, 0x5e45, 0x5e3d, 0x5e40, 0x5e43, // 0xb450 - 0xb457
		0x5e7e, 0x5eca, 0x5ec1, 0x5ec2, 0x5ec4, 0x5f3c, 0x5f6d, 0x5fa9, // 0xb458 - 0xb45f
		0x5faa, 0x5fa8, 0x60d1, 0x60e1, 0x60b2, 0x60b6, 0x60e0, 0x611c, // 0xb460 - 0xb467
		0x6123, 0x60fa, 0x6115, 0x60f0, 0x60fb, 0x60f4, 0x6168, 0x60f1, // 0xb468 - 0xb46f
		0x610e, 0x60f6, 0x6109, 0x6100, 0x6112, 0x621f, 0x6249, 0x63a3, // 0xb470 - 0xb477
		0x638c, 0x63cf, 0x63c0, 0x63e9, 0x63c9, 0x63c6, 0x63cd, 0x0000, // 0xb478 - 0xb47f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb480 - 0xb487
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb488 - 0xb48f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb490 - 0xb497
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb498 - 0xb49f
		0x0000, 0x7840, 0x50a8, 0x77d7, 0x6410, 0x89e6, 0x5904, 0x63e3, // 0xb4a0 - 0xb4a7
		0x5ddd, 0x7a7f, 0x693d, 0x4f20, 0x8239, 0x5598, 0x4e32, 0x75ae, // 0xb4a8 - 0xb4af
		0x7a97, 0x5e62, 0x5e8a, 0x95ef, 0x521b, 0x5439, 0x708a, 0x6376, // 0xb4b0 - 0xb4b7
		0x9524, 0x5782, 0x6625, 0x693f, 0x9187, 0x5507, 0x6df3, 0x7eaf, // 0xb4b8 - 0xb4bf
		0x8822, 0x6233, 0x7ef0, 0x75b5, 0x8328, 0x78c1, 0x96cc, 0x8f9e, // 0xb4c0 - 0xb4c7
		0x6148, 0x74f7, 0x8bcd, 0x6b64, 0x523a, 0x8d50, 0x6b21, 0x806a, // 0xb4c8 - 0xb4cf
		0x8471, 0x56f1, 0x5306, 0x4ece, 0x4e1b, 0x51d1, 0x7c97, 0x918b, // 0xb4d0 - 0xb4d7
		0x7c07, 0x4fc3, 0x8e7f, 0x7be1, 0x7a9c, 0x6467, 0x5d14, 0x50ac, // 0xb4d8 - 0xb4df
		0x8106, 0x7601, 0x7cb9, 0x6dec, 0x7fe0, 0x6751, 0x5b58, 0x5bf8, // 0xb4e0 - 0xb4e7
		0x78cb, 0x64ae, 0x6413, 0x63aa, 0x632b, 0x9519, 0x642d, 0x8fbe, // 0xb4e8 - 0xb4ef
		0x7b54, 0x7629, 0x6253, 0x5927, 0x5446, 0x6b79, 0x50a3, 0x6234, // 0xb4f0 - 0xb4f7
		0x5e26, 0x6b86, 0x4ee3, 0x8d37, 0x888b, 0x5f85, 0x902e, 0x0000, // 0xb4f8 - 0xb4ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb500 - 0xb507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb508 - 0xb50f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb510 - 0xb517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb518 - 0xb51f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb520 - 0xb527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb528 - 0xb52f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb530 - 0xb537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb538 - 0xb53f
		0x6e89, 0x6e19, 0x6e4e, 0x6e63, 0x6e44, 0x6e72, 0x6e69, 0x6e5f, // 0xb540 - 0xb547
		0x7119, 0x711a, 0x7126, 0x7130, 0x7121, 0x7136, 0x716e, 0x711c, // 0xb548 - 0xb54f
		0x724c, 0x7284, 0x7280, 0x7336, 0x7325, 0x7334, 0x7329, 0x743a, // 0xb550 - 0xb557
		0x742a, 0x7433, 0x7422, 0x7425, 0x7435, 0x7436, 0x7434, 0x742f, // 0xb558 - 0xb55f
		0x741b, 0x7426, 0x7428, 0x7525, 0x7526, 0x756b, 0x756a, 0x75e2, // 0xb560 - 0xb567
		0x75db, 0x75e3, 0x75d9, 0x75d8, 0x75de, 0x75e0, 0x767b, 0x767c, // 0xb568 - 0xb56f
		0x7696, 0x7693, 0x76b4, 0x76dc, 0x774f, 0x77ed, 0x785d, 0x786c, // 0xb570 - 0xb577
		0x786f, 0x7a0d, 0x7a08, 0x7a0b, 0x7a05, 0x7a00, 0x7a98, 0x0000, // 0xb578 - 0xb57f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb580 - 0xb587
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb588 - 0xb58f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb590 - 0xb597
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb598 - 0xb59f
		0x0000, 0x6020, 0x803d, 0x62c5, 0x4e39, 0x5355, 0x90f8, 0x63b8, // 0xb5a0 - 0xb5a7
		0x80c6, 0x65e6, 0x6c2e, 0x4f46, 0x60ee, 0x6de1, 0x8bde, 0x5f39, // 0xb5a8 - 0xb5af
		0x86cb, 0x5f53, 0x6321, 0x515a, 0x8361, 0x6863, 0x5200, 0x6363, // 0xb5b0 - 0xb5b7
		0x8e48, 0x5012, 0x5c9b, 0x7977, 0x5bfc, 0x5230, 0x7a3b, 0x60bc, // 0xb5b8 - 0xb5bf
		0x9053, 0x76d7, 0x5fb7, 0x5f97, 0x7684, 0x8e6c, 0x706f, 0x767b, // 0xb5c0 - 0xb5c7
		0x7b49, 0x77aa, 0x51f3, 0x9093, 0x5824, 0x4f4e, 0x6ef4, 0x8fea, // 0xb5c8 - 0xb5cf
		0x654c, 0x7b1b, 0x72c4, 0x6da4, 0x7fdf, 0x5ae1, 0x62b5, 0x5e95, // 0xb5d0 - 0xb5d7
		0x5730, 0x8482, 0x7b2c, 0x5e1d, 0x5f1f, 0x9012, 0x7f14, 0x98a0, // 0xb5d8 - 0xb5df
		0x6382, 0x6ec7, 0x7898, 0x70b9, 0x5178, 0x975b, 0x57ab, 0x7535, // 0xb5e0 - 0xb5e7
		0x4f43, 0x7538, 0x5e97, 0x60e6, 0x5960, 0x6dc0, 0x6bbf, 0x7889, // 0xb5e8 - 0xb5ef
		0x53fc, 0x96d5, 0x51cb, 0x5201, 0x6389, 0x540a, 0x9493, 0x8c03, // 0xb5f0 - 0xb5f7
		0x8dcc, 0x7239, 0x789f, 0x8776, 0x8fed, 0x8c0d, 0x53e0, 0x0000, // 0xb5f8 - 0xb5ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb600 - 0xb607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb608 - 0xb60f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb610 - 0xb617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb618 - 0xb61f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb620 - 0xb627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb628 - 0xb62f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb630 - 0xb637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb638 - 0xb63f
		0x8a54, 0x8a5b, 0x8a50, 0x8a46, 0x8a34, 0x8a3a, 0x8a36, 0x8a56, // 0xb640 - 0xb647
		0x8c61, 0x8c82, 0x8caf, 0x8cbc, 0x8cb3, 0x8cbd, 0x8cc1, 0x8cbb, // 0xb648 - 0xb64f
		0x8cc0, 0x8cb4, 0x8cb7, 0x8cb6, 0x8cbf, 0x8cb8, 0x8d8a, 0x8d85, // 0xb650 - 0xb657
		0x8d81, 0x8dce, 0x8ddd, 0x8dcb, 0x8dda, 0x8dd1, 0x8dcc, 0x8ddb, // 0xb658 - 0xb65f
		0x8dc6, 0x8efb, 0x8ef8, 0x8efc, 0x8f9c, 0x902e, 0x9035, 0x9031, // 0xb660 - 0xb667
		0x9038, 0x9032, 0x9036, 0x9102, 0x90f5, 0x9109, 0x90fe, 0x9163, // 0xb668 - 0xb66f
		0x9165, 0x91cf, 0x9214, 0x9215, 0x9223, 0x9209, 0x921e, 0x920d, // 0xb670 - 0xb677
		0x9210, 0x9207, 0x9211, 0x9594, 0x958f, 0x958b, 0x9591, 0x0000, // 0xb678 - 0xb67f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb680 - 0xb687
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb688 - 0xb68f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb690 - 0xb697
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb698 - 0xb69f
		0x0000, 0x4e01, 0x76ef, 0x53ee, 0x9489, 0x9876, 0x9f0e, 0x952d, // 0xb6a0 - 0xb6a7
		0x5b9a, 0x8ba2, 0x4e22, 0x4e1c, 0x51ac, 0x8463, 0x61c2, 0x52a8, // 0xb6a8 - 0xb6af
		0x680b, 0x4f97, 0x606b, 0x51bb, 0x6d1e, 0x515c, 0x6296, 0x6597, // 0xb6b0 - 0xb6b7
		0x9661, 0x8c46, 0x9017, 0x75d8, 0x90fd, 0x7763, 0x6bd2, 0x728a, // 0xb6b8 - 0xb6bf
		0x72ec, 0x8bfb, 0x5835, 0x7779, 0x8d4c, 0x675c, 0x9540, 0x809a, // 0xb6c0 - 0xb6c7
		0x5ea6, 0x6e21, 0x5992, 0x7aef, 0x77ed, 0x953b, 0x6bb5, 0x65ad, // 0xb6c8 - 0xb6cf
		0x7f0e, 0x5806, 0x5151, 0x961f, 0x5bf9, 0x58a9, 0x5428, 0x8e72, // 0xb6d0 - 0xb6d7
		0x6566, 0x987f, 0x56e4, 0x949d, 0x76fe, 0x9041, 0x6387, 0x54c6, // 0xb6d8 - 0xb6df
		0x591a, 0x593a, 0x579b, 0x8eb2, 0x6735, 0x8dfa, 0x8235, 0x5241, // 0xb6e0 - 0xb6e7
		0x60f0, 0x5815, 0x86fe, 0x5ce8, 0x9e45, 0x4fc4, 0x989d, 0x8bb9, // 0xb6e8 - 0xb6ef
		0x5a25, 0x6076, 0x5384, 0x627c, 0x904f, 0x9102, 0x997f, 0x6069, // 0xb6f0 - 0xb6f7
		0x800c, 0x513f, 0x8033, 0x5c14, 0x9975, 0x6d31, 0x4e8c, 0x0000, // 0xb6f8 - 0xb6ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb700 - 0xb707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb708 - 0xb70f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb710 - 0xb717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb718 - 0xb71f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb720 - 0xb727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb728 - 0xb72f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb730 - 0xb737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb738 - 0xb73f
		0x5ab3, 0x5ac2, 0x5ab2, 0x5d69, 0x5d6f, 0x5e4c, 0x5e79, 0x5ec9, // 0xb740 - 0xb747
		0x5ec8, 0x5f12, 0x5f59, 0x5fac, 0x5fae, 0x611a, 0x610f, 0x6148, // 0xb748 - 0xb74f
		0x611f, 0x60f3, 0x611b, 0x60f9, 0x6101, 0x6108, 0x614e, 0x614c, // 0xb750 - 0xb757
		0x6144, 0x614d, 0x613e, 0x6134, 0x6127, 0x610d, 0x6106, 0x6137, // 0xb758 - 0xb75f
		0x6221, 0x6222, 0x6413, 0x643e, 0x641e, 0x642a, 0x642d, 0x643d, // 0xb760 - 0xb767
		0x642c, 0x640f, 0x641c, 0x6414, 0x640d, 0x6436, 0x6416, 0x6417, // 0xb768 - 0xb76f
		0x6406, 0x656c, 0x659f, 0x65b0, 0x6697, 0x6689, 0x6687, 0x6688, // 0xb770 - 0xb777
		0x6696, 0x6684, 0x6698, 0x668d, 0x6703, 0x6994, 0x696d, 0x0000, // 0xb778 - 0xb77f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb780 - 0xb787
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb788 - 0xb78f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb790 - 0xb797
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb798 - 0xb79f
		0x0000, 0x8d30, 0x53d1, 0x7f5a, 0x7b4f, 0x4f10, 0x4e4f, 0x9600, // 0xb7a0 - 0xb7a7
		0x6cd5, 0x73d0, 0x85e9, 0x5e06, 0x756a, 0x7ffb, 0x6a0a, 0x77fe, // 0xb7a8 - 0xb7af
		0x9492, 0x7e41, 0x51e1, 0x70e6, 0x53cd, 0x8fd4, 0x8303, 0x8d29, // 0xb7b0 - 0xb7b7
		0x72af, 0x996d, 0x6cdb, 0x574a, 0x82b3, 0x65b9, 0x80aa, 0x623f, // 0xb7b8 - 0xb7bf
		0x9632, 0x59a8, 0x4eff, 0x8bbf, 0x7eba, 0x653e, 0x83f2, 0x975e, // 0xb7c0 - 0xb7c7
		0x5561, 0x98de, 0x80a5, 0x532a, 0x8bfd, 0x5420, 0x80ba, 0x5e9f, // 0xb7c8 - 0xb7cf
		0x6cb8, 0x8d39, 0x82ac, 0x915a, 0x5429, 0x6c1b, 0x5206, 0x7eb7, // 0xb7d0 - 0xb7d7
		0x575f, 0x711a, 0x6c7e, 0x7c89, 0x594b, 0x4efd, 0x5fff, 0x6124, // 0xb7d8 - 0xb7df
		0x7caa, 0x4e30, 0x5c01, 0x67ab, 0x8702, 0x5cf0, 0x950b, 0x98ce, // 0xb7e0 - 0xb7e7
		0x75af, 0x70fd, 0x9022, 0x51af, 0x7f1d, 0x8bbd, 0x5949, 0x51e4, // 0xb7e8 - 0xb7ef
		0x4f5b, 0x5426, 0x592b, 0x6577, 0x80a4, 0x5b75, 0x6276, 0x62c2, // 0xb7f0 - 0xb7f7
		0x8f90, 0x5e45, 0x6c1f, 0x7b26, 0x4f0f, 0x4fd8, 0x670d, 0x0000, // 0xb7f8 - 0xb7ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb800 - 0xb807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb808 - 0xb80f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb810 - 0xb817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb818 - 0xb81f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb820 - 0xb827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb828 - 0xb82f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb830 - 0xb837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb838 - 0xb83f
		0x7779, 0x776a, 0x776c, 0x775c, 0x7765, 0x7768, 0x7762, 0x77ee, // 0xb840 - 0xb847
		0x788e, 0x78b0, 0x7897, 0x7898, 0x788c, 0x7889, 0x787c, 0x7891, // 0xb848 - 0xb84f
		0x7893, 0x787f, 0x797a, 0x797f, 0x7981, 0x842c, 0x79bd, 0x7a1c, // 0xb850 - 0xb857
		0x7a1a, 0x7a20, 0x7a14, 0x7a1f, 0x7a1e, 0x7a9f, 0x7aa0, 0x7b77, // 0xb858 - 0xb85f
		0x7bc0, 0x7b60, 0x7b6e, 0x7b67, 0x7cb1, 0x7cb3, 0x7cb5, 0x7d93, // 0xb860 - 0xb867
		0x7d79, 0x7d91, 0x7d81, 0x7d8f, 0x7d5b, 0x7f6e, 0x7f69, 0x7f6a, // 0xb868 - 0xb86f
		0x7f72, 0x7fa9, 0x7fa8, 0x7fa4, 0x8056, 0x8058, 0x8086, 0x8084, // 0xb870 - 0xb877
		0x8171, 0x8170, 0x8178, 0x8165, 0x816e, 0x8173, 0x816b, 0x0000, // 0xb878 - 0xb87f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb880 - 0xb887
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb888 - 0xb88f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb890 - 0xb897
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb898 - 0xb89f
		0x0000, 0x6d6e, 0x6daa, 0x798f, 0x88b1, 0x5f17, 0x752b, 0x629a, // 0xb8a0 - 0xb8a7
		0x8f85, 0x4fef, 0x91dc, 0x65a7, 0x812f, 0x8151, 0x5e9c, 0x8150, // 0xb8a8 - 0xb8af
		0x8d74, 0x526f, 0x8986, 0x8d4b, 0x590d, 0x5085, 0x4ed8, 0x961c, // 0xb8b0 - 0xb8b7
		0x7236, 0x8179, 0x8d1f, 0x5bcc, 0x8ba3, 0x9644, 0x5987, 0x7f1a, // 0xb8b8 - 0xb8bf
		0x5490, 0x5676, 0x560e, 0x8be5, 0x6539, 0x6982, 0x9499, 0x76d6, // 0xb8c0 - 0xb8c7
		0x6e89, 0x5e72, 0x7518, 0x6746, 0x67d1, 0x7aff, 0x809d, 0x8d76, // 0xb8c8 - 0xb8cf
		0x611f, 0x79c6, 0x6562, 0x8d63, 0x5188, 0x521a, 0x94a2, 0x7f38, // 0xb8d0 - 0xb8d7
		0x809b, 0x7eb2, 0x5c97, 0x6e2f, 0x6760, 0x7bd9, 0x768b, 0x9ad8, // 0xb8d8 - 0xb8df
		0x818f, 0x7f94, 0x7cd5, 0x641e, 0x9550, 0x7a3f, 0x544a, 0x54e5, // 0xb8e0 - 0xb8e7
		0x6b4c, 0x6401, 0x6208, 0x9e3d, 0x80f3, 0x7599, 0x5272, 0x9769, // 0xb8e8 - 0xb8ef
		0x845b, 0x683c, 0x86e4, 0x9601, 0x9694, 0x94ec, 0x4e2a, 0x5404, // 0xb8f0 - 0xb8f7
		0x7ed9, 0x6839, 0x8ddf, 0x8015, 0x66f4, 0x5e9a, 0x7fb9, 0x0000, // 0xb8f8 - 0xb8ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb900 - 0xb907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb908 - 0xb90f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb910 - 0xb917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb918 - 0xb91f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb920 - 0xb927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb928 - 0xb92f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb930 - 0xb937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb938 - 0xb93f
		0x8f9f, 0x8fb2, 0x904b, 0x904a, 0x9053, 0x9042, 0x9054, 0x903c, // 0xb940 - 0xb947
		0x9055, 0x9050, 0x9047, 0x904f, 0x904e, 0x904d, 0x9051, 0x903e, // 0xb948 - 0xb94f
		0x9041, 0x9112, 0x9117, 0x916c, 0x916a, 0x9169, 0x91c9, 0x9237, // 0xb950 - 0xb957
		0x9257, 0x9238, 0x923d, 0x9240, 0x923e, 0x925b, 0x924b, 0x9264, // 0xb958 - 0xb95f
		0x9251, 0x9234, 0x9249, 0x924d, 0x9245, 0x9239, 0x923f, 0x925a, // 0xb960 - 0xb967
		0x9598, 0x9698, 0x9694, 0x9695, 0x96cd, 0x96cb, 0x96c9, 0x96ca, // 0xb968 - 0xb96f
		0x96f7, 0x96fb, 0x96f9, 0x96f6, 0x9756, 0x9774, 0x9776, 0x9810, // 0xb970 - 0xb977
		0x9811, 0x9813, 0x980a, 0x9812, 0x980c, 0x98fc, 0x98f4, 0x0000, // 0xb978 - 0xb97f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb980 - 0xb987
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb988 - 0xb98f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb990 - 0xb997
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xb998 - 0xb99f
		0x0000, 0x57c2, 0x803f, 0x6897, 0x5de5, 0x653b, 0x529f, 0x606d, // 0xb9a0 - 0xb9a7
		0x9f9a, 0x4f9b, 0x8eac, 0x516c, 0x5bab, 0x5f13, 0x5de9, 0x6c5e, // 0xb9a8 - 0xb9af
		0x62f1, 0x8d21, 0x5171, 0x94a9, 0x52fe, 0x6c9f, 0x82df, 0x72d7, // 0xb9b0 - 0xb9b7
		0x57a2, 0x6784, 0x8d2d, 0x591f, 0x8f9c, 0x83c7, 0x5495, 0x7b8d, // 0xb9b8 - 0xb9bf
		0x4f30, 0x6cbd, 0x5b64, 0x59d1, 0x9f13, 0x53e4, 0x86ca, 0x9aa8, // 0xb9c0 - 0xb9c7
		0x8c37, 0x80a1, 0x6545, 0x987e, 0x56fa, 0x96c7, 0x522e, 0x74dc, // 0xb9c8 - 0xb9cf
		0x5250, 0x5be1, 0x6302, 0x8902, 0x4e56, 0x62d0, 0x602a, 0x68fa, // 0xb9d0 - 0xb9d7
		0x5173, 0x5b98, 0x51a0, 0x89c2, 0x7ba1, 0x9986, 0x7f50, 0x60ef, // 0xb9d8 - 0xb9df
		0x704c, 0x8d2f, 0x5149, 0x5e7f, 0x901b, 0x7470, 0x89c4, 0x572d, // 0xb9e0 - 0xb9e7
		0x7845, 0x5f52, 0x9f9f, 0x95fa, 0x8f68, 0x9b3c, 0x8be1, 0x7678, // 0xb9e8 - 0xb9ef
		0x6842, 0x67dc, 0x8dea, 0x8d35, 0x523d, 0x8f8a, 0x6eda, 0x68cd, // 0xb9f0 - 0xb9f7
		0x9505, 0x90ed, 0x56fd, 0x679c, 0x88f9, 0x8fc7, 0x54c8, 0x0000, // 0xb9f8 - 0xb9ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba00 - 0xba07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba08 - 0xba0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba10 - 0xba17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba18 - 0xba1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba20 - 0xba27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba28 - 0xba2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba30 - 0xba37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba38 - 0xba3f
		0x613f, 0x614b, 0x6177, 0x6162, 0x6163, 0x615f, 0x615a, 0x6158, // 0xba40 - 0xba47
		0x6175, 0x622a, 0x6487, 0x6458, 0x6454, 0x64a4, 0x6478, 0x645f, // 0xba48 - 0xba4f
		0x647a, 0x6451, 0x6467, 0x6434, 0x646d, 0x647b, 0x6572, 0x65a1, // 0xba50 - 0xba57
		0x65d7, 0x65d6, 0x66a2, 0x66a8, 0x669d, 0x699c, 0x69a8, 0x6995, // 0xba58 - 0xba5f
		0x69c1, 0x69ae, 0x69d3, 0x69cb, 0x699b, 0x69b7, 0x69bb, 0x69ab, // 0xba60 - 0xba67
		0x69b4, 0x69d0, 0x69cd, 0x69ad, 0x69cc, 0x69a6, 0x69c3, 0x69a3, // 0xba68 - 0xba6f
		0x6b49, 0x6b4c, 0x6c33, 0x6f33, 0x6f14, 0x6efe, 0x6f13, 0x6ef4, // 0xba70 - 0xba77
		0x6f29, 0x6f3e, 0x6f20, 0x6f2c, 0x6f0f, 0x6f02, 0x6f22, 0x0000, // 0xba78 - 0xba7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba80 - 0xba87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba88 - 0xba8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba90 - 0xba97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xba98 - 0xba9f
		0x0000, 0x9ab8, 0x5b69, 0x6d77, 0x6c26, 0x4ea5, 0x5bb3, 0x9a87, // 0xbaa0 - 0xbaa7
		0x9163, 0x61a8, 0x90af, 0x97e9, 0x542b, 0x6db5, 0x5bd2, 0x51fd, // 0xbaa8 - 0xbaaf
		0x558a, 0x7f55, 0x7ff0, 0x64bc, 0x634d, 0x65f1, 0x61be, 0x608d, // 0xbab0 - 0xbab7
		0x710a, 0x6c57, 0x6c49, 0x592f, 0x676d, 0x822a, 0x58d5, 0x568e, // 0xbab8 - 0xbabf
		0x8c6a, 0x6beb, 0x90dd, 0x597d, 0x8017, 0x53f7, 0x6d69, 0x5475, // 0xbac0 - 0xbac7
		0x559d, 0x8377, 0x83cf, 0x6838, 0x79be, 0x548c, 0x4f55, 0x5408, // 0xbac8 - 0xbacf
		0x76d2, 0x8c89, 0x9602, 0x6cb3, 0x6db8, 0x8d6b, 0x8910, 0x9e64, // 0xbad0 - 0xbad7
		0x8d3a, 0x563f, 0x9ed1, 0x75d5, 0x5f88, 0x72e0, 0x6068, 0x54fc, // 0xbad8 - 0xbadf
		0x4ea8, 0x6a2a, 0x8861, 0x6052, 0x8f70, 0x54c4, 0x70d8, 0x8679, // 0xbae0 - 0xbae7
		0x9e3f, 0x6d2a, 0x5b8f, 0x5f18, 0x7ea2, 0x5589, 0x4faf, 0x7334, // 0xbae8 - 0xbaef
		0x543c, 0x539a, 0x5019, 0x540e, 0x547c, 0x4e4e, 0x5ffd, 0x745a, // 0xbaf0 - 0xbaf7
		0x58f6, 0x846b, 0x80e1, 0x8774, 0x72d0, 0x7cca, 0x6e56, 0x0000, // 0xbaf8 - 0xbaff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb00 - 0xbb07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb08 - 0xbb0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb10 - 0xbb17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb18 - 0xbb1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb20 - 0xbb27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb28 - 0xbb2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb30 - 0xbb37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb38 - 0xbb3f
		0x7f70, 0x7fe0, 0x7fe1, 0x7fdf, 0x805e, 0x805a, 0x8087, 0x8150, // 0xbb40 - 0xbb47
		0x8180, 0x818f, 0x8188, 0x818a, 0x817f, 0x8182, 0x81e7, 0x81fa, // 0xbb48 - 0xbb4f
		0x8207, 0x8214, 0x821e, 0x824b, 0x84c9, 0x84bf, 0x84c6, 0x84c4, // 0xbb50 - 0xbb57
		0x8499, 0x849e, 0x84b2, 0x849c, 0x84cb, 0x84b8, 0x84c0, 0x84d3, // 0xbb58 - 0xbb5f
		0x8490, 0x84bc, 0x84d1, 0x84ca, 0x873f, 0x871c, 0x873b, 0x8722, // 0xbb60 - 0xbb67
		0x8725, 0x8734, 0x8718, 0x8755, 0x8737, 0x8729, 0x88f3, 0x8902, // 0xbb68 - 0xbb6f
		0x88f4, 0x88f9, 0x88f8, 0x88fd, 0x88e8, 0x891a, 0x88ef, 0x8aa6, // 0xbb70 - 0xbb77
		0x8a8c, 0x8a9e, 0x8aa3, 0x8a8d, 0x8aa1, 0x8a93, 0x8aa4, 0x0000, // 0xbb78 - 0xbb7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb80 - 0xbb87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb88 - 0xbb8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb90 - 0xbb97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbb98 - 0xbb9f
		0x0000, 0x5f27, 0x864e, 0x552c, 0x62a4, 0x4e92, 0x6caa, 0x6237, // 0xbba0 - 0xbba7
		0x82b1, 0x54d7, 0x534e, 0x733e, 0x6ed1, 0x753b, 0x5212, 0x5316, // 0xbba8 - 0xbbaf
		0x8bdd, 0x69d0, 0x5f8a, 0x6000, 0x6dee, 0x574f, 0x6b22, 0x73af, // 0xbbb0 - 0xbbb7
		0x6853, 0x8fd8, 0x7f13, 0x6362, 0x60a3, 0x5524, 0x75ea, 0x8c62, // 0xbbb8 - 0xbbbf
		0x7115, 0x6da3, 0x5ba6, 0x5e7b, 0x8352, 0x614c, 0x9ec4, 0x78fa, // 0xbbc0 - 0xbbc7
		0x8757, 0x7c27, 0x7687, 0x51f0, 0x60f6, 0x714c, 0x6643, 0x5e4c, // 0xbbc8 - 0xbbcf
		0x604d, 0x8c0e, 0x7070, 0x6325, 0x8f89, 0x5fbd, 0x6062, 0x86d4, // 0xbbd0 - 0xbbd7
		0x56de, 0x6bc1, 0x6094, 0x6167, 0x5349, 0x60e0, 0x6666, 0x8d3f, // 0xbbd8 - 0xbbdf
		0x79fd, 0x4f1a, 0x70e9, 0x6c47, 0x8bb3, 0x8bf2, 0x7ed8, 0x8364, // 0xbbe0 - 0xbbe7
		0x660f, 0x5a5a, 0x9b42, 0x6d51, 0x6df7, 0x8c41, 0x6d3b, 0x4f19, // 0xbbe8 - 0xbbef
		0x706b, 0x83b7, 0x6216, 0x60d1, 0x970d, 0x8d27, 0x7978, 0x51fb, // 0xbbf0 - 0xbbf7
		0x573e, 0x57fa, 0x673a, 0x7578, 0x7a3d, 0x79ef, 0x7b95, 0x0000, // 0xbbf8 - 0xbbff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc00 - 0xbc07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc08 - 0xbc0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc10 - 0xbc17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc18 - 0xbc1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc20 - 0xbc27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc28 - 0xbc2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc30 - 0xbc37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc38 - 0xbc3f
		0x5287, 0x5288, 0x5289, 0x528d, 0x528a, 0x52f0, 0x53b2, 0x562e, // 0xbc40 - 0xbc47
		0x563b, 0x5639, 0x5632, 0x563f, 0x5634, 0x5629, 0x5653, 0x564e, // 0xbc48 - 0xbc4f
		0x5657, 0x5674, 0x5636, 0x562f, 0x5630, 0x5880, 0x589f, 0x589e, // 0xbc50 - 0xbc57
		0x58b3, 0x589c, 0x58ae, 0x58a9, 0x58a6, 0x596d, 0x5b09, 0x5afb, // 0xbc58 - 0xbc5f
		0x5b0b, 0x5af5, 0x5b0c, 0x5b08, 0x5bee, 0x5bec, 0x5be9, 0x5beb, // 0xbc60 - 0xbc67
		0x5c64, 0x5c65, 0x5d9d, 0x5d94, 0x5e62, 0x5e5f, 0x5e61, 0x5ee2, // 0xbc68 - 0xbc6f
		0x5eda, 0x5edf, 0x5edd, 0x5ee3, 0x5ee0, 0x5f48, 0x5f71, 0x5fb7, // 0xbc70 - 0xbc77
		0x5fb5, 0x6176, 0x6167, 0x616e, 0x615d, 0x6155, 0x6182, 0x0000, // 0xbc78 - 0xbc7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc80 - 0xbc87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc88 - 0xbc8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc90 - 0xbc97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbc98 - 0xbc9f
		0x0000, 0x808c, 0x9965, 0x8ff9, 0x6fc0, 0x8ba5, 0x9e21, 0x59ec, // 0xbca0 - 0xbca7
		0x7ee9, 0x7f09, 0x5409, 0x6781, 0x68d8, 0x8f91, 0x7c4d, 0x96c6, // 0xbca8 - 0xbcaf
		0x53ca, 0x6025, 0x75be, 0x6c72, 0x5373, 0x5ac9, 0x7ea7, 0x6324, // 0xbcb0 - 0xbcb7
		0x51e0, 0x810a, 0x5df1, 0x84df, 0x6280, 0x5180, 0x5b63, 0x4f0e, // 0xbcb8 - 0xbcbf
		0x796d, 0x5242, 0x60b8, 0x6d4e, 0x5bc4, 0x5bc2, 0x8ba1, 0x8bb0, // 0xbcc0 - 0xbcc7
		0x65e2, 0x5fcc, 0x9645, 0x5993, 0x7ee7, 0x7eaa, 0x5609, 0x67b7, // 0xbcc8 - 0xbccf
		0x5939, 0x4f73, 0x5bb6, 0x52a0, 0x835a, 0x988a, 0x8d3e, 0x7532, // 0xbcd0 - 0xbcd7
		0x94be, 0x5047, 0x7a3c, 0x4ef7, 0x67b6, 0x9a7e, 0x5ac1, 0x6b7c, // 0xbcd8 - 0xbcdf
		0x76d1, 0x575a, 0x5c16, 0x7b3a, 0x95f4, 0x714e, 0x517c, 0x80a9, // 0xbce0 - 0xbce7
		0x8270, 0x5978, 0x7f04, 0x8327, 0x68c0, 0x67ec, 0x78b1, 0x7877, // 0xbce8 - 0xbcef
		0x62e3, 0x6361, 0x7b80, 0x4fed, 0x526a, 0x51cf, 0x8350, 0x69db, // 0xbcf0 - 0xbcf7
		0x9274, 0x8df5, 0x8d31, 0x89c1, 0x952e, 0x7bad, 0x4ef6, 0x0000, // 0xbcf8 - 0xbcff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd00 - 0xbd07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd08 - 0xbd0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd10 - 0xbd17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd18 - 0xbd1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd20 - 0xbd27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd28 - 0xbd2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd30 - 0xbd37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd38 - 0xbd3f
		0x747e, 0x7480, 0x757f, 0x7620, 0x7629, 0x761f, 0x7624, 0x7626, // 0xbd40 - 0xbd47
		0x7621, 0x7622, 0x769a, 0x76ba, 0x76e4, 0x778e, 0x7787, 0x778c, // 0xbd48 - 0xbd4f
		0x7791, 0x778b, 0x78cb, 0x78c5, 0x78ba, 0x78ca, 0x78be, 0x78d5, // 0xbd50 - 0xbd57
		0x78bc, 0x78d0, 0x7a3f, 0x7a3c, 0x7a40, 0x7a3d, 0x7a37, 0x7a3b, // 0xbd58 - 0xbd5f
		0x7aaf, 0x7aae, 0x7bad, 0x7bb1, 0x7bc4, 0x7bb4, 0x7bc6, 0x7bc7, // 0xbd60 - 0xbd67
		0x7bc1, 0x7ba0, 0x7bcc, 0x7cca, 0x7de0, 0x7df4, 0x7def, 0x7dfb, // 0xbd68 - 0xbd6f
		0x7dd8, 0x7dec, 0x7ddd, 0x7de8, 0x7de3, 0x7dda, 0x7dde, 0x7de9, // 0xbd70 - 0xbd77
		0x7d9e, 0x7dd9, 0x7df2, 0x7df9, 0x7f75, 0x7f77, 0x7faf, 0x0000, // 0xbd78 - 0xbd7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd80 - 0xbd87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd88 - 0xbd8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd90 - 0xbd97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbd98 - 0xbd9f
		0x0000, 0x5065, 0x8230, 0x5251, 0x996f, 0x6e10, 0x6e85, 0x6da7, // 0xbda0 - 0xbda7
		0x5efa, 0x50f5, 0x59dc, 0x5c06, 0x6d46, 0x6c5f, 0x7586, 0x848b, // 0xbda8 - 0xbdaf
		0x6868, 0x5956, 0x8bb2, 0x5320, 0x9171, 0x964d, 0x8549, 0x6912, // 0xbdb0 - 0xbdb7
		0x7901, 0x7126, 0x80f6, 0x4ea4, 0x90ca, 0x6d47, 0x9a84, 0x5a07, // 0xbdb8 - 0xbdbf
		0x56bc, 0x6405, 0x94f0, 0x77eb, 0x4fa5, 0x811a, 0x72e1, 0x89d2, // 0xbdc0 - 0xbdc7
		0x997a, 0x7f34, 0x7ede, 0x527f, 0x6559, 0x9175, 0x8f7f, 0x8f83, // 0xbdc8 - 0xbdcf
		0x53eb, 0x7a96, 0x63ed, 0x63a5, 0x7686, 0x79f8, 0x8857, 0x9636, // 0xbdd0 - 0xbdd7
		0x622a, 0x52ab, 0x8282, 0x6854, 0x6770, 0x6377, 0x776b, 0x7aed, // 0xbdd8 - 0xbddf
		0x6d01, 0x7ed3, 0x89e3, 0x59d0, 0x6212, 0x85c9, 0x82a5, 0x754c, // 0xbde0 - 0xbde7
		0x501f, 0x4ecb, 0x75a5, 0x8beb, 0x5c4a, 0x5dfe, 0x7b4b, 0x65a4, // 0xbde8 - 0xbdef
		0x91d1, 0x4eca, 0x6d25, 0x895f, 0x7d27, 0x9526, 0x4ec5, 0x8c28, // 0xbdf0 - 0xbdf7
		0x8fdb, 0x9773, 0x664b, 0x7981, 0x8fd1, 0x70ec, 0x6d78, 0x0000, // 0xbdf8 - 0xbdff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe00 - 0xbe07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe08 - 0xbe0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe10 - 0xbe17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe18 - 0xbe1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe20 - 0xbe27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe28 - 0xbe2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe30 - 0xbe37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe38 - 0xbe3f
		0x8f25, 0x9069, 0x906e, 0x9068, 0x906d, 0x9077, 0x9130, 0x912d, // 0xbe40 - 0xbe47
		0x9127, 0x9131, 0x9187, 0x9189, 0x918b, 0x9183, 0x92c5, 0x92bb, // 0xbe48 - 0xbe4f
		0x92b7, 0x92ea, 0x92ac, 0x92e4, 0x92c1, 0x92b3, 0x92bc, 0x92d2, // 0xbe50 - 0xbe57
		0x92c7, 0x92f0, 0x92b2, 0x95ad, 0x95b1, 0x9704, 0x9706, 0x9707, // 0xbe58 - 0xbe5f
		0x9709, 0x9760, 0x978d, 0x978b, 0x978f, 0x9821, 0x982b, 0x981c, // 0xbe60 - 0xbe67
		0x98b3, 0x990a, 0x9913, 0x9912, 0x9918, 0x99dd, 0x99d0, 0x99df, // 0xbe68 - 0xbe6f
		0x99db, 0x99d1, 0x99d5, 0x99d2, 0x99d9, 0x9ab7, 0x9aee, 0x9aef, // 0xbe70 - 0xbe77
		0x9b27, 0x9b45, 0x9b44, 0x9b77, 0x9b6f, 0x9d06, 0x9d09, 0x0000, // 0xbe78 - 0xbe7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe80 - 0xbe87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe88 - 0xbe8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe90 - 0xbe97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbe98 - 0xbe9f
		0x0000, 0x5c3d, 0x52b2, 0x8346, 0x5162, 0x830e, 0x775b, 0x6676, // 0xbea0 - 0xbea7
		0x9cb8, 0x4eac, 0x60ca, 0x7cbe, 0x7cb3, 0x7ecf, 0x4e95, 0x8b66, // 0xbea8 - 0xbeaf
		0x666f, 0x9888, 0x9759, 0x5883, 0x656c, 0x955c, 0x5f84, 0x75c9, // 0xbeb0 - 0xbeb7
		0x9756, 0x7adf, 0x7ade, 0x51c0, 0x70af, 0x7a98, 0x63ea, 0x7a76, // 0xbeb8 - 0xbebf
		0x7ea0, 0x7396, 0x97ed, 0x4e45, 0x7078, 0x4e5d, 0x9152, 0x53a9, // 0xbec0 - 0xbec7
		0x6551, 0x65e7, 0x81fc, 0x8205, 0x548e, 0x5c31, 0x759a, 0x97a0, // 0xbec8 - 0xbecf
		0x62d8, 0x72d9, 0x75bd, 0x5c45, 0x9a79, 0x83ca, 0x5c40, 0x5480, // 0xbed0 - 0xbed7
		0x77e9, 0x4e3e, 0x6cae, 0x805a, 0x62d2, 0x636e, 0x5de8, 0x5177, // 0xbed8 - 0xbedf
		0x8ddd, 0x8e1e, 0x952f, 0x4ff1, 0x53e5, 0x60e7, 0x70ac, 0x5267, // 0xbee0 - 0xbee7
		0x6350, 0x9e43, 0x5a1f, 0x5026, 0x7737, 0x5377, 0x7ee2, 0x6485, // 0xbee8 - 0xbeef
		0x652b, 0x6289, 0x6398, 0x5014, 0x7235, 0x89c9, 0x51b3, 0x8bc0, // 0xbef0 - 0xbef7
		0x7edd, 0x5747, 0x83cc, 0x94a7, 0x519b, 0x541b, 0x5cfb, 0x0000, // 0xbef8 - 0xbeff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf00 - 0xbf07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf08 - 0xbf0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf10 - 0xbf17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf18 - 0xbf1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf20 - 0xbf27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf28 - 0xbf2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf30 - 0xbf37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf38 - 0xbf3f
		0x6fc3, 0x6fa4, 0x6fc1, 0x6fa7, 0x6fb3, 0x6fc0, 0x6fb9, 0x6fb6, // 0xbf40 - 0xbf47
		0x6fa6, 0x6fa0, 0x6fb4, 0x71be, 0x71c9, 0x71d0, 0x71d2, 0x71c8, // 0xbf48 - 0xbf4f
		0x71d5, 0x71b9, 0x71ce, 0x71d9, 0x71dc, 0x71c3, 0x71c4, 0x7368, // 0xbf50 - 0xbf57
		0x749c, 0x74a3, 0x7498, 0x749f, 0x749e, 0x74e2, 0x750c, 0x750d, // 0xbf58 - 0xbf5f
		0x7634, 0x7638, 0x763a, 0x76e7, 0x76e5, 0x77a0, 0x779e, 0x779f, // 0xbf60 - 0xbf67
		0x77a5, 0x78e8, 0x78da, 0x78ec, 0x78e7, 0x79a6, 0x7a4d, 0x7a4e, // 0xbf68 - 0xbf6f
		0x7a46, 0x7a4c, 0x7a4b, 0x7aba, 0x7bd9, 0x7c11, 0x7bc9, 0x7be4, // 0xbf70 - 0xbf77
		0x7bdb, 0x7be1, 0x7be9, 0x7be6, 0x7cd5, 0x7cd6, 0x7e0a, 0x0000, // 0xbf78 - 0xbf7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf80 - 0xbf87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf88 - 0xbf8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf90 - 0xbf97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xbf98 - 0xbf9f
		0x0000, 0x4fca, 0x7ae3, 0x6d5a, 0x90e1, 0x9a8f, 0x5580, 0x5496, // 0xbfa0 - 0xbfa7
		0x5361, 0x54af, 0x5f00, 0x63e9, 0x6977, 0x51ef, 0x6168, 0x520a, // 0xbfa8 - 0xbfaf
		0x582a, 0x52d8, 0x574e, 0x780d, 0x770b, 0x5eb7, 0x6177, 0x7ce0, // 0xbfb0 - 0xbfb7
		0x625b, 0x6297, 0x4ea2, 0x7095, 0x8003, 0x62f7, 0x70e4, 0x9760, // 0xbfb8 - 0xbfbf
		0x5777, 0x82db, 0x67ef, 0x68f5, 0x78d5, 0x9897, 0x79d1, 0x58f3, // 0xbfc0 - 0xbfc7
		0x54b3, 0x53ef, 0x6e34, 0x514b, 0x523b, 0x5ba2, 0x8bfe, 0x80af, // 0xbfc8 - 0xbfcf
		0x5543, 0x57a6, 0x6073, 0x5751, 0x542d, 0x7a7a, 0x6050, 0x5b54, // 0xbfd0 - 0xbfd7
		0x63a7, 0x62a0, 0x53e3, 0x6263, 0x5bc7, 0x67af, 0x54ed, 0x7a9f, // 0xbfd8 - 0xbfdf
		0x82e6, 0x9177, 0x5e93, 0x88e4, 0x5938, 0x57ae, 0x630e, 0x8de8, // 0xbfe0 - 0xbfe7
		0x80ef, 0x5757, 0x7b77, 0x4fa9, 0x5feb, 0x5bbd, 0x6b3e, 0x5321, // 0xbfe8 - 0xbfef
		0x7b50, 0x72c2, 0x6846, 0x77ff, 0x7736, 0x65f7, 0x51b5, 0x4e8f, // 0xbff0 - 0xbff7
		0x76d4, 0x5cbf, 0x7aa5, 0x8475, 0x594e, 0x9b41, 0x5080, 0x0000, // 0xbff8 - 0xbfff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc000 - 0xc007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc008 - 0xc00f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc010 - 0xc017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc018 - 0xc01f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc020 - 0xc027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc028 - 0xc02f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc030 - 0xc037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc038 - 0xc03f
		0x9310, 0x9326, 0x9321, 0x9315, 0x932e, 0x9319, 0x95bb, 0x96a7, // 0xc040 - 0xc047
		0x96a8, 0x96aa, 0x96d5, 0x970e, 0x9711, 0x9716, 0x970d, 0x9713, // 0xc048 - 0xc04f
		0x970f, 0x975b, 0x975c, 0x9766, 0x9798, 0x9830, 0x9838, 0x983b, // 0xc050 - 0xc057
		0x9837, 0x982d, 0x9839, 0x9824, 0x9910, 0x9928, 0x991e, 0x991b, // 0xc058 - 0xc05f
		0x9921, 0x991a, 0x99ed, 0x99e2, 0x99f1, 0x9ab8, 0x9abc, 0x9afb, // 0xc060 - 0xc067
		0x9aed, 0x9b28, 0x9b91, 0x9d15, 0x9d23, 0x9d26, 0x9d28, 0x9d12, // 0xc068 - 0xc06f
		0x9d1b, 0x9ed8, 0x9ed4, 0x9f8d, 0x9f9c, 0x512a, 0x511f, 0x5121, // 0xc070 - 0xc077
		0x5132, 0x52f5, 0x568e, 0x5680, 0x5690, 0x5685, 0x5687, 0x0000, // 0xc078 - 0xc07f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc080 - 0xc087
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc088 - 0xc08f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc090 - 0xc097
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc098 - 0xc09f
		0x0000, 0x9988, 0x6127, 0x6e83, 0x5764, 0x6606, 0x6346, 0x56f0, // 0xc0a0 - 0xc0a7
		0x62ec, 0x6269, 0x5ed3, 0x9614, 0x5783, 0x62c9, 0x5587, 0x8721, // 0xc0a8 - 0xc0af
		0x814a, 0x8fa3, 0x5566, 0x83b1, 0x6765, 0x8d56, 0x84dd, 0x5a6a, // 0xc0b0 - 0xc0b7
		0x680f, 0x62e6, 0x7bee, 0x9611, 0x5170, 0x6f9c, 0x8c30, 0x63fd, // 0xc0b8 - 0xc0bf
		0x89c8, 0x61d2, 0x7f06, 0x70c2, 0x6ee5, 0x7405, 0x6994, 0x72fc, // 0xc0c0 - 0xc0c7
		0x5eca, 0x90ce, 0x6717, 0x6d6a, 0x635e, 0x52b3, 0x7262, 0x8001, // 0xc0c8 - 0xc0cf
		0x4f6c, 0x59e5, 0x916a, 0x70d9, 0x6d9d, 0x52d2, 0x4e50, 0x96f7, // 0xc0d0 - 0xc0d7
		0x956d, 0x857e, 0x78ca, 0x7d2f, 0x5121, 0x5792, 0x64c2, 0x808b, // 0xc0d8 - 0xc0df
		0x7c7b, 0x6cea, 0x68f1, 0x695e, 0x51b7, 0x5398, 0x68a8, 0x7281, // 0xc0e0 - 0xc0e7
		0x9ece, 0x7bf1, 0x72f8, 0x79bb, 0x6f13, 0x7406, 0x674e, 0x91cc, // 0xc0e8 - 0xc0ef
		0x9ca4, 0x793c, 0x8389, 0x8354, 0x540f, 0x6817, 0x4e3d, 0x5389, // 0xc0f0 - 0xc0f7
		0x52b1, 0x783e, 0x5386, 0x5229, 0x5088, 0x4f8b, 0x4fd0, 0x0000, // 0xc0f8 - 0xc0ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc100 - 0xc107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc108 - 0xc10f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc110 - 0xc117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc118 - 0xc11f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc120 - 0xc127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc128 - 0xc12f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc130 - 0xc137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc138 - 0xc13f
		0x77a7, 0x77ad, 0x77ef, 0x78f7, 0x78fa, 0x78f4, 0x78ef, 0x7901, // 0xc140 - 0xc147
		0x79a7, 0x79aa, 0x7a57, 0x7abf, 0x7c07, 0x7c0d, 0x7bfe, 0x7bf7, // 0xc148 - 0xc14f
		0x7c0c, 0x7be0, 0x7ce0, 0x7cdc, 0x7cde, 0x7ce2, 0x7cdf, 0x7cd9, // 0xc150 - 0xc157
		0x7cdd, 0x7e2e, 0x7e3e, 0x7e46, 0x7e37, 0x7e32, 0x7e43, 0x7e2b, // 0xc158 - 0xc15f
		0x7e3d, 0x7e31, 0x7e45, 0x7e41, 0x7e34, 0x7e39, 0x7e48, 0x7e35, // 0xc160 - 0xc167
		0x7e3f, 0x7e2f, 0x7f44, 0x7ff3, 0x7ffc, 0x8071, 0x8072, 0x8070, // 0xc168 - 0xc16f
		0x806f, 0x8073, 0x81c6, 0x81c3, 0x81ba, 0x81c2, 0x81c0, 0x81bf, // 0xc170 - 0xc177
		0x81bd, 0x81c9, 0x81be, 0x81e8, 0x8209, 0x8271, 0x85aa, 0x0000, // 0xc178 - 0xc17f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc180 - 0xc187
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc188 - 0xc18f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc190 - 0xc197
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc198 - 0xc19f
		0x0000, 0x75e2, 0x7acb, 0x7c92, 0x6ca5, 0x96b6, 0x529b, 0x7483, // 0xc1a0 - 0xc1a7
		0x54e9, 0x4fe9, 0x8054, 0x83b2, 0x8fde, 0x9570, 0x5ec9, 0x601c, // 0xc1a8 - 0xc1af
		0x6d9f, 0x5e18, 0x655b, 0x8138, 0x94fe, 0x604b, 0x70bc, 0x7ec3, // 0xc1b0 - 0xc1b7
		0x7cae, 0x51c9, 0x6881, 0x7cb1, 0x826f, 0x4e24, 0x8f86, 0x91cf, // 0xc1b8 - 0xc1bf
		0x667e, 0x4eae, 0x8c05, 0x64a9, 0x804a, 0x50da, 0x7597, 0x71ce, // 0xc1c0 - 0xc1c7
		0x5be5, 0x8fbd, 0x6f66, 0x4e86, 0x6482, 0x9563, 0x5ed6, 0x6599, // 0xc1c8 - 0xc1cf
		0x5217, 0x88c2, 0x70c8, 0x52a3, 0x730e, 0x7433, 0x6797, 0x78f7, // 0xc1d0 - 0xc1d7
		0x9716, 0x4e34, 0x90bb, 0x9cde, 0x6dcb, 0x51db, 0x8d41, 0x541d, // 0xc1d8 - 0xc1df
		0x62ce, 0x73b2, 0x83f1, 0x96f6, 0x9f84, 0x94c3, 0x4f36, 0x7f9a, // 0xc1e0 - 0xc1e7
		0x51cc, 0x7075, 0x9675, 0x5cad, 0x9886, 0x53e6, 0x4ee4, 0x6e9c, // 0xc1e8 - 0xc1ef
		0x7409, 0x69b4, 0x786b, 0x998f, 0x7559, 0x5218, 0x7624, 0x6d41, // 0xc1f0 - 0xc1f7
		0x67f3, 0x516d, 0x9f99, 0x804b, 0x5499, 0x7b3c, 0x7abf, 0x0000, // 0xc1f8 - 0xc1ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc200 - 0xc207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc208 - 0xc20f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc210 - 0xc217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc218 - 0xc21f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc220 - 0xc227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc228 - 0xc22f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc230 - 0xc237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc238 - 0xc23f
		0x99ff, 0x9bae, 0x9bab, 0x9baa, 0x9bad, 0x9d3b, 0x9d3f, 0x9e8b, // 0xc240 - 0xc247
		0x9ecf, 0x9ede, 0x9edc, 0x9edd, 0x9edb, 0x9f3e, 0x9f4b, 0x53e2, // 0xc248 - 0xc24f
		0x5695, 0x56ae, 0x58d9, 0x58d8, 0x5b38, 0x5f5d, 0x61e3, 0x6233, // 0xc250 - 0xc257
		0x64f4, 0x64f2, 0x64fe, 0x6506, 0x64fa, 0x64fb, 0x64f7, 0x65b7, // 0xc258 - 0xc25f
		0x66dc, 0x6726, 0x6ab3, 0x6aac, 0x6ac3, 0x6abb, 0x6ab8, 0x6ac2, // 0xc260 - 0xc267
		0x6aae, 0x6aaf, 0x6b5f, 0x6b78, 0x6baf, 0x7009, 0x700b, 0x6ffe, // 0xc268 - 0xc26f
		0x7006, 0x6ffa, 0x7011, 0x700f, 0x71fb, 0x71fc, 0x71fe, 0x71f8, // 0xc270 - 0xc277
		0x7377, 0x7375, 0x74a7, 0x74bf, 0x7515, 0x7656, 0x7658, 0x0000, // 0xc278 - 0xc27f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc280 - 0xc287
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc288 - 0xc28f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc290 - 0xc297
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc298 - 0xc29f
		0x0000, 0x9686, 0x5784, 0x62e2, 0x9647, 0x697c, 0x5a04, 0x6402, // 0xc2a0 - 0xc2a7
		0x7bd3, 0x6f0f, 0x964b, 0x82a6, 0x5362, 0x9885, 0x5e90, 0x7089, // 0xc2a8 - 0xc2af
		0x63b3, 0x5364, 0x864f, 0x9c81, 0x9e93, 0x788c, 0x9732, 0x8def, // 0xc2b0 - 0xc2b7
		0x8d42, 0x9e7f, 0x6f5e, 0x7984, 0x5f55, 0x9646, 0x622e, 0x9a74, // 0xc2b8 - 0xc2bf
		0x5415, 0x94dd, 0x4fa3, 0x65c5, 0x5c65, 0x5c61, 0x7f15, 0x8651, // 0xc2c0 - 0xc2c7
		0x6c2f, 0x5f8b, 0x7387, 0x6ee4, 0x7eff, 0x5ce6, 0x631b, 0x5b6a, // 0xc2c8 - 0xc2cf
		0x6ee6, 0x5375, 0x4e71, 0x63a0, 0x7565, 0x62a1, 0x8f6e, 0x4f26, // 0xc2d0 - 0xc2d7
		0x4ed1, 0x6ca6, 0x7eb6, 0x8bba, 0x841d, 0x87ba, 0x7f57, 0x903b, // 0xc2d8 - 0xc2df
		0x9523, 0x7ba9, 0x9aa1, 0x88f8, 0x843d, 0x6d1b, 0x9a86, 0x7edc, // 0xc2e0 - 0xc2e7
		0x5988, 0x9ebb, 0x739b, 0x7801, 0x8682, 0x9a6c, 0x9a82, 0x561b, // 0xc2e8 - 0xc2ef
		0x5417, 0x57cb, 0x4e70, 0x9ea6, 0x5356, 0x8fc8, 0x8109, 0x7792, // 0xc2f0 - 0xc2f7
		0x9992, 0x86ee, 0x6ee1, 0x8513, 0x66fc, 0x6162, 0x6f2b, 0x0000, // 0xc2f8 - 0xc2ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc300 - 0xc307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc308 - 0xc30f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc310 - 0xc317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc318 - 0xc31f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc320 - 0xc327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc328 - 0xc32f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc330 - 0xc337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc338 - 0xc33f
		0x97ad, 0x97f9, 0x984d, 0x984f, 0x984c, 0x984e, 0x9853, 0x98ba, // 0xc340 - 0xc347
		0x993e, 0x993f, 0x993d, 0x992e, 0x99a5, 0x9a0e, 0x9ac1, 0x9b03, // 0xc348 - 0xc34f
		0x9b06, 0x9b4f, 0x9b4e, 0x9b4d, 0x9bca, 0x9bc9, 0x9bfd, 0x9bc8, // 0xc350 - 0xc357
		0x9bc0, 0x9d51, 0x9d5d, 0x9d60, 0x9ee0, 0x9f15, 0x9f2c, 0x5133, // 0xc358 - 0xc35f
		0x56a5, 0x58de, 0x58df, 0x58e2, 0x5bf5, 0x9f90, 0x5eec, 0x61f2, // 0xc360 - 0xc367
		0x61f7, 0x61f6, 0x61f5, 0x6500, 0x650f, 0x66e0, 0x66dd, 0x6ae5, // 0xc368 - 0xc36f
		0x6add, 0x6ada, 0x6ad3, 0x701b, 0x701f, 0x7028, 0x701a, 0x701d, // 0xc370 - 0xc377
		0x7015, 0x7018, 0x7206, 0x720d, 0x7258, 0x72a2, 0x7378, 0x0000, // 0xc378 - 0xc37f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc380 - 0xc387
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc388 - 0xc38f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc390 - 0xc397
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc398 - 0xc39f
		0x0000, 0x8c29, 0x8292, 0x832b, 0x76f2, 0x6c13, 0x5fd9, 0x83bd, // 0xc3a0 - 0xc3a7
		0x732b, 0x8305, 0x951a, 0x6bdb, 0x77db, 0x94c6, 0x536f, 0x8302, // 0xc3a8 - 0xc3af
		0x5192, 0x5e3d, 0x8c8c, 0x8d38, 0x4e48, 0x73ab, 0x679a, 0x6885, // 0xc3b0 - 0xc3b7
		0x9176, 0x9709, 0x7164, 0x6ca1, 0x7709, 0x5a92, 0x9541, 0x6bcf, // 0xc3b8 - 0xc3bf
		0x7f8e, 0x6627, 0x5bd0, 0x59b9, 0x5a9a, 0x95e8, 0x95f7, 0x4eec, // 0xc3c0 - 0xc3c7
		0x840c, 0x8499, 0x6aac, 0x76df, 0x9530, 0x731b, 0x68a6, 0x5b5f, // 0xc3c8 - 0xc3cf
		0x772f, 0x919a, 0x9761, 0x7cdc, 0x8ff7, 0x8c1c, 0x5f25, 0x7c73, // 0xc3d0 - 0xc3d7
		0x79d8, 0x89c5, 0x6ccc, 0x871c, 0x5bc6, 0x5e42, 0x68c9, 0x7720, // 0xc3d8 - 0xc3df
		0x7ef5, 0x5195, 0x514d, 0x52c9, 0x5a29, 0x7f05, 0x9762, 0x82d7, // 0xc3e0 - 0xc3e7
		0x63cf, 0x7784, 0x85d0, 0x79d2, 0x6e3a, 0x5e99, 0x5999, 0x8511, // 0xc3e8 - 0xc3ef
		0x706d, 0x6c11, 0x62bf, 0x76bf, 0x654f, 0x60af, 0x95fd, 0x660e, // 0xc3f0 - 0xc3f7
		0x879f, 0x9e23, 0x94ed, 0x540d, 0x547d, 0x8c2c, 0x6478, 0x0000, // 0xc3f8 - 0xc3ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc400 - 0xc407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc408 - 0xc40f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc410 - 0xc417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc418 - 0xc41f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc420 - 0xc427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc428 - 0xc42f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc430 - 0xc437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc438 - 0xc43f
		0x9858, 0x985b, 0x98bc, 0x9945, 0x9949, 0x9a16, 0x9a19, 0x9b0d, // 0xc440 - 0xc447
		0x9be8, 0x9be7, 0x9bd6, 0x9bdb, 0x9d89, 0x9d61, 0x9d72, 0x9d6a, // 0xc448 - 0xc44f
		0x9d6c, 0x9e92, 0x9e97, 0x9e93, 0x9eb4, 0x52f8, 0x56a8, 0x56b7, // 0xc450 - 0xc457
		0x56b6, 0x56b4, 0x56bc, 0x58e4, 0x5b40, 0x5b43, 0x5b7d, 0x5bf6, // 0xc458 - 0xc45f
		0x5dc9, 0x61f8, 0x61fa, 0x6518, 0x6514, 0x6519, 0x66e6, 0x6727, // 0xc460 - 0xc467
		0x6aec, 0x703e, 0x7030, 0x7032, 0x7210, 0x737b, 0x74cf, 0x7662, // 0xc468 - 0xc46f
		0x7665, 0x7926, 0x792a, 0x792c, 0x792b, 0x7ac7, 0x7af6, 0x7c4c, // 0xc470 - 0xc477
		0x7c43, 0x7c4d, 0x7cef, 0x7cf0, 0x8fae, 0x7e7d, 0x7e7c, 0x0000, // 0xc478 - 0xc47f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc480 - 0xc487
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc488 - 0xc48f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc490 - 0xc497
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc498 - 0xc49f
		0x0000, 0x6479, 0x8611, 0x6a21, 0x819c, 0x78e8, 0x6469, 0x9b54, // 0xc4a0 - 0xc4a7
		0x62b9, 0x672b, 0x83ab, 0x58a8, 0x9ed8, 0x6cab, 0x6f20, 0x5bde, // 0xc4a8 - 0xc4af
		0x964c, 0x8c0b, 0x725f, 0x67d0, 0x62c7, 0x7261, 0x4ea9, 0x59c6, // 0xc4b0 - 0xc4b7
		0x6bcd, 0x5893, 0x66ae, 0x5e55, 0x52df, 0x6155, 0x6728, 0x76ee, // 0xc4b8 - 0xc4bf
		0x7766, 0x7267, 0x7a46, 0x62ff, 0x54ea, 0x5450, 0x94a0, 0x90a3, // 0xc4c0 - 0xc4c7
		0x5a1c, 0x7eb3, 0x6c16, 0x4e43, 0x5976, 0x8010, 0x5948, 0x5357, // 0xc4c8 - 0xc4cf
		0x7537, 0x96be, 0x56ca, 0x6320, 0x8111, 0x607c, 0x95f9, 0x6dd6, // 0xc4d0 - 0xc4d7
		0x5462, 0x9981, 0x5185, 0x5ae9, 0x80fd, 0x59ae, 0x9713, 0x502a, // 0xc4d8 - 0xc4df
		0x6ce5, 0x5c3c, 0x62df, 0x4f60, 0x533f, 0x817b, 0x9006, 0x6eba, // 0xc4e0 - 0xc4e7
		0x852b, 0x62c8, 0x5e74, 0x78be, 0x64b5, 0x637b, 0x5ff5, 0x5a18, // 0xc4e8 - 0xc4ef
		0x917f, 0x9e1f, 0x5c3f, 0x634f, 0x8042, 0x5b7d, 0x556e, 0x954a, // 0xc4f0 - 0xc4f7
		0x954d, 0x6d85, 0x60a8, 0x67e0, 0x72de, 0x51dd, 0x5b81, 0x0000, // 0xc4f8 - 0xc4ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc500 - 0xc507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc508 - 0xc50f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc510 - 0xc517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc518 - 0xc51f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc520 - 0xc527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc528 - 0xc52f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc530 - 0xc537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc538 - 0xc53f
		0x8b77, 0x8b7d, 0x8d13, 0x8e8a, 0x8e8d, 0x8e8b, 0x8f5f, 0x8faf, // 0xc540 - 0xc547
		0x91ba, 0x942e, 0x9433, 0x9435, 0x943a, 0x9438, 0x9432, 0x942b, // 0xc548 - 0xc54f
		0x95e2, 0x9738, 0x9739, 0x9732, 0x97ff, 0x9867, 0x9865, 0x9957, // 0xc550 - 0xc557
		0x9a45, 0x9a43, 0x9a40, 0x9a3e, 0x9acf, 0x9b54, 0x9b51, 0x9c2d, // 0xc558 - 0xc55f
		0x9c25, 0x9daf, 0x9db4, 0x9dc2, 0x9db8, 0x9e9d, 0x9eef, 0x9f19, // 0xc560 - 0xc567
		0x9f5c, 0x9f66, 0x9f67, 0x513c, 0x513b, 0x56c8, 0x56ca, 0x56c9, // 0xc568 - 0xc56f
		0x5b7f, 0x5dd4, 0x5dd2, 0x5f4e, 0x61ff, 0x6524, 0x6b0a, 0x6b61, // 0xc570 - 0xc577
		0x7051, 0x7058, 0x7380, 0x74e4, 0x758a, 0x766e, 0x766c, 0x0000, // 0xc578 - 0xc57f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc580 - 0xc587
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc588 - 0xc58f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc590 - 0xc597
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc598 - 0xc59f
		0x0000, 0x62e7, 0x6cde, 0x725b, 0x626d, 0x94ae, 0x7ebd, 0x8113, // 0xc5a0 - 0xc5a7
		0x6d53, 0x519c, 0x5f04, 0x5974, 0x52aa, 0x6012, 0x5973, 0x6696, // 0xc5a8 - 0xc5af
		0x8650, 0x759f, 0x632a, 0x61e6, 0x7cef, 0x8bfa, 0x54e6, 0x6b27, // 0xc5b0 - 0xc5b7
		0x9e25, 0x6bb4, 0x85d5, 0x5455, 0x5076, 0x6ca4, 0x556a, 0x8db4, // 0xc5b8 - 0xc5bf
		0x722c, 0x5e15, 0x6015, 0x7436, 0x62cd, 0x6392, 0x724c, 0x5f98, // 0xc5c0 - 0xc5c7
		0x6e43, 0x6d3e, 0x6500, 0x6f58, 0x76d8, 0x78d0, 0x76fc, 0x7554, // 0xc5c8 - 0xc5cf
		0x5224, 0x53db, 0x4e53, 0x5e9e, 0x65c1, 0x802a, 0x80d6, 0x629b, // 0xc5d0 - 0xc5d7
		0x5486, 0x5228, 0x70ae, 0x888d, 0x8dd1, 0x6ce1, 0x5478, 0x80da, // 0xc5d8 - 0xc5df
		0x57f9, 0x88f4, 0x8d54, 0x966a, 0x914d, 0x4f69, 0x6c9b, 0x55b7, // 0xc5e0 - 0xc5e7
		0x76c6, 0x7830, 0x62a8, 0x70f9, 0x6f8e, 0x5f6d, 0x84ec, 0x68da, // 0xc5e8 - 0xc5ef
		0x787c, 0x7bf7, 0x81a8, 0x670b, 0x9e4f, 0x6367, 0x78b0, 0x576f, // 0xc5f0 - 0xc5f7
		0x7812, 0x9739, 0x6279, 0x62ab, 0x5288, 0x7435, 0x6bd7, 0x0000, // 0xc5f8 - 0xc5ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc600 - 0xc607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc608 - 0xc60f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc610 - 0xc617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc618 - 0xc61f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc620 - 0xc627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc628 - 0xc62f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc630 - 0xc637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc638 - 0xc63f
		0x8b96, 0x8277, 0x8d1b, 0x91c0, 0x946a, 0x9742, 0x9748, 0x9744, // 0xc640 - 0xc647
		0x97c6, 0x9870, 0x9a5f, 0x9b22, 0x9b58, 0x9c5f, 0x9df9, 0x9dfa, // 0xc648 - 0xc64f
		0x9e7c, 0x9e7d, 0x9f07, 0x9f77, 0x9f72, 0x5ef3, 0x6b16, 0x7063, // 0xc650 - 0xc657
		0x7c6c, 0x7c6e, 0x883b, 0x89c0, 0x8ea1, 0x91c1, 0x9472, 0x9470, // 0xc658 - 0xc65f
		0x9871, 0x995e, 0x9ad6, 0x9b23, 0x9ecc, 0x7064, 0x77da, 0x8b9a, // 0xc660 - 0xc667
		0x9477, 0x97c9, 0x9a62, 0x9a65, 0x7e9c, 0x8b9c, 0x8eaa, 0x91c5, // 0xc668 - 0xc66f
		0x947d, 0x947e, 0x947c, 0x9c77, 0x9c78, 0x9ef7, 0x8c54, 0x947f, // 0xc670 - 0xc677
		0x9e1a, 0x7228, 0x9a6a, 0x9b31, 0x9e1b, 0x9e1e, 0x7c72, 0x0000, // 0xc678 - 0xc67f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc680 - 0xc687
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc688 - 0xc68f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc690 - 0xc697
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc698 - 0xc69f
		0x0000, 0x5564, 0x813e, 0x75b2, 0x76ae, 0x5339, 0x75de, 0x50fb, // 0xc6a0 - 0xc6a7
		0x5c41, 0x8b6c, 0x7bc7, 0x504f, 0x7247, 0x9a97, 0x98d8, 0x6f02, // 0xc6a8 - 0xc6af
		0x74e2, 0x7968, 0x6487, 0x77a5, 0x62fc, 0x9891, 0x8d2b, 0x54c1, // 0xc6b0 - 0xc6b7
		0x8058, 0x4e52, 0x576a, 0x82f9, 0x840d, 0x5e73, 0x51ed, 0x74f6, // 0xc6b8 - 0xc6bf
		0x8bc4, 0x5c4f, 0x5761, 0x6cfc, 0x9887, 0x5a46, 0x7834, 0x9b44, // 0xc6c0 - 0xc6c7
		0x8feb, 0x7c95, 0x5256, 0x6251, 0x94fa, 0x4ec6, 0x8386, 0x8461, // 0xc6c8 - 0xc6cf
		0x83e9, 0x84b2, 0x57d4, 0x6734, 0x5703, 0x666e, 0x6d66, 0x8c31, // 0xc6d0 - 0xc6d7
		0x66dd, 0x7011, 0x671f, 0x6b3a, 0x6816, 0x621a, 0x59bb, 0x4e03, // 0xc6d8 - 0xc6df
		0x51c4, 0x6f06, 0x67d2, 0x6c8f, 0x5176, 0x68cb, 0x5947, 0x6b67, // 0xc6e0 - 0xc6e7
		0x7566, 0x5d0e, 0x8110, 0x9f50, 0x65d7, 0x7948, 0x7941, 0x9a91, // 0xc6e8 - 0xc6ef
		0x8d77, 0x5c82, 0x4e5e, 0x4f01, 0x542f, 0x5951, 0x780c, 0x5668, // 0xc6f0 - 0xc6f7
		0x6c14, 0x8fc4, 0x5f03, 0x6c7d, 0x6ce3, 0x8bab, 0x6390, 0x0000, // 0xc6f8 - 0xc6ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc700 - 0xc707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc708 - 0xc70f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc710 - 0xc717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc718 - 0xc71f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc720 - 0xc727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc728 - 0xc72f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc730 - 0xc737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc738 - 0xc73f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc740 - 0xc747
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc748 - 0xc74f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc750 - 0xc757
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc758 - 0xc75f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc760 - 0xc767
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc768 - 0xc76f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc770 - 0xc777
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc778 - 0xc77f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc780 - 0xc787
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc788 - 0xc78f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc790 - 0xc797
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc798 - 0xc79f
		0x0000, 0x6070, 0x6d3d, 0x7275, 0x6266, 0x948e, 0x94c5, 0x5343, // 0xc7a0 - 0xc7a7
		0x8fc1, 0x7b7e, 0x4edf, 0x8c26, 0x4e7e, 0x9ed4, 0x94b1, 0x94b3, // 0xc7a8 - 0xc7af
		0x524d, 0x6f5c, 0x9063, 0x6d45, 0x8c34, 0x5811, 0x5d4c, 0x6b20, // 0xc7b0 - 0xc7b7
		0x6b49, 0x67aa, 0x545b, 0x8154, 0x7f8c, 0x5899, 0x8537, 0x5f3a, // 0xc7b8 - 0xc7bf
		0x62a2, 0x6a47, 0x9539, 0x6572, 0x6084, 0x6865, 0x77a7, 0x4e54, // 0xc7c0 - 0xc7c7
		0x4fa8, 0x5de7, 0x9798, 0x64ac, 0x7fd8, 0x5ced, 0x4fcf, 0x7a8d, // 0xc7c8 - 0xc7cf
		0x5207, 0x8304, 0x4e14, 0x602f, 0x7a83, 0x94a6, 0x4fb5, 0x4eb2, // 0xc7d0 - 0xc7d7
		0x79e6, 0x7434, 0x52e4, 0x82b9, 0x64d2, 0x79bd, 0x5bdd, 0x6c81, // 0xc7d8 - 0xc7df
		0x9752, 0x8f7b, 0x6c22, 0x503e, 0x537f, 0x6e05, 0x64ce, 0x6674, // 0xc7e0 - 0xc7e7
		0x6c30, 0x60c5, 0x9877, 0x8bf7, 0x5e86, 0x743c, 0x7a77, 0x79cb, // 0xc7e8 - 0xc7ef
		0x4e18, 0x90b1, 0x7403, 0x6c42, 0x56da, 0x914b, 0x6cc5, 0x8d8b, // 0xc7f0 - 0xc7f7
		0x533a, 0x86c6, 0x66f2, 0x8eaf, 0x5c48, 0x9a71, 0x6e20, 0x0000, // 0xc7f8 - 0xc7ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc800 - 0xc807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc808 - 0xc80f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc810 - 0xc817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc818 - 0xc81f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc820 - 0xc827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc828 - 0xc82f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc830 - 0xc837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc838 - 0xc83f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc840 - 0xc847
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc848 - 0xc84f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc850 - 0xc857
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc858 - 0xc85f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc860 - 0xc867
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc868 - 0xc86f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc870 - 0xc877
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc878 - 0xc87f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc880 - 0xc887
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc888 - 0xc88f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc890 - 0xc897
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc898 - 0xc89f
		0x0000, 0x53d6, 0x5a36, 0x9f8b, 0x8da3, 0x53bb, 0x5708, 0x98a7, // 0xc8a0 - 0xc8a7
		0x6743, 0x919b, 0x6cc9, 0x5168, 0x75ca, 0x62f3, 0x72ac, 0x5238, // 0xc8a8 - 0xc8af
		0x529d, 0x7f3a, 0x7094, 0x7638, 0x5374, 0x9e4a, 0x69b7, 0x786e, // 0xc8b0 - 0xc8b7
		0x96c0, 0x88d9, 0x7fa4, 0x7136, 0x71c3, 0x5189, 0x67d3, 0x74e4, // 0xc8b8 - 0xc8bf
		0x58e4, 0x6518, 0x56b7, 0x8ba9, 0x9976, 0x6270, 0x7ed5, 0x60f9, // 0xc8c0 - 0xc8c7
		0x70ed, 0x58ec, 0x4ec1, 0x4eba, 0x5fcd, 0x97e7, 0x4efb, 0x8ba4, // 0xc8c8 - 0xc8cf
		0x5203, 0x598a, 0x7eab, 0x6254, 0x4ecd, 0x65e5, 0x620e, 0x8338, // 0xc8d0 - 0xc8d7
		0x84c9, 0x8363, 0x878d, 0x7194, 0x6eb6, 0x5bb9, 0x7ed2, 0x5197, // 0xc8d8 - 0xc8df
		0x63c9, 0x67d4, 0x8089, 0x8339, 0x8815, 0x5112, 0x5b7a, 0x5982, // 0xc8e0 - 0xc8e7
		0x8fb1, 0x4e73, 0x6c5d, 0x5165, 0x8925, 0x8f6f, 0x962e, 0x854a, // 0xc8e8 - 0xc8ef
		0x745e, 0x9510, 0x95f0, 0x6da6, 0x82e5, 0x5f31, 0x6492, 0x6d12, // 0xc8f0 - 0xc8f7
		0x8428, 0x816e, 0x9cc3, 0x585e, 0x8d5b, 0x4e09, 0x53c1, 0x0000, // 0xc8f8 - 0xc8ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc900 - 0xc907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc908 - 0xc90f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc910 - 0xc917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc918 - 0xc91f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc920 - 0xc927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc928 - 0xc92f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc930 - 0xc937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc938 - 0xc93f
		0x4e42, 0x4e5c, 0x51f5, 0x531a, 0x5382, 0x4e07, 0x4e0c, 0x4e47, // 0xc940 - 0xc947
		0x4e8d, 0x56d7, 0xfa0c, 0x5c6e, 0x5f73, 0x4e0f, 0x5187, 0x4e0e, // 0xc948 - 0xc94f
		0x4e2e, 0x4e93, 0x4ec2, 0x4ec9, 0x4ec8, 0x5198, 0x52fc, 0x536c, // 0xc950 - 0xc957
		0x53b9, 0x5720, 0x5903, 0x592c, 0x5c10, 0x5dff, 0x65e1, 0x6bb3, // 0xc958 - 0xc95f
		0x6bcc, 0x6c14, 0x723f, 0x4e31, 0x4e3c, 0x4ee8, 0x4edc, 0x4ee9, // 0xc960 - 0xc967
		0x4ee1, 0x4edd, 0x4eda, 0x520c, 0x531c, 0x534c, 0x5722, 0x5723, // 0xc968 - 0xc96f
		0x5917, 0x592f, 0x5b81, 0x5b84, 0x5c12, 0x5c3b, 0x5c74, 0x5c73, // 0xc970 - 0xc977
		0x5e04, 0x5e80, 0x5e82, 0x5fc9, 0x6209, 0x6250, 0x6c15, 0x0000, // 0xc978 - 0xc97f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc980 - 0xc987
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc988 - 0xc98f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc990 - 0xc997
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xc998 - 0xc99f
		0x0000, 0x4f1e, 0x6563, 0x6851, 0x55d3, 0x4e27, 0x6414, 0x9a9a, // 0xc9a0 - 0xc9a7
		0x626b, 0x5ac2, 0x745f, 0x8272, 0x6da9, 0x68ee, 0x50e7, 0x838e, // 0xc9a8 - 0xc9af
		0x7802, 0x6740, 0x5239, 0x6c99, 0x7eb1, 0x50bb, 0x5565, 0x715e, // 0xc9b0 - 0xc9b7
		0x7b5b, 0x6652, 0x73ca, 0x82eb, 0x6749, 0x5c71, 0x5220, 0x717d, // 0xc9b8 - 0xc9bf
		0x886b, 0x95ea, 0x9655, 0x64c5, 0x8d61, 0x81b3, 0x5584, 0x6c55, // 0xc9c0 - 0xc9c7
		0x6247, 0x7f2e, 0x5892, 0x4f24, 0x5546, 0x8d4f, 0x664c, 0x4e0a, // 0xc9c8 - 0xc9cf
		0x5c1a, 0x88f3, 0x68a2, 0x634e, 0x7a0d, 0x70e7, 0x828d, 0x52fa, // 0xc9d0 - 0xc9d7
		0x97f6, 0x5c11, 0x54e8, 0x90b5, 0x7ecd, 0x5962, 0x8d4a, 0x86c7, // 0xc9d8 - 0xc9df
		0x820c, 0x820d, 0x8d66, 0x6444, 0x5c04, 0x6151, 0x6d89, 0x793e, // 0xc9e0 - 0xc9e7
		0x8bbe, 0x7837, 0x7533, 0x547b, 0x4f38, 0x8eab, 0x6df1, 0x5a20, // 0xc9e8 - 0xc9ef
		0x7ec5, 0x795e, 0x6c88, 0x5ba1, 0x5a76, 0x751a, 0x80be, 0x614e, // 0xc9f0 - 0xc9f7
		0x6e17, 0x58f0, 0x751f, 0x7525, 0x7272, 0x5347, 0x7ef3, 0x0000, // 0xc9f8 - 0xc9ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca00 - 0xca07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca08 - 0xca0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca10 - 0xca17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca18 - 0xca1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca20 - 0xca27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca28 - 0xca2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca30 - 0xca37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca38 - 0xca3f
		0x6c4c, 0x7071, 0x725e, 0x72b4, 0x72b5, 0x738e, 0x752a, 0x767f, // 0xca40 - 0xca47
		0x7a75, 0x7f51, 0x8278, 0x827c, 0x8280, 0x827d, 0x827f, 0x864d, // 0xca48 - 0xca4f
		0x897e, 0x9099, 0x9097, 0x9098, 0x909b, 0x9094, 0x9622, 0x9624, // 0xca50 - 0xca57
		0x9620, 0x9623, 0x4f56, 0x4f3b, 0x4f62, 0x4f49, 0x4f53, 0x4f64, // 0xca58 - 0xca5f
		0x4f3e, 0x4f67, 0x4f52, 0x4f5f, 0x4f41, 0x4f58, 0x4f2d, 0x4f33, // 0xca60 - 0xca67
		0x4f3f, 0x4f61, 0x518f, 0x51b9, 0x521c, 0x521e, 0x5221, 0x52ad, // 0xca68 - 0xca6f
		0x52ae, 0x5309, 0x5363, 0x5372, 0x538e, 0x538f, 0x5430, 0x5437, // 0xca70 - 0xca77
		0x542a, 0x5454, 0x5445, 0x5419, 0x541c, 0x5425, 0x5418, 0x0000, // 0xca78 - 0xca7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca80 - 0xca87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca88 - 0xca8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca90 - 0xca97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xca98 - 0xca9f
		0x0000, 0x7701, 0x76db, 0x5269, 0x80dc, 0x5723, 0x5e08, 0x5931, // 0xcaa0 - 0xcaa7
		0x72ee, 0x65bd, 0x6e7f, 0x8bd7, 0x5c38, 0x8671, 0x5341, 0x77f3, // 0xcaa8 - 0xcaaf
		0x62fe, 0x65f6, 0x4ec0, 0x98df, 0x8680, 0x5b9e, 0x8bc6, 0x53f2, // 0xcab0 - 0xcab7
		0x77e2, 0x4f7f, 0x5c4e, 0x9a76, 0x59cb, 0x5f0f, 0x793a, 0x58eb, // 0xcab8 - 0xcabf
		0x4e16, 0x67ff, 0x4e8b, 0x62ed, 0x8a93, 0x901d, 0x52bf, 0x662f, // 0xcac0 - 0xcac7
		0x55dc, 0x566c, 0x9002, 0x4ed5, 0x4f8d, 0x91ca, 0x9970, 0x6c0f, // 0xcac8 - 0xcacf
		0x5e02, 0x6043, 0x5ba4, 0x89c6, 0x8bd5, 0x6536, 0x624b, 0x9996, // 0xcad0 - 0xcad7
		0x5b88, 0x5bff, 0x6388, 0x552e, 0x53d7, 0x7626, 0x517d, 0x852c, // 0xcad8 - 0xcadf
		0x67a2, 0x68b3, 0x6b8a, 0x6292, 0x8f93, 0x53d4, 0x8212, 0x6dd1, // 0xcae0 - 0xcae7
		0x758f, 0x4e66, 0x8d4e, 0x5b70, 0x719f, 0x85af, 0x6691, 0x66d9, // 0xcae8 - 0xcaef
		0x7f72, 0x8700, 0x9ecd, 0x9f20, 0x5c5e, 0x672f, 0x8ff0, 0x6811, // 0xcaf0 - 0xcaf7
		0x675f, 0x620d, 0x7ad6, 0x5885, 0x5eb6, 0x6570, 0x6f31, 0x0000, // 0xcaf8 - 0xcaff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb00 - 0xcb07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb08 - 0xcb0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb10 - 0xcb17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb18 - 0xcb1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb20 - 0xcb27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb28 - 0xcb2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb30 - 0xcb37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb38 - 0xcb3f
		0x6759, 0x6755, 0x674c, 0x6748, 0x675d, 0x674d, 0x675a, 0x674b, // 0xcb40 - 0xcb47
		0x6bd0, 0x6c19, 0x6c1a, 0x6c78, 0x6c67, 0x6c6b, 0x6c84, 0x6c8b, // 0xcb48 - 0xcb4f
		0x6c8f, 0x6c71, 0x6c6f, 0x6c69, 0x6c9a, 0x6c6d, 0x6c87, 0x6c95, // 0xcb50 - 0xcb57
		0x6c9c, 0x6c66, 0x6c73, 0x6c65, 0x6c7b, 0x6c8e, 0x7074, 0x707a, // 0xcb58 - 0xcb5f
		0x7263, 0x72bf, 0x72bd, 0x72c3, 0x72c6, 0x72c1, 0x72ba, 0x72c5, // 0xcb60 - 0xcb67
		0x7395, 0x7397, 0x7393, 0x7394, 0x7392, 0x753a, 0x7539, 0x7594, // 0xcb68 - 0xcb6f
		0x7595, 0x7681, 0x793d, 0x8034, 0x8095, 0x8099, 0x8090, 0x8092, // 0xcb70 - 0xcb77
		0x809c, 0x8290, 0x828f, 0x8285, 0x828e, 0x8291, 0x8293, 0x0000, // 0xcb78 - 0xcb7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb80 - 0xcb87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb88 - 0xcb8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb90 - 0xcb97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcb98 - 0xcb9f
		0x0000, 0x6055, 0x5237, 0x800d, 0x6454, 0x8870, 0x7529, 0x5e05, // 0xcba0 - 0xcba7
		0x6813, 0x62f4, 0x971c, 0x53cc, 0x723d, 0x8c01, 0x6c34, 0x7761, // 0xcba8 - 0xcbaf
		0x7a0e, 0x542e, 0x77ac, 0x987a, 0x821c, 0x8bf4, 0x7855, 0x6714, // 0xcbb0 - 0xcbb7
		0x70c1, 0x65af, 0x6495, 0x5636, 0x601d, 0x79c1, 0x53f8, 0x4e1d, // 0xcbb8 - 0xcbbf
		0x6b7b, 0x8086, 0x5bfa, 0x55e3, 0x56db, 0x4f3a, 0x4f3c, 0x9972, // 0xcbc0 - 0xcbc7
		0x5df3, 0x677e, 0x8038, 0x6002, 0x9882, 0x9001, 0x5b8b, 0x8bbc, // 0xcbc8 - 0xcbcf
		0x8bf5, 0x641c, 0x8258, 0x64de, 0x55fd, 0x82cf, 0x9165, 0x4fd7, // 0xcbd0 - 0xcbd7
		0x7d20, 0x901f, 0x7c9f, 0x50f3, 0x5851, 0x6eaf, 0x5bbf, 0x8bc9, // 0xcbd8 - 0xcbdf
		0x8083, 0x9178, 0x849c, 0x7b97, 0x867d, 0x968b, 0x968f, 0x7ee5, // 0xcbe0 - 0xcbe7
		0x9ad3, 0x788e, 0x5c81, 0x7a57, 0x9042, 0x96a7, 0x795f, 0x5b59, // 0xcbe8 - 0xcbef
		0x635f, 0x7b0b, 0x84d1, 0x68ad, 0x5506, 0x7f29, 0x7410, 0x7d22, // 0xcbf0 - 0xcbf7
		0x9501, 0x6240, 0x584c, 0x4ed6, 0x5b83, 0x5979, 0x5854, 0x0000, // 0xcbf8 - 0xcbff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc00 - 0xcc07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc08 - 0xcc0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc10 - 0xcc17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc18 - 0xcc1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc20 - 0xcc27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc28 - 0xcc2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc30 - 0xcc37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc38 - 0xcc3f
		0x5768, 0x577d, 0x590c, 0x5945, 0x59b5, 0x59ba, 0x59cf, 0x59ce, // 0xcc40 - 0xcc47
		0x59b2, 0x59cc, 0x59c1, 0x59b6, 0x59bc, 0x59c3, 0x59d6, 0x59b1, // 0xcc48 - 0xcc4f
		0x59bd, 0x59c0, 0x59c8, 0x59b4, 0x59c7, 0x5b62, 0x5b65, 0x5b93, // 0xcc50 - 0xcc57
		0x5b95, 0x5c44, 0x5c47, 0x5cae, 0x5ca4, 0x5ca0, 0x5cb5, 0x5caf, // 0xcc58 - 0xcc5f
		0x5ca8, 0x5cac, 0x5c9f, 0x5ca3, 0x5cad, 0x5ca2, 0x5caa, 0x5ca7, // 0xcc60 - 0xcc67
		0x5c9d, 0x5ca5, 0x5cb6, 0x5cb0, 0x5ca6, 0x5e17, 0x5e14, 0x5e19, // 0xcc68 - 0xcc6f
		0x5f28, 0x5f22, 0x5f23, 0x5f24, 0x5f54, 0x5f82, 0x5f7e, 0x5f7d, // 0xcc70 - 0xcc77
		0x5fde, 0x5fe5, 0x602d, 0x6026, 0x6019, 0x6032, 0x600b, 0x0000, // 0xcc78 - 0xcc7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc80 - 0xcc87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc88 - 0xcc8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc90 - 0xcc97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcc98 - 0xcc9f
		0x0000, 0x736d, 0x631e, 0x8e4b, 0x8e0f, 0x80ce, 0x82d4, 0x62ac, // 0xcca0 - 0xcca7
		0x53f0, 0x6cf0, 0x915e, 0x592a, 0x6001, 0x6c70, 0x574d, 0x644a, // 0xcca8 - 0xccaf
		0x8d2a, 0x762b, 0x6ee9, 0x575b, 0x6a80, 0x75f0, 0x6f6d, 0x8c2d, // 0xccb0 - 0xccb7
		0x8c08, 0x5766, 0x6bef, 0x8892, 0x78b3, 0x63a2, 0x53f9, 0x70ad, // 0xccb8 - 0xccbf
		0x6c64, 0x5858, 0x642a, 0x5802, 0x68e0, 0x819b, 0x5510, 0x7cd6, // 0xccc0 - 0xccc7
		0x5018, 0x8eba, 0x6dcc, 0x8d9f, 0x70eb, 0x638f, 0x6d9b, 0x6ed4, // 0xccc8 - 0xcccf
		0x7ee6, 0x8404, 0x6843, 0x9003, 0x6dd8, 0x9676, 0x8ba8, 0x5957, // 0xccd0 - 0xccd7
		0x7279, 0x85e4, 0x817e, 0x75bc, 0x8a8a, 0x68af, 0x5254, 0x8e22, // 0xccd8 - 0xccdf
		0x9511, 0x63d0, 0x9898, 0x8e44, 0x557c, 0x4f53, 0x66ff, 0x568f, // 0xcce0 - 0xcce7
		0x60d5, 0x6d95, 0x5243, 0x5c49, 0x5929, 0x6dfb, 0x586b, 0x7530, // 0xcce8 - 0xccef
		0x751c, 0x606c, 0x8214, 0x8146, 0x6311, 0x6761, 0x8fe2, 0x773a, // 0xccf0 - 0xccf7
		0x8df3, 0x8d34, 0x94c1, 0x5e16, 0x5385, 0x542c, 0x70c3, 0x0000, // 0xccf8 - 0xccff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd00 - 0xcd07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd08 - 0xcd0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd10 - 0xcd17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd18 - 0xcd1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd20 - 0xcd27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd28 - 0xcd2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd30 - 0xcd37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd38 - 0xcd3f
		0x6cd2, 0x6cdd, 0x6cb4, 0x6c8a, 0x6c9d, 0x6c80, 0x6cde, 0x6cc0, // 0xcd40 - 0xcd47
		0x6d30, 0x6ccd, 0x6cc7, 0x6cb0, 0x6cf9, 0x6ccf, 0x6ce9, 0x6cd1, // 0xcd48 - 0xcd4f
		0x7094, 0x7098, 0x7085, 0x7093, 0x7086, 0x7084, 0x7091, 0x7096, // 0xcd50 - 0xcd57
		0x7082, 0x709a, 0x7083, 0x726a, 0x72d6, 0x72cb, 0x72d8, 0x72c9, // 0xcd58 - 0xcd5f
		0x72dc, 0x72d2, 0x72d4, 0x72da, 0x72cc, 0x72d1, 0x73a4, 0x73a1, // 0xcd60 - 0xcd67
		0x73ad, 0x73a6, 0x73a2, 0x73a0, 0x73ac, 0x739d, 0x74dd, 0x74e8, // 0xcd68 - 0xcd6f
		0x753f, 0x7540, 0x753e, 0x758c, 0x7598, 0x76af, 0x76f3, 0x76f1, // 0xcd70 - 0xcd77
		0x76f0, 0x76f5, 0x77f8, 0x77fc, 0x77f9, 0x77fb, 0x77fa, 0x0000, // 0xcd78 - 0xcd7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd80 - 0xcd87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd88 - 0xcd8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd90 - 0xcd97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcd98 - 0xcd9f
		0x0000, 0x6c40, 0x5ef7, 0x505c, 0x4ead, 0x5ead, 0x633a, 0x8247, // 0xcda0 - 0xcda7
		0x901a, 0x6850, 0x916e, 0x77b3, 0x540c, 0x94dc, 0x5f64, 0x7ae5, // 0xcda8 - 0xcdaf
		0x6876, 0x6345, 0x7b52, 0x7edf, 0x75db, 0x5077, 0x6295, 0x5934, // 0xcdb0 - 0xcdb7
		0x900f, 0x51f8, 0x79c3, 0x7a81, 0x56fe, 0x5f92, 0x9014, 0x6d82, // 0xcdb8 - 0xcdbf
		0x5c60, 0x571f, 0x5410, 0x5154, 0x6e4d, 0x56e2, 0x63a8, 0x9893, // 0xcdc0 - 0xcdc7
		0x817f, 0x8715, 0x892a, 0x9000, 0x541e, 0x5c6f, 0x81c0, 0x62d6, // 0xcdc8 - 0xcdcf
		0x6258, 0x8131, 0x9e35, 0x9640, 0x9a6e, 0x9a7c, 0x692d, 0x59a5, // 0xcdd0 - 0xcdd7
		0x62d3, 0x553e, 0x6316, 0x54c7, 0x86d9, 0x6d3c, 0x5a03, 0x74e6, // 0xcdd8 - 0xcddf
		0x889c, 0x6b6a, 0x5916, 0x8c4c, 0x5f2f, 0x6e7e, 0x73a9, 0x987d, // 0xcde0 - 0xcde7
		0x4e38, 0x70f7, 0x5b8c, 0x7897, 0x633d, 0x665a, 0x7696, 0x60cb, // 0xcde8 - 0xcdef
		0x5b9b, 0x5a49, 0x4e07, 0x8155, 0x6c6a, 0x738b, 0x4ea1, 0x6789, // 0xcdf0 - 0xcdf7
		0x7f51, 0x5f80, 0x65fa, 0x671b, 0x5fd8, 0x5984, 0x5a01, 0x0000, // 0xcdf8 - 0xcdff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce00 - 0xce07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce08 - 0xce0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce10 - 0xce17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce18 - 0xce1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce20 - 0xce27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce28 - 0xce2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce30 - 0xce37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce38 - 0xce3f
		0x54c3, 0x830d, 0x54b7, 0x54ae, 0x54d6, 0x54b6, 0x54c5, 0x54c6, // 0xce40 - 0xce47
		0x54a0, 0x5470, 0x54bc, 0x54a2, 0x54be, 0x5472, 0x54de, 0x54b0, // 0xce48 - 0xce4f
		0x57b5, 0x579e, 0x579f, 0x57a4, 0x578c, 0x5797, 0x579d, 0x579b, // 0xce50 - 0xce57
		0x5794, 0x5798, 0x578f, 0x5799, 0x57a5, 0x579a, 0x5795, 0x58f4, // 0xce58 - 0xce5f
		0x590d, 0x5953, 0x59e1, 0x59de, 0x59ee, 0x5a00, 0x59f1, 0x59dd, // 0xce60 - 0xce67
		0x59fa, 0x59fd, 0x59fc, 0x59f6, 0x59e4, 0x59f2, 0x59f7, 0x59db, // 0xce68 - 0xce6f
		0x59e9, 0x59f3, 0x59f5, 0x59e0, 0x59fe, 0x59f4, 0x59ed, 0x5ba8, // 0xce70 - 0xce77
		0x5c4c, 0x5cd0, 0x5cd8, 0x5ccc, 0x5cd7, 0x5ccb, 0x5cdb, 0x0000, // 0xce78 - 0xce7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce80 - 0xce87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce88 - 0xce8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce90 - 0xce97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xce98 - 0xce9f
		0x0000, 0x5dcd, 0x5fae, 0x5371, 0x97e6, 0x8fdd, 0x6845, 0x56f4, // 0xcea0 - 0xcea7
		0x552f, 0x60df, 0x4e3a, 0x6f4d, 0x7ef4, 0x82c7, 0x840e, 0x59d4, // 0xcea8 - 0xceaf
		0x4f1f, 0x4f2a, 0x5c3e, 0x7eac, 0x672a, 0x851a, 0x5473, 0x754f, // 0xceb0 - 0xceb7
		0x80c3, 0x5582, 0x9b4f, 0x4f4d, 0x6e2d, 0x8c13, 0x5c09, 0x6170, // 0xceb8 - 0xcebf
		0x536b, 0x761f, 0x6e29, 0x868a, 0x6587, 0x95fb, 0x7eb9, 0x543b, // 0xcec0 - 0xcec7
		0x7a33, 0x7d0a, 0x95ee, 0x55e1, 0x7fc1, 0x74ee, 0x631d, 0x8717, // 0xcec8 - 0xcecf
		0x6da1, 0x7a9d, 0x6211, 0x65a1, 0x5367, 0x63e1, 0x6c83, 0x5deb, // 0xced0 - 0xced7
		0x545c, 0x94a8, 0x4e4c, 0x6c61, 0x8bec, 0x5c4b, 0x65e0, 0x829c, // 0xced8 - 0xcedf
		0x68a7, 0x543e, 0x5434, 0x6bcb, 0x6b66, 0x4e94, 0x6342, 0x5348, // 0xcee0 - 0xcee7
		0x821e, 0x4f0d, 0x4fae, 0x575e, 0x620a, 0x96fe, 0x6664, 0x7269, // 0xcee8 - 0xceef
		0x52ff, 0x52a1, 0x609f, 0x8bef, 0x6614, 0x7199, 0x6790, 0x897f, // 0xcef0 - 0xcef7
		0x7852, 0x77fd, 0x6670, 0x563b, 0x5438, 0x9521, 0x727a, 0x0000, // 0xcef8 - 0xceff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf00 - 0xcf07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf08 - 0xcf0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf10 - 0xcf17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf18 - 0xcf1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf20 - 0xcf27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf28 - 0xcf2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf30 - 0xcf37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf38 - 0xcf3f
		0x67dc, 0x67bb, 0x67f8, 0x67d8, 0x67c0, 0x67b7, 0x67c5, 0x67eb, // 0xcf40 - 0xcf47
		0x67e4, 0x67df, 0x67b5, 0x67cd, 0x67b3, 0x67f7, 0x67f6, 0x67ee, // 0xcf48 - 0xcf4f
		0x67e3, 0x67c2, 0x67b9, 0x67ce, 0x67e7, 0x67f0, 0x67b2, 0x67fc, // 0xcf50 - 0xcf57
		0x67c6, 0x67ed, 0x67cc, 0x67ae, 0x67e6, 0x67db, 0x67fa, 0x67c9, // 0xcf58 - 0xcf5f
		0x67ca, 0x67c3, 0x67ea, 0x67cb, 0x6b28, 0x6b82, 0x6b84, 0x6bb6, // 0xcf60 - 0xcf67
		0x6bd6, 0x6bd8, 0x6be0, 0x6c20, 0x6c21, 0x6d28, 0x6d34, 0x6d2d, // 0xcf68 - 0xcf6f
		0x6d1f, 0x6d3c, 0x6d3f, 0x6d12, 0x6d0a, 0x6cda, 0x6d33, 0x6d04, // 0xcf70 - 0xcf77
		0x6d19, 0x6d3a, 0x6d1a, 0x6d11, 0x6d00, 0x6d1d, 0x6d42, 0x0000, // 0xcf78 - 0xcf7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf80 - 0xcf87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf88 - 0xcf8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf90 - 0xcf97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xcf98 - 0xcf9f
		0x0000, 0x7a00, 0x606f, 0x5e0c, 0x6089, 0x819d, 0x5915, 0x60dc, // 0xcfa0 - 0xcfa7
		0x7184, 0x70ef, 0x6eaa, 0x6c50, 0x7280, 0x6a84, 0x88ad, 0x5e2d, // 0xcfa8 - 0xcfaf
		0x4e60, 0x5ab3, 0x559c, 0x94e3, 0x6d17, 0x7cfb, 0x9699, 0x620f, // 0xcfb0 - 0xcfb7
		0x7ec6, 0x778e, 0x867e, 0x5323, 0x971e, 0x8f96, 0x6687, 0x5ce1, // 0xcfb8 - 0xcfbf
		0x4fa0, 0x72ed, 0x4e0b, 0x53a6, 0x590f, 0x5413, 0x6380, 0x9528, // 0xcfc0 - 0xcfc7
		0x5148, 0x4ed9, 0x9c9c, 0x7ea4, 0x54b8, 0x8d24, 0x8854, 0x8237, // 0xcfc8 - 0xcfcf
		0x95f2, 0x6d8e, 0x5f26, 0x5acc, 0x663e, 0x9669, 0x73b0, 0x732e, // 0xcfd0 - 0xcfd7
		0x53bf, 0x817a, 0x9985, 0x7fa1, 0x5baa, 0x9677, 0x9650, 0x7ebf, // 0xcfd8 - 0xcfdf
		0x76f8, 0x53a2, 0x9576, 0x9999, 0x7bb1, 0x8944, 0x6e58, 0x4e61, // 0xcfe0 - 0xcfe7
		0x7fd4, 0x7965, 0x8be6, 0x60f3, 0x54cd, 0x4eab, 0x9879, 0x5df7, // 0xcfe8 - 0xcfef
		0x6a61, 0x50cf, 0x5411, 0x8c61, 0x8427, 0x785d, 0x9704, 0x524a, // 0xcff0 - 0xcff7
		0x54ee, 0x56a3, 0x9500, 0x6d88, 0x5bb5, 0x6dc6, 0x6653, 0x0000, // 0xcff8 - 0xcfff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd000 - 0xd007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd008 - 0xd00f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd010 - 0xd017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd018 - 0xd01f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd020 - 0xd027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd028 - 0xd02f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd030 - 0xd037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd038 - 0xd03f
		0x7a7e, 0x7ad1, 0x7b00, 0x7b01, 0x7c7a, 0x7c78, 0x7c79, 0x7c7f, // 0xd040 - 0xd047
		0x7c80, 0x7c81, 0x7d03, 0x7d08, 0x7d01, 0x7f58, 0x7f91, 0x7f8d, // 0xd048 - 0xd04f
		0x7fbe, 0x8007, 0x800e, 0x800f, 0x8014, 0x8037, 0x80d8, 0x80c7, // 0xd050 - 0xd057
		0x80e0, 0x80d1, 0x80c8, 0x80c2, 0x80d0, 0x80c5, 0x80e3, 0x80d9, // 0xd058 - 0xd05f
		0x80dc, 0x80ca, 0x80d5, 0x80c9, 0x80cf, 0x80d7, 0x80e6, 0x80cd, // 0xd060 - 0xd067
		0x81ff, 0x8221, 0x8294, 0x82d9, 0x82fe, 0x82f9, 0x8307, 0x82e8, // 0xd068 - 0xd06f
		0x8300, 0x82d5, 0x833a, 0x82eb, 0x82d6, 0x82f4, 0x82ec, 0x82e1, // 0xd070 - 0xd077
		0x82f2, 0x82f5, 0x830c, 0x82fb, 0x82f6, 0x82f0, 0x82ea, 0x0000, // 0xd078 - 0xd07f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd080 - 0xd087
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd088 - 0xd08f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd090 - 0xd097
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd098 - 0xd09f
		0x0000, 0x5c0f, 0x5b5d, 0x6821, 0x8096, 0x5578, 0x7b11, 0x6548, // 0xd0a0 - 0xd0a7
		0x6954, 0x4e9b, 0x6b47, 0x874e, 0x978b, 0x534f, 0x631f, 0x643a, // 0xd0a8 - 0xd0af
		0x90aa, 0x659c, 0x80c1, 0x8c10, 0x5199, 0x68b0, 0x5378, 0x87f9, // 0xd0b0 - 0xd0b7
		0x61c8, 0x6cc4, 0x6cfb, 0x8c22, 0x5c51, 0x85aa, 0x82af, 0x950c, // 0xd0b8 - 0xd0bf
		0x6b23, 0x8f9b, 0x65b0, 0x5ffb, 0x5fc3, 0x4fe1, 0x8845, 0x661f, // 0xd0c0 - 0xd0c7
		0x8165, 0x7329, 0x60fa, 0x5174, 0x5211, 0x578b, 0x5f62, 0x90a2, // 0xd0c8 - 0xd0cf
		0x884c, 0x9192, 0x5e78, 0x674f, 0x6027, 0x59d3, 0x5144, 0x51f6, // 0xd0d0 - 0xd0d7
		0x80f8, 0x5308, 0x6c79, 0x96c4, 0x718a, 0x4f11, 0x4fee, 0x7f9e, // 0xd0d8 - 0xd0df
		0x673d, 0x55c5, 0x9508, 0x79c0, 0x8896, 0x7ee3, 0x589f, 0x620c, // 0xd0e0 - 0xd0e7
		0x9700, 0x865a, 0x5618, 0x987b, 0x5f90, 0x8bb8, 0x84c4, 0x9157, // 0xd0e8 - 0xd0ef
		0x53d9, 0x65ed, 0x5e8f, 0x755c, 0x6064, 0x7d6e, 0x5a7f, 0x7eea, // 0xd0f0 - 0xd0f7
		0x7eed, 0x8f69, 0x55a7, 0x5ba3, 0x60ac, 0x65cb, 0x7384, 0x0000, // 0xd0f8 - 0xd0ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd100 - 0xd107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd108 - 0xd10f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd110 - 0xd117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd118 - 0xd11f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd120 - 0xd127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd128 - 0xd12f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd130 - 0xd137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd138 - 0xd13f
		0x550a, 0x54fb, 0x54f7, 0x54f8, 0x54e0, 0x550e, 0x5503, 0x550b, // 0xd140 - 0xd147
		0x5701, 0x5702, 0x57cc, 0x5832, 0x57d5, 0x57d2, 0x57ba, 0x57c6, // 0xd148 - 0xd14f
		0x57bd, 0x57bc, 0x57b8, 0x57b6, 0x57bf, 0x57c7, 0x57d0, 0x57b9, // 0xd150 - 0xd157
		0x57c1, 0x590e, 0x594a, 0x5a19, 0x5a16, 0x5a2d, 0x5a2e, 0x5a15, // 0xd158 - 0xd15f
		0x5a0f, 0x5a17, 0x5a0a, 0x5a1e, 0x5a33, 0x5b6c, 0x5ba7, 0x5bad, // 0xd160 - 0xd167
		0x5bac, 0x5c03, 0x5c56, 0x5c54, 0x5cec, 0x5cff, 0x5cee, 0x5cf1, // 0xd168 - 0xd16f
		0x5cf7, 0x5d00, 0x5cf9, 0x5e29, 0x5e28, 0x5ea8, 0x5eae, 0x5eaa, // 0xd170 - 0xd177
		0x5eac, 0x5f33, 0x5f30, 0x5f67, 0x605d, 0x605a, 0x6067, 0x0000, // 0xd178 - 0xd17f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd180 - 0xd187
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd188 - 0xd18f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd190 - 0xd197
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd198 - 0xd19f
		0x0000, 0x9009, 0x7663, 0x7729, 0x7eda, 0x9774, 0x859b, 0x5b66, // 0xd1a0 - 0xd1a7
		0x7a74, 0x96ea, 0x8840, 0x52cb, 0x718f, 0x5faa, 0x65ec, 0x8be2, // 0xd1a8 - 0xd1af
		0x5bfb, 0x9a6f, 0x5de1, 0x6b89, 0x6c5b, 0x8bad, 0x8baf, 0x900a, // 0xd1b0 - 0xd1b7
		0x8fc5, 0x538b, 0x62bc, 0x9e26, 0x9e2d, 0x5440, 0x4e2b, 0x82bd, // 0xd1b8 - 0xd1bf
		0x7259, 0x869c, 0x5d16, 0x8859, 0x6daf, 0x96c5, 0x54d1, 0x4e9a, // 0xd1c0 - 0xd1c7
		0x8bb6, 0x7109, 0x54bd, 0x9609, 0x70df, 0x6df9, 0x76d0, 0x4e25, // 0xd1c8 - 0xd1cf
		0x7814, 0x8712, 0x5ca9, 0x5ef6, 0x8a00, 0x989c, 0x960e, 0x708e, // 0xd1d0 - 0xd1d7
		0x6cbf, 0x5944, 0x63a9, 0x773c, 0x884d, 0x6f14, 0x8273, 0x5830, // 0xd1d8 - 0xd1df
		0x71d5, 0x538c, 0x781a, 0x96c1, 0x5501, 0x5f66, 0x7130, 0x5bb4, // 0xd1e0 - 0xd1e7
		0x8c1a, 0x9a8c, 0x6b83, 0x592e, 0x9e2f, 0x79e7, 0x6768, 0x626c, // 0xd1e8 - 0xd1ef
		0x4f6f, 0x75a1, 0x7f8a, 0x6d0b, 0x9633, 0x6c27, 0x4ef0, 0x75d2, // 0xd1f0 - 0xd1f7
		0x517b, 0x6837, 0x6f3e, 0x9080, 0x8170, 0x5996, 0x7476, 0x0000, // 0xd1f8 - 0xd1ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd200 - 0xd207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd208 - 0xd20f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd210 - 0xd217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd218 - 0xd21f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd220 - 0xd227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd228 - 0xd22f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd230 - 0xd237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd238 - 0xd23f
		0x6be8, 0x6be3, 0x6be2, 0x6be7, 0x6c25, 0x6d7a, 0x6d63, 0x6d64, // 0xd240 - 0xd247
		0x6d76, 0x6d0d, 0x6d61, 0x6d92, 0x6d58, 0x6d62, 0x6d6d, 0x6d6f, // 0xd248 - 0xd24f
		0x6d91, 0x6d8d, 0x6def, 0x6d7f, 0x6d86, 0x6d5e, 0x6d67, 0x6d60, // 0xd250 - 0xd257
		0x6d97, 0x6d70, 0x6d7c, 0x6d5f, 0x6d82, 0x6d98, 0x6d2f, 0x6d68, // 0xd258 - 0xd25f
		0x6d8b, 0x6d7e, 0x6d80, 0x6d84, 0x6d16, 0x6d83, 0x6d7b, 0x6d7d, // 0xd260 - 0xd267
		0x6d75, 0x6d90, 0x70dc, 0x70d3, 0x70d1, 0x70dd, 0x70cb, 0x7f39, // 0xd268 - 0xd26f
		0x70e2, 0x70d7, 0x70d2, 0x70de, 0x70e0, 0x70d4, 0x70cd, 0x70c5, // 0xd270 - 0xd277
		0x70c6, 0x70c7, 0x70da, 0x70ce, 0x70e1, 0x7242, 0x7278, 0x0000, // 0xd278 - 0xd27f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd280 - 0xd287
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd288 - 0xd28f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd290 - 0xd297
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd298 - 0xd29f
		0x0000, 0x6447, 0x5c27, 0x9065, 0x7a91, 0x8c23, 0x59da, 0x54ac, // 0xd2a0 - 0xd2a7
		0x8200, 0x836f, 0x8981, 0x8000, 0x6930, 0x564e, 0x8036, 0x7237, // 0xd2a8 - 0xd2af
		0x91ce, 0x51b6, 0x4e5f, 0x9875, 0x6396, 0x4e1a, 0x53f6, 0x66f3, // 0xd2b0 - 0xd2b7
		0x814b, 0x591c, 0x6db2, 0x4e00, 0x58f9, 0x533b, 0x63d6, 0x94f1, // 0xd2b8 - 0xd2bf
		0x4f9d, 0x4f0a, 0x8863, 0x9890, 0x5937, 0x9057, 0x79fb, 0x4eea, // 0xd2c0 - 0xd2c7
		0x80f0, 0x7591, 0x6c82, 0x5b9c, 0x59e8, 0x5f5d, 0x6905, 0x8681, // 0xd2c8 - 0xd2cf
		0x501a, 0x5df2, 0x4e59, 0x77e3, 0x4ee5, 0x827a, 0x6291, 0x6613, // 0xd2d0 - 0xd2d7
		0x9091, 0x5c79, 0x4ebf, 0x5f79, 0x81c6, 0x9038, 0x8084, 0x75ab, // 0xd2d8 - 0xd2df
		0x4ea6, 0x88d4, 0x610f, 0x6bc5, 0x5fc6, 0x4e49, 0x76ca, 0x6ea2, // 0xd2e0 - 0xd2e7
		0x8be3, 0x8bae, 0x8c0a, 0x8bd1, 0x5f02, 0x7ffc, 0x7fcc, 0x7ece, // 0xd2e8 - 0xd2ef
		0x8335, 0x836b, 0x56e0, 0x6bb7, 0x97f3, 0x9634, 0x59fb, 0x541f, // 0xd2f0 - 0xd2f7
		0x94f6, 0x6deb, 0x5bc5, 0x996e, 0x5c39, 0x5f15, 0x9690, 0x0000, // 0xd2f8 - 0xd2ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd300 - 0xd307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd308 - 0xd30f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd310 - 0xd317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd318 - 0xd31f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd320 - 0xd327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd328 - 0xd32f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd330 - 0xd337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd338 - 0xd33f
		0x7b04, 0x7b13, 0x7b05, 0x7b0f, 0x7b08, 0x7b0a, 0x7b0e, 0x7b09, // 0xd340 - 0xd347
		0x7b12, 0x7c84, 0x7c91, 0x7c8a, 0x7c8c, 0x7c88, 0x7c8d, 0x7c85, // 0xd348 - 0xd34f
		0x7d1e, 0x7d1d, 0x7d11, 0x7d0e, 0x7d18, 0x7d16, 0x7d13, 0x7d1f, // 0xd350 - 0xd357
		0x7d12, 0x7d0f, 0x7d0c, 0x7f5c, 0x7f61, 0x7f5e, 0x7f60, 0x7f5d, // 0xd358 - 0xd35f
		0x7f5b, 0x7f96, 0x7f92, 0x7fc3, 0x7fc2, 0x7fc0, 0x8016, 0x803e, // 0xd360 - 0xd367
		0x8039, 0x80fa, 0x80f2, 0x80f9, 0x80f5, 0x8101, 0x80fb, 0x8100, // 0xd368 - 0xd36f
		0x8201, 0x822f, 0x8225, 0x8333, 0x832d, 0x8344, 0x8319, 0x8351, // 0xd370 - 0xd377
		0x8325, 0x8356, 0x833f, 0x8341, 0x8326, 0x831c, 0x8322, 0x0000, // 0xd378 - 0xd37f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd380 - 0xd387
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd388 - 0xd38f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd390 - 0xd397
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd398 - 0xd39f
		0x0000, 0x5370, 0x82f1, 0x6a31, 0x5a74, 0x9e70, 0x5e94, 0x7f28, // 0xd3a0 - 0xd3a7
		0x83b9, 0x8424, 0x8425, 0x8367, 0x8747, 0x8fce, 0x8d62, 0x76c8, // 0xd3a8 - 0xd3af
		0x5f71, 0x9896, 0x786c, 0x6620, 0x54df, 0x62e5, 0x4f63, 0x81c3, // 0xd3b0 - 0xd3b7
		0x75c8, 0x5eb8, 0x96cd, 0x8e0a, 0x86f9, 0x548f, 0x6cf3, 0x6d8c, // 0xd3b8 - 0xd3bf
		0x6c38, 0x607f, 0x52c7, 0x7528, 0x5e7d, 0x4f18, 0x60a0, 0x5fe7, // 0xd3c0 - 0xd3c7
		0x5c24, 0x7531, 0x90ae, 0x94c0, 0x72b9, 0x6cb9, 0x6e38, 0x9149, // 0xd3c8 - 0xd3cf
		0x6709, 0x53cb, 0x53f3, 0x4f51, 0x91c9, 0x8bf1, 0x53c8, 0x5e7c, // 0xd3d0 - 0xd3d7
		0x8fc2, 0x6de4, 0x4e8e, 0x76c2, 0x6986, 0x865e, 0x611a, 0x8206, // 0xd3d8 - 0xd3df
		0x4f59, 0x4fde, 0x903e, 0x9c7c, 0x6109, 0x6e1d, 0x6e14, 0x9685, // 0xd3e0 - 0xd3e7
		0x4e88, 0x5a31, 0x96e8, 0x4e0e, 0x5c7f, 0x79b9, 0x5b87, 0x8bed, // 0xd3e8 - 0xd3ef
		0x7fbd, 0x7389, 0x57df, 0x828b, 0x90c1, 0x5401, 0x9047, 0x55bb, // 0xd3f0 - 0xd3f7
		0x5cea, 0x5fa1, 0x6108, 0x6b32, 0x72f1, 0x80b2, 0x8a89, 0x0000, // 0xd3f8 - 0xd3ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd400 - 0xd407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd408 - 0xd40f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd410 - 0xd417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd418 - 0xd41f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd420 - 0xd427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd428 - 0xd42f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd430 - 0xd437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd438 - 0xd43f
		0x914e, 0x914f, 0x91d5, 0x91e2, 0x91da, 0x965c, 0x965f, 0x96bc, // 0xd440 - 0xd447
		0x98e3, 0x9adf, 0x9b2f, 0x4e7f, 0x5070, 0x506a, 0x5061, 0x505e, // 0xd448 - 0xd44f
		0x5060, 0x5053, 0x504b, 0x505d, 0x5072, 0x5048, 0x504d, 0x5041, // 0xd450 - 0xd457
		0x505b, 0x504a, 0x5062, 0x5015, 0x5045, 0x505f, 0x5069, 0x506b, // 0xd458 - 0xd45f
		0x5063, 0x5064, 0x5046, 0x5040, 0x506e, 0x5073, 0x5057, 0x5051, // 0xd460 - 0xd467
		0x51d0, 0x526b, 0x526d, 0x526c, 0x526e, 0x52d6, 0x52d3, 0x532d, // 0xd468 - 0xd46f
		0x539c, 0x5575, 0x5576, 0x553c, 0x554d, 0x5550, 0x5534, 0x552a, // 0xd470 - 0xd477
		0x5551, 0x5562, 0x5536, 0x5535, 0x5530, 0x5552, 0x5545, 0x0000, // 0xd478 - 0xd47f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd480 - 0xd487
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd488 - 0xd48f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd490 - 0xd497
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd498 - 0xd49f
		0x0000, 0x6d74, 0x5bd3, 0x88d5, 0x9884, 0x8c6b, 0x9a6d, 0x9e33, // 0xd4a0 - 0xd4a7
		0x6e0a, 0x51a4, 0x5143, 0x57a3, 0x8881, 0x539f, 0x63f4, 0x8f95, // 0xd4a8 - 0xd4af
		0x56ed, 0x5458, 0x5706, 0x733f, 0x6e90, 0x7f18, 0x8fdc, 0x82d1, // 0xd4b0 - 0xd4b7
		0x613f, 0x6028, 0x9662, 0x66f0, 0x7ea6, 0x8d8a, 0x8dc3, 0x94a5, // 0xd4b8 - 0xd4bf
		0x5cb3, 0x7ca4, 0x6708, 0x60a6, 0x9605, 0x8018, 0x4e91, 0x90e7, // 0xd4c0 - 0xd4c7
		0x5300, 0x9668, 0x5141, 0x8fd0, 0x8574, 0x915d, 0x6655, 0x97f5, // 0xd4c8 - 0xd4cf
		0x5b55, 0x531d, 0x7838, 0x6742, 0x683d, 0x54c9, 0x707e, 0x5bb0, // 0xd4d0 - 0xd4d7
		0x8f7d, 0x518d, 0x5728, 0x54b1, 0x6512, 0x6682, 0x8d5e, 0x8d43, // 0xd4d8 - 0xd4df
		0x810f, 0x846c, 0x906d, 0x7cdf, 0x51ff, 0x85fb, 0x67a3, 0x65e9, // 0xd4e0 - 0xd4e7
		0x6fa1, 0x86a4, 0x8e81, 0x566a, 0x9020, 0x7682, 0x7076, 0x71e5, // 0xd4e8 - 0xd4ef
		0x8d23, 0x62e9, 0x5219, 0x6cfd, 0x8d3c, 0x600e, 0x589e, 0x618e, // 0xd4f0 - 0xd4f7
		0x66fe, 0x8d60, 0x624e, 0x55b3, 0x6e23, 0x672d, 0x8f67, 0x0000, // 0xd4f8 - 0xd4ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd500 - 0xd507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd508 - 0xd50f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd510 - 0xd517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd518 - 0xd51f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd520 - 0xd527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd528 - 0xd52f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd530 - 0xd537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd538 - 0xd53f
		0x5d30, 0x5d12, 0x5d23, 0x5d1f, 0x5d2e, 0x5e3e, 0x5e34, 0x5eb1, // 0xd540 - 0xd547
		0x5eb4, 0x5eb9, 0x5eb2, 0x5eb3, 0x5f36, 0x5f38, 0x5f9b, 0x5f96, // 0xd548 - 0xd54f
		0x5f9f, 0x608a, 0x6090, 0x6086, 0x60be, 0x60b0, 0x60ba, 0x60d3, // 0xd550 - 0xd557
		0x60d4, 0x60cf, 0x60e4, 0x60d9, 0x60dd, 0x60c8, 0x60b1, 0x60db, // 0xd558 - 0xd55f
		0x60b7, 0x60ca, 0x60bf, 0x60c3, 0x60cd, 0x60c0, 0x6332, 0x6365, // 0xd560 - 0xd567
		0x638a, 0x6382, 0x637d, 0x63bd, 0x639e, 0x63ad, 0x639d, 0x6397, // 0xd568 - 0xd56f
		0x63ab, 0x638e, 0x636f, 0x6387, 0x6390, 0x636e, 0x63af, 0x6375, // 0xd570 - 0xd577
		0x639c, 0x636d, 0x63ae, 0x637c, 0x63a4, 0x633b, 0x639f, 0x0000, // 0xd578 - 0xd57f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd580 - 0xd587
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd588 - 0xd58f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd590 - 0xd597
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd598 - 0xd59f
		0x0000, 0x94e1, 0x95f8, 0x7728, 0x6805, 0x69a8, 0x548b, 0x4e4d, // 0xd5a0 - 0xd5a7
		0x70b8, 0x8bc8, 0x6458, 0x658b, 0x5b85, 0x7a84, 0x503a, 0x5be8, // 0xd5a8 - 0xd5af
		0x77bb, 0x6be1, 0x8a79, 0x7c98, 0x6cbe, 0x76cf, 0x65a9, 0x8f97, // 0xd5b0 - 0xd5b7
		0x5d2d, 0x5c55, 0x8638, 0x6808, 0x5360, 0x6218, 0x7ad9, 0x6e5b, // 0xd5b8 - 0xd5bf
		0x7efd, 0x6a1f, 0x7ae0, 0x5f70, 0x6f33, 0x5f20, 0x638c, 0x6da8, // 0xd5c0 - 0xd5c7
		0x6756, 0x4e08, 0x5e10, 0x8d26, 0x4ed7, 0x80c0, 0x7634, 0x969c, // 0xd5c8 - 0xd5cf
		0x62db, 0x662d, 0x627e, 0x6cbc, 0x8d75, 0x7167, 0x7f69, 0x5146, // 0xd5d0 - 0xd5d7
		0x8087, 0x53ec, 0x906e, 0x6298, 0x54f2, 0x86f0, 0x8f99, 0x8005, // 0xd5d8 - 0xd5df
		0x9517, 0x8517, 0x8fd9, 0x6d59, 0x73cd, 0x659f, 0x771f, 0x7504, // 0xd5e0 - 0xd5e7
		0x7827, 0x81fb, 0x8d1e, 0x9488, 0x4fa6, 0x6795, 0x75b9, 0x8bca, // 0xd5e8 - 0xd5ef
		0x9707, 0x632f, 0x9547, 0x9635, 0x84b8, 0x6323, 0x7741, 0x5f81, // 0xd5f0 - 0xd5f7
		0x72f0, 0x4e89, 0x6014, 0x6574, 0x62ef, 0x6b63, 0x653f, 0x0000, // 0xd5f8 - 0xd5ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd600 - 0xd607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd608 - 0xd60f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd610 - 0xd617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd618 - 0xd61f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd620 - 0xd627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd628 - 0xd62f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd630 - 0xd637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd638 - 0xd63f
		0x6dd0, 0x6df2, 0x6dd3, 0x6dfd, 0x6dd7, 0x6dcd, 0x6de3, 0x6dbb, // 0xd640 - 0xd647
		0x70fa, 0x710d, 0x70f7, 0x7117, 0x70f4, 0x710c, 0x70f0, 0x7104, // 0xd648 - 0xd64f
		0x70f3, 0x7110, 0x70fc, 0x70ff, 0x7106, 0x7113, 0x7100, 0x70f8, // 0xd650 - 0xd657
		0x70f6, 0x710b, 0x7102, 0x710e, 0x727e, 0x727b, 0x727c, 0x727f, // 0xd658 - 0xd65f
		0x731d, 0x7317, 0x7307, 0x7311, 0x7318, 0x730a, 0x7308, 0x72ff, // 0xd660 - 0xd667
		0x730f, 0x731e, 0x7388, 0x73f6, 0x73f8, 0x73f5, 0x7404, 0x7401, // 0xd668 - 0xd66f
		0x73fd, 0x7407, 0x7400, 0x73fa, 0x73fc, 0x73ff, 0x740c, 0x740b, // 0xd670 - 0xd677
		0x73f4, 0x7408, 0x7564, 0x7563, 0x75ce, 0x75d2, 0x75cf, 0x0000, // 0xd678 - 0xd67f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd680 - 0xd687
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd688 - 0xd68f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd690 - 0xd697
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd698 - 0xd69f
		0x0000, 0x5e27, 0x75c7, 0x90d1, 0x8bc1, 0x829d, 0x679d, 0x652f, // 0xd6a0 - 0xd6a7
		0x5431, 0x8718, 0x77e5, 0x80a2, 0x8102, 0x6c41, 0x4e4b, 0x7ec7, // 0xd6a8 - 0xd6af
		0x804c, 0x76f4, 0x690d, 0x6b96, 0x6267, 0x503c, 0x4f84, 0x5740, // 0xd6b0 - 0xd6b7
		0x6307, 0x6b62, 0x8dbe, 0x53ea, 0x65e8, 0x7eb8, 0x5fd7, 0x631a, // 0xd6b8 - 0xd6bf
		0x63b7, 0x81f3, 0x81f4, 0x7f6e, 0x5e1c, 0x5cd9, 0x5236, 0x667a, // 0xd6c0 - 0xd6c7
		0x79e9, 0x7a1a, 0x8d28, 0x7099, 0x75d4, 0x6ede, 0x6cbb, 0x7a92, // 0xd6c8 - 0xd6cf
		0x4e2d, 0x76c5, 0x5fe0, 0x949f, 0x8877, 0x7ec8, 0x79cd, 0x80bf, // 0xd6d0 - 0xd6d7
		0x91cd, 0x4ef2, 0x4f17, 0x821f, 0x5468, 0x5dde, 0x6d32, 0x8bcc, // 0xd6d8 - 0xd6df
		0x7ca5, 0x8f74, 0x8098, 0x5e1a, 0x5492, 0x76b1, 0x5b99, 0x663c, // 0xd6e0 - 0xd6e7
		0x9aa4, 0x73e0, 0x682a, 0x86db, 0x6731, 0x732a, 0x8bf8, 0x8bdb, // 0xd6e8 - 0xd6ef
		0x9010, 0x7af9, 0x70db, 0x716e, 0x62c4, 0x77a9, 0x5631, 0x4e3b, // 0xd6f0 - 0xd6f7
		0x8457, 0x67f1, 0x52a9, 0x86c0, 0x8d2e, 0x94f8, 0x7b51, 0x0000, // 0xd6f8 - 0xd6ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd700 - 0xd707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd708 - 0xd70f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd710 - 0xd717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd718 - 0xd71f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd720 - 0xd727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd728 - 0xd72f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd730 - 0xd737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd738 - 0xd73f
		0x801e, 0x801b, 0x8047, 0x8043, 0x8048, 0x8118, 0x8125, 0x8119, // 0xd740 - 0xd747
		0x811b, 0x812d, 0x811f, 0x812c, 0x811e, 0x8121, 0x8115, 0x8127, // 0xd748 - 0xd74f
		0x811d, 0x8122, 0x8211, 0x8238, 0x8233, 0x823a, 0x8234, 0x8232, // 0xd750 - 0xd757
		0x8274, 0x8390, 0x83a3, 0x83a8, 0x838d, 0x837a, 0x8373, 0x83a4, // 0xd758 - 0xd75f
		0x8374, 0x838f, 0x8381, 0x8395, 0x8399, 0x8375, 0x8394, 0x83a9, // 0xd760 - 0xd767
		0x837d, 0x8383, 0x838c, 0x839d, 0x839b, 0x83aa, 0x838b, 0x837e, // 0xd768 - 0xd76f
		0x83a5, 0x83af, 0x8388, 0x8397, 0x83b0, 0x837f, 0x83a6, 0x8387, // 0xd770 - 0xd777
		0x83ae, 0x8376, 0x839a, 0x8659, 0x8656, 0x86bf, 0x86b7, 0x0000, // 0xd778 - 0xd77f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd780 - 0xd787
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd788 - 0xd78f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd790 - 0xd797
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd798 - 0xd79f
		0x0000, 0x4f4f, 0x6ce8, 0x795d, 0x9a7b, 0x6293, 0x722a, 0x62fd, // 0xd7a0 - 0xd7a7
		0x4e13, 0x7816, 0x8f6c, 0x64b0, 0x8d5a, 0x7bc6, 0x6869, 0x5e84, // 0xd7a8 - 0xd7af
		0x88c5, 0x5986, 0x649e, 0x58ee, 0x72b6, 0x690e, 0x9525, 0x8ffd, // 0xd7b0 - 0xd7b7
		0x8d58, 0x5760, 0x7f00, 0x8c06, 0x51c6, 0x6349, 0x62d9, 0x5353, // 0xd7b8 - 0xd7bf
		0x684c, 0x7422, 0x8301, 0x914c, 0x5544, 0x7740, 0x707c, 0x6d4a, // 0xd7c0 - 0xd7c7
		0x5179, 0x54a8, 0x8d44, 0x59ff, 0x6ecb, 0x6dc4, 0x5b5c, 0x7d2b, // 0xd7c8 - 0xd7cf
		0x4ed4, 0x7c7d, 0x6ed3, 0x5b50, 0x81ea, 0x6e0d, 0x5b57, 0x9b03, // 0xd7d0 - 0xd7d7
		0x68d5, 0x8e2a, 0x5b97, 0x7efc, 0x603b, 0x7eb5, 0x90b9, 0x8d70, // 0xd7d8 - 0xd7df
		0x594f, 0x63cd, 0x79df, 0x8db3, 0x5352, 0x65cf, 0x7956, 0x8bc5, // 0xd7e0 - 0xd7e7
		0x963b, 0x7ec4, 0x94bb, 0x7e82, 0x5634, 0x9189, 0x6700, 0x7f6a, // 0xd7e8 - 0xd7ef
		0x5c0a, 0x9075, 0x6628, 0x5de6, 0x4f50, 0x67de, 0x505a, 0x4f5c, // 0xd7f0 - 0xd7f7
		0x5750, 0x5ea7, 0x91f3, 0x91f8, 0x91e4, 0x91f9, 0x91ea, 0x0000, // 0xd7f8 - 0xd7ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd800 - 0xd807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd808 - 0xd80f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd810 - 0xd817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd818 - 0xd81f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd820 - 0xd827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd828 - 0xd82f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd830 - 0xd837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd838 - 0xd83f
		0x91eb, 0x91f7, 0x91e8, 0x91ee, 0x957a, 0x9586, 0x9588, 0x967c, // 0xd840 - 0xd847
		0x966d, 0x966b, 0x9671, 0x966f, 0x96bf, 0x976a, 0x9804, 0x98e5, // 0xd848 - 0xd84f
		0x9997, 0x509b, 0x5095, 0x5094, 0x509e, 0x508b, 0x50a3, 0x5083, // 0xd850 - 0xd857
		0x508c, 0x508e, 0x509d, 0x5068, 0x509c, 0x5092, 0x5082, 0x5087, // 0xd858 - 0xd85f
		0x515f, 0x51d4, 0x5312, 0x5311, 0x53a4, 0x53a7, 0x5591, 0x55a8, // 0xd860 - 0xd867
		0x55a5, 0x55ad, 0x5577, 0x5645, 0x55a2, 0x5593, 0x5588, 0x558f, // 0xd868 - 0xd86f
		0x55b5, 0x5581, 0x55a3, 0x5592, 0x55a4, 0x557d, 0x558c, 0x55a6, // 0xd870 - 0xd877
		0x557f, 0x5595, 0x55a1, 0x558e, 0x570c, 0x5829, 0x5837, 0x0000, // 0xd878 - 0xd87f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd880 - 0xd887
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd888 - 0xd88f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd890 - 0xd897
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd898 - 0xd89f
		0x0000, 0x4e8d, 0x4e0c, 0x5140, 0x4e10, 0x5eff, 0x5345, 0x4e15, // 0xd8a0 - 0xd8a7
		0x4e98, 0x4e1e, 0x9b32, 0x5b6c, 0x5669, 0x4e28, 0x79ba, 0x4e3f, // 0xd8a8 - 0xd8af
		0x5315, 0x4e47, 0x592d, 0x723b, 0x536e, 0x6c10, 0x56df, 0x80e4, // 0xd8b0 - 0xd8b7
		0x9997, 0x6bd3, 0x777e, 0x9f17, 0x4e36, 0x4e9f, 0x9f10, 0x4e5c, // 0xd8b8 - 0xd8bf
		0x4e69, 0x4e93, 0x8288, 0x5b5b, 0x556c, 0x560f, 0x4ec4, 0x538d, // 0xd8c0 - 0xd8c7
		0x539d, 0x53a3, 0x53a5, 0x53ae, 0x9765, 0x8d5d, 0x531a, 0x53f5, // 0xd8c8 - 0xd8cf
		0x5326, 0x532e, 0x533e, 0x8d5c, 0x5366, 0x5363, 0x5202, 0x5208, // 0xd8d0 - 0xd8d7
		0x520e, 0x522d, 0x5233, 0x523f, 0x5240, 0x524c, 0x525e, 0x5261, // 0xd8d8 - 0xd8df
		0x525c, 0x84af, 0x527d, 0x5282, 0x5281, 0x5290, 0x5293, 0x5182, // 0xd8e0 - 0xd8e7
		0x7f54, 0x4ebb, 0x4ec3, 0x4ec9, 0x4ec2, 0x4ee8, 0x4ee1, 0x4eeb, // 0xd8e8 - 0xd8ef
		0x4ede, 0x4f1b, 0x4ef3, 0x4f22, 0x4f64, 0x4ef5, 0x4f25, 0x4f27, // 0xd8f0 - 0xd8f7
		0x4f09, 0x4f2b, 0x4f5e, 0x4f67, 0x6538, 0x4f5a, 0x4f5d, 0x0000, // 0xd8f8 - 0xd8ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd900 - 0xd907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd908 - 0xd90f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd910 - 0xd917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd918 - 0xd91f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd920 - 0xd927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd928 - 0xd92f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd930 - 0xd937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd938 - 0xd93f
		0x60f2, 0x610a, 0x6116, 0x6105, 0x60f5, 0x6113, 0x60f8, 0x60fc, // 0xd940 - 0xd947
		0x60fe, 0x60c1, 0x6103, 0x6118, 0x611d, 0x6110, 0x60ff, 0x6104, // 0xd948 - 0xd94f
		0x610b, 0x624a, 0x6394, 0x63b1, 0x63b0, 0x63ce, 0x63e5, 0x63e8, // 0xd950 - 0xd957
		0x63ef, 0x63c3, 0x649d, 0x63f3, 0x63ca, 0x63e0, 0x63f6, 0x63d5, // 0xd958 - 0xd95f
		0x63f2, 0x63f5, 0x6461, 0x63df, 0x63be, 0x63dd, 0x63dc, 0x63c4, // 0xd960 - 0xd967
		0x63d8, 0x63d3, 0x63c2, 0x63c7, 0x63cc, 0x63cb, 0x63c8, 0x63f0, // 0xd968 - 0xd96f
		0x63d7, 0x63d9, 0x6532, 0x6567, 0x656a, 0x6564, 0x655c, 0x6568, // 0xd970 - 0xd977
		0x6565, 0x658c, 0x659d, 0x659e, 0x65ae, 0x65d0, 0x65d2, 0x0000, // 0xd978 - 0xd97f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd980 - 0xd987
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd988 - 0xd98f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd990 - 0xd997
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xd998 - 0xd99f
		0x0000, 0x4f5f, 0x4f57, 0x4f32, 0x4f3d, 0x4f76, 0x4f74, 0x4f91, // 0xd9a0 - 0xd9a7
		0x4f89, 0x4f83, 0x4f8f, 0x4f7e, 0x4f7b, 0x4faa, 0x4f7c, 0x4fac, // 0xd9a8 - 0xd9af
		0x4f94, 0x4fe6, 0x4fe8, 0x4fea, 0x4fc5, 0x4fda, 0x4fe3, 0x4fdc, // 0xd9b0 - 0xd9b7
		0x4fd1, 0x4fdf, 0x4ff8, 0x5029, 0x504c, 0x4ff3, 0x502c, 0x500f, // 0xd9b8 - 0xd9bf
		0x502e, 0x502d, 0x4ffe, 0x501c, 0x500c, 0x5025, 0x5028, 0x507e, // 0xd9c0 - 0xd9c7
		0x5043, 0x5055, 0x5048, 0x504e, 0x506c, 0x507b, 0x50a5, 0x50a7, // 0xd9c8 - 0xd9cf
		0x50a9, 0x50ba, 0x50d6, 0x5106, 0x50ed, 0x50ec, 0x50e6, 0x50ee, // 0xd9d0 - 0xd9d7
		0x5107, 0x510b, 0x4edd, 0x6c3d, 0x4f58, 0x4f65, 0x4fce, 0x9fa0, // 0xd9d8 - 0xd9df
		0x6c46, 0x7c74, 0x516e, 0x5dfd, 0x9ec9, 0x9998, 0x5181, 0x5914, // 0xd9e0 - 0xd9e7
		0x52f9, 0x530d, 0x8a07, 0x5310, 0x51eb, 0x5919, 0x5155, 0x4ea0, // 0xd9e8 - 0xd9ef
		0x5156, 0x4eb3, 0x886e, 0x88a4, 0x4eb5, 0x8114, 0x88d2, 0x7980, // 0xd9f0 - 0xd9f7
		0x5b34, 0x8803, 0x7fb8, 0x51ab, 0x51b1, 0x51bd, 0x51bc, 0x0000, // 0xd9f8 - 0xd9ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda00 - 0xda07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda08 - 0xda0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda10 - 0xda17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda18 - 0xda1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda20 - 0xda27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda28 - 0xda2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda30 - 0xda37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda38 - 0xda3f
		0x6e68, 0x6e5c, 0x6e61, 0x6e31, 0x6e28, 0x6e60, 0x6e71, 0x6e6b, // 0xda40 - 0xda47
		0x6e39, 0x6e22, 0x6e30, 0x6e53, 0x6e65, 0x6e27, 0x6e78, 0x6e64, // 0xda48 - 0xda4f
		0x6e77, 0x6e55, 0x6e79, 0x6e52, 0x6e66, 0x6e35, 0x6e36, 0x6e5a, // 0xda50 - 0xda57
		0x7120, 0x711e, 0x712f, 0x70fb, 0x712e, 0x7131, 0x7123, 0x7125, // 0xda58 - 0xda5f
		0x7122, 0x7132, 0x711f, 0x7128, 0x713a, 0x711b, 0x724b, 0x725a, // 0xda60 - 0xda67
		0x7288, 0x7289, 0x7286, 0x7285, 0x728b, 0x7312, 0x730b, 0x7330, // 0xda68 - 0xda6f
		0x7322, 0x7331, 0x7333, 0x7327, 0x7332, 0x732d, 0x7326, 0x7323, // 0xda70 - 0xda77
		0x7335, 0x730c, 0x742e, 0x742c, 0x7430, 0x742b, 0x7416, 0x0000, // 0xda78 - 0xda7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda80 - 0xda87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda88 - 0xda8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda90 - 0xda97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xda98 - 0xda9f
		0x0000, 0x51c7, 0x5196, 0x51a2, 0x51a5, 0x8ba0, 0x8ba6, 0x8ba7, // 0xdaa0 - 0xdaa7
		0x8baa, 0x8bb4, 0x8bb5, 0x8bb7, 0x8bc2, 0x8bc3, 0x8bcb, 0x8bcf, // 0xdaa8 - 0xdaaf
		0x8bce, 0x8bd2, 0x8bd3, 0x8bd4, 0x8bd6, 0x8bd8, 0x8bd9, 0x8bdc, // 0xdab0 - 0xdab7
		0x8bdf, 0x8be0, 0x8be4, 0x8be8, 0x8be9, 0x8bee, 0x8bf0, 0x8bf3, // 0xdab8 - 0xdabf
		0x8bf6, 0x8bf9, 0x8bfc, 0x8bff, 0x8c00, 0x8c02, 0x8c04, 0x8c07, // 0xdac0 - 0xdac7
		0x8c0c, 0x8c0f, 0x8c11, 0x8c12, 0x8c14, 0x8c15, 0x8c16, 0x8c19, // 0xdac8 - 0xdacf
		0x8c1b, 0x8c18, 0x8c1d, 0x8c1f, 0x8c20, 0x8c21, 0x8c25, 0x8c27, // 0xdad0 - 0xdad7
		0x8c2a, 0x8c2b, 0x8c2e, 0x8c2f, 0x8c32, 0x8c33, 0x8c35, 0x8c36, // 0xdad8 - 0xdadf
		0x5369, 0x537a, 0x961d, 0x9622, 0x9621, 0x9631, 0x962a, 0x963d, // 0xdae0 - 0xdae7
		0x963c, 0x9642, 0x9649, 0x9654, 0x965f, 0x9667, 0x966c, 0x9672, // 0xdae8 - 0xdaef
		0x9674, 0x9688, 0x968d, 0x9697, 0x96b0, 0x9097, 0x909b, 0x909d, // 0xdaf0 - 0xdaf7
		0x9099, 0x90ac, 0x90a1, 0x90b4, 0x90b3, 0x90b6, 0x90ba, 0x0000, // 0xdaf8 - 0xdaff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb00 - 0xdb07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb08 - 0xdb0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb10 - 0xdb17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb18 - 0xdb1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb20 - 0xdb27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb28 - 0xdb2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb30 - 0xdb37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb38 - 0xdb3f
		0x7f66, 0x7fa2, 0x7fa0, 0x7fa1, 0x7fd7, 0x8051, 0x804f, 0x8050, // 0xdb40 - 0xdb47
		0x80fe, 0x80d4, 0x8143, 0x814a, 0x8152, 0x814f, 0x8147, 0x813d, // 0xdb48 - 0xdb4f
		0x814d, 0x813a, 0x81e6, 0x81ee, 0x81f7, 0x81f8, 0x81f9, 0x8204, // 0xdb50 - 0xdb57
		0x823c, 0x823d, 0x823f, 0x8275, 0x833b, 0x83cf, 0x83f9, 0x8423, // 0xdb58 - 0xdb5f
		0x83c0, 0x83e8, 0x8412, 0x83e7, 0x83e4, 0x83fc, 0x83f6, 0x8410, // 0xdb60 - 0xdb67
		0x83c6, 0x83c8, 0x83eb, 0x83e3, 0x83bf, 0x8401, 0x83dd, 0x83e5, // 0xdb68 - 0xdb6f
		0x83d8, 0x83ff, 0x83e1, 0x83cb, 0x83ce, 0x83d6, 0x83f5, 0x83c9, // 0xdb70 - 0xdb77
		0x8409, 0x840f, 0x83de, 0x8411, 0x8406, 0x83c2, 0x83f3, 0x0000, // 0xdb78 - 0xdb7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb80 - 0xdb87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb88 - 0xdb8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb90 - 0xdb97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdb98 - 0xdb9f
		0x0000, 0x90b8, 0x90b0, 0x90cf, 0x90c5, 0x90be, 0x90d0, 0x90c4, // 0xdba0 - 0xdba7
		0x90c7, 0x90d3, 0x90e6, 0x90e2, 0x90dc, 0x90d7, 0x90db, 0x90eb, // 0xdba8 - 0xdbaf
		0x90ef, 0x90fe, 0x9104, 0x9122, 0x911e, 0x9123, 0x9131, 0x912f, // 0xdbb0 - 0xdbb7
		0x9139, 0x9143, 0x9146, 0x520d, 0x5942, 0x52a2, 0x52ac, 0x52ad, // 0xdbb8 - 0xdbbf
		0x52be, 0x54ff, 0x52d0, 0x52d6, 0x52f0, 0x53df, 0x71ee, 0x77cd, // 0xdbc0 - 0xdbc7
		0x5ef4, 0x51f5, 0x51fc, 0x9b2f, 0x53b6, 0x5f01, 0x755a, 0x5def, // 0xdbc8 - 0xdbcf
		0x574c, 0x57a9, 0x57a1, 0x587e, 0x58bc, 0x58c5, 0x58d1, 0x5729, // 0xdbd0 - 0xdbd7
		0x572c, 0x572a, 0x5733, 0x5739, 0x572e, 0x572f, 0x575c, 0x573b, // 0xdbd8 - 0xdbdf
		0x5742, 0x5769, 0x5785, 0x576b, 0x5786, 0x577c, 0x577b, 0x5768, // 0xdbe0 - 0xdbe7
		0x576d, 0x5776, 0x5773, 0x57ad, 0x57a4, 0x578c, 0x57b2, 0x57cf, // 0xdbe8 - 0xdbef
		0x57a7, 0x57b4, 0x5793, 0x57a0, 0x57d5, 0x57d8, 0x57da, 0x57d9, // 0xdbf0 - 0xdbf7
		0x57d2, 0x57b8, 0x57f4, 0x57ef, 0x57f8, 0x57e4, 0x57dd, 0x0000, // 0xdbf8 - 0xdbff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc00 - 0xdc07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc08 - 0xdc0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc10 - 0xdc17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc18 - 0xdc1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc20 - 0xdc27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc28 - 0xdc2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc30 - 0xdc37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc38 - 0xdc3f
		0x8ef9, 0x8ee6, 0x8eee, 0x8ee5, 0x8ef5, 0x8ee7, 0x8ee8, 0x8ef6, // 0xdc40 - 0xdc47
		0x8eeb, 0x8ef1, 0x8eec, 0x8ef4, 0x8ee9, 0x902d, 0x9034, 0x902f, // 0xdc48 - 0xdc4f
		0x9106, 0x912c, 0x9104, 0x90ff, 0x90fc, 0x9108, 0x90f9, 0x90fb, // 0xdc50 - 0xdc57
		0x9101, 0x9100, 0x9107, 0x9105, 0x9103, 0x9161, 0x9164, 0x915f, // 0xdc58 - 0xdc5f
		0x9162, 0x9160, 0x9201, 0x920a, 0x9225, 0x9203, 0x921a, 0x9226, // 0xdc60 - 0xdc67
		0x920f, 0x920c, 0x9200, 0x9212, 0x91ff, 0x91fd, 0x9206, 0x9204, // 0xdc68 - 0xdc6f
		0x9227, 0x9202, 0x921c, 0x9224, 0x9219, 0x9217, 0x9205, 0x9216, // 0xdc70 - 0xdc77
		0x957b, 0x958d, 0x958c, 0x9590, 0x9687, 0x967e, 0x9688, 0x0000, // 0xdc78 - 0xdc7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc80 - 0xdc87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc88 - 0xdc8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc90 - 0xdc97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdc98 - 0xdc9f
		0x0000, 0x580b, 0x580d, 0x57fd, 0x57ed, 0x5800, 0x581e, 0x5819, // 0xdca0 - 0xdca7
		0x5844, 0x5820, 0x5865, 0x586c, 0x5881, 0x5889, 0x589a, 0x5880, // 0xdca8 - 0xdcaf
		0x99a8, 0x9f19, 0x61ff, 0x8279, 0x827d, 0x827f, 0x828f, 0x828a, // 0xdcb0 - 0xdcb7
		0x82a8, 0x8284, 0x828e, 0x8291, 0x8297, 0x8299, 0x82ab, 0x82b8, // 0xdcb8 - 0xdcbf
		0x82be, 0x82b0, 0x82c8, 0x82ca, 0x82e3, 0x8298, 0x82b7, 0x82ae, // 0xdcc0 - 0xdcc7
		0x82cb, 0x82cc, 0x82c1, 0x82a9, 0x82b4, 0x82a1, 0x82aa, 0x829f, // 0xdcc8 - 0xdccf
		0x82c4, 0x82ce, 0x82a4, 0x82e1, 0x8309, 0x82f7, 0x82e4, 0x830f, // 0xdcd0 - 0xdcd7
		0x8307, 0x82dc, 0x82f4, 0x82d2, 0x82d8, 0x830c, 0x82fb, 0x82d3, // 0xdcd8 - 0xdcdf
		0x8311, 0x831a, 0x8306, 0x8314, 0x8315, 0x82e0, 0x82d5, 0x831c, // 0xdce0 - 0xdce7
		0x8351, 0x835b, 0x835c, 0x8308, 0x8392, 0x833c, 0x8334, 0x8331, // 0xdce8 - 0xdcef
		0x839b, 0x835e, 0x832f, 0x834f, 0x8347, 0x8343, 0x835f, 0x8340, // 0xdcf0 - 0xdcf7
		0x8317, 0x8360, 0x832d, 0x833a, 0x8333, 0x8366, 0x8365, 0x0000, // 0xdcf8 - 0xdcff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd00 - 0xdd07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd08 - 0xdd0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd10 - 0xdd17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd18 - 0xdd1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd20 - 0xdd27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd28 - 0xdd2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd30 - 0xdd37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd38 - 0xdd3f
		0x5ab7, 0x5ac0, 0x5aca, 0x5ab4, 0x5ab6, 0x5acd, 0x5ab9, 0x5a90, // 0xdd40 - 0xdd47
		0x5bd6, 0x5bd8, 0x5bd9, 0x5c1f, 0x5c33, 0x5d71, 0x5d63, 0x5d4a, // 0xdd48 - 0xdd4f
		0x5d65, 0x5d72, 0x5d6c, 0x5d5e, 0x5d68, 0x5d67, 0x5d62, 0x5df0, // 0xdd50 - 0xdd57
		0x5e4f, 0x5e4e, 0x5e4a, 0x5e4d, 0x5e4b, 0x5ec5, 0x5ecc, 0x5ec6, // 0xdd58 - 0xdd5f
		0x5ecb, 0x5ec7, 0x5f40, 0x5faf, 0x5fad, 0x60f7, 0x6149, 0x614a, // 0xdd60 - 0xdd67
		0x612b, 0x6145, 0x6136, 0x6132, 0x612e, 0x6146, 0x612f, 0x614f, // 0xdd68 - 0xdd6f
		0x6129, 0x6140, 0x6220, 0x9168, 0x6223, 0x6225, 0x6224, 0x63c5, // 0xdd70 - 0xdd77
		0x63f1, 0x63eb, 0x6410, 0x6412, 0x6409, 0x6420, 0x6424, 0x0000, // 0xdd78 - 0xdd7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd80 - 0xdd87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd88 - 0xdd8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd90 - 0xdd97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdd98 - 0xdd9f
		0x0000, 0x8368, 0x831b, 0x8369, 0x836c, 0x836a, 0x836d, 0x836e, // 0xdda0 - 0xdda7
		0x83b0, 0x8378, 0x83b3, 0x83b4, 0x83a0, 0x83aa, 0x8393, 0x839c, // 0xdda8 - 0xddaf
		0x8385, 0x837c, 0x83b6, 0x83a9, 0x837d, 0x83b8, 0x837b, 0x8398, // 0xddb0 - 0xddb7
		0x839e, 0x83a8, 0x83ba, 0x83bc, 0x83c1, 0x8401, 0x83e5, 0x83d8, // 0xddb8 - 0xddbf
		0x5807, 0x8418, 0x840b, 0x83dd, 0x83fd, 0x83d6, 0x841c, 0x8438, // 0xddc0 - 0xddc7
		0x8411, 0x8406, 0x83d4, 0x83df, 0x840f, 0x8403, 0x83f8, 0x83f9, // 0xddc8 - 0xddcf
		0x83ea, 0x83c5, 0x83c0, 0x8426, 0x83f0, 0x83e1, 0x845c, 0x8451, // 0xddd0 - 0xddd7
		0x845a, 0x8459, 0x8473, 0x8487, 0x8488, 0x847a, 0x8489, 0x8478, // 0xddd8 - 0xdddf
		0x843c, 0x8446, 0x8469, 0x8476, 0x848c, 0x848e, 0x8431, 0x846d, // 0xdde0 - 0xdde7
		0x84c1, 0x84cd, 0x84d0, 0x84e6, 0x84bd, 0x84d3, 0x84ca, 0x84bf, // 0xdde8 - 0xddef
		0x84ba, 0x84e0, 0x84a1, 0x84b9, 0x84b4, 0x8497, 0x84e5, 0x84e3, // 0xddf0 - 0xddf7
		0x850c, 0x750d, 0x8538, 0x84f0, 0x8539, 0x851f, 0x853a, 0x0000, // 0xddf8 - 0xddff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde00 - 0xde07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde08 - 0xde0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde10 - 0xde17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde18 - 0xde1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde20 - 0xde27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde28 - 0xde2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde30 - 0xde37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde38 - 0xde3f
		0x6bf9, 0x6bf7, 0x6bf8, 0x6e9b, 0x6ed6, 0x6ec8, 0x6e8f, 0x6ec0, // 0xde40 - 0xde47
		0x6e9f, 0x6e93, 0x6e94, 0x6ea0, 0x6eb1, 0x6eb9, 0x6ec6, 0x6ed2, // 0xde48 - 0xde4f
		0x6ebd, 0x6ec1, 0x6e9e, 0x6ec9, 0x6eb7, 0x6eb0, 0x6ecd, 0x6ea6, // 0xde50 - 0xde57
		0x6ecf, 0x6eb2, 0x6ebe, 0x6ec3, 0x6edc, 0x6ed8, 0x6e99, 0x6e92, // 0xde58 - 0xde5f
		0x6e8e, 0x6e8d, 0x6ea4, 0x6ea1, 0x6ebf, 0x6eb3, 0x6ed0, 0x6eca, // 0xde60 - 0xde67
		0x6e97, 0x6eae, 0x6ea3, 0x7147, 0x7154, 0x7152, 0x7163, 0x7160, // 0xde68 - 0xde6f
		0x7141, 0x715d, 0x7162, 0x7172, 0x7178, 0x716a, 0x7161, 0x7142, // 0xde70 - 0xde77
		0x7158, 0x7143, 0x714b, 0x7170, 0x715f, 0x7150, 0x7153, 0x0000, // 0xde78 - 0xde7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde80 - 0xde87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde88 - 0xde8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde90 - 0xde97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xde98 - 0xde9f
		0x0000, 0x8556, 0x853b, 0x84ff, 0x84fc, 0x8559, 0x8548, 0x8568, // 0xdea0 - 0xdea7
		0x8564, 0x855e, 0x857a, 0x77a2, 0x8543, 0x8572, 0x857b, 0x85a4, // 0xdea8 - 0xdeaf
		0x85a8, 0x8587, 0x858f, 0x8579, 0x85ae, 0x859c, 0x8585, 0x85b9, // 0xdeb0 - 0xdeb7
		0x85b7, 0x85b0, 0x85d3, 0x85c1, 0x85dc, 0x85ff, 0x8627, 0x8605, // 0xdeb8 - 0xdebf
		0x8629, 0x8616, 0x863c, 0x5efe, 0x5f08, 0x593c, 0x5941, 0x8037, // 0xdec0 - 0xdec7
		0x5955, 0x595a, 0x5958, 0x530f, 0x5c22, 0x5c25, 0x5c2c, 0x5c34, // 0xdec8 - 0xdecf
		0x624c, 0x626a, 0x629f, 0x62bb, 0x62ca, 0x62da, 0x62d7, 0x62ee, // 0xded0 - 0xded7
		0x6322, 0x62f6, 0x6339, 0x634b, 0x6343, 0x63ad, 0x63f6, 0x6371, // 0xded8 - 0xdedf
		0x637a, 0x638e, 0x63b4, 0x636d, 0x63ac, 0x638a, 0x6369, 0x63ae, // 0xdee0 - 0xdee7
		0x63bc, 0x63f2, 0x63f8, 0x63e0, 0x63ff, 0x63c4, 0x63de, 0x63ce, // 0xdee8 - 0xdeef
		0x6452, 0x63c6, 0x63be, 0x6445, 0x6441, 0x640b, 0x641b, 0x6420, // 0xdef0 - 0xdef7
		0x640c, 0x6426, 0x6421, 0x645e, 0x6484, 0x646d, 0x6496, 0x0000, // 0xdef8 - 0xdeff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf00 - 0xdf07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf08 - 0xdf0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf10 - 0xdf17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf18 - 0xdf1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf20 - 0xdf27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf28 - 0xdf2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf30 - 0xdf37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf38 - 0xdf3f
		0x7a1b, 0x7a10, 0x7aa3, 0x7aa2, 0x7a9e, 0x7aeb, 0x7b66, 0x7b64, // 0xdf40 - 0xdf47
		0x7b6d, 0x7b74, 0x7b69, 0x7b72, 0x7b65, 0x7b73, 0x7b71, 0x7b70, // 0xdf48 - 0xdf4f
		0x7b61, 0x7b78, 0x7b76, 0x7b63, 0x7cb2, 0x7cb4, 0x7caf, 0x7d88, // 0xdf50 - 0xdf57
		0x7d86, 0x7d80, 0x7d8d, 0x7d7f, 0x7d85, 0x7d7a, 0x7d8e, 0x7d7b, // 0xdf58 - 0xdf5f
		0x7d83, 0x7d7c, 0x7d8c, 0x7d94, 0x7d84, 0x7d7d, 0x7d92, 0x7f6d, // 0xdf60 - 0xdf67
		0x7f6b, 0x7f67, 0x7f68, 0x7f6c, 0x7fa6, 0x7fa5, 0x7fa7, 0x7fdb, // 0xdf68 - 0xdf6f
		0x7fdc, 0x8021, 0x8164, 0x8160, 0x8177, 0x815c, 0x8169, 0x815b, // 0xdf70 - 0xdf77
		0x8162, 0x8172, 0x6721, 0x815e, 0x8176, 0x8167, 0x816f, 0x0000, // 0xdf78 - 0xdf7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf80 - 0xdf87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf88 - 0xdf8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf90 - 0xdf97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xdf98 - 0xdf9f
		0x0000, 0x647a, 0x64b7, 0x64b8, 0x6499, 0x64ba, 0x64c0, 0x64d0, // 0xdfa0 - 0xdfa7
		0x64d7, 0x64e4, 0x64e2, 0x6509, 0x6525, 0x652e, 0x5f0b, 0x5fd2, // 0xdfa8 - 0xdfaf
		0x7519, 0x5f11, 0x535f, 0x53f1, 0x53fd, 0x53e9, 0x53e8, 0x53fb, // 0xdfb0 - 0xdfb7
		0x5412, 0x5416, 0x5406, 0x544b, 0x5452, 0x5453, 0x5454, 0x5456, // 0xdfb8 - 0xdfbf
		0x5443, 0x5421, 0x5457, 0x5459, 0x5423, 0x5432, 0x5482, 0x5494, // 0xdfc0 - 0xdfc7
		0x5477, 0x5471, 0x5464, 0x549a, 0x549b, 0x5484, 0x5476, 0x5466, // 0xdfc8 - 0xdfcf
		0x549d, 0x54d0, 0x54ad, 0x54c2, 0x54b4, 0x54d2, 0x54a7, 0x54a6, // 0xdfd0 - 0xdfd7
		0x54d3, 0x54d4, 0x5472, 0x54a3, 0x54d5, 0x54bb, 0x54bf, 0x54cc, // 0xdfd8 - 0xdfdf
		0x54d9, 0x54da, 0x54dc, 0x54a9, 0x54aa, 0x54a4, 0x54dd, 0x54cf, // 0xdfe0 - 0xdfe7
		0x54de, 0x551b, 0x54e7, 0x5520, 0x54fd, 0x5514, 0x54f3, 0x5522, // 0xdfe8 - 0xdfef
		0x5523, 0x550f, 0x5511, 0x5527, 0x552a, 0x5567, 0x558f, 0x55b5, // 0xdff0 - 0xdff7
		0x5549, 0x556d, 0x5541, 0x5555, 0x553f, 0x5550, 0x553c, 0x0000, // 0xdff8 - 0xdfff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe000 - 0xe007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe008 - 0xe00f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe010 - 0xe017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe018 - 0xe01f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe020 - 0xe027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe028 - 0xe02f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe030 - 0xe037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe038 - 0xe03f
		0x6f3e, 0x6f13, 0x6ef7, 0x6f86, 0x6f7a, 0x6f78, 0x6f81, 0x6f80, // 0xe040 - 0xe047
		0x6f6f, 0x6f5b, 0x6ff3, 0x6f6d, 0x6f82, 0x6f7c, 0x6f58, 0x6f8e, // 0xe048 - 0xe04f
		0x6f91, 0x6fc2, 0x6f66, 0x6fb3, 0x6fa3, 0x6fa1, 0x6fa4, 0x6fb9, // 0xe050 - 0xe057
		0x6fc6, 0x6faa, 0x6fdf, 0x6fd5, 0x6fec, 0x6fd4, 0x6fd8, 0x6ff1, // 0xe058 - 0xe05f
		0x6fee, 0x6fdb, 0x7009, 0x700b, 0x6ffa, 0x7011, 0x7001, 0x700f, // 0xe060 - 0xe067
		0x6ffe, 0x701b, 0x701a, 0x6f74, 0x701d, 0x7018, 0x701f, 0x7030, // 0xe068 - 0xe06f
		0x703e, 0x7032, 0x7051, 0x7063, 0x7099, 0x7092, 0x70af, 0x70f1, // 0xe070 - 0xe077
		0x70ac, 0x70b8, 0x70b3, 0x70ae, 0x70df, 0x70cb, 0x70dd, 0x0000, // 0xe078 - 0xe07f
		0x70d9, 0x7109, 0x70fd, 0x711c, 0x7119, 0x7165, 0x7155, 0x7188, // 0xe080 - 0xe087
		0x7166, 0x7162, 0x714c, 0x7156, 0x716c, 0x718f, 0x71fb, 0x7184, // 0xe088 - 0xe08f
		0x7195, 0x71a8, 0x71ac, 0x71d7, 0x71b9, 0x71be, 0x71d2, 0x71c9, // 0xe090 - 0xe097
		0x71d4, 0x71ce, 0x71e0, 0x71ec, 0x71e7, 0x71f5, 0x71fc, 0x71f9, // 0xe098 - 0xe09f
		0x71ff, 0x720d, 0x7210, 0x721b, 0x7228, 0x722d, 0x722c, 0x7230, // 0xe0a0 - 0xe0a7
		0x7232, 0x723b, 0x723c, 0x723f, 0x7240, 0x7246, 0x724b, 0x7258, // 0xe0a8 - 0xe0af
		0x7274, 0x727e, 0x7282, 0x7281, 0x7287, 0x7292, 0x7296, 0x72a2, // 0xe0b0 - 0xe0b7
		0x72a7, 0x72b9, 0x72b2, 0x72c3, 0x72c6, 0x72c4, 0x72ce, 0x72d2, // 0xe0b8 - 0xe0bf
		0x72e2, 0x72e0, 0x72e1, 0x72f9, 0x72f7, 0x500f, 0x7317, 0x730a, // 0xe0c0 - 0xe0c7
		0x731c, 0x7316, 0x731d, 0x7334, 0x732f, 0x7329, 0x7325, 0x733e, // 0xe0c8 - 0xe0cf
		0x734e, 0x734f, 0x9ed8, 0x7357, 0x736a, 0x7368, 0x7370, 0x7378, // 0xe0d0 - 0xe0d7
		0x7375, 0x737b, 0x737a, 0x73c8, 0x73b3, 0x73ce, 0x73bb, 0x73c0, // 0xe0d8 - 0xe0df
		0x73e5, 0x73ee, 0x73de, 0x74a2, 0x7405, 0x746f, 0x7425, 0x73f8, // 0xe0e0 - 0xe0e7
		0x7432, 0x743a, 0x7455, 0x743f, 0x745f, 0x7459, 0x7441, 0x745c, // 0xe0e8 - 0xe0ef
		0x7469, 0x7470, 0x7463, 0x746a, 0x7476, 0x747e, 0x748b, 0x749e, // 0xe0f0 - 0xe0f7
		0x74a7, 0x74ca, 0x74cf, 0x74d4, 0x73f1, 0x5e3b, 0x5e3c, 0x0000, // 0xe0f8 - 0xe0ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe100 - 0xe107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe108 - 0xe10f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe110 - 0xe117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe118 - 0xe11f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe120 - 0xe127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe128 - 0xe12f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe130 - 0xe137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe138 - 0xe13f
		0x74e0, 0x74e3, 0x74e7, 0x74e9, 0x74ee, 0x74f2, 0x74f0, 0x74f1, // 0xe140 - 0xe147
		0x74f8, 0x74f7, 0x7504, 0x7503, 0x7505, 0x750c, 0x750e, 0x750d, // 0xe148 - 0xe14f
		0x7515, 0x7513, 0x751e, 0x7526, 0x752c, 0x753c, 0x7544, 0x754d, // 0xe150 - 0xe157
		0x754a, 0x7549, 0x755b, 0x7546, 0x755a, 0x7569, 0x7564, 0x7567, // 0xe158 - 0xe15f
		0x756b, 0x756d, 0x7578, 0x7576, 0x7586, 0x7587, 0x7574, 0x758a, // 0xe160 - 0xe167
		0x7589, 0x7582, 0x7594, 0x759a, 0x759d, 0x75a5, 0x75a3, 0x75c2, // 0xe168 - 0xe16f
		0x75b3, 0x75c3, 0x75b5, 0x75bd, 0x75b8, 0x75bc, 0x75b1, 0x75cd, // 0xe170 - 0xe177
		0x75ca, 0x75d2, 0x75d9, 0x75e3, 0x75de, 0x75fe, 0x75ff, 0x0000, // 0xe178 - 0xe17f
		0x75fc, 0x7601, 0x75f0, 0x75fa, 0x75f2, 0x75f3, 0x760b, 0x760d, // 0xe180 - 0xe187
		0x7609, 0x761f, 0x7627, 0x7620, 0x7621, 0x7622, 0x7624, 0x7634, // 0xe188 - 0xe18f
		0x7630, 0x763b, 0x7647, 0x7648, 0x7646, 0x765c, 0x7658, 0x7661, // 0xe190 - 0xe197
		0x7662, 0x7668, 0x7669, 0x766a, 0x7667, 0x766c, 0x7670, 0x7672, // 0xe198 - 0xe19f
		0x7676, 0x7678, 0x767c, 0x7680, 0x7683, 0x7688, 0x768b, 0x768e, // 0xe1a0 - 0xe1a7
		0x7696, 0x7693, 0x7699, 0x769a, 0x76b0, 0x76b4, 0x76b8, 0x76b9, // 0xe1a8 - 0xe1af
		0x76ba, 0x76c2, 0x76cd, 0x76d6, 0x76d2, 0x76de, 0x76e1, 0x76e5, // 0xe1b0 - 0xe1b7
		0x76e7, 0x76ea, 0x862f, 0x76fb, 0x7708, 0x7707, 0x7704, 0x7729, // 0xe1b8 - 0xe1bf
		0x7724, 0x771e, 0x7725, 0x7726, 0x771b, 0x7737, 0x7738, 0x7747, // 0xe1c0 - 0xe1c7
		0x775a, 0x7768, 0x776b, 0x775b, 0x7765, 0x777f, 0x777e, 0x7779, // 0xe1c8 - 0xe1cf
		0x778e, 0x778b, 0x7791, 0x77a0, 0x779e, 0x77b0, 0x77b6, 0x77b9, // 0xe1d0 - 0xe1d7
		0x77bf, 0x77bc, 0x77bd, 0x77bb, 0x77c7, 0x77cd, 0x77d7, 0x77da, // 0xe1d8 - 0xe1df
		0x77dc, 0x77e3, 0x77ee, 0x77fc, 0x780c, 0x7812, 0x7926, 0x7820, // 0xe1e0 - 0xe1e7
		0x792a, 0x7845, 0x788e, 0x7874, 0x7886, 0x787c, 0x789a, 0x788c, // 0xe1e8 - 0xe1ef
		0x78a3, 0x78b5, 0x78aa, 0x78af, 0x78d1, 0x78c6, 0x78cb, 0x78d4, // 0xe1f0 - 0xe1f7
		0x78be, 0x78bc, 0x78c5, 0x78ca, 0x78ec, 0x7303, 0x72fa, 0x0000, // 0xe1f8 - 0xe1ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe200 - 0xe207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe208 - 0xe20f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe210 - 0xe217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe218 - 0xe21f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe220 - 0xe227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe228 - 0xe22f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe230 - 0xe237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe238 - 0xe23f
		0x78e7, 0x78da, 0x78fd, 0x78f4, 0x7907, 0x7912, 0x7911, 0x7919, // 0xe240 - 0xe247
		0x792c, 0x792b, 0x7940, 0x7960, 0x7957, 0x795f, 0x795a, 0x7955, // 0xe248 - 0xe24f
		0x7953, 0x797a, 0x797f, 0x798a, 0x799d, 0x79a7, 0x9f4b, 0x79aa, // 0xe250 - 0xe257
		0x79ae, 0x79b3, 0x79b9, 0x79ba, 0x79c9, 0x79d5, 0x79e7, 0x79ec, // 0xe258 - 0xe25f
		0x79e1, 0x79e3, 0x7a08, 0x7a0d, 0x7a18, 0x7a19, 0x7a20, 0x7a1f, // 0xe260 - 0xe267
		0x7980, 0x7a31, 0x7a3b, 0x7a3e, 0x7a37, 0x7a43, 0x7a57, 0x7a49, // 0xe268 - 0xe26f
		0x7a61, 0x7a62, 0x7a69, 0x9f9d, 0x7a70, 0x7a79, 0x7a7d, 0x7a88, // 0xe270 - 0xe277
		0x7a97, 0x7a95, 0x7a98, 0x7a96, 0x7aa9, 0x7ac8, 0x7ab0, 0x0000, // 0xe278 - 0xe27f
		0x7ab6, 0x7ac5, 0x7ac4, 0x7abf, 0x9083, 0x7ac7, 0x7aca, 0x7acd, // 0xe280 - 0xe287
		0x7acf, 0x7ad5, 0x7ad3, 0x7ad9, 0x7ada, 0x7add, 0x7ae1, 0x7ae2, // 0xe288 - 0xe28f
		0x7ae6, 0x7aed, 0x7af0, 0x7b02, 0x7b0f, 0x7b0a, 0x7b06, 0x7b33, // 0xe290 - 0xe297
		0x7b18, 0x7b19, 0x7b1e, 0x7b35, 0x7b28, 0x7b36, 0x7b50, 0x7b7a, // 0xe298 - 0xe29f
		0x7b04, 0x7b4d, 0x7b0b, 0x7b4c, 0x7b45, 0x7b75, 0x7b65, 0x7b74, // 0xe2a0 - 0xe2a7
		0x7b67, 0x7b70, 0x7b71, 0x7b6c, 0x7b6e, 0x7b9d, 0x7b98, 0x7b9f, // 0xe2a8 - 0xe2af
		0x7b8d, 0x7b9c, 0x7b9a, 0x7b8b, 0x7b92, 0x7b8f, 0x7b5d, 0x7b99, // 0xe2b0 - 0xe2b7
		0x7bcb, 0x7bc1, 0x7bcc, 0x7bcf, 0x7bb4, 0x7bc6, 0x7bdd, 0x7be9, // 0xe2b8 - 0xe2bf
		0x7c11, 0x7c14, 0x7be6, 0x7be5, 0x7c60, 0x7c00, 0x7c07, 0x7c13, // 0xe2c0 - 0xe2c7
		0x7bf3, 0x7bf7, 0x7c17, 0x7c0d, 0x7bf6, 0x7c23, 0x7c27, 0x7c2a, // 0xe2c8 - 0xe2cf
		0x7c1f, 0x7c37, 0x7c2b, 0x7c3d, 0x7c4c, 0x7c43, 0x7c54, 0x7c4f, // 0xe2d0 - 0xe2d7
		0x7c40, 0x7c50, 0x7c58, 0x7c5f, 0x7c64, 0x7c56, 0x7c65, 0x7c6c, // 0xe2d8 - 0xe2df
		0x7c75, 0x7c83, 0x7c90, 0x7ca4, 0x7cad, 0x7ca2, 0x7cab, 0x7ca1, // 0xe2e0 - 0xe2e7
		0x7ca8, 0x7cb3, 0x7cb2, 0x7cb1, 0x7cae, 0x7cb9, 0x7cbd, 0x7cc0, // 0xe2e8 - 0xe2ef
		0x7cc5, 0x7cc2, 0x7cd8, 0x7cd2, 0x7cdc, 0x7ce2, 0x9b3b, 0x7cef, // 0xe2f0 - 0xe2f7
		0x7cf2, 0x7cf4, 0x7cf6, 0x7cfa, 0x7d06, 0x607a, 0x6042, 0x0000, // 0xe2f8 - 0xe2ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe300 - 0xe307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe308 - 0xe30f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe310 - 0xe317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe318 - 0xe31f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe320 - 0xe327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe328 - 0xe32f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe330 - 0xe337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe338 - 0xe33f
		0x7d02, 0x7d1c, 0x7d15, 0x7d0a, 0x7d45, 0x7d4b, 0x7d2e, 0x7d32, // 0xe340 - 0xe347
		0x7d3f, 0x7d35, 0x7d46, 0x7d73, 0x7d56, 0x7d4e, 0x7d72, 0x7d68, // 0xe348 - 0xe34f
		0x7d6e, 0x7d4f, 0x7d63, 0x7d93, 0x7d89, 0x7d5b, 0x7d8f, 0x7d7d, // 0xe350 - 0xe357
		0x7d9b, 0x7dba, 0x7dae, 0x7da3, 0x7db5, 0x7dc7, 0x7dbd, 0x7dab, // 0xe358 - 0xe35f
		0x7e3d, 0x7da2, 0x7daf, 0x7ddc, 0x7db8, 0x7d9f, 0x7db0, 0x7dd8, // 0xe360 - 0xe367
		0x7ddd, 0x7de4, 0x7dde, 0x7dfb, 0x7df2, 0x7de1, 0x7e05, 0x7e0a, // 0xe368 - 0xe36f
		0x7e23, 0x7e21, 0x7e12, 0x7e31, 0x7e1f, 0x7e09, 0x7e0b, 0x7e22, // 0xe370 - 0xe377
		0x7e46, 0x7e66, 0x7e3b, 0x7e35, 0x7e39, 0x7e43, 0x7e37, 0x0000, // 0xe378 - 0xe37f
		0x7e32, 0x7e3a, 0x7e67, 0x7e5d, 0x7e56, 0x7e5e, 0x7e59, 0x7e5a, // 0xe380 - 0xe387
		0x7e79, 0x7e6a, 0x7e69, 0x7e7c, 0x7e7b, 0x7e83, 0x7dd5, 0x7e7d, // 0xe388 - 0xe38f
		0x8fae, 0x7e7f, 0x7e88, 0x7e89, 0x7e8c, 0x7e92, 0x7e90, 0x7e93, // 0xe390 - 0xe397
		0x7e94, 0x7e96, 0x7e8e, 0x7e9b, 0x7e9c, 0x7f38, 0x7f3a, 0x7f45, // 0xe398 - 0xe39f
		0x7f4c, 0x7f4d, 0x7f4e, 0x7f50, 0x7f51, 0x7f55, 0x7f54, 0x7f58, // 0xe3a0 - 0xe3a7
		0x7f5f, 0x7f60, 0x7f68, 0x7f69, 0x7f67, 0x7f78, 0x7f82, 0x7f86, // 0xe3a8 - 0xe3af
		0x7f83, 0x7f88, 0x7f87, 0x7f8c, 0x7f94, 0x7f9e, 0x7f9d, 0x7f9a, // 0xe3b0 - 0xe3b7
		0x7fa3, 0x7faf, 0x7fb2, 0x7fb9, 0x7fae, 0x7fb6, 0x7fb8, 0x8b71, // 0xe3b8 - 0xe3bf
		0x7fc5, 0x7fc6, 0x7fca, 0x7fd5, 0x7fd4, 0x7fe1, 0x7fe6, 0x7fe9, // 0xe3c0 - 0xe3c7
		0x7ff3, 0x7ff9, 0x98dc, 0x8006, 0x8004, 0x800b, 0x8012, 0x8018, // 0xe3c8 - 0xe3cf
		0x8019, 0x801c, 0x8021, 0x8028, 0x803f, 0x803b, 0x804a, 0x8046, // 0xe3d0 - 0xe3d7
		0x8052, 0x8058, 0x805a, 0x805f, 0x8062, 0x8068, 0x8073, 0x8072, // 0xe3d8 - 0xe3df
		0x8070, 0x8076, 0x8079, 0x807d, 0x807f, 0x8084, 0x8086, 0x8085, // 0xe3e0 - 0xe3e7
		0x809b, 0x8093, 0x809a, 0x80ad, 0x5190, 0x80ac, 0x80db, 0x80e5, // 0xe3e8 - 0xe3ef
		0x80d9, 0x80dd, 0x80c4, 0x80da, 0x80d6, 0x8109, 0x80ef, 0x80f1, // 0xe3f0 - 0xe3f7
		0x811b, 0x8129, 0x8123, 0x812f, 0x814b, 0x6cef, 0x6cfe, 0x0000, // 0xe3f8 - 0xe3ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe400 - 0xe407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe408 - 0xe40f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe410 - 0xe417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe418 - 0xe41f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe420 - 0xe427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe428 - 0xe42f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe430 - 0xe437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe438 - 0xe43f
		0x968b, 0x8146, 0x813e, 0x8153, 0x8151, 0x80fc, 0x8171, 0x816e, // 0xe440 - 0xe447
		0x8165, 0x8166, 0x8174, 0x8183, 0x8188, 0x818a, 0x8180, 0x8182, // 0xe448 - 0xe44f
		0x81a0, 0x8195, 0x81a4, 0x81a3, 0x815f, 0x8193, 0x81a9, 0x81b0, // 0xe450 - 0xe457
		0x81b5, 0x81be, 0x81b8, 0x81bd, 0x81c0, 0x81c2, 0x81ba, 0x81c9, // 0xe458 - 0xe45f
		0x81cd, 0x81d1, 0x81d9, 0x81d8, 0x81c8, 0x81da, 0x81df, 0x81e0, // 0xe460 - 0xe467
		0x81e7, 0x81fa, 0x81fb, 0x81fe, 0x8201, 0x8202, 0x8205, 0x8207, // 0xe468 - 0xe46f
		0x820a, 0x820d, 0x8210, 0x8216, 0x8229, 0x822b, 0x8238, 0x8233, // 0xe470 - 0xe477
		0x8240, 0x8259, 0x8258, 0x825d, 0x825a, 0x825f, 0x8264, 0x0000, // 0xe478 - 0xe47f
		0x8262, 0x8268, 0x826a, 0x826b, 0x822e, 0x8271, 0x8277, 0x8278, // 0xe480 - 0xe487
		0x827e, 0x828d, 0x8292, 0x82ab, 0x829f, 0x82bb, 0x82ac, 0x82e1, // 0xe488 - 0xe48f
		0x82e3, 0x82df, 0x82d2, 0x82f4, 0x82f3, 0x82fa, 0x8393, 0x8303, // 0xe490 - 0xe497
		0x82fb, 0x82f9, 0x82de, 0x8306, 0x82dc, 0x8309, 0x82d9, 0x8335, // 0xe498 - 0xe49f
		0x8334, 0x8316, 0x8332, 0x8331, 0x8340, 0x8339, 0x8350, 0x8345, // 0xe4a0 - 0xe4a7
		0x832f, 0x832b, 0x8317, 0x8318, 0x8385, 0x839a, 0x83aa, 0x839f, // 0xe4a8 - 0xe4af
		0x83a2, 0x8396, 0x8323, 0x838e, 0x8387, 0x838a, 0x837c, 0x83b5, // 0xe4b0 - 0xe4b7
		0x8373, 0x8375, 0x83a0, 0x8389, 0x83a8, 0x83f4, 0x8413, 0x83eb, // 0xe4b8 - 0xe4bf
		0x83ce, 0x83fd, 0x8403, 0x83d8, 0x840b, 0x83c1, 0x83f7, 0x8407, // 0xe4c0 - 0xe4c7
		0x83e0, 0x83f2, 0x840d, 0x8422, 0x8420, 0x83bd, 0x8438, 0x8506, // 0xe4c8 - 0xe4cf
		0x83fb, 0x846d, 0x842a, 0x843c, 0x855a, 0x8484, 0x8477, 0x846b, // 0xe4d0 - 0xe4d7
		0x84ad, 0x846e, 0x8482, 0x8469, 0x8446, 0x842c, 0x846f, 0x8479, // 0xe4d8 - 0xe4df
		0x8435, 0x84ca, 0x8462, 0x84b9, 0x84bf, 0x849f, 0x84d9, 0x84cd, // 0xe4e0 - 0xe4e7
		0x84bb, 0x84da, 0x84d0, 0x84c1, 0x84c6, 0x84d6, 0x84a1, 0x8521, // 0xe4e8 - 0xe4ef
		0x84ff, 0x84f4, 0x8517, 0x8518, 0x852c, 0x851f, 0x8515, 0x8514, // 0xe4f0 - 0xe4f7
		0x84fc, 0x8540, 0x8563, 0x8558, 0x8548, 0x6f7a, 0x6fd1, 0x0000, // 0xe4f8 - 0xe4ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe500 - 0xe507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe508 - 0xe50f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe510 - 0xe517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe518 - 0xe51f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe520 - 0xe527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe528 - 0xe52f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe530 - 0xe537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe538 - 0xe53f
		0x8541, 0x8602, 0x854b, 0x8555, 0x8580, 0x85a4, 0x8588, 0x8591, // 0xe540 - 0xe547
		0x858a, 0x85a8, 0x856d, 0x8594, 0x859b, 0x85ea, 0x8587, 0x859c, // 0xe548 - 0xe54f
		0x8577, 0x857e, 0x8590, 0x85c9, 0x85ba, 0x85cf, 0x85b9, 0x85d0, // 0xe550 - 0xe557
		0x85d5, 0x85dd, 0x85e5, 0x85dc, 0x85f9, 0x860a, 0x8613, 0x860b, // 0xe558 - 0xe55f
		0x85fe, 0x85fa, 0x8606, 0x8622, 0x861a, 0x8630, 0x863f, 0x864d, // 0xe560 - 0xe567
		0x4e55, 0x8654, 0x865f, 0x8667, 0x8671, 0x8693, 0x86a3, 0x86a9, // 0xe568 - 0xe56f
		0x86aa, 0x868b, 0x868c, 0x86b6, 0x86af, 0x86c4, 0x86c6, 0x86b0, // 0xe570 - 0xe577
		0x86c9, 0x8823, 0x86ab, 0x86d4, 0x86de, 0x86e9, 0x86ec, 0x0000, // 0xe578 - 0xe57f
		0x86df, 0x86db, 0x86ef, 0x8712, 0x8706, 0x8708, 0x8700, 0x8703, // 0xe580 - 0xe587
		0x86fb, 0x8711, 0x8709, 0x870d, 0x86f9, 0x870a, 0x8734, 0x873f, // 0xe588 - 0xe58f
		0x8737, 0x873b, 0x8725, 0x8729, 0x871a, 0x8760, 0x875f, 0x8778, // 0xe590 - 0xe597
		0x874c, 0x874e, 0x8774, 0x8757, 0x8768, 0x876e, 0x8759, 0x8753, // 0xe598 - 0xe59f
		0x8763, 0x876a, 0x8805, 0x87a2, 0x879f, 0x8782, 0x87af, 0x87cb, // 0xe5a0 - 0xe5a7
		0x87bd, 0x87c0, 0x87d0, 0x96d6, 0x87ab, 0x87c4, 0x87b3, 0x87c7, // 0xe5a8 - 0xe5af
		0x87c6, 0x87bb, 0x87ef, 0x87f2, 0x87e0, 0x880f, 0x880d, 0x87fe, // 0xe5b0 - 0xe5b7
		0x87f6, 0x87f7, 0x880e, 0x87d2, 0x8811, 0x8816, 0x8815, 0x8822, // 0xe5b8 - 0xe5bf
		0x8821, 0x8831, 0x8836, 0x8839, 0x8827, 0x883b, 0x8844, 0x8842, // 0xe5c0 - 0xe5c7
		0x8852, 0x8859, 0x885e, 0x8862, 0x886b, 0x8881, 0x887e, 0x889e, // 0xe5c8 - 0xe5cf
		0x8875, 0x887d, 0x88b5, 0x8872, 0x8882, 0x8897, 0x8892, 0x88ae, // 0xe5d0 - 0xe5d7
		0x8899, 0x88a2, 0x888d, 0x88a4, 0x88b0, 0x88bf, 0x88b1, 0x88c3, // 0xe5d8 - 0xe5df
		0x88c4, 0x88d4, 0x88d8, 0x88d9, 0x88dd, 0x88f9, 0x8902, 0x88fc, // 0xe5e0 - 0xe5e7
		0x88f4, 0x88e8, 0x88f2, 0x8904, 0x890c, 0x890a, 0x8913, 0x8943, // 0xe5e8 - 0xe5ef
		0x891e, 0x8925, 0x892a, 0x892b, 0x8941, 0x8944, 0x893b, 0x8936, // 0xe5f0 - 0xe5f7
		0x8938, 0x894c, 0x891d, 0x8960, 0x895e, 0x59aa, 0x59a3, 0x0000, // 0xe5f8 - 0xe5ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe600 - 0xe607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe608 - 0xe60f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe610 - 0xe617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe618 - 0xe61f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe620 - 0xe627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe628 - 0xe62f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe630 - 0xe637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe638 - 0xe63f
		0x8966, 0x8964, 0x896d, 0x896a, 0x896f, 0x8974, 0x8977, 0x897e, // 0xe640 - 0xe647
		0x8983, 0x8988, 0x898a, 0x8993, 0x8998, 0x89a1, 0x89a9, 0x89a6, // 0xe648 - 0xe64f
		0x89ac, 0x89af, 0x89b2, 0x89ba, 0x89bd, 0x89bf, 0x89c0, 0x89da, // 0xe650 - 0xe657
		0x89dc, 0x89dd, 0x89e7, 0x89f4, 0x89f8, 0x8a03, 0x8a16, 0x8a10, // 0xe658 - 0xe65f
		0x8a0c, 0x8a1b, 0x8a1d, 0x8a25, 0x8a36, 0x8a41, 0x8a5b, 0x8a52, // 0xe660 - 0xe667
		0x8a46, 0x8a48, 0x8a7c, 0x8a6d, 0x8a6c, 0x8a62, 0x8a85, 0x8a82, // 0xe668 - 0xe66f
		0x8a84, 0x8aa8, 0x8aa1, 0x8a91, 0x8aa5, 0x8aa6, 0x8a9a, 0x8aa3, // 0xe670 - 0xe677
		0x8ac4, 0x8acd, 0x8ac2, 0x8ada, 0x8aeb, 0x8af3, 0x8ae7, 0x0000, // 0xe678 - 0xe67f
		0x8ae4, 0x8af1, 0x8b14, 0x8ae0, 0x8ae2, 0x8af7, 0x8ade, 0x8adb, // 0xe680 - 0xe687
		0x8b0c, 0x8b07, 0x8b1a, 0x8ae1, 0x8b16, 0x8b10, 0x8b17, 0x8b20, // 0xe688 - 0xe68f
		0x8b33, 0x97ab, 0x8b26, 0x8b2b, 0x8b3e, 0x8b28, 0x8b41, 0x8b4c, // 0xe690 - 0xe697
		0x8b4f, 0x8b4e, 0x8b49, 0x8b56, 0x8b5b, 0x8b5a, 0x8b6b, 0x8b5f, // 0xe698 - 0xe69f
		0x8b6c, 0x8b6f, 0x8b74, 0x8b7d, 0x8b80, 0x8b8c, 0x8b8e, 0x8b92, // 0xe6a0 - 0xe6a7
		0x8b93, 0x8b96, 0x8b99, 0x8b9a, 0x8c3a, 0x8c41, 0x8c3f, 0x8c48, // 0xe6a8 - 0xe6af
		0x8c4c, 0x8c4e, 0x8c50, 0x8c55, 0x8c62, 0x8c6c, 0x8c78, 0x8c7a, // 0xe6b0 - 0xe6b7
		0x8c82, 0x8c89, 0x8c85, 0x8c8a, 0x8c8d, 0x8c8e, 0x8c94, 0x8c7c, // 0xe6b8 - 0xe6bf
		0x8c98, 0x621d, 0x8cad, 0x8caa, 0x8cbd, 0x8cb2, 0x8cb3, 0x8cae, // 0xe6c0 - 0xe6c7
		0x8cb6, 0x8cc8, 0x8cc1, 0x8ce4, 0x8ce3, 0x8cda, 0x8cfd, 0x8cfa, // 0xe6c8 - 0xe6cf
		0x8cfb, 0x8d04, 0x8d05, 0x8d0a, 0x8d07, 0x8d0f, 0x8d0d, 0x8d10, // 0xe6d0 - 0xe6d7
		0x9f4e, 0x8d13, 0x8ccd, 0x8d14, 0x8d16, 0x8d67, 0x8d6d, 0x8d71, // 0xe6d8 - 0xe6df
		0x8d73, 0x8d81, 0x8d99, 0x8dc2, 0x8dbe, 0x8dba, 0x8dcf, 0x8dda, // 0xe6e0 - 0xe6e7
		0x8dd6, 0x8dcc, 0x8ddb, 0x8dcb, 0x8dea, 0x8deb, 0x8ddf, 0x8de3, // 0xe6e8 - 0xe6ef
		0x8dfc, 0x8e08, 0x8e09, 0x8dff, 0x8e1d, 0x8e1e, 0x8e10, 0x8e1f, // 0xe6f0 - 0xe6f7
		0x8e42, 0x8e35, 0x8e30, 0x8e34, 0x8e4a, 0x7ea8, 0x7ea9, 0x0000, // 0xe6f8 - 0xe6ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe700 - 0xe707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe708 - 0xe70f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe710 - 0xe717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe718 - 0xe71f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe720 - 0xe727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe728 - 0xe72f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe730 - 0xe737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe738 - 0xe73f
		0x8e47, 0x8e49, 0x8e4c, 0x8e50, 0x8e48, 0x8e59, 0x8e64, 0x8e60, // 0xe740 - 0xe747
		0x8e2a, 0x8e63, 0x8e55, 0x8e76, 0x8e72, 0x8e7c, 0x8e81, 0x8e87, // 0xe748 - 0xe74f
		0x8e85, 0x8e84, 0x8e8b, 0x8e8a, 0x8e93, 0x8e91, 0x8e94, 0x8e99, // 0xe750 - 0xe757
		0x8eaa, 0x8ea1, 0x8eac, 0x8eb0, 0x8ec6, 0x8eb1, 0x8ebe, 0x8ec5, // 0xe758 - 0xe75f
		0x8ec8, 0x8ecb, 0x8edb, 0x8ee3, 0x8efc, 0x8efb, 0x8eeb, 0x8efe, // 0xe760 - 0xe767
		0x8f0a, 0x8f05, 0x8f15, 0x8f12, 0x8f19, 0x8f13, 0x8f1c, 0x8f1f, // 0xe768 - 0xe76f
		0x8f1b, 0x8f0c, 0x8f26, 0x8f33, 0x8f3b, 0x8f39, 0x8f45, 0x8f42, // 0xe770 - 0xe777
		0x8f3e, 0x8f4c, 0x8f49, 0x8f46, 0x8f4e, 0x8f57, 0x8f5c, 0x0000, // 0xe778 - 0xe77f
		0x8f62, 0x8f63, 0x8f64, 0x8f9c, 0x8f9f, 0x8fa3, 0x8fad, 0x8faf, // 0xe780 - 0xe787
		0x8fb7, 0x8fda, 0x8fe5, 0x8fe2, 0x8fea, 0x8fef, 0x9087, 0x8ff4, // 0xe788 - 0xe78f
		0x9005, 0x8ff9, 0x8ffa, 0x9011, 0x9015, 0x9021, 0x900d, 0x901e, // 0xe790 - 0xe797
		0x9016, 0x900b, 0x9027, 0x9036, 0x9035, 0x9039, 0x8ff8, 0x904f, // 0xe798 - 0xe79f
		0x9050, 0x9051, 0x9052, 0x900e, 0x9049, 0x903e, 0x9056, 0x9058, // 0xe7a0 - 0xe7a7
		0x905e, 0x9068, 0x906f, 0x9076, 0x96a8, 0x9072, 0x9082, 0x907d, // 0xe7a8 - 0xe7af
		0x9081, 0x9080, 0x908a, 0x9089, 0x908f, 0x90a8, 0x90af, 0x90b1, // 0xe7b0 - 0xe7b7
		0x90b5, 0x90e2, 0x90e4, 0x6248, 0x90db, 0x9102, 0x9112, 0x9119, // 0xe7b8 - 0xe7bf
		0x9132, 0x9130, 0x914a, 0x9156, 0x9158, 0x9163, 0x9165, 0x9169, // 0xe7c0 - 0xe7c7
		0x9173, 0x9172, 0x918b, 0x9189, 0x9182, 0x91a2, 0x91ab, 0x91af, // 0xe7c8 - 0xe7cf
		0x91aa, 0x91b5, 0x91b4, 0x91ba, 0x91c0, 0x91c1, 0x91c9, 0x91cb, // 0xe7d0 - 0xe7d7
		0x91d0, 0x91d6, 0x91df, 0x91e1, 0x91db, 0x91fc, 0x91f5, 0x91f6, // 0xe7d8 - 0xe7df
		0x921e, 0x91ff, 0x9214, 0x922c, 0x9215, 0x9211, 0x925e, 0x9257, // 0xe7e0 - 0xe7e7
		0x9245, 0x9249, 0x9264, 0x9248, 0x9295, 0x923f, 0x924b, 0x9250, // 0xe7e8 - 0xe7ef
		0x929c, 0x9296, 0x9293, 0x929b, 0x925a, 0x92cf, 0x92b9, 0x92b7, // 0xe7f0 - 0xe7f7
		0x92e9, 0x930f, 0x92fa, 0x9344, 0x932e, 0x742e, 0x742c, 0x0000, // 0xe7f8 - 0xe7ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe800 - 0xe807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe808 - 0xe80f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe810 - 0xe817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe818 - 0xe81f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe820 - 0xe827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe828 - 0xe82f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe830 - 0xe837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe838 - 0xe83f
		0x9319, 0x9322, 0x931a, 0x9323, 0x933a, 0x9335, 0x933b, 0x935c, // 0xe840 - 0xe847
		0x9360, 0x937c, 0x936e, 0x9356, 0x93b0, 0x93ac, 0x93ad, 0x9394, // 0xe848 - 0xe84f
		0x93b9, 0x93d6, 0x93d7, 0x93e8, 0x93e5, 0x93d8, 0x93c3, 0x93dd, // 0xe850 - 0xe857
		0x93d0, 0x93c8, 0x93e4, 0x941a, 0x9414, 0x9413, 0x9403, 0x9407, // 0xe858 - 0xe85f
		0x9410, 0x9436, 0x942b, 0x9435, 0x9421, 0x943a, 0x9441, 0x9452, // 0xe860 - 0xe867
		0x9444, 0x945b, 0x9460, 0x9462, 0x945e, 0x946a, 0x9229, 0x9470, // 0xe868 - 0xe86f
		0x9475, 0x9477, 0x947d, 0x945a, 0x947c, 0x947e, 0x9481, 0x947f, // 0xe870 - 0xe877
		0x9582, 0x9587, 0x958a, 0x9594, 0x9596, 0x9598, 0x9599, 0x0000, // 0xe878 - 0xe87f
		0x95a0, 0x95a8, 0x95a7, 0x95ad, 0x95bc, 0x95bb, 0x95b9, 0x95be, // 0xe880 - 0xe887
		0x95ca, 0x6ff6, 0x95c3, 0x95cd, 0x95cc, 0x95d5, 0x95d4, 0x95d6, // 0xe888 - 0xe88f
		0x95dc, 0x95e1, 0x95e5, 0x95e2, 0x9621, 0x9628, 0x962e, 0x962f, // 0xe890 - 0xe897
		0x9642, 0x964c, 0x964f, 0x964b, 0x9677, 0x965c, 0x965e, 0x965d, // 0xe898 - 0xe89f
		0x965f, 0x9666, 0x9672, 0x966c, 0x968d, 0x9698, 0x9695, 0x9697, // 0xe8a0 - 0xe8a7
		0x96aa, 0x96a7, 0x96b1, 0x96b2, 0x96b0, 0x96b4, 0x96b6, 0x96b8, // 0xe8a8 - 0xe8af
		0x96b9, 0x96ce, 0x96cb, 0x96c9, 0x96cd, 0x894d, 0x96dc, 0x970d, // 0xe8b0 - 0xe8b7
		0x96d5, 0x96f9, 0x9704, 0x9706, 0x9708, 0x9713, 0x970e, 0x9711, // 0xe8b8 - 0xe8bf
		0x970f, 0x9716, 0x9719, 0x9724, 0x972a, 0x9730, 0x9739, 0x973d, // 0xe8c0 - 0xe8c7
		0x973e, 0x9744, 0x9746, 0x9748, 0x9742, 0x9749, 0x975c, 0x9760, // 0xe8c8 - 0xe8cf
		0x9764, 0x9766, 0x9768, 0x52d2, 0x976b, 0x9771, 0x9779, 0x9785, // 0xe8d0 - 0xe8d7
		0x977c, 0x9781, 0x977a, 0x9786, 0x978b, 0x978f, 0x9790, 0x979c, // 0xe8d8 - 0xe8df
		0x97a8, 0x97a6, 0x97a3, 0x97b3, 0x97b4, 0x97c3, 0x97c6, 0x97c8, // 0xe8e0 - 0xe8e7
		0x97cb, 0x97dc, 0x97ed, 0x9f4f, 0x97f2, 0x7adf, 0x97f6, 0x97f5, // 0xe8e8 - 0xe8ef
		0x980f, 0x980c, 0x9838, 0x9824, 0x9821, 0x9837, 0x983d, 0x9846, // 0xe8f0 - 0xe8f7
		0x984f, 0x984b, 0x986b, 0x986f, 0x9870, 0x6920, 0x68f9, 0x0000, // 0xe8f8 - 0xe8ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe900 - 0xe907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe908 - 0xe90f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe910 - 0xe917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe918 - 0xe91f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe920 - 0xe927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe928 - 0xe92f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe930 - 0xe937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xe938 - 0xe93f
		0x9871, 0x9874, 0x9873, 0x98aa, 0x98af, 0x98b1, 0x98b6, 0x98c4, // 0xe940 - 0xe947
		0x98c3, 0x98c6, 0x98e9, 0x98eb, 0x9903, 0x9909, 0x9912, 0x9914, // 0xe948 - 0xe94f
		0x9918, 0x9921, 0x991d, 0x991e, 0x9924, 0x9920, 0x992c, 0x992e, // 0xe950 - 0xe957
		0x993d, 0x993e, 0x9942, 0x9949, 0x9945, 0x9950, 0x994b, 0x9951, // 0xe958 - 0xe95f
		0x9952, 0x994c, 0x9955, 0x9997, 0x9998, 0x99a5, 0x99ad, 0x99ae, // 0xe960 - 0xe967
		0x99bc, 0x99df, 0x99db, 0x99dd, 0x99d8, 0x99d1, 0x99ed, 0x99ee, // 0xe968 - 0xe96f
		0x99f1, 0x99f2, 0x99fb, 0x99f8, 0x9a01, 0x9a0f, 0x9a05, 0x99e2, // 0xe970 - 0xe977
		0x9a19, 0x9a2b, 0x9a37, 0x9a45, 0x9a42, 0x9a40, 0x9a43, 0x0000, // 0xe978 - 0xe97f
		0x9a3e, 0x9a55, 0x9a4d, 0x9a5b, 0x9a57, 0x9a5f, 0x9a62, 0x9a65, // 0xe980 - 0xe987
		0x9a64, 0x9a69, 0x9a6b, 0x9a6a, 0x9aad, 0x9ab0, 0x9abc, 0x9ac0, // 0xe988 - 0xe98f
		0x9acf, 0x9ad1, 0x9ad3, 0x9ad4, 0x9ade, 0x9adf, 0x9ae2, 0x9ae3, // 0xe990 - 0xe997
		0x9ae6, 0x9aef, 0x9aeb, 0x9aee, 0x9af4, 0x9af1, 0x9af7, 0x9afb, // 0xe998 - 0xe99f
		0x9b06, 0x9b18, 0x9b1a, 0x9b1f, 0x9b22, 0x9b23, 0x9b25, 0x9b27, // 0xe9a0 - 0xe9a7
		0x9b28, 0x9b29, 0x9b2a, 0x9b2e, 0x9b2f, 0x9b32, 0x9b44, 0x9b43, // 0xe9a8 - 0xe9af
		0x9b4f, 0x9b4d, 0x9b4e, 0x9b51, 0x9b58, 0x9b74, 0x9b93, 0x9b83, // 0xe9b0 - 0xe9b7
		0x9b91, 0x9b96, 0x9b97, 0x9b9f, 0x9ba0, 0x9ba8, 0x9bb4, 0x9bc0, // 0xe9b8 - 0xe9bf
		0x9bca, 0x9bb9, 0x9bc6, 0x9bcf, 0x9bd1, 0x9bd2, 0x9be3, 0x9be2, // 0xe9c0 - 0xe9c7
		0x9be4, 0x9bd4, 0x9be1, 0x9c3a, 0x9bf2, 0x9bf1, 0x9bf0, 0x9c15, // 0xe9c8 - 0xe9cf
		0x9c14, 0x9c09, 0x9c13, 0x9c0c, 0x9c06, 0x9c08, 0x9c12, 0x9c0a, // 0xe9d0 - 0xe9d7
		0x9c04, 0x9c2e, 0x9c1b, 0x9c25, 0x9c24, 0x9c21, 0x9c30, 0x9c47, // 0xe9d8 - 0xe9df
		0x9c32, 0x9c46, 0x9c3e, 0x9c5a, 0x9c60, 0x9c67, 0x9c76, 0x9c78, // 0xe9e0 - 0xe9e7
		0x9ce7, 0x9cec, 0x9cf0, 0x9d09, 0x9d08, 0x9ceb, 0x9d03, 0x9d06, // 0xe9e8 - 0xe9ef
		0x9d2a, 0x9d26, 0x9daf, 0x9d23, 0x9d1f, 0x9d44, 0x9d15, 0x9d12, // 0xe9f0 - 0xe9f7
		0x9d41, 0x9d3f, 0x9d3e, 0x9d46, 0x9d48, 0x8f87, 0x8f8b, 0x0000, // 0xe9f8 - 0xe9ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea00 - 0xea07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea08 - 0xea0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea10 - 0xea17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea18 - 0xea1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea20 - 0xea27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea28 - 0xea2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea30 - 0xea37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xea38 - 0xea3f
		0x9d5d, 0x9d5e, 0x9d64, 0x9d51, 0x9d50, 0x9d59, 0x9d72, 0x9d89, // 0xea40 - 0xea47
		0x9d87, 0x9dab, 0x9d6f, 0x9d7a, 0x9d9a, 0x9da4, 0x9da9, 0x9db2, // 0xea48 - 0xea4f
		0x9dc4, 0x9dc1, 0x9dbb, 0x9db8, 0x9dba, 0x9dc6, 0x9dcf, 0x9dc2, // 0xea50 - 0xea57
		0x9dd9, 0x9dd3, 0x9df8, 0x9de6, 0x9ded, 0x9def, 0x9dfd, 0x9e1a, // 0xea58 - 0xea5f
		0x9e1b, 0x9e1e, 0x9e75, 0x9e79, 0x9e7d, 0x9e81, 0x9e88, 0x9e8b, // 0xea60 - 0xea67
		0x9e8c, 0x9e92, 0x9e95, 0x9e91, 0x9e9d, 0x9ea5, 0x9ea9, 0x9eb8, // 0xea68 - 0xea6f
		0x9eaa, 0x9ead, 0x9761, 0x9ecc, 0x9ece, 0x9ecf, 0x9ed0, 0x9ed4, // 0xea70 - 0xea77
		0x9edc, 0x9ede, 0x9edd, 0x9ee0, 0x9ee5, 0x9ee8, 0x9eef, 0x0000, // 0xea78 - 0xea7f
		0x9ef4, 0x9ef6, 0x9ef7, 0x9ef9, 0x9efb, 0x9efc, 0x9efd, 0x9f07, // 0xea80 - 0xea87
		0x9f08, 0x76b7, 0x9f15, 0x9f21, 0x9f2c, 0x9f3e, 0x9f4a, 0x9f52, // 0xea88 - 0xea8f
		0x9f54, 0x9f63, 0x9f5f, 0x9f60, 0x9f61, 0x9f66, 0x9f67, 0x9f6c, // 0xea90 - 0xea97
		0x9f6a, 0x9f77, 0x9f72, 0x9f76, 0x9f95, 0x9f9c, 0x9fa0, 0x582f, // 0xea98 - 0xea9f
		0x69c7, 0x9059, 0x7464, 0x51dc, 0x7199, 0x8f9a, 0x8ece, 0x620b, // 0xeaa0 - 0xeaa7
		0x6217, 0x621b, 0x621f, 0x6222, 0x6221, 0x6225, 0x6224, 0x622c, // 0xeaa8 - 0xeaaf
		0x81e7, 0x74ef, 0x74f4, 0x74ff, 0x750f, 0x7511, 0x7513, 0x6534, // 0xeab0 - 0xeab7
		0x65ee, 0x65ef, 0x65f0, 0x660a, 0x6619, 0x6772, 0x6603, 0x6615, // 0xeab8 - 0xeabf
		0x6600, 0x7085, 0x66f7, 0x661d, 0x6634, 0x6631, 0x6636, 0x6635, // 0xeac0 - 0xeac7
		0x8006, 0x665f, 0x6654, 0x6641, 0x664f, 0x6656, 0x6661, 0x6657, // 0xeac8 - 0xeacf
		0x6677, 0x6684, 0x668c, 0x66a7, 0x669d, 0x66be, 0x66db, 0x66dc, // 0xead0 - 0xead7
		0x66e6, 0x66e9, 0x8d32, 0x8d33, 0x8d36, 0x8d3b, 0x8d3d, 0x8d40, // 0xead8 - 0xeadf
		0x8d45, 0x8d46, 0x8d48, 0x8d49, 0x8d47, 0x8d4d, 0x8d55, 0x8d59, // 0xeae0 - 0xeae7
		0x89c7, 0x89ca, 0x89cb, 0x89cc, 0x89ce, 0x89cf, 0x89d0, 0x89d1, // 0xeae8 - 0xeaef
		0x726e, 0x729f, 0x725d, 0x7266, 0x726f, 0x727e, 0x727f, 0x7284, // 0xeaf0 - 0xeaf7
		0x728b, 0x728d, 0x728f, 0x7292, 0x6308, 0x6332, 0x63b0, 0x0000, // 0xeaf8 - 0xeaff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb00 - 0xeb07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb08 - 0xeb0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb10 - 0xeb17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb18 - 0xeb1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb20 - 0xeb27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb28 - 0xeb2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb30 - 0xeb37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb38 - 0xeb3f
		0x8540, 0x8546, 0x8564, 0x8541, 0x8562, 0x8544, 0x8551, 0x8547, // 0xeb40 - 0xeb47
		0x8563, 0x853e, 0x855b, 0x8571, 0x854e, 0x856e, 0x8575, 0x8555, // 0xeb48 - 0xeb4f
		0x8567, 0xfe33, 0x858c, 0x8566, 0x855d, 0x8554, 0x8565, 0x856c, // 0xeb50 - 0xeb57
		0x8663, 0x8665, 0x8664, 0x879b, 0xfe31, 0x8797, 0x8793, 0x8792, // 0xeb58 - 0xeb5f
		0x8788, 0x8781, 0x8796, 0x8798, 0xfe30, 0x8787, 0x87a3, 0x8785, // 0xeb60 - 0xeb67
		0x8790, 0xfe35, 0xfe36, 0xfe39, 0xfe3a, 0x879c, 0x879a, 0xfe37, // 0xeb68 - 0xeb6f
		0xfe38, 0xfe3f, 0xfe40, 0xfe3d, 0xfe3e, 0xfe41, 0xfe42, 0xfe43, // 0xeb70 - 0xeb77
		0xfe44, 0xfe3b, 0xfe3c, 0x892c, 0x891f, 0x89f1, 0x8ae0, 0x0000, // 0xeb78 - 0xeb7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb80 - 0xeb87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb88 - 0xeb8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb90 - 0xeb97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xeb98 - 0xeb9f
		0x0000, 0x643f, 0x64d8, 0x8004, 0x6bea, 0x6bf3, 0x6bfd, 0x6bf5, // 0xeba0 - 0xeba7
		0x6bf9, 0x6c05, 0x6c07, 0x6c06, 0x6c0d, 0x6c15, 0x6c18, 0x6c19, // 0xeba8 - 0xebaf
		0x6c1a, 0x6c21, 0x6c29, 0x6c24, 0x6c2a, 0x6c32, 0x6535, 0x6555, // 0xebb0 - 0xebb7
		0x656b, 0x724d, 0x7252, 0x7256, 0x7230, 0x8662, 0x5216, 0x809f, // 0xebb8 - 0xebbf
		0x809c, 0x8093, 0x80bc, 0x670a, 0x80bd, 0x80b1, 0x80ab, 0x80ad, // 0xebc0 - 0xebc7
		0x80b4, 0x80b7, 0x80e7, 0x80e8, 0x80e9, 0x80ea, 0x80db, 0x80c2, // 0xebc8 - 0xebcf
		0x80c4, 0x80d9, 0x80cd, 0x80d7, 0x6710, 0x80dd, 0x80eb, 0x80f1, // 0xebd0 - 0xebd7
		0x80f4, 0x80ed, 0x810d, 0x810e, 0x80f2, 0x80fc, 0x6715, 0x8112, // 0xebd8 - 0xebdf
		0x8c5a, 0x8136, 0x811e, 0x812c, 0x8118, 0x8132, 0x8148, 0x814c, // 0xebe0 - 0xebe7
		0x8153, 0x8174, 0x8159, 0x815a, 0x8171, 0x8160, 0x8169, 0x817c, // 0xebe8 - 0xebef
		0x817d, 0x816d, 0x8167, 0x584d, 0x5ab5, 0x8188, 0x8182, 0x8191, // 0xebf0 - 0xebf7
		0x6ed5, 0x81a3, 0x81aa, 0x81cc, 0x6726, 0x81ca, 0x81bb, 0x0000, // 0xebf8 - 0xebff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec00 - 0xec07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec08 - 0xec0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec10 - 0xec17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec18 - 0xec1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec20 - 0xec27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec28 - 0xec2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec30 - 0xec37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec38 - 0xec3f
		0x930b, 0x92fe, 0x9309, 0x9300, 0x92fb, 0x9316, 0x95bc, 0x95cd, // 0xec40 - 0xec47
		0x95be, 0x95b9, 0x95ba, 0x95b6, 0x95bf, 0x95b5, 0x95bd, 0x96a9, // 0xec48 - 0xec4f
		0x96d4, 0x970b, 0x9712, 0x9710, 0x9799, 0x9797, 0x9794, 0x97f0, // 0xec50 - 0xec57
		0x97f8, 0x9835, 0x982f, 0x9832, 0x9924, 0x991f, 0x9927, 0x9929, // 0xec58 - 0xec5f
		0x999e, 0x99ee, 0x99ec, 0x99e5, 0x99e4, 0x99f0, 0x99e3, 0x99ea, // 0xec60 - 0xec67
		0x99e9, 0x99e7, 0x9ab9, 0x9abf, 0x9ab4, 0x9abb, 0x9af6, 0x9afa, // 0xec68 - 0xec6f
		0x9af9, 0x9af7, 0x9b33, 0x9b80, 0x9b85, 0x9b87, 0x9b7c, 0x9b7e, // 0xec70 - 0xec77
		0x9b7b, 0x9b82, 0x9b93, 0x9b92, 0x9b90, 0x9b7a, 0x9b95, 0x0000, // 0xec78 - 0xec7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec80 - 0xec87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec88 - 0xec8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec90 - 0xec97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xec98 - 0xec9f
		0x0000, 0x81c1, 0x81a6, 0x6b24, 0x6b37, 0x6b39, 0x6b43, 0x6b46, // 0xeca0 - 0xeca7
		0x6b59, 0x98d1, 0x98d2, 0x98d3, 0x98d5, 0x98d9, 0x98da, 0x6bb3, // 0xeca8 - 0xecaf
		0x5f40, 0x6bc2, 0x89f3, 0x6590, 0x9f51, 0x6593, 0x65bc, 0x65c6, // 0xecb0 - 0xecb7
		0x65c4, 0x65c3, 0x65cc, 0x65ce, 0x65d2, 0x65d6, 0x7080, 0x709c, // 0xecb8 - 0xecbf
		0x7096, 0x709d, 0x70bb, 0x70c0, 0x70b7, 0x70ab, 0x70b1, 0x70e8, // 0xecc0 - 0xecc7
		0x70ca, 0x7110, 0x7113, 0x7116, 0x712f, 0x7131, 0x7173, 0x715c, // 0xecc8 - 0xeccf
		0x7168, 0x7145, 0x7172, 0x714a, 0x7178, 0x717a, 0x7198, 0x71b3, // 0xecd0 - 0xecd7
		0x71b5, 0x71a8, 0x71a0, 0x71e0, 0x71d4, 0x71e7, 0x71f9, 0x721d, // 0xecd8 - 0xecdf
		0x7228, 0x706c, 0x7118, 0x7166, 0x71b9, 0x623e, 0x623d, 0x6243, // 0xece0 - 0xece7
		0x6248, 0x6249, 0x793b, 0x7940, 0x7946, 0x7949, 0x795b, 0x795c, // 0xece8 - 0xecef
		0x7953, 0x795a, 0x7962, 0x7957, 0x7960, 0x796f, 0x7967, 0x797a, // 0xecf0 - 0xecf7
		0x7985, 0x798a, 0x799a, 0x79a7, 0x79b3, 0x5fd1, 0x5fd0, 0x0000, // 0xecf8 - 0xecff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed00 - 0xed07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed08 - 0xed0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed10 - 0xed17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed18 - 0xed1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed20 - 0xed27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed28 - 0xed2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed30 - 0xed37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed38 - 0xed3f
		0x6a95, 0x6a83, 0x6aa8, 0x6aa4, 0x6a91, 0x6a7f, 0x6aa6, 0x6a9a, // 0xed40 - 0xed47
		0x6a85, 0x6a8c, 0x6a92, 0x6b5b, 0x6bad, 0x6c09, 0x6fcc, 0x6fa9, // 0xed48 - 0xed4f
		0x6ff4, 0x6fd4, 0x6fe3, 0x6fdc, 0x6fed, 0x6fe7, 0x6fe6, 0x6fde, // 0xed50 - 0xed57
		0x6ff2, 0x6fdd, 0x6fe2, 0x6fe8, 0x71e1, 0x71f1, 0x71e8, 0x71f2, // 0xed58 - 0xed5f
		0x71e4, 0x71f0, 0x71e2, 0x7373, 0x736e, 0x736f, 0x7497, 0x74b2, // 0xed60 - 0xed67
		0x74ab, 0x7490, 0x74aa, 0x74ad, 0x74b1, 0x74a5, 0x74af, 0x7510, // 0xed68 - 0xed6f
		0x7511, 0x7512, 0x750f, 0x7584, 0x7643, 0x7648, 0x7649, 0x7647, // 0xed70 - 0xed77
		0x76a4, 0x76e9, 0x77b5, 0x77ab, 0x77b2, 0x77b7, 0x77b6, 0x0000, // 0xed78 - 0xed7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed80 - 0xed87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed88 - 0xed8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed90 - 0xed97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xed98 - 0xed9f
		0x0000, 0x603c, 0x605d, 0x605a, 0x6067, 0x6041, 0x6059, 0x6063, // 0xeda0 - 0xeda7
		0x60ab, 0x6106, 0x610d, 0x615d, 0x61a9, 0x619d, 0x61cb, 0x61d1, // 0xeda8 - 0xedaf
		0x6206, 0x8080, 0x807f, 0x6c93, 0x6cf6, 0x6dfc, 0x77f6, 0x77f8, // 0xedb0 - 0xedb7
		0x7800, 0x7809, 0x7817, 0x7818, 0x7811, 0x65ab, 0x782d, 0x781c, // 0xedb8 - 0xedbf
		0x781d, 0x7839, 0x783a, 0x783b, 0x781f, 0x783c, 0x7825, 0x782c, // 0xedc0 - 0xedc7
		0x7823, 0x7829, 0x784e, 0x786d, 0x7856, 0x7857, 0x7826, 0x7850, // 0xedc8 - 0xedcf
		0x7847, 0x784c, 0x786a, 0x789b, 0x7893, 0x789a, 0x7887, 0x789c, // 0xedd0 - 0xedd7
		0x78a1, 0x78a3, 0x78b2, 0x78b9, 0x78a5, 0x78d4, 0x78d9, 0x78c9, // 0xedd8 - 0xeddf
		0x78ec, 0x78f2, 0x7905, 0x78f4, 0x7913, 0x7924, 0x791e, 0x7934, // 0xede0 - 0xede7
		0x9f9b, 0x9ef9, 0x9efb, 0x9efc, 0x76f1, 0x7704, 0x770d, 0x76f9, // 0xede8 - 0xedef
		0x7707, 0x7708, 0x771a, 0x7722, 0x7719, 0x772d, 0x7726, 0x7735, // 0xedf0 - 0xedf7
		0x7738, 0x7750, 0x7751, 0x7747, 0x7743, 0x775a, 0x7768, 0x0000, // 0xedf8 - 0xedff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee00 - 0xee07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee08 - 0xee0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee10 - 0xee17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee18 - 0xee1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee20 - 0xee27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee28 - 0xee2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee30 - 0xee37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee38 - 0xee3f
		0x8577, 0x857c, 0x8589, 0x85a1, 0x857a, 0x8578, 0x8557, 0x858e, // 0xee40 - 0xee47
		0x8596, 0x8586, 0x858d, 0x8599, 0x859d, 0x8581, 0x85a2, 0x8582, // 0xee48 - 0xee4f
		0x8588, 0x8585, 0x8579, 0x8576, 0x8598, 0x8590, 0x859f, 0x8668, // 0xee50 - 0xee57
		0x87be, 0x87aa, 0x87ad, 0x87c5, 0x87b0, 0x87ac, 0x87b9, 0x87b5, // 0xee58 - 0xee5f
		0x87bc, 0x87ae, 0x87c9, 0x87c3, 0x87c2, 0x87cc, 0x87b7, 0x87af, // 0xee60 - 0xee67
		0x87c4, 0x87ca, 0x87b4, 0x87b6, 0x87bf, 0x87b8, 0x87bd, 0x87de, // 0xee68 - 0xee6f
		0x87b2, 0x8935, 0x8933, 0x893c, 0x893e, 0x8941, 0x8952, 0x8937, // 0xee70 - 0xee77
		0x8942, 0x89ad, 0x89af, 0x89ae, 0x89f2, 0x89f3, 0x8b1e, 0x0000, // 0xee78 - 0xee7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee80 - 0xee87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee88 - 0xee8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee90 - 0xee97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xee98 - 0xee9f
		0x0000, 0x7762, 0x7765, 0x777f, 0x778d, 0x777d, 0x7780, 0x778c, // 0xeea0 - 0xeea7
		0x7791, 0x779f, 0x77a0, 0x77b0, 0x77b5, 0x77bd, 0x753a, 0x7540, // 0xeea8 - 0xeeaf
		0x754e, 0x754b, 0x7548, 0x755b, 0x7572, 0x7579, 0x7583, 0x7f58, // 0xeeb0 - 0xeeb7
		0x7f61, 0x7f5f, 0x8a48, 0x7f68, 0x7f74, 0x7f71, 0x7f79, 0x7f81, // 0xeeb8 - 0xeebf
		0x7f7e, 0x76cd, 0x76e5, 0x8832, 0x9485, 0x9486, 0x9487, 0x948b, // 0xeec0 - 0xeec7
		0x948a, 0x948c, 0x948d, 0x948f, 0x9490, 0x9494, 0x9497, 0x9495, // 0xeec8 - 0xeecf
		0x949a, 0x949b, 0x949c, 0x94a3, 0x94a4, 0x94ab, 0x94aa, 0x94ad, // 0xeed0 - 0xeed7
		0x94ac, 0x94af, 0x94b0, 0x94b2, 0x94b4, 0x94b6, 0x94b7, 0x94b8, // 0xeed8 - 0xeedf
		0x94b9, 0x94ba, 0x94bc, 0x94bd, 0x94bf, 0x94c4, 0x94c8, 0x94c9, // 0xeee0 - 0xeee7
		0x94ca, 0x94cb, 0x94cc, 0x94cd, 0x94ce, 0x94d0, 0x94d1, 0x94d2, // 0xeee8 - 0xeeef
		0x94d5, 0x94d6, 0x94d7, 0x94d9, 0x94d8, 0x94db, 0x94de, 0x94df, // 0xeef0 - 0xeef7
		0x94e0, 0x94e2, 0x94e4, 0x94e5, 0x94e7, 0x94e8, 0x94ea, 0x0000, // 0xeef8 - 0xeeff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef00 - 0xef07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef08 - 0xef0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef10 - 0xef17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef18 - 0xef1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef20 - 0xef27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef28 - 0xef2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef30 - 0xef37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef38 - 0xef3f
		0x979e, 0x979d, 0x97d5, 0x97d4, 0x97f1, 0x9841, 0x9844, 0x984a, // 0xef40 - 0xef47
		0x9849, 0x9845, 0x9843, 0x9925, 0x992b, 0x992c, 0x992a, 0x9933, // 0xef48 - 0xef4f
		0x9932, 0x992f, 0x992d, 0x9931, 0x9930, 0x9998, 0x99a3, 0x99a1, // 0xef50 - 0xef57
		0x9a02, 0x99fa, 0x99f4, 0x99f7, 0x99f9, 0x99f8, 0x99f6, 0x99fb, // 0xef58 - 0xef5f
		0x99fd, 0x99fe, 0x99fc, 0x9a03, 0x9abe, 0x9afe, 0x9afd, 0x9b01, // 0xef60 - 0xef67
		0x9afc, 0x9b48, 0x9b9a, 0x9ba8, 0x9b9e, 0x9b9b, 0x9ba6, 0x9ba1, // 0xef68 - 0xef6f
		0x9ba5, 0x9ba4, 0x9b86, 0x9ba2, 0x9ba0, 0x9baf, 0x9d33, 0x9d41, // 0xef70 - 0xef77
		0x9d67, 0x9d36, 0x9d2e, 0x9d2f, 0x9d31, 0x9d38, 0x9d30, 0x0000, // 0xef78 - 0xef7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef80 - 0xef87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef88 - 0xef8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef90 - 0xef97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xef98 - 0xef9f
		0x0000, 0x94e9, 0x94eb, 0x94ee, 0x94ef, 0x94f3, 0x94f4, 0x94f5, // 0xefa0 - 0xefa7
		0x94f7, 0x94f9, 0x94fc, 0x94fd, 0x94ff, 0x9503, 0x9502, 0x9506, // 0xefa8 - 0xefaf
		0x9507, 0x9509, 0x950a, 0x950d, 0x950e, 0x950f, 0x9512, 0x9513, // 0xefb0 - 0xefb7
		0x9514, 0x9515, 0x9516, 0x9518, 0x951b, 0x951d, 0x951e, 0x951f, // 0xefb8 - 0xefbf
		0x9522, 0x952a, 0x952b, 0x9529, 0x952c, 0x9531, 0x9532, 0x9534, // 0xefc0 - 0xefc7
		0x9536, 0x9537, 0x9538, 0x953c, 0x953e, 0x953f, 0x9542, 0x9535, // 0xefc8 - 0xefcf
		0x9544, 0x9545, 0x9546, 0x9549, 0x954c, 0x954e, 0x954f, 0x9552, // 0xefd0 - 0xefd7
		0x9553, 0x9554, 0x9556, 0x9557, 0x9558, 0x9559, 0x955b, 0x955e, // 0xefd8 - 0xefdf
		0x955f, 0x955d, 0x9561, 0x9562, 0x9564, 0x9565, 0x9566, 0x9567, // 0xefe0 - 0xefe7
		0x9568, 0x9569, 0x956a, 0x956b, 0x956c, 0x956f, 0x9571, 0x9572, // 0xefe8 - 0xefef
		0x9573, 0x953a, 0x77e7, 0x77ec, 0x96c9, 0x79d5, 0x79ed, 0x79e3, // 0xeff0 - 0xeff7
		0x79eb, 0x7a06, 0x5d47, 0x7a03, 0x7a02, 0x7a1e, 0x7a14, 0x0000, // 0xeff8 - 0xefff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf000 - 0xf007
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf008 - 0xf00f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf010 - 0xf017
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf018 - 0xf01f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf020 - 0xf027
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf028 - 0xf02f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf030 - 0xf037
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf038 - 0xf03f
		0x74b8, 0x74c0, 0x74b5, 0x74c1, 0x74be, 0x74b6, 0x74bb, 0x74c2, // 0xf040 - 0xf047
		0x7514, 0x7513, 0x765c, 0x7664, 0x7659, 0x7650, 0x7653, 0x7657, // 0xf048 - 0xf04f
		0x765a, 0x76a6, 0x76bd, 0x76ec, 0x77c2, 0x77ba, 0x78ff, 0x790c, // 0xf050 - 0xf057
		0x7913, 0x7914, 0x7909, 0x7910, 0x7912, 0x7911, 0x79ad, 0x79ac, // 0xf058 - 0xf05f
		0x7a5f, 0x7c1c, 0x7c29, 0x7c19, 0x7c20, 0x7c1f, 0x7c2d, 0x7c1d, // 0xf060 - 0xf067
		0x7c26, 0x7c28, 0x7c22, 0x7c25, 0x7c30, 0x7e5c, 0x7e50, 0x7e56, // 0xf068 - 0xf06f
		0x7e63, 0x7e58, 0x7e62, 0x7e5f, 0x7e51, 0x7e60, 0x7e57, 0x7e53, // 0xf070 - 0xf077
		0x7fb5, 0x7fb3, 0x7ff7, 0x7ff8, 0x8075, 0x81d1, 0x81d2, 0x0000, // 0xf078 - 0xf07f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf080 - 0xf087
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf088 - 0xf08f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf090 - 0xf097
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf098 - 0xf09f
		0x0000, 0x7a39, 0x7a37, 0x7a51, 0x9ecf, 0x99a5, 0x7a70, 0x7688, // 0xf0a0 - 0xf0a7
		0x768e, 0x7693, 0x7699, 0x76a4, 0x74de, 0x74e0, 0x752c, 0x9e20, // 0xf0a8 - 0xf0af
		0x9e22, 0x9e28, 0x9e29, 0x9e2a, 0x9e2b, 0x9e2c, 0x9e32, 0x9e31, // 0xf0b0 - 0xf0b7
		0x9e36, 0x9e38, 0x9e37, 0x9e39, 0x9e3a, 0x9e3e, 0x9e41, 0x9e42, // 0xf0b8 - 0xf0bf
		0x9e44, 0x9e46, 0x9e47, 0x9e48, 0x9e49, 0x9e4b, 0x9e4c, 0x9e4e, // 0xf0c0 - 0xf0c7
		0x9e51, 0x9e55, 0x9e57, 0x9e5a, 0x9e5b, 0x9e5c, 0x9e5e, 0x9e63, // 0xf0c8 - 0xf0cf
		0x9e66, 0x9e67, 0x9e68, 0x9e69, 0x9e6a, 0x9e6b, 0x9e6c, 0x9e71, // 0xf0d0 - 0xf0d7
		0x9e6d, 0x9e73, 0x7592, 0x7594, 0x7596, 0x75a0, 0x759d, 0x75ac, // 0xf0d8 - 0xf0df
		0x75a3, 0x75b3, 0x75b4, 0x75b8, 0x75c4, 0x75b1, 0x75b0, 0x75c3, // 0xf0e0 - 0xf0e7
		0x75c2, 0x75d6, 0x75cd, 0x75e3, 0x75e8, 0x75e6, 0x75e4, 0x75eb, // 0xf0e8 - 0xf0ef
		0x75e7, 0x7603, 0x75f1, 0x75fc, 0x75ff, 0x7610, 0x7600, 0x7605, // 0xf0f0 - 0xf0f7
		0x760c, 0x7617, 0x760a, 0x7625, 0x7618, 0x7615, 0x7619, 0x0000, // 0xf0f8 - 0xf0ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf100 - 0xf107
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf108 - 0xf10f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf110 - 0xf117
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf118 - 0xf11f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf120 - 0xf127
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf128 - 0xf12f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf130 - 0xf137
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf138 - 0xf13f
		0x8e5b, 0x8e5a, 0x8e61, 0x8e5d, 0x8e69, 0x8e54, 0x8f46, 0x8f47, // 0xf140 - 0xf147
		0x8f48, 0x8f4b, 0x9128, 0x913a, 0x913b, 0x913e, 0x91a8, 0x91a5, // 0xf148 - 0xf14f
		0x91a7, 0x91af, 0x91aa, 0x93b5, 0x938c, 0x9392, 0x93b7, 0x939b, // 0xf150 - 0xf157
		0x939d, 0x9389, 0x93a7, 0x938e, 0x93aa, 0x939e, 0x93a6, 0x9395, // 0xf158 - 0xf15f
		0x9388, 0x9399, 0x939f, 0x938d, 0x93b1, 0x9391, 0x93b2, 0x93a4, // 0xf160 - 0xf167
		0x93a8, 0x93b4, 0x93a3, 0x93a5, 0x95d2, 0x95d3, 0x95d1, 0x96b3, // 0xf168 - 0xf16f
		0x96d7, 0x96da, 0x5dc2, 0x96df, 0x96d8, 0x96dd, 0x9723, 0x9722, // 0xf170 - 0xf177
		0x9725, 0x97ac, 0x97ae, 0x97a8, 0x97ab, 0x97a4, 0x97aa, 0x0000, // 0xf178 - 0xf17f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf180 - 0xf187
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf188 - 0xf18f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf190 - 0xf197
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf198 - 0xf19f
		0x0000, 0x761b, 0x763c, 0x7622, 0x7620, 0x7640, 0x762d, 0x7630, // 0xf1a0 - 0xf1a7
		0x763f, 0x7635, 0x7643, 0x763e, 0x7633, 0x764d, 0x765e, 0x7654, // 0xf1a8 - 0xf1af
		0x765c, 0x7656, 0x766b, 0x766f, 0x7fca, 0x7ae6, 0x7a78, 0x7a79, // 0xf1b0 - 0xf1b7
		0x7a80, 0x7a86, 0x7a88, 0x7a95, 0x7aa6, 0x7aa0, 0x7aac, 0x7aa8, // 0xf1b8 - 0xf1bf
		0x7aad, 0x7ab3, 0x8864, 0x8869, 0x8872, 0x887d, 0x887f, 0x8882, // 0xf1c0 - 0xf1c7
		0x88a2, 0x88c6, 0x88b7, 0x88bc, 0x88c9, 0x88e2, 0x88ce, 0x88e3, // 0xf1c8 - 0xf1cf
		0x88e5, 0x88f1, 0x891a, 0x88fc, 0x88e8, 0x88fe, 0x88f0, 0x8921, // 0xf1d0 - 0xf1d7
		0x8919, 0x8913, 0x891b, 0x890a, 0x8934, 0x892b, 0x8936, 0x8941, // 0xf1d8 - 0xf1df
		0x8966, 0x897b, 0x758b, 0x80e5, 0x76b2, 0x76b4, 0x77dc, 0x8012, // 0xf1e0 - 0xf1e7
		0x8014, 0x8016, 0x801c, 0x8020, 0x8022, 0x8025, 0x8026, 0x8027, // 0xf1e8 - 0xf1ef
		0x8029, 0x8028, 0x8031, 0x800b, 0x8035, 0x8043, 0x8046, 0x804d, // 0xf1f0 - 0xf1f7
		0x8052, 0x8069, 0x8071, 0x8983, 0x9878, 0x9880, 0x9883, 0x0000, // 0xf1f8 - 0xf1ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf200 - 0xf207
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf208 - 0xf20f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf210 - 0xf217
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf218 - 0xf21f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf220 - 0xf227
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf228 - 0xf22f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf230 - 0xf237
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf238 - 0xf23f
		0x5fbf, 0x61fb, 0x6507, 0x6510, 0x650d, 0x6509, 0x650c, 0x650e, // 0xf240 - 0xf247
		0x6584, 0x65de, 0x65dd, 0x66de, 0x6ae7, 0x6ae0, 0x6acc, 0x6ad1, // 0xf248 - 0xf24f
		0x6ad9, 0x6acb, 0x6adf, 0x6adc, 0x6ad0, 0x6aeb, 0x6acf, 0x6acd, // 0xf250 - 0xf257
		0x6ade, 0x6b60, 0x6bb0, 0x6c0c, 0x7019, 0x7027, 0x7020, 0x7016, // 0xf258 - 0xf25f
		0x702b, 0x7021, 0x7022, 0x7023, 0x7029, 0x7017, 0x7024, 0x701c, // 0xf260 - 0xf267
		0x702a, 0x720c, 0x720a, 0x7207, 0x7202, 0x7205, 0x72a5, 0x72a6, // 0xf268 - 0xf26f
		0x72a4, 0x72a3, 0x72a1, 0x74cb, 0x74c5, 0x74b7, 0x74c3, 0x7516, // 0xf270 - 0xf277
		0x7660, 0x77c9, 0x77ca, 0x77c4, 0x77f1, 0x791d, 0x791b, 0x0000, // 0xf278 - 0xf27f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf280 - 0xf287
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf288 - 0xf28f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf290 - 0xf297
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf298 - 0xf29f
		0x0000, 0x9889, 0x988c, 0x988d, 0x988f, 0x9894, 0x989a, 0x989b, // 0xf2a0 - 0xf2a7
		0x989e, 0x989f, 0x98a1, 0x98a2, 0x98a5, 0x98a6, 0x864d, 0x8654, // 0xf2a8 - 0xf2af
		0x866c, 0x866e, 0x867f, 0x867a, 0x867c, 0x867b, 0x86a8, 0x868d, // 0xf2b0 - 0xf2b7
		0x868b, 0x86ac, 0x869d, 0x86a7, 0x86a3, 0x86aa, 0x8693, 0x86a9, // 0xf2b8 - 0xf2bf
		0x86b6, 0x86c4, 0x86b5, 0x86ce, 0x86b0, 0x86ba, 0x86b1, 0x86af, // 0xf2c0 - 0xf2c7
		0x86c9, 0x86cf, 0x86b4, 0x86e9, 0x86f1, 0x86f2, 0x86ed, 0x86f3, // 0xf2c8 - 0xf2cf
		0x86d0, 0x8713, 0x86de, 0x86f4, 0x86df, 0x86d8, 0x86d1, 0x8703, // 0xf2d0 - 0xf2d7
		0x8707, 0x86f8, 0x8708, 0x870a, 0x870d, 0x8709, 0x8723, 0x873b, // 0xf2d8 - 0xf2df
		0x871e, 0x8725, 0x872e, 0x871a, 0x873e, 0x8748, 0x8734, 0x8731, // 0xf2e0 - 0xf2e7
		0x8729, 0x8737, 0x873f, 0x8782, 0x8722, 0x877d, 0x877e, 0x877b, // 0xf2e8 - 0xf2ef
		0x8760, 0x8770, 0x874c, 0x876e, 0x878b, 0x8753, 0x8763, 0x877c, // 0xf2f0 - 0xf2f7
		0x8764, 0x8759, 0x8765, 0x8793, 0x87af, 0x87a8, 0x87d2, 0x0000, // 0xf2f8 - 0xf2ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf300 - 0xf307
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf308 - 0xf30f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf310 - 0xf317
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf318 - 0xf31f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf320 - 0xf327
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf328 - 0xf32f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf330 - 0xf337
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf338 - 0xf33f
		0x8b51, 0x8b42, 0x8b52, 0x8b57, 0x8c43, 0x8c77, 0x8c76, 0x8c9a, // 0xf340 - 0xf347
		0x8d06, 0x8d07, 0x8d09, 0x8dac, 0x8daa, 0x8dad, 0x8dab, 0x8e6d, // 0xf348 - 0xf34f
		0x8e78, 0x8e73, 0x8e6a, 0x8e6f, 0x8e7b, 0x8ec2, 0x8f52, 0x8f51, // 0xf350 - 0xf357
		0x8f4f, 0x8f50, 0x8f53, 0x8fb4, 0x9140, 0x913f, 0x91b0, 0x91ad, // 0xf358 - 0xf35f
		0x93de, 0x93c7, 0x93cf, 0x93c2, 0x93da, 0x93d0, 0x93f9, 0x93ec, // 0xf360 - 0xf367
		0x93cc, 0x93d9, 0x93a9, 0x93e6, 0x93ca, 0x93d4, 0x93ee, 0x93e3, // 0xf368 - 0xf36f
		0x93d5, 0x93c4, 0x93ce, 0x93c0, 0x93d2, 0x93e7, 0x957d, 0x95da, // 0xf370 - 0xf377
		0x95db, 0x96e1, 0x9729, 0x972b, 0x972c, 0x9728, 0x9726, 0x0000, // 0xf378 - 0xf37f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf380 - 0xf387
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf388 - 0xf38f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf390 - 0xf397
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf398 - 0xf39f
		0x0000, 0x87c6, 0x8788, 0x8785, 0x87ad, 0x8797, 0x8783, 0x87ab, // 0xf3a0 - 0xf3a7
		0x87e5, 0x87ac, 0x87b5, 0x87b3, 0x87cb, 0x87d3, 0x87bd, 0x87d1, // 0xf3a8 - 0xf3af
		0x87c0, 0x87ca, 0x87db, 0x87ea, 0x87e0, 0x87ee, 0x8816, 0x8813, // 0xf3b0 - 0xf3b7
		0x87fe, 0x880a, 0x881b, 0x8821, 0x8839, 0x883c, 0x7f36, 0x7f42, // 0xf3b8 - 0xf3bf
		0x7f44, 0x7f45, 0x8210, 0x7afa, 0x7afd, 0x7b08, 0x7b03, 0x7b04, // 0xf3c0 - 0xf3c7
		0x7b15, 0x7b0a, 0x7b2b, 0x7b0f, 0x7b47, 0x7b38, 0x7b2a, 0x7b19, // 0xf3c8 - 0xf3cf
		0x7b2e, 0x7b31, 0x7b20, 0x7b25, 0x7b24, 0x7b33, 0x7b3e, 0x7b1e, // 0xf3d0 - 0xf3d7
		0x7b58, 0x7b5a, 0x7b45, 0x7b75, 0x7b4c, 0x7b5d, 0x7b60, 0x7b6e, // 0xf3d8 - 0xf3df
		0x7b7b, 0x7b62, 0x7b72, 0x7b71, 0x7b90, 0x7ba6, 0x7ba7, 0x7bb8, // 0xf3e0 - 0xf3e7
		0x7bac, 0x7b9d, 0x7ba8, 0x7b85, 0x7baa, 0x7b9c, 0x7ba2, 0x7bab, // 0xf3e8 - 0xf3ef
		0x7bb4, 0x7bd1, 0x7bc1, 0x7bcc, 0x7bdd, 0x7bda, 0x7be5, 0x7be6, // 0xf3f0 - 0xf3f7
		0x7bea, 0x7c0c, 0x7bfe, 0x7bfc, 0x7c0f, 0x7c16, 0x7c0b, 0x0000, // 0xf3f8 - 0xf3ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf400 - 0xf407
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf408 - 0xf40f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf410 - 0xf417
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf418 - 0xf41f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf420 - 0xf427
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf428 - 0xf42f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf430 - 0xf437
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf438 - 0xf43f
		0x56b5, 0x56b3, 0x58e3, 0x5b45, 0x5dc6, 0x5dc7, 0x5eee, 0x5eef, // 0xf440 - 0xf447
		0x5fc0, 0x5fc1, 0x61f9, 0x6517, 0x6516, 0x6515, 0x6513, 0x65df, // 0xf448 - 0xf44f
		0x66e8, 0x66e3, 0x66e4, 0x6af3, 0x6af0, 0x6aea, 0x6ae8, 0x6af9, // 0xf450 - 0xf457
		0x6af1, 0x6aee, 0x6aef, 0x703c, 0x7035, 0x702f, 0x7037, 0x7034, // 0xf458 - 0xf45f
		0x7031, 0x7042, 0x7038, 0x703f, 0x703a, 0x7039, 0x7040, 0x703b, // 0xf460 - 0xf467
		0x7033, 0x7041, 0x7213, 0x7214, 0x72a8, 0x737d, 0x737c, 0x74ba, // 0xf468 - 0xf46f
		0x76ab, 0x76aa, 0x76be, 0x76ed, 0x77cc, 0x77ce, 0x77cf, 0x77cd, // 0xf470 - 0xf477
		0x77f2, 0x7925, 0x7923, 0x7927, 0x7928, 0x7924, 0x7929, 0x0000, // 0xf478 - 0xf47f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf480 - 0xf487
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf488 - 0xf48f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf490 - 0xf497
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf498 - 0xf49f
		0x0000, 0x7c1f, 0x7c2a, 0x7c26, 0x7c38, 0x7c41, 0x7c40, 0x81fe, // 0xf4a0 - 0xf4a7
		0x8201, 0x8202, 0x8204, 0x81ec, 0x8844, 0x8221, 0x8222, 0x8223, // 0xf4a8 - 0xf4af
		0x822d, 0x822f, 0x8228, 0x822b, 0x8238, 0x823b, 0x8233, 0x8234, // 0xf4b0 - 0xf4b7
		0x823e, 0x8244, 0x8249, 0x824b, 0x824f, 0x825a, 0x825f, 0x8268, // 0xf4b8 - 0xf4bf
		0x887e, 0x8885, 0x8888, 0x88d8, 0x88df, 0x895e, 0x7f9d, 0x7f9f, // 0xf4c0 - 0xf4c7
		0x7fa7, 0x7faf, 0x7fb0, 0x7fb2, 0x7c7c, 0x6549, 0x7c91, 0x7c9d, // 0xf4c8 - 0xf4cf
		0x7c9c, 0x7c9e, 0x7ca2, 0x7cb2, 0x7cbc, 0x7cbd, 0x7cc1, 0x7cc7, // 0xf4d0 - 0xf4d7
		0x7ccc, 0x7ccd, 0x7cc8, 0x7cc5, 0x7cd7, 0x7ce8, 0x826e, 0x66a8, // 0xf4d8 - 0xf4df
		0x7fbf, 0x7fce, 0x7fd5, 0x7fe5, 0x7fe1, 0x7fe6, 0x7fe9, 0x7fee, // 0xf4e0 - 0xf4e7
		0x7ff3, 0x7cf8, 0x7d77, 0x7da6, 0x7dae, 0x7e47, 0x7e9b, 0x9eb8, // 0xf4e8 - 0xf4ef
		0x9eb4, 0x8d73, 0x8d84, 0x8d94, 0x8d91, 0x8db1, 0x8d67, 0x8d6d, // 0xf4f0 - 0xf4f7
		0x8c47, 0x8c49, 0x914a, 0x9150, 0x914e, 0x914f, 0x9164, 0x0000, // 0xf4f8 - 0xf4ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf500 - 0xf507
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf508 - 0xf50f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf510 - 0xf517
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf518 - 0xf51f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf520 - 0xf527
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf528 - 0xf52f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf530 - 0xf537
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf538 - 0xf53f
		0x93fc, 0x940c, 0x93f6, 0x9411, 0x9406, 0x95de, 0x95e0, 0x95df, // 0xf540 - 0xf547
		0x972e, 0x972f, 0x97b9, 0x97bb, 0x97fd, 0x97fe, 0x9860, 0x9862, // 0xf548 - 0xf54f
		0x9863, 0x985f, 0x98c1, 0x98c2, 0x9950, 0x994e, 0x9959, 0x994c, // 0xf550 - 0xf557
		0x994b, 0x9953, 0x9a32, 0x9a34, 0x9a31, 0x9a2c, 0x9a2a, 0x9a36, // 0xf558 - 0xf55f
		0x9a29, 0x9a2e, 0x9a38, 0x9a2d, 0x9ac7, 0x9aca, 0x9ac6, 0x9b10, // 0xf560 - 0xf567
		0x9b12, 0x9b11, 0x9c0b, 0x9c08, 0x9bf7, 0x9c05, 0x9c12, 0x9bf8, // 0xf568 - 0xf56f
		0x9c40, 0x9c07, 0x9c0e, 0x9c06, 0x9c17, 0x9c14, 0x9c09, 0x9d9f, // 0xf570 - 0xf577
		0x9d99, 0x9da4, 0x9d9d, 0x9d92, 0x9d98, 0x9d90, 0x9d9b, 0x0000, // 0xf578 - 0xf57f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf580 - 0xf587
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf588 - 0xf58f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf590 - 0xf597
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf598 - 0xf59f
		0x0000, 0x9162, 0x9161, 0x9170, 0x9169, 0x916f, 0x917d, 0x917e, // 0xf5a0 - 0xf5a7
		0x9172, 0x9174, 0x9179, 0x918c, 0x9185, 0x9190, 0x918d, 0x9191, // 0xf5a8 - 0xf5af
		0x91a2, 0x91a3, 0x91aa, 0x91ad, 0x91ae, 0x91af, 0x91b5, 0x91b4, // 0xf5b0 - 0xf5b7
		0x91ba, 0x8c55, 0x9e7e, 0x8db8, 0x8deb, 0x8e05, 0x8e59, 0x8e69, // 0xf5b8 - 0xf5bf
		0x8db5, 0x8dbf, 0x8dbc, 0x8dba, 0x8dc4, 0x8dd6, 0x8dd7, 0x8dda, // 0xf5c0 - 0xf5c7
		0x8dde, 0x8dce, 0x8dcf, 0x8ddb, 0x8dc6, 0x8dec, 0x8df7, 0x8df8, // 0xf5c8 - 0xf5cf
		0x8de3, 0x8df9, 0x8dfb, 0x8de4, 0x8e09, 0x8dfd, 0x8e14, 0x8e1d, // 0xf5d0 - 0xf5d7
		0x8e1f, 0x8e2c, 0x8e2e, 0x8e23, 0x8e2f, 0x8e3a, 0x8e40, 0x8e39, // 0xf5d8 - 0xf5df
		0x8e35, 0x8e3d, 0x8e31, 0x8e49, 0x8e41, 0x8e42, 0x8e51, 0x8e52, // 0xf5e0 - 0xf5e7
		0x8e4a, 0x8e70, 0x8e76, 0x8e7c, 0x8e6f, 0x8e74, 0x8e85, 0x8e8f, // 0xf5e8 - 0xf5ef
		0x8e94, 0x8e90, 0x8e9c, 0x8e9e, 0x8c78, 0x8c82, 0x8c8a, 0x8c85, // 0xf5f0 - 0xf5f7
		0x8c98, 0x8c94, 0x659b, 0x89d6, 0x89de, 0x89da, 0x89dc, 0x0000, // 0xf5f8 - 0xf5ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf600 - 0xf607
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf608 - 0xf60f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf610 - 0xf617
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf618 - 0xf61f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf620 - 0xf627
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf628 - 0xf62f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf630 - 0xf637
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf638 - 0xf63f
		0x8829, 0x881d, 0x881b, 0x8820, 0x8824, 0x881c, 0x882b, 0x884a, // 0xf640 - 0xf647
		0x896d, 0x8969, 0x896e, 0x896b, 0x89fa, 0x8b79, 0x8b78, 0x8b45, // 0xf648 - 0xf64f
		0x8b7a, 0x8b7b, 0x8d10, 0x8d14, 0x8daf, 0x8e8e, 0x8e8c, 0x8f5e, // 0xf650 - 0xf657
		0x8f5b, 0x8f5d, 0x9146, 0x9144, 0x9145, 0x91b9, 0x943f, 0x943b, // 0xf658 - 0xf65f
		0x9436, 0x9429, 0x943d, 0x943c, 0x9430, 0x9439, 0x942a, 0x9437, // 0xf660 - 0xf667
		0x942c, 0x9440, 0x9431, 0x95e5, 0x95e4, 0x95e3, 0x9735, 0x973a, // 0xf668 - 0xf66f
		0x97bf, 0x97e1, 0x9864, 0x98c9, 0x98c6, 0x98c0, 0x9958, 0x9956, // 0xf670 - 0xf677
		0x9a39, 0x9a3d, 0x9a46, 0x9a44, 0x9a42, 0x9a41, 0x9a3a, 0x0000, // 0xf678 - 0xf67f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf680 - 0xf687
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf688 - 0xf68f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf690 - 0xf697
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf698 - 0xf69f
		0x0000, 0x89e5, 0x89eb, 0x89ef, 0x8a3e, 0x8b26, 0x9753, 0x96e9, // 0xf6a0 - 0xf6a7
		0x96f3, 0x96ef, 0x9706, 0x9701, 0x9708, 0x970f, 0x970e, 0x972a, // 0xf6a8 - 0xf6af
		0x972d, 0x9730, 0x973e, 0x9f80, 0x9f83, 0x9f85, 0x9f86, 0x9f87, // 0xf6b0 - 0xf6b7
		0x9f88, 0x9f89, 0x9f8a, 0x9f8c, 0x9efe, 0x9f0b, 0x9f0d, 0x96b9, // 0xf6b8 - 0xf6bf
		0x96bc, 0x96bd, 0x96ce, 0x96d2, 0x77bf, 0x96e0, 0x928e, 0x92ae, // 0xf6c0 - 0xf6c7
		0x92c8, 0x933e, 0x936a, 0x93ca, 0x938f, 0x943e, 0x946b, 0x9c7f, // 0xf6c8 - 0xf6cf
		0x9c82, 0x9c85, 0x9c86, 0x9c87, 0x9c88, 0x7a23, 0x9c8b, 0x9c8e, // 0xf6d0 - 0xf6d7
		0x9c90, 0x9c91, 0x9c92, 0x9c94, 0x9c95, 0x9c9a, 0x9c9b, 0x9c9e, // 0xf6d8 - 0xf6df
		0x9c9f, 0x9ca0, 0x9ca1, 0x9ca2, 0x9ca3, 0x9ca5, 0x9ca6, 0x9ca7, // 0xf6e0 - 0xf6e7
		0x9ca8, 0x9ca9, 0x9cab, 0x9cad, 0x9cae, 0x9cb0, 0x9cb1, 0x9cb2, // 0xf6e8 - 0xf6ef
		0x9cb3, 0x9cb4, 0x9cb5, 0x9cb6, 0x9cb7, 0x9cba, 0x9cbb, 0x9cbc, // 0xf6f0 - 0xf6f7
		0x9cbd, 0x9cc4, 0x9cc5, 0x9cc6, 0x9cc7, 0x9cca, 0x9ccb, 0x0000, // 0xf6f8 - 0xf6ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf700 - 0xf707
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf708 - 0xf70f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf710 - 0xf717
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf718 - 0xf71f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf720 - 0xf727
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf728 - 0xf72f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf730 - 0xf737
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf738 - 0xf73f
		0x7cf4, 0x7cf1, 0x7e91, 0x7f4f, 0x7f87, 0x81de, 0x826b, 0x8634, // 0xf740 - 0xf747
		0x8635, 0x8633, 0x862c, 0x8632, 0x8636, 0x882c, 0x8828, 0x8826, // 0xf748 - 0xf74f
		0x882a, 0x8825, 0x8971, 0x89bf, 0x89be, 0x89fb, 0x8b7e, 0x8b84, // 0xf750 - 0xf757
		0x8b82, 0x8b86, 0x8b85, 0x8b7f, 0x8d15, 0x8e95, 0x8e94, 0x8e9a, // 0xf758 - 0xf75f
		0x8e92, 0x8e90, 0x8e96, 0x8e97, 0x8f60, 0x8f62, 0x9147, 0x944c, // 0xf760 - 0xf767
		0x9450, 0x944a, 0x944b, 0x944f, 0x9447, 0x9445, 0x9448, 0x9449, // 0xf768 - 0xf76f
		0x9446, 0x973f, 0x97e3, 0x986a, 0x9869, 0x98cb, 0x9954, 0x995b, // 0xf770 - 0xf777
		0x9a4e, 0x9a53, 0x9a54, 0x9a4c, 0x9a4f, 0x9a48, 0x9a4a, 0x0000, // 0xf778 - 0xf77f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf780 - 0xf787
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf788 - 0xf78f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf790 - 0xf797
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf798 - 0xf79f
		0x0000, 0x9ccc, 0x9ccd, 0x9cce, 0x9ccf, 0x9cd0, 0x9cd3, 0x9cd4, // 0xf7a0 - 0xf7a7
		0x9cd5, 0x9cd7, 0x9cd8, 0x9cd9, 0x9cdc, 0x9cdd, 0x9cdf, 0x9ce2, // 0xf7a8 - 0xf7af
		0x977c, 0x9785, 0x9791, 0x9792, 0x9794, 0x97af, 0x97ab, 0x97a3, // 0xf7b0 - 0xf7b7
		0x97b2, 0x97b4, 0x9ab1, 0x9ab0, 0x9ab7, 0x9e58, 0x9ab6, 0x9aba, // 0xf7b8 - 0xf7bf
		0x9abc, 0x9ac1, 0x9ac0, 0x9ac5, 0x9ac2, 0x9acb, 0x9acc, 0x9ad1, // 0xf7c0 - 0xf7c7
		0x9b45, 0x9b43, 0x9b47, 0x9b49, 0x9b48, 0x9b4d, 0x9b51, 0x98e8, // 0xf7c8 - 0xf7cf
		0x990d, 0x992e, 0x9955, 0x9954, 0x9adf, 0x9ae1, 0x9ae6, 0x9aef, // 0xf7d0 - 0xf7d7
		0x9aeb, 0x9afb, 0x9aed, 0x9af9, 0x9b08, 0x9b0f, 0x9b13, 0x9b1f, // 0xf7d8 - 0xf7df
		0x9b23, 0x9ebd, 0x9ebe, 0x7e3b, 0x9e82, 0x9e87, 0x9e88, 0x9e8b, // 0xf7e0 - 0xf7e7
		0x9e92, 0x93d6, 0x9e9d, 0x9e9f, 0x9edb, 0x9edc, 0x9edd, 0x9ee0, // 0xf7e8 - 0xf7ef
		0x9edf, 0x9ee2, 0x9ee9, 0x9ee7, 0x9ee5, 0x9eea, 0x9eef, 0x9f22, // 0xf7f0 - 0xf7f7
		0x9f2c, 0x9f2f, 0x9f39, 0x9f37, 0x9f3d, 0x9f3e, 0x9f44, 0x0000, // 0xf7f8 - 0xf7ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf800 - 0xf807
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf808 - 0xf80f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf810 - 0xf817
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf818 - 0xf81f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf820 - 0xf827
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf828 - 0xf82f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf830 - 0xf837
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf838 - 0xf83f
		0x8b8c, 0x8b8e, 0x8b8b, 0x8b88, 0x8c45, 0x8d19, 0x8e98, 0x8f64, // 0xf840 - 0xf847
		0x8f63, 0x91bc, 0x9462, 0x9455, 0x945d, 0x9457, 0x945e, 0x97c4, // 0xf848 - 0xf84f
		0x97c5, 0x9800, 0x9a56, 0x9a59, 0x9b1e, 0x9b1f, 0x9b20, 0x9c52, // 0xf850 - 0xf857
		0x9c58, 0x9c50, 0x9c4a, 0x9c4d, 0x9c4b, 0x9c55, 0x9c59, 0x9c4c, // 0xf858 - 0xf85f
		0x9c4e, 0x9dfb, 0x9df7, 0x9def, 0x9de3, 0x9deb, 0x9df8, 0x9de4, // 0xf860 - 0xf867
		0x9df6, 0x9de1, 0x9dee, 0x9de6, 0x9df2, 0x9df0, 0x9de2, 0x9dec, // 0xf868 - 0xf86f
		0x9df4, 0x9df3, 0x9de8, 0x9ded, 0x9ec2, 0x9ed0, 0x9ef2, 0x9ef3, // 0xf870 - 0xf877
		0x9f06, 0x9f1c, 0x9f38, 0x9f37, 0x9f36, 0x9f43, 0x9f4f, 0x0000, // 0xf878 - 0xf87f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf880 - 0xf887
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf888 - 0xf88f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf890 - 0xf897
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf898 - 0xf89f
		0x0000, 0x9f71, 0x9f70, 0x9f6e, 0x9f6f, 0x56d3, 0x56cd, 0x5b4e, // 0xf8a0 - 0xf8a7
		0x5c6d, 0x652d, 0x66ed, 0x66ee, 0x6b13, 0x705f, 0x7061, 0x705d, // 0xf8a8 - 0xf8af
		0x7060, 0x7223, 0x74db, 0x74e5, 0x77d5, 0x7938, 0x79b7, 0x79b6, // 0xf8b0 - 0xf8b7
		0x7c6a, 0x7e97, 0x7f89, 0x826d, 0x8643, 0x8838, 0x8837, 0x8835, // 0xf8b8 - 0xf8bf
		0x884b, 0x8b94, 0x8b95, 0x8e9e, 0x8e9f, 0x8ea0, 0x8e9d, 0x91be, // 0xf8c0 - 0xf8c7
		0x91bd, 0x91c2, 0x946b, 0x9468, 0x9469, 0x96e5, 0x9746, 0x9743, // 0xf8c8 - 0xf8cf
		0x9747, 0x97c7, 0x97e5, 0x9a5e, 0x9ad5, 0x9b59, 0x9c63, 0x9c67, // 0xf8d0 - 0xf8d7
		0x9c66, 0x9c62, 0x9c5e, 0x9c60, 0x9e02, 0x9dfe, 0x9e07, 0x9e03, // 0xf8d8 - 0xf8df
		0x9e06, 0x9e05, 0x9e00, 0x9e01, 0x9e09, 0x9dff, 0x9dfd, 0x9e04, // 0xf8e0 - 0xf8e7
		0x9ea0, 0x9f1e, 0x9f46, 0x9f74, 0x9f75, 0x9f76, 0x56d4, 0x652e, // 0xf8e8 - 0xf8ef
		0x65b8, 0x6b18, 0x6b19, 0x6b17, 0x6b1a, 0x7062, 0x7226, 0x72aa, // 0xf8f0 - 0xf8f7
		0x77d8, 0x77d9, 0x7939, 0x7c69, 0x7c6b, 0x7cf6, 0x7e9a, 0x0000, // 0xf8f8 - 0xf8ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf900 - 0xf907
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf908 - 0xf90f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf910 - 0xf917
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf918 - 0xf91f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf920 - 0xf927
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf928 - 0xf92f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf930 - 0xf937
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf938 - 0xf93f
		0x7e98, 0x7e9b, 0x7e99, 0x81e0, 0x81e1, 0x8646, 0x8647, 0x8648, // 0xf940 - 0xf947
		0x8979, 0x897a, 0x897c, 0x897b, 0x89ff, 0x8b98, 0x8b99, 0x8ea5, // 0xf948 - 0xf94f
		0x8ea4, 0x8ea3, 0x946e, 0x946d, 0x946f, 0x9471, 0x9473, 0x9749, // 0xf950 - 0xf957
		0x9872, 0x995f, 0x9c68, 0x9c6e, 0x9c6d, 0x9e0b, 0x9e0d, 0x9e10, // 0xf958 - 0xf95f
		0x9e0f, 0x9e12, 0x9e11, 0x9ea1, 0x9ef5, 0x9f09, 0x9f47, 0x9f78, // 0xf960 - 0xf967
		0x9f7b, 0x9f7a, 0x9f79, 0x571e, 0x7066, 0x7c6f, 0x883c, 0x8db2, // 0xf968 - 0xf96f
		0x8ea6, 0x91c3, 0x9474, 0x9478, 0x9476, 0x9475, 0x9a60, 0x9c74, // 0xf970 - 0xf977
		0x9c73, 0x9c71, 0x9c75, 0x9e14, 0x9e13, 0x9ef6, 0x9f0a, 0x0000, // 0xf978 - 0xf97f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf980 - 0xf987
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf988 - 0xf98f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf990 - 0xf997
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xf998 - 0xf99f
		0x0000, 0x9fa4, 0x7068, 0x7065, 0x7cf7, 0x866a, 0x883e, 0x883d, // 0xf9a0 - 0xf9a7
		0x883f, 0x8b9e, 0x8c9c, 0x8ea9, 0x8ec9, 0x974b, 0x9873, 0x9874, // 0xf9a8 - 0xf9af
		0x98cc, 0x9961, 0x99ab, 0x9a64, 0x9a66, 0x9a67, 0x9b24, 0x9e15, // 0xf9b0 - 0xf9b7
		0x9e17, 0x9f48, 0x6207, 0x6b1e, 0x7227, 0x864c, 0x8ea8, 0x9482, // 0xf9b8 - 0xf9bf
		0x9480, 0x9481, 0x9a69, 0x9a68, 0x9b2e, 0x9e19, 0x7229, 0x864b, // 0xf9c0 - 0xf9c7
		0x8b9f, 0x9483, 0x9c79, 0x9eb7, 0x7675, 0x9a6b, 0x9c7a, 0x9e1d, // 0xf9c8 - 0xf9cf
		0x7069, 0x706a, 0x9ea4, 0x9f7e, 0x9f49, 0x9f98, 0x7f55, 0x7ff0, // 0xf9d0 - 0xf9d7
		0x9591, 0x9592, 0x9650, 0x97d3, 0x5272, 0x8f44, 0x51fd, 0x542b, // 0xf9d8 - 0xf9df
		0x54b8, 0x5563, 0x558a, 0x6abb, 0x6db5, 0x7dd8, 0x8266, 0x929c, // 0xf9e0 - 0xf9e7
		0x9677, 0x9e79, 0x5408, 0x54c8, 0x76d2, 0x86e4, 0x95a4, 0x95d4, // 0xf9e8 - 0xf9ef
		0x965c, 0x4ea2, 0x4f09, 0x59ee, 0x5ae6, 0x5df7, 0x6052, 0x6297, // 0xf9f0 - 0xf9f7
		0x676d, 0x6841, 0x6c86, 0x6e2f, 0x7f38, 0x809b, 0x822a, 0x0000, // 0xf9f8 - 0xf9ff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa00 - 0xfa07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa08 - 0xfa0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa10 - 0xfa17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa18 - 0xfa1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa20 - 0xfa27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa28 - 0xfa2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa30 - 0xfa37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa38 - 0xfa3f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa40 - 0xfa47
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa48 - 0xfa4f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa50 - 0xfa57
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa58 - 0xfa5f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa60 - 0xfa67
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa68 - 0xfa6f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa70 - 0xfa77
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa78 - 0xfa7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa80 - 0xfa87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa88 - 0xfa8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa90 - 0xfa97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfa98 - 0xfa9f
		0x0000, 0xfa08, 0xfa09, 0x9805, 0x4ea5, 0x5055, 0x54b3, 0x5793, // 0xfaa0 - 0xfaa7
		0x595a, 0x5b69, 0x5bb3, 0x61c8, 0x6977, 0x6d77, 0x7023, 0x87f9, // 0xfaa8 - 0xfaaf
		0x89e3, 0x8a72, 0x8ae7, 0x9082, 0x99ed, 0x9ab8, 0x52be, 0x6838, // 0xfab0 - 0xfab7
		0x5016, 0x5e78, 0x674f, 0x8347, 0x884c, 0x4eab, 0x5411, 0x56ae, // 0xfab8 - 0xfabf
		0x73e6, 0x9115, 0x97ff, 0x9909, 0x9957, 0x9999, 0x5653, 0x589f, // 0xfac0 - 0xfac7
		0x865b, 0x8a31, 0x61b2, 0x6af6, 0x737b, 0x8ed2, 0x6b47, 0x96aa, // 0xfac8 - 0xfacf
		0x9a57, 0x5955, 0x7200, 0x8d6b, 0x9769, 0x4fd4, 0x5cf4, 0x5f26, // 0xfad0 - 0xfad7
		0x61f8, 0x665b, 0x6ceb, 0x70ab, 0x7384, 0x73b9, 0x73fe, 0x7729, // 0xfad8 - 0xfadf
		0x774d, 0x7d43, 0x7d62, 0x7e23, 0x8237, 0x8852, 0xfa0a, 0x8ce2, // 0xfae0 - 0xfae7
		0x9249, 0x986f, 0x5b51, 0x7a74, 0x8840, 0x9801, 0x5acc, 0x4fe0, // 0xfae8 - 0xfaef
		0x5354, 0x593e, 0x5cfd, 0x633e, 0x6d79, 0x72f9, 0x8105, 0x8107, // 0xfaf0 - 0xfaf7
		0x83a2, 0x92cf, 0x9830, 0x4ea8, 0x5144, 0x5211, 0x578b, 0x0000, // 0xfaf8 - 0xfaff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb00 - 0xfb07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb08 - 0xfb0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb10 - 0xfb17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb18 - 0xfb1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb20 - 0xfb27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb28 - 0xfb2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb30 - 0xfb37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb38 - 0xfb3f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb40 - 0xfb47
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb48 - 0xfb4f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb50 - 0xfb57
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb58 - 0xfb5f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb60 - 0xfb67
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb68 - 0xfb6f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb70 - 0xfb77
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb78 - 0xfb7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb80 - 0xfb87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb88 - 0xfb8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb90 - 0xfb97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfb98 - 0xfb9f
		0x0000, 0x5f62, 0x6cc2, 0x6ece, 0x7005, 0x7050, 0x70af, 0x7192, // 0xfba0 - 0xfba7
		0x73e9, 0x7469, 0x834a, 0x87a2, 0x8861, 0x9008, 0x90a2, 0x93a3, // 0xfba8 - 0xfbaf
		0x99a8, 0x516e, 0x5f57, 0x60e0, 0x6167, 0x66b3, 0x8559, 0x8e4a, // 0xfbb0 - 0xfbb7
		0x91af, 0x978b, 0x4e4e, 0x4e92, 0x547c, 0x58d5, 0x58fa, 0x597d, // 0xfbb8 - 0xfbbf
		0x5cb5, 0x5f27, 0x6236, 0x6248, 0x660a, 0x6667, 0x6beb, 0x6d69, // 0xfbc0 - 0xfbc7
		0x6dcf, 0x6e56, 0x6ef8, 0x6f94, 0x6fe0, 0x6fe9, 0x705d, 0x72d0, // 0xfbc8 - 0xfbcf
		0x7425, 0x745a, 0x74e0, 0x7693, 0x795c, 0x7cca, 0x7e1e, 0x80e1, // 0xfbd0 - 0xfbd7
		0x82a6, 0x846b, 0x84bf, 0x864e, 0x865f, 0x8774, 0x8b77, 0x8c6a, // 0xfbd8 - 0xfbdf
		0x93ac, 0x9800, 0x9865, 0x60d1, 0x6216, 0x9177, 0x5a5a, 0x660f, // 0xfbe0 - 0xfbe7
		0x6df7, 0x6e3e, 0x743f, 0x9b42, 0x5ffd, 0x60da, 0x7b0f, 0x54c4, // 0xfbe8 - 0xfbef
		0x5f18, 0x6c5e, 0x6cd3, 0x6d2a, 0x70d8, 0x7d05, 0x8679, 0x8a0c, // 0xfbf0 - 0xfbf7
		0x9d3b, 0x5316, 0x548c, 0x5b05, 0x6a3a, 0x706b, 0x7575, 0x0000, // 0xfbf8 - 0xfbff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc00 - 0xfc07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc08 - 0xfc0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc10 - 0xfc17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc18 - 0xfc1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc20 - 0xfc27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc28 - 0xfc2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc30 - 0xfc37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc38 - 0xfc3f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc40 - 0xfc47
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc48 - 0xfc4f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc50 - 0xfc57
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc58 - 0xfc5f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc60 - 0xfc67
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc68 - 0xfc6f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc70 - 0xfc77
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc78 - 0xfc7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc80 - 0xfc87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc88 - 0xfc8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc90 - 0xfc97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfc98 - 0xfc9f
		0x0000, 0x798d, 0x79be, 0x82b1, 0x83ef, 0x8a71, 0x8b41, 0x8ca8, // 0xfca0 - 0xfca7
		0x9774, 0xfa0b, 0x64f4, 0x652b, 0x78ba, 0x78bb, 0x7a6b, 0x4e38, // 0xfca8 - 0xfcaf
		0x559a, 0x5950, 0x5ba6, 0x5e7b, 0x60a3, 0x63db, 0x6b61, 0x6665, // 0xfcb0 - 0xfcb7
		0x6853, 0x6e19, 0x7165, 0x74b0, 0x7d08, 0x9084, 0x9a69, 0x9c25, // 0xfcb8 - 0xfcbf
		0x6d3b, 0x6ed1, 0x733e, 0x8c41, 0x95ca, 0x51f0, 0x5e4c, 0x5fa8, // 0xfcc0 - 0xfcc7
		0x604d, 0x60f6, 0x6130, 0x614c, 0x6643, 0x6644, 0x69a5, 0x6cc1, // 0xfcc8 - 0xfccf
		0x6e5f, 0x6ec9, 0x6f62, 0x714c, 0x749c, 0x7687, 0x7bc1, 0x7c27, // 0xfcd0 - 0xfcd7
		0x8352, 0x8757, 0x9051, 0x968d, 0x9ec3, 0x532f, 0x56de, 0x5efb, // 0xfcd8 - 0xfcdf
		0x5f8a, 0x6062, 0x6094, 0x61f7, 0x6666, 0x6703, 0x6a9c, 0x6dee, // 0xfce0 - 0xfce7
		0x6fae, 0x7070, 0x736a, 0x7e6a, 0x81be, 0x8334, 0x86d4, 0x8aa8, // 0xfce8 - 0xfcef
		0x8cc4, 0x5283, 0x7372, 0x5b96, 0x6a6b, 0x9404, 0x54ee, 0x5686, // 0xfcf0 - 0xfcf7
		0x5b5d, 0x6548, 0x6585, 0x66c9, 0x689f, 0x6d8d, 0x6dc6, 0x0000, // 0xfcf8 - 0xfcff
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd00 - 0xfd07
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd08 - 0xfd0f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd10 - 0xfd17
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd18 - 0xfd1f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd20 - 0xfd27
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd28 - 0xfd2f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd30 - 0xfd37
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd38 - 0xfd3f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd40 - 0xfd47
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd48 - 0xfd4f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd50 - 0xfd57
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd58 - 0xfd5f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd60 - 0xfd67
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd68 - 0xfd6f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd70 - 0xfd77
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd78 - 0xfd7f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd80 - 0xfd87
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd88 - 0xfd8f
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd90 - 0xfd97
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, // 0xfd98 - 0xfd9f
		0x0000, 0x723b, 0x80b4, 0x9175, 0x9a4d, 0x4faf, 0x5019, 0x539a, // 0xfda0 - 0xfda7
		0x540e, 0x543c, 0x5589, 0x55c5, 0x5e3f, 0x5f8c, 0x673d, 0x7166, // 0xfda8 - 0xfdaf
		0x73dd, 0x9005, 0x52db, 0x52f3, 0x5864, 0x58ce, 0x7104, 0x718f, // 0xfdb0 - 0xfdb7
		0x71fb, 0x85b0, 0x8a13, 0x6688, 0x85a8, 0x55a7, 0x6684, 0x714a, // 0xfdb8 - 0xfdbf
		0x8431, 0x5349, 0x5599, 0x6bc1, 0x5f59, 0x5fbd, 0x63ee, 0x6689, // 0xfdc0 - 0xfdc7
		0x7147, 0x8af1, 0x8f1d, 0x9ebe, 0x4f11, 0x643a, 0x70cb, 0x7566, // 0xfdc8 - 0xfdcf
		0x8667, 0x6064, 0x8b4e, 0x9df8, 0x5147, 0x51f6, 0x5308, 0x6d36, // 0xfdd0 - 0xfdd7
		0x80f8, 0x9ed1, 0x6615, 0x6b23, 0x7098, 0x75d5, 0x5403, 0x5c79, // 0xfdd8 - 0xfddf
		0x7d07, 0x8a16, 0x6b20, 0x6b3d, 0x6b46, 0x5438, 0x6070, 0x6d3d, // 0xfde0 - 0xfde7
		0x7fd5, 0x8208, 0x50d6, 0x51de, 0x559c, 0x566b, 0x56cd, 0x59ec, // 0xfde8 - 0xfdef
		0x5b09, 0x5e0c, 0x6199, 0x6198, 0x6231, 0x665e, 0x66e6, 0x7199, // 0xfdf0 - 0xfdf7
		0x71b9, 0x71ba, 0x72a7, 0x79a7, 0x7a00, 0x7fb2, 0x8a70, 0x0000  // 0xfdf8 - 0xfdff
	};

	static const WPXComplexMap charComplexMap[] =
	{
		{ 0x8591, { 0xF860, 0x0030, 0x002E, 0x0000 } },
		{ 0x85AB, { 0xF862, 0x0058, 0x0049, 0x0049, 0x0049, 0x0000 } },
		{ 0x85AC, { 0xF861, 0x0058, 0x0049, 0x0056, 0x0000 } },
		{ 0x85AD, { 0xF860, 0x0058, 0x0056, 0x0000 } },
		{ 0x85BF, { 0xF862, 0x0078, 0x0069, 0x0069, 0x0069, 0x0000 } },
		{ 0x85C0, { 0xF861, 0x0078, 0x0069, 0x0076, 0x0000 } },
		{ 0x85C1, { 0xF860, 0x0078, 0x0076, 0x0000 } },
		{ 0x8645, { 0xFF4D, 0xF87F, 0x0000 } },
		{ 0x864B, { 0xFF47, 0xF87F, 0x0000 } },
		{ 0x865D, { 0xF860, 0x0054, 0x0042, 0x0000 } },
		{ 0x869E, { 0xF861, 0x0046, 0x0041, 0x0058, 0x0000 } },
		{ 0x86CE, { 0xF860, 0x2193, 0x2191, 0x0000 } },
		{ 0x86D3, { 0x21E8, 0xF87A, 0x0000 } },
		{ 0x86D4, { 0x21E6, 0xF87A, 0x0000 } },
		{ 0x86D5, { 0x21E7, 0xF87A, 0x0000 } },
		{ 0x86D6, { 0x21E9, 0xF87A, 0x0000 } },
		{ 0x8791, { 0x5927, 0x20DD, 0x0000 } },
		{ 0x8792, { 0x5C0F, 0x20DD, 0x0000 } },
		{ 0x879D, { 0x63A7, 0x20DD, 0x0000 } },
		{ 0x87FB, { 0xF862, 0x6709, 0x9650, 0x4F1A, 0x793E, 0x0000 } },
		{ 0x87FC, { 0xF862, 0x8CA1, 0x56E3, 0x6CD5, 0x4EBA, 0x0000 } },
		{ 0xA141, { 0x300C, 0xF87F, 0x0000 } },
		{ 0xA142, { 0x300D, 0xF87F, 0x0000 } },
		{ 0xA143, { 0x300C, 0xF87B, 0x0000 } },
		{ 0xA144, { 0x300D, 0xF87B, 0x0000 } },
		{ 0xA145, { 0x300C, 0xF87C, 0x0000 } },
		{ 0xA146, { 0x300D, 0xF87C, 0x0000 } },
		{ 0xA147, { 0x300E, 0xF87C, 0x0000 } },
		{ 0xA148, { 0x300F, 0xF87C, 0x0000 } },
		{ 0xA149, { 0x300A, 0xF878, 0x0000 } },
		{ 0xA14A, { 0x300B, 0xF878, 0x0000 } },
		{ 0xA14B, { 0x3008, 0xF878, 0x0000 } },
		{ 0xA14C, { 0x3009, 0xF878, 0x0000 } },
		{ 0xA14D, { 0xFF0C, 0xF87D, 0x0000 } },
		{ 0xA14E, { 0x3001, 0xF87D, 0x0000 } },
		{ 0xA14F, { 0xFE59, 0xF87F, 0x0000 } },
		{ 0xA14F, { 0xFF0E, 0xF87D, 0x0000 } },
		{ 0xA150, { 0xFE5A, 0xF87F, 0x0000 } },
		{ 0xA150, { 0xFF0E, 0xF87E, 0x0000 } },
		{ 0xA151, { 0x2985, 0xF87F, 0x0000 } },
		{ 0xA151, { 0xFF1B, 0xF87D, 0x0000 } },
		{ 0xA152, { 0x2986, 0xF87F, 0x0000 } },
		{ 0xA152, { 0xFF1A, 0xF87D, 0x0000 } },
		{ 0xA153, { 0x2985, 0xF879, 0x0000 } },
		{ 0xA153, { 0xFF1F, 0xF87D, 0x0000 } },
		{ 0xA154, { 0x2986, 0xF879, 0x0000 } },
		{ 0xA154, { 0xFF01, 0xF87D, 0x0000 } },
		{ 0xA155, { 0x2985, 0xF87C, 0x0000 } },
		{ 0xA156, { 0x2986, 0xF87C, 0x0000 } },
		{ 0xA157, { 0x0028, 0xF87C, 0x0000 } },
		{ 0xA158, { 0x0029, 0xF87C, 0x0000 } },
		{ 0xA15A, { 0xFF3F, 0xF87F, 0x0000 } },
		{ 0xA15B, { 0x3010, 0xF87F, 0x0000 } },
		{ 0xA15C, { 0x3011, 0xF87F, 0x0000 } },
		{ 0xA161, { 0x005B, 0xF87B, 0x0000 } },
		{ 0xA162, { 0x005D, 0xF87B, 0x0000 } },
		{ 0xA163, { 0x005B, 0xF87C, 0x0000 } },
		{ 0xA164, { 0x005D, 0xF87C, 0x0000 } },
		{ 0xA165, { 0x2985, 0xF87B, 0x0000 } },
		{ 0xA166, { 0x2986, 0xF87B, 0x0000 } },
		{ 0xA167, { 0x2020, 0xF87F, 0x0000 } },
		{ 0xA168, { 0x2021, 0xF87F, 0x0000 } },
		{ 0xA169, { 0x2020, 0xF87B, 0x0000 } },
		{ 0xA16A, { 0x2021, 0xF87C, 0x0000 } },
		{ 0xA16B, { 0x2020, 0xF877, 0x0000 } },
		{ 0xA16C, { 0x002A, 0xF877, 0x0000 } },
		{ 0xA173, { 0x00D7, 0xF877, 0x0000 } },
		{ 0xA174, { 0x221E, 0xF877, 0x0000 } },
		{ 0xA175, { 0x223D, 0xF877, 0x0000 } },
		{ 0xA176, { 0x00B1, 0xF877, 0x0000 } },
		{ 0xA177, { 0x2213, 0xF877, 0x0000 } },
		{ 0xA178, { 0x003E, 0xF877, 0x0000 } },
		{ 0xA179, { 0x003C, 0xF877, 0x0000 } },
		{ 0xA17B, { 0x2260, 0xF877, 0x0000 } },
		{ 0xA17D, { 0xFF08, 0xF87F, 0x0000 } },
		{ 0xA17E, { 0xFF09, 0xF87F, 0x0000 } },
		{ 0xA181, { 0x007B, 0xF877, 0x0000 } },
		{ 0xA182, { 0x007D, 0xF877, 0x0000 } },
		{ 0xA183, { 0x005B, 0xF877, 0x0000 } },
		{ 0xA184, { 0x005D, 0xF877, 0x0000 } },
		{ 0xA185, { 0x2229, 0xF877, 0x0000 } },
		{ 0xA186, { 0x222A, 0xF877, 0x0000 } },
		{ 0xA187, { 0x2282, 0xF877, 0x0000 } },
		{ 0xA188, { 0x2208, 0xF877, 0x0000 } },
		{ 0xA189, { 0x2211, 0xF877, 0x0000 } },
		{ 0xA18A, { 0x0021, 0xF877, 0x0000 } },
		{ 0xA18B, { 0x3007, 0xF876, 0x0000 } },
		{ 0xA18C, { 0x4E00, 0xF876, 0x0000 } },
		{ 0xA18D, { 0x4E8C, 0xF876, 0x0000 } },
		{ 0xA18E, { 0x4E09, 0xF876, 0x0000 } },
		{ 0xA18F, { 0x56DB, 0xF876, 0x0000 } },
		{ 0xA190, { 0x4E94, 0xF876, 0x0000 } },
		{ 0xA191, { 0x516D, 0xF876, 0x0000 } },
		{ 0xA192, { 0x4E03, 0xF876, 0x0000 } },
		{ 0xA193, { 0x516B, 0xF876, 0x0000 } },
		{ 0xA194, { 0x4E5D, 0xF876, 0x0000 } },
		{ 0xA195, { 0x5341, 0xF876, 0x0000 } },
		{ 0xA19C, { 0xFF0A, 0xF874, 0x0000 } },
		{ 0xA19D, { 0x2051, 0xF87C, 0x0000 } },
		{ 0xA19E, { 0xFF0A, 0xF875, 0x0000 } },
		{ 0xA19F, { 0xFF0A, 0xF871, 0x0000 } },
		{ 0xA1A0, { 0x2051, 0xF879, 0x0000 } },
		{ 0xA1A1, { 0xFF5B, 0xF87F, 0x0000 } },
		{ 0xA1A2, { 0xFF5D, 0xF87F, 0x0000 } },
		{ 0xA1A3, { 0x3014, 0xF87F, 0x0000 } },
		{ 0xA1A4, { 0x3015, 0xF87F, 0x0000 } },
		{ 0xA1C3, { 0x203E, 0xF87C, 0x0000 } },
		{ 0xA1C5, { 0xFF3F, 0xF87C, 0x0000 } },
		{ 0xA1CB, { 0xFE4B, 0xF87C, 0x0000 } },
		{ 0xA1FE, { 0xFF0F, 0xF87F, 0x0000 } },
		{ 0xA240, { 0xFF3C, 0xF87F, 0x0000 } },
		{ 0xA241, { 0x2985, 0xF878, 0x0000 } },
		{ 0xA242, { 0x2986, 0xF878, 0x0000 } },
		{ 0xA243, { 0x2985, 0xF873, 0x0000 } },
		{ 0xA244, { 0x2986, 0xF873, 0x0000 } },
		{ 0xA245, { 0xFE59, 0xF87C, 0x0000 } },
		{ 0xA246, { 0xFE5A, 0xF87C, 0x0000 } },
		{ 0xA247, { 0x3016, 0xF878, 0x0000 } },
		{ 0xA248, { 0x3017, 0xF878, 0x0000 } },
		{ 0xA249, { 0x3010, 0xF878, 0x0000 } },
		{ 0xA24A, { 0x3011, 0xF878, 0x0000 } },
		{ 0xA24B, { 0x0028, 0xF87F, 0x0000 } },
		{ 0xA24C, { 0x0029, 0xF87F, 0x0000 } },
		{ 0xA279, { 0x2502, 0xF87F, 0x0000 } },
		{ 0xA2CC, { 0x5341, 0xF87F, 0x0000 } },
		{ 0xA2CE, { 0x5345, 0xF87F, 0x0000 } },
		{ 0xA2E6, { 0x0031, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2E7, { 0x0032, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2E8, { 0x0033, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2E9, { 0x0034, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2EA, { 0x0035, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2EB, { 0x0036, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2EC, { 0x0037, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2ED, { 0x0038, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2EE, { 0x0039, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xA2EF, { 0xF863, 0x005B, 0x0031, 0x0030, 0x005D, 0x0000 } },
		{ 0xA2F0, { 0xF863, 0x005B, 0x0031, 0x0031, 0x005D, 0x0000 } },
		{ 0xA2F1, { 0xF863, 0x005B, 0x0031, 0x0032, 0x005D, 0x0000 } },
		{ 0xA2F2, { 0xF863, 0x005B, 0x0031, 0x0033, 0x005D, 0x0000 } },
		{ 0xA2F3, { 0xF863, 0x005B, 0x0031, 0x0034, 0x005D, 0x0000 } },
		{ 0xA2F4, { 0xF863, 0x005B, 0x0031, 0x0035, 0x005D, 0x0000 } },
		{ 0xA2F5, { 0xF863, 0x005B, 0x0031, 0x0036, 0x005D, 0x0000 } },
		{ 0xA2F6, { 0xF863, 0x005B, 0x0031, 0x0037, 0x005D, 0x0000 } },
		{ 0xA2F7, { 0xF863, 0x005B, 0x0031, 0x0038, 0x005D, 0x0000 } },
		{ 0xA2F8, { 0xF863, 0x005B, 0x0031, 0x0039, 0x005D, 0x0000 } },
		{ 0xA2F9, { 0xF863, 0x005B, 0x0032, 0x0030, 0x005D, 0x0000 } },
		{ 0xA2FA, { 0x00B6, 0xF87F, 0x0000 } },
		{ 0xA2FB, { 0x2016, 0xF87B, 0x0000 } },
		{ 0xA2FC, { 0x2016, 0xF87C, 0x0000 } },
		{ 0xA341, { 0x0031, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA342, { 0x0032, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA343, { 0x0033, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA344, { 0x0034, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA345, { 0x0035, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA346, { 0x0036, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA347, { 0x0037, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA348, { 0x0038, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA349, { 0x0039, 0x20DE, 0xF87B, 0x0000 } },
		{ 0xA34A, { 0xF864, 0x005B, 0x0031, 0x0030, 0x005D, 0x0000 } },
		{ 0xA34B, { 0xF864, 0x005B, 0x0031, 0x0031, 0x005D, 0x0000 } },
		{ 0xA34C, { 0xF864, 0x005B, 0x0031, 0x0032, 0x005D, 0x0000 } },
		{ 0xA34D, { 0xF864, 0x005B, 0x0031, 0x0033, 0x005D, 0x0000 } },
		{ 0xA34E, { 0xF864, 0x005B, 0x0031, 0x0034, 0x005D, 0x0000 } },
		{ 0xA34F, { 0xF864, 0x005B, 0x0031, 0x0035, 0x005D, 0x0000 } },
		{ 0xA350, { 0xF864, 0x005B, 0x0031, 0x0036, 0x005D, 0x0000 } },
		{ 0xA351, { 0xF864, 0x005B, 0x0031, 0x0037, 0x005D, 0x0000 } },
		{ 0xA352, { 0xF864, 0x005B, 0x0031, 0x0038, 0x005D, 0x0000 } },
		{ 0xA353, { 0xF864, 0x005B, 0x0031, 0x0039, 0x005D, 0x0000 } },
		{ 0xA354, { 0xF864, 0x005B, 0x0032, 0x0030, 0x005D, 0x0000 } },
		{ 0xA35F, { 0x24EB, 0xF87F, 0x0000 } },
		{ 0xA360, { 0x24EC, 0xF87F, 0x0000 } },
		{ 0xA361, { 0x24ED, 0xF87F, 0x0000 } },
		{ 0xA362, { 0x24EE, 0xF87F, 0x0000 } },
		{ 0xA363, { 0x24EF, 0xF87F, 0x0000 } },
		{ 0xA364, { 0x24F0, 0xF87F, 0x0000 } },
		{ 0xA365, { 0x24F1, 0xF87F, 0x0000 } },
		{ 0xA366, { 0x24F2, 0xF87F, 0x0000 } },
		{ 0xA367, { 0x24F3, 0xF87F, 0x0000 } },
		{ 0xA368, { 0x24F4, 0xF87F, 0x0000 } },
		{ 0xA369, { 0xF861, 0x0028, 0x0041, 0x0029, 0x0000 } },
		{ 0xA36A, { 0xF861, 0x0028, 0x0042, 0x0029, 0x0000 } },
		{ 0xA36B, { 0xF861, 0x0028, 0x0043, 0x0029, 0x0000 } },
		{ 0xA36C, { 0xF861, 0x0028, 0x0044, 0x0029, 0x0000 } },
		{ 0xA36D, { 0xF861, 0x0028, 0x0045, 0x0029, 0x0000 } },
		{ 0xA36E, { 0xF861, 0x0028, 0x0046, 0x0029, 0x0000 } },
		{ 0xA36F, { 0xF861, 0x0028, 0x0047, 0x0029, 0x0000 } },
		{ 0xA370, { 0xF861, 0x0028, 0x0048, 0x0029, 0x0000 } },
		{ 0xA371, { 0xF861, 0x0028, 0x0049, 0x0029, 0x0000 } },
		{ 0xA372, { 0xF861, 0x0028, 0x004A, 0x0029, 0x0000 } },
		{ 0xA373, { 0xF861, 0x0028, 0x004B, 0x0029, 0x0000 } },
		{ 0xA374, { 0xF861, 0x0028, 0x004C, 0x0029, 0x0000 } },
		{ 0xA375, { 0xF861, 0x0028, 0x004D, 0x0029, 0x0000 } },
		{ 0xA376, { 0xF861, 0x0028, 0x004E, 0x0029, 0x0000 } },
		{ 0xA377, { 0xF861, 0x0028, 0x004F, 0x0029, 0x0000 } },
		{ 0xA378, { 0xF861, 0x0028, 0x0050, 0x0029, 0x0000 } },
		{ 0xA379, { 0xF861, 0x0028, 0x0051, 0x0029, 0x0000 } },
		{ 0xA37A, { 0xF861, 0x0028, 0x0052, 0x0029, 0x0000 } },
		{ 0xA37B, { 0xF861, 0x0028, 0x0053, 0x0029, 0x0000 } },
		{ 0xA37C, { 0xF861, 0x0028, 0x0054, 0x0029, 0x0000 } },
		{ 0xA37D, { 0xF861, 0x0028, 0x0055, 0x0029, 0x0000 } },
		{ 0xA381, { 0xF861, 0x0028, 0x0056, 0x0029, 0x0000 } },
		{ 0xA382, { 0xF861, 0x0028, 0x0057, 0x0029, 0x0000 } },
		{ 0xA383, { 0xF861, 0x0028, 0x0058, 0x0029, 0x0000 } },
		{ 0xA384, { 0xF861, 0x0028, 0x0059, 0x0029, 0x0000 } },
		{ 0xA385, { 0xF861, 0x0028, 0x005A, 0x0029, 0x0000 } },
		{ 0xA441, { 0x0031, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA442, { 0x0032, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA443, { 0x0033, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA444, { 0x0034, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA445, { 0x0035, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA446, { 0x0036, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA447, { 0x0037, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA448, { 0x0038, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA449, { 0x0039, 0x20DE, 0xF87F, 0x0000 } },
		{ 0xA44A, { 0xF862, 0x005B, 0x0031, 0x0030, 0x005D, 0x0000 } },
		{ 0xA44B, { 0xF862, 0x005B, 0x0031, 0x0031, 0x005D, 0x0000 } },
		{ 0xA44C, { 0xF862, 0x005B, 0x0031, 0x0032, 0x005D, 0x0000 } },
		{ 0xA44D, { 0xF862, 0x005B, 0x0031, 0x0033, 0x005D, 0x0000 } },
		{ 0xA44E, { 0xF862, 0x005B, 0x0031, 0x0034, 0x005D, 0x0000 } },
		{ 0xA44F, { 0xF862, 0x005B, 0x0031, 0x0035, 0x005D, 0x0000 } },
		{ 0xA450, { 0xF862, 0x005B, 0x0031, 0x0036, 0x005D, 0x0000 } },
		{ 0xA451, { 0xF862, 0x005B, 0x0031, 0x0037, 0x005D, 0x0000 } },
		{ 0xA452, { 0xF862, 0x005B, 0x0031, 0x0038, 0x005D, 0x0000 } },
		{ 0xA453, { 0xF862, 0x005B, 0x0031, 0x0039, 0x005D, 0x0000 } },
		{ 0xA454, { 0xF862, 0x005B, 0x0032, 0x0030, 0x005D, 0x0000 } },
		{ 0xA455, { 0x0031, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA456, { 0x0032, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA457, { 0x0033, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA458, { 0x0034, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA459, { 0x0035, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA45A, { 0x0036, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA45B, { 0x0037, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA45C, { 0x0038, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA45D, { 0x0039, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xA45E, { 0xF865, 0x005B, 0x0031, 0x0030, 0x005D, 0x0000 } },
		{ 0xA45F, { 0xF865, 0x005B, 0x0031, 0x0031, 0x005D, 0x0000 } },
		{ 0xA460, { 0xF865, 0x005B, 0x0031, 0x0032, 0x005D, 0x0000 } },
		{ 0xA461, { 0xF865, 0x005B, 0x0031, 0x0033, 0x005D, 0x0000 } },
		{ 0xA462, { 0xF865, 0x005B, 0x0031, 0x0034, 0x005D, 0x0000 } },
		{ 0xA463, { 0xF865, 0x005B, 0x0031, 0x0035, 0x005D, 0x0000 } },
		{ 0xA464, { 0xF865, 0x005B, 0x0031, 0x0036, 0x005D, 0x0000 } },
		{ 0xA465, { 0xF865, 0x005B, 0x0031, 0x0037, 0x005D, 0x0000 } },
		{ 0xA466, { 0xF865, 0x005B, 0x0031, 0x0038, 0x005D, 0x0000 } },
		{ 0xA467, { 0xF865, 0x005B, 0x0031, 0x0039, 0x005D, 0x0000 } },
		{ 0xA468, { 0xF865, 0x005B, 0x0032, 0x0030, 0x005D, 0x0000 } },
		{ 0xA469, { 0x278A, 0xF87F, 0x0000 } },
		{ 0xA46A, { 0x278B, 0xF87F, 0x0000 } },
		{ 0xA46B, { 0x278C, 0xF87F, 0x0000 } },
		{ 0xA46C, { 0x278D, 0xF87F, 0x0000 } },
		{ 0xA46D, { 0x278E, 0xF87F, 0x0000 } },
		{ 0xA46E, { 0x278F, 0xF87F, 0x0000 } },
		{ 0xA46F, { 0x2790, 0xF87F, 0x0000 } },
		{ 0xA470, { 0x2791, 0xF87F, 0x0000 } },
		{ 0xA471, { 0x2792, 0xF87F, 0x0000 } },
		{ 0xA472, { 0x2793, 0xF87F, 0x0000 } },
		{ 0xA473, { 0x24EB, 0xF878, 0x0000 } },
		{ 0xA474, { 0x24EC, 0xF878, 0x0000 } },
		{ 0xA475, { 0x24ED, 0xF878, 0x0000 } },
		{ 0xA476, { 0x24EE, 0xF878, 0x0000 } },
		{ 0xA477, { 0x24EF, 0xF878, 0x0000 } },
		{ 0xA478, { 0x24F0, 0xF878, 0x0000 } },
		{ 0xA479, { 0x24F1, 0xF878, 0x0000 } },
		{ 0xA47A, { 0x24F2, 0xF878, 0x0000 } },
		{ 0xA47B, { 0x24F3, 0xF878, 0x0000 } },
		{ 0xA47C, { 0x24F4, 0xF878, 0x0000 } },
		{ 0xA498, { 0x2222, 0xF87F, 0x0000 } },
		{ 0xA49E, { 0x2225, 0x0347, 0x0000 } },
		{ 0xA49F, { 0x2AFD, 0x0347, 0x0000 } },
		{ 0xA541, { 0x0030, 0x20DE, 0x0000 } },
		{ 0xA542, { 0x0031, 0x20DE, 0x0000 } },
		{ 0xA543, { 0x0032, 0x20DE, 0x0000 } },
		{ 0xA544, { 0x0033, 0x20DE, 0x0000 } },
		{ 0xA545, { 0x0034, 0x20DE, 0x0000 } },
		{ 0xA546, { 0x0035, 0x20DE, 0x0000 } },
		{ 0xA547, { 0x0036, 0x20DE, 0x0000 } },
		{ 0xA548, { 0x0037, 0x20DE, 0x0000 } },
		{ 0xA549, { 0x0038, 0x20DE, 0x0000 } },
		{ 0xA54A, { 0x0039, 0x20DE, 0x0000 } },
		{ 0xA54B, { 0x24EA, 0xF87F, 0x0000 } },
		{ 0xA54C, { 0x2460, 0xF87F, 0x0000 } },
		{ 0xA54D, { 0x2461, 0xF87F, 0x0000 } },
		{ 0xA54E, { 0x2462, 0xF87F, 0x0000 } },
		{ 0xA54F, { 0x2463, 0xF87F, 0x0000 } },
		{ 0xA550, { 0x2464, 0xF87F, 0x0000 } },
		{ 0xA551, { 0x2465, 0xF87F, 0x0000 } },
		{ 0xA552, { 0x2466, 0xF87F, 0x0000 } },
		{ 0xA553, { 0x2467, 0xF87F, 0x0000 } },
		{ 0xA554, { 0x2468, 0xF87F, 0x0000 } },
		{ 0xA555, { 0xF860, 0x0041, 0x0029, 0x0000 } },
		{ 0xA556, { 0xF860, 0x0042, 0x0029, 0x0000 } },
		{ 0xA557, { 0xF860, 0x0043, 0x0029, 0x0000 } },
		{ 0xA558, { 0xF860, 0x0044, 0x0029, 0x0000 } },
		{ 0xA559, { 0xF860, 0x0045, 0x0029, 0x0000 } },
		{ 0xA55A, { 0xF860, 0x0046, 0x0029, 0x0000 } },
		{ 0xA55B, { 0xF860, 0x0047, 0x0029, 0x0000 } },
		{ 0xA55C, { 0xF860, 0x0048, 0x0029, 0x0000 } },
		{ 0xA55D, { 0xF860, 0x0049, 0x0029, 0x0000 } },
		{ 0xA55E, { 0xF860, 0x004A, 0x0029, 0x0000 } },
		{ 0xA55F, { 0xF860, 0x004B, 0x0029, 0x0000 } },
		{ 0xA560, { 0xF860, 0x004C, 0x0029, 0x0000 } },
		{ 0xA561, { 0xF860, 0x004D, 0x0029, 0x0000 } },
		{ 0xA562, { 0xF860, 0x004E, 0x0029, 0x0000 } },
		{ 0xA563, { 0xF860, 0x004F, 0x0029, 0x0000 } },
		{ 0xA564, { 0xF860, 0x0050, 0x0029, 0x0000 } },
		{ 0xA565, { 0xF860, 0x0051, 0x0029, 0x0000 } },
		{ 0xA566, { 0xF860, 0x0052, 0x0029, 0x0000 } },
		{ 0xA567, { 0xF860, 0x0053, 0x0029, 0x0000 } },
		{ 0xA568, { 0xF860, 0x0054, 0x0029, 0x0000 } },
		{ 0xA569, { 0xF860, 0x0055, 0x0029, 0x0000 } },
		{ 0xA56A, { 0xF860, 0x0056, 0x0029, 0x0000 } },
		{ 0xA56B, { 0xF860, 0x0057, 0x0029, 0x0000 } },
		{ 0xA56C, { 0xF860, 0x0058, 0x0029, 0x0000 } },
		{ 0xA56D, { 0xF860, 0x0059, 0x0029, 0x0000 } },
		{ 0xA56E, { 0xF860, 0x005A, 0x0029, 0x0000 } },
		{ 0xA56F, { 0xF860, 0x0061, 0x0029, 0x0000 } },
		{ 0xA570, { 0xF860, 0x0062, 0x0029, 0x0000 } },
		{ 0xA571, { 0xF860, 0x0063, 0x0029, 0x0000 } },
		{ 0xA572, { 0xF860, 0x0064, 0x0029, 0x0000 } },
		{ 0xA573, { 0xF860, 0x0065, 0x0029, 0x0000 } },
		{ 0xA574, { 0xF860, 0x0066, 0x0029, 0x0000 } },
		{ 0xA575, { 0xF860, 0x0067, 0x0029, 0x0000 } },
		{ 0xA576, { 0xF860, 0x0068, 0x0029, 0x0000 } },
		{ 0xA577, { 0xF860, 0x0069, 0x0029, 0x0000 } },
		{ 0xA578, { 0xF860, 0x006A, 0x0029, 0x0000 } },
		{ 0xA579, { 0xF860, 0x006B, 0x0029, 0x0000 } },
		{ 0xA57A, { 0xF860, 0x006C, 0x0029, 0x0000 } },
		{ 0xA57B, { 0xF860, 0x006D, 0x0029, 0x0000 } },
		{ 0xA57C, { 0xF860, 0x006E, 0x0029, 0x0000 } },
		{ 0xA57D, { 0xF860, 0x006F, 0x0029, 0x0000 } },
		{ 0xA581, { 0xF860, 0x0070, 0x0029, 0x0000 } },
		{ 0xA582, { 0xF860, 0x0071, 0x0029, 0x0000 } },
		{ 0xA583, { 0xF860, 0x0072, 0x0029, 0x0000 } },
		{ 0xA584, { 0xF860, 0x0073, 0x0029, 0x0000 } },
		{ 0xA585, { 0xF860, 0x0074, 0x0029, 0x0000 } },
		{ 0xA586, { 0xF860, 0x0075, 0x0029, 0x0000 } },
		{ 0xA587, { 0xF860, 0x0076, 0x0029, 0x0000 } },
		{ 0xA588, { 0xF860, 0x0077, 0x0029, 0x0000 } },
		{ 0xA589, { 0xF860, 0x0078, 0x0029, 0x0000 } },
		{ 0xA58A, { 0xF860, 0x0079, 0x0029, 0x0000 } },
		{ 0xA58B, { 0xF860, 0x007A, 0x0029, 0x0000 } },
		{ 0xA5DA, { 0xFF01, 0xF874, 0x0000 } },
		{ 0xA5DB, { 0x3002, 0xF87D, 0x0000 } },
		{ 0xA5DC, { 0x2032, 0xF87F, 0x0000 } },
		{ 0xA5DD, { 0x2033, 0xF87F, 0x0000 } },
		{ 0xA5F9, { 0x3257, 0xF87A, 0x0000 } },
		{ 0xA5FA, { 0x3258, 0xF87A, 0x0000 } },
		{ 0xA5FB, { 0x3259, 0xF87A, 0x0000 } },
		{ 0xA5FC, { 0x325A, 0xF87A, 0x0000 } },
		{ 0xA641, { 0xF83D, 0xF87F, 0x0000 } },
		{ 0xA643, { 0x2020, 0xF87C, 0x0000 } },
		{ 0xA644, { 0xF860, 0x2020, 0x2020, 0x0000 } },
		{ 0xA645, { 0xF860, 0x2021, 0x2021, 0x0000 } },
		{ 0xA646, { 0xF861, 0x2020, 0x2020, 0x2020, 0x0000 } },
		{ 0xA647, { 0x00A7, 0xF87C, 0x0000 } },
		{ 0xA649, { 0xFF0A, 0xF87F, 0x0000 } },
		{ 0xA64A, { 0xFF0A, 0xF873, 0x0000 } },
		{ 0xA64B, { 0x2051, 0xF874, 0x0000 } },
		{ 0xA64C, { 0xF860, 0x002A, 0x002A, 0x0000 } },
		{ 0xA64F, { 0x2051, 0xF871, 0x0000 } },
		{ 0xA650, { 0xF867, 0x002A, 0x002A, 0x0000 } },
		{ 0xA651, { 0x2042, 0xF879, 0x0000 } },
		{ 0xA656, { 0x25A0, 0x20DF, 0x0000 } },
		{ 0xA657, { 0x25C7, 0x20DF, 0x0000 } },
		{ 0xA659, { 0x25A1, 0x20DF, 0x0000 } },
		{ 0xA65E, { 0x261C, 0xF87F, 0x0000 } },
		{ 0xA65F, { 0x261E, 0xF87F, 0x0000 } },
		{ 0xA660, { 0xF806, 0x20DF, 0x0000 } },
		{ 0xA661, { 0x25C7, 0x20DF, 0x20DF, 0x0000 } },
		{ 0xA662, { 0x25C7, 0x20DE, 0x0000 } },
		{ 0xA665, { 0x25C6, 0x20DE, 0x0000 } },
		{ 0xA666, { 0xF805, 0x20DE, 0x0000 } },
		{ 0xA667, { 0x29C8, 0x20DE, 0x0000 } },
		{ 0xA669, { 0x25CE, 0x20DD, 0x0000 } },
		{ 0xA66A, { 0x25B3, 0x20DD, 0x0000 } },
		{ 0xA66B, { 0x25B2, 0x20DD, 0x0000 } },
		{ 0xA670, { 0x25EF, 0xF87C, 0x0000 } },
		{ 0xA671, { 0x2610, 0xF87C, 0x0000 } },
		{ 0xA676, { 0x2610, 0xF87F, 0x0000 } },
		{ 0xA679, { 0x2723, 0xF87A, 0x0000 } },
		{ 0xA67A, { 0x2756, 0xF87A, 0x0000 } },
		{ 0xA67B, { 0x273F, 0xF87A, 0x0000 } },
		{ 0xA67D, { 0x3013, 0xF87C, 0x0000 } },
		{ 0xA682, { 0x25C9, 0x20DD, 0x0000 } },
		{ 0xA685, { 0x27E1, 0x20DD, 0x0000 } },
		{ 0xA686, { 0xF80B, 0xF87F, 0x0000 } },
		{ 0xA687, { 0x2720, 0xF87A, 0x0000 } },
		{ 0xA689, { 0x25C8, 0xF87F, 0x0000 } },
		{ 0xA68A, { 0x25A8, 0xF87F, 0x0000 } },
		{ 0xA68E, { 0x2756, 0xF87F, 0x0000 } },
		{ 0xA690, { 0x20A9, 0xF87F, 0x0000 } },
		{ 0xA691, { 0xF809, 0xF87A, 0x0000 } },
		{ 0xA692, { 0x534D, 0xF87F, 0x0000 } },
		{ 0xA697, { 0x262F, 0xF87A, 0x0000 } },
		{ 0xA698, { 0x262F, 0xF876, 0x0000 } },
		{ 0xA69B, { 0x2748, 0x20D8, 0x0000 } },
		{ 0xA6D9, { 0xFF0C, 0xF87E, 0x0000 } },
		{ 0xA6DA, { 0x3002, 0xF87E, 0x0000 } },
		{ 0xA6DB, { 0x3001, 0xF87E, 0x0000 } },
		{ 0xA6DC, { 0xFF1A, 0xF87E, 0x0000 } },
		{ 0xA6DD, { 0xFF1B, 0xF87E, 0x0000 } },
		{ 0xA6DE, { 0xFF01, 0xF87E, 0x0000 } },
		{ 0xA6DF, { 0xFF1F, 0xF87E, 0x0000 } },
		{ 0xA6EC, { 0x3016, 0xF87E, 0x0000 } },
		{ 0xA6ED, { 0x3017, 0xF87E, 0x0000 } },
		{ 0xA6F3, { 0x22EF, 0xF87E, 0x0000 } },
		{ 0xA6F9, { 0x3251, 0xF87A, 0x0000 } },
		{ 0xA6FA, { 0x3252, 0xF87A, 0x0000 } },
		{ 0xA6FB, { 0x3253, 0xF87A, 0x0000 } },
		{ 0xA6FC, { 0x3254, 0xF87A, 0x0000 } },
		{ 0xA6FD, { 0x3255, 0xF87A, 0x0000 } },
		{ 0xA6FE, { 0x3256, 0xF87A, 0x0000 } },
		{ 0xA741, { 0x2642, 0xF87F, 0x0000 } },
		{ 0xA744, { 0x25CB, 0xF87F, 0x0000 } },
		{ 0xA745, { 0x25B3, 0xF87F, 0x0000 } },
		{ 0xA748, { 0x2394, 0xF876, 0x0000 } },
		{ 0xA749, { 0x25AD, 0xF878, 0x0000 } },
		{ 0xA74E, { 0x25C7, 0xF87F, 0x0000 } },
		{ 0xA74F, { 0x51F9, 0xF87F, 0x0000 } },
		{ 0xA750, { 0x51F8, 0xF87F, 0x0000 } },
		{ 0xA752, { 0x2206, 0xF87F, 0x0000 } },
		{ 0xA754, { 0x222A, 0xF87F, 0x0000 } },
		{ 0xA757, { 0x2229, 0xF87F, 0x0000 } },
		{ 0xA762, { 0x003D, 0x20E5, 0x0000 } },
		{ 0xA763, { 0x2261, 0x20E5, 0x0000 } },
		{ 0xA765, { 0x003D, 0x20D2, 0x0000 } },
		{ 0xA767, { 0x002D, 0x0308, 0x0000 } },
		{ 0xA76D, { 0x22A5, 0x0338, 0x0000 } },
		{ 0xA76E, { 0x2261, 0x20D2, 0x0000 } },
		{ 0xA779, { 0x223D, 0x0336, 0x0000 } },
		{ 0xA77A, { 0x2314, 0xF87F, 0x0000 } },
		{ 0xA77C, { 0x88DC, 0x20E4, 0x0000 } },
		{ 0xA77D, { 0xF862, 0xC8FC, 0xC2DD, 0xD68C, 0xC0AC, 0x0000 } },
		{ 0xA781, { 0xF863, 0xC8FC, 0xC2DD, 0xD68C, 0xC0AC, 0x0000 } },
		{ 0xA783, { 0x329E, 0xF87F, 0x0000 } },
		{ 0xA786, { 0x203C, 0xF87F, 0x0000 } },
		{ 0xA788, { 0x25C7, 0xF87C, 0x0000 } },
		{ 0xA789, { 0x25C7, 0xF879, 0x0000 } },
		{ 0xA78A, { 0x25C7, 0xF87B, 0x0000 } },
		{ 0xA78B, { 0x25C6, 0xF879, 0x0000 } },
		{ 0xA78C, { 0x25A1, 0xF87C, 0x0000 } },
		{ 0xA78D, { 0x25A1, 0xF879, 0x0000 } },
		{ 0xA78E, { 0x25A1, 0xF87B, 0x0000 } },
		{ 0xA791, { 0x25CB, 0xF879, 0x0000 } },
		{ 0xA792, { 0x25CB, 0xF87B, 0x0000 } },
		{ 0xA793, { 0x25CF, 0xF879, 0x0000 } },
		{ 0xA79B, { 0x25B4, 0x20E4, 0x0000 } },
		{ 0xA841, { 0x2192, 0xF87B, 0x0000 } },
		{ 0xA842, { 0x2190, 0xF87B, 0x0000 } },
		{ 0xA843, { 0x2191, 0xF87B, 0x0000 } },
		{ 0xA844, { 0x2193, 0xF87B, 0x0000 } },
		{ 0xA845, { 0x2196, 0xF87B, 0x0000 } },
		{ 0xA846, { 0x2197, 0xF87B, 0x0000 } },
		{ 0xA847, { 0x2198, 0xF87B, 0x0000 } },
		{ 0xA848, { 0x2199, 0xF87B, 0x0000 } },
		{ 0xA84C, { 0x21D4, 0xF87F, 0x0000 } },
		{ 0xA84D, { 0x2192, 0xF87C, 0x0000 } },
		{ 0xA84E, { 0x2190, 0xF87C, 0x0000 } },
		{ 0xA84F, { 0x2191, 0xF87C, 0x0000 } },
		{ 0xA850, { 0x2193, 0xF87C, 0x0000 } },
		{ 0xA851, { 0x2194, 0xF87C, 0x0000 } },
		{ 0xA852, { 0x2195, 0xF87C, 0x0000 } },
		{ 0xA853, { 0x2190, 0xF879, 0x0000 } },
		{ 0xA854, { 0x2192, 0xF879, 0x0000 } },
		{ 0xA855, { 0x2191, 0xF879, 0x0000 } },
		{ 0xA856, { 0x2193, 0xF879, 0x0000 } },
		{ 0xA857, { 0x21E6, 0x20DE, 0x0000 } },
		{ 0xA858, { 0x21E8, 0x20DE, 0x0000 } },
		{ 0xA859, { 0x21E7, 0x20DE, 0x0000 } },
		{ 0xA85A, { 0x21E9, 0x20DE, 0x0000 } },
		{ 0xA85B, { 0x21E6, 0x20DD, 0x0000 } },
		{ 0xA85D, { 0x21E7, 0x20DD, 0x0000 } },
		{ 0xA85E, { 0x21E9, 0x20DD, 0x0000 } },
		{ 0xA85F, { 0x2190, 0xF87F, 0x0000 } },
		{ 0xA861, { 0x2191, 0xF87F, 0x0000 } },
		{ 0xA862, { 0x2193, 0xF87F, 0x0000 } },
		{ 0xA863, { 0x2190, 0xF875, 0x0000 } },
		{ 0xA864, { 0x2192, 0xF875, 0x0000 } },
		{ 0xA865, { 0x2191, 0xF875, 0x0000 } },
		{ 0xA866, { 0x2193, 0xF875, 0x0000 } },
		{ 0xA869, { 0x2190, 0xF871, 0x0000 } },
		{ 0xA86B, { 0x2190, 0xF872, 0x0000 } },
		{ 0xA86C, { 0x2192, 0xF872, 0x0000 } },
		{ 0xA86D, { 0x2191, 0xF872, 0x0000 } },
		{ 0xA86E, { 0x2193, 0xF872, 0x0000 } },
		{ 0xA873, { 0x21E6, 0xF87A, 0x0000 } },
		{ 0xA875, { 0x21E7, 0xF87A, 0x0000 } },
		{ 0xA876, { 0x21E9, 0xF87A, 0x0000 } },
		{ 0xA877, { 0x21E6, 0xF87B, 0x0000 } },
		{ 0xA879, { 0x21E7, 0xF87B, 0x0000 } },
		{ 0xA87A, { 0x21E9, 0xF87B, 0x0000 } },
		{ 0xA87D, { 0x21BB, 0xF87B, 0x0000 } },
		{ 0xA884, { 0x2939, 0xF87F, 0x0000 } },
		{ 0xA885, { 0x2934, 0xF87F, 0x0000 } },
		{ 0xA887, { 0x21B1, 0xF87F, 0x0000 } },
		{ 0xA888, { 0x21BB, 0xF87F, 0x0000 } },
		{ 0xA88A, { 0x21B0, 0xF87F, 0x0000 } },
		{ 0xA88E, { 0x21E6, 0xF879, 0x0000 } },
		{ 0xA88F, { 0x21E8, 0xF879, 0x0000 } },
		{ 0xA890, { 0x21E7, 0xF879, 0x0000 } },
		{ 0xA891, { 0x21E9, 0xF879, 0x0000 } },
		{ 0xA895, { 0x21D4, 0xF879, 0x0000 } },
		{ 0xA896, { 0x21E8, 0xF874, 0x0000 } },
		{ 0xA897, { 0x21E6, 0xF874, 0x0000 } },
		{ 0xA898, { 0x21C0, 0xF879, 0x0000 } },
		{ 0xA899, { 0x21BC, 0xF879, 0x0000 } },
		{ 0xA89A, { 0x21D2, 0xF87C, 0x0000 } },
		{ 0xA89B, { 0x21D0, 0xF87C, 0x0000 } },
		{ 0xA941, { 0xF860, 0x0041, 0x002E, 0x0000 } },
		{ 0xA942, { 0xF860, 0x0042, 0x002E, 0x0000 } },
		{ 0xA943, { 0xF860, 0x0043, 0x002E, 0x0000 } },
		{ 0xA944, { 0xF860, 0x0044, 0x002E, 0x0000 } },
		{ 0xA945, { 0xF860, 0x0045, 0x002E, 0x0000 } },
		{ 0xA946, { 0xF860, 0x0046, 0x002E, 0x0000 } },
		{ 0xA947, { 0xF860, 0x0047, 0x002E, 0x0000 } },
		{ 0xA948, { 0xF860, 0x0048, 0x002E, 0x0000 } },
		{ 0xA949, { 0xF860, 0x0049, 0x002E, 0x0000 } },
		{ 0xA94A, { 0xF860, 0x004A, 0x002E, 0x0000 } },
		{ 0xA94B, { 0xF860, 0x004B, 0x002E, 0x0000 } },
		{ 0xA94C, { 0xF860, 0x004C, 0x002E, 0x0000 } },
		{ 0xA94D, { 0xF860, 0x004D, 0x002E, 0x0000 } },
		{ 0xA94E, { 0xF860, 0x004E, 0x002E, 0x0000 } },
		{ 0xA94F, { 0xF860, 0x004F, 0x002E, 0x0000 } },
		{ 0xA950, { 0xF860, 0x0050, 0x002E, 0x0000 } },
		{ 0xA951, { 0xF860, 0x0051, 0x002E, 0x0000 } },
		{ 0xA952, { 0xF860, 0x0052, 0x002E, 0x0000 } },
		{ 0xA953, { 0xF860, 0x0053, 0x002E, 0x0000 } },
		{ 0xA954, { 0xF860, 0x0054, 0x002E, 0x0000 } },
		{ 0xA955, { 0xF860, 0x0055, 0x002E, 0x0000 } },
		{ 0xA956, { 0xF860, 0x0056, 0x002E, 0x0000 } },
		{ 0xA957, { 0xF860, 0x0057, 0x002E, 0x0000 } },
		{ 0xA958, { 0xF860, 0x0058, 0x002E, 0x0000 } },
		{ 0xA959, { 0xF860, 0x0059, 0x002E, 0x0000 } },
		{ 0xA95A, { 0xF860, 0x005A, 0x002E, 0x0000 } },
		{ 0xA95B, { 0xF860, 0x0061, 0x002E, 0x0000 } },
		{ 0xA95C, { 0xF860, 0x0062, 0x002E, 0x0000 } },
		{ 0xA95D, { 0xF860, 0x0063, 0x002E, 0x0000 } },
		{ 0xA95E, { 0xF860, 0x0064, 0x002E, 0x0000 } },
		{ 0xA95F, { 0xF860, 0x0065, 0x002E, 0x0000 } },
		{ 0xA960, { 0xF860, 0x0066, 0x002E, 0x0000 } },
		{ 0xA961, { 0xF860, 0x0067, 0x002E, 0x0000 } },
		{ 0xA962, { 0xF860, 0x0068, 0x002E, 0x0000 } },
		{ 0xA963, { 0xF860, 0x0069, 0x002E, 0x0000 } },
		{ 0xA964, { 0xF860, 0x006A, 0x002E, 0x0000 } },
		{ 0xA965, { 0xF860, 0x006B, 0x002E, 0x0000 } },
		{ 0xA966, { 0xF860, 0x006C, 0x002E, 0x0000 } },
		{ 0xA967, { 0xF860, 0x006D, 0x002E, 0x0000 } },
		{ 0xA968, { 0xF860, 0x006E, 0x002E, 0x0000 } },
		{ 0xA969, { 0xF860, 0x006F, 0x002E, 0x0000 } },
		{ 0xA96A, { 0xF860, 0x0070, 0x002E, 0x0000 } },
		{ 0xA96B, { 0xF860, 0x0071, 0x002E, 0x0000 } },
		{ 0xA96C, { 0xF860, 0x0072, 0x002E, 0x0000 } },
		{ 0xA96D, { 0xF860, 0x0073, 0x002E, 0x0000 } },
		{ 0xA96E, { 0xF860, 0x0074, 0x002E, 0x0000 } },
		{ 0xA96F, { 0xF860, 0x0075, 0x002E, 0x0000 } },
		{ 0xA970, { 0xF860, 0x0076, 0x002E, 0x0000 } },
		{ 0xA971, { 0xF860, 0x0077, 0x002E, 0x0000 } },
		{ 0xA972, { 0xF860, 0x0078, 0x002E, 0x0000 } },
		{ 0xA973, { 0xF860, 0x0079, 0x002E, 0x0000 } },
		{ 0xA974, { 0xF860, 0x007A, 0x002E, 0x0000 } },
		{ 0xAA41, { 0xC6B4, 0x20DE, 0x0000 } },
		{ 0xAA42, { 0xB2F5, 0x20DE, 0x0000 } },
		{ 0xAA43, { 0xC8FC, 0x20DE, 0x0000 } },
		{ 0xAA44, { 0xBA85, 0x20DE, 0x0000 } },
		{ 0xAA45, { 0xB300, 0x20DE, 0x0000 } },
		{ 0xAA46, { 0xD615, 0x20DE, 0x0000 } },
		{ 0xAA47, { 0xBD80, 0x20DE, 0x0000 } },
		{ 0xAA48, { 0xC804, 0x20DE, 0x0000 } },
		{ 0xAA49, { 0xC811, 0x20DE, 0x0000 } },
		{ 0xAA4A, { 0xC218, 0x20DE, 0x0000 } },
		{ 0xAA4B, { 0xB3D9, 0x20DE, 0x0000 } },
		{ 0xAA4C, { 0xBE44, 0x20DE, 0x0000 } },
		{ 0xAA4D, { 0xBC18, 0x20DE, 0x0000 } },
		{ 0xAA4E, { 0xC790, 0x20DE, 0x0000 } },
		{ 0xAA4F, { 0xD0C0, 0x20DE, 0x0000 } },
		{ 0xAA50, { 0xAC10, 0x20DE, 0x0000 } },
		{ 0xAA51, { 0xC57D, 0x20DE, 0x0000 } },
		{ 0xAA52, { 0xC778, 0x20DE, 0x0000 } },
		{ 0xAA53, { 0xB73B, 0x20DE, 0x0000 } },
		{ 0xAA54, { 0x5370, 0x20DE, 0x0000 } },
		{ 0xAA55, { 0x8A3B, 0x20DE, 0x0000 } },
		{ 0xAA56, { 0xC608, 0x20DE, 0x0000 } },
		{ 0xAA57, { 0x611F, 0x20DE, 0x0000 } },
		{ 0xAA58, { 0x51A0, 0x20DE, 0x0000 } },
		{ 0xAA59, { 0x7B54, 0x20DE, 0x0000 } },
		{ 0xAA5A, { 0x4EE3, 0x20DE, 0x0000 } },
		{ 0xAA5B, { 0x982D, 0x20DE, 0x0000 } },
		{ 0xAA5C, { 0x52D5, 0x20DE, 0x0000 } },
		{ 0xAA5D, { 0x540D, 0x20DE, 0x0000 } },
		{ 0xAA5E, { 0x76EE, 0x20DE, 0x0000 } },
		{ 0xAA5F, { 0x53CD, 0x20DE, 0x0000 } },
		{ 0xAA60, { 0x88DC, 0x20DE, 0x0000 } },
		{ 0xAA61, { 0x672C, 0x20DE, 0x0000 } },
		{ 0xAA62, { 0x526F, 0x20DE, 0x0000 } },
		{ 0xAA63, { 0x5E8F, 0x20DE, 0x0000 } },
		{ 0xAA64, { 0x9023, 0x20DE, 0x0000 } },
		{ 0xAA65, { 0x5F71, 0x20DE, 0x0000 } },
		{ 0xAA66, { 0x4F8B, 0x20DE, 0x0000 } },
		{ 0xAA67, { 0x6E90, 0x20DE, 0x0000 } },
		{ 0xAA68, { 0x5B50, 0x20DE, 0x0000 } },
		{ 0xAA69, { 0x524D, 0x20DE, 0x0000 } },
		{ 0xAA6A, { 0x7BC0, 0x20DE, 0x0000 } },
		{ 0xAA6B, { 0x63A5, 0x20DE, 0x0000 } },
		{ 0xAA6C, { 0x52A9, 0x20DE, 0x0000 } },
		{ 0xAA6D, { 0x6307, 0x20DE, 0x0000 } },
		{ 0xAA6E, { 0x4ED6, 0x20DE, 0x0000 } },
		{ 0xAA6F, { 0x6D3E, 0x20DE, 0x0000 } },
		{ 0xAA70, { 0x5F62, 0x20DE, 0x0000 } },
		{ 0xAA71, { 0xC870, 0x20DE, 0x0000 } },
		{ 0xAA72, { 0xBB38, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA73, { 0xB2F5, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA74, { 0xC8FC, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA75, { 0xB73B, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA76, { 0x8A3B, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA77, { 0xAD50, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA78, { 0xC5ED, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA79, { 0xC74C, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA7A, { 0xC815, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA7B, { 0xD574, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA7C, { 0xC608, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAA7D, { 0xC874, 0x20DD, 0x0000 } },
		{ 0xAA81, { 0xB77C, 0x20DD, 0x0000 } },
		{ 0xAA82, { 0xB9C8, 0x20DD, 0x0000 } },
		{ 0xAA83, { 0xBC14, 0x20DD, 0x0000 } },
		{ 0xAA84, { 0xC0AC, 0x20DD, 0x0000 } },
		{ 0xAA85, { 0xC544, 0x20DD, 0x0000 } },
		{ 0xAA86, { 0xC790, 0x20DD, 0x0000 } },
		{ 0xAA87, { 0xCC28, 0x20DD, 0x0000 } },
		{ 0xAA88, { 0xCE74, 0x20DD, 0x0000 } },
		{ 0xAA89, { 0xD0C0, 0x20DD, 0x0000 } },
		{ 0xAA8A, { 0xD30C, 0x20DD, 0x0000 } },
		{ 0xAA8B, { 0xB192, 0x20DD, 0x0000 } },
		{ 0xAA8C, { 0xB0AE, 0x20DD, 0x0000 } },
		{ 0xAA8D, { 0xBA85, 0x20DD, 0x0000 } },
		{ 0xAA8E, { 0xB300, 0x20DD, 0x0000 } },
		{ 0xAA8F, { 0xD615, 0x20DD, 0x0000 } },
		{ 0xAA90, { 0xBD80, 0x20DD, 0x0000 } },
		{ 0xAA91, { 0xC804, 0x20DD, 0x0000 } },
		{ 0xAA92, { 0xC811, 0x20DD, 0x0000 } },
		{ 0xAA93, { 0xC218, 0x20DD, 0x0000 } },
		{ 0xAA94, { 0xB3D9, 0x20DD, 0x0000 } },
		{ 0xAA95, { 0xBE44, 0x20DD, 0x0000 } },
		{ 0xAA96, { 0xAC8C, 0x20DD, 0x0000 } },
		{ 0xAA97, { 0xBC18, 0x20DD, 0x0000 } },
		{ 0xAA98, { 0xC18D, 0x20DD, 0x0000 } },
		{ 0xAA99, { 0xC778, 0x20DD, 0x0000 } },
		{ 0xAA9A, { 0xBCF8, 0x20DD, 0x0000 } },
		{ 0xAA9B, { 0xC57D, 0x20DD, 0x0000 } },
		{ 0xAA9C, { 0xC219, 0x20DD, 0x0000 } },
		{ 0xAA9D, { 0xC720, 0x20DD, 0x0000 } },
		{ 0xAA9E, { 0xAD00, 0x20DD, 0x0000 } },
		{ 0xAA9F, { 0x51A0, 0x20DD, 0x0000 } },
		{ 0xAAF9, { 0xF862, 0x0028, 0x0032, 0x0031, 0x0029, 0x0000 } },
		{ 0xAAFA, { 0xF862, 0x0028, 0x0032, 0x0032, 0x0029, 0x0000 } },
		{ 0xAAFB, { 0xF862, 0x0028, 0x0032, 0x0033, 0x0029, 0x0000 } },
		{ 0xAAFC, { 0xF862, 0x0028, 0x0032, 0x0034, 0x0029, 0x0000 } },
		{ 0xAAFD, { 0xF862, 0x0028, 0x0032, 0x0035, 0x0029, 0x0000 } },
		{ 0xAAFE, { 0xF862, 0x0028, 0x0032, 0x0036, 0x0029, 0x0000 } },
		{ 0xAB41, { 0xC870, 0x20DD, 0x0000 } },
		{ 0xAB42, { 0xAD6D, 0x20DD, 0x0000 } },
		{ 0xAB43, { 0xAC10, 0x20DD, 0x0000 } },
		{ 0xAB44, { 0x5370, 0x20DD, 0x0000 } },
		{ 0xAB45, { 0x8863, 0x20DD, 0x0000 } },
		{ 0xAB46, { 0x672B, 0x20DD, 0x0000 } },
		{ 0xAB47, { 0xAC70, 0x20DD, 0x0000 } },
		{ 0xAB48, { 0xB2F5, 0x20DD, 0x0000 } },
		{ 0xAB49, { 0xBCC0, 0x20DD, 0x0000 } },
		{ 0xAB4A, { 0xC0C1, 0x20DD, 0x0000 } },
		{ 0xAB4B, { 0xC13C, 0x20DD, 0x0000 } },
		{ 0xAB4C, { 0xC2E0, 0x20DD, 0x0000 } },
		{ 0xAB4D, { 0xC5EC, 0x20DD, 0x0000 } },
		{ 0xAB4E, { 0xC608, 0x20DD, 0x0000 } },
		{ 0xAB4F, { 0xC6D0, 0x20DD, 0x0000 } },
		{ 0xAB50, { 0xC791, 0x20DD, 0x0000 } },
		{ 0xAB51, { 0xC900, 0x20DD, 0x0000 } },
		{ 0xAB52, { 0xD0B9, 0x20DD, 0x0000 } },
		{ 0xAB53, { 0xC678, 0x20DD, 0x0000 } },
		{ 0xAB54, { 0xD65C, 0x20DD, 0x0000 } },
		{ 0xAB55, { 0xAC04, 0x20DD, 0x0000 } },
		{ 0xAB56, { 0xAC19, 0x20DD, 0x0000 } },
		{ 0xAB57, { 0xC2E4, 0x20DD, 0x0000 } },
		{ 0xAB58, { 0x611F, 0x20DD, 0x0000 } },
		{ 0xAB59, { 0x6163, 0x20DD, 0x0000 } },
		{ 0xAB5A, { 0x4EE3, 0x20DD, 0x0000 } },
		{ 0xAB5B, { 0x52D5, 0x20DD, 0x0000 } },
		{ 0xAB5D, { 0x53CD, 0x20DD, 0x0000 } },
		{ 0xAB5E, { 0x526F, 0x20DD, 0x0000 } },
		{ 0xAB5F, { 0x81EA, 0x20DD, 0x0000 } },
		{ 0xAB60, { 0x524D, 0x20DD, 0x0000 } },
		{ 0xAB61, { 0x96FB, 0x20DD, 0x0000 } },
		{ 0xAB62, { 0x63A5, 0x20DD, 0x0000 } },
		{ 0xAB63, { 0x52A9, 0x20DD, 0x0000 } },
		{ 0xAB64, { 0x6CE8, 0x20DD, 0x0000 } },
		{ 0xAB65, { 0x53C3, 0x20DD, 0x0000 } },
		{ 0xAB66, { 0x672C, 0x20DD, 0x0000 } },
		{ 0xAB67, { 0x65B0, 0x20DD, 0x0000 } },
		{ 0xAB68, { 0x73FE, 0x20DD, 0x0000 } },
		{ 0xAB69, { 0x5F62, 0x20DD, 0x0000 } },
		{ 0xAB6A, { 0x9593, 0x20DD, 0x0000 } },
		{ 0xAB6B, { 0x570B, 0x20DD, 0x0000 } },
		{ 0xAB6D, { 0x4ED6, 0x20DD, 0x0000 } },
		{ 0xAB6E, { 0xBE60, 0x20DD, 0x0000 } },
		{ 0xAB6F, { 0xC2DC, 0x20DD, 0x0000 } },
		{ 0xAB70, { 0xC785, 0x20DD, 0x0000 } },
		{ 0xAB71, { 0xC73C, 0x20DD, 0x0000 } },
		{ 0xAB72, { 0xC74C, 0x20DD, 0x0000 } },
		{ 0xAB73, { 0xC9C1, 0x20DD, 0x0000 } },
		{ 0xAB74, { 0xD45C, 0x20DD, 0x0000 } },
		{ 0xAB75, { 0xAC00, 0x20DD, 0x0000 } },
		{ 0xAB76, { 0xB098, 0x20DD, 0x0000 } },
		{ 0xAB77, { 0xB2E4, 0x20DD, 0x0000 } },
		{ 0xAB78, { 0xD558, 0x20DD, 0x0000 } },
		{ 0xAB79, { 0xB9C8, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB7A, { 0xBC14, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB7B, { 0xC0AC, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB7C, { 0xC544, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB7D, { 0xC790, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB81, { 0xCC28, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB82, { 0xCE74, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB83, { 0xD0C0, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB84, { 0xD30C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB85, { 0xD558, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB86, { 0xBE44, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB87, { 0xB2F5, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB88, { 0xBE60, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB89, { 0xBCF8, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB8A, { 0xB2E8, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB8B, { 0xC13C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB8C, { 0xC2DC, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB8D, { 0xC5EC, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB8E, { 0xC608, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB8F, { 0xC73C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB90, { 0xC74C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB91, { 0xC785, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB92, { 0xC81C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB93, { 0xC874, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB94, { 0xC900, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB95, { 0xD45C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB96, { 0xD574, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB97, { 0xB290, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB98, { 0xB192, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB99, { 0xB0AE, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB9A, { 0xBC18, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB9B, { 0xAC00, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB9C, { 0xB098, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB9D, { 0xB2E4, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB9E, { 0xB77C, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xAB9F, { 0xC678, 0x20DD, 0xF87A, 0x0000 } },
		{ 0xABF7, { 0xF862, 0x0028, 0x0032, 0x0037, 0x0029, 0x0000 } },
		{ 0xABF8, { 0xF862, 0x0028, 0x0032, 0x0038, 0x0029, 0x0000 } },
		{ 0xABF9, { 0xF862, 0x0028, 0x0032, 0x0039, 0x0029, 0x0000 } },
		{ 0xABFA, { 0xF862, 0x0028, 0x0033, 0x0030, 0x0029, 0x0000 } },
		{ 0xAC44, { 0x2964, 0xF87F, 0x0000 } },
		{ 0xAC45, { 0x2962, 0xF87F, 0x0000 } },
		{ 0xAC46, { 0x21E8, 0xF870, 0x0000 } },
		{ 0xAC47, { 0x21E6, 0xF870, 0x0000 } },
		{ 0xAC4C, { 0x21C0, 0xF87F, 0x0000 } },
		{ 0xAC4D, { 0x21BC, 0xF87F, 0x0000 } },
		{ 0xAC4E, { 0x21E8, 0xF87F, 0x0000 } },
		{ 0xAC4F, { 0x21E6, 0xF87F, 0x0000 } },
		{ 0xAC55, { 0x2190, 0xF87A, 0x0000 } },
		{ 0xAC56, { 0x2192, 0xF87A, 0x0000 } },
		{ 0xAC57, { 0x2191, 0xF87A, 0x0000 } },
		{ 0xAC58, { 0x2193, 0xF87A, 0x0000 } },
		{ 0xAC59, { 0x21E6, 0xF87C, 0x0000 } },
		{ 0xAC5A, { 0x21E8, 0xF87C, 0x0000 } },
		{ 0xAC5B, { 0x21E7, 0xF87C, 0x0000 } },
		{ 0xAC5C, { 0x21E9, 0xF87C, 0x0000 } },
		{ 0xAC5D, { 0x2190, 0xF873, 0x0000 } },
		{ 0xAC60, { 0x2191, 0xF873, 0x0000 } },
		{ 0xAC61, { 0x2193, 0xF873, 0x0000 } },
		{ 0xAC62, { 0x2190, 0xF878, 0x0000 } },
		{ 0xAC63, { 0x2192, 0xF878, 0x0000 } },
		{ 0xAC64, { 0x2191, 0xF878, 0x0000 } },
		{ 0xAC65, { 0x2193, 0xF878, 0x0000 } },
		{ 0xAC66, { 0x2190, 0xF874, 0x0000 } },
		{ 0xAC67, { 0x2192, 0xF874, 0x0000 } },
		{ 0xAC68, { 0x2191, 0xF874, 0x0000 } },
		{ 0xAC69, { 0x2193, 0xF874, 0x0000 } },
		{ 0xAC6E, { 0x21E6, 0xF875, 0x0000 } },
		{ 0xAC6F, { 0x21E8, 0xF875, 0x0000 } },
		{ 0xAC70, { 0x21E7, 0xF875, 0x0000 } },
		{ 0xAC71, { 0x21E9, 0xF875, 0x0000 } },
		{ 0xAC76, { 0x2936, 0xF87A, 0x0000 } },
		{ 0xAC77, { 0x21B1, 0xF87A, 0x0000 } },
		{ 0xAC78, { 0x21BB, 0xF87A, 0x0000 } },
		{ 0xAC79, { 0x2935, 0xF87A, 0x0000 } },
		{ 0xAC7A, { 0x21B0, 0xF87A, 0x0000 } },
		{ 0xAC7B, { 0x2937, 0xF87A, 0x0000 } },
		{ 0xAC7C, { 0x2939, 0xF87A, 0x0000 } },
		{ 0xAC7D, { 0x2934, 0xF87A, 0x0000 } },
		{ 0xAC81, { 0x2936, 0xF87C, 0x0000 } },
		{ 0xAC82, { 0x21B1, 0xF87C, 0x0000 } },
		{ 0xAC83, { 0x21BB, 0xF87C, 0x0000 } },
		{ 0xAC84, { 0x2935, 0xF87C, 0x0000 } },
		{ 0xAC85, { 0x21B0, 0xF87C, 0x0000 } },
		{ 0xAC86, { 0x2937, 0xF87C, 0x0000 } },
		{ 0xAC87, { 0x2939, 0xF87C, 0x0000 } },
		{ 0xAC88, { 0x2934, 0xF87C, 0x0000 } },
		{ 0xAC89, { 0x2190, 0xF870, 0x0000 } },
		{ 0xAC8A, { 0x2192, 0xF870, 0x0000 } },
		{ 0xAC8B, { 0x2191, 0xF870, 0x0000 } },
		{ 0xAC8C, { 0x2193, 0xF870, 0x0000 } },
		{ 0xAC8F, { 0x261D, 0xF87F, 0x0000 } },
		{ 0xAC90, { 0x261F, 0xF87F, 0x0000 } },
		{ 0xAC91, { 0xB2E8, 0x20DD, 0x0000 } },
		{ 0xAC92, { 0xCC38, 0x20DD, 0x0000 } },
		{ 0xAC93, { 0xC18C, 0x20DD, 0x0000 } },
		{ 0xAC94, { 0xC911, 0x20DD, 0x0000 } },
		{ 0xAC95, { 0xC77C, 0x20DD, 0x0000 } },
		{ 0xAC96, { 0xC774, 0x20DD, 0x0000 } },
		{ 0xAC97, { 0xD734, 0x20DD, 0x0000 } },
		{ 0xACC2, { 0x0031, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC3, { 0x0032, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC4, { 0x0033, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC5, { 0x0034, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC6, { 0x0035, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC7, { 0x0036, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC8, { 0x0037, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACC9, { 0x0038, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACCA, { 0x0039, 0x20DE, 0xF875, 0x0000 } },
		{ 0xACCB, { 0xF866, 0x005B, 0x0031, 0x0030, 0x005D, 0x0000 } },
		{ 0xACCC, { 0xF866, 0x005B, 0x0031, 0x0031, 0x005D, 0x0000 } },
		{ 0xACCD, { 0xF866, 0x005B, 0x0031, 0x0032, 0x005D, 0x0000 } },
		{ 0xACCE, { 0xF866, 0x005B, 0x0031, 0x0033, 0x005D, 0x0000 } },
		{ 0xACCF, { 0xF866, 0x005B, 0x0031, 0x0034, 0x005D, 0x0000 } },
		{ 0xACD0, { 0xF866, 0x005B, 0x0031, 0x0035, 0x005D, 0x0000 } },
		{ 0xACF2, { 0xF866, 0x005B, 0x0031, 0x0036, 0x005D, 0x0000 } },
		{ 0xACF3, { 0xF866, 0x005B, 0x0031, 0x0037, 0x005D, 0x0000 } },
		{ 0xACF4, { 0xF866, 0x005B, 0x0031, 0x0038, 0x005D, 0x0000 } },
		{ 0xACF5, { 0xF866, 0x005B, 0x0031, 0x0039, 0x005D, 0x0000 } },
		{ 0xACF6, { 0xF866, 0x005B, 0x0032, 0x0030, 0x005D, 0x0000 } },
		{ 0xAD41, { 0x4E00, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD42, { 0x4E8C, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD43, { 0x4E09, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD44, { 0x56DB, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD45, { 0x4E94, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD46, { 0x516D, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD47, { 0x4E03, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD48, { 0x516B, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD49, { 0x4E5D, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD4A, { 0x5341, 0x20DE, 0xF87A, 0x0000 } },
		{ 0xAD4B, { 0xF863, 0x005B, 0x5341, 0x4E00, 0x005D, 0x0000 } },
		{ 0xAD4C, { 0xF863, 0x005B, 0x5341, 0x4E8C, 0x005D, 0x0000 } },
		{ 0xAD4D, { 0xF863, 0x005B, 0x5341, 0x4E09, 0x005D, 0x0000 } },
		{ 0xAD4E, { 0xF863, 0x005B, 0x5341, 0x56DB, 0x005D, 0x0000 } },
		{ 0xAD4F, { 0xF863, 0x005B, 0x5341, 0x4E94, 0x005D, 0x0000 } },
		{ 0xAD50, { 0xF863, 0x005B, 0x5341, 0x516D, 0x005D, 0x0000 } },
		{ 0xAD51, { 0xF863, 0x005B, 0x5341, 0x4E03, 0x005D, 0x0000 } },
		{ 0xAD52, { 0xF863, 0x005B, 0x5341, 0x516B, 0x005D, 0x0000 } },
		{ 0xAD53, { 0xF863, 0x005B, 0x5341, 0x4E5D, 0x005D, 0x0000 } },
		{ 0xAD54, { 0xF863, 0x005B, 0x4E8C, 0x5341, 0x005D, 0x0000 } },
		{ 0xAD55, { 0x4E00, 0x20DE, 0x0000 } },
		{ 0xAD56, { 0x4E8C, 0x20DE, 0x0000 } },
		{ 0xAD57, { 0x4E09, 0x20DE, 0x0000 } },
		{ 0xAD58, { 0x56DB, 0x20DE, 0x0000 } },
		{ 0xAD59, { 0x4E94, 0x20DE, 0x0000 } },
		{ 0xAD5A, { 0x516D, 0x20DE, 0x0000 } },
		{ 0xAD5B, { 0x4E03, 0x20DE, 0x0000 } },
		{ 0xAD5C, { 0x516B, 0x20DE, 0x0000 } },
		{ 0xAD5D, { 0x4E5D, 0x20DE, 0x0000 } },
		{ 0xAD5E, { 0x5341, 0x20DE, 0x0000 } },
		{ 0xAD5F, { 0xF862, 0x005B, 0x5341, 0x4E00, 0x005D, 0x0000 } },
		{ 0xAD60, { 0xF862, 0x005B, 0x5341, 0x4E8C, 0x005D, 0x0000 } },
		{ 0xAD61, { 0xF862, 0x005B, 0x5341, 0x4E09, 0x005D, 0x0000 } },
		{ 0xAD62, { 0xF862, 0x005B, 0x5341, 0x56DB, 0x005D, 0x0000 } },
		{ 0xAD63, { 0xF862, 0x005B, 0x5341, 0x4E94, 0x005D, 0x0000 } },
		{ 0xAD64, { 0xF862, 0x005B, 0x5341, 0x516D, 0x005D, 0x0000 } },
		{ 0xAD65, { 0xF862, 0x005B, 0x5341, 0x4E03, 0x005D, 0x0000 } },
		{ 0xAD66, { 0xF862, 0x005B, 0x5341, 0x516B, 0x005D, 0x0000 } },
		{ 0xAD67, { 0xF862, 0x005B, 0x5341, 0x4E5D, 0x005D, 0x0000 } },
		{ 0xAD68, { 0xF862, 0x005B, 0x4E8C, 0x5341, 0x005D, 0x0000 } },
		{ 0xAD69, { 0x65E5, 0x20DE, 0x0000 } },
		{ 0xAD6A, { 0x6708, 0x20DE, 0x0000 } },
		{ 0xAD6B, { 0x706B, 0x20DE, 0x0000 } },
		{ 0xAD6C, { 0x6C34, 0x20DE, 0x0000 } },
		{ 0xAD6D, { 0x6728, 0x20DE, 0x0000 } },
		{ 0xAD6E, { 0x91D1, 0x20DE, 0x0000 } },
		{ 0xAD6F, { 0x571F, 0x20DE, 0x0000 } },
		{ 0xAD77, { 0x65E5, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xAD78, { 0x6708, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xAD79, { 0x706B, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xAD7A, { 0x6C34, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xAD7B, { 0x6728, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xAD7C, { 0x91D1, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xAD7D, { 0x571F, 0x20DE, 0xF87C, 0x0000 } },
		{ 0xADA1, { 0x300C, 0xF879, 0x0000 } },
		{ 0xADA2, { 0x300D, 0xF879, 0x0000 } },
		{ 0xADA3, { 0x300E, 0xF879, 0x0000 } },
		{ 0xADA4, { 0x300F, 0xF879, 0x0000 } },
		{ 0xADA5, { 0x21E8, 0xF878, 0x0000 } },
		{ 0xADA6, { 0x21E6, 0xF878, 0x0000 } },
		{ 0xADA7, { 0x21E7, 0xF878, 0x0000 } },
		{ 0xADA8, { 0x21E9, 0xF878, 0x0000 } },
		{ 0xADAC, { 0x2033, 0xF873, 0x0000 } },
		{ 0xADAE, { 0x2032, 0xF873, 0x0000 } },
		{ 0xADAF, { 0x21E7, 0xF87F, 0x0000 } },
		{ 0xADB0, { 0x0021, 0xF87F, 0x0000 } },
		{ 0xEB41, { 0x3001, 0xF87E, 0x0000 } },
		{ 0xEB42, { 0x3002, 0xF87E, 0x0000 } },
		{ 0xEB50, { 0xFFE3, 0xF87E, 0x0000 } },
		{ 0xEB5B, { 0x30FC, 0xF87E, 0x0000 } },
		{ 0xEB5D, { 0x2010, 0xF87E, 0x0000 } },
		{ 0xEB60, { 0x301C, 0xF87E, 0x0000 } },
		{ 0xEB61, { 0x2016, 0xF87E, 0x0000 } },
		{ 0xEB62, { 0xFF5C, 0xF87E, 0x0000 } },
		{ 0xEB63, { 0x2026, 0xF87E, 0x0000 } },
		{ 0xEB6D, { 0xFF3B, 0xF87E, 0x0000 } },
		{ 0xEB6E, { 0xFF3D, 0xF87E, 0x0000 } },
		{ 0xEB81, { 0xFF1D, 0xF87E, 0x0000 } },
		{ 0xEC9F, { 0x3041, 0xF87E, 0x0000 } },
		{ 0xECA1, { 0x3043, 0xF87E, 0x0000 } },
		{ 0xECA3, { 0x3045, 0xF87E, 0x0000 } },
		{ 0xECA5, { 0x3047, 0xF87E, 0x0000 } },
		{ 0xECA7, { 0x3049, 0xF87E, 0x0000 } },
		{ 0xECC1, { 0x3063, 0xF87E, 0x0000 } },
		{ 0xECE1, { 0x3083, 0xF87E, 0x0000 } },
		{ 0xECE3, { 0x3085, 0xF87E, 0x0000 } },
		{ 0xECE5, { 0x3087, 0xF87E, 0x0000 } },
		{ 0xECEC, { 0x308E, 0xF87E, 0x0000 } },
		{ 0xED40, { 0x30A1, 0xF87E, 0x0000 } },
		{ 0xED42, { 0x30A3, 0xF87E, 0x0000 } },
		{ 0xED44, { 0x30A5, 0xF87E, 0x0000 } },
		{ 0xED46, { 0x30A7, 0xF87E, 0x0000 } },
		{ 0xED48, { 0x30A9, 0xF87E, 0x0000 } },
		{ 0xED62, { 0x30C3, 0xF87E, 0x0000 } },
		{ 0xED83, { 0x30E3, 0xF87E, 0x0000 } },
		{ 0xED85, { 0x30E5, 0xF87E, 0x0000 } },
		{ 0xED87, { 0x30E7, 0xF87E, 0x0000 } },
		{ 0xED8E, { 0x30EE, 0xF87E, 0x0000 } },
		{ 0xED95, { 0x30F5, 0xF87E, 0x0000 } },
		{ 0xED96, { 0x30F6, 0xF87E, 0x0000 } },
		{ 0x0000, { 0x0000 } }
	};

	if (character < 0x8140 || character >= 0xfdff)
	{
		// character outside our known range: return whitespace
		*chars = &asciiMap[0x00];
		return 1;
	}

	int retVal = 0;

	// Find the entry corresponding to the WorldScript character
	if ((retVal = findSimpleMap((unsigned short)(character - 0x8140), chars, charSimpleMap, WPD_NUM_ELEMENTS(charSimpleMap))))
		return retVal;

	if ((retVal = findComplexMap(character, chars, charComplexMap)))
		return retVal;

	*chars = &asciiMap[0x00];
	return 1;

}
/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
