#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2024 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Syberia series:
# - Syberia 1
# - Syberia 2
# send your bug reports to bugs@dotslashplay.it
###

script_version=20250311.1

PLAYIT_COMPATIBILITY_LEVEL='2.32'

GAME_ID_SYBERIA1='syberia-1'
GAME_NAME_SYBERIA1='Syberia'

GAME_ID_SYBERIA2='syberia-2'
GAME_NAME_SYBERIA2='Syberia Ⅱ'

ARCHIVE_BASE_SYBERIA1_0_NAME='syberia_en_2.0.0.1.pkg'
ARCHIVE_BASE_SYBERIA1_0_MD5='61fb5374919b9a5cc9eeda91924b6def'
## TODO: ./play.it 2.32 has no support for Mac OS .pkg installers.
ARCHIVE_BASE_SYBERIA1_0_EXTRACTOR='unar'
ARCHIVE_BASE_SYBERIA1_0_SIZE='3542062'
ARCHIVE_BASE_SYBERIA1_0_VERSION='1.0.2-gog2.0.0.1'
ARCHIVE_BASE_SYBERIA1_0_URL='https://www.gog.com/game/syberia'

ARCHIVE_BASE_SYBERIA2_0_NAME='syberia_2_en_1_0_0_gog_15710.pkg'
ARCHIVE_BASE_SYBERIA2_0_MD5='b331a2e6ddf1d8f2d127a081ef0cab69'
## TODO: ./play.it 2.32 has no support for Mac OS .pkg installers.
ARCHIVE_BASE_SYBERIA2_0_EXTRACTOR='unar'
ARCHIVE_BASE_SYBERIA2_0_SIZE='2620541'
ARCHIVE_BASE_SYBERIA2_0_VERSION='1.0.0-gog15710'
ARCHIVE_BASE_SYBERIA2_0_URL='https://www.gog.com/game/syberia_2'

CONTENT_PATH_DEFAULT='payload/Contents'
CONTENT_GAME_MAIN_FILES='
_CodeSignature
Documents
MacOS
Resources
Info.plist
PkgInfo'

APP_MAIN_SCUMMID_SYBERIA1='tetraedge:syberia'
APP_MAIN_SCUMMID_SYBERIA2='tetraedge:syberia2'
## TODO: ./play.it 2.32 has no support for Mac OS ICNS icons.
APP_MAIN_ICON='Resources/Icon.icns'
## Set the requirements list to extract the game icons
REQUIREMENTS_LIST="${REQUIREMENTS_LIST:-}
icns2png"

# Load common functions

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run the default initialization actions

initialization_default "$@"

# Extract game data

archive_extraction_default
ARCHIVE_INNER_PATH="${PLAYIT_WORKDIR}/gamedata/$(game_id).cpio.gz"
ARCHIVE_INNER_EXTRACTOR='unar'
mv "${PLAYIT_WORKDIR}/gamedata/package.pkg/Scripts" "$ARCHIVE_INNER_PATH"
archive_extraction 'ARCHIVE_INNER'
rm "$ARCHIVE_INNER_PATH"

# Include game data

## Include the game icons
icons_temporary_directory="${PLAYIT_WORKDIR}/icons"
mkdir --parents "$icons_temporary_directory"
icns2png --extract --output "$icons_temporary_directory" "$(icon_full_path 'APP_MAIN_ICON')" >/dev/null 2>&1
icons_include_from_directory "$(current_package)" 'APP_MAIN' "$icons_temporary_directory"
rmdir "$icons_temporary_directory"

content_inclusion_default

# Write launchers

## Set the game language based on the current locale
scummvm_id=$(application_scummvm_scummid 'APP_MAIN')
game_exec_line() {
	cat <<- 'EOF'
	case "${LANG%_*}" in
	    ('fr')
	        scummvm_options='--language=fr'
	    ;;
	    ('en'|*)
	        scummvm_options='--language=en'
	    ;;
	esac
	EOF
	cat <<- EOF
	scummvm --path="\$PATH_GAME_DATA" \$scummvm_options "\$@" $scummvm_id
	EOF
}

launchers_generation

# Build packages

packages_generation
print_instructions

# Clean up

working_directory_cleanup

exit 0
