cmake_minimum_required(VERSION 3.21)

if(NOT CMAKE_SCRIPT_MODE_FILE)
	message(FATAL_ERROR "Usage: cmake -P cmake/Merge.cmake")
endif()

set(prefix ausweisapp)
set(extension .aar)

set(XML_LOAD_LOCAL_LIBS "<array name=\"load_local_libs\">")
set(XML_QT_LIBS "<array name=\"qt_libs\">")
set(XML_END_TAG "</array>")

function(remove_arch filename output)
	list(APPEND ARCHS x86_64 x86 armeabi-v7a arm64-v8a)
	foreach(arch ${ARCHS})
		string(REPLACE "-${arch}" "" filename "${filename}")
	endforeach()
	set(${output} ${filename} PARENT_SCOPE)
endfunction()

function(get_version_filename _files _output)
	foreach(filename ${_files})
		message(STATUS "Check AAR: ${filename}")
		get_filename_component(filename ${filename} NAME)
		remove_arch(${filename} filenameWithoutArch)
		if(filename STREQUAL filenameWithoutArch)
			message(STATUS "Ignore merged AAR: ${filename}")
			continue()
		else()
			set(filename ${filenameWithoutArch})
		endif()

		# Check that any AAR has same version
		string(REPLACE "${extension}" "" aar_version "${filename}")
		string(REPLACE "${prefix}-" "" aar_version "${aar_version}")
		if(APP_NAME_VERSION AND NOT APP_NAME_VERSION STREQUAL aar_version)
			message(STATUS "Different package version: ${APP_NAME_VERSION} / ${aar_version}")
			set(APP_NAME_VERSION version-mismatch)
		else()
			set(APP_NAME_VERSION ${aar_version})
		endif()
	endforeach()

	set(${_output} ${APP_NAME_VERSION} PARENT_SCOPE)
endfunction()

function(extract_aars _files _dir _dir_values)
	foreach(filename ${_files})
		if(NOT EXISTS "${filename}")
			continue()
		endif()

		# Extract the .aar file to package the merged aar itself
		if(EXISTS "${_dir}")
			set(patterns "jni")
		else()
			set(patterns "")
		endif()

		file(ARCHIVE_EXTRACT INPUT "${filename}" DESTINATION "${_dir}" PATTERNS ${patterns})
		get_filename_component(dirname ${filename} NAME)
		file(ARCHIVE_EXTRACT INPUT "${filename}" DESTINATION "${_dir_values}/${dirname}" PATTERNS "res/values/values.xml")
	endforeach()
endfunction()

function(append_value _line)
	cmake_parse_arguments(_PARAM "NOLF" "" "" ${ARGN})
	if(NOT _PARAM_NOLF)
		set(LF "\n")
	endif()
	file(APPEND "${VALUES_XML}" "${_line}${LF}")
endfunction()

function(read_values _all_values _out_local_libs _out_qt_libs)
	foreach(valueFile ${_all_values})
		set(addLineLocalLibs FALSE)
		set(addLineQtLibs FALSE)
		file(STRINGS "${valueFile}" filecontent)
		foreach(line ${filecontent})
			if(line MATCHES "${XML_LOAD_LOCAL_LIBS}")
				set(addLineLocalLibs TRUE)
				continue()
			elseif(line MATCHES "${XML_QT_LIBS}")
				set(addLineQtLibs TRUE)
				continue()
			elseif(line MATCHES "${XML_END_TAG}")
				set(addLineLocalLibs FALSE)
				set(addLineQtLibs FALSE)
			endif()

			if(addLineLocalLibs)
				string(APPEND LOCAL_LIBS "${line}\n")
			endif()
			if(addLineQtLibs)
				string(APPEND QT_LIBS "${line}\n")
			endif()
		endforeach()
	endforeach()

	set(${_out_local_libs} "${LOCAL_LIBS}" PARENT_SCOPE)
	set(${_out_qt_libs} "${QT_LIBS}" PARENT_SCOPE)
endfunction()

function(merge_values _dir _all_values)
	set(VALUES_XML "${_dir}/res/values/values.xml")
	if(NOT EXISTS "${VALUES_XML}")
		message(FATAL_ERROR "Cannot find ${VALUES_XML}")
	endif()

	read_values("${_all_values}" LOCAL_LIBS QT_LIBS)

	set(writeLine TRUE)
	file(STRINGS "${VALUES_XML}" filecontent)
	file(WRITE "${VALUES_XML}" "")
	foreach(line ${filecontent})
		if(line MATCHES "${XML_LOAD_LOCAL_LIBS}")
			set(writeLine FALSE)
			append_value("${line}")
			append_value("${LOCAL_LIBS}" NOLF)
		elseif(line MATCHES "${XML_QT_LIBS}")
			set(writeLine FALSE)
			append_value("${line}")
			append_value("${QT_LIBS}" NOLF)
		elseif(line MATCHES "${XML_END_TAG}")
			set(writeLine TRUE)
		endif()

		if(writeLine)
			append_value("${line}")
		endif()
	endforeach()
endfunction()

function(copy_file _extension _destination)
	file(GLOB files "${CMAKE_BINARY_DIR}/*.${_extension}")
	if(files)
		list(GET files 0 file)
		message(STATUS "Copy ${file} to ${_destination}")
		file(COPY_FILE "${file}" "${_destination}" ONLY_IF_DIFFERENT)
	else()
		message(WARNING "Missing files(s) (${_extension}) in: ${CMAKE_BINARY_DIR}")
	endif()
endfunction()

file(GLOB AAR_FILES "${CMAKE_BINARY_DIR}/*${extension}")
if(NOT AAR_FILES)
	message(FATAL_ERROR "Missing AAR(s) in: ${CMAKE_BINARY_DIR}")
endif()

get_version_filename("${AAR_FILES}" VERSION)
set(AAR_TMP_DIR ${CMAKE_BINARY_DIR}/${VERSION})
set(AAR_VALUES_TMP_DIR ${AAR_TMP_DIR}-values)
if(DIST_DIR)
	get_filename_component(DIST_DIR "${DIST_DIR}" ABSOLUTE)
else()
	set(DIST_DIR ${CMAKE_BINARY_DIR}/dist)
endif()
set(BASE_NAME ${DIST_DIR}/${prefix}-${VERSION})
set(AAR ${BASE_NAME}${extension})
if(EXISTS "${AAR_TMP_DIR}")
	file(REMOVE_RECURSE "${AAR_TMP_DIR}")
endif()
if(EXISTS "${AAR_VALUES_TMP_DIR}")
	file(REMOVE_RECURSE "${AAR_VALUES_TMP_DIR}")
endif()
if(EXISTS "${AAR}")
	message(STATUS "Remove AAR: ${AAR}")
	file(REMOVE "${AAR}")
endif()
if(NOT EXISTS "${DIST_DIR}")
	file(MAKE_DIRECTORY ${DIST_DIR})
endif()

extract_aars("${AAR_FILES}" "${AAR_TMP_DIR}" "${AAR_VALUES_TMP_DIR}")
file(GLOB_RECURSE ALL_VALUES_XML "${AAR_VALUES_TMP_DIR}/values.xml")
merge_values("${AAR_TMP_DIR}" "${ALL_VALUES_XML}")

message(STATUS "Create AAR: ${AAR}")
execute_process(COMMAND ${CMAKE_COMMAND} -E tar cf "${AAR}" --format=zip . WORKING_DIRECTORY "${AAR_TMP_DIR}")
file(REMOVE_RECURSE "${AAR_TMP_DIR}")
file(REMOVE_RECURSE "${AAR_VALUES_TMP_DIR}")

copy_file("pom" "${BASE_NAME}.pom")
copy_file("jar" "${BASE_NAME}-sources.jar")
