//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Data/Datafield.h
//! @brief     Defines class Datafield.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_DEVICE_DATA_DATAFIELD_H
#define BORNAGAIN_DEVICE_DATA_DATAFIELD_H

#include <memory>
#include <vector>

#ifdef BORNAGAIN_PYTHON
#include "PyCore/Embed/PyObjectDecl.h"
#endif

using std::size_t;

class Scale;
class Frame;

//! Stores radiation power per bin.

class Datafield {
public:
    Datafield(); // required by SimulationResult() required by Swig
    //! Constructor that takes ownership of supplied frame and initializes values and errorbars
    Datafield(const Frame* frame, const std::vector<double>& values = {},
              const std::vector<double>& errSigmas = {});

    //! Constructor that takes ownership of supplied axes and initializes values and errorbars
    Datafield(std::vector<const Scale*>&& axes, const std::vector<double>& values = {},
              const std::vector<double>& errSigmas = {});

    Datafield(const Datafield&);
    Datafield(Datafield&&);
    virtual ~Datafield();

    Datafield& operator=(Datafield&& other) noexcept;

    Datafield* clone() const;

    // for use from Python:

    void setAt(size_t i, double val);
    double valAt(size_t i) const;

    // retrieve basic info

    const Frame& frame() const;
    size_t rank() const;
    const Scale& axis(size_t k) const;
    const Scale& xAxis() const;
    const Scale& yAxis() const;

    //! Returns total size of data buffer (product of bin number in every dimension).
    size_t size() const;
    bool empty() const;

    //! Returns copy of raw data vector
    const std::vector<double>& flatVector() const;

    double maxVal() const;
    double minVal() const;

    // modifiers

    //! Multiplies contents by constant factor, e.g. to shift curves in log plot
    void scale(double factor);

    // helpers

    Datafield* crop(double xmin, double ymin, double xmax, double ymax) const;
    Datafield* crop(double xmin, double xmax) const;

#ifdef BORNAGAIN_PYTHON
    //! Returns data as Python numpy array
    PyObject* npArray() const;
#endif

    //! Project a 2D histogram into 1D histogram along X. The projection is made
    //! from all bins along y-axis.
    Datafield* xProjection() const;

    //! @brief Project a 2D histogram into 1D histogram along X. The projection is made
    //! from the y-bin closest to given ordinate yvalue.
    //! @param yvalue the value on y-axis at which projection is taken
    Datafield* xProjection(double yvalue) const;

    //! @brief Project a 2D histogram into 1D histogram along X. The projection is made from
    //! all y-bins corresponding to ordinate between ylow and yup.
    //! @param ylow lower edje on y-axis
    //! @param yup upper edje on y-axis
    Datafield* xProjection(double ylow, double yup) const;

    //! Project a 2D histogram into 1D histogram along Y. The projection is made
    //! from all bins along x-axis.
    Datafield* yProjection() const;

    //! @brief Project a 2D histogram into 1D histogram along Y. The projection is made
    //! from the x-bin closest to given abscissa xvalue.
    //! @param xvalue the value on x-axis at which projection is taken
    Datafield* yProjection(double xvalue) const;

    //! @brief Project a 2D histogram into 1D histogram along Y. The projection is made from
    //! all x-bins corresponding to abscissa between xlow and xup.
    //! @param xlow lower edje on x-axis
    //! @param xup upper edje on x-axis
    Datafield* yProjection(double xlow, double xup) const;

#ifndef SWIG
    double& operator[](size_t i);
    const double& operator[](size_t i) const;

    //! Sets new values to raw data vector
    void setVector(const std::vector<double>& data_vector);

    bool hasErrorSigmas() const;
    const std::vector<double>& errorSigmas() const;
    std::vector<double>& errorSigmas();

    //! Sets content of output data to specific value
    void setAllTo(const double& value);

protected:
    std::unique_ptr<const Frame> m_frame;

private:
    //! Creates projection along X. The projections is made by collecting the data in the range
    //! between [ybinlow, ybinup].
    Datafield* create_xProjection(int ybinlow, int ybinup) const;

    //! Creates projection along Y. The projections is made by collecting the data in the range
    //! between [xbinlow, xbinup].
    Datafield* create_yProjection(int xbinlow, int xbinup) const;

    std::vector<double> m_values;
    std::vector<double> m_errSigmas;

#endif // SWIG
};

#endif // BORNAGAIN_DEVICE_DATA_DATAFIELD_H
