//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/ItemWithParticles.cpp
//! @brief     Implements class ItemWithParticles
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/ItemWithParticles.h"
#include "Base/Vector/RotMatrix.h"
#include "Sample/Particle/IParticle.h"
#include "Sample/Scattering/Rotations.h"

namespace {
namespace Tag {

const QString Abundance("Abundance");
const QString Position("Position");
const QString Rotation("Rotation");

} // namespace Tag
} // namespace

ItemWithParticles::ItemWithParticles(const QString& abundanceTooltip,
                                     const QString& positionTooltip)
{
    m_abundance.init("Abundance", abundanceTooltip, 1.0, Unit::unitless, 3 /* decimals */,
                     RealLimits::limited(0.0, 1.0), "abundance");
    m_position.init("Position Offset", positionTooltip, Unit::nanometer, "pos");
    m_rotation.init("Rotation", "");
}

std::unique_ptr<IRotation> ItemWithParticles::createRotation() const
{
    if (!m_rotation.currentItem())
        return {};
    return m_rotation->createRotation();
}

void ItemWithParticles::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // abundance
    w->writeStartElement(Tag::Abundance);
    m_abundance.writeTo(w);
    w->writeEndElement();

    // position
    w->writeStartElement(Tag::Position);
    m_position.writeTo(w);
    w->writeEndElement();

    // rotation
    w->writeStartElement(Tag::Rotation);
    m_rotation.writeTo(w);
    w->writeEndElement();
}

void ItemWithParticles::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // abundance
        if (tag == Tag::Abundance) {
            m_abundance.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // position
        } else if (tag == Tag::Position) {
            m_position.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // rotation
        } else if (tag == Tag::Rotation) {
            m_rotation.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
