//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Mask/EllipseView.cpp
//! @brief     Implements EllipseView class
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Mask/EllipseView.h"
#include "GUI/Model/Data/MaskItems.h"
#include <QGraphicsSceneMouseEvent>
#include <QPainter>

EllipseView::EllipseView(EllipseItem* item)
    : RectangleBaseView(item)
    , m_item(item)
{
    m_item->maskVisibilityChanged();
}

void EllipseView::onChangedX()
{
    setBlockOnProperty(true);
    m_item->setXCenter(fromSceneX(this->x()));
    setBlockOnProperty(false);
}

void EllipseView::onChangedY()
{
    setBlockOnProperty(true);
    m_item->setYCenter(fromSceneY(this->y()));
    setBlockOnProperty(false);
}

void EllipseView::onPropertyChange()
{
    update_view();
    setX(toSceneX(m_item->xCenter()));
    setY(toSceneY(m_item->yCenter()));
    setTransform(QTransform().rotate(-1.0 * m_item->angle()));
}

void EllipseView::mouseMoveEvent(QGraphicsSceneMouseEvent* event)
{
    if (m_activeHandleElement) {
        QPointF opposPos = mapFromScene(m_resize_opposite_origin);
        qreal xmin = std::min(event->pos().x(), opposPos.x());
        qreal xmax = std::max(event->pos().x(), opposPos.x());
        qreal ymin = std::min(event->pos().y(), opposPos.y());
        qreal ymax = std::max(event->pos().y(), opposPos.y());
        qreal width = xmax - xmin;
        qreal height = ymax - ymin;

        qreal xcenter = xmin + (xmax - xmin) / 2.;
        qreal ycenter = ymin + (ymax - ymin) / 2.;

        QPointF center(xcenter, ycenter);
        QPointF centerInScene = mapToScene(center);

        if (m_activeHandleElement->getHandleType() == SizeHandleElement::RESIZE) {
            m_item->setXCenter(fromSceneX(centerInScene.x()));
            m_item->setYCenter(fromSceneY(centerInScene.y()));

            m_item->setXRadius((fromSceneX(centerInScene.x() + width / 2.)
                                - fromSceneX(centerInScene.x() - width / 2.))
                               / 2.);
            m_item->setYRadius((fromSceneY(centerInScene.y() - height / 2.)
                                - fromSceneY(centerInScene.y() + height / 2.))
                               / 2.);
        } else if (m_activeHandleElement->getHandleType() == SizeHandleElement::RESIZE_HEIGHT) {
            m_item->setYCenter(fromSceneY(centerInScene.y()));
            m_item->setYRadius((fromSceneY(centerInScene.y() - height / 2.)
                                - fromSceneY(centerInScene.y() + height / 2.))
                               / 2.);
        } else if (m_activeHandleElement->getHandleType() == SizeHandleElement::RESIZE_WIDTH) {
            m_item->setXCenter(fromSceneX(centerInScene.x()));
            m_item->setXRadius((fromSceneX(centerInScene.x() + width / 2.)
                                - fromSceneX(centerInScene.x() - width / 2.))
                               / 2.);
        }
    } else {
        IShape2DView::mouseMoveEvent(event);
    }
}

QPainterPath EllipseView::shape() const
{
    QPainterPath path;
    path.addEllipse(m_mask_rect);
    return path;
}

MaskItemObject* EllipseView::parameterizedItem() const
{
    return m_item;
}

//! updates position of view using item properties

void EllipseView::update_position()
{
    disconnect(this, &EllipseView::xChanged, this, &EllipseView::onChangedX);
    disconnect(this, &EllipseView::yChanged, this, &EllipseView::onChangedY);

    setX(toSceneX(m_item->xCenter()));
    setY(toSceneY(m_item->yCenter()));

    connect(this, &EllipseView::xChanged, this, &EllipseView::onChangedX);
    connect(this, &EllipseView::yChanged, this, &EllipseView::onChangedY);

    if (m_item->angle() != 0.0)
        setTransform(QTransform().rotate(-1.0 * m_item->angle()));
}

QRectF EllipseView::mask_rectangle()
{
    return QRectF(-width() / 2., -height() / 2., width(), height());
}

//! Returns the x-coordinate of the rectangle's left edge

qreal EllipseView::left() const
{
    return toSceneX(m_item->xCenter() - m_item->xRadius());
}

//! Returns the x-coordinate of the rectangle's right edge

qreal EllipseView::right() const
{
    return toSceneX(m_item->xCenter() + m_item->xRadius());
}

//! Returns the y-coordinate of the rectangle's top edge.

qreal EllipseView::top() const
{
    return toSceneY(m_item->yCenter() + m_item->yRadius());
}

//! Returns the y-coordinate of the rectangle's bottom edge.

qreal EllipseView::bottom() const
{
    return toSceneY(m_item->yCenter() - m_item->yRadius());
}
