//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/PlotSpecular/SpecularDataCanvas.cpp
//! @brief     Implements class SpecularDataCanvas
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/PlotSpecular/SpecularDataCanvas.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Data/SpecularDataItem.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/View/PlotSpecular/SpecularPlotCanvas.h"
#include "GUI/View/PlotUtil/SavePlotAssistant.h"
#include <QVBoxLayout>
#include <qcustomplot.h>

SpecularDataCanvas::SpecularDataCanvas(QWidget* parent)
    : DataAccessWidget(parent)
    , m_plot_canvas(new SpecularPlotCanvas)
    , m_reset_view_action(nullptr)
    , m_save_plot_action(nullptr)
{
    QVBoxLayout* vlayout = new QVBoxLayout(this);
    vlayout->setSpacing(0);
    vlayout->setContentsMargins(0, 0, 0, 0);
    vlayout->addWidget(m_plot_canvas);
    setLayout(vlayout);
    setStyleSheet("background-color:white;");

    m_plot_canvas->setStatusLabelEnabled(true);

    m_reset_view_action = new QAction(this);
    m_reset_view_action->setText("Center view");
    m_reset_view_action->setIcon(QIcon(":/images/camera-metering-center.svg"));
    m_reset_view_action->setToolTip("Reset view\n"
                                    "x,y axes range will be set to default");
    connect(m_reset_view_action, &QAction::triggered, this, &SpecularDataCanvas::onResetViewAction,
            Qt::UniqueConnection);

    m_save_plot_action = new QAction(this);
    m_save_plot_action->setText("Save");
    m_save_plot_action->setIcon(QIcon(":/images/content-save-outline.svg"));
    m_save_plot_action->setToolTip("Save plot");
    connect(m_save_plot_action, &QAction::triggered, this, &SpecularDataCanvas::onSavePlotAction,
            Qt::UniqueConnection);

    enableDeprecatedOnMousePress(true); // true for legacy reasons
}

void SpecularDataCanvas::setJobOrRealItem(QObject* job_or_real_Item)
{
    DataAccessWidget::setJobOrRealItem(job_or_real_Item);
    m_plot_canvas->setSpecularItems({currentSpecularDataItem()});
}

QSize SpecularDataCanvas::sizeHint() const
{
    return QSize(500, 400);
}

QSize SpecularDataCanvas::minimumSizeHint() const
{
    return QSize(128, 128);
}

QList<QAction*> SpecularDataCanvas::actionList()
{
    return QList<QAction*>() << m_reset_view_action << m_save_plot_action;
}

QCustomPlot* SpecularDataCanvas::customPlot()
{
    return m_plot_canvas->customPlot();
}

void SpecularDataCanvas::enableDeprecatedOnMousePress(bool b)
{
    if (b)
        connect(m_plot_canvas->customPlot(), &QCustomPlot::mousePress, this,
                &SpecularDataCanvas::onMousePress, Qt::UniqueConnection);
    else
        disconnect(m_plot_canvas->customPlot(), &QCustomPlot::mousePress, this,
                   &SpecularDataCanvas::onMousePress);
}

void SpecularDataCanvas::onResetViewAction()
{
    for (auto item : allSpecularDataItems())
        item->resetView();
    gProjectDocument.value()->setModified();
}

void SpecularDataCanvas::onSavePlotAction()
{
    ASSERT(gProjectDocument.has_value());
    QString dirname = gProjectDocument.value()->userExportDir();
    GUI::Plot::savePlot(dirname, m_plot_canvas->customPlot(), currentSpecularDataItem()->c_field());
}

void SpecularDataCanvas::onMousePress(QMouseEvent* event)
{
    if (event->button() == Qt::RightButton)
        emit customContextMenuRequested(event->globalPos());
}
