//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sim/Scan/AlphaScan.h
//! @brief     Declares AlphaScan class.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_SIM_SCAN_ALPHASCAN_H
#define BORNAGAIN_SIM_SCAN_ALPHASCAN_H

#include "Sim/Scan/IBeamScan.h"
#include <memory>

struct ParameterSample;
class IDistribution1D;

//! Scan type with inclination angles as coordinate values and a unique wavelength.
//! Features footprint correction.
class AlphaScan : public IBeamScan {
public:
    AlphaScan(const Scale& alpha_axis);
    AlphaScan(int nbins, double alpha_i_min, double alpha_i_max);
    ~AlphaScan() override;
    AlphaScan* clone() const override;
    std::string className() const final { return "AlphaScan"; }
    std::vector<const INode*> nodeChildren() const override;

    void setWavelengthDistribution(const IDistribution1D& distr);
    void setAngleDistribution(const IDistribution1D& distr);

    void setWavelength(double lambda);
    void setAlphaOffset(double offset) { m_alpha_offset = offset; }

#ifndef SWIG
    //! Generates simulation elements for specular simulations
    std::vector<SpecularElement> generateElements() const override;

    size_t nDistributionSamples() const override;

    CoordSystem1D* scanCoordSystem() const override;

    // needed for export
    const IDistribution1D* wavelengthDistribution() const
    {
        return m_lambda_distrib.get();
    }
    const IDistribution1D* angleDistribution() const
    {
        return m_alpha_distrib.get();
    }

    double alphaOffset() const
    {
        return m_alpha_offset;
    }

private:
    void checkInitialization();

    std::unique_ptr<const IDistribution1D> m_lambda_distrib;
    std::unique_ptr<const IDistribution1D> m_alpha_distrib;

    double m_alpha_offset{0};
#endif // SWIG
};

#endif // BORNAGAIN_SIM_SCAN_ALPHASCAN_H
