// Copyright 2017 Canonical Ltd.
// Licensed under the AGPLv3, see LICENCE file for details.

package discharger_test

import (
	"encoding/json"
	"html/template"
	"net/http"
	"net/http/httptest"
	"time"

	jc "github.com/juju/testing/checkers"
	"golang.org/x/net/context"
	"gopkg.in/CanonicalLtd/candidclient.v1/params"
	gc "gopkg.in/check.v1"
	errgo "gopkg.in/errgo.v1"
	"gopkg.in/macaroon-bakery.v2/bakery"
	"gopkg.in/macaroon-bakery.v2/httpbakery"

	"github.com/CanonicalLtd/candid/idp"
	"github.com/CanonicalLtd/candid/internal/auth"
	"github.com/CanonicalLtd/candid/internal/candidtest"
	"github.com/CanonicalLtd/candid/internal/discharger"
	"github.com/CanonicalLtd/candid/internal/identity"
	"github.com/CanonicalLtd/candid/internal/monitoring"
	"github.com/CanonicalLtd/candid/meeting"
	"github.com/CanonicalLtd/candid/store"
)

type idpSuite struct {
	candidtest.StoreSuite

	// template is used to configure the output generated by success
	// following a login. if there is a template called "login" in
	// template then it will be processed and the output returned.
	template     *template.Template
	meetingPlace *meeting.Place

	vc idp.VisitCompleter
}

var _ = gc.Suite(&idpSuite{})

func (s *idpSuite) SetUpTest(c *gc.C) {
	s.StoreSuite.SetUpTest(c)

	s.template = template.New("")

	oven := bakery.NewOven(bakery.OvenParams{
		Namespace: auth.Namespace,
		RootKeyStoreForOps: func([]bakery.Op) bakery.RootKeyStore {
			return s.BakeryRootKeyStore
		},
		Key:      bakery.MustGenerateKey(),
		Location: "candidtest",
	})
	var err error
	s.meetingPlace, err = meeting.NewPlace(meeting.Params{
		Store:      s.MeetingStore,
		Metrics:    monitoring.NewMeetingMetrics(),
		ListenAddr: "localhost",
	})
	c.Assert(err, gc.Equals, nil)

	s.vc = discharger.NewVisitCompleter(identity.HandlerParams{
		ServerParams: identity.ServerParams{
			Store:        s.Store,
			MeetingStore: s.MeetingStore,
			RootKeyStore: s.BakeryRootKeyStore,
			Template:     s.template,
		},
		MeetingPlace: s.meetingPlace,
		Oven:         oven,
	})
}

func (s *idpSuite) TearDownTest(c *gc.C) {
	s.meetingPlace.Close()
	s.StoreSuite.TearDownTest(c)
}

func (s *idpSuite) TestLoginFailure(c *gc.C) {
	rr := httptest.NewRecorder()
	s.vc.Failure(context.Background(), rr, nil, "", errgo.WithCausef(nil, params.ErrForbidden, "test error"))
	c.Assert(rr.Code, gc.Equals, http.StatusForbidden)
	var perr params.Error
	err := json.Unmarshal(rr.Body.Bytes(), &perr)
	c.Assert(err, gc.Equals, nil)
	c.Assert(perr, jc.DeepEquals, params.Error{
		Code:    params.ErrForbidden,
		Message: "test error",
	})
}

func (s *idpSuite) TestLoginFailureWithWait(c *gc.C) {
	id := "test"
	err := s.meetingPlace.NewRendezvous(context.Background(), id, []byte("test"))
	c.Assert(err, gc.Equals, nil)

	rr := httptest.NewRecorder()
	s.vc.Failure(context.Background(), rr, nil, id, errgo.WithCausef(nil, params.ErrForbidden, "test error"))
	c.Assert(rr.Code, gc.Equals, http.StatusForbidden)
	var perr params.Error
	err = json.Unmarshal(rr.Body.Bytes(), &perr)
	c.Assert(err, gc.Equals, nil)
	c.Assert(perr, jc.DeepEquals, params.Error{
		Code:    params.ErrForbidden,
		Message: "test error",
	})

	ctx, cancel := context.WithTimeout(context.Background(), 100*time.Millisecond)
	defer cancel()
	d1, d2, err := s.meetingPlace.Wait(ctx, id)
	c.Assert(err, gc.Equals, nil)
	c.Assert(string(d1), gc.Equals, "test")
	var li discharger.LoginInfo
	err = json.Unmarshal(d2, &li)
	c.Assert(err, gc.Equals, nil)
	c.Assert(li.DischargeToken, gc.IsNil)
	c.Assert(li.Error, jc.DeepEquals, &httpbakery.Error{
		Message: "test error",
	})
}

func (s *idpSuite) TestLoginSuccess(c *gc.C) {
	req, err := http.NewRequest("GET", "", nil)
	c.Assert(err, gc.Equals, nil)
	rr := httptest.NewRecorder()
	s.vc.Success(context.Background(), rr, req, "", &store.Identity{
		Username: "test-user",
	})
	c.Assert(rr.Code, gc.Equals, http.StatusOK)
	c.Assert(rr.HeaderMap.Get("Content-Type"), gc.Equals, "text/plain; charset=utf-8")
	c.Assert(rr.Body.String(), gc.Equals, "Login successful as test-user")
}

func (s *idpSuite) TestLoginSuccessWithTemplate(c *gc.C) {
	_, err := s.template.New("login").Parse("<h1>Login successful as {{.Username}}</h1>")
	c.Assert(err, gc.Equals, nil)
	req, err := http.NewRequest("GET", "", nil)
	c.Assert(err, gc.Equals, nil)
	rr := httptest.NewRecorder()
	s.vc.Success(context.Background(), rr, req, "", &store.Identity{
		Username: "test-user",
	})
	c.Assert(rr.Code, gc.Equals, http.StatusOK)
	c.Assert(rr.HeaderMap.Get("Content-Type"), gc.Equals, "text/html;charset=utf-8")
	c.Assert(rr.Body.String(), gc.Equals, "<h1>Login successful as test-user</h1>")
}
