#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>

#ifdef WIN32
#include <windows.h>
#include <winsock2.h>
#include <ws2tcpip.h>
#undef USE_WINSOCK
#define USE_WINSOCK 2
#include <io.h>
#include <winsock2.h>
#else
#include <sys/socket.h>
#endif

#include <sys/time.h>
#include <unistd.h>
#include <errno.h>

#include <sstream>

#include "canlxx.h"
#include "opensslutil.h"
#include "opensslgenericutil.h"
#include "cache.h"

namespace AuthN {

using namespace AuthN::OpenSSL;

static int ssl_no_passphrase_callback(char*, int, int, void*) {
   return -1;
}

static int ssl_verify_callback(int ok,X509_STORE_CTX *sctx) {
  if(!ok) {
    X509_STORE_CTX_set_error(sctx,X509_V_OK); 
    //SSL* ssl = (SSL*)X509_STORE_CTX_get_ex_data(sctx,SSL_get_ex_data_X509_STORE_CTX_idx());
    //if(ssl) SSL_set_verify_result(ssl,X509_V_OK);
    ok = 1;
  };
  return ok;
}

static std::string tostring(unsigned int val) {
  std::stringstream ss;
  ss << val;
  return ss.str();
}

//#if (OPENSSL_VERSION_NUMBER > 0x009080ffL)
#ifdef HAVE_OCSP_STAPLING
static Status initialize_ocsp_stapling(SSL_CTX* ctx, Context* logctx, X509* cert, 
    const std::string& cache_file, long cache_interval, const std::string& resp_url) {

  ssl_stapling_init();
  SSL_CTX_set_tlsext_status_cb(ctx, ocsp_stapling_server_callback);

  return ssl_stapling_init_CertStatusContext(logctx, ctx, cert, cache_file, cache_interval, resp_url);
}
#endif

static void set_ssl_ctx(SSL_CTX* ctx, const X509* cert, 
    const EVP_PKEY* key, const STACK_OF(X509)* chain) {

  // Setting compatibility options
  SSL_CTX_set_mode(ctx,SSL_MODE_ENABLE_PARTIAL_WRITE);
#ifdef SSL_OP_NO_TICKET
  SSL_CTX_set_options(ctx,
      SSL_OP_SINGLE_DH_USE | SSL_OP_NO_SSLv2 | SSL_OP_ALL | SSL_OP_NO_TICKET);
#else
  SSL_CTX_set_options(ctx,
      SSL_OP_SINGLE_DH_USE | SSL_OP_NO_SSLv2 | SSL_OP_ALL);
#endif
  SSL_CTX_set_session_cache_mode(ctx,SSL_SESS_CACHE_OFF);

  // Using as little verification as possible. Verification of remote peer 
  // will be done after connection established using Validator.
  //   CAPath
  //if(!SSL_CTX_load_verify_locations(ctx, NULL, NULL)) goto error;
  //   SSL_VERIFY_PEER |  SSL_VERIFY_FAIL_IF_NO_PEER_CERT
  // SSL_VERIFY_PEER is needed to make server request certificate from client
  SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, &ssl_verify_callback);
  //SSL_CTX_set_cert_verify_callback(ctx_,&verify_cert_callback,NULL);
  //   X509_V_FLAG_CRL_CHECK | X509_V_FLAG_ALLOW_PROXY_CERTS

  //X509_VERIFY_PARAM belongs to the changes between 0.9.7h and 0.9.8
#ifndef HAVE_OPENSSL_X509_VERIFY_PARAM
    X509_STORE* store = NULL;
    int vflags = 0;
    vflags |= X509_V_FLAG_IGNORE_CRITICAL;
#ifdef X509_V_FLAG_ALLOW_PROXY_CERTS
    vflags |= X509_V_FLAG_ALLOW_PROXY_CERTS;
#endif
    store = SSL_CTX_get_cert_store(ctx);
    X509_STORE_set_flags(store, vflags);
#else
  if(ctx->param) X509_VERIFY_PARAM_set_flags(ctx->param,
      X509_V_FLAG_IGNORE_CRITICAL | X509_V_FLAG_ALLOW_PROXY_CERTS);
  if(ctx->param) X509_VERIFY_PARAM_clear_flags(ctx->param,
      X509_V_FLAG_CRL_CHECK | X509_V_FLAG_CRL_CHECK_ALL | X509_V_FLAG_POLICY_CHECK);
#endif

  if(cert) {
    if(!cert) throw Status(-1,"Missing certificate");
    if(!key) throw Status(-1,"Missing private key");
    if(SSL_CTX_use_certificate(ctx,(X509*)cert) != 1) throw Status(-1,"Failed to accept certificate");
    if(SSL_CTX_use_PrivateKey(ctx,(EVP_PKEY*)key) != 1) throw Status(-1,"Failed to accept private key");
    if(chain != NULL) {
      for(int idx = 0;;++idx) {
        if(idx >= sk_X509_num(chain)) break;
        X509* cert = sk_X509_value(chain,idx);
        if(cert) cert = X509_dup(cert);
        if(cert) {
          if(SSL_CTX_add_extra_chain_cert(ctx,cert) != 1) {
            X509_free(cert);
            throw Status(-1,"Failed to accept certificate from chain");
          };
        };
      };
    };
  };

  SSL_CTX_set_default_passwd_cb(ctx, &ssl_no_passphrase_callback);

}

static struct timeval time_left(const struct timeval& t) {
  struct timeval now;
  struct timeval td;
  td.tv_sec = 0;
  td.tv_usec = 0;
  if(gettimeofday(&now,NULL) != 0) return td;
  if(now.tv_sec > t.tv_sec) return td;
  td.tv_sec = t.tv_sec - now.tv_sec;
  if(now.tv_usec > t.tv_usec) {
    if(td.tv_sec > 0) {
      td.tv_usec = 1000000 + t.tv_usec - now.tv_usec;
      --td.tv_sec;
    };
  } else {
    td.tv_usec = t.tv_usec - now.tv_usec;
  };
  return td;
}

static struct timeval time_inc(int ms) {
  if(ms < 0) ms = 24*60*60*1000; // long enough
  struct timeval td;
  td.tv_sec = time(NULL) + (ms?(ms/1000 + 1):0); // backup
  td.tv_usec = 0;
  if(gettimeofday(&td,NULL) != 0) return td;
  td.tv_sec += ms/1000;
  td.tv_usec += (ms%1000)*1000;
  td.tv_sec += td.tv_usec/1000000;
  td.tv_usec %= 1000000;
  return td;
}

static bool BIO_wait_activity(BIO* bio, struct timeval& till) {
  if((!BIO_should_read(bio)) && (!BIO_should_write(bio))) return true;
  int fd = -1;
  BIO_get_fd(bio,&fd);
  if(fd == -1) return false;
  fd_set ifd;
  fd_set ofd;
  FD_ZERO(&ifd);
  FD_ZERO(&ofd);
  if(BIO_should_read(bio)) FD_SET(fd,&ifd);
  if(BIO_should_write(bio)) FD_SET(fd,&ofd);
  struct timeval to = time_left(till); 
  if((fd=select(fd+1,&ifd,&ofd,NULL,&to)) < 0) return false;
  if(fd == 0) return false;
  return true;
}

static bool BIO_wait_read(BIO* bio, struct timeval& till) {
  int fd = -1;
  BIO_get_fd(bio,&fd);
  if(fd == -1) return false;
  fd_set ifd;
  fd_set ofd;
  FD_ZERO(&ifd);
  FD_ZERO(&ofd);
  FD_SET(fd,&ifd);
  struct timeval to = time_left(till); 
  if((fd=select(fd+1,&ifd,&ofd,NULL,&to)) < 0) return false;
  if(fd == 0) return false;
  return true;
}

static Credentials* get_peer(SSL* ssl) {
  X509* cert = NULL;
  STACK_OF(X509)* chain = NULL;

  // Analyze peer's credentials
  if(!(cert = SSL_get_peer_certificate(ssl))) throw Status(9);
  chain = SSL_get_peer_cert_chain(ssl);
  AuthN::OpenSSL::CertContext* certctx = NULL;
  AuthN::Context ctx(AuthN::Context::EmptyContext);
  certctx = new AuthN::OpenSSL::CertContext(ctx);
  certctx->certFromX509(cert, chain);
  std::string peer_str;
  certctx->certToPEM(peer_str);
  if(certctx!=NULL) delete certctx;
  Credentials* peer = new Credentials(Context());
  peer->Assign(peer_str);
  return peer;
}

// ----------------------------------------------------------------------------

IONetwork::IONetwork(const AuthN::Context& ctx):IO(ctx),ctx_(NULL),bio_(NULL) {
}

IONetwork::~IONetwork(void) {
  if(bio_) BIO_free_all(bio_);
  if(ctx_) SSL_CTX_free(ctx_);
}

AuthN::Status IONetwork::Connect(const std::string& host, int port) {
  //int err = SSL_ERROR_NONE;
  SSL* ssl = NULL;
  
  try {

  if(bio_ || ctx_) throw Status(-1,"Connection already established");  

  ctx_=SSL_CTX_new(SSLv23_client_method());
  //ctx_=SSL_CTX_new(SSLv3_client_method());
  if(!ctx_) throw Status(-1);

  //set_ssl_ctx(ctx_,
  //    node_?node_->cert_:NULL,node_?node_->priv_key_:NULL,node_?node_->cert_chain_:NULL);
  set_ssl_ctx(ctx_, node_?node_->certctx_->cert_.cert : NULL, 
      node_?node_->certctx_->key_.pkey : NULL, node_?node_->certctx_->cert_.cert_chain : NULL);

  // Communication layer
  bio_ = BIO_new_ssl_connect(ctx_);
  if(!bio_) throw Status(-1);
  BIO_get_ssl(bio_, &ssl);
  if(!ssl) throw Status(6);
  BIO_set_conn_hostname(bio_, (host+":"+tostring(port)).c_str());
  if(!BIO_set_nbio(bio_,1)) throw Status(5);
  SSL_set_mode(ssl, SSL_MODE_AUTO_RETRY);

//#if (OPENSSL_VERSION_NUMBER > 0x009080ffL)
#ifdef HAVE_OCSP_STAPLING
  // Setup the ocsp cert status request value from RFC 3546
  SSL_CTX_set_tlsext_status_cb(ctx_, ocsp_stapling_client_callback);
  SSL_CTX_set_tlsext_status_arg(ctx_, context_);

  // Trigger on the OCSP stapling protocol
  // TODO: Look for the OCSP response in local cache at first, 
  // if not found, trigger on the OCSP stapling protocol,
  // if stapling fails, contact the OCSP responder directly

  // Current implementation status:
  // Trigger on the OCSP stapling protocol at first,
  // if stapling fails, look for the OCSP response in local cache,
  // if not found, contact the OCSP responder directly.

  SSL_set_tlsext_status_type(ssl, TLSEXT_STATUSTYPE_ocsp);
#endif

  //SSL_set_verify(ssl, SSL_VERIFY_PEER,  ssl_verify_callback);

  struct timeval till = time_inc(timeout_);
  for(;;) {
    if(BIO_do_connect(bio_) == 1) break;
    if(!BIO_should_retry(bio_)) return Status(-1);
    if(!BIO_wait_activity(bio_,till)) throw Status(CommunicationTimeout,"Timeout waiting for remote peer activity");
  };
  for(;;) {
    if(BIO_do_handshake(bio_) == 1) break;
    if(!BIO_should_retry(bio_)) return Status(-1);
    if(!BIO_wait_activity(bio_,till)) throw Status(CommunicationTimeout,"Timeout waiting for handshake");
  };

  // Analyze peer's credentials
/*
  if(!(cert = SSL_get_peer_certificate(ssl))) throw Status(9);
  chain = SSL_get_peer_cert_chain(ssl);
  if(peer_) delete peer_;
  peer_ = new Credentials(Context());
  peer_->Assign(cert,chain,NULL);
*/
  peer_ = get_peer(ssl);
  validator_->SetMode(AuthN::Validator::ValidationOCSPIfPresent);

  Status valid = validator_->Validate(*peer_);
  if(!valid) throw valid;

  valid_ = true;
  return Status();

  } catch(Status& err) {
    std::string openssl_err = GetOpenSSLError();
    if(bio_) BIO_free_all(bio_); bio_ = NULL;
    if(ctx_) SSL_CTX_free(ctx_); ctx_ = NULL;
    valid_ = false;
    if(openssl_err.empty()) {
      last_error_ = err;
    } else {
      last_error_ = Status(err.GetCode(),err.GetDescription()+": "+openssl_err);
    };
    return last_error_;
  };
}

AuthN::IO& IONetwork::Accept(const std::string& host, int port) {
  SSL *ssl;
  BIO* conn = NULL;
  BIO* sbio = NULL;

  try {

  if(bio_) {
    if(!ctx_) throw Status(-1);
  } else {
    if(ctx_) throw Status(-1);

    ctx_ = SSL_CTX_new(SSLv23_server_method());
    if(!ctx_) throw Status(-1);

    //set_ssl_ctx(ctx_,
    //     node_?node_->cert_:NULL,node_?node_->priv_key_:NULL,node_?node_->cert_chain_:NULL);
    set_ssl_ctx(ctx_, node_?node_->certctx_->cert_.cert : NULL, 
         node_?node_->certctx_->key_.pkey : NULL, node_?node_->certctx_->cert_.cert_chain : NULL);



    X509* cert = node_ ? node_->certctx_->cert_.cert : NULL;
    Context* logctx_ = context_;
    // TODO: give the paramters
//#if (OPENSSL_VERSION_NUMBER > 0x009080ffL)
#ifdef HAVE_OCSP_STAPLING
    std::string cache_file = "/tmp/ocsp_stapling_cache";
    Status status = initialize_ocsp_stapling(ctx_, logctx_, cert, cache_file, 0, "");
    if(!status) context_->Log(Context::LogError, status.GetDescription());
#endif


    bio_ = BIO_new_accept((char*)((host.empty()?"*":host)+":"+tostring(port)).c_str());
    if(!bio_) throw Status(-1);
    if(BIO_do_accept(bio_) <= 0) throw Status(-1);
  };

  if(BIO_do_accept(bio_) <= 0) throw Status(-1);

  conn = BIO_pop(bio_);
  if(!conn) throw Status(-1);
  //if(!BIO_set_nbio(conn,1)) throw Status(5);
  BIO_set_nbio(conn,1);

  sbio=BIO_new_ssl(ctx_,0); // Or copy ctx_ ?
  if(!sbio) throw Status(-1);
  BIO_get_ssl(sbio, &ssl);
  if(!ssl) throw Status(-1);
  SSL_set_mode(ssl, SSL_MODE_AUTO_RETRY);
  sbio = BIO_push(sbio, conn); conn = NULL;

  if(BIO_do_handshake(sbio) <= 0) throw Status(-1);

  // Analyze peer's credentials
/*
  if(!(cert = SSL_get_peer_certificate(ssl))) throw Status(9);
  chain = SSL_get_peer_cert_chain(ssl);
  Credentials* peer = new Credentials(Context());
  peer->Assign(cert,chain,NULL);
*/
  Credentials* peer = get_peer(ssl);
  Status valid = validator_->Validate(*peer);
  if(!valid) { delete peer; throw valid; };

  IONetwork* nio = new IONetwork(validator_->GetContext());
  nio->ctx_ = NULL; // not assigning ctx_ because not to be freed explicitely
  nio->bio_ = sbio;
  nio->peer_ = peer;
// BIO_flush(sbio);
// BIO_free_all(sbio);

  valid_ = true;
  return *nio;

  } catch(Status& err) {
    std::string openssl_err = GetOpenSSLError();
    // TODO: not every error requires to kill accepting bio
    if(bio_) BIO_free_all(bio_); bio_ = NULL;
    if(ctx_) SSL_CTX_free(ctx_); ctx_ = NULL;
    if(conn) BIO_free_all(conn);
    if(sbio) BIO_free_all(sbio);
    if(openssl_err.empty()) {
      last_error_ = err;
    } else {
      last_error_ = Status(err.GetCode(),err.GetDescription()+": "+openssl_err);
    };
    valid_ = false;
    return *(new IO(Context()));
  };
}

IONetwork::operator bool(void) const {
  return valid_;
}

bool IONetwork::operator!(void) const {
  return !valid_;
}

AuthN::Status IONetwork::WireWrite(const void *buffer, size_t size) {
  if(!bio_) return Status(-1);
  struct timeval till = time_inc(timeout_);
  for(;size > 0;) {
    int l = BIO_write(bio_, buffer, size);
    if(l <= 0) {
      if(!BIO_should_retry(bio_)) return Status(-1);
      if(!BIO_wait_activity(bio_,till)) return Status(CommunicationTimeout,"Timeout writing to remote peer");
    } else {
      buffer = ((char*)buffer) + l;
      size -= l;
    };
  };
  return Status();
/*
  if(socket_ < 0) return Status(-1,"Not valid");
  for(;size > 0;) {
#ifdef MSG_NOSIGNAL
    ssize_t l =send(socket_,buffer,size,MSG_NOSIGNAL);
#else
    ssize_t l =send(socket_,buffer,size,0);
#endif
    if(l < 0) {
      if(errno = EINTR) continue;
      close(socket_);
      socket_ = -1;
      valid_ = false;
      return Status(-1,"Failed");
    };
    buffer = ((char*)buffer) + l;
    size -= l;
  };
*/
}

AuthN::Status IONetwork::WireRead(void *buffer, size_t& size) {
  if(!bio_) return Status(-1);
  struct timeval till = time_inc(timeout_);
  for(;;) {
    int l = BIO_read(bio_, buffer, size);
    if(l <= 0) {
      if(BIO_should_retry(bio_)) {
        if(!BIO_wait_activity(bio_,till)) {
          size = 0;
          return (last_error_ = Status(CommunicationTimeout,"Timeout reading from remote peer"));
        };
      } else {
        // TODO: check for real error
        if(!BIO_wait_read(bio_,till)) {
          size = 0;
          return (last_error_ = Status(CommunicationTimeout,"Timeout reading from remote peer"));
        };
      };
    } else {
      size = l;
      break;
    };
  };
  return Status();
}

AuthN::Status IONetwork::WireClose(void) {
  if(bio_) BIO_free_all(bio_); bio_ = NULL;
  if(ctx_) SSL_CTX_free(ctx_); ctx_ = NULL;
  valid_ = false;
  return Status(-1);
}

// ----------------------------------------------------------------------------

IOSocket::IOSocket(const AuthN::Context& ctx):IO(ctx),ctx_(NULL),bio_(NULL) {
}

IOSocket::~IOSocket(void) {
  if(bio_) BIO_free_all(bio_);
  if(ctx_) SSL_CTX_free(ctx_);
}

AuthN::Status IOSocket::Connect(int socket) {
  //int err = SSL_ERROR_NONE;
  SSL* ssl = NULL;
  
  try {

  if(bio_ || ctx_) throw Status(-1);  

  ctx_=SSL_CTX_new(SSLv23_client_method());
  //ctx_=SSL_CTX_new(SSLv3_client_method());
  if(!ctx_) throw Status(-1);

  //set_ssl_ctx(ctx_,
  //     node_?node_->cert_:NULL,node_?node_->priv_key_:NULL,node_?node_->cert_chain_:NULL);
  set_ssl_ctx(ctx_, node_?node_->certctx_->cert_.cert : NULL, 
       node_?node_->certctx_->key_.pkey : NULL, node_?node_->certctx_->cert_.cert_chain : NULL);

  // Communication layer
  bio_ = BIO_new_ssl(ctx_,1);
  if(!bio_) throw Status(-1);
  BIO* bio_socket = BIO_new_socket(socket,0);
  if(!bio_socket) throw Status(-1);
  bio_ = BIO_push(bio_,bio_socket);
  BIO_get_ssl(bio_, &ssl);
  if(!ssl) throw Status(6);
  BIO_set_nbio(bio_socket,1);
  //if(!BIO_set_nbio(bio_socket,1)) throw Status(5);
  SSL_set_mode(ssl, SSL_MODE_AUTO_RETRY);
  struct timeval till = time_inc(timeout_);
  for(;;) {
    if(BIO_do_handshake(bio_) == 1) break;
    if(!BIO_should_retry(bio_)) return Status(-1);
    if(!BIO_wait_activity(bio_,till)) throw Status(CommunicationTimeout);
  };

  // Analyze peer's credentials
/*
  if(!(cert = SSL_get_peer_certificate(ssl))) throw Status(9);
  chain = SSL_get_peer_cert_chain(ssl);
  if(peer_) delete peer_;
  peer_ = new Credentials(Context());
  peer_->Assign(cert,chain,NULL);
*/
  peer_ = get_peer(ssl);
  Status valid = validator_->Validate(*peer_);
  if(!valid) throw valid;

  valid_ = true;
  return Status();

  } catch(Status& err) {
    std::string openssl_err = GetOpenSSLError();
    //HandleError(err);
    if(bio_) BIO_free_all(bio_); bio_ = NULL;
    if(ctx_) SSL_CTX_free(ctx_); ctx_ = NULL;
    valid_ = false;
    if(openssl_err.empty()) {
      last_error_ = err;
    } else {
      last_error_ = Status(err.GetCode(),err.GetDescription()+": "+openssl_err);
    };
    return last_error_;
  };
}

AuthN::Status IOSocket::Accept(int socket) {
  SSL *ssl;
  BIO* conn = NULL;

  try {

  if(bio_ || ctx_) throw Status(-1);  

  ctx_ = SSL_CTX_new(SSLv23_server_method());
  if(!ctx_) throw Status(-1);

  //set_ssl_ctx(ctx_,
  //    node_?node_->cert_:NULL,node_?node_->priv_key_:NULL,node_?node_->cert_chain_:NULL);
  set_ssl_ctx(ctx_, node_?node_->certctx_->cert_.cert : NULL, 
      node_?node_->certctx_->key_.pkey : NULL, node_?node_->certctx_->cert_.cert_chain : NULL);

  bio_ = BIO_new_ssl(ctx_,0);
  if(!bio_) throw Status(-1);
  BIO* bio_socket = BIO_new_socket(socket,0);
  if(!bio_socket) throw Status(-1);
  bio_ = BIO_push(bio_,bio_socket);
  BIO_get_ssl(bio_, &ssl);
  if(!ssl) throw Status(-1);
  BIO_set_nbio(conn,1); 
 //if(!BIO_set_nbio(conn,1)) throw Status(5);
  SSL_set_mode(ssl, SSL_MODE_AUTO_RETRY);
  struct timeval till = time_inc(timeout_);
  for(;;) {
    if(BIO_do_handshake(bio_) == 1) break;
    if(!BIO_should_retry(bio_)) return Status(-1);
    if(!BIO_wait_activity(bio_,till)) throw Status(CommunicationTimeout);
  };

  // Analyze peer's credentials
/*
  if(!(cert = SSL_get_peer_certificate(ssl))) throw Status(9);
  chain = SSL_get_peer_cert_chain(ssl);
  if(peer_) delete peer_;
  peer_ = new Credentials(Context());
  peer_->Assign(cert,chain,NULL);
*/
  peer_ = get_peer(ssl);
  Status valid = validator_->Validate(*peer_);
  if(!valid) throw valid;

  valid_ = true;
  return Status();

  } catch(Status& err) {
    std::string openssl_err = GetOpenSSLError();
    // TODO: not every error requires to kill accepting bio
    if(bio_) BIO_free_all(bio_); bio_ = NULL;
    if(ctx_) SSL_CTX_free(ctx_); ctx_ = NULL;
    if(openssl_err.empty()) {
      last_error_ = err;
    } else {
      last_error_ = Status(err.GetCode(),err.GetDescription()+": "+openssl_err);
    };
    valid_ = false;
    return last_error_;
  };
}

IOSocket::operator bool(void) const {
  return valid_;
}

bool IOSocket::operator!(void) const {
  return !valid_;
}

AuthN::Status IOSocket::WireWrite(const void *buffer, size_t size) {
  if(!bio_) return Status(-1);
  struct timeval till = time_inc(timeout_);
  for(;size > 0;) {
    int l = BIO_write(bio_, buffer, size);
    if(l <= 0) {
      if(!BIO_should_retry(bio_)) return Status(-1);
      if(!BIO_wait_activity(bio_,till)) return Status(CommunicationTimeout);
    } else {
      buffer = ((char*)buffer) + l;
      size -= l;
    };
  };
  return Status();
}

AuthN::Status IOSocket::WireRead(void *buffer, size_t& size) {
  if(!bio_) return Status(-1);
  struct timeval till = time_inc(timeout_);
  for(;;) {
    int l = BIO_read(bio_, buffer, size);
    if(l <= 0) {
      if(!BIO_should_retry(bio_)) {
        size = 0;
        return Status(-1);
      };
      if(!BIO_wait_activity(bio_,till)) {
        size = 0;
        return Status(CommunicationTimeout);
      };
    } else {
      size = l;
      break;
    };
  };
  return Status();
}

AuthN::Status IOSocket::WireClose(void) {
  if(bio_) BIO_free_all(bio_); bio_ = NULL;
  if(ctx_) SSL_CTX_free(ctx_); ctx_ = NULL;
  valid_ = false;
  return Status(-1);
}

} // namespace AuthN

