(*:Name: Geometry *)

(*:Title: Geometry *)

(*:Author: Tom Wickham-Jones*)

(*:Package Version: 1.0 *)

(*:Mathematica Version: 2.2 *)


(*:History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

(*:Summary:
	This package provides functions to perform a variety of
	two-dimensional geometric computations.
*)



BeginPackage[ "ExtendGraphics`Geometry`"]

ImplicitLine::usage =
	"ImplicitLine[ c, n] represents the implicit line that passes
	 through c and is normal to n."

NormalVector::usage = 
	"NormalVector[ v] returns the normal vector to v."
	
ToImplicit::usage = 
	"ToImplicit[ Line[ {p1, p2}]] returns the ImplicitLine
	 defined by Line[ {p1, p2}]."
	
ToLine::usage = 
	"ToLine[ ImplicitLine[ c, n]] returns the Line
	 defined by ImplicitLine[ c, n]."
	
VectorLength::usage = 
	"VectorLength[ v] returns the length of the vector v."
	
LineLength::usage = 
	"LineLength[ l] returns the length of the line l."
	
IntersectionPoint::usage = 
	"IntersectionPoint returns the point of
	intersection of various geometric objects."
	
PointOnLineQ::usage = 
	"PointOnLineQ[ p, i] tests whether the point p lies
	 on the implicit line i."
	
ClosestPointOnLine::usage = 
	"ClosestPointOnLine[ p, l] returns the point on the line
	 l closest to the point p."
	
SideOfLine::usage = 
	"SideOfLine[ p, l] tests whether the point p lies
	 on the right or the left or is collinear of the line l."

Collinear::usage =
	"Collinear is a possible result from SideOfLine."

PointInTriangle::usage = 
	"PointInTriangle[ p, tri] tests whether the point p lies
	 inside the triangle tri."
	
CenterOfGravity::usage = 
	"CenterOfGravity[ tri] returns the center of gravity of
	 the triangle tri."
	
Boundary::usage = 
	"Boundary[ tri] returns the boundary of the triangle tri."
	
Medians::usage = 
	"Medians[ tri] returns the medians of the triangle tri."
	
Perimeter::usage = 
	"Perimeter[ tri] returns the perimeter of the triangle tri."
	
Area::usage = 
	"Area[ tri] returns the area of the triangle tri."
	
InCircle::usage = 
	"InCircle[ tri] returns the inscribing circle of the triangle tri."
	
Bisector::usage = 
	"Bisector[ l1, l2] returns the bisector line of the lines
	 l1 and l2."
	
OutCircle::usage = 
	"OutCircle[ tri] returns the circumscribing circle 
	 of the triangle tri."	

PointInCircleQ::usage =
	"PointInCircleQ[ x, a, b, c] tests whether the point x
	 lies in the circle through a, b, c."

Begin[ "`Private`"]

NormalVector[ {px_, py_}] :=
    {py, -px}

ToImplicit[ Line[ {p1_, p2_}]] := 
	ImplicitLine[ p1, NormalVector[ p2-p1]]

ToLine[ ImplicitLine[ {cx_, cy_}, {nx_, ny_}]] :=
	Line[ {{cx, cy}, {cx, cy} + {-ny, nx}}]

VectorLength[ p_] :=
	Sqrt[ p.p]


LineLength[ Line[ {p1_, p2_}]] := 
	VectorLength[ p2-p1]

LineLength[ l_ImplicitLine] := 
	LineLength[ ToLine[l]]


IntersectionPoint[ Line[ {p1_, p2_}], ImplicitLine[ c_, n_]] :=
    Block[ {tst},
        tst = Chop[(p2 - p1).n] ;
        If[ tst === 0,
			Print[ "Parallel Lines"];
			{Infinity, Infinity},
	    	p1 - (p2 - p1) (p1 - c).n/tst
			]
		]

PointOnLineQ[ p_, ImplicitLine[ c_, n_]] :=
    Chop[(p-c).n] === 0

ClosestPointOnLine[ p_, ImplicitLine[ c_, n_]] :=
    p - (p-c).n/n.n n

ClosestPointOnLine[ p_, Line[ {p1_, p2_}]] :=
	ClosestPointOnLine[ p, ToImplicit[ Line[ {p1, p2}]]]

SideOfLine[ p_, Line[ {p1_, p2_}]] :=
	Block[{u, v, test},
		u = (p2 - p1) ;
		u = u/VectorLength[ u] ;
		v = (p - p1) ;
		v = v/VectorLength[ v] ;
		test = u.NormalVector[v] ;
		If[ N[test] > 10^(-15.),
			Left,
			If[ test < -10^(-15.),
				Right,
				Collinear]]
		]

PointInTriangle[ 
              p_, Polygon[ {p1_, p2_, p3_}]] :=
	Block[{test},
		test = 
		  Map[ SideOfLine[ p,Line[#]]&, {{p1,p2},{p2,p3},{p3,p1}}] ;
		If[ MemberQ[ test, Collinear],
				test = DeleteCases[ test, Collinear] ;
				If[ Apply[ SameQ, test],
						Collinear,
						False]
				,
				Apply[ SameQ, test]]
		]

CenterOfGravity[ Polygon[{ p1_, p2_, p3_}]] :=
	(p1+p2+p3)/3


Boundary[ Polygon[ {a_, b__}]] := Line[ {a,b,a}]

Medians[ Polygon[{p1_, p2_, p3_}]] :=
    {
	Line[ {p1, (p2+p3)/2}],
	Line[ {p2, (p3+p1)/2}],
	Line[ {p3, (p1+p2)/2}]
	}

Perimeter[ Polygon[ p_List]] :=
		Apply[ Plus, Map[ VectorLength, p - RotateLeft[p]]]


Area[ Polygon[ {{p1x_, p1y_}, {p2x_, p2y_}, {p3x_, p3y_}}]] :=
    Area[ Polygon[ {{p1x,p1y,0}, {p2x,p2y,0},{p3x,p3y,0}}]]
	
Area[ Polygon[ {p1_, p2_, p3_}]] :=
	Block[{res},
		res = Partition[{p1, p2, p3, p1}, 2,1] ;
		res = Map[ Cross[ #[[1]], #[[2]]]&, res] ;
		VectorLength[ Apply[ Plus, res]]/2
	]

InCircle[ t:Polygon[ {p1_, p2_, p3_}]] :=
	Block[{per, rad, cent},
		per = Perimeter[ t] ;
		rad = 2 Area[ t]/per ;
		cent =  VectorLength[ p2 - p3] p1 + 
				VectorLength[ p3 - p1] p2 +
				VectorLength[ p1 - p2] p3 ;
		cent = cent / per ;
		Circle[ cent, rad]
		]


Bisector[ Line[ {p_, p1_}], Line[ {p_, p2_}]] :=
	Block[{pn1, pn2, vl1, vl2},
		pn1 = p1 - p ;
		pn2 = p2 - p ;
		vl1 = VectorLength[ pn1] ;
		vl2 = VectorLength[ pn2] ;
		pn1 = pn1 * (vl1+vl2)/2/vl1 ;
		pn2 = pn2 * (vl1+vl2)/2/vl2 ;
		Line[ {p, p+(pn1+pn2)/2}]
		]

OutCircle[ t:Polygon[ {p1_, p2_, p3_}]] :=
	Block[{d1, d2, d3, 
		   c1, c2, c3, c,
		   rad, cent},
		d1 = (p3 - p1).(p2 - p1) ;
		d2 = (p1 - p2).(p3 - p2) ;
		d3 = (p2 - p3).(p1 - p3) ;
		c1 = d2 d3 ;
		c2 = d3 d1 ;
		c3 = d1 d2 ;
		c = c1 + c2 + c3 ;
		rad = Sqrt[ (d1+d2)(d2+d3)(d3+d1)/c]/2 ;
		cent = (c2+c3)p1 + (c3+c1)p2 + (c1+c2)p3 ;
		cent = cent/2/c;
		Circle[ cent, rad]
		]

PointInCircleQ[ 
    {xr_,xi_},{ar_,ai_},
    {br_,bi_},{cr_,ci_}] /;
                RightOfLineQ[ 
                    {br,bi}, 
                    Line[ {{ar,ai}, {cr,ci}}]] := 
    Module[{numr, numi, denr, deni},
        numr = (br - cr)(xr - ar) - (bi - ci)(xi - ai);
        numi = (br - cr)(xi - ai) + (bi - ci)(xr - ar);
        denr = (br - ar)(xr - cr) - (bi - ai)(xi - ci);
        deni = (br - ar)(xi - ci) + (bi - ai)(xr - cr);
        denr numi - deni numr > 0
        ]
		
PointInCircleQ[ x_, a_, b_, c_] := 
      PointInCircleQ[ x, a, c, b]

End[]

EndPackage[]


(*:Examples:

<<ExtendGraphics`Geometry`
		
*)

