/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "process_int.h"

void *
Complextorect(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  const auto COMPLEXTORECT = cdoOperatorAdd("complextorect", 0, 0, nullptr);
  const auto COMPLEXTOPOL = cdoOperatorAdd("complextopol", 0, 0, nullptr);

  const auto operatorID = cdoOperatorID();

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);
  const auto vlistID3 = vlistDuplicate(vlistID1);

  const auto nvars = vlistNvars(vlistID2);
  for (varID = 0; varID < nvars; ++varID)
    {
      auto datatype = vlistInqVarDatatype(vlistID2, varID);
      datatype = (datatype == CDI_DATATYPE_CPX64) ? CDI_DATATYPE_FLT64 : CDI_DATATYPE_FLT32;
      vlistDefVarDatatype(vlistID2, varID, datatype);
      vlistDefVarDatatype(vlistID3, varID, datatype);
    }

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  const auto taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);
  vlistDefTaxis(vlistID3, taxisID3);

  const auto streamID2 = cdoOpenWrite(1);
  const auto streamID3 = cdoOpenWrite(2);

  cdoDefVlist(streamID2, vlistID2);
  cdoDefVlist(streamID3, vlistID3);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> array1(2 * gridsizemax);
  Varray<double> array2(gridsizemax), array3(gridsizemax);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      taxisCopyTimestep(taxisID3, taxisID1);

      cdoDefTimestep(streamID2, tsID);
      cdoDefTimestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoDefRecord(streamID2, varID, levelID);
          cdoDefRecord(streamID3, varID, levelID);

          const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));

          cdoReadRecord(streamID1, array1.data(), &nmiss);

          const auto missval1 = vlistInqVarMissval(vlistID1, varID);
          const auto missval2 = missval1;

          if (operatorID == COMPLEXTORECT)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  array2[i] = array1[2 * i];
                  array3[i] = array1[2 * i + 1];
                }
            }
          else if (operatorID == COMPLEXTOPOL)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  array2[i] = SQRTMN(ADDMN(MULMN(array1[2 * i], array1[2 * i]), MULMN(array1[2 * i + 1], array1[2 * i + 1])));
                  array3[i] = (DBL_IS_EQUAL(array1[2 * i], missval1) || DBL_IS_EQUAL(array1[2 * i + 1], missval1))
                                  ? missval1
                                  : atan2(array1[2 * i + 1], array1[2 * i]);
                }
            }

          cdoWriteRecord(streamID2, array2.data(), nmiss);
          cdoWriteRecord(streamID3, array3.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);
  vlistDestroy(vlistID3);

  cdoFinish();

  return nullptr;
}
