/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <muse.h>
#include <string.h>

/*----------------------------------------------------------------------------*/
/**
  @brief    Test program to check that functions from muse_utils work when
            called with the necessary data.

  This program explicitely tests
    muse_utils_frameset_merge_frames()   (but only for trivial cases!)
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
  UNUSED_ARGUMENTS(argc, argv);
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

  /* Test trivial cases of muse_utils_frameset_merge_frames() */
  cpl_errorstate ps = cpl_errorstate_get();
  cpl_error_code rc = muse_utils_frameset_merge_frames(NULL, CPL_TRUE);
  cpl_test(!cpl_errorstate_is_equal(ps) && rc == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(ps);
  /* check with existing but empty frameset */
  cpl_frameset *frameset = cpl_frameset_new();
  rc = muse_utils_frameset_merge_frames(frameset, CPL_TRUE);
  cpl_test(!cpl_errorstate_is_equal(ps) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(ps);
  /* since testing on real files is too costly in development, *
   * make up a minimal frameset, first with one, then with two *
   * frames for very simple testing                            */
  cpl_frame *frame = cpl_frame_new();
  cpl_frame_set_filename(frame, "WAVECAL_TABLE-10.fits");
  cpl_frame_set_tag(frame, MUSE_TAG_WAVECAL_TABLE);
  cpl_frameset_insert(frameset, frame);
  rc = muse_utils_frameset_merge_frames(frameset, CPL_TRUE);
  cpl_test(rc == CPL_ERROR_NONE); /* nothing done (WARNING output) but OK */
  cpl_test_zero(strcmp(cpl_frame_get_filename(cpl_frameset_get_position(frameset, 0)),
                       "WAVECAL_TABLE-10.fits")); /* filename unchanged */
  /* now with 2nd frame */
  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, "WAVECAL_TABLE-11.fits");
  cpl_frame_set_tag(frame, MUSE_TAG_WAVECAL_TABLE);
  cpl_frameset_insert(frameset, frame);
  int errstate = errno;
  rc = muse_utils_frameset_merge_frames(frameset, CPL_TRUE);
  cpl_test_noneq(errstate, errno); /* the C error state has changed! */
  cpl_test(rc == CPL_ERROR_NONE); /* nothing done (WARNING output) but OK */
  cpl_test_zero(strcmp(cpl_frame_get_filename(cpl_frameset_get_position(frameset, 0)),
                       "WAVECAL_TABLE-10.fits")); /* filename unchanged */
  cpl_test_zero(strcmp(cpl_frame_get_filename(cpl_frameset_get_position(frameset, 1)),
                       "WAVECAL_TABLE-11.fits")); /* filename unchanged */
  cpl_frameset_dump(frameset, stdout);
  fflush(stdout);
  cpl_frameset_delete(frameset);

  /* now test the same with pixel table frames */
  frameset = cpl_frameset_new();
  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, "PIXTABLE_OBJECT-10.fits");
  cpl_frame_set_tag(frame, "PIXTABLE_OBJECT");
  cpl_frameset_insert(frameset, frame);
  rc = muse_utils_frameset_merge_frames(frameset, CPL_TRUE);
  cpl_test(rc == CPL_ERROR_NONE); /* nothing done (WARNING output) but OK */
  cpl_test_zero(strcmp(cpl_frame_get_filename(cpl_frameset_get_position(frameset, 0)),
                       "PIXTABLE_OBJECT-10.fits")); /* filename unchanged */

  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, "PIXTABLE_OBJECT-11.fits");
  cpl_frame_set_tag(frame, "PIXTABLE_OBJECT");
  cpl_frameset_insert(frameset, frame);
  errstate = errno;
  rc = muse_utils_frameset_merge_frames(frameset, CPL_TRUE);
  cpl_test_eq(errstate, errno); /* no change this time, since merging not done */
  cpl_test(rc == CPL_ERROR_NONE); /* nothing done (WARNING output) but OK */
  cpl_test_zero(strcmp(cpl_frame_get_filename(cpl_frameset_get_position(frameset, 0)),
                       "PIXTABLE_OBJECT-10.fits")); /* filename unchanged */
  cpl_test_zero(strcmp(cpl_frame_get_filename(cpl_frameset_get_position(frameset, 1)),
                       "PIXTABLE_OBJECT-11.fits")); /* filename unchanged */
  cpl_frameset_dump(frameset, stdout);
  fflush(stdout);
  cpl_frameset_delete(frameset);

  return cpl_test_end(0);
}
