package org.eclipse.webdav.internal.kernel.utils;

/*
 * (c) Copyright IBM Corp. 2000, 2001.
 * All Rights Reserved.
 */

import org.eclipse.webdav.internal.kernel.Policy;

/**
 * <code>Assert</code> is useful for for embedding runtime sanity checks
 * in code.
 * The predicate methods all test a condition and throw some
 * type of unchecked exception if the condition does not hold.
 * <p>
 * Assertion failure exceptions, like most runtime exceptions, are
 * thrown when something is misbehaving. Assertion failures are invariably
 * unspecified behavior; consequently, clients should never rely on
 * these being thrown (and certainly should not being catching them
 * specifically).
 * 
 */
public class Assert {
/** Block instantiation. */
private Assert() {
}
/** Asserts that an argument is legal. If the given boolean is
 * not <code>true</code>, an <code>IllegalArgumentException</code>
 * is thrown.
 *
 * @exception IllegalArgumentException Thrown if the legality test failed
 */
static public boolean isLegal(boolean expression) {
	return isLegal(expression, "");
}
/** Asserts that an argument is legal. If the given boolean is
 * not <code>true</code>, an <code>IllegalArgumentException</code>
 * is thrown.
 * The given message is included in that exception, to aid debugging.
 *
 * @exception IllegalArgumentException Thrown if the legality test failed
 */
static public boolean isLegal(boolean expression, String message) {
	if (!expression)
		throw new IllegalArgumentException();
	return expression;
}
/** Asserts that the given object is not <code>null</code>. If this
 * is not the case, some kind of unchecked exception is thrown.
 */
static public void isNotNull(Object o) {
	isNotNull(o, "");
}
/** Asserts that the given object is not <code>null</code>. If this
 * is not the case, some kind of unchecked exception is thrown.
 * The given message is included in that exception, to aid debugging.
 */
static public void isNotNull(Object o, String message) {
	if (o == null)
		throw new AssertionFailedException(Policy.bind("assert.null",message));
}
/** Asserts that the given boolean is <code>true</code>. If this
 * is not the case, some kind of unchecked exception is thrown.
 */
static public boolean isTrue(boolean expression) {
	return isTrue(expression, "");
}
/** Asserts that the given boolean is <code>true</code>. If this
 * is not the case, some kind of unchecked exception is thrown.
 * The given message is included in that exception, to aid debugging.
 */
static public boolean isTrue(boolean expression, String message) {
	if (!expression)
		throw new AssertionFailedException(Policy.bind("assert.fail",message));
	return expression;
}
}
