/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Tone mapping function from this paper:
 *   Erik Reinhard and Kate Devlin. Dynamic Range Reduction Inspired by
 *   Photoreceptor Physiology.  IEEE Transactions on Visualization and
 *   Computer Graphics (2004).
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <limits.h>
#include <values.h>
#include <getopt.h>
#include "exrinput.h"
#include "exroutput.h"

/**
 * Luminance function for sRGB.
 */
static double luminance( double r, double g, double b )
{
    return ( 0.2126 * r ) + ( 0.7152 * g ) + ( 0.0722 * b );
}

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    exrinput_t *exrin;
    double *scanline;
    int width, height;
    double m, f, c, a;
    double llav, lav, minl, maxl;
    double rav, gav, bav;
    int option_index = 0;
    char cur;
    int y;

    f = 0.0;
    m = 0.0;
    c = 0.0;
    a = 1.0;

    if( argc < 3 ) {
        fprintf( stderr, "Usage: exrpptm [-f fval] [-m mval] "
                         "[-c cval] [-a aval] input.exr output.exr\n" );
        return 1;
    }

    while( (cur = getopt( argc, argv, "f:m:c:a:" )) != -1 ) {
        switch( cur ) {
        case 'f': f = atof( optarg ); break;
        case 'm': m = atof( optarg ); break;
        case 'c': c = atof( optarg ); break;
        case 'a': a = atof( optarg ); break;
        default:
            fprintf( stderr, "Usage: exrpptm input.exr output.exr\n" );
            return 1;
        }
    }

    exrin = exrinput_new( argv[ optind ] );
    if( !exrin ) {
        fprintf( stderr, "exrpptm: Cannot open %s.\n", argv[ optind ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exrpptm: %s [%dx%d]\n", argv[ optind ], width, height );
    optind++;

    scanline = malloc( width * 4 * sizeof( double ) );
    if( !scanline ) {
        fprintf( stderr, "exrpptm: Cannot allocate memory.\n" );
        exrinput_delete( exrin );
        return 1;
    }

    exrout = exroutput_new( argv[ optind ], width, height );
    if( !exrout ) {
        exrinput_delete( exrin );
        fprintf( stderr, "exrpptm: Cannot open %s.\n", argv[ optind ] );
        return 1;
    }

    if( f < -8.0 ) f = -8.0; if( f > 8.0 ) f = 8.0;
    if( m <  0.0 ) m =  0.0; if( m > 1.0 ) m = 1.0;
    if( c <  0.0 ) c =  0.0; if( c > 1.0 ) c = 1.0;
    if( a <  0.0 ) a =  0.0; if( a > 1.0 ) a = 1.0;

    fprintf( stderr, "exrpptm: Using f=%.3f, m=%.3f, c=%.3f and a=%.3f\n",
             f, m, c, a );

    maxl = -DBL_MAX;
    minl = DBL_MAX;
    lav  = rav = gav = bav = 0.0;

    for( y = 0; y < height; y++ ) {
        double *input = exrinput_get_scanline( exrin, y );
        int x;

        fprintf( stderr, "exrpptm: Scanning %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r = input[ (x * 4) + 0 ];
            double g = input[ (x * 4) + 1 ];
            double b = input[ (x * 4) + 2 ];
            double Y = luminance( r, g, b );

            rav += r;
            gav += g;
            bav += b;
            lav += Y;

            if( Y > 0.0 ) {
                llav += log( Y );
                if( Y < minl ) minl = Y; if( Y > maxl ) maxl = Y;
            }
        }
    }
    llav /= (double) (width * height);
    lav /= (double) (width * height);
    rav /= (double) (width * height);
    gav /= (double) (width * height);
    bav /= (double) (width * height);

    fprintf( stderr, "\nexrpptm: Luminance min is %7.3f, avg %7.3f, max %7.3f\n",
             minl, lav, maxl );
    fprintf( stderr, "exrpptm: Luminance log min is %7.3f, avg log %7.3f, log max %7.3f\n",
             log( minl ), llav, log( maxl ) );

    f = exp( -f );
    if( m == 0.0 ) m = 0.3 + 0.7 * pow((log(maxl) - llav) / (log(maxl) - log(minl)), 1.4);

    for( y = 0; y < height; y++ ) {
        double *input = exrinput_get_scanline( exrin, y );
        int x;

        fprintf( stderr, "exrpptm: Correcting scanline %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r, g, b;
            double Y;
            double ig, il, ia;

            r = input[ (x * 4) + 0 ];
            g = input[ (x * 4) + 1 ];
            b = input[ (x * 4) + 2 ];

            Y = luminance( r, g, b );

            ig = ((1.0 - c) * lav) + (c * rav);
            il = ((1.0 - c) *   Y) + (c * r);
            ia = ((1.0 - a) *  ig) + (a * il);
            r /= r + pow( f * ia, m );

            ig = ((1.0 - c) * lav) + (c * gav);
            il = ((1.0 - c) *   Y) + (c * g);
            ia = ((1.0 - a) *  ig) + (a * il);
            g /= g + pow( f * ia, m );

            ig = ((1.0 - c) * lav) + (c * bav);
            il = ((1.0 - c) *   Y) + (c * b);
            ia = ((1.0 - a) *  ig) + (a * il);
            b /= b + pow( f * ia, m );

            scanline[ (x * 4) + 0 ] = r;
            scanline[ (x * 4) + 1 ] = g;
            scanline[ (x * 4) + 2 ] = b;
            scanline[ (x * 4) + 3 ] = input[ (x * 4) + 3 ];
        }
        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\nexrpptm: Done.\n" );

    exroutput_delete( exrout );
    exrinput_delete( exrin );
    return 0;
}

