/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_config.h>
#include <globus_io.h>
#include "globus_rls_client.h"
#include "misc.h"
#include <stdarg.h>
#include <termios.h>
#include <string.h>

#define GLOBUS_I_RLS_LOG_MASK   255

extern char	                    *pidfile;
extern char                     *logtype;
static int	                    detached        = 0;
static globus_logging_handle_t  log_handle      = NULL;

/*
 * writepid - Write pid to file.  Globus isn't activated yet so use standard
 *   C lib functions.
 */
void
writepid(char *prog)

{
  FILE	*f;

  if ((f = fopen(pidfile, "w")) == NULL)
    logit(LOG_WARNING, "writepid: Can't write pid file %s", pidfile);
  else {
    fchmod(fileno(f), 0644);
    fprintf(f, "%d\n", (int) getpid());
    fclose(f);
  }
}

/*
 * detach - Detach from controlling tty, open syslog.  Globus isn't activated
 *   yet so use C lib functions.
 */
void
detach(char *prog)

{
  int	i;

  if (fork())
    exit(0);
#ifndef TIOCNOTTY
  setsid();
#else
  if ((i = open("/dev/tty", O_RDONLY)) >= 0)
    ioctl(i, TIOCNOTTY, 0);
#endif
  for (i = 0; i < 10; i++)	/* Close existing file descriptors	*/
    close(i);
  open("/dev/null", 0);
  dup2(0, 1);
  dup2(0, 2);
  detached++;
  openlog(prog, LOG_PID, LOG_DAEMON);
}

/*
 * log - initialize
 */
globus_result_t
log_init()
{
    globus_result_t res;
    if (!detached)
    {
        res = globus_logging_init(
                &log_handle,
                NULL,
                1024,
                GLOBUS_I_RLS_LOG_MASK,
                &globus_logging_stdio_module,
                stdout);
    }
    else if (logtype && strcasecmp(logtype, "syslog-ng") == 0)
    {
        res = globus_logging_init(
                &log_handle,
                NULL,
                1024,
                GLOBUS_I_RLS_LOG_MASK,
                &globus_logging_syslog_ng_module,
                NULL);
    }
    else
    {
        res = globus_logging_init(
                &log_handle,
                NULL,
                1024,
                GLOBUS_I_RLS_LOG_MASK,
                &globus_logging_syslog_module,
                NULL);
    }
    return res;
}

/*
 * log - destroy
 */
globus_result_t
log_destroy()
{
    return (log_handle) ? globus_logging_destroy(log_handle) : GLOBUS_SUCCESS;
}

/*
 * log - Write log message.
 */
void
logit(int level, char *fmt, ...)

{
  va_list	ap;
  char		buf[BUFLEN];
  char		tbuf[100];

  va_start(ap, fmt);
  vsnprintf(buf, BUFLEN, fmt, ap);
  globus_logging_write(log_handle, level,
          "T%lu: %s%s", (unsigned long int) pthread_self(), buf,
          (detached)?"":"\n");
  va_end(ap);
}

char *
globuserr(globus_result_t r, char *errmsg)

{
  globus_object_t	*eo;
  char			*s;

  if ((eo = globus_error_get(r)) == NULL)
    return "unknown globus error";
  s = globus_error_print_friendly(eo);
  globus_object_free(eo);
  if (!s)
    return "unknown globus error";
  strncpy(errmsg, s, MAXERRMSG);
  globus_libc_free(s);
  return errmsg;
}
