-- with Ada.Text_IO;

with Interfaces.C;
with Interfaces.C.Strings;
with Interfaces.C.Pointers;

with System;
with Ada.Exceptions;

package body GNU.DB.SQLite3 is


   package C   renames Interfaces.C;
   package CS  renames Interfaces.C.Strings;

   ----------------------------------------------------------------------------
   -- local support -----------------------------------------------------------
   ----------------------------------------------------------------------------

   package C_Wide_Strings is
     new C.Pointers (C.size_t,
                     C.wchar_t,
                     C.wchar_array,
                     C.wide_nul);
   package CWS renames C_Wide_Strings;

   function To_Ada
     (Thing : CWS.Pointer)
      return Wide_String
   is
   begin
      return C.To_Ada (CWS.Value (Thing));
   end To_Ada;
   pragma Inline (To_Ada);

   function To_Ada
     (Thing : CS.chars_ptr)
      return String
   is
   begin
      return CS.Value (Thing);
   end To_Ada;
   pragma Inline (To_Ada);


   function To_C
     (Thing : Wide_String)
      return C.wchar_array
   is
   begin
      return C.To_C (Thing) (0 .. C.size_t (Thing'Length));
   end To_C;
   pragma Inline (To_C);
   type Error_Description is record
      Id    : Ada.Exceptions.Exception_Id;
      Code  : Return_Value;
      Msg   : access String;
   end record;
   Retcode_2_Exception_Map : constant array (Return_Value) of Error_Description :=
                               (SQLITE_OK         => (Ada.Exceptions.Null_Id, SQLITE_OK, new String'("")),
                                SQLITE_ERROR      => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_INTERNAL   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_PERM       => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_ABORT      => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_BUSY       => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_LOCKED     => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_NOMEM      => (Storage_Error'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_READONLY   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_INTERRUPT  => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_IOERR      => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_CORRUPT    => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_NOTFOUND   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_FULL       => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_CANTOPEN   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_PROTOCOL   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_EMPTY      => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_SCHEMA     => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_TOOBIG     => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_CONSTRAINT => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_MISMATCH   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_MISUSE     => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_NOLFS      => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_AUTH       => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_FORMAT     => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_RANGE      => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_NOTADB     => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_ROW        => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                SQLITE_DONE       => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")),
                                UNDEFINED_ERROR   => (SQLITE_Exception'Identity, SQLITE_ERROR, new String'("SQL error or missing database")));
   procedure Retcode_2_Exception (Code : Return_Value; Info : String := "") is
   begin
      if Code /= SQLITE_OK then
         Ada.Exceptions.Raise_Exception (Retcode_2_Exception_Map (Code).Id,
                                         "[" & Code'Img & "] " &
                                         Retcode_2_Exception_Map (Code).Msg.all &
                                         " " & Info);
      end if;
   end Retcode_2_Exception;

   function Int_To_Ret_Val
     (Rc : Integer)
      return Return_Value
   is
   begin
      case Rc is
         when  0           => return SQLITE_OK;
         when  1           => return SQLITE_ERROR;
         when  2           => return SQLITE_INTERNAL;
         when  3           => return SQLITE_PERM;
         when  4           => return SQLITE_ABORT;
         when  5           => return SQLITE_BUSY;
         when  6           => return SQLITE_LOCKED;
         when  7           => return SQLITE_NOMEM;
         when  8           => return SQLITE_READONLY;
         when  9           => return SQLITE_INTERRUPT;
         when 10           => return SQLITE_IOERR;
         when 11           => return SQLITE_CORRUPT;
         when 12           => return SQLITE_NOTFOUND;
         when 13           => return SQLITE_FULL;
         when 14           => return SQLITE_CANTOPEN;
         when 15           => return SQLITE_PROTOCOL;
         when 16           => return SQLITE_EMPTY;
         when 17           => return SQLITE_SCHEMA;
         when 18           => return SQLITE_TOOBIG;
         when 19           => return SQLITE_CONSTRAINT;
         when 20           => return SQLITE_MISMATCH;
         when 21           => return SQLITE_MISUSE;
         when 22           => return SQLITE_NOLFS;
         when 23           => return SQLITE_AUTH;
         when 24           => return SQLITE_FORMAT;
         when 25           => return SQLITE_RANGE;
         when 26           => return SQLITE_NOTADB;
         when 100          => return SQLITE_ROW;
         when 101          => return SQLITE_DONE;
         when others       => return UNDEFINED_ERROR;
      end case;
   end Int_To_Ret_Val;

   function Int_To_Ret_Val
     (int : C.int)
      return Return_Value
   is
   begin
      return Int_To_Ret_Val (Integer (int));
   end Int_To_Ret_Val;

   -- pragma Import(C, SQLITE_STATIC, "SQLITE_STATIC");

   ----------------------------------------------------------------------------
   -- body --------------------------------------------------------------------
   ----------------------------------------------------------------------------

   -----------------
   -- bind_double --
   -----------------

   function Bind_Double
     (Stmt : access Statement;
      N    : Integer;
      D    : Long_Float)
      return Return_Value
   is

      function Sqlite3_Bind_Double
        (Stmt : Stmt_Access;
         N    : C.int;
         D    : C.double)
         return C.int;
      pragma Import (C, Sqlite3_Bind_Double, "sqlite3_bind_double");
      -- sqlite3.h:627
      -- int sqlite3_bind_double(sqlite3_stmt*, int, double);

   begin

      return Int_To_Ret_Val (Sqlite3_Bind_Double (Stmt.Stmt, C.int (N), C.double (D)));

   end Bind_Double;

   --------------
   -- bind_int --
   --------------

   function Bind_Int
     (Stmt : access Statement;
      N    : Integer;
      I    : Integer)
      return Return_Value
   is

      function Sqlite3_Bind_Int
        (Stmt : Stmt_Access;
         N    : C.int;
         I    : C.int)
         return C.int;
      pragma Import (C, Sqlite3_Bind_Int, "sqlite3_bind_int");
      -- sqlite3.h:628
      -- int sqlite3_bind_int(sqlite3_stmt*, int, int);

   begin

      return Int_To_Ret_Val (Sqlite3_Bind_Int (Stmt.Stmt, C.int (N), C.int (I)));

   end Bind_Int;

   ----------------
   -- bind_int64 --
   ----------------

   function Bind_Int64
     (Stmt : access Statement;
      N    : Integer;
      I    : Int64)
      return Return_Value
   is

      function Sqlite3_Bind_Int64
        (Stmt : Stmt_Access;
         N    : C.int;
         I    : Int64)
         return C.int;
      pragma Import (C, Sqlite3_Bind_Int64, "sqlite3_bind_int64");
      -- sqlite3.h:629
      -- int sqlite3_bind_int64(sqlite3_stmt*, int, sqlite_int64);

   begin

      return Int_To_Ret_Val (Sqlite3_Bind_Int64 (Stmt.Stmt, C.int (N), I));

   end Bind_Int64;

   ---------------
   -- bind_null --
   ---------------

   function Bind_Null
     (Stmt : access Statement;
      N    : Integer)
      return Return_Value
   is

      function Sqlite3_Bind_Null
        (Stmt : Stmt_Access;
         N    : C.int)
         return C.int;
      pragma Import (C, Sqlite3_Bind_Null, "sqlite3_bind_null");
      -- sqlite3.h:630
      -- int sqlite3_bind_null(sqlite3_stmt*, int);

   begin

      return Int_To_Ret_Val (Sqlite3_Bind_Null (Stmt.Stmt, C.int (N)));

   end Bind_Null;

   ---------------
   -- bind_text --
   ---------------

   function Bind_Text
     (Stmt : access Statement;
      N    : Integer;
      T    : String)
      return Return_Value
   is

      function Sqlite3_Bind_Text
        (Stmt : Stmt_Access;
         N    : C.int;
         T    : C.char_array;
         N2   : C.int;
         A    : System.Address)
         return C.int;
      pragma Import (C, Sqlite3_Bind_Text, "sqlite3_bind_text");
      -- sqlite3.h:631
      -- int sqlite3_bind_text(sqlite3_stmt*, int, const char*, int n, void(*)(void*));

   begin

      return Int_To_Ret_Val (Sqlite3_Bind_Text (
        Stmt.Stmt,
        C.int (N),
        C.To_C (T),
        0,
        System.Null_Address --SQLITE_STATIC
       ));

   end Bind_Text;

   -----------------
   -- bind_text16 --
   -----------------

   function Bind_Text16
     (Stmt : access Statement;
      N    : Integer;
      T    : Wide_String)
      return Return_Value
   is

      function Sqlite3_Bind_Text16
        (Stmt : Stmt_Access;
         N    : C.int;
         T    : C.wchar_array;
         N2   : C.int;
         A    : System.Address)
         return C.int;
      pragma Import (C, Sqlite3_Bind_Text16, "sqlite3_bind_text16");
      -- sqlite3.h:632
      -- int sqlite3_bind_text16(sqlite3_stmt*, int, const void*, int, void(*)(void*));

   begin

      return Int_To_Ret_Val (Sqlite3_Bind_Text16 (
        Stmt.Stmt,
        C.int (N),
        To_C (T),
        0,
        System.Null_Address
       ));

   end Bind_Text16;

   -----------
   -- close --
   -----------

   function Close
     (Self : access Object)
      return Return_Value
   is

      function Sqlite3_Close
        (Db : DB_Access)
         return C.int;
      pragma Import (C, Sqlite3_Close, "sqlite3_close");
      -- sqlite3.h:75
      -- int sqlite3_close(sqlite3 *);

      Return_Code : Return_Value;

   begin

      Return_Code := Int_To_Ret_Val (Sqlite3_Close (Self.Db));

      if (Return_Code = SQLITE_OK) then
         Self.Db := null;
      end if;

      return Return_Code;

   end Close;
   procedure Close (Self : access Object) is
   begin
      Retcode_2_Exception (Self.Close);
   end Close;
   -------------
   -- Changes --
   -------------

   function Changes
     (Self : access Object)
      return Integer
   is

      function Sqlite3_Changes
        (Db : DB_Access)
         return C.int;
      pragma Import (C, Sqlite3_Changes, "sqlite3_changes");
      -- sqlite3.h:194
      -- int sqlite3_changes(sqlite3*);

   begin

      return Integer (Sqlite3_Changes (Self.Db));

   end Changes;

   ------------------
   -- column_count --
   ------------------

   function Column_Count
     (Stmt : access Statement)
      return Integer
   is

      function Sqlite3_Column_Count
        (Stmt : Stmt_Access)
         return C.int;
      pragma Import (C, Sqlite3_Column_Count, "sqlite3_column_count");
      -- sqlite3.h:661
      -- int sqlite3_column_count(sqlite3_stmt *pStmt);

   begin

      return Integer (Sqlite3_Column_Count (Stmt.Stmt));

   end Column_Count;

   ------------------
   -- column_count --
   ------------------

   function Column_Count
     (Self : Table_Reference;
      Row  : Natural)
      return Natural
   is

      use String_Vectors;
      use String_Tables;

   begin

      if Row <= Row_Count (Self) then
         return Natural (Length (Element (Self.all, Row)));
      end if;

      return 0;

   end Column_Count;

   ---------------------
   -- column_decltype --
   ---------------------

   function Column_Decltype
     (Stmt : access Statement;
      ICol : Integer)
      return String
   is
      function Sqlite3_Column_Decltype
        (Stmt : Stmt_Access;
         ICol : C.int)
         return CS.chars_ptr;
      pragma Import (C, Sqlite3_Column_Decltype, "sqlite3_column_decltype");
      -- sqlite3.h:690
      -- const char *sqlite3_column_decltype(sqlite3_stmt *, int i);

   begin

      return To_Ada (Sqlite3_Column_Decltype (Stmt.Stmt, C.int (ICol)));

   end Column_Decltype;

   -----------------------
   -- column_decltype16 --
   -----------------------

   function Column_Decltype16
     (Stmt : access Statement;
      ICol : Integer)
      return Wide_String
   is

      function Sqlite3_Column_Decltype16
        (Stmt : Stmt_Access;
         ICol : C.int)
         return CWS.Pointer;
      pragma Import (C, Sqlite3_Column_Decltype16, "sqlite3_column_decltype16");
      -- sqlite3.h:710
      -- const void *sqlite3_column_decltype16(sqlite3_stmt*,int);

   begin

      return To_Ada (Sqlite3_Column_Decltype16 (Stmt.Stmt, C.int (ICol)));

   end Column_Decltype16;

   ----------------
   -- column_int --
   ----------------

   function Column_Int
     (Stmt : access Statement;
      ICol : Integer)
      return Integer
   is

      function Sqlite3_Column_Int
        (Stmt : Stmt_Access;
         ICol : C.int)
         return C.int;
      pragma Import (C, Sqlite3_Column_Int, "sqlite3_column_int");
      -- sqlite3.h:828
      -- int sqlite3_column_int(sqlite3_stmt*, int iCol);

   begin

      return Integer (Sqlite3_Column_Int (Stmt.Stmt, C.int (ICol)));

   end Column_Int;

   -----------------
   -- column_text --
   -----------------

   function Column_Text
     (Stmt : access Statement;
      ICol : Integer)
      return String
   is

      function Sqlite3_Column_Text
        (Stmt : Stmt_Access;
         ICol : C.int)
         return CS.chars_ptr;
      pragma Import (C, Sqlite3_Column_Text, "sqlite3_column_text");
      -- sqlite3.h:830
      -- const unsigned char *sqlite3_column_text(sqlite3_stmt*, int iCol);

   begin

      return To_Ada (Sqlite3_Column_Text (Stmt.Stmt, C.int (ICol)));

   end Column_Text;

   -------------------
   -- column_text16 --
   -------------------

   function Column_Text16
     (Stmt : access Statement;
      ICol : Integer)
      return Wide_String
   is

      function Sqlite3_Column_Text16
        (Stmt : Stmt_Access;
         ICol : C.int)
         return CWS.Pointer;
      pragma Import (C, Sqlite3_Column_Text16, "sqlite3_column_text16");
      -- sqlite3.h:831
      -- const void *sqlite3_column_text16(sqlite3_stmt*, int iCol);

   begin

      return To_Ada (Sqlite3_Column_Text16 (Stmt.Stmt, C.int (ICol)));

   end Column_Text16;

   -----------------
   -- column_name --
   -----------------

   function Column_Name
     (Stmt : access Statement;
      ICol : Integer)
      return String
   is

      function Sqlite3_Column_Name
        (Stmt : Stmt_Access;
         ICol : C.int)
         return CS.chars_ptr;
      pragma Import (C, Sqlite3_Column_Name, "sqlite3_column_name");
      -- sqlite3.h:669
      -- const char *sqlite3_column_name(sqlite3_stmt*,int);

   begin

      return To_Ada (Sqlite3_Column_Name (Stmt.Stmt, C.int (ICol)));

   end Column_Name;

   -------------------
   -- column_name16 --
   -------------------

   function Column_Name16
     (Stmt : access Statement;
      ICol : Integer)
      return Wide_String
   is

      function Sqlite3_Column_Name16
        (Stmt : Stmt_Access;
         ICol : C.int)
         return CWS.Pointer;
      pragma Import (C, Sqlite3_Column_Name16, "sqlite3_column_name16");
      -- sqlite3.h:670
      -- const void *sqlite3_column_name16(sqlite3_stmt*,int);

   begin

      return To_Ada (Sqlite3_Column_Name16 (Stmt.Stmt, C.int (ICol)));

   end Column_Name16;

   -----------------
   -- column_type --
   -----------------

   function Column_Type
     (Stmt : access Statement;
      ICol : Integer)
      return Integer
   is

      function Sqlite3_Column_Type
        (Stmt : Stmt_Access;
         ICol : C.int)
         return C.int;
      pragma Import (C, Sqlite3_Column_Type, "sqlite3_column_type");
      -- sqlite3.h:832
      -- int sqlite3_column_type(sqlite3_stmt*, int iCol);

   begin

      return Integer (Sqlite3_Column_Type (Stmt.Stmt, C.int (ICol)));

   end Column_Type;

   --------------
   -- Complete --
   --------------

   function Complete
     (Sql : String)
      return Boolean
   is

      function Sqlite3_Complete
        (Sql : C.char_array)
         return C.int;
      pragma Import (C, Sqlite3_Complete, "sqlite3_complete");
      -- sqlite3.h:233
      -- int sqlite3_complete(const char *sql);

   begin

      if (Integer (Sqlite3_Complete (C.To_C (Sql))) = 0) then
         return False;
      else
         return True;
      end if;

   end Complete;

   ----------------
   -- Complete16 --
   ----------------

   function Complete16
     (Sql : Wide_String)
      return Boolean
   is

      function Sqlite3_Complete16
        (Sql : C.wchar_array)
         return C.int;
      pragma Import (C, Sqlite3_Complete16, "sqlite3_complete16");
      -- sqlite3.h:234
      -- int sqlite3_complete16(const void *sql);

   begin

      if (Integer (Sqlite3_Complete16 (C.To_C (Sql))) = 0) then
         return False;
      else
         return True;
      end if;

   end Complete16;

   ----------------
   -- data_count --
   ----------------

   function Data_Count
     (Stmt : access Statement)
      return Integer
   is

      function Sqlite3_Data_Count
        (Stmt : Stmt_Access)
         return C.int;
      pragma Import (C, Sqlite3_Data_Count, "sqlite3_data_count");
      -- sqlite3.h:755
      -- int sqlite3_data_count(sqlite3_stmt *pStmt);

   begin

      return Integer (Sqlite3_Data_Count (Stmt.Stmt));

   end Data_Count;

   -------------
   -- element --
   -------------

   function Element
     (Self : Table_Reference;
      Row  : Natural;
      Col  : Natural)
      return String
   is

      use String_Tables;
      use String_Vectors;

   begin

      return Element (Element (Self.all, Row), Col);

   end Element;

   ---------------
   -- element16 --
   ---------------

   function Element16
     (Self : Table16_Reference;
      Row  : Natural;
      Col  : Natural)
      return Wide_String
   is

      use Wide_String_Tables;
      use Wide_String_Vectors;

   begin

      return Element (Element (Self.all, Row), Col);

   end Element16;

   -------------
   -- errcode --
   -------------

   function Errcode
     (Self : access Object)
      return Integer
   is

      function Sqlite3_Errcode
        (Db : DB_Access)
         return C.int;
      pragma Import (C, Sqlite3_Errcode, "sqlite3_errcode");
      -- sqlite3.h:528
      -- int sqlite3_errcode(sqlite3 *db);

   begin

      return Integer (Sqlite3_Errcode (Self.Db));

   end Errcode;

   ------------
   -- errmsg --
   ------------

   function Errmsg
     (Self : access Object)
      return String
   is

      function Sqlite3_Errmsg
        (Db : DB_Access)
         return CS.chars_ptr;
      pragma Import (C, Sqlite3_Errmsg, "sqlite3_errmsg");
      -- sqlite3.h:538
      -- const char *sqlite3_errmsg(sqlite3*);

   begin

      return To_Ada (Sqlite3_Errmsg (Self.Db));

   end Errmsg;

   --------------
   -- errmsg16 --
   --------------

   function Errmsg16
     (Self : access Object)
      return Wide_String
   is

      function Sqlite3_Errmsg16
        (Db : DB_Access)
         return CWS.Pointer;
      pragma Import (C, Sqlite3_Errmsg16, "sqlite3_errmsg16");
      -- sqlite3.h:548
      -- const void *sqlite3_errmsg16(sqlite3*);

   begin

      return To_Ada (Sqlite3_Errmsg16 (Self.Db));

   end Errmsg16;

   ----------
   -- exec --
   ----------

   function Exec
     (Self : access Object;
      Sql  : String;
      Cb   : Callback)
      return Return_Value
   is

      function Proxy
        (PArg        : Handle;
         Argc        : Integer;
         Argv        : String_Vectors.Vector;
         ColumnNames : String_Vectors.Vector)
         return Return_Value
      is
         pragma Unreferenced (PArg);
      begin
         return Cb (Argc - 1, Argv, ColumnNames);
      end Proxy;
      pragma Inline (Proxy);

      package Real_Exec
      is new Generic_Exec (Object, Handle);

   begin

      return Real_Exec.Exec (Self, Sql, Proxy'Access, null);

   end Exec;
   procedure Exec
     (Self : access Object;
      Sql  : String;
      Cb   : Callback) is
   begin
      Retcode_2_Exception (Self.Exec (Sql, Cb));
   end Exec;
   ------------
   -- exec16 --
   ------------

   function Exec16
     (Self : access Object;
      Sql  : Wide_String;
      Cb   : Callback16)
      return Return_Value
   is

      function Proxy
        (PArg        : Handle;
         Argc        : Integer;
         Argv        : Wide_String_Vectors.Vector;
         ColumnNames : Wide_String_Vectors.Vector)
         return Return_Value
      is
         pragma Unreferenced (PArg);
      begin
         return Cb (Argc, Argv, ColumnNames);
      end Proxy;
      pragma Inline (Proxy);

      package Real_Exec16
      is new Generic_Exec16 (Object, Handle);

   begin

      return Real_Exec16.Exec16 (Self, Sql, Proxy'Access, null);

   end Exec16;

   ----------
   -- exec --
   ----------

   function Exec
     (Self : access Object;
      Sql  : String)
      return Return_Value
   is

      function Nothing
        (Argc        : Integer;
         Argv        : String_Vectors.Vector;
         ColumnNames : String_Vectors.Vector)
         return Return_Value
      is
         pragma Unreferenced (Argc, Argv, ColumnNames);
      begin
         return SQLITE_OK;
      end Nothing;
      pragma Inline (Nothing);

   begin

      return Exec (Self, Sql, Nothing'Unrestricted_Access);

   end Exec;
   procedure Exec
     (Self : access Object;
      Sql  : String) is
   begin
      Retcode_2_Exception (Self.Exec (Sql), Sql);
   end Exec;
   ------------
   -- exec16 --
   ------------

   function Exec16
     (Self : access Object;
      Sql  : Wide_String)
      return Return_Value
   is

      function Nothing
        (Argc        : Integer;
         Argv        : Wide_String_Vectors.Vector;
         ColumnNames : Wide_String_Vectors.Vector)
         return Return_Value
      is
         pragma Unreferenced (Argc, Argv, ColumnNames);
      begin
         return SQLITE_OK;
      end Nothing;
      pragma Inline (Nothing);

   begin

      return Exec16 (Self, Sql, Nothing'Unrestricted_Access);

   end Exec16;

   --------------
   -- finalize --
   --------------

   function Finalize
     (Stmt : access Statement)
      return Return_Value
   is

      function Sqlite3_Finalize
        (Stmt : Stmt_Access)
         return C.int;
      pragma Import (C, Sqlite3_Finalize, "sqlite3_finalize");
      -- sqlite3.h:848
      -- int sqlite3_finalize(sqlite3_stmt *pStmt);

   begin
      return Int_To_Ret_Val (Sqlite3_Finalize (Stmt.Stmt));
   end Finalize;

   procedure Finalize
     (Stmt : access Statement) is
   begin
      Retcode_2_Exception (Stmt.Finalize);
   end Finalize;
   ------------------
   -- Generic_Exec --
   ------------------

   package body Generic_Exec is

      use String_Vectors;

      function Exec
        (Self    : access Object;
         Sql     : String;
         Cb      : Callback;
         Arg     : Data_Handle)
         return Return_Value
      is

         St  : aliased Statement;
         RSt : constant Statement_Reference := St'Unrestricted_Access;
         Rc  : Return_Value;

         Vals      : String_Vectors.Vector;
         Cols      : String_Vectors.Vector;
         NCol      : Integer;
         Cols_Done : Boolean := False;

      begin

         Rc := Prepare (Self, Sql, RSt);
         if not (Rc = SQLITE_OK) then
            return Rc;
         end if;

         NCol := Column_Count (RSt);

         loop

            Rc := Step (RSt);

            if (Rc = SQLITE_ROW) then

               Clear (Vals);

               if not Cols_Done then
                  for I in 0 .. (NCol - 1) loop
                     Append (Cols, Column_Name (RSt, I));
                  end loop;

                  Cols_Done := True;
               end if;

               for I in 0 .. (NCol - 1) loop
                  Append (Vals, Column_Text (RSt, I));
               end loop;

               Rc := Cb (Arg, NCol, Vals, Cols);
               if not (Rc = SQLITE_OK) then
                  Rc := Finalize (RSt);
                  Rc := SQLITE_ABORT;
                  exit;
               end if;

            else

               Rc := Finalize (RSt);
               exit;

            end if;

         end loop;

         return Rc;

      end Exec;

   end Generic_Exec;

   --------------------
   -- Generic_Exec16 --
   --------------------

   package body Generic_Exec16 is

      use Wide_String_Vectors;

      function Exec16
        (Self    : access Object;
         Sql     : Wide_String;
         Cb      : Callback16;
         Arg     : Data_Handle)
         return Return_Value
      is

         St  : aliased Statement;
         RSt : constant Statement_Reference := St'Unrestricted_Access;
         Rc  : Return_Value;

         Vals      : Wide_String_Vectors.Vector;
         Cols      : Wide_String_Vectors.Vector;
         NCol      : Integer;
         Cols_Done : Boolean := False;

      begin

         Rc := Prepare16 (Self, Sql, RSt);
         if not (Rc = SQLITE_OK) then
            return Rc;
         end if;

         NCol := Column_Count (RSt);

         loop

            Rc := Step (RSt);

            if (Rc = SQLITE_ROW) then

               Clear (Vals);

               if not Cols_Done then
                  for I in 0 .. (NCol - 1) loop
                     Append (Cols, Column_Name16 (RSt, I));
                  end loop;

                  Cols_Done := True;
               end if;

               for I in 0 .. (NCol - 1) loop
                  Append (Vals, Column_Text16 (RSt, I));
               end loop;

               Rc := Cb (Arg, NCol, Vals, Cols);
               if not (Rc = SQLITE_OK) then
                  Rc := Finalize (RSt);
                  Rc := SQLITE_ABORT;
                  exit;
               end if;

            else

               Rc := Finalize (RSt);
               exit;

            end if;

         end loop;

         return Rc;

      end Exec16;

   end Generic_Exec16;

   ---------------
   -- Get_Table --
   ---------------

   function Get_Table
     (Self  : access Object;
      Sql   : String;
      Table : Table_Reference)
      return Return_Value
   is

      use String_Tables;
      use String_Vectors;

      package Tables_Exec
      is new Generic_Exec (String_Tables.Vector, Table_Reference);

      function Build_Table
        (PArg        : Table_Reference;
         Argc        : Integer;
         Argv        : String_Vectors.Vector;
         ColumnNames : String_Vectors.Vector)
         return Return_Value
      is
         pragma Unreferenced (Argc);
      begin


         if Is_Empty (PArg.all) then
            Append (PArg.all, ColumnNames);
         end if;

         Append (PArg.all, Argv);

         return SQLITE_OK;

      end Build_Table;

   begin

      Clear (Table.all);
      return Tables_Exec.Exec (Self, Sql, Build_Table'Unrestricted_Access, Table);

   end Get_Table;

   -----------------
   -- Get_Table16 --
   -----------------

   function Get_Table16
     (Self  : access Object;
      Sql   : Wide_String;
      Table : Table16_Reference)
      return Return_Value
   is

      use Wide_String_Tables;
      use Wide_String_Vectors;

      package Tables16_Exec16
      is new Generic_Exec16 (Wide_String_Tables.Vector, Table16_Reference);

      function Build_Table16
        (PArg        : Table16_Reference;
         Argc        : Integer;
         Argv        : Wide_String_Vectors.Vector;
         ColumnNames : Wide_String_Vectors.Vector)
         return Return_Value
      is
         pragma Unreferenced (Argc);
      begin


         if Is_Empty (PArg.all) then
            Append (PArg.all, ColumnNames);
         end if;

         Append (PArg.all, Argv);

         return SQLITE_OK;

      end Build_Table16;

   begin

      Clear (Table.all);
      return Tables16_Exec16.Exec16 (Self,
                                     Sql,
                                     Build_Table16'Unrestricted_Access,
                                     Table);

   end Get_Table16;

   ---------------
   -- Interrupt --
   ---------------

   procedure Interrupt
     (Self : access Object)
   is

      procedure Sqlite3_Interrupt
        (Db : DB_Access);
      pragma Import (C, Sqlite3_Interrupt, "sqlite3_interrupt");
      -- sqlite3.h:220
      -- void sqlite3_interrupt(sqlite3*);

   begin

      Sqlite3_Interrupt (Self.Db);

   end Interrupt;

   -----------------------
   -- Last_Insert_Rowid --
   -----------------------

   function Last_Insert_Rowid
     (Self : access Object)
      return Uint64
   is

      function Sqlite3_Last_Insert_Rowid
        (Db : DB_Access)
         return Uint64;
      pragma Import (C, Sqlite3_Last_Insert_Rowid, "sqlite3_last_insert_rowid");
      -- sqlite3.h:172
      -- sqlite_int64 sqlite3_last_insert_rowid(sqlite3*);

   begin

      return Sqlite3_Last_Insert_Rowid (Self.Db);

   end Last_Insert_Rowid;

   ----------
   -- open --
   ----------

   function Open
     (Self     : access Object;
      Filename : String)
      return Return_Value
   is

      function Sqlite3_Open
        (Filename  : C.char_array;
         DB_Handle : access DB_Access)
         return C.int;
      pragma Import (C, Sqlite3_Open, "sqlite3_open");
      -- sqlite3.h:504
      -- int sqlite3_open(
      --   const char *filename,   /* Database filename (UTF-8) */
      --   sqlite3 **ppDb          /* OUT: SQLite db handle */
      -- );

   begin

      return Int_To_Ret_Val
        (Sqlite3_Open (C.To_C (Filename),
         Self.all.Db'Unchecked_Access));

   end Open;
   procedure Open
     (Self     : access Object;
      Filename : String) is
   begin
      Retcode_2_Exception (Self.Open (Filename));
   end Open;

   ------------
   -- open16 --
   ------------

   function Open16
     (Self     : access Object;
      Filename : Wide_String)
      return Return_Value
   is

      function Sqlite3_Open16
        (Filename  : C.wchar_array;
         DB_Handle : access DB_Access)
         return C.int;
      pragma Import (C, Sqlite3_Open16, "sqlite3_open16");
      -- sqlite3.h:508
      -- int sqlite3_open16(
      --   const void *filename,   /* Database filename (UTF-16) */
      --   sqlite3 **ppDb          /* OUT: SQLite db handle */
      -- );

   begin

      return Int_To_Ret_Val (Sqlite3_Open16 (To_C (Filename), Self.Db'Access));

   end Open16;


   -------------
   -- prepare --
   -------------

   function Prepare
     (Self : access Object;
      Sql  : String;
      Stmt : Statement_Reference)
      return Return_Value
   is

      function Sqlite3_Prepare
        (Db     : DB_Access;
         ZSql   : C.char_array;
         NBytes : C.int;
         PpStmt : access Stmt_Access;
         PzTail : System.Address)
         return C.int;
      pragma Import (C, Sqlite3_Prepare, "sqlite3_prepare");
      -- sqlite3.h:581
      -- int sqlite3_prepare(
      --   sqlite3 *db,            /* Database handle */
      --   const char *zSql,       /* SQL statement, UTF-8 encoded */
      --   int nBytes,             /* Length of zSql in bytes. */
      --   sqlite3_stmt **ppStmt,  /* OUT: Statement handle */
      --   const char **pzTail     /* OUT: Pointer to unused portion of zSql */
      -- );

   begin

      return Int_To_Ret_Val (Sqlite3_Prepare (Self.Db,
        C.To_C (Sql),
        C.int (-1),
        Stmt.Stmt'Access,
        System.Null_Address)
       );

   end Prepare;

   procedure  Prepare
     (Self : access Object;
      Sql  : String;
      Stmt : Statement_Reference) is
   begin
      Retcode_2_Exception (Prepare (Self, Sql, Stmt), Sql);
   end Prepare;
   ---------------
   -- prepare16 --
   ---------------

   function Prepare16
     (Self : access Object;
      Sql  : Wide_String;
      Stmt : Statement_Reference)
      return Return_Value
   is

      function Sqlite3_Prepare16
        (Db     : DB_Access;
         ZSql   : C.wchar_array;
         NBytes : C.int;
         PpStmt : access Stmt_Access;
         PzTail : System.Address)
         return C.int;
      pragma Import (C, Sqlite3_Prepare16, "sqlite3_prepare16");
      -- sqlite3.h:588
      -- int sqlite3_prepare16(
      --   sqlite3 *db,            /* Database handle */
      --   const void *zSql,       /* SQL statement, UTF-16 encoded */
      --   int nBytes,             /* Length of zSql in bytes. */
      --   sqlite3_stmt **ppStmt,  /* OUT: Statement handle */
      --   const void **pzTail     /* OUT: Pointer to unused portion of zSql */
      -- );

   begin

      return Int_To_Ret_Val (Sqlite3_Prepare16 (Self.Db,
        To_C (Sql),
        C.int (-1),
        Stmt.Stmt'Access,
        System.Null_Address)
       );

   end Prepare16;

   -----------
   -- Reset --
   -----------

   function Reset
     (Stmt : access Statement)
      return Return_Value
   is

      function Sqlite3_Reset
        (Stmt : Stmt_Access)
         return C.int;
      pragma Import (C, Sqlite3_Reset, "sqlite3_reset");
      -- sqlite3.h:857
      -- int sqlite3_reset(sqlite3_stmt *pStmt);

   begin

      return Int_To_Ret_Val (Sqlite3_Reset (Stmt.Stmt));

   end Reset;

   ---------------
   -- Row_Count --
   ---------------

   function Row_Count
     (Self : Table_Reference)
      return Natural
   is

      use String_Tables;

   begin

      return Natural (Length (Self.all));

   end Row_Count;

   ----------
   -- Step --
   ----------

   function Step
     (Stmt : access Statement)
      return Return_Value
   is

      function Sqlite3_Step
        (Stmt : Stmt_Access)
         return C.int;
      pragma Import (C, Sqlite3_Step, "sqlite3_step");
      -- sqlite3.h:744
      -- int sqlite3_step(sqlite3_stmt*);

   begin

      return Int_To_Ret_Val (Sqlite3_Step (Stmt.Stmt));

   end Step;
   function Step
     (Stmt : access Statement)
      return Boolean is
   begin
      return Stmt.Step = SQLITE_OK;
   end Step;

   procedure Step
     (Stmt : access Statement) is
   begin
      Retcode_2_Exception (Stmt.Step);
   end Step;

   -------------
   -- Total_Changes --
   -------------

   function Total_Changes
     (Self : access Object)
      return Integer
   is

      function Sqlite3_Total_Changes
        (Db : DB_Access)
         return C.int;
      pragma Import (C, Sqlite3_Total_Changes, "sqlite3_total_changes");
      -- sqlite3.h:212
      -- int sqlite3_total_changes(sqlite3*);

   begin

      return Integer (Sqlite3_Total_Changes (Self.Db));

   end Total_Changes;

end GNU.DB.SQLite3;
