/*
 *  This file is part of GNUDoQ, Copyright (C) 2005-2006 Luc Vo Van
 *
 *   GNUDoQ is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   GNUDoQ is distributed in the hope that it will be useful, but WITHOUT
 *   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *   for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with GNUDoQ; see the file COPYING.  If not, write to
 *   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *   Boston, MA 02111-1307, USA.
 *
 *   Class          :   Main UI (implementation)
 *   Author         :   Luc Vo Van
 *   Original Date  :   September 2005
 *
 **/

#include "GNUDoQ.H"
#include "GNUDoQBoxWidget.H"
#include "ui_GNUDoQAboutForm.h"
#include "GNUDoku.xpm"

#include "sudoku.H"
using sudoku::Sudoku;
#include "sudoku-solve.H"

#include <QDialog>
#include <QApplication>
#include <QDateTime>
#include <QMessageBox>
#include <QSettings>
#include <QPainter>
#include <QPrinter>
#include <QFileDialog>
#include <QPrintDialog>

#include <iostream>
#include <sstream>
#include <vector>

#define max(a,b) (((a)>(b))?(a):(b))

QString GNUDoQ::VERSION = "0.94";

GNUDoQ::GNUDoQ (QWidget * parent):QWidget (parent)
{
  int
    iGroup;

  ui.setupUi (this);
  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  iGroup = ((int) (i / 3)) * 3 + (j / 3);
	  m_Boxes[i][j] = new GNUDoQBoxWidget (iGroup);
	  QObject::connect (m_Boxes[i][j], SIGNAL (valueChanged ()),
			    this, SLOT (TestSolveOrVerify ()));
	  ui.gridLayout1->addWidget (m_Boxes[i][j], i, j);
	}
    }

  ui.sldDifficulty->setValue (50);


  // Load the previously played puzzle, or generate a new one
  QSettings
  settings ("thelemmings.net", "GNUDoQ");
  if (settings.contains ("current/puzzle"))
    LoadPuzzle ();
  else
    GNUDoQ::on_btnGenerate_clicked ();
}

/*****************************************************************************
 * UI methods
 *****************************************************************************/

void
GNUDoQ::on_btnGenerate_clicked ()
{
  int seed = time (NULL);
  std::ostringstream seedstr;
  int diff = (int) (ui.sldDifficulty->value () / 10.0);
  if (diff < 10)
    seedstr << "0";
  seedstr << diff << seed;
  ui.lePuzzleCode->setText (QString::fromStdString (seedstr.str ()));
  float difficulty = (float) diff / 10.0;

  GeneratePuzzle (difficulty, seed);
}

void
GNUDoQ::on_btnQuit_clicked ()
{
  close ();
}

void
GNUDoQ::on_btnSolve_clicked ()
{
  SolvePuzzle ();
}

void
GNUDoQ::on_btnClear_clicked ()
{
  ResetPuzzle ();
}

void
GNUDoQ::on_btnPrint_clicked ()
{
  // Prints out
  QPrintDialog dialog (&printer, this);
  if (dialog.exec ())
    {
      QPainter painter (&printer);

      // Don't do a thing if there is no printer configured
      if (!painter.isActive ())
	return;

      // Scaling stuff
	  extern int qt_defaultDpi ();
	  qreal sourceDpiX = qt_defaultDpi ();
	  qreal sourceDpiY = sourceDpiX;

	  QPaintDevice *dev = painter.device ();
	  if (dev)
	    {
	      sourceDpiX = dev->logicalDpiX ();
	      sourceDpiY = dev->logicalDpiY ();
	    }

	  const qreal dpiScaleX = qreal (printer.logicalDpiX ()) / sourceDpiX;
	  const qreal dpiScaleY = qreal (printer.logicalDpiY ()) / sourceDpiY;
	  painter.scale (dpiScaleX, dpiScaleY);

      painter.setWindow (printer.pageRect ());

      // Measures
      int iOneInchH = printer.logicalDpiX ();
      int iOneInchV = printer.logicalDpiY ();
      int iLineSpacing = iOneInchV / 20;
      int iPageLeft = printer.pageRect ().x ();
      int iPageTop = printer.pageRect ().y ();
      int iPageWidth = printer.pageRect ().width ();
      int iPageBottom = printer.pageRect ().height ();
      int i, j;

      // Pen
      QPen pen (Qt::SolidLine);

      // Fonts
      QFont fTitle ("Times-Roman", 18);
      QFontMetrics fmTitle (fTitle);
      fTitle.setBold (true);

      QFont fNormal ("Times-Roman", 12);
      QFontMetrics fmNormal (fNormal);

      QFont fSmall ("Times-Roman", 10);
      QFontMetrics fmSmall (fSmall);

      QFont fNumbers ("Times-Roman", 24);
      QFontMetrics fmNumbers (fNumbers);

      // Header
      QString sBigTitle = tr ("GNUDoQ Puzzle Sheet");
      QString sSubTitle =
	tr
	("GNUDoQ is freely available for download at http://www.thelemmings.net");
      QRect r0 = fmTitle.boundingRect (sBigTitle);
      QRect r1 = fmSmall.boundingRect (sSubTitle);
      i =
	(iPageWidth -
	 (iPageLeft + max (r0.width (), r1.width ()) + 2 * iOneInchH)) / 2;
      j =
	iLineSpacing + r0.height () + 2 * iLineSpacing + r1.height () +
	2 * iLineSpacing;

      // Shadowy rectangles
      painter.fillRect (i + iOneInchH / 30,
		        iPageTop + iOneInchV / 30,
			iPageWidth - i + iOneInchH / 30,
			j + iOneInchV / 30,
			QBrush (Qt::gray));
      painter.fillRect (i, iPageTop, iPageWidth - i, j, QBrush (Qt::white));
      pen.setWidth (2);
      painter.setPen (pen);
      painter.drawRect (i, iPageTop, iPageWidth - i, j);

      // Big Title
      painter.setFont (fTitle);
      painter.drawText (iPageLeft + iOneInchH +
			(iPageWidth - (r0.width () + 2 * iOneInchH)) / 2,
			iPageTop + iLineSpacing + r0.height (),
			sBigTitle);
      // Small Title
      painter.setFont (fSmall);
      painter.drawText (iPageLeft + iOneInchH +
			(iPageWidth - (r1.width () + 2 * iOneInchH)) / 2,
			iPageTop + iLineSpacing + r0.height () +
			2 * iLineSpacing + r1.height (),
			sSubTitle);

      // Footer
      QString sFooter =
	tr
	("Generated on %1. GNUDoQ and GNUDoku are free programs, share and enjoy!").
	arg (QDateTime::currentDateTime ().toString ("ddd MMMM d yyyy"));
	  QString sFooter2 =
	tr("This software is powered by the GNUDoku engine: http://icculus.org/~jcspray/GNUDoku/");
      QRect r2 = fmSmall.boundingRect (sFooter);
      QRect r4 = fmSmall.boundingRect (sFooter2);
      i = (iPageWidth - r2.width ()) / 2;
      j = iLineSpacing + r2.height () + iLineSpacing + r4.height () + iLineSpacing;
      painter.drawLine (iPageLeft, iPageTop + iPageBottom - j, iPageWidth,
			iPageTop + iPageBottom - j);
      painter.setFont (fSmall);
      painter.drawText (iPageLeft + i,
			iPageTop + iPageBottom - iLineSpacing - r4.height () - iLineSpacing, sFooter);
      painter.drawText (iPageLeft + (iPageWidth - r4.width ()) / 2,
			iPageTop + iPageBottom - iLineSpacing, sFooter2);

      // The puzzle number
      pen.setWidth (1);
      painter.setPen (pen);
      QString sPuzzleNumber = tr ("Puzzle Code: %1").arg (m_sCurrentCode);
      QRect r3 = fmNormal.boundingRect (sPuzzleNumber);
      i = (iPageWidth - r3.width ()) / 2;
      painter.drawRoundRect (iPageLeft + i - 2 * iLineSpacing,
			     (int) (1.5f * (float) iOneInchV) - r3.height () - 2 * iLineSpacing,
			     r3.width () + 4 * iLineSpacing,
			     r3.height () + 4 * iLineSpacing);
      painter.setFont (fNormal);
      painter.drawText (iPageLeft + i,
			(int) (1.5f * (float) iOneInchV), sPuzzleNumber);

      // The puzzle grid
      int iPuzzleSize = 5;	// puzzle size of a side in inches
      int iGridSpacingH, iGridSpacingV;
      pen.setWidth (4);
      painter.setPen (pen);

      i = iPageLeft + (iPageWidth - iPuzzleSize * iOneInchH) / 2;
      j = iPageTop + 2 * iOneInchV;

      // The large lines
      iGridSpacingH = iPuzzleSize * iOneInchH / 3;
      iGridSpacingV = iPuzzleSize * iOneInchV / 3;
      for (int k = 1; k < 3; k++)
	{
	  // Horizontal
	  painter.drawLine (i + k * iGridSpacingH, j,
			    i + k * iGridSpacingH, j + iPuzzleSize * iOneInchV);
	  painter.drawLine (i, j + k * iGridSpacingV,
			    i + iPuzzleSize * iOneInchH, j + k * iGridSpacingV);
	}

      // The thin lines
      iGridSpacingH = iPuzzleSize * iOneInchH / 9;
      iGridSpacingV = iPuzzleSize * iOneInchV / 9;
      pen.setWidth (1);
      painter.setPen (pen);
      for (int k = 0; k < 9; k++)
	{
	  if (k % 3 != 0)
	    {
	      // vertical
	      painter.drawLine (i + k * iGridSpacingH, j,
			        i + k * iGridSpacingH, j + iPuzzleSize * iOneInchV);

	      // horizontal
	      painter.drawLine (i, j + k * iGridSpacingV,
				i + iPuzzleSize * iOneInchH, j + k * iGridSpacingV);
	    }
	}

      // The numbers
      painter.setFont (fNumbers);
      for (int y = 0; y < 9; ++y)
	{
	  for (int x = 0; x < 9; ++x)
	    {
	      // if not editable, it's part of the original puzzle,
	      // we print it.
	      if (!m_Boxes[x][y]->editable ())
		{
		  QString sNumber = QString::number (m_Boxes[x][y]->value ());
		  QRect rTmp = fmNumbers.boundingRect (sNumber);
		  painter.drawText (i + y * iGridSpacingH + (iGridSpacingH - rTmp.width ()) / 2,
				    j + x * iGridSpacingV + (iGridSpacingV -
					    rTmp.height ()) / 2 + rTmp.height (), sNumber);
		}

	    }
	}

    }
}

void
GNUDoQ::on_lePuzzleCode_returnPressed ()
{
  // First two characters are difficulty
  int diffint = ui.lePuzzleCode->text ().left (2).toInt ();
  float diff = (float) diffint / 10.0f;
  if (diff > 1.0f)
    diff = 1.0f;
  else if (diff < 0.0f)
    diff = 0.0f;

  int seed = ui.lePuzzleCode->text ().mid (2).toInt ();
  GeneratePuzzle (diff, seed);
}

void
GNUDoQ::on_btnLogoAbout_clicked ()
{
  Ui::GNUDoQAboutForm aboutForm;
  QDialog dialog;

  aboutForm.setupUi (&dialog);
  aboutForm.lblLogo->setPixmap (QPixmap ((const char **) GNUDoku_xpm));
  aboutForm.lblVersion->setText ("GNUDoQ v" + VERSION);
  aboutForm.teAbout->setHtml (tr ("<b>GNUDoQ</b> (C) 2005-2006 Luc Vo Van<br/>"
				  "Based on the GNUDoku engine by<br/>"
				  "&nbsp;&nbsp;Phillip Jordan<br/>"
				  "&nbsp;&nbsp;Colin Lewis<br/>"
				  "&nbsp;&nbsp;John Spray.<br/>"
				  "<a>http://icculus.org/~jcspray/GNUDoku/</a><br/>"
				  "<a>http://www.thelemmings.net</a><br/>"
				  "Complete source code for this software is<br/>"
				  "available from <a>http://www.thelemmings.net</a><br/>"
				  "<br/>"
				  "GNUDoQ is a free game allowing you to play, create and solve"
				  " Su Doku puzzles. Su Doku is a Japanese number placement puzzle"
				  " requiring logic and combinatory skills. The goal of the game"
				  " is to place numerals from 1 to 9 in the grid so that each"
				  " number appears exactly once in each row, column, and 3x3"
				  " subgrid.<br/><br/>"
				  "You can get more information regarding Su Doku puzzles from"
				  " <a>http://en.wikipedia.org/wiki/Sudoku</a><br/><br/>"
				  "This program is free software; you can redistribute it and/or "
				  "modify it under the terms of the GNU General Public License "
				  "as published by the Free Software Foundation; either version 2 "
				  "of the License, or (at your option) any later version.<br/><br/>"
				  "This program is distributed in the hope that it will be useful, "
				  "but WITHOUT ANY WARRANTY; without even the implied warranty of "
				  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
				  "GNU General Public License for more details.<br/><br/>"
				  "You should have received a copy of the GNU General Public License "
				  "along with this program; if not, write to the Free Software "
				  "Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA "
				  "02110-1301, USA."));
  dialog.exec ();
}

void
GNUDoQ::on_sldDifficulty_valueChanged ()
{
  if (ui.sldDifficulty->value () < 30)
    ui.lblDifficulty->setText (tr ("Easy"));
  else if (ui.sldDifficulty->value () < 51)
    ui.lblDifficulty->setText (tr ("Normal"));
  else if (ui.sldDifficulty->value () < 75)
    ui.lblDifficulty->setText (tr ("Hard"));
  else
    ui.lblDifficulty->setText (tr ("Very hard"));
}

/*****************************************************************************
 * GNUDoku engine methods
 *****************************************************************************/

void
GNUDoQ::GeneratePuzzle (float const difficulty, int const seed)
{
  m_sCurrentCode =
    QString::number ((int) (difficulty * 10.0f)) + QString::number (seed);
  srand (seed);

  int top = 0;
  Sudoku::attempt stack[81];
  Sudoku::flags flag_data;
  char visited[81] = { 0 };

  for (int i = 0; i < 81; ++i)
    stack[i].index = -1;

  memset (&flag_data, 0, sizeof (flag_data));

  Sudoku::solve (stack, top, &flag_data, visited);

  int const blanks = static_cast < int >(difficulty * 45.0) + 18;
  Sudoku::blank (visited, blanks);

  ClearBoxes ();
  FillBoxes (stack, visited);
  TestSolveOrVerify ();
}

void
GNUDoQ::SolvePuzzle ()
{
  int top = 0;
  Sudoku::attempt stack[81];
  Sudoku::flags flag_data;
  char visited[81] = { 0 };

  this->setCursor (Qt::WaitCursor);

  // This indicated wether are not there are still
  // empty boxes. If there aren't any then the
  // player is probably checking his moves
  // Otherwise the player is looking for a solution
  bool bStillEmptyBoxes = false;
  bool bWin = true;

  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  if (m_Boxes[i][j]->editable () && m_Boxes[i][j]->value () == 0)
	    bStillEmptyBoxes = true;
	  if (m_Boxes[i][j]->value () == 0)
	    m_Boxes[i][j]->setCpusolved (true);
	}
    }


  for (int i = 0; i < 81; ++i)
    stack[i].index = -1;

  memset (&flag_data, 0, sizeof (flag_data));

  top = GetFromBoxes (stack, &flag_data, visited);
  if (top == -1)
    {
      this->setCursor (Qt::ArrowCursor);
      return;
    }

  // unroll the old stack structure into the new Board object,
  // solve, then convert back (!)
  sudoku::Board board;
  for (int i = 0; i < top; ++i)
    {
      sudoku::move fixed (stack[i], sudoku::MOVE_FIXED);
      fixed.Do (board);
    }

  std::vector < sudoku::move > moves;
  bool solved = sudoku::Solve (board, moves);
  if (solved)
    {
      // convert moves necessary for solution into old-style attempts
      for (std::vector < sudoku::move >::const_iterator cur = moves.begin ();
	   cur != moves.end (); ++cur)
	{
	  stack[top] = Sudoku::attempt (*cur);
	  visited[cur->index] = 1;
	  ++top;
	}
      FillBoxes (stack, visited);
    }
  else
    {
      QMessageBox::warning (this, "GNUDoQ",
			    tr ("Solving failed!\n\nThe problem could not be"
				" solved.  This is probably because it is"
				" insolvable.  If you are absolutely certain"
				" that the initial grid is valid, please send it"
				"to the GNUDoku developers."),
			    QMessageBox::Ok,
			    QMessageBox::NoButton, QMessageBox::NoButton);
    }

  // If the player was looking for a solution, he doesn't get to lose
  // or win. He just gets the solution (if any)
  if (bStillEmptyBoxes)
    {
      this->setCursor (Qt::ArrowCursor);
      return;
    }
  // Check if we win
  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  if (m_Boxes[i][j]->erroneous ())
	    bWin = false;
	}
    }

  this->setCursor (Qt::ArrowCursor);

  if (bWin)
    {
      QMessageBox::information (this, tr ("You win!"),
				tr
				("Congratulations, you solved the puzzle!"),
				QMessageBox::Ok);
    }
  else
    {
      QMessageBox::information (this, tr ("Sorry..."),
				tr
				("There were errors in the puzzle, try again!"),
				QMessageBox::Ok);
    }
}

/*****************************************************************************
 * Box methods
 *****************************************************************************/

int
GNUDoQ::GetFromBoxes (Sudoku::attempt * top,
		      Sudoku::flags * flags, char visited[])
{
  //    int res = 0;
  int count = 0;
  //    Sudoku::attempt* const stackbase = top;
  bool errors_found = false;
  Sudoku::flags firstoccu;

  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  if (m_Boxes[j][i]->value () == 0)
	    {
	      // Set background to default
	      m_Boxes[j][i]->setErroneous (false);
	      continue;
	    }

	  top->col = j;
	  top->row = i;

	  top->value = m_Boxes[j][i]->value () - 1;

	  top->index = top->row * 9 + top->col;
	  top->box = (top->row / 3) * 3 + (top->col / 3);
	  top->tries = 0;

	  if (flags->col[top->col][top->value] ||
	      flags->row[top->row][top->value] ||
	      flags->box[top->box][top->value])
	    {
	      errors_found = true;
	      m_Boxes[j][i]->setErroneous (true);
	      if (flags->col[top->col][top->value])
		{
		  int index = firstoccu.col[top->col][top->value];
		  m_Boxes[index % 9][index / 9]->setErroneous (true);
		}
	      if (flags->row[top->row][top->value])
		{
		  int index = firstoccu.row[top->row][top->value];
		  m_Boxes[index % 9][index / 9]->setErroneous (true);
		}
	      if (flags->box[top->box][top->value])
		{
		  int index = firstoccu.box[top->box][top->value];
		  m_Boxes[index % 9][index / 9]->setErroneous (true);
		}

	    }
	  else
	    {
	      m_Boxes[j][i]->setErroneous (false);
	    }
	  flags->col[top->col][top->value] = 1;
	  flags->row[top->row][top->value] = 1;
	  flags->box[top->box][top->value] = 1;
	  firstoccu.col[top->col][top->value] = top->index;
	  firstoccu.row[top->row][top->value] = top->index;
	  firstoccu.box[top->box][top->value] = top->index;
	  visited[top->index] = 1;

	  ++top;
	  ++count;
	}
    }

  if (errors_found)
    return -1;
  else
    return count;
}


void
GNUDoQ::ClearBoxes ()
{
  DisconnectBoxesValueChanged ();
  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  m_Boxes[j][i]->setValue (0);
	  m_Boxes[j][i]->setEditable (false);
	  m_Boxes[j][i]->setErroneous (false);
	  m_Boxes[j][i]->setCpusolved (false);
	}
    }
  ConnectBoxesValueChanged ();
}


void
GNUDoQ::ResetPuzzle ()
{
  DisconnectBoxesValueChanged ();
  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  m_Boxes[j][i]->setErroneous (false);
	  if (m_Boxes[j][i]->cpusolved ())
	    {
	      m_Boxes[j][i]->setCpusolved (false);
	      m_Boxes[j][i]->setEditable (true);
	    }
	  if (m_Boxes[j][i]->editable ())
	    m_Boxes[j][i]->setValue (0);
	}
    }
  ConnectBoxesValueChanged ();
}

void
GNUDoQ::TestSolveOrVerify ()
{
  bool bSolve = false;

  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  // m_Boxes[j][i]->setErroneous (false);
	  bSolve = bSolve | (m_Boxes[j][i]->value () == 0);
	}
    }

  QFont font;
  if (bSolve)
    {
      font = ui.btnSolve->font ();
      font.setWeight (QFont::Normal);
      ui.btnSolve->setFont (font);
      ui.btnSolve->setText (tr ("Solve"));
      ui.btnSolve->setToolTip (tr ("Solves or tests the current grid."));
    }
  else
    {
      font = ui.btnSolve->font ();
      font.setWeight (QFont::Bold);
      ui.btnSolve->setFont (font);
      ui.btnSolve->setText (tr ("Verify"));
      ui.btnSolve->setToolTip (tr ("Verifies your solution"));
    }
}

void
GNUDoQ::FillBoxes (const Sudoku::attempt stack[], const char visited[])
{
  DisconnectBoxesValueChanged ();
  int grid[81];

  memset (grid, 0, sizeof (grid));
  for (int i = 0; i < 81; ++i)
    {
      if (stack[i].index >= 0 && visited[stack[i].index])
	grid[stack[i].index] = 1 + stack[i].value;
    }


  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  if (grid[i * 9 + j] == 0)
	    {
	      m_Boxes[j][i]->setValue (0);
	      m_Boxes[j][i]->setEditable (true);
	    }
	  else
	    {
	      m_Boxes[j][i]->setValue (grid[i * 9 + j]);
	      m_Boxes[j][i]->setEditable (false);
	    }
	  m_Boxes[j][i]->setErroneous (false);
	}
    }

  ConnectBoxesValueChanged ();
}


void
GNUDoQ::DisconnectBoxesValueChanged ()
{
  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  disconnect (m_Boxes[i][j], 0, this, 0);
	}
    }
}

void
GNUDoQ::ConnectBoxesValueChanged ()
{
  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  QObject::connect (m_Boxes[i][j], SIGNAL (valueChanged ()),
			    this, SLOT (TestSolveOrVerify ()));
	}
    }
}

/*****************************************************************************
 * Save and load methods
 ****************************************************************************/

QString GNUDoQ::DumpToString ()
{
  QString
  puzzlestring ("");
  QString
    sText;

  for (int i = 0; i < 81; ++i)
    {
      sText = QString::number (m_Boxes[i % 9][i / 9]->value ());
      if (sText.isEmpty ())
	puzzlestring += QChar ('0');
      else
	{
	  if (!m_Boxes[i % 9][i / 9]->editable ())
	    puzzlestring += QChar ('+');

	  puzzlestring += sText[0];
	}
    }

  return puzzlestring;
}

void
GNUDoQ::SavePuzzle ()
{
  QSettings settings ("thelemmings.net", "GNUDoQ");
  settings.setValue ("current/puzzle", DumpToString ());
  settings.setValue ("current/puzzlecode", m_sCurrentCode);
}

void
GNUDoQ::LoadPuzzle ()
{
  QSettings settings ("thelemmings.net", "GNUDoQ");
  QString puzzle = settings.value ("current/puzzle").toString ();
  m_sCurrentCode = settings.value ("current/puzzlecode").toString ();
  ui.lePuzzleCode->setText (m_sCurrentCode);
  int x = 0, y = 0;

  DisconnectBoxesValueChanged ();

  QChar tmp[2];
  if (puzzle.length () < 81)
    return;

  for (int i = 0; i < 9; ++i)
    {
      for (int j = 0; j < 9; ++j)
	{
	  m_Boxes[j][i]->setEditable (true);
	  m_Boxes[j][i]->setErroneous (false);
	}
    }

  tmp[1] = 0;
  for (int i = 0; i < puzzle.length (); ++i)
    {
      if (puzzle[i] == 43)
	m_Boxes[x][y]->setEditable (false);
      else if (puzzle[i] < 48 || puzzle[i] > 57)
	return;
      else
	{
	  tmp[0] = puzzle[i];
	  m_Boxes[x][y]->setValue (QString (puzzle[i]).toInt ());
	  x++;
	  if (x > 8)
	    {
	      x = 0;
	      y++;
	    }
	}
    }

  ConnectBoxesValueChanged ();
}
