//*****************************************************************************
//                              PnlGnuCapDC.cpp                               *
//                             -----------------                              *
// Started     : 18/08/2003                                                   *
// Last Update : 06/09/2011                                                   *
// Copyright   : (C) 2003 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapDC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapDC, PnlAnaBase )

  EVT_CHOICE  ( ID_CHO_SRCNAME, PnlGnuCapDC::OnSrcName )
  EVT_RADIOBOX( ID_RBX_SCALE,   PnlGnuCapDC::OnScale   )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapDC::PnlGnuCapDC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrEng( eSIMR_GNUCAP );
  bSetAnaType( eCMD_DC );

  Create( );        // Create the analysis panel
  InitSwpUnits( );  // Initialize the sweep units
  bClear( );        // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapDC::~PnlGnuCapDC( )
{
}

//*****************************************************************************
// Create the scale display objects.

void  PnlGnuCapDC::CreateSrc( void )
{
  wxPanel * poPnl;

  // Create the temperature panel and controls
#ifdef LAYOUT_MNGR
  poPnl = new wxPanel( this );
#else
  poPnl = this;
#endif // LAYOUT_MNGR

  // Create input source
#ifdef LAYOUT_MNGR
  m_oLblSrcName.Create( poPnl, ID_UNUSED, wxT("Signal Source") );
  m_oChoSrcName.Create( poPnl, ID_CHO_SRCNAME );
#else
  new wxLabel( poPnl, ID_UNUSED, wxT("Signal Source"), wxPoint( 18, 176 ) );
  m_oChoSrcName.Create( poPnl, ID_CHO_SRCNAME,
                        wxPoint( 144, 171 ), wxSize( 110, 29 ) );
#endif // LAYOUT_MNGR
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapDC::Create( void )
{
#ifndef LAYOUT_MNGR
  int  ix, iy;

  // Set the sweep parameters labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 201 );
  m_oSbxSwpPars.SetLabel( wxT(" DC Sweep ") );
#endif // LAYOUT_MNGR
  m_oPnlStart  .bSetName( wxT("Start Value") );
  m_oPnlStop   .bSetName( wxT("Stop Value") );

  // Create scale controls but disable the scale option GNU-Cap doesn't support
  PnlAnaBase::CreateScale( );
  m_oRbxScale.Enable( eSCALE_OCT, FALSE );

  PnlAnaBase::CreateTemp( );  // Create the analysis temperature controls
  PnlAnaBase::DoLayout( );    // Layout the panel's GUI objects
  CreateSrc( );               // Create input source
  DoLayout( );                // Local layout tasks

#ifdef LAYOUT_MNGR
  SetSwpPnlLbl( wxT(" DC Sweep ") );
#endif // LAYOUT_MNGR
}

//*****************************************************************************
// Layout the panel display objects.

void  PnlGnuCapDC::DoLayout( void )
{
#ifndef LAYOUT_MNGR
  return;
#endif

  wxWindow    * poWin1;
  wxWindow    * poWin2;
  wxSizer     * poSzr;
  wxSizerFlags  oFlags;

  if( ! bIsCreated( eCTLGRP_BASE ) ) return;

  // Create and set the sizer for the signal source panel
  poWin1 = m_oLblSrcName.GetParent( );
  poSzr = new wxBoxSizer( wxHORIZONTAL );
  poWin1->SetSizer( poSzr );

  // Layout the signal source panel
  oFlags.Align( wxALIGN_LEFT );
  oFlags.Border( wxALL, 3 );
  poSzr->Add( &m_oLblSrcName, oFlags );
  poSzr->Add( &m_oChoSrcName, oFlags );

  // Create and set the sizer for the signal source panel
  poWin2 = m_oPnlStart.GetParent( );
  poSzr = poWin2->GetSizer( );

  // Add the control to the sweep panel
  oFlags.Align( wxALIGN_LEFT );
  oFlags.Border( wxALL & ~wxBOTTOM, 3 );
  poSzr->Add( poWin1, oFlags );
}

//*****************************************************************************
// Initialize the sweep parameter units.

void  PnlGnuCapDC::InitSwpUnits( void )
{
  wxString    os1;
  eUnitsType  eUnits;

  // Determine the sweep parameter units
  os1 = m_oChoSrcName.GetStringSelection( );

  eUnits = Component::eGetUnitsType( os1 );

  switch( eUnits )
  {
    case eUNITS_VOLT :  // Units of voltage
      os1 = wxT("Voltage");
      break;

    case eUNITS_CURR :  // Units of current
      os1 = wxT("Current");
      break;

    case eUNITS_CAP  :  // Units of capacitance
      os1 = wxT("Capacitance");
      break;

    case eUNITS_IND  :  // Units of inductance
      os1 = wxT("Inductance");
      break;

    case eUNITS_RES  :  // Units of resistance
      os1 = wxT("Resistance");
      break;

    default          :  // No units
      os1 = wxT("Value");
  }

  // Set the sweep parameter units
  m_oPnlStart.bSetName( wxString( wxT("Start ") ) + os1 );
  m_oPnlStop .bSetName( wxString( wxT("Stop ")  ) + os1 );
  m_oPnlStart.bSetUnitsType( eUnits );
  m_oPnlStop .bSetUnitsType( eUnits );
  m_oPnlStep .bSetUnitsType( eUnits );
}

//*****************************************************************************
// Initialize the step scale.

void  PnlGnuCapDC::InitScale( void )
{
  switch( m_oRbxScale.GetSelection( ) )
  {
    case eSCALE_LIN :  // Increment the step value by adding constant value
      m_oPnlStep.bSetName( wxT("Step Increment") );
      m_oPnlStep.bSetVarType( eVAR_FLT );
      m_oPnlStep.bSetParms( 10.0, 0.0, 1000.0, 0.01, 100.0 );
      m_oPnlStep.bShowUnits( TRUE );
      break;

    case eSCALE_LOG :  // Increase the step by a constant multiplier
      m_oPnlStep.bSetName( wxT("Step Multiplier") );
      m_oPnlStep.bSetVarType( eVAR_FLT );
      m_oPnlStep.bSetParms( 1.1, 0.0, 100.0, 0.01, 10.0 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    case eSCALE_DEC :  // Specify a fixed number of steps per decade
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetVarType( eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 1000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    default :
      break;
  }
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapDC::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )   bRtn = FALSE;

  // Set default step scale type and sweep values
  if( ! bSetScale( eSCALE_LIN ) ) bRtn = FALSE;
  m_oPnlStart.bSetValue( (float)   0.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)  10.0 );

  // Set input source default values
  m_oChoSrcName.Clear( );
  m_oChoSrcName.Append( wxT("None") );
  m_oChoSrcName.SetSelection( 0 );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapDC::bLoad( SimnGnuCap & roSimn )
{
  bool  bRtn=TRUE;

  // Load the components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VIRLC") );

  // Go no further if the DC command isn't valid
  if( ! roSimn.m_oCmdDC.bIsValid( ) )                        return( FALSE );

  // Set the source component label
  if( m_oChoSrcName.SetStringSelection( roSimn.m_oCmdDC.m_osSource ) )
       InitSwpUnits( );
  else bRtn = FALSE;

  // Set the step scale (do this before setting the sweep step)
  if( roSimn.m_oCmdDC.m_eScale != eSCALE_NONE )
  {
    m_oRbxScale.SetSelection( roSimn.m_oCmdDC.m_eScale );
    InitScale( );
  }

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdDC.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdDC.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdDC.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  if( roSimn.eGetAnaType( ) == eCMD_DC )
  {
    // Set the parameters to derive
    m_oCbxVoltage.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] );
    m_oCbxCurrent.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] );
    m_oCbxPower  .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] );
    m_oCbxResist .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] );
  }

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdDC.m_osTempC ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapDC::bSave( SimnGnuCap & roSimn )
{
  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdDC.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdDC.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdDC.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the step scale
  roSimn.m_oCmdDC.m_eScale = (eScaleType) m_oRbxScale.GetSelection( );

  // Set the sweep source (a sweep source is not compulsary for a DC analysis)
  roSimn.m_oCmdDC.m_osSource = wxT("");
  if( m_oChoSrcName.GetStringSelection( ) != wxT("None") )
    roSimn.m_oCmdDC.m_osSource = m_oChoSrcName.GetStringSelection( );

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_DC;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Set the analysis temperature
  roSimn.m_oCmdDC.m_osTempC = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdDC.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdDC.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdDC.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapDC::OnScale( wxCommandEvent & roEvtCmd )
{
  InitScale( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapDC::OnSrcName( wxCommandEvent & roEvtCmd )
{
  InitSwpUnits( );

  // Permit the base class to process this event as well
  roEvtCmd.Skip( );
}

//*****************************************************************************
