/*
 *  $Id: trimmed-mean.c 21830 2019-01-14 17:38:07Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/datafield.h>
#include <libprocess/arithmetic.h>
#include <libprocess/elliptic.h>
#include <libprocess/filters.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define TRIMMED_MEAN_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    LOWEST_CHANGED = 0,
    HIGHEST_CHANGED = 1,
    JUST_FIX_IT = 2,
};

typedef struct {
    gdouble percentlowest;
    gdouble percenthighest;
    gint size;
    gboolean trim_symm;
    gboolean do_extract;
} TrimmedMeanArgs;

typedef struct {
    TrimmedMeanArgs *args;

    GtkObject *radius;
    GtkObject *size;
    GtkObject *percentlowest;
    GtkObject *nlowest;
    GtkObject *percenthighest;
    GtkObject *nhighest;
    GtkWidget *highest_header;
    GtkWidget *do_extract;
    GtkWidget *trim_symm;
    gboolean in_update;

    GwySIValueFormat *vfpix;
    gdouble pixelscale;
    gdouble percentscale;
} TrimmedMeanControls;

static gboolean module_register             (void);
static void     trimmed_mean                (GwyContainer *data,
                                             GwyRunType run);
static gboolean trimmed_mean_dialogue       (TrimmedMeanArgs *args,
                                             GwyDataField *dfield);
static void     radius_changed              (GtkAdjustment *adj,
                                             TrimmedMeanControls *controls);
static void     size_changed                (GtkAdjustment *adj,
                                             TrimmedMeanControls *controls);
static void     percentlowest_changed       (GtkAdjustment *adj,
                                             TrimmedMeanControls *controls);
static void     nlowest_changed             (GtkAdjustment *adj,
                                             TrimmedMeanControls *controls);
static void     percenthighest_changed      (GtkAdjustment *adj,
                                             TrimmedMeanControls *controls);
static void     nhighest_changed            (GtkAdjustment *adj,
                                             TrimmedMeanControls *controls);
static void     trim_symm_changed           (GtkToggleButton *toggle,
                                             TrimmedMeanControls *controls);
static void     do_extract_changed          (GtkToggleButton *toggle,
                                             TrimmedMeanControls *controls);
static void     update_sensitivity          (TrimmedMeanControls *controls);
static void     update_rank_ranges          (TrimmedMeanControls *controls);
static void     trimmed_mean_dialogue_update(TrimmedMeanControls *controls,
                                             const TrimmedMeanArgs *args);
static void     trimmed_mean_sanitize_args  (TrimmedMeanArgs *args);
static void     trimmed_mean_load_args      (GwyContainer *container,
                                             TrimmedMeanArgs *args);
static void     trimmed_mean_save_args      (GwyContainer *container,
                                             TrimmedMeanArgs *args);

static const TrimmedMeanArgs trimmed_mean_defaults = {
    5.0, 5.0,
    20,
    TRUE, TRUE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Trimmed mean filtering and leveling."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, trimmed_mean)

static gboolean
module_register(void)
{
    gwy_process_func_register("trimmed_mean",
                              (GwyProcessFunc)&trimmed_mean,
                              N_("/_Level/_Trimmed Mean..."),
                              NULL,
                              TRIMMED_MEAN_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Trimmed mean leveling and filter"));

    return TRUE;
}

static void
trimmed_mean(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *kernel, *background;
    GtkWindow *window;
    TrimmedMeanArgs args;
    gint oldid, newid, kres, n, nlowest, nhighest, xres, yres;
    GQuark dquark;
    gboolean ok = TRUE;

    g_return_if_fail(run & TRIMMED_MEAN_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_KEY, &dquark,
                                     GWY_APP_DATA_FIELD_ID, &oldid,
                                     0);
    g_return_if_fail(dfield && dquark);

    trimmed_mean_load_args(gwy_app_settings_get(), &args);

    if (run == GWY_RUN_INTERACTIVE) {
        ok = trimmed_mean_dialogue(&args, dfield);
        trimmed_mean_save_args(gwy_app_settings_get(), &args);
    }

    if (!ok)
        return;

    kres = 2*args.size + 1;
    n = gwy_data_field_get_elliptic_area_size(kres, kres);

    window = gwy_app_find_window_for_channel(data, oldid);
    gwy_app_wait_start(window, _("Filtering..."));

    /* TODO */
    nlowest = (gint)floor(args.percentlowest/100.0 * n);
    nhighest = (gint)floor(args.percenthighest/100.0 * n);

    kernel = gwy_data_field_new(kres, kres, 1.0, 1.0, TRUE);
    gwy_data_field_elliptic_area_fill(kernel, 0, 0, kres, kres, 1.0);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    background = gwy_data_field_duplicate(dfield);
    ok = gwy_data_field_area_filter_trimmed_mean(background, kernel, 0, 0,
                                                 xres, yres, nlowest, nhighest,
                                                 gwy_app_wait_set_fraction);
    g_object_unref(kernel);
    gwy_app_wait_finish();
    if (!background)
        return;

    gwy_app_undo_qcheckpointv(data, 1, &dquark);
    gwy_data_field_subtract_fields(dfield, dfield, background);
    gwy_data_field_data_changed(dfield);
    gwy_app_channel_log_add_proc(data, oldid, oldid);

    if (!args.do_extract) {
        g_object_unref(background);
        return;
    }

    newid = gwy_app_data_browser_add_data_field(background, data, TRUE);
    g_object_unref(background);
    gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            0);
    gwy_app_set_data_field_title(data, newid, _("Background"));
    gwy_app_channel_log_add(data, oldid, newid, NULL, NULL);
}

static gboolean
trimmed_mean_dialogue(TrimmedMeanArgs *args, GwyDataField *dfield)
{
    GtkWidget *dialog, *table, *spin;
    TrimmedMeanControls controls;
    gint response, row;
    gdouble q, xr, yr;

    gwy_clear(&controls, 1);
    controls.args = args;

    controls.vfpix
        = gwy_data_field_get_value_format_xy(dfield,
                                             GWY_SI_UNIT_FORMAT_VFMARKUP, NULL);

    /* FIXME: this is bogus for non-square pixels anyway */
    xr = gwy_data_field_get_dx(dfield);
    yr = gwy_data_field_get_dy(dfield);
    controls.pixelscale = hypot(xr, yr)/controls.vfpix->magnitude;

    dialog = gtk_dialog_new_with_buttons(_("Trimmed Mean"), NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    table = gtk_table_new(11, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table,
                       FALSE, FALSE, 4);
    row = 0;
    controls.in_update = TRUE;

    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Kernel Size")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    q = controls.pixelscale;
    controls.radius = gtk_adjustment_new(q*args->size, q, 1024*q, q, 10*q, 0);
    spin = gwy_table_attach_adjbar(table, row, _("Real _radius:"),
                                   controls.vfpix->units,
                                   controls.radius, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin),
                               controls.vfpix->precision);
    g_signal_connect(controls.radius, "value-changed",
                     G_CALLBACK(radius_changed), &controls);
    row++;

    controls.size = gtk_adjustment_new(args->size, 1, 1024, 1, 10, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Pixel radius:"), _("px"),
                                   controls.size, GWY_HSCALE_SQRT);
    g_signal_connect(controls.size, "value-changed",
                     G_CALLBACK(size_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Trim Lowest")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.percentlowest = gtk_adjustment_new(args->percentlowest,
                                                0.0, 100.0, 0.001, 0.1, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Percentile:"), "%",
                                   controls.percentlowest, GWY_HSCALE_SQRT);
    g_signal_connect(controls.percentlowest, "value-changed",
                     G_CALLBACK(percentlowest_changed), &controls);
    row++;

    /* We will fix the range later. */
    controls.nlowest = gtk_adjustment_new(0.0, 0.0, 1024.0, 1.0, 10.0, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Number of values:"), NULL,
                                   controls.nlowest,
                                   GWY_HSCALE_SQRT | GWY_HSCALE_SNAP);
    g_signal_connect(controls.nlowest, "value-changed",
                     G_CALLBACK(nlowest_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.trim_symm
        = gtk_check_button_new_with_mnemonic(_("_Trim symmetrically"));
    gtk_table_attach(GTK_TABLE(table), controls.trim_symm,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.trim_symm),
                                 args->trim_symm);
    g_signal_connect(controls.trim_symm, "toggled",
                     G_CALLBACK(trim_symm_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.highest_header = gwy_label_new_header(_("Trim Highest"));
    gtk_table_attach(GTK_TABLE(table), controls.highest_header,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.percenthighest = gtk_adjustment_new(args->percenthighest,
                                                 0.0, 100.0, 0.001, 0.1, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Percentile:"), "%",
                                   controls.percenthighest, GWY_HSCALE_SQRT);
    g_signal_connect(controls.percenthighest, "value-changed",
                     G_CALLBACK(percenthighest_changed), &controls);
    row++;

    /* We will fix the range later. */
    controls.nhighest = gtk_adjustment_new(0.0, 0.0, 1024.0, 1.0, 10.0, 0);
    spin = gwy_table_attach_adjbar(table, row, _("_Number of values:"), NULL,
                                   controls.nhighest,
                                   GWY_HSCALE_SQRT | GWY_HSCALE_SNAP);
    g_signal_connect(controls.nhighest, "value-changed",
                     G_CALLBACK(nhighest_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Options")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.do_extract
        = gtk_check_button_new_with_mnemonic(_("E_xtract background"));
    gtk_table_attach(GTK_TABLE(table), controls.do_extract,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.do_extract),
                                 args->do_extract);
    g_signal_connect(controls.do_extract, "toggled",
                     G_CALLBACK(do_extract_changed), &controls);
    row++;

    controls.in_update = FALSE;
    update_rank_ranges(&controls);
    update_sensitivity(&controls);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            gwy_si_unit_value_format_free(controls.vfpix);
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            trimmed_mean_dialogue_update(&controls, &trimmed_mean_defaults);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);
    gwy_si_unit_value_format_free(controls.vfpix);

    return TRUE;
}

static void
enforce_constraints(TrimmedMeanControls *controls, gint what)
{
    TrimmedMeanArgs *args = controls->args;
    gdouble ps = controls->percentscale;
    gint n1, n2, s;

    g_return_if_fail(!controls->in_update);

    controls->in_update = TRUE;
    if (what == LOWEST_CHANGED) {
        n1 = (gint)floor(args->percentlowest/ps);
        if (!args->trim_symm) {
            n2 = (gint)floor(args->percenthighest/ps);
            if (n1 + n2 > 100.0/ps - 0.5)
                n2 = GWY_ROUND(100.0/ps - 1.0 - n1);
        }
        else
            n2 = n1;
        args->percenthighest = n2*ps;
    }
    else if (what == HIGHEST_CHANGED) {
        n2 = (gint)floor(args->percenthighest/ps);
        if (!args->trim_symm) {
            n1 = (gint)floor(args->percentlowest/ps);
            if (n1 + n2 > 100.0/ps - 0.5)
                n1 = GWY_ROUND(100.0/ps - 1.0 - n2);
        }
        else
            n1 = n2;
        args->percentlowest = n1*ps;
    }
    else {
        n1 = (gint)floor(args->percentlowest/ps);
        n2 = (gint)floor(args->percenthighest/ps);
        if (args->trim_symm && n2 != n1)
            n1 = n2 = (n1 + n2)/2;
        s = n1 + n2;
        if (s > 100.0/ps - 0.5) {
            n1 = (gint)floor(100.0*n1/s/ps);
            n2 = (gint)floor(100.0*n2/s/ps);
            if (n1 + n2 > 100.0/ps - 0.5 && n2 > n1)
                n2--;
            if (n1 + n2 > 100.0/ps - 0.5 && n1 > n2)
                n1--;
            if (n1 + n2 > 100.0/ps - 0.5 && n2 > 0)
                n2--;
            if (n1 + n2 > 100.0/ps - 0.5 && n1 > 0)
                n1--;
            if (n1 + n2 > 100.0/ps - 0.5)
                n1 = n2 = 0;   /* Can this happen? */
        }
        args->percentlowest = n1*ps;
        args->percenthighest = n2*ps;
    }
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percentlowest),
                             args->percentlowest);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percenthighest),
                             args->percenthighest);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->nlowest), n1);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->nhighest), n2);
    controls->in_update = FALSE;
}

static void
radius_changed(GtkAdjustment *adj, TrimmedMeanControls *controls)
{
    TrimmedMeanArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->size = GWY_ROUND(gtk_adjustment_get_value(adj)/controls->pixelscale);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->size), args->size);
    controls->in_update = FALSE;
    update_rank_ranges(controls);
}

static void
size_changed(GtkAdjustment *adj, TrimmedMeanControls *controls)
{
    TrimmedMeanArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->size = gwy_adjustment_get_int(adj);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->radius),
                             args->size*controls->pixelscale);
    controls->in_update = FALSE;
    update_rank_ranges(controls);
}

static void
percentlowest_changed(GtkAdjustment *adj, TrimmedMeanControls *controls)
{
    if (controls->in_update)
        return;

    controls->args->percentlowest = gtk_adjustment_get_value(adj);
    enforce_constraints(controls, LOWEST_CHANGED);
}

static void
nlowest_changed(GtkAdjustment *adj, TrimmedMeanControls *controls)
{
    if (controls->in_update)
        return;

    controls->args->percentlowest
        = gtk_adjustment_get_value(adj) * controls->percentscale;
    enforce_constraints(controls, LOWEST_CHANGED);
}

static void
percenthighest_changed(GtkAdjustment *adj, TrimmedMeanControls *controls)
{
    if (controls->in_update)
        return;

    controls->args->percenthighest = gtk_adjustment_get_value(adj);
    enforce_constraints(controls, HIGHEST_CHANGED);
}

static void
nhighest_changed(GtkAdjustment *adj, TrimmedMeanControls *controls)
{
    if (controls->in_update)
        return;

    if (controls->in_update)
        return;

    controls->args->percenthighest
        = gtk_adjustment_get_value(adj) * controls->percentscale;
    enforce_constraints(controls, HIGHEST_CHANGED);
}

static void
do_extract_changed(GtkToggleButton *toggle, TrimmedMeanControls *controls)
{
    TrimmedMeanArgs *args = controls->args;

    args->do_extract = gtk_toggle_button_get_active(toggle);
}

static void
trim_symm_changed(GtkToggleButton *toggle, TrimmedMeanControls *controls)
{
    TrimmedMeanArgs *args = controls->args;

    args->trim_symm = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
    update_rank_ranges(controls);
    enforce_constraints(controls, JUST_FIX_IT);
}

static void
update_sensitivity(TrimmedMeanControls *controls)
{
    TrimmedMeanArgs *args = controls->args;
    gboolean is_asymm = !args->trim_symm;

    gwy_table_hscale_set_sensitive(controls->percenthighest, is_asymm);
    gwy_table_hscale_set_sensitive(controls->nhighest, is_asymm);
    gtk_widget_set_sensitive(controls->highest_header, is_asymm);
}

static void
trimmed_mean_dialogue_update(TrimmedMeanControls *controls,
                             const TrimmedMeanArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->size), args->size);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->trim_symm),
                                 args->trim_symm);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percentlowest),
                             args->percentlowest);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->percenthighest),
                             args->percenthighest);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->do_extract),
                                 args->do_extract);
}

static void
update_rank_ranges(TrimmedMeanControls *controls)
{
    TrimmedMeanArgs *args = controls->args;
    gint kres = 2*args->size + 1;
    gint n = gwy_data_field_get_elliptic_area_size(kres, kres);

    controls->percentscale = 100.0/n;
    controls->in_update = TRUE;
    if (args->trim_symm) {
        n = (n-1)/2;
        g_object_set(controls->nlowest, "upper", 1.0*n, NULL);
        g_object_set(controls->nhighest, "upper", 1.0*n, NULL);
        g_object_set(controls->percentlowest, "upper", 50.0, NULL);
        g_object_set(controls->percenthighest, "upper", 50.0, NULL);
    }
    else {
        g_object_set(controls->nlowest, "upper", n - 1.0, NULL);
        g_object_set(controls->nhighest, "upper", n - 1.0, NULL);
        g_object_set(controls->percentlowest, "upper", 100.0, NULL);
        g_object_set(controls->percenthighest, "upper", 100.0, NULL);
    }
    controls->in_update = FALSE;
    enforce_constraints(controls, JUST_FIX_IT);
}

static const gchar do_extract_key[]     = "/module/trimmed-mean/do_extract";
static const gchar percenthighest_key[] = "/module/trimmed-mean/percenthighest";
static const gchar percentlowest_key[]  = "/module/trimmed-mean/percentlowest";
static const gchar size_key[]           = "/module/trimmed-mean/size";
static const gchar trim_symm_key[]      = "/module/trimmed-mean/trim_symm";

static void
trimmed_mean_sanitize_args(TrimmedMeanArgs *args)
{
    args->percentlowest = CLAMP(args->percentlowest, 0.0, 100.0);
    args->percenthighest = CLAMP(args->percenthighest, 0.0, 100.0);
    args->size = CLAMP(args->size, 1, 1024);
    args->do_extract = !!args->do_extract;
    args->trim_symm = !!args->trim_symm;
    if (args->trim_symm) {
        args->percenthighest
            = args->percentlowest
            = MIN(args->percentlowest, args->percenthighest);
    }
    if (args->percentlowest + args->percenthighest >= 100.0)
        args->percenthighest = args->percentlowest = 49.0;
}

static void
trimmed_mean_load_args(GwyContainer *container,
                       TrimmedMeanArgs *args)
{
    *args = trimmed_mean_defaults;

    gwy_container_gis_int32_by_name(container, size_key, &args->size);
    gwy_container_gis_double_by_name(container, percentlowest_key,
                                     &args->percentlowest);
    gwy_container_gis_double_by_name(container, percenthighest_key,
                                     &args->percenthighest);
    gwy_container_gis_boolean_by_name(container, do_extract_key,
                                      &args->do_extract);
    gwy_container_gis_boolean_by_name(container, trim_symm_key,
                                      &args->trim_symm);
    trimmed_mean_sanitize_args(args);
}

static void
trimmed_mean_save_args(GwyContainer *container,
                       TrimmedMeanArgs *args)
{
    gwy_container_set_int32_by_name(container, size_key, args->size);
    gwy_container_set_double_by_name(container, percentlowest_key,
                                     args->percentlowest);
    gwy_container_set_double_by_name(container, percenthighest_key,
                                     args->percenthighest);
    gwy_container_set_boolean_by_name(container, do_extract_key,
                                      args->do_extract);
    gwy_container_set_boolean_by_name(container, trim_symm_key,
                                      args->trim_symm);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
