/*
   Hotswapform.cpp - Implementation of the HotswapForm class

   Copyright (C) 2003 Tim Stadelmann

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/



/** HEADERS **/

/* KDE headers */
#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmainwindow.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>

/* QT headers */
#include <qhgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qsizepolicy.h>
#include <qvbox.h>
#include <qvgroupbox.h>
#include <qwhatsthis.h>

/* Program headers */
#include "hotswapform.h"
#include "device.h"


/** MACROS **/

#define SYSTRAY_SIZE 22



/** FUNCTIONS **/

HotswapForm::HotswapForm (QWidget *parent,
			  const unsigned int ide_controller,
			  QString hotswap_path)
  : KMainWindow (parent, 0, WDestructiveClose)
{
  /* Create the actions for the menu items and main window buttons and
     connect them to the appropriate slots.  */
  swap_action = new KAction (i18n ("&Swap"), "khotswap_swap",
			     CTRL + Key_S,
			     this, SLOT (swap ()),
			     actionCollection (), "device_swap");
  remove_action = new KAction (i18n ("&Remove"), "khotswap_remove",
			       CTRL + Key_R,
			       this, SLOT (remove ()),
			       actionCollection (), "device_remove");
  insert_action = new KAction (i18n ("&Insert"), "khotswap_insert",
			       CTRL + Key_I,
			       this, SLOT (insert ()),
			       actionCollection (), "device_insert");
  quit_action = KStdAction::quit (this, SLOT (quit ()),
				  actionCollection ());

  /* Construct the "Device" menu.  */
  device_menu = new KPopupMenu (this);
  menuBar ()->insertItem (i18n ("&Device"), device_menu);
  swap_action->plug (device_menu);
  remove_action->plug (device_menu);
  insert_action->plug (device_menu);
  device_menu->insertSeparator ();
  quit_action->plug (device_menu);
  
  /* Construct the "Help" menu.  */
  help_menu = helpMenu ();
  menuBar ()->insertItem (i18n ("&Help"), help_menu);

  /* Construct and show the central widget.  */

  QVBox *main_form;
  QHGroupBox *device_frame;
  QVGroupBox *action_frame;

  /* Create a QVBox as the container.  */
  main_form = new QVBox (this);
  main_form->setMargin (10);
  main_form->setSpacing (10);

  /* Create a titled box containing the device label.  */
  device_frame = new QHGroupBox (i18n ("Configured Device"), main_form);
  device_icon = new QLabel (device_frame);
  device_icon->setSizePolicy (QSizePolicy::Fixed, QSizePolicy::Fixed, false);
  device_text = new QLabel (device_frame);
  device_text->setAlignment (AlignCenter);

  QWhatsThis::add (device_frame,
		   i18n ("Shows the currently configured device, if one "
			 "is present."));
  
  /* Create a titled box containing the action buttons, and connect
     their clicked () signals to the appropriate slots.  */

  KIconLoader *icon_loader = KGlobal::iconLoader ();

  action_frame = new QVGroupBox (i18n ("Action"), main_form);
  swap_button = new QPushButton (icon_loader->loadIconSet ("khotswap_swap",
							   KIcon::Small),
				 i18n ("Swap"), action_frame);
  connect (swap_button, SIGNAL (clicked ()), this, SLOT (swap ()));
  remove_button = new QPushButton (icon_loader->loadIconSet ("khotswap_remove",
							     KIcon::Small),
				   i18n ("Remove"), action_frame);
  connect (remove_button, SIGNAL (clicked ()), this, SLOT (remove ()));
  insert_button = new QPushButton (icon_loader->loadIconSet ("khotswap_insert",
							     KIcon::Small),
				   i18n ("Insert"), action_frame);
  connect (insert_button, SIGNAL (clicked ()), this, SLOT (insert ()));

  QWhatsThis::add (action_frame,
		   i18n ("Select 'Swap' if you want to exchange your device "
			 "for a different or and 'Remove' if you want to "
			 "disconnect it.  If no device is present and you "
			 "want to connect one choose 'Insert'."));

  setCentralWidget (main_form);
  main_form->show ();

  /* The layout of the main window looks awkward if the user enlarges
     it.  */
  layout ()->setResizeMode (QLayout::Fixed);
  
  /* Create and show the system tray icon.  */
  systray_icon = new HotswapSystemTray (this);
  swap_action->plug (systray_icon->get_context_menu ());
  remove_action->plug (systray_icon->get_context_menu ());
  insert_action->plug (systray_icon->get_context_menu ());
  systray_icon->show ();

  /* Create a Device object and update the user interface according to
     the information it provides.  */
  device = new Device (this, ide_controller, hotswap_path);
  update ();
}

/* The destructor of the HotswapForm class.  Deletes objects created
   in the constructor.  */
HotswapForm:: ~HotswapForm ()
{
  /* Delete the objects created in the constructor.  */
  delete device_menu;
  delete help_menu;
  delete swap_action;
  delete remove_action;
  delete insert_action;
  delete quit_action;
  delete device_icon;
  delete device_text;
  delete swap_button;
  delete remove_button;
  delete insert_button;
  delete device;
}

/* This method updates the user interface to reflect the state of the
   Device object.  */
void
HotswapForm::update ()
{
  KIconLoader *icon_loader = KGlobal::iconLoader ();

  device_text->setText (device->name ());

  if (device->present ())
    {
      device_icon->setPixmap (icon_loader->loadIcon ("hdd_unmount",  
						     KIcon::Desktop));
      systray_icon->setPixmap (icon_loader->loadIcon ("hdd_unmount",  
						      KIcon::Panel,
						      SYSTRAY_SIZE));

      swap_action->setEnabled (true);
      remove_action->setEnabled (true);
      insert_action->setEnabled (false);
      swap_button->setEnabled (true);
      remove_button->setEnabled (true);
      insert_button->setEnabled (false);
    }
  else
    {
      device_icon->setPixmap (icon_loader->loadIcon ("khotswap_no_device",  
						     KIcon::Desktop));
      systray_icon->setPixmap (icon_loader->loadIcon ("khotswap_no_device",  
						      KIcon::Panel,
						      SYSTRAY_SIZE));

      swap_action->setEnabled (false);
      remove_action->setEnabled (false);
      insert_action->setEnabled (true);
      swap_button->setEnabled (false);
      remove_button->setEnabled (false);
      insert_button->setEnabled (true);
    }
}

void
HotswapForm::swap ()
{
  device->remove ();
  
  if (KMessageBox::warningContinueCancel
      (this, i18n ("You can now exchange the device."),
       i18n ("Exchange Device"))
      == KMessageBox::Continue)
    device->insert ();
  else
    update ();
}

/* This method unregisters the device and tells the user that he can
   now physically disconnect it.  */
void
HotswapForm::remove ()
{
  device->remove ();
  update ();

  KMessageBox::information
    (this, i18n ("You can now remove the device."),
     i18n ("Remove Device"));
}

/* This method prompts the users to connect a device and proceeds to
   rescan the IDE bus upon receiving a confirmation.  */
void
HotswapForm::insert ()
{
  if (KMessageBox::warningContinueCancel
      (this, i18n ("Please insert the new device and click `Continue'"),
       i18n ("Insert Device"))
      == KMessageBox::Continue)
    {
      device->insert ();
      update ();
    }
}


/* This method terminates the application.  */
void
HotswapForm::quit ()
{
  KApplication::kApplication ()->quit ();
}


void
HotswapForm::saveProperties (KConfig *config)
{
  config->writeEntry ("systray", isHidden ());
}


void
HotswapForm::readProperties (KConfig *config)
{
  systray = config->readEntry ("systray");
}
