/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ilightkit.h"

#include "ivtk.h"

#include <vtkLight.h>
#include <vtkMath.h>
#include <vtkRenderer.h>


namespace iLightKit_Private
{
	void UpdateLight(vtkLight *l, double elevation, double azimuth, double ex[3], double ey[3], double ez[3])
	{
		int i;
		double org[3], pos[3];

#ifdef IVTK_PRE54
		elevation *= vtkMath::DegreesToRadians();
		azimuth *= vtkMath::DegreesToRadians();
#else
		elevation = vtkMath::RadiansFromDegrees(elevation);
		azimuth = vtkMath::RadiansFromDegrees(azimuth);
#endif
		org[0] = cos(elevation)*sin(azimuth);
		org[1] = sin(elevation); 
		org[2] = cos(elevation)*cos(azimuth);
 
		for(i=0; i<3; i++)
		{
			pos[i] = org[0]*ex[i] + org[1]*ey[i] + org[2]*ez[i];
		}
		l->SetPosition(pos);
	}
};


using namespace iLightKit_Private;


//
//  Main class
//
iLightKit* iLightKit::New()
{
	return new iLightKit;
}


iLightKit::iLightKit()
{
	this->vtkLightKit::SetKeyLightAngle(45.0,0.0);
	this->vtkLightKit::SetFillLightAngle(-75.0,0.0);
	this->vtkLightKit::SetBackLightAngle(0.0,110.0);

	this->vtkLightKit::SetKeyLightIntensity(1.0);
	this->vtkLightKit::SetKeyToFillRatio(3.0);
	this->vtkLightKit::SetKeyToHeadRatio(6.0);
	this->vtkLightKit::SetKeyToBackRatio(3.0);

	mAngles[0] = mAngles[1] = 0.0;

	this->UpdateAngles();
}


iLightKit::~iLightKit()
{
}


//
//  This is to fix a bug with off-screen rednering: the key light must be first.
//
void iLightKit::AddLightsToRenderer(vtkRenderer *renderer) 
{
	if(renderer != 0) 
	{
		renderer->AddLight(this->KeyLight);
		renderer->AddLight(this->FillLight);
		renderer->AddLight(this->HeadLight);
		renderer->AddLight(this->BackLight0);
		renderer->AddLight(this->BackLight1);
	}
}


void iLightKit::SetAngles(float e, float a)
{
	if(fabs(e-mAngles[0])>1.0e-5 || fabs(a-mAngles[1])>1.0e-5)
	{
		mAngles[0] = e;
		mAngles[1] = a;
		this->UpdateAngles();
	}
}


void iLightKit::SetIntensities(float k, float f, float h)
{
	if(k > 1.0e-10)
	{
		this->SetKeyLightIntensity(k);
		if(f > 1.0e-10) this->SetKeyToFillRatio(k/f);
		if(h > 1.0e-10) this->SetKeyToHeadRatio(k/h);
	}
}


void iLightKit::GetIntensities(float i[3]) const
{
	i[0] = this->KeyLightIntensity;
	i[1] = this->KeyLightIntensity/this->KeyToFillRatio;
	i[2] = this->KeyLightIntensity/this->KeyToHeadRatio;
}


void iLightKit::UpdateAngles()
{
	double ex[3], ey[3], ez[3];

#ifdef IVTK_PRE54
	double ce = cos(mAngles[0]*vtkMath::DegreesToRadians());
	double se = sin(mAngles[0]*vtkMath::DegreesToRadians());
	double ca = cos(mAngles[1]*vtkMath::DegreesToRadians());
	double sa = sin(mAngles[1]*vtkMath::DegreesToRadians());
#else
	double ce = cos(vtkMath::RadiansFromDegrees(mAngles[0]));
	double se = sin(vtkMath::RadiansFromDegrees(mAngles[0]));
	double ca = cos(vtkMath::RadiansFromDegrees(mAngles[1]));
	double sa = sin(vtkMath::RadiansFromDegrees(mAngles[1]));
#endif

	ex[0] =  ce*ca; ex[1] = 0.0; ex[2] = -ce*sa;
	ey[0] = -se*sa; ey[1] =  ce; ey[2] = -se*ca;
	ez[0] =  ce*sa; ez[1] =  se; ez[2] =  ce*ca;

	UpdateLight(this->KeyLight,this->KeyLightAngle[0],this->KeyLightAngle[1],ex,ey,ez);
	UpdateLight(this->FillLight,this->FillLightAngle[0],this->FillLightAngle[1],ex,ey,ez);
//	UpdateLight(this->Headlight,0.0,0.0,ex,ey,ez);
	UpdateLight(this->BackLight0,this->BackLightAngle[0],this->BackLightAngle[1],ex,ey,ez);
	UpdateLight(this->BackLight1,this->BackLightAngle[0],this->BackLightAngle[1],ex,ey,ez);
}
