# - Build binary and source package installers
#
# The CPack module generates binary and source installers in a variety
# of formats using the cpack program. Inclusion of the CPack module
# adds two new targets to the resulting makefiles, package and
# package_source, which build the binary and source installers,
# respectively. The generated binary installers contain everything
# installed via CMake's INSTALL command (and the deprecated
# INSTALL_FILES, INSTALL_PROGRAMS, and INSTALL_TARGETS commands).
#
# For certain kinds of binary installers (including the graphical
# installers on Mac OS X and Windows), CPack generates installers that
# allow users to select individual application components to
# install. The contents of each of the components are identified by
# the COMPONENT argument of CMake's INSTALL command. These components
# can be annotated with user-friendly names and descriptions,
# inter-component dependencies, etc., and grouped in various ways to
# customize the resulting installer. See the cpack_add_* commands,
# described below, for more information about component-specific
# installations.
#
# The CPACK_GENERATOR variable has different meanings in different
# contexts. In your CMakeLists.txt file, CPACK_GENERATOR is a
# *list of generators*: when run with no other arguments, CPack
# will iterate over that list and produce one package for each
# generator. In a CPACK_PROJECT_CONFIG_FILE, though, CPACK_GENERATOR
# is a *string naming a single generator*. If you need per-cpack-
# generator logic to control *other* cpack settings, then you need
# a CPACK_PROJECT_CONFIG_FILE.
#
# The CMake source tree itself contains a CPACK_PROJECT_CONFIG_FILE.
# See the top level file CMakeCPackOptions.cmake.in for an example.
#
# If set, the CPACK_PROJECT_CONFIG_FILE is included automatically
# on a per-generator basis. It only need contain overrides.
#
# Here's how it works:
# - cpack runs
# - it includes CPackConfig.cmake
# - it iterates over the generators listed in that file's
#     CPACK_GENERATOR list variable (unless told to use just a
#     specific one via -G on the command line...)
#
# - foreach generator, it then
#   - sets CPACK_GENERATOR to the one currently being iterated
#   - includes the CPACK_PROJECT_CONFIG_FILE
#   - produces the package for that generator
#
# This is the key: For each generator listed in CPACK_GENERATOR
# in CPackConfig.cmake, cpack will *reset* CPACK_GENERATOR
# internally to *the one currently being used* and then include
# the CPACK_PROJECT_CONFIG_FILE.
#
# Before including this CPack module in your CMakeLists.txt file,
# there are a variety of variables that can be set to customize
# the resulting installers. The most commonly-used variables are:
#
#   CPACK_PACKAGE_NAME - The name of the package (or application). If
#   not specified, defaults to the project name.
#
#   CPACK_PACKAGE_VENDOR - The name of the package vendor (e.g.,
#   "Kitware").
#
#   CPACK_PACKAGE_VERSION_MAJOR - Package major Version
#
#   CPACK_PACKAGE_VERSION_MINOR - Package minor Version
#
#   CPACK_PACKAGE_VERSION_PATCH - Package patch Version
#
#   CPACK_PACKAGE_DESCRIPTION_FILE - A text file used to describe the
#   project. Used, for example, the introduction screen of a
#   CPack-generated Windows installer to describe the project.
#
#   CPACK_PACKAGE_DESCRIPTION_SUMMARY - Short description of the
#   project (only a few words).
#
#   CPACK_PACKAGE_FILE_NAME - The name of the package file to generate,
#   not including the extension. For example, cmake-2.6.1-Linux-i686.
#
#   CPACK_PACKAGE_INSTALL_DIRECTORY - Installation directory on the
#   target system, e.g., "CMake 2.5".
#
#   CPACK_PROJECT_CONFIG_FILE - File included at cpack time, once per
#   generator after setting CPACK_GENERATOR to the actual generator
#   being used. Allows per-generator setting of CPACK_* variables at
#   cpack time.
#
#   CPACK_RESOURCE_FILE_LICENSE - License file for the project, which
#   will typically be displayed to the user (often with an explicit
#   "Accept" button, for graphical installers) prior to installation.
#
#   CPACK_RESOURCE_FILE_README - ReadMe file for the project, which
#   typically describes in some detail
#
#   CPACK_RESOURCE_FILE_WELCOME - Welcome file for the project, which
#   welcomes users to this installer. Typically used in the graphical
#   installers on Windows and Mac OS X.
#
#   CPACK_MONOLITHIC_INSTALL - Disables the component-based 
#   installation mechanism, so that all components are always installed.
#
#   CPACK_GENERATOR - List of CPack generators to use. If not
#   specified, CPack will create a set of options (e.g.,
#   CPACK_BINARY_NSIS) allowing the user to enable/disable individual
#   generators.
#
#   CPACK_OUTPUT_CONFIG_FILE - The name of the CPack configuration file
#   for binary installers that will be generated by the CPack
#   module. Defaults to CPackConfig.cmake.
#
#   CPACK_PACKAGE_EXECUTABLES - Lists each of the executables along
#   with a text label, to be used to create Start Menu shortcuts on
#   Windows. For example, setting this to the list ccmake;CMake will
#   create a shortcut named "CMake" that will execute the installed
#   executable ccmake.
#
#   CPACK_STRIP_FILES - List of files to be stripped. Starting with
#   CMake 2.6.0 CPACK_STRIP_FILES will be a boolean variable which
#   enables stripping of all files (a list of files evaluates to TRUE
#   in CMake, so this change is compatible).
#
# The following CPack variables are specific to source packages, and 
# will not affect binary packages:
#
#   CPACK_SOURCE_PACKAGE_FILE_NAME - The name of the source package,
#   e.g., cmake-2.6.1
#
#   CPACK_SOURCE_STRIP_FILES - List of files in the source tree that
#   will be stripped. Starting with CMake 2.6.0
#   CPACK_SOURCE_STRIP_FILES will be a boolean variable which enables
#   stripping of all files (a list of files evaluates to TRUE in CMake,
#   so this change is compatible).
#
#   CPACK_SOURCE_GENERATOR - List of generators used for the source
#   packages. As with CPACK_GENERATOR, if this is not specified then
#   CPack will create a set of options (e.g., CPACK_SOURCE_ZIP)
#   allowing users to select which packages will be generated.
#
#   CPACK_SOURCE_OUTPUT_CONFIG_FILE - The name of the CPack
#   configuration file for source installers that will be generated by
#   the CPack module. Defaults to CPackSourceConfig.cmake.
#
#   CPACK_SOURCE_IGNORE_FILES - Pattern of files in the source tree
#   that won't be packaged when building a source package. This is a
#   list of patterns, e.g., /CVS/;/\\.svn/;\\.swp$;\\.#;/#;.*~;cscope.*
#
# The following variables are specific to the DragNDrop installers
# built on Mac OS X:
#
#   CPACK_DMG_VOLUME_NAME - The volume name of the generated disk
#   image. Defaults to CPACK_PACKAGE_FILE_NAME.
#
#   CPACK_DMG_FORMAT - The disk image format. Common values are UDRO
#   (UDIF read-only), UDZO (UDIF zlib-compressed) or UDBZ (UDIF
#   bzip2-compressed). Refer to hdiutil(1) for more information on
#   other available formats.
#
#   CPACK_DMG_DS_STORE - Path to a custom .DS_Store file which e.g.
#   can be used to specify the Finder window position/geometry and
#   layout (such as hidden toolbars, placement of the icons etc.).
#   This file has to be generated by the Finder (either manually or
#   through OSA-script) using a normal folder from which the .DS_Store
#   file can then be extracted.
#
#   CPACK_DMG_BACKGROUND_IMAGE - Path to an image file which is to be
#   used as the background for the Finder Window when the disk image
#   is opened.  By default no background image is set. The background
#   image is applied after applying the custom .DS_Store file.
#
#   CPACK_COMMAND_HDIUTIL - Path to the hdiutil(1) command used to
#   operate on disk image files on Mac OS X. This variable can be used
#   to override the automatically detected command (or specify its
#   location if the auto-detection fails to find it.)
#
#   CPACK_COMMAND_SETFILE - Path to the SetFile(1) command used to set
#   extended attributes on files and directories on Mac OS X. This
#   variable can be used to override the automatically detected
#   command (or specify its location if the auto-detection fails to
#   find it.)
#
#   CPACK_COMMAND_REZ - Path to the Rez(1) command used to compile
#   resources on Mac OS X. This variable can be used to override the
#   automatically detected command (or specify its location if the
#   auto-detection fails to find it.)
#
# Installers built on Mac OS X using the Bundle generator use the
# aforementioned DragNDrop variables, plus the following Bundle-specific
# parameters:
#
#   CPACK_BUNDLE_NAME - The name of the generated bundle.  This
#   appears in the OSX finder as the bundle name.  Required.
#
#   CPACK_BUNDLE_PLIST - Path to an OSX plist file that will be used
#   as the Info.plist for the generated bundle.  This assumes that
#   the caller has generated or specified their own Info.plist file.
#   Required.
#
#   CPACK_BUNDLE_ICON - Path to an OSX icns file that will be used as
#   the icon for the generated bundle.  This is the icon that appears
#   in the OSX finder for the bundle, and in the OSX dock when the
#   bundle is opened.  Required.
#
#   CPACK_BUNDLE_STARTUP_SCRIPT - Path to an executable or script that
#   will be run whenever an end-user double-clicks the generated bundle
#   in the OSX Finder.  Optional.
#
# The following variables are specific to the graphical installers built
# on Windows using the Nullsoft Installation System.
#
#   CPACK_PACKAGE_INSTALL_REGISTRY_KEY - Registry key used when
#   installing this project.
#
#   CPACK_NSIS_INSTALL_ROOT - The default installation directory presented
#   to the end user by the NSIS installer is under this root dir. The full
#   directory presented to the end user is:
#   ${CPACK_NSIS_INSTALL_ROOT}/${CPACK_PACKAGE_INSTALL_DIRECTORY}
#
#   CPACK_NSIS_MUI_ICON - The icon file (.ico) for the generated
#   install program.
#
#   CPACK_NSIS_MUI_UNIICON - The icon file (.ico) for the generated
#   uninstall program.
#
#   CPACK_PACKAGE_ICON - A branding image that will be displayed inside
#   the installer.
#
#   CPACK_NSIS_EXTRA_INSTALL_COMMANDS - Extra NSIS commands that will
#   be added to the install Section.
#
#   CPACK_NSIS_EXTRA_UNINSTALL_COMMANDS - Extra NSIS commands that will
#   be added to the uninstall Section.
#
#   CPACK_NSIS_COMPRESSOR - The arguments that will be passed to the
#   NSIS SetCompressor command.
#
#   CPACK_NSIS_MODIFY_PATH - If this is set to "ON", then an extra page
#   will appear in the installer that will allow the user to choose
#   whether the program directory should be added to the system PATH
#   variable.
#
#   CPACK_NSIS_DISPLAY_NAME - The display name string that appears in
#   the Windows Add/Remove Program control panel
#
#   CPACK_NSIS_PACKAGE_NAME - The title displayed at the top of the
#   installer.
#
#   CPACK_NSIS_INSTALLED_ICON_NAME - A path to the executable that
#   contains the installer icon.
#
#   CPACK_NSIS_HELP_LINK - URL to a web site providing assistance in
#   installing your application.
#
#   CPACK_NSIS_URL_INFO_ABOUT - URL to a web site providing more
#   information about your application.
#
#   CPACK_NSIS_CONTACT - Contact information for questions and comments
#   about the installation process.
#
#   CPACK_NSIS_CREATE_ICONS_EXTRA - Additional NSIS commands for
#   creating start menu shortcuts.
#
#   CPACK_NSIS_DELETE_ICONS_EXTRA -Additional NSIS commands to
#   uninstall start menu shortcuts.
#
#   CPACK_NSIS_EXECUTABLES_DIRECTORY - Creating NSIS start menu links
#   assumes that they are in 'bin' unless this variable is set.
#   For example, you would set this to 'exec' if your executables are
#   in an exec directory.
#
#   CPACK_NSIS_MUI_FINISHPAGE_RUN - Specify an executable to add an option
#   to run on the finish page of the NSIS installer.
#
# The following variable is specific to installers build on Mac OS X
# using PackageMaker:
#
#   CPACK_OSX_PACKAGE_VERSION - The version of Mac OS X that the
#   resulting PackageMaker archive should be compatible
#   with. Different versions of Mac OS X support different
#   features. For example, CPack can only build component-based
#   installers for Mac OS X 10.4 or newer, and can only build
#   installers that download component son-the-fly for Mac OS X 10.5
#   or newer. If left blank, this value will be set to the minimum
#   version of Mac OS X that supports the requested features. Set this
#   variable to some value (e.g., 10.4) only if you want to guarantee
#   that your installer will work on that version of Mac OS X, and
#   don't mind missing extra features available in the installer
#   shipping with later versions of Mac OS X.
#
# The following variables are for advanced uses of CPack:
#
#   CPACK_CMAKE_GENERATOR - What CMake generator should be used if the
#   project is CMake project. Defaults to the value of CMAKE_GENERATOR;
#   few users will want to change this setting.
#
#   CPACK_INSTALL_CMAKE_PROJECTS - List of four values that specify
#   what project to install. The four values are: Build directory,
#   Project Name, Project Component, Directory. If omitted, CPack will
#   build an installer that installers everything.
#
#   CPACK_SYSTEM_NAME - System name, defaults to the value of
#   ${CMAKE_SYSTEM_NAME}.
#
#   CPACK_PACKAGE_VERSION - Package full version, used internally. By
#   default, this is built from CPACK_PACKAGE_VERSION_MAJOR,
#   CPACK_PACKAGE_VERSION_MINOR, and CPACK_PACKAGE_VERSION_PATCH.
#
#   CPACK_TOPLEVEL_TAG - Directory for the installed files.
#
#   CPACK_INSTALL_COMMANDS - Extra commands to install components.
#
#   CPACK_INSTALL_DIRECTORIES - Extra directories to install.
#
# Component-specific installation allows users to select specific sets
# of components to install during the install process. Installation
# components are identified by the COMPONENT argument of CMake's
# INSTALL commands, and should be further described by the following
# CPack commands:
#
#   cpack_add_component - Describes a CPack installation component
#   named by the COMPONENT argument to a CMake INSTALL command.
#
#     cpack_add_component(compname
#                         [DISPLAY_NAME name]
#                         [DESCRIPTION description]
#                         [HIDDEN | REQUIRED | DISABLED ]
#                         [GROUP group]
#                         [DEPENDS comp1 comp2 ... ]
#                         [INSTALL_TYPES type1 type2 ... ]
#                         [DOWNLOADED]
#                         [ARCHIVE_FILE filename])
#
#   The cmake_add_component command describes an installation
#   component, which the user can opt to install or remove as part of
#   the graphical installation process. compname is the name of the
#   component, as provided to the COMPONENT argument of one or more
#   CMake INSTALL commands.
#
#   DISPLAY_NAME is the displayed name of the component, used in
#   graphical installers to display the component name. This value can
#   be any string.
#
#   DESCRIPTION is an extended description of the component, used in
#   graphical installers to give the user additional information about
#   the component. Descriptions can span multiple lines using "\n" as
#   the line separator. Typically, these descriptions should be no
#   more than a few lines long.
#
#   HIDDEN indicates that this component will be hidden in the
#   graphical installer, so that the user cannot directly change
#   whether it is installed or not.
#
#   REQUIRED indicates that this component is required, and therefore
#   will always be installed. It will be visible in the graphical
#   installer, but it cannot be unselected. (Typically, required
#   components are shown greyed out).
#
#   DISABLED indicates that this component should be disabled
#   (unselected) by default. The user is free to select this component
#   for installation, unless it is also HIDDEN.
#
#   DEPENDS lists the components on which this component depends. If
#   this component is selected, then each of the components listed
#   must also be selected. The dependency information is encoded
#   within the installer itself, so that users cannot install
#   inconsitent sets of components.
#
#   GROUP names the component group of which this component is a
#   part. If not provided, the component will be a standalone
#   component, not part of any component group. Component groups are
#   described with the cpack_add_component_group command, detailed
#   below.
#
#   INSTALL_TYPES lists the installation types of which this component
#   is a part. When one of these installations types is selected, this
#   component will automatically be selected. Installation types are
#   described with the cpack_add_install_type command, detailed below.
#
#   DOWNLOADED indicates that this component should be downloaded
#   on-the-fly by the installer, rather than packaged in with the
#   installer itself. For more information, see the cpack_configure_downloads
#   command.
#
#   ARCHIVE_FILE provides a name for the archive file created by CPack
#   to be used for downloaded components. If not supplied, CPack will
#   create a file with some name based on CPACK_PACKAGE_FILE_NAME and
#   the name of the component. See cpack_configure_downloads for more
#   information.
#
#   cpack_add_component_group - Describes a group of related CPack
#   installation components.
#
#     cpack_add_component_group(groupname
#                              [DISPLAY_NAME name]
#                              [DESCRIPTION description]
#                              [PARENT_GROUP parent]
#                              [EXPANDED]
#                              [BOLD_TITLE])
#
#   The cpack_add_component_group describes a group of installation
#   components, which will be placed together within the listing of
#   options. Typically, component groups allow the user to
#   select/deselect all of the components within a single group via a
#   single group-level option. Use component groups to reduce the
#   complexity of installers with many options. groupname is an
#   arbitrary name used to identify the group in the GROUP argument of
#   the cpack_add_component command, which is used to place a
#   component in a group. The name of the group must not conflict with
#   the name of any component.
#
#   DISPLAY_NAME is the displayed name of the component group, used in
#   graphical installers to display the component group name. This
#   value can be any string.
#
#   DESCRIPTION is an extended description of the component group,
#   used in graphical installers to give the user additional
#   information about the components within that group. Descriptions
#   can span multiple lines using "\n" as the line
#   separator. Typically, these descriptions should be no more than a
#   few lines long.
#
#   PARENT_GROUP, if supplied, names the parent group of this group. 
#   Parent groups are used to establish a hierarchy of groups, 
#   providing an arbitrary hierarchy of groups.
#
#   EXPANDED indicates that, by default, the group should show up as
#   "expanded", so that the user immediately sees all of the
#   components within the group. Otherwise, the group will initially
#   show up as a single entry.
#
#   BOLD_TITLE indicates that the group title should appear in bold,
#   to call the user's attention to the group.
#
#   cpack_add_install_type - Add a new installation type containing a
#   set of predefined component selections to the graphical installer.
#                      
#     cpack_add_install_type(typename
#                            [DISPLAY_NAME name])   
#
#   The cpack_add_install_type command identifies a set of preselected
#   components that represents a common use case for an
#   application. For example, a "Developer" install type might include
#   an application along with its header and library files, while an
#   "End user" install type might just include the application's
#   executable. Each component identifies itself with one or more
#   install types via the INSTALL_TYPES argument to
#   cpack_add_component.
#
#   DISPLAY_NAME is the displayed name of the install type, which will
#   typically show up in a drop-down box within a graphical
#   installer. This value can be any string.
#
#   cpack_configure_downloads - Configure CPack to download selected
#   components on-the-fly as part of the installation process.
#
#     cpack_configure_downloads(site
#                               [UPLOAD_DIRECTORY dirname]
#                               [ALL]
#                               [ADD_REMOVE|NO_ADD_REMOVE])
#
#   The cpack_configure_downloads command configures installation-time
#   downloads of selected components. For each downloadable component,
#   CPack will create an archive containing the contents of that
#   component, which should be uploaded to the given site. When the
#   user selects that component for installation, the installer will
#   download and extract the component in place. This feature is
#   useful for creating small installers that only download the
#   requested components, saving bandwidth. Additionally, the
#   installers are small enough that they will be installed as part of
#   the normal installation process, and the "Change" button in
#   Windows Add/Remove Programs control panel will allow one to add or
#   remove parts of the application after the original
#   installation. On Windows, the downloaded-components functionality
#   requires the ZipDLL plug-in for NSIS, available at:
#
#     http://nsis.sourceforge.net/ZipDLL_plug-in
#
#   On Mac OS X, installers that download components on-the-fly can
#   only be built and installed on system using Mac OS X 10.5 or
#   later.
#
#   The site argument is a URL where the archives for downloadable 
#   components will reside, e.g., http://www.cmake.org/files/2.6.1/installer/
#   All of the archives produced by CPack should be uploaded to that location.
#
#   UPLOAD_DIRECTORY is the local directory where CPack will create the 
#   various archives for each of the components. The contents of this
#   directory should be uploaded to a location accessible by the URL given
#   in the site argument. If omitted, CPack will use the directory 
#   CPackUploads inside the CMake binary directory to store the generated
#   archives.
#
#   The ALL flag indicates that all components be downloaded. Otherwise, only 
#   those components explicitly marked as DOWNLOADED or that have a specified 
#   ARCHIVE_FILE will be downloaded. Additionally, the ALL option implies
#   ADD_REMOVE (unless NO_ADD_REMOVE is specified).
#
#   ADD_REMOVE indicates that CPack should install a copy of the installer
#   that can be called from Windows' Add/Remove Programs dialog (via the 
#   "Modify" button) to change the set of installed components. NO_ADD_REMOVE
#   turns off this behavior. This option is ignored on Mac OS X.

#=============================================================================
# Copyright 2006-2009 Kitware, Inc.
#
# Distributed under the OSI-approved BSD License (the "License");
# see accompanying file Copyright.txt for details.
#
# This software is distributed WITHOUT ANY WARRANTY; without even the
# implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the License for more information.
#=============================================================================
# (To distribute this file outside of CMake, substitute the full
#  License text for the above reference.)

# Pick a configuration file
SET(cpack_input_file "${CMAKE_ROOT}/Templates/CPackConfig.cmake.in")
IF(EXISTS "${CMAKE_SOURCE_DIR}/CPackConfig.cmake.in")
  SET(cpack_input_file "${CMAKE_SOURCE_DIR}/CPackConfig.cmake.in")
ENDIF(EXISTS "${CMAKE_SOURCE_DIR}/CPackConfig.cmake.in")
SET(cpack_source_input_file "${CMAKE_ROOT}/Templates/CPackConfig.cmake.in")
IF(EXISTS "${CMAKE_SOURCE_DIR}/CPackSourceConfig.cmake.in")
  SET(cpack_source_input_file "${CMAKE_SOURCE_DIR}/CPackSourceConfig.cmake.in")
ENDIF(EXISTS "${CMAKE_SOURCE_DIR}/CPackSourceConfig.cmake.in")

# Argument-parsing macro from http://www.cmake.org/Wiki/CMakeMacroParseArguments
MACRO(cpack_parse_arguments prefix arg_names option_names)
  SET(${prefix}_DEFAULT_ARGS)
  FOREACH(arg_name ${arg_names})    
    SET(${prefix}_${arg_name})
  ENDFOREACH(arg_name)
  FOREACH(option ${option_names})
    SET(${prefix}_${option} FALSE)
  ENDFOREACH(option)

  SET(current_arg_name DEFAULT_ARGS)
  SET(current_arg_list)
  FOREACH(arg ${ARGN})            
    SET(larg_names ${arg_names})    
    LIST(FIND larg_names "${arg}" is_arg_name)                   
    IF (is_arg_name GREATER -1)
      SET(${prefix}_${current_arg_name} ${current_arg_list})
      SET(current_arg_name ${arg})
      SET(current_arg_list)
    ELSE (is_arg_name GREATER -1)
      SET(loption_names ${option_names})    
      LIST(FIND loption_names "${arg}" is_option)            
      IF (is_option GREATER -1)
        SET(${prefix}_${arg} TRUE)
      ELSE (is_option GREATER -1)
        SET(current_arg_list ${current_arg_list} ${arg})
      ENDIF (is_option GREATER -1)
    ENDIF (is_arg_name GREATER -1)
  ENDFOREACH(arg)
  SET(${prefix}_${current_arg_name} ${current_arg_list})
ENDMACRO(cpack_parse_arguments)

# Macro that appends a SET command for the given variable name (var)
# to the macro named strvar, but only if the variable named "var"
# has been defined. The string will eventually be appended to a CPack
# configuration file.
MACRO(cpack_append_variable_set_command var strvar)
  IF (DEFINED ${var})
    SET(${strvar} "${${strvar}}SET(${var}")
    FOREACH(APPENDVAL ${${var}})
      SET(${strvar} "${${strvar}} ${APPENDVAL}")
    ENDFOREACH(APPENDVAL)
    SET(${strvar} "${${strvar}})\n")
  ENDIF (DEFINED ${var})
ENDMACRO(cpack_append_variable_set_command)

# Macro that appends a SET command for the given variable name (var)
# to the macro named strvar, but only if the variable named "var"
# has been defined and is a string. The string will eventually be
# appended to a CPack configuration file.
MACRO(cpack_append_string_variable_set_command var strvar)
  IF (DEFINED ${var})
    LIST(LENGTH ${var} CPACK_APP_VALUE_LEN)
    IF(${CPACK_APP_VALUE_LEN} EQUAL 1)
      SET(${strvar} "${${strvar}}SET(${var} \"${${var}}\")\n")
    ENDIF(${CPACK_APP_VALUE_LEN} EQUAL 1)
  ENDIF (DEFINED ${var})
ENDMACRO(cpack_append_string_variable_set_command)

# Macro that appends a SET command for the given variable name (var)
# to the macro named strvar, but only if the variable named "var"
# has been set to true. The string will eventually be
# appended to a CPack configuration file.
MACRO(cpack_append_option_set_command var strvar)
  IF (${var})
    LIST(LENGTH ${var} CPACK_APP_VALUE_LEN)
    IF(${CPACK_APP_VALUE_LEN} EQUAL 1)
      SET(${strvar} "${${strvar}}SET(${var} TRUE)\n")
    ENDIF(${CPACK_APP_VALUE_LEN} EQUAL 1)
  ENDIF (${var})
ENDMACRO(cpack_append_option_set_command)

# Macro that adds a component to the CPack installer
MACRO(cpack_add_component compname)
  STRING(TOUPPER ${compname} CPACK_ADDCOMP_UNAME)
  cpack_parse_arguments(CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}
    "DISPLAY_NAME;DESCRIPTION;GROUP;DEPENDS;INSTALL_TYPES;ARCHIVE_FILE"
    "HIDDEN;REQUIRED;DISABLED;DOWNLOADED"
    ${ARGN}
    )
  
  if (CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_DOWNLOADED)
    SET(CPACK_ADDCOMP_STR "\n# Configuration for downloaded component \"${compname}\"\n")
  else ()
    SET(CPACK_ADDCOMP_STR "\n# Configuration for component \"${compname}\"\n")
  endif ()

  IF(NOT CPACK_MONOLITHIC_INSTALL)
    # If the user didn't set CPACK_COMPONENTS_ALL explicitly, update the
    # value of CPACK_COMPONENTS_ALL in the configuration file. This will
    # take care of any components that have been added after the CPack
    # moduled was included.
    IF(NOT CPACK_COMPONENTS_ALL_SET_BY_USER)
      GET_CMAKE_PROPERTY(CPACK_ADDCOMP_COMPONENTS COMPONENTS)
      SET(CPACK_ADDCOMP_STR "${CPACK_ADDCOMP_STR}\nSET(CPACK_COMPONENTS_ALL")
      FOREACH(COMP ${CPACK_ADDCOMP_COMPONENTS})
        SET(CPACK_ADDCOMP_STR "${CPACK_ADDCOMP_STR} ${COMP}")
      ENDFOREACH(COMP)
      SET(CPACK_ADDCOMP_STR "${CPACK_ADDCOMP_STR})\n")
    ENDIF(NOT CPACK_COMPONENTS_ALL_SET_BY_USER)
  ENDIF(NOT CPACK_MONOLITHIC_INSTALL)

  cpack_append_string_variable_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_DISPLAY_NAME
    CPACK_ADDCOMP_STR)
  cpack_append_string_variable_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_DESCRIPTION
    CPACK_ADDCOMP_STR)
  cpack_append_variable_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_GROUP
    CPACK_ADDCOMP_STR)
  cpack_append_variable_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_DEPENDS
    CPACK_ADDCOMP_STR)
  cpack_append_variable_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_INSTALL_TYPES
    CPACK_ADDCOMP_STR)
  cpack_append_string_variable_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_ARCHIVE_FILE
    CPACK_ADDCOMP_STR)    
  cpack_append_option_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_HIDDEN
    CPACK_ADDCOMP_STR)
  cpack_append_option_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_REQUIRED
    CPACK_ADDCOMP_STR)
  cpack_append_option_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_DISABLED
    CPACK_ADDCOMP_STR)
  cpack_append_option_set_command(
    CPACK_COMPONENT_${CPACK_ADDCOMP_UNAME}_DOWNLOADED
    CPACK_ADDCOMP_STR)
  FILE(APPEND "${CPACK_OUTPUT_CONFIG_FILE}" "${CPACK_ADDCOMP_STR}")
ENDMACRO(cpack_add_component)

# Macro that adds a component group to the CPack installer
MACRO(cpack_add_component_group grpname)
  STRING(TOUPPER ${grpname} CPACK_ADDGRP_UNAME)
  cpack_parse_arguments(CPACK_COMPONENT_GROUP_${CPACK_ADDGRP_UNAME}
    "DISPLAY_NAME;DESCRIPTION"
    "EXPANDED;BOLD_TITLE"
    ${ARGN}
    )

  SET(CPACK_ADDGRP_STR "\n# Configuration for component group \"${grpname}\"\n")
  cpack_append_string_variable_set_command(
    CPACK_COMPONENT_GROUP_${CPACK_ADDGRP_UNAME}_DISPLAY_NAME
    CPACK_ADDGRP_STR)
  cpack_append_string_variable_set_command(
    CPACK_COMPONENT_GROUP_${CPACK_ADDGRP_UNAME}_DESCRIPTION
    CPACK_ADDGRP_STR)
  cpack_append_option_set_command(
    CPACK_COMPONENT_GROUP_${CPACK_ADDGRP_UNAME}_EXPANDED
    CPACK_ADDGRP_STR)
  cpack_append_option_set_command(
    CPACK_COMPONENT_GROUP_${CPACK_ADDGRP_UNAME}_BOLD_TITLE
    CPACK_ADDGRP_STR)
  FILE(APPEND "${CPACK_OUTPUT_CONFIG_FILE}" "${CPACK_ADDGRP_STR}")
ENDMACRO(cpack_add_component_group)

# Macro that adds an installation type to the CPack installer
MACRO(cpack_add_install_type insttype)
  STRING(TOUPPER ${insttype} CPACK_INSTTYPE_UNAME)
  cpack_parse_arguments(CPACK_INSTALL_TYPE_${CPACK_INSTTYPE_UNAME}
    "DISPLAY_NAME"
    ""
    ${ARGN}
    )

  SET(CPACK_INSTTYPE_STR 
    "\n# Configuration for installation type \"${insttype}\"\n")
  SET(CPACK_INSTTYPE_STR 
    "${CPACK_INSTTYPE_STR}LIST(APPEND CPACK_ALL_INSTALL_TYPES ${insttype})\n")
  cpack_append_string_variable_set_command(
    CPACK_INSTALL_TYPE_${CPACK_INSTTYPE_UNAME}_DISPLAY_NAME
    CPACK_INSTTYPE_STR)
  FILE(APPEND "${CPACK_OUTPUT_CONFIG_FILE}" "${CPACK_INSTTYPE_STR}")
ENDMACRO(cpack_add_install_type)

MACRO(cpack_configure_downloads site)
  cpack_parse_arguments(CPACK_DOWNLOAD
    "UPLOAD_DIRECTORY"
    "ALL;ADD_REMOVE;NO_ADD_REMOVE"
    ${ARGN}
    )
    
  SET(CPACK_CONFIG_DL_STR
    "\n# Downloaded components configuration\n")
  SET(CPACK_UPLOAD_DIRECTORY ${CPACK_DOWNLOAD_UPLOAD_DIRECTORY})
  SET(CPACK_DOWNLOAD_SITE ${site})
  cpack_append_string_variable_set_command(
    CPACK_DOWNLOAD_SITE
    CPACK_CONFIG_DL_STR)  
  cpack_append_string_variable_set_command(
    CPACK_UPLOAD_DIRECTORY
    CPACK_CONFIG_DL_STR)
  cpack_append_option_set_command(
    CPACK_DOWNLOAD_ALL
    CPACK_CONFIG_DL_STR)
  IF (${CPACK_DOWNLOAD_ALL} AND NOT ${CPACK_DOWNLOAD_NO_ADD_REMOVE})
    SET(CPACK_DOWNLOAD_ADD_REMOVE ON)
  ENDIF (${CPACK_DOWNLOAD_ALL} AND NOT ${CPACK_DOWNLOAD_NO_ADD_REMOVE})
  SET(CPACK_ADD_REMOVE ${CPACK_DOWNLOAD_ADD_REMOVE})
  cpack_append_option_set_command(
    CPACK_ADD_REMOVE
    CPACK_CONFIG_DL_STR)
  FILE(APPEND "${CPACK_OUTPUT_CONFIG_FILE}" "${CPACK_CONFIG_DL_STR}")
ENDMACRO(cpack_configure_downloads)

# Macro for setting values if a user did not overwrite them
MACRO(cpack_set_if_not_set name value)
  IF(NOT DEFINED "${name}")
    SET(${name} "${value}")
  ENDIF(NOT DEFINED "${name}")
ENDMACRO(cpack_set_if_not_set)

# Macro to encode variables for the configuration file
# find any varable that stars with CPACK and create a variable
# _CPACK_OTHER_VARIABLES_ that contains SET commands for
# each cpack variable.  _CPACK_OTHER_VARIABLES_ is then
# used as an @ replacment in configure_file for the CPackConfig.
MACRO(cpack_encode_variables)
  SET(_CPACK_OTHER_VARIABLES_)
  GET_CMAKE_PROPERTY(res VARIABLES)
  FOREACH(var ${res})
    IF("xxx${var}" MATCHES "xxxCPACK")  
      SET(_CPACK_OTHER_VARIABLES_
        "${_CPACK_OTHER_VARIABLES_}\nSET(${var} \"${${var}}\")")
      ENDIF("xxx${var}" MATCHES "xxxCPACK")
  ENDFOREACH(var ${res})
ENDMACRO(cpack_encode_variables)

# Set the package name
cpack_set_if_not_set(CPACK_PACKAGE_NAME "${CMAKE_PROJECT_NAME}")
cpack_set_if_not_set(CPACK_PACKAGE_VERSION_MAJOR "0")
cpack_set_if_not_set(CPACK_PACKAGE_VERSION_MINOR "1")
cpack_set_if_not_set(CPACK_PACKAGE_VERSION_PATCH "1")
cpack_set_if_not_set(CPACK_PACKAGE_VERSION
  "${CPACK_PACKAGE_VERSION_MAJOR}.${CPACK_PACKAGE_VERSION_MINOR}.${CPACK_PACKAGE_VERSION_PATCH}")
cpack_set_if_not_set(CPACK_PACKAGE_VENDOR "Humanity")
cpack_set_if_not_set(CPACK_PACKAGE_DESCRIPTION_SUMMARY
  "${CMAKE_PROJECT_NAME} built using CMake")

cpack_set_if_not_set(CPACK_PACKAGE_DESCRIPTION_FILE
  "${CMAKE_ROOT}/Templates/CPack.GenericDescription.txt")
cpack_set_if_not_set(CPACK_RESOURCE_FILE_LICENSE
  "${CMAKE_ROOT}/Templates/CPack.GenericLicense.txt")
cpack_set_if_not_set(CPACK_RESOURCE_FILE_README
  "${CMAKE_ROOT}/Templates/CPack.GenericDescription.txt")
cpack_set_if_not_set(CPACK_RESOURCE_FILE_WELCOME
  "${CMAKE_ROOT}/Templates/CPack.GenericWelcome.txt")

cpack_set_if_not_set(CPACK_MODULE_PATH "${CMAKE_MODULE_PATH}")

IF(CPACK_NSIS_MODIFY_PATH)
  SET(CPACK_NSIS_MODIFY_PATH ON)
ENDIF(CPACK_NSIS_MODIFY_PATH)

SET(__cpack_system_name ${CMAKE_SYSTEM_NAME})
IF(${__cpack_system_name} MATCHES Windows)
  IF(CMAKE_CL_64)
    SET(__cpack_system_name win64)
  ELSE(CMAKE_CL_64)
    SET(__cpack_system_name win32)
  ENDIF(CMAKE_CL_64)
ENDIF(${__cpack_system_name} MATCHES Windows)
cpack_set_if_not_set(CPACK_SYSTEM_NAME "${__cpack_system_name}")

# Root dir: default value should be the string literal "$PROGRAMFILES"
# for backwards compatibility. Projects may set this value to anything.
if(CMAKE_CL_64)
set(__cpack_root_default "$PROGRAMFILES64")
else(CMAKE_CL_64)
set(__cpack_root_default "$PROGRAMFILES")
endif(CMAKE_CL_64)
cpack_set_if_not_set(CPACK_NSIS_INSTALL_ROOT "${__cpack_root_default}")

# <project>-<major>.<minor>.<patch>-<release>-<platform>.<pkgtype>
cpack_set_if_not_set(CPACK_PACKAGE_FILE_NAME
  "${CPACK_PACKAGE_NAME}-${CPACK_PACKAGE_VERSION}-${CPACK_SYSTEM_NAME}")
cpack_set_if_not_set(CPACK_PACKAGE_INSTALL_DIRECTORY
  "${CPACK_PACKAGE_NAME} ${CPACK_PACKAGE_VERSION}")
cpack_set_if_not_set(CPACK_PACKAGE_INSTALL_REGISTRY_KEY
  "${CPACK_PACKAGE_NAME} ${CPACK_PACKAGE_VERSION}")
cpack_set_if_not_set(CPACK_PACKAGE_DEFAULT_LOCATION "/")
cpack_set_if_not_set(CPACK_PACKAGE_RELOCATABLE "true")

# always force to exactly "true" or "false" for CPack.Info.plist.in:
if(CPACK_PACKAGE_RELOCATABLE)
  set(CPACK_PACKAGE_RELOCATABLE "true")
else(CPACK_PACKAGE_RELOCATABLE)
  set(CPACK_PACKAGE_RELOCATABLE "false")
endif(CPACK_PACKAGE_RELOCATABLE)

macro(cpack_check_file_exists file description)
  if(NOT EXISTS "${file}")
    message(SEND_ERROR "CPack ${description} file: \"${file}\" could not be found.")
  endif(NOT EXISTS "${file}")
endmacro(cpack_check_file_exists)

cpack_check_file_exists("${CPACK_PACKAGE_DESCRIPTION_FILE}" "package description")
cpack_check_file_exists("${CPACK_RESOURCE_FILE_LICENSE}"    "license resource")
cpack_check_file_exists("${CPACK_RESOURCE_FILE_README}"     "readme resource")
cpack_check_file_exists("${CPACK_RESOURCE_FILE_WELCOME}"    "welcome resource")

macro(cpack_optional_append _list _cond _item)
  if(${_cond})
    set(${_list} ${${_list}} ${_item})
  endif(${_cond})
endmacro(cpack_optional_append _list _cond _item)

# Provide options to choose generators
# we might check here if the required tools for the generates exist
# and set the defaults according to the results
if(NOT CPACK_GENERATOR)
  if(UNIX)
    if(CYGWIN)
      option(CPACK_BINARY_CYGWIN "Enable to build Cygwin binary packages" ON)
    else(CYGWIN)
      if(APPLE)
        option(CPACK_BINARY_BUNDLE       "Enable to build OSX bundles"      OFF)
        option(CPACK_BINARY_DRAGNDROP    "Enable to build OSX Drag And Drop package" OFF)
        option(CPACK_BINARY_PACKAGEMAKER "Enable to build PackageMaker packages" ON)
        option(CPACK_BINARY_OSXX11       "Enable to build OSX X11 packages"      OFF)
      else(APPLE)
        option(CPACK_BINARY_TZ  "Enable to build TZ packages"     ON)
      endif(APPLE)
      option(CPACK_BINARY_STGZ "Enable to build STGZ packages"    ON)
      option(CPACK_BINARY_TGZ  "Enable to build TGZ packages"     ON)
      option(CPACK_BINARY_TBZ2 "Enable to build TBZ2 packages"    OFF)
      option(CPACK_BINARY_DEB  "Enable to build Debian packages"  OFF)
      option(CPACK_BINARY_RPM  "Enable to build RPM packages"     OFF)
      option(CPACK_BINARY_NSIS "Enable to build NSIS packages"    OFF)
    endif(CYGWIN)
  else(UNIX)
    option(CPACK_BINARY_NSIS "Enable to build NSIS packages" ON)
    option(CPACK_BINARY_ZIP  "Enable to build ZIP packages" OFF)
  endif(UNIX)
  
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_BUNDLE       Bundle)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_DRAGNDROP    DragNDrop)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_PACKAGEMAKER PackageMaker)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_OSXX11       OSXX11)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_CYGWIN       CygwinBinary)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_DEB          DEB)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_RPM          RPM)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_NSIS         NSIS)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_STGZ         STGZ)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_TGZ          TGZ)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_TBZ2         TBZ2)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_TZ           TZ)
  cpack_optional_append(CPACK_GENERATOR  CPACK_BINARY_ZIP          ZIP)
  
endif(NOT CPACK_GENERATOR)

# Provide options to choose source generators
if(NOT CPACK_SOURCE_GENERATOR)
  if(UNIX)
    if(CYGWIN)
      option(CPACK_SOURCE_CYGWIN "Enable to build Cygwin source packages" ON)
    else(CYGWIN)
      option(CPACK_SOURCE_TBZ2 "Enable to build TBZ2 source packages" ON)
      option(CPACK_SOURCE_TGZ  "Enable to build TGZ source packages"  ON)
      option(CPACK_SOURCE_TZ   "Enable to build TZ source packages"   ON)
      option(CPACK_SOURCE_ZIP  "Enable to build ZIP source packages"  OFF)
    endif(CYGWIN)
  else(UNIX)
    option(CPACK_SOURCE_ZIP "Enable to build ZIP source packages" ON)
  endif(UNIX)

  cpack_optional_append(CPACK_SOURCE_GENERATOR  CPACK_SOURCE_CYGWIN  CygwinSource)
  cpack_optional_append(CPACK_SOURCE_GENERATOR  CPACK_SOURCE_TGZ     TGZ)
  cpack_optional_append(CPACK_SOURCE_GENERATOR  CPACK_SOURCE_TBZ2    TBZ2)
  cpack_optional_append(CPACK_SOURCE_GENERATOR  CPACK_SOURCE_TZ      TZ)
  cpack_optional_append(CPACK_SOURCE_GENERATOR  CPACK_SOURCE_ZIP     ZIP)
endif(NOT CPACK_SOURCE_GENERATOR)

# mark the above options as advanced
mark_as_advanced(CPACK_BINARY_CYGWIN CPACK_BINARY_PACKAGEMAKER CPACK_BINARY_OSXX11
                 CPACK_BINARY_STGZ   CPACK_BINARY_TGZ          CPACK_BINARY_TBZ2 
                 CPACK_BINARY_DEB    CPACK_BINARY_RPM          CPACK_BINARY_TZ     
                 CPACK_BINARY_NSIS CPACK_BINARY_ZIP CPACK_BINARY_BUNDLE
                 CPACK_SOURCE_CYGWIN CPACK_SOURCE_TBZ2 CPACK_SOURCE_TGZ 
                 CPACK_SOURCE_TZ CPACK_SOURCE_ZIP CPACK_BINARY_DRAGNDROP)

# Set some other variables
cpack_set_if_not_set(CPACK_INSTALL_CMAKE_PROJECTS
  "${CMAKE_BINARY_DIR};${CMAKE_PROJECT_NAME};ALL;/")
cpack_set_if_not_set(CPACK_CMAKE_GENERATOR "${CMAKE_GENERATOR}")
cpack_set_if_not_set(CPACK_TOPLEVEL_TAG "${CPACK_SYSTEM_NAME}")
# if the user has set CPACK_NSIS_DISPLAY_NAME remember it
if(DEFINED CPACK_NSIS_DISPLAY_NAME)
  SET(CPACK_NSIS_DISPLAY_NAME_SET TRUE)
endif()
# if the user has set CPACK_NSIS_DISPLAY
# explicitly, then use that as the default
# value of CPACK_NSIS_PACKAGE_NAME  instead
# of CPACK_PACKAGE_INSTALL_DIRECTORY 
cpack_set_if_not_set(CPACK_NSIS_DISPLAY_NAME "${CPACK_PACKAGE_INSTALL_DIRECTORY}")

if(CPACK_NSIS_DISPLAY_NAME_SET)
  string(REPLACE "\\" "\\\\" 
    _NSIS_DISPLAY_NAME_TMP  "${CPACK_NSIS_DISPLAY_NAME}")
  cpack_set_if_not_set(CPACK_NSIS_PACKAGE_NAME "${_NSIS_DISPLAY_NAME_TMP}")
else()
  cpack_set_if_not_set(CPACK_NSIS_PACKAGE_NAME "${CPACK_PACKAGE_INSTALL_DIRECTORY}")
endif()

cpack_set_if_not_set(CPACK_OUTPUT_CONFIG_FILE
  "${CMAKE_BINARY_DIR}/CPackConfig.cmake")

cpack_set_if_not_set(CPACK_SOURCE_OUTPUT_CONFIG_FILE
  "${CMAKE_BINARY_DIR}/CPackSourceConfig.cmake")

cpack_set_if_not_set(CPACK_SET_DESTDIR OFF)
cpack_set_if_not_set(CPACK_INSTALL_PREFIX "${CMAKE_INSTALL_PREFIX}")

cpack_set_if_not_set(CPACK_NSIS_INSTALLER_ICON_CODE "")
cpack_set_if_not_set(CPACK_NSIS_INSTALLER_MUI_ICON_CODE "")

IF(DEFINED CPACK_COMPONENTS_ALL)
  IF(CPACK_MONOLITHIC_INSTALL)
    MESSAGE("CPack warning: both CPACK_COMPONENTS_ALL and CPACK_MONOLITHIC_INSTALL have been set.\nDefaulting to a monolithic installation.")
    SET(CPACK_COMPONENTS_ALL)
  ELSE(CPACK_MONOLITHIC_INSTALL)
    # The user has provided the set of components to be installed as
    # part of a component-based installation; trust her.
    SET(CPACK_COMPONENTS_ALL_SET_BY_USER TRUE)
  ENDIF(CPACK_MONOLITHIC_INSTALL)
ELSE(DEFINED CPACK_COMPONENTS_ALL)
  # If the user has not specifically requested a monolithic installer
  # but has specified components in various "install" commands, tell
  # CPack about those components.
  IF(NOT CPACK_MONOLITHIC_INSTALL)
    GET_CMAKE_PROPERTY(CPACK_COMPONENTS_ALL COMPONENTS)
    LIST(LENGTH CPACK_COMPONENTS_ALL CPACK_COMPONENTS_LEN)
    IF(CPACK_COMPONENTS_LEN EQUAL 1)
      # Only one component: this is not a component-based installation
      # (at least, it isn't a component-based installation, but may
      # become one later if the user uses the cpack_add_* commands).
      SET(CPACK_COMPONENTS_ALL)
    ENDIF(CPACK_COMPONENTS_LEN EQUAL 1)
    SET(CPACK_COMPONENTS_LEN)
  ENDIF(NOT CPACK_MONOLITHIC_INSTALL)
ENDIF(DEFINED CPACK_COMPONENTS_ALL)

# CMake always generates a component named "Unspecified", which is
# used to install everything that doesn't have an explicitly-provided
# component. Since these files should always be installed, we'll make
# them hidden and required.
set(CPACK_COMPONENT_UNSPECIFIED_HIDDEN TRUE)
set(CPACK_COMPONENT_UNSPECIFIED_REQUIRED TRUE)

cpack_encode_variables()
configure_file("${cpack_input_file}" "${CPACK_OUTPUT_CONFIG_FILE}" @ONLY IMMEDIATE)

# Generate source file
cpack_set_if_not_set(CPACK_SOURCE_INSTALLED_DIRECTORIES
  "${CMAKE_SOURCE_DIR};/")
cpack_set_if_not_set(CPACK_SOURCE_TOPLEVEL_TAG "${CPACK_SYSTEM_NAME}-Source")
cpack_set_if_not_set(CPACK_SOURCE_PACKAGE_FILE_NAME
  "${CPACK_PACKAGE_NAME}-${CPACK_PACKAGE_VERSION}-Source")
cpack_set_if_not_set(CPACK_SOURCE_IGNORE_FILES
  "/CVS/;/\\\\\\\\.svn/;/\\\\\\\\.bzr/;/\\\\\\\\.hg/;/\\\\\\\\.git/;\\\\\\\\.swp$;\\\\\\\\.#;/#")
SET(CPACK_INSTALL_CMAKE_PROJECTS "${CPACK_SOURCE_INSTALL_CMAKE_PROJECTS}")
SET(CPACK_INSTALLED_DIRECTORIES "${CPACK_SOURCE_INSTALLED_DIRECTORIES}")
SET(CPACK_GENERATOR "${CPACK_SOURCE_GENERATOR}")
SET(CPACK_TOPLEVEL_TAG "${CPACK_SOURCE_TOPLEVEL_TAG}")
SET(CPACK_PACKAGE_FILE_NAME "${CPACK_SOURCE_PACKAGE_FILE_NAME}")
SET(CPACK_IGNORE_FILES "${CPACK_SOURCE_IGNORE_FILES}")
SET(CPACK_STRIP_FILES "${CPACK_SOURCE_STRIP_FILES}")

cpack_encode_variables()
configure_file("${cpack_source_input_file}"
  "${CPACK_SOURCE_OUTPUT_CONFIG_FILE}" @ONLY IMMEDIATE)
