"""Reads /etc/init/juju-machine-agent.conf to determine Juju agent config"""

from txzookeeper.client import ZookeeperClient

import juju
from juju.environment.config import EnvironmentsConfig


def get_upstart_env(conf_path="/etc/init/juju-machine-agent.conf"):
    # Find environment variables as follows
    # env JUJU_MACHINE_ID="2"
    # env JUJU_ZOOKEEPER="ip-10-171-74-191.us-west-1.compute.internal:2181"

    env = {}
    with open(conf_path) as conf:
        for line in conf:
            if line.startswith("env "):
                kv = line[4:]
                k, v = kv.split("=")
                env[k] = v[1:-2]  # remove quotes
    return env


def get_zk_client_connector(options):
    # Running on a Juju machine or on an admin system? Distinguish by
    # first checking for an ~/.juju/environments.yaml file; if not
    # check the machine for an upstart conf file
    env_config = EnvironmentsConfig()
    try:
        env_config.load()
    except juju.errors.FileNotFound:
        return get_juju_machine_zk_client_connector()

    if options.environment is None:
        environment = env_config.get_default()
    else:
        environment = env_config.get(options.environment)
    if environment is None:
        raise juju.errors.EnvironmentNotFound(
            "Environment %r not configured in environments.yaml" % options.environment)
    provider = environment.get_machine_provider()

    def connector():
        # Note: returns a Deferred
        return provider.connect()

    return connector


def get_juju_machine_zk_client_connector():
    env = get_upstart_env()

    # When principals for ACL support is added, this is the place to do it.
    def connector():
        return ZookeeperClient().connect(env["JUJU_ZOOKEEPER"])

    return connector
