#ifndef _HEAP_H // -*-C++-*-
#define _HEAP_H

/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

using namespace std;

#ifndef MEM_POOLSIZE
#define MEM_POOLSIZE 1
#endif

#ifndef WIN32
#define SWPRINTF swprintf
#else
#define SWPRINTF snwprintf
#endif
#ifndef NOCHECK
#define GC_DEBUG
#endif
#ifndef WIN32
#include <pthread.h>
#define GC_THREADS
#else
#define GC_WIN32_THREADS
#endif
#include <gc/gc_cpp.h>

//#include <gc/gc_allocator.h>
#include <vector>
#include <cstdlib>
#include <string.h>
#include "sizes.h"
#include "ValueFuns.h"

class VMState;

typedef void(*func)(VMState*);

class Value;
class String;
class Array;
class Exception;
class Union;
class Real;
class Closure;

class ValuePool;
class StringPool;
class UnionPool;
class RealPool;
class ArrayPool;

#define KMALLOC(x) GC_MALLOC(x)
#define KMALLOC_BLOCK(x) GC_MALLOC_ATOMIC(x)
#define KMALLOC_ROOT(x) GC_MALLOC_UNCOLLECTABLE(x)
#define KREALLOC(x) GC_REALLOC(x)

#define NEWVAL (new Value(NULL,KVT_INT))
//(Value*)(KMALLOC(sizeof(Value))))
#define MKVAL(ptr,ft) new Value(ptr,ft)
#define MKINITVAL(ptr,ft) new(GCInit) Value(ptr,ft)
#define MKPVAL(place,ptr,ft) new(place) Value(ptr,ft)
//NEWVAL->setPtr(ptr,ft)

#define NEWREAL(n) new Real(n)
#define NEWPREAL(p,n) new(p) Real(n)
//(((Real*)(KMALLOC(sizeof(Real))))->setNum(n))


// If we're allocating globals, we'd better do a GC_init first. Use this
// as a placement to the 'new' operator on Values.
enum DoInit { GCInit, NoGCInit };


class ValuePool {
public:
  void *operator new(size_t num_bytes) {
    return (KMALLOC_ROOT(num_bytes));
  }

  ValuePool();
  void* next();
private:
  Value* m_pool;
  int m_usage;
  void refill();
};

class StringPool {
public:
  void *operator new(size_t num_bytes) {
    return (KMALLOC_ROOT(num_bytes));
  }

  StringPool();
  void* next();
private:
  String* m_pool;
  int m_usage;
  void refill();
};

class UnionPool {
public:
  void *operator new(size_t num_bytes) {
    return (KMALLOC_ROOT(num_bytes));
  }

  UnionPool();
  void* next();
private:
  Union* m_pool;
  int m_usage;
  void refill();
};

class RealPool {
public:
  void *operator new(size_t num_bytes) {
    return (KMALLOC_ROOT(num_bytes));
  }

  RealPool();
  void* next();
private:
  Real* m_pool;
  int m_usage;
  void refill();
};

class ArrayPool {
public:
  void *operator new(size_t num_bytes) {
    return (KMALLOC_ROOT(num_bytes));
  }

  ArrayPool();
  void* next();
private:
  Array* m_pool;
  int m_usage;
  void refill();
};


// I really shouldn't call this Real! Float/Double better.
class Real
{
public:
    void *operator new(size_t num_bytes) {
	return (KMALLOC(num_bytes));
    }
    void *operator new(size_t num_bytes, void* place) {
      return place;
    }
  // create in preallocated memory
    void *operator new(size_t num_bytes, RealPool* allocator) {
      return allocator->next();
    }


    Real(double n):number(n) {}
//    Real* setNum(double n) { number = n; return this; }
    double number;
};

// cached values
extern Value *zero, *one, *minusone;

extern "C" {
  wchar_t* emptyString();
}


//#define U_TAG(x) ((x->tag_arity) >> 16)
#define U_ARITY(x) x->arity
//#define THIS_TAG (tag_arity >> 16)
#define THIS_ARITY arity

class Union : public gc_cleanup
{
public:
    // If getargs is true, get arguments from the stack in vm.
  // create in preallocated memory
    void *operator new(size_t num_bytes, UnionPool* allocator) {
      return allocator->next();
    }
    void *operator new(size_t num_bytes) {
      return (void*)GC_MALLOC(num_bytes);
    }

    Union(VMState* vm, ukint arity, bool getargs = true);
    ~Union() { if (args != NULL) { GC_FREE(args); } }

    bool eq(Union* x, map<Value*,Value*>& done);
    kint cmp(Union* x);
    ukint memusage();
    Union* copy(VMState* vm, map<Value*,Value*>& done);

    ukint arity; 
    Value** args;
};


#ifdef WIN32
/* No, still doesn't work. Error is:
d:/MINgw/include/c++/3.2.3/bits/stl_construct.h: In function `void 
   std::_Construct(_T1*, const _T2&) [with _T1 = CallStackEntry, _T2 = 
   CallStackEntry]':
Heap.cc:494:   instantiated from `void std::vector<_Tp, _Alloc>::push_back(const _Tp&) [with _Tp = CallStackEntry, _Alloc = std::allocator<CallStackEntry>]'
Heap.cc:455:   instantiated from here
d:/MINgw/include/c++/3.2.3/bits/stl_construct.h:78: no matching function for 
   call to `CallStackEntry::operator new(unsigned int, void*&)'
Heap.h:104: candidates are: static void* CallStackEntry::operator new(unsigned 
   int)
*/
struct CallStackEntry : public gc {
# else
class CallStackEntry {
public:
    void *operator new(size_t num_bytes) {
	return (KMALLOC(num_bytes));
    }
#endif
    const wchar_t* fn_name;
    const wchar_t* file;
    int line;
    const wchar_t* call_file;
    int call_line;
};

class Exception
{
public:
    void *operator new(size_t num_bytes) {
	return (KMALLOC(num_bytes));
    }
    void *operator new(size_t num_bytes, void* place) {
      return place;
    }

    Exception(VMState* vm);
    Exception(VMState* vm, const char* tag, kint arity);
    void show();
    bool eq(Exception* x);
    kint cmp(Exception* x);

    void dumpBacktrace();
    
    String* err;
    kint code;

    const char* tag;
    kint arity;
    Value** args;

    wchar_t* throwfile;
    kint throwline;

    // Need a copy of entire backtrace. Ick.
    vector<CallStackEntry> backtrace;
};

class Value
{
public:
    /// Create an initialised value.
    Value(void* ptr, valtype ft):m_val(ptr),m_funtable(ft) {}

    void *operator new(size_t num_bytes) {
	return (KMALLOC(num_bytes));
    }
  // create in preallocated memory
    void *operator new(size_t num_bytes, void* allocptr) {
      return allocptr;
    }
  // create in preallocated memory
    void *operator new(size_t num_bytes, ValuePool* allocator) {
      return allocator->next();
    }
 // Create a global, better do initialisation first
    void *operator new(size_t num_bytes, DoInit init) {
	if (init==GCInit) {
	    GC_init();
	}
	return KMALLOC(num_bytes);
    }


    valtype getType() { return m_funtable; }

  // returns a new value pointing to the same thing
    Value* clone() {
	return MKVAL(m_val,m_funtable);
    }
    Value* cloneTo(ValuePool* valpool) {
	return MKPVAL(valpool,m_val,m_funtable);
    }

    /// Get an integer out. Assumes you've checked/know.
    kint getInt() { return (kint)m_val; };
    /// Get a real number out
    double getReal() { 
// FIXME: I don't know if turning this check off is 'safe'. It doesn't
// throw an exception after all. Maybe it should?
#ifndef NOCHECK
	if (m_funtable==KVT_REAL)
	    return ((Real*)m_val)->number;
	else
	    return 0.0; // Not too much of an overhead here.
#else
	    return ((Real*)m_val)->number;
#endif	
    }

    /// Get the function pointer. Assumes you've checked/know.
    Closure* getFunc() { return (Closure*)m_val; }

    /// Get the string
    String* getString();
    /// Get the array.
    Array* getArray();
    /// Get the union.
    Union* getUnion();
    /// Get the exception.
    Exception* getExcept();

    /// Run the closure stored here.
    void runClosure(VMState *vm);

    /// Add more arguments from the stack to the closure stored here.
    Value* addClosureArgs(VMState *vm, kint i, bool lam);

    /// Coercions
    void int2str();
    void int2str(StringPool*);
    void str2int();
    void str2real();
    void str2real(RealPool*);
    void chr2str();
    void chr2str(StringPool*);
    void bool2str();
    void bool2str(StringPool*);
    void str2chr();
    void int2real();
    void int2real(RealPool*);
    void real2int();
    void real2str();
    void real2str(StringPool*);

    /// Get the tag out of a union
    kint getTag() { return m_funtable-KVT_UNION; }


    /// Get the raw pointer out.
    void* getRaw() { return m_val; }


    /// Update to be an integer.
    void setInt(kint i)
    {
	m_val=(void*)i;
	m_funtable = KVT_INT;
    }

    /// Update to be an integer and return a pointer
    Value* setIntReturn(kint i) {
	setInt(i);
	return this;
    }

    /// Update to be a real
    void setReal(double i);
    void setReal(RealPool* r, double v);

    /// Update to be a function.
    void setFunc(Closure* f);

    /// Update to be ana rray.
    void setString(String* str)
    {
	m_val=(void*)str;
	m_funtable = KVT_STRING;
    }

    void setArray(Array* a);

    /// Set a raw pointer.
    void setPtr(Value* p) {
//    cout << "Making " << this << " to " << p->m_val << endl;
//	memcpy(this,p,sizeof(Value*));
	m_val=p->m_val;
	m_funtable=p->m_funtable;
    }

    Value* setPtr(void* raw, valtype ft) {
	m_val = raw;
	m_funtable = ft;
	return this;
    }

    void addInt(kint x) {
        m_val = (void*)((kint)m_val+x);
    }

    void addVal(Value* v) {
	m_val=(void*)((kint)m_val+(kint)(v->m_val));
    }

    void subVal(Value* v) {
	m_val=(void*)((kint)m_val-(kint)(v->m_val));
    }

    void mulVal(Value* v) {
	m_val=(void*)((kint)m_val*(kint)(v->m_val));
    }

    void divVal(Value* v) {
	m_val=(void*)((kint)m_val/(kint)(v->m_val));
    }

    /// Set a value from the projection of the top stack item
    void project(VMState* vm, kint i);
    void fastproject(Value* topitem, kint i) {
	Union* top=(Union*)(topitem->m_val);
	Value* arg=top->args[i];
	m_val=arg->m_val;
	m_funtable = arg->m_funtable;
    }
    void exceptionproject(Value* topitem, kint i) {
	Exception* top=(Exception*)(topitem->m_val);
	Value* arg=top->args[i];
	m_val=arg->m_val;
	m_funtable = arg->m_funtable;
    }

    void fastproject(VMState* vm, kint i);

    /// Lookup a value in an array --- resize/create if necessary.
    Value* lookup(VMState* vm, kint i);
    /// Get the length of an array
    kint length();

    /// Get the function table out
    valtype getFunTable() { return m_funtable; }
    
private:
    void* m_val;
    valtype m_funtable;
};

// Get the actual string, considering offset
#define M_STR (m_str+m_offset)
// Get the actual length available, considering offset
#define M_ALLOC (m_alloc-m_offset)
// Prepared to waste 8k before reallocating a string (this is quite a lot...)
#define OFFSET_LIMIT 8192

#define NEWSTRING(x) new String(x)
#define NEWPSTRING(p,x) new(p) String(x)
//(((String*)(KMALLOC(sizeof(String))))->setVal(x))

class String
{
public:
    String(const wchar_t* val);
    String(wchar_t val);
    String(kint len);
    String(VMState* vm);

    void *operator new(size_t num_bytes) {
	return (KMALLOC(num_bytes));
    }
    void *operator new(size_t num_bytes, void* place) {
      return place;
    }
  // create in preallocated memory
    void *operator new(size_t num_bytes, StringPool* allocator) {
      return allocator->next();
    }

    String* setVal(const wchar_t* val);
    String* setVal(wchar_t val);
    String* setVal(kint len);

    wchar_t* getVal() { return M_STR; }

    kint length() {
        return m_len; 
    }
    wchar_t getIndex(kint i) { return M_STR[i]; }
    void setIndex(kint i,wchar_t c) { M_STR[i]=c; }
    void append(String* s);
    void append(wchar_t c);
    void append(const wchar_t* s);
    void getForeign(const char* c);
    bool eq(String* s);
    bool eq(const wchar_t* s);
    kint cmp(String* s);
    // Increase the offset. If it goes over OFFSET_LIMIT, reallocate
    void offset(kint x);
    // Chop last x characters off.
    void chop(kint x);

  // debug/marshalling purposes
    kint space() { return M_ALLOC; }

private:
    wchar_t* m_str;
    kint m_alloc;
    kint m_offset; // How much of the start of the string to ignore
                  // (Optimisation if we're just chopping off the start of
                  //  the string)
    kint m_len;
};






#endif
