/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_localaccount.mod voms local account plugin

    \section vomslocalaccountsyn SYNOPSIS
    \b lcmaps_voms_localaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]
        [-use_voms_gid]

    \section vomslocalaccountdesc DESCRIPTION

    This localaccount acquisition plugin is a 'VOMS-aware' modification of the 'localaccount' plugin.
    The plugin tries to find a local account (more specifically a UID) based on the VOMS information
    that has been retrieved by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod"
    from the user's grid credential.
    It will try to match a VO-GROUP-ROLE combination from the user's grid credential with an entry
    in a gridmapfile (most likely the traditional gridmapfile, used by the localaccount and
    poolaccount plugins)
    In this file VO-GROUP-ROLE combinations are listed next to the name of an existing account,
    as shown in the following example.

    EXAMPLE:

    \c "/VO=wilma/GROUP=management" \c wilmamgr

    \c "/VO=fred/GROUP=*"  \c fredmgr

    If the first matching VO-GROUP-ROLE combination is \c "/VO=wilma/GROUP=management" the plugin
    will map the user to the account wilmamgr.
    If the first matching VO-GROUP-ROLE combination is
    \c "/VO=fred/GROUP=*" the plugin will map the user to the account fredmgr.

    \section vomslocalaccountnote1 NOTE 1
        This plugin should only be used in combination with the \e 'voms_localgroup'
        and/or \e 'voms_poolgroup' plugins.

    \section vomslocalaccountoptions OPTIONS
    \subsection vomslocalaccountoptie1 -GRIDMAPFILE \<gridmapfile\>
        See \ref vomslocalaccountoptie4 "-gridmap"
 
    \subsection vomslocalaccountoptie2 -gridmapfile \<gridmapfile\>
        See \ref vomslocalaccountoptie4 "-gridmap"
 
    \subsection vomslocalaccountoptie3 -GRIDMAP \<gridmapfile\>
        See \ref vomslocalaccountoptie4 "-gridmap"
 
    \subsection vomslocalaccountoptie4 -gridmap \<gridmapfile\>
        When this option is set it will override the default path to the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

    \subsection vomslocalaccountoptie5 -use_voms_gid
        When this option is set the LCMAPS system relies on other VOMS plugins such as
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod" and \ref lcmaps_voms_poolgroup.mod
        "lcmaps_voms_poolgroup.mod" to assign a primary GID based on the VOMS attributes contained in the 
        user proxy instead of taking the default primary GID that comes with the local account.

\section vomslocalaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure
 
 
\section vomslocalaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
\section vomslocalaccountSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_localaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_localaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/* Define both _XOPEN_SOURCE for strdup and _GNU_SOURCE, this makes Solaris
 * happier */
#define _XOPEN_SOURCE	500
/* Try to provide RTLD_DEFAULT */
/* MacOS needs no POSIX or _DARWIN_C_SOURCE to have RTLD_DEFAULT */
#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>

#include "lcmaps_voms_config.h"

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include "lcmaps_gridmapfile.h"


/************************************************************************
 * defines
 ************************************************************************/

#define PLUGIN_PREFIX	"lcmaps_voms_localaccount"

#define PLUGIN_RUN	0   /* full run mode */
#define PLUGIN_VERIFY	1   /* verify-only mode */


/************************************************************************
 * global variables
 ************************************************************************/

static char *gridmapfile = NULL; /* filename of gridmapfile */
static int pgid_mapping = 1;	/* default: map pGID from account */
static int sgid_mapping = 1;	/* default: map sGIDs from account */


/************************************************************************
 * private prototypes
 ************************************************************************/

/* called by plugin_run() and plugin_verify() */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode);


/************************************************************************
 * public functions
 ************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(int argc, char **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_initialize()";
    int i;
    int  use_voms_gid                           = 0; /* use GIDs from VOMS */
    int  use_account_gid                        = 0; /* use GIDs from acct */
    int  do_not_map_primary_gid                 = 0; /* pGID not from acct */
    int  add_primary_gid_from_mapped_account    = 0; /* pGID from acct */
    int  add_primary_gid_as_secondary_gid_from_mapped_account = 0; /* pGID from acct as sGID */
    int  do_not_add_secondary_gids_from_mapped_account = 0; /* sGIDs not from acct */
    int  add_secondary_gids_from_mapped_account = 0; /* sGIDs from acct */

    /* Log commandline parameters on debug */
    lcmaps_log(LOG_DEBUG,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
	lcmaps_log(LOG_DEBUG,"%s: arg %d is %s\n", logstr, i, argv[i]);

    /* Parse arguments, argv[0] = name of plugin, so start with i = 1 */
    for (i = 1; i < argc; i++) {
	/* check grid-mapfile option (or similar variations) */
        if ( strcmp(argv[i], "-gridmapfile") == 0 ||
             strcmp(argv[i], "-GRIDMAPFILE") == 0 ||
             strcmp(argv[i], "-gridmap") == 0 ||
             strcmp(argv[i], "-GRIDMAP") == 0 )
	{
	    /* check valid filename argument */
            if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by valid filename\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(gridmapfile); gridmapfile=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (gridmapfile = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &gridmapfile)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using grid-mapfile \"%s\".\n",
		    logstr, gridmapfile);
            i++;
        }
        else if (strcmp(argv[i], "--do-not-add-primary-gid-from-mapped-account") == 0)
        {
	    /* check consistency with other pGID flags */
	    if (add_primary_gid_from_mapped_account ||
		add_primary_gid_as_secondary_gid_from_mapped_account)
	    {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--add-primary-gid-from-mapped-account or "
		    "--add-primary-gid-as-secondary-gid-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            do_not_map_primary_gid = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-from-mapped-account") == 0)
        {
	    /* check consistency with other pGID flags */
	    if (do_not_map_primary_gid) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-add-primary-gid-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            add_primary_gid_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-as-secondary-gid-from-mapped-account") == 0)
        {
	    /* check consistency with other pGID flags */
	    if (do_not_map_primary_gid) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-add-primary-gid-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            add_primary_gid_as_secondary_gid_from_mapped_account = 1;
	}
        else if (strcmp(argv[i], "--do-not-add-secondary-gids-from-mapped-account") == 0)
        {
	    /* check consistency with other sGID flag */
	    if (add_secondary_gids_from_mapped_account) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--add-secondary-gids-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            do_not_add_secondary_gids_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-secondary-gids-from-mapped-account") == 0)
        {
	    /* check consistency with other sGID flag */
	    if (do_not_add_secondary_gids_from_mapped_account) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-add-secondary-gids-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            add_secondary_gids_from_mapped_account = 1;
        }
        else if ((strcmp(argv[i], "--use-voms-gid") == 0) ||
                 (strcmp(argv[i], "--use_voms_gid") == 0) ||
                 (strcmp(argv[i], "-use_voms_gid") == 0))
        {
	    /* check consistency with other voms/account defaults flag */
	    if (use_account_gid)    {
		lcmaps_log(LOG_ERR,
		    "%s: cannot specify both %s and --use-account-gid\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            use_voms_gid = 1;
        }
        else if (strcmp(argv[i], "--use-account-gid") == 0) {
	    /* check consistency with other voms/account defaults flag */
	    if (use_voms_gid)    {
		lcmaps_log(LOG_ERR,
		    "%s: cannot specify both %s and --use-voms-gid\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            use_account_gid = 1;
        }
	/* any other argument is an error */
	else
	{
            lcmaps_log(LOG_ERR,
		    "%s: Unknown argument for plugin: %s (failure)\n",
		    logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Check we have a grid-mapfile */
    if (gridmapfile==NULL || gridmapfile[0]=='\0')  {
	free(gridmapfile); gridmapfile=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No grid-mapfile was provided, will use default.\n", logstr);
    }

    /* Set the correct pGID and sGID mapping flags, first defaults */
    if (use_voms_gid)
	/* no implicit mapping from account GIDs */
	pgid_mapping=sgid_mapping=0;
    else if (use_account_gid)
	/* implicit mapping from account GIDs */
	pgid_mapping=sgid_mapping=1;

    /* handle pGID options */
    if (do_not_map_primary_gid)
	pgid_mapping=0;
    else {
	if (add_primary_gid_from_mapped_account)
	    /* add (not replace) pGID->pGID flag */
	    pgid_mapping|=1;
	if (add_primary_gid_as_secondary_gid_from_mapped_account)
	    /* add (not replace) pGID->sGID flag */
	    pgid_mapping|=2;
    }

    /* handle sGID options */
    if (do_not_add_secondary_gids_from_mapped_account)
	sgid_mapping=0;
    else if (add_secondary_gids_from_mapped_account)
	sgid_mapping=1;

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    free(gridmapfile);
    gridmapfile = NULL;

    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(int *argc, lcmaps_argument_t **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
	{"fqan_list"    ,"char **", 0,NULL},
	{"nfqan"        ,"int"    , 0,NULL},
	{"requested_uid","uid_t"  , 0,NULL},
	{NULL           ,NULL     ,-1,NULL},
	{NULL           ,NULL     ,-1,NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log(LOG_DEBUG,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log(LOG_DEBUG,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[3].argName="requested_username";
	argList[3].argType="char *";
	argList[3].argInOut=1;
	argList[3].value=NULL;
    }

    lcmaps_log(LOG_DEBUG,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log(LOG_DEBUG,"%s: address first argument: %p\n",
	    logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void) {
    const char * logstr = PLUGIN_PREFIX"-plugin_terminate()";

    lcmaps_log(LOG_DEBUG,"%s: terminating\n", logstr);

    free(gridmapfile);
    gridmapfile=NULL;

    return LCMAPS_MOD_SUCCESS;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Actual run/verify function. Called by both plugin_run and plugin_verify
 * with different lcmaps_mode.
 */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode) {
    const char *        logstr       = NULL;
    const char *        logmapfile   = gridmapfile ? gridmapfile
						   : "default grid-mapfile";
    void *              value        = NULL;
    int                 nfqan        = -1;
    char **             fqan_list    = NULL;
    int                 i            = 0;
    char *              req_username = NULL;
    int                 req_username_needs_free=0;
    uid_t               req_uid      = (uid_t)(-1);
    struct passwd *     pw           = NULL;
    unsigned short      options      = 0;
    const char *        searchstr    = NULL;
    int                 rc           = 0;
    char *              username     = NULL;
    int                 found_mapping= 0;
    struct passwd       *user_info   = NULL;
    int                 cnt_sec_gid  = 0;
    gid_t *             sec_gid      = NULL;

    /* Set suitable logstr */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = PLUGIN_PREFIX"-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = PLUGIN_PREFIX"-plugin_verify()";
    else {
        lcmaps_log(LOG_ERR, PLUGIN_PREFIX"-plugin_run_or_verify(): "
		"attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_plugin;
    }

    /* Try to get FQANs from LCMAPS values: */
    /* First try to obtain FQANs from the credential data (i.e. stored by other
     * plugins */
    fqan_list = getCredentialData(LCMAPS_VO_CRED_STRING, &nfqan);
    if (nfqan>0) {
	lcmaps_log(LOG_DEBUG, "%s: found %d FQAN(s) in credential data\n",
		logstr, nfqan);
    } else {
	/* No FQANs registered, use the introspect/run arguments */
	lcmaps_log(LOG_DEBUG,
	    "%s: no FQANs registered by other plugins, trying run/introspect args\n",
	    logstr);
	if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) ) {
	    /* get number of FQANs */
	    if ( (nfqan = *(int *) value) < 1 ) {
		lcmaps_log(LOG_INFO,
			"%s: no (valid) VOMS groups found --> no mapping\n",
			logstr);
		goto fail_plugin;
	    }
	    /* Log number of FQANs */
	    lcmaps_log(LOG_DEBUG,
		    "%s: the list of FQANs should contain %d elements\n",
		    logstr, nfqan);
	    /* get FQAN list */
	    value = lcmaps_getArgValue("fqan_list", "char **", argc, argv);
	    if ( value==NULL || (fqan_list = *(char ***) value)==NULL )   {
		lcmaps_log(LOG_WARNING,
			"%s: could not retrieve list of %d FQANs!\n",
			logstr, nfqan);
		goto fail_plugin;
	    }
	}
    }

    /* Log the found FQANs */
    for (i = 0; i < nfqan; i++)
	lcmaps_log(LOG_DEBUG, "%s: FQAN %d: %s\n", logstr, i+1, fqan_list[i]);

    /* In verify mode, requested account is typically in requested_uid, in run
     * mode, it can be provided as requested_username. Let's just see if either
     * is set. */

    /* Get requested_username value */
    value = lcmaps_getArgValue("requested_username", "char *", argc, argv);
    if ( value != NULL && *(char **)value !=NULL ) {
	/* copy and log resulting account name */
	req_username=*(char **)value;
	lcmaps_log(LOG_DEBUG,"%s: the requested user is %s\n",
		logstr, req_username);
    } else { /* No (valid) requested_username, try requested_uid */
	/* Get requested_uid value */
	value = lcmaps_getArgValue("requested_uid", "uid_t", argc, argv);
	if ( value!=NULL && *(int *)value != -1)  { /* undefined value -> -1 */
	    req_uid = *(uid_t *)value;
	    /* Basic sanity check */
	    if (req_uid == 0) {
		lcmaps_log(LOG_ERR,
			"%s: illegal request for uid == 0 (failure)\n", logstr);
		goto fail_plugin;
	    }
	    /* Get passwd info */
	    if ( (pw = getpwuid(req_uid )) == NULL ||
		 pw->pw_name == NULL )
	    {
		lcmaps_log(LOG_WARNING,
			"%s: the requested uid %d is illegal.\n",
			logstr, (int) req_uid);
		goto fail_plugin;
	    }
	    /* copy resulting account name */
	    if ( (req_username=strdup(pw->pw_name))==NULL ) {
		lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
		goto fail_plugin;
	    }
	    /* keep track whether it needs free */
	    req_username_needs_free=1;
	    /* Log the result */
	    lcmaps_log(LOG_DEBUG,"%s: the requested user is %s(%d)\n",
		    logstr, req_username, (int)req_uid);
	} else if (lcmaps_mode == PLUGIN_VERIFY)    {
	    lcmaps_log(LOG_WARNING,
		    "%s: both requested_username and requested_uid are unset, "
		    "need at least one in PLUGIN_VERIFY mode\n", logstr);
	    goto fail_plugin;
	}
    }

    /* Set the matching options */
    if (req_username)	{
	options = MATCH_EXACT|MATCH_WILD_CHARS|REQUIRE_MAPFILE;
	searchstr=req_username;
    } else {
	options = MATCH_WILD_CHARS|REQUIRE_MAPFILE;
	searchstr=".";
    }

    /* Try to match the VO strings with the grid-mapfile, normally the first
     * available VO string should match */
    for (i = 0; i < nfqan; i++) {
        rc = lcmaps_gridmapfile(gridmapfile, fqan_list[i], searchstr, options,
				&username);

	/* parse return value of lcmaps_gridmapfile */
	if (rc==-1)
	    /* error */
	    goto fail_plugin;
	if (rc==0)  {
	    /* default also returns rc==0, but we don't use: REQUIRE_MAPFILE */
	    /* no match: log and try next FQAN */
	    lcmaps_log(LOG_DEBUG,
		"%s: No match for %s localaccount for FQAN %d, \"%s\"\n",
		logstr,
		(req_username ? "the requested" : "a VOMS"), i+1, fqan_list[i]);
	    continue; /* next FQAN */
	}

	/* match */
	found_mapping = 1;
	break;  /* out of FQAN loop */
    }

    /* Log if when we haven't found a match */
    if (found_mapping == 0) {
	/* no match, this should be at most a NOTICE */
	if (req_username)
	    lcmaps_log(LOG_NOTICE,
		"%s: No match for requested localaccount %s for any of the "
		"FQANs, in %s\n", logstr, req_username, logmapfile);
	else
	    lcmaps_log(LOG_NOTICE,
		"%s: No match for a VOMS localaccount for any of the "
		"FQANs, in %s\n", logstr, logmapfile);
	goto fail_plugin;
    }

    /* found match: log */
    lcmaps_log(LOG_DEBUG,"%s: found %susername %s for FQAN %d, \"%s\"\n",
	logstr, req_username ? "requested " : "", username, i+1, fqan_list[i]);

    /* Convert username to uid, pgid and sgids */

    /* Get account info for found username */
    if ( (user_info = getpwnam(username)) == NULL ) {
	lcmaps_log(LOG_WARNING,
		"%s: no user account found with the name \"%s\"\n",
		logstr, username);
	goto fail_plugin;
    }

    /* Log resulting account */
    lcmaps_log(LOG_DEBUG,"%s: username : %s\n", logstr, user_info->pw_name);
    lcmaps_log(LOG_DEBUG,"%s: user_id  : %d\n", logstr, user_info->pw_uid);
    lcmaps_log(LOG_DEBUG,"%s: group_id : %d\n", logstr, user_info->pw_gid);
    lcmaps_log(LOG_DEBUG,"%s: home dir : %s\n", logstr, user_info->pw_dir);

    /* Add this credential data to the credential data repository in the plugin
     * manager */
    addCredentialData(UID, &(user_info->pw_uid));

    /* Handle primary GID from account.
     * NOTE we can add pGID both as pGID and as sGID */
    if ( (pgid_mapping&1) == 1) {
	/* Map primary Unix GID from the account info */
	lcmaps_log(LOG_DEBUG,
		"%s: adding primary GID (%d) from local account\n",
		logstr, user_info->pw_gid);
	addCredentialData(PRI_GID, &(user_info->pw_gid));
    }
    if ( (pgid_mapping&2) == 2 ) {
	/* Add the primary GID from the mapped account as a secondary GID to the
	 * result */
	lcmaps_log(LOG_DEBUG,
		"%s: adding primary GID (%d) from local account "
		"as a secondary GID\n", logstr, user_info->pw_gid);
	addCredentialData(SEC_GID, &(user_info->pw_gid));
    }

    /* Handle secondary GIDs from account */
    if (sgid_mapping==1 &&
	lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
    {
	/* Add secondary Unix group IDs from the mapped local account */
	lcmaps_log(LOG_DEBUG,
		"%s: adding secondary GIDs (%d) from local account\n",
		logstr, user_info->pw_gid);
	for (i = 0; i < cnt_sec_gid; i++)
	    addCredentialData(SEC_GID, &(sec_gid[i]));
	free(sec_gid);
    }

    /* success */
    if (req_username_needs_free)
	free(req_username);
    free(username);

    lcmaps_log(LOG_INFO,"%s: voms_localaccount plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_plugin:
    if (req_username_needs_free)
	free(req_username);
    free(username);

    lcmaps_log(LOG_INFO,"%s: voms_localaccount plugin failed\n", logstr);

    return LCMAPS_MOD_FAIL;
}
