/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_account.h
    \brief  Utility functions to initialize, fill and clean an lcmaps account structure.
    \author Martijn Steenbakkers for the EU DataGrid.

    Utility functions to initialize, fill and clean an lcmaps account structure.
    This account structure is used by several lcmaps interfaces.

    \ingroup LcmapsInterface
*/

#ifndef LCMAPS_ACCOUNT_H
#define LCMAPS_ACCOUNT_H

#ifndef LCMAPS_ACCOUNT_C
#   define LCMAPS_ACCOUNT_C_STORAGE_CLASS extern
#else
#   define LCMAPS_ACCOUNT_C_STORAGE_CLASS
#endif

/******************************************************************************
                             Include header files
******************************************************************************/
#include <stdlib.h>
#include <sys/types.h>
#include <pwd.h>

/******************************************************************************
                               Type definitions
******************************************************************************/

/*!
    \struct lcmaps_account_info_s
    \brief structure that contains the local account information gathered by LCMAPS
*/
/*!
    \brief Type of LCMAPS account info struct
*/
typedef struct lcmaps_account_info_s
{
    uid_t          uid;         /*!< the uid of the local account */
    gid_t *        pgid_list;   /*!< the list of primary gids */
    int            npgid;       /*!< the number of primary gids found */
    gid_t *        sgid_list;   /*!< the list of secondary gids */
    int            nsgid;       /*!< the number of secondary gids found */
    char *         poolindex;   /*!< the pool index */
} lcmaps_account_info_t;


/******************************************************************************
                   Module definition
 *****************************************************************************/

/******************************************************************************
Function:       lcmaps_account_info_init()
Description:    Initialize the account info structure
Parameters:
                plcmaps_account: pointer to the account info struct to be filled
Returns:        0: succes
                1: failed to initialize the account info structure
******************************************************************************/
/*!
    \fn lcmaps_account_info_init(
        lcmaps_account_info_t * plcmaps_account
    )
    \brief Initialize the account info structure
    \param plcmaps_account  pointer to the account info struct to be filled

    \retval 0 success.
    \retval 1 failed to clean the account info structure
*/
typedef int
lcmaps_account_info_init_t(
        lcmaps_account_info_t * plcmaps_account
);
lcmaps_account_info_init_t lcmaps_account_info_init;


/******************************************************************************
Function:       lcmaps_account_info_fill()
Description:    Fill structure with account info
Parameters:
                puid:            ptr to the uid of the local account
                ppgid_list:      ptr to the list of primary gids
                pnpgid:          ptr to the number of primary gids found
                psgid_list:      ptr to the list of secondary gids
                pnsgid:          ptr to the number of secondary gids found
                ppoolindex:      ptr to the pool index
                plcmaps_account: pointer to the account info struct to be filled
Returns:        0: succes
                1: failed to fill the account info structure
                -1: other failure
******************************************************************************/
/*!
    \fn lcmaps_account_info_fill(
        uid_t *                 puid,
        gid_t **                ppgid_list,
        int *                   pnpgid,
        gid_t **                psgid_list,
        int *                   pnsgid,
        char **                 ppoolindex,
        lcmaps_account_info_t * plcmaps_account
    )
    \brief Fill structure with account info
    \param puid             ptr to the uid of the local account
    \param ppgid_list       ptr to the list of primary gids
    \param pnpgid           ptr to the number of primary gids found
    \param psgid_list       ptr to the list of secondary gids
    \param pnsgid           ptr to the number of secondary gids found
    \param ppoolindex       ptr to the pool index
    \param plcmaps_account  pointer to the account info struct to be filled

    \retval 0 success.
    \retval 1 failed to fill the account info structure
    \retval -1 other failure
    \internal
*/
LCMAPS_ACCOUNT_C_STORAGE_CLASS int
lcmaps_account_info_fill(
        uid_t *                 puid,
        gid_t **                ppgid_list,
        int *                   pnpgid,
        gid_t **                psgid_list,
        int *                   pnsgid,
        char **                 ppoolindex,
        lcmaps_account_info_t * plcmaps_account
);


/*!
    \brief Clean the account info structure
    \param plcmaps_account  pointer to the account info struct to be filled

    \retval 0 success.
    \retval 1 failed to clean the account info structure
*/
typedef int
lcmaps_account_info_clean_t(
        lcmaps_account_info_t * plcmaps_account
);

/**
   \brief Clean the account info structure
 */
#ifndef LCMAPS_USE_DLOPEN
lcmaps_account_info_clean_t lcmaps_account_info_clean;
#endif /* LCMAPS_USE_DLOPEN */

#endif /* LCMAPS_ACCOUNT_H */
