/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_voms_attributes.c
    \brief  retrieve VOMS attributes (FQANs) from x509
    \author Oscar Koeroo for EMI/IGE

    Last change: support for VOMS' generic attributes  (Placi Flury flury@switch.ch)
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include "gssapi.h"

#include "lcmaps_log.h"
#include "lcmaps_types.h"
#include "_lcmaps_gsi_utils.h"
#include "_lcmaps_voms_attributes.h"


/*#include "voms_apic.h"*/
#include <voms/voms_apic.h>

/******************************************************************************
                                Definitions
******************************************************************************/

/* #define VOMS_BUFFER_SIZE 1024 */
#define DEFAULT_CERT_DIR "/etc/grid-security/certificates"
#define DEFAULT_VOMS_DIR "/etc/grid-security/vomsdir"

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

static void set_vomsdir_certdir(void);
static void print_lcmaps_vomsdata(lcmaps_vomsdata_t * lcmaps_vomsdata);
#ifdef LCMAPS_DEPRECATED
static void print_vomsdata(struct vomsdata *);
#endif /* LCMAPS_DEPRECATED */

/*
static void lcmaps_fetch_VOMS_credentials(STACK_OF(X509) *,
                                          char ***,
                                          int *, lcmaps_vomsdata_t **);
*/
static int lcmaps_get_voms_generic_attributes(struct voms *vo,
                                              struct vomsdata *vd,
                                              lcmaps_voms_generic_attr_t **attr_list,
                                              int *nattrs);
void lcmaps_print_stack_of_x509_to_string(STACK_OF(X509) * px509_chain,
                                          char *output_file);
void lcmaps_print_x509_to_string(X509 * px509_cert, char *output_file);


/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *certdir = NULL;
static char *vomsdir = NULL;

/*
 * Always verify VOMS credentials by default
 * as designed by the VOMS developers
 */
static int   verify_voms_credentials = 1;


/* Everything ok (= OpenSSL X509_V_OK) */
#define GRST_RET_OK             0

/* Failed for unspecified reason */
#define GRST_RET_FAILED         1000

#define MAXTAG 500

#define GRST_PROXYCERTINFO_OID  "1.3.6.1.4.1.3536.1.222"
#define GRST_VOMS_OID           "1.3.6.1.4.1.8005.100.100.5"
#define GRST_VOMS_DIR           "/etc/grid-security/vomsdir"

#define GRST_ASN1_MAXCOORDLEN   50
#define GRST_ASN1_MAXTAGS       500


#ifdef GRST_ACTIVE
struct GRSTasn1TagList
{
    char treecoords[GRST_ASN1_MAXCOORDLEN + 1];
    int start;
    int headerlength;
    int length;
    int tag;
};

int GRSTasn1GetX509Name(char *, int, char *, char *,
                        struct GRSTasn1TagList taglist[], int);
#endif

/* reverse error codes from voms_apic.h */
static char *retmsg[] =
    { "VERR_NONE", "VERR_NOSOCKET", "VERR_NOIDENT", "VERR_COMM",
    "VERR_PARAM", "VERR_NOEXT", "VERR_NOINIT",
    "VERR_TIME", "VERR_IDCHECK", "VERR_EXTRAINFO",
    "VERR_FORMAT", "VERR_NODATA", "VERR_PARSE",
    "VERR_DIR", "VERR_SIGN", "VERR_SERVER",
    "VERR_MEM", "VERR_VERIFY", "VERR_IDENT", "VERR_TYPE"
};




/******************************************************************************
Function:   lcmaps_disable_voms_attributes_verification
Description:
            Disables the verification in the VOMS API
Parameters:
            Void
Returns:
            Void
******************************************************************************/
void lcmaps_disable_voms_attributes_verification (void)
{
    /* disable verification */
    verify_voms_credentials = 0;
}

/******************************************************************************
Function:   lcmaps_enable_voms_attributes_verification
Description:
            Enables the verification in the VOMS API
Parameters:
            Void
Returns:
            Void
******************************************************************************/
void lcmaps_enable_voms_attributes_verification (void)
{
    /* enable verification */
    verify_voms_credentials = 1;
}

/******************************************************************************
Function:   lcmaps_is_set_to_verify_voms_attributes
Description:
            Will return the current setting to enable or disable the
            verification of the VOMS credentials by the VOMS API
Parameters:
            Void
Returns:
            1 : Verification is enabled
            0 : Verification is disabled
******************************************************************************/
int lcmaps_is_set_to_verify_voms_attributes (void)
{
    /* Will verify VOMS credentials? */
    return verify_voms_credentials;
}


/**************************************************************************
 * Read the generic attributes from the usercert.
 *
 * Paramters:
 *      vo: the VO data structure
 *      vd: the overall VO information
 *      attr_list: list of the generic attributes
 *      nattrs: number of attributes
 * Return:
 *      0: success (notice, the attribute list can still be empty)
 *      1: failure (could not read certificate attributes)
 **************************************************************************/
static int lcmaps_get_voms_generic_attributes(struct voms *vo, struct vomsdata *vd,
                                              lcmaps_voms_generic_attr_t **attr_list, int *nattrs)
{
    int handle, attrSrcNum;
    int errno_local = 0;
    int i, j, k;
    /* struct attribute *tmp_list; */
    struct attribute voms_ga;


    /* Could be the amount of vomsdata_t sources for Generic Attributes - need to check semantics */
    attrSrcNum = VOMS_GetAttributeSourcesNumber(vo, vd, &errno_local);
    if (errno_local != 0) {
        lcmaps_log(LOG_ERR, "%s Error: VOMS_GetAttributeSourcesNumber(), %s", __func__, retmsg[errno_local]);
        return 1;
    }

    /* Count the list of (struct attribute)'s which each hold one Generic Attribute */
    for (i = 0, *nattrs = 0; i < attrSrcNum; i++) {
        handle = VOMS_GetAttributeSourceHandle(vo, i, vd, &errno_local);
        if (errno_local != 0) {
            lcmaps_log(LOG_ERR, "%s Error: VOMS_AttributeSourceHandle(), %s", __func__, retmsg[errno_local]);
            *nattrs = 0;
            return 1;
        }
        *nattrs += VOMS_GetAttributesNumber(vo, handle, vd, &errno_local);
    }
    lcmaps_log_debug(3, "%s: found %d generic attributes. \n", __func__, *nattrs);

    /* Allocating lcmaps_voms_generic_attr_t's */
    *attr_list = (lcmaps_voms_generic_attr_t *) calloc (*nattrs, sizeof(lcmaps_voms_generic_attr_t));
    if (*attr_list == NULL) {
        lcmaps_log(LOG_ERR, "%s: Error: Could not allocate more memory: %s\n", __func__, strerror(errno));
        return 1;
    }

    /* Loop the Generic Attributes in VOMS and copy them to LCMAPS - We want to own that memory */
    for (i = 0, k = 0; i < attrSrcNum; i++) {
        /* Select a VOMS source object */
        handle = VOMS_GetAttributeSourceHandle(vo, i, vd, &errno_local);
        if (errno_local) {
            lcmaps_log(LOG_ERR, "%s Error: VOMS_GetAttributeSourceHandle, %s \n", __func__, retmsg[errno_local]);
            goto lcmaps_get_attributes_cleanup;
        }

        /* Loop the VOMS source object for its VOMS Generic Attributes */
        for (j = 0; j < VOMS_GetAttributesNumber(vo, handle, vd, &errno_local); j++) {
            if (errno_local) {
                lcmaps_log(LOG_ERR, "%s Error: VOMS_GetAttributesNumber, %s \n", __func__, retmsg[errno_local]);
                goto lcmaps_get_attributes_cleanup;
            }
            /* VOMS_GetAttribute(vo, handle, j, &(tmp_list[k++]), vd, &errno_local); */
            voms_ga.name=voms_ga.value=voms_ga.qualifier=NULL;
            VOMS_GetAttribute(vo, handle, j, &voms_ga, vd, &errno_local);
            if (errno_local) {
                lcmaps_log(LOG_ERR, "%s Error: VOMS_GetAttribute, %s \n", __func__, retmsg[errno_local]);
                goto lcmaps_get_attributes_cleanup;
            }

            /* Copying the VOMS Generic Attributes */
            if (voms_ga.name) {
                (*attr_list)[k].name      = strdup(voms_ga.name);
            } else {
                (*attr_list)[k].name      = NULL;
            }
            if (voms_ga.value) {
                (*attr_list)[k].value     = strdup(voms_ga.value);
            } else {
                (*attr_list)[k].value     = NULL;
            }
            if (voms_ga.qualifier) {
                (*attr_list)[k].qualifier = strdup(voms_ga.qualifier);
            } else {
                (*attr_list)[k].qualifier = NULL;
            }

            k++;
        }
    }
    return 0;

lcmaps_get_attributes_cleanup:
    for (i = 0; (i < *nattrs) && *attr_list; i++) {
        free((*attr_list)[i].name);
        free((*attr_list)[i].value);
        free((*attr_list)[i].qualifier);
    }
    free(*attr_list);
    *attr_list = NULL;
    *nattrs = 0;
    return 1;
}


/******************************************************************************
Function:    lcmaps_x509_to_voms_fqans
Description:
    Extract from X509 certificate chain (of types X509, STACK_OF(X509)) a list of VOMS
    fully qualified attribute names (FQANs)
Parameters:
    px509_cred:     pointer to a x509 credential
    px509_chain:    pointer to a x509 chain
    pnfqan:         the number of fqans found. If the proxy is empty (VERR_NOEXT),
                    this will be set to zero. Not an immediate error !
Returns:
    A list of fqans or NULL
******************************************************************************/

char **lcmaps_x509_to_voms_fqans(X509 * px509_cred,
                                 STACK_OF(X509) * px509_chain,
                                 lcmaps_vomsdata_t ** lcmaps_vomsdata,
                                 int *pnfqan)
{
    struct vomsdata *vd = NULL;
    struct voms **volist = NULL;
    struct voms *vo = NULL;
    lcmaps_vomsdata_t *tmp_lcmaps_vomsdata = NULL;
    lcmaps_voms_t *tmp_lcmaps_voms = NULL;
    lcmaps_fqan_unix_t *tmp_fqan = NULL;
    /* lcmaps_voms_generic_attr_t * attr_list = NULL; */
    char **fqans_list = NULL;
    int fqan_count = 0;
    int errNo = 0;
    int k,j;
    int nattrs = 0;
    const char *template_hint = "LCMAPS will ignore the VOMS extensions in the mapping sequence";

#ifdef DEPRECATED_VOMS_VERSION_CHECK
    int vomsmajor = -1;
    int vomsminor = -1;
    int vomspatch = -1;
#endif

#ifdef DEPRECATED_VOMS_VERSION_CHECK
    vomsmajor = getMajorVersionNumber();
    vomsminor = getMinorVersionNumber();
    vomspatch = getPatchVersionNumber();
#endif


    /* VOMS dir through API takes presedence over environment */

    /* CA dir through API takes presedence over environment */

    set_vomsdir_certdir();

    if ((vd = VOMS_Init(vomsdir, certdir)) == NULL) {
        lcmaps_log(LOG_ERR, "%s(): failed to initialize voms data structure. Either the VOMSDIR directory (%s) or the CA certificate directory (%s) does not exist.\n", __func__, vomsdir, certdir);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    lcmaps_log_debug(5, "%s(): VOMS data structure initialized\n", __func__);

#ifdef GUMSOSG
    /*
     * OSG GUMS - without verification
     */
    /*
     * Switching off the verification of the VOMS Attributes, just reading...
     */

    /* Warning! Warning! Warning! - Disabling the VOMS Verification here */

    /* VOMS_SetVerificationType(VERIFY_NONE, vd, &errNo); */
    lcmaps_disable_voms_attributes_verification();
#endif

    /* Enables or disables the VOMS attribute verification */
    if (lcmaps_is_set_to_verify_voms_attributes() == 0)
    {
        VOMS_SetVerificationType(VERIFY_NONE, vd, &errNo);
        lcmaps_log_debug (1, "%s(): VOMS attribute verification is disabled.\n", __func__);
    }

    /* Extract the VOMS attributes */
    if (VOMS_Retrieve(px509_cred, px509_chain, RECURSE_CHAIN, vd, &errNo))
    {
        volist = vd->data;
        tmp_lcmaps_vomsdata =(lcmaps_vomsdata_t *) calloc(1, sizeof(lcmaps_vomsdata_t));

        for(k = 0, (vo = volist[k]); (vo = volist[k]); k++){
            lcmaps_log_debug(3, "%s(): setting voms data for VO == %s\n", __func__, vo->voname);
            lcmaps_log_debug(3, "%s(): setting voms data for VO server == %s\n", __func__, vo->server);

            tmp_lcmaps_voms = (lcmaps_voms_t *) realloc(tmp_lcmaps_voms,sizeof(lcmaps_voms_t) * (k+1));
            tmp_lcmaps_voms[k].user_dn = strdup(vo->user);
            tmp_lcmaps_voms[k].user_ca = strdup(vo->userca);
            tmp_lcmaps_voms[k].voms_issuer_dn = strdup(vo->server);
            tmp_lcmaps_voms[k].voms_issuer_ca = strdup(vo->serverca);
            tmp_lcmaps_voms[k].uri = strdup(vo->uri);
            tmp_lcmaps_voms[k].date1 = strdup(vo->date1);
            tmp_lcmaps_voms[k].date2 = strdup(vo->date2);
            tmp_lcmaps_voms[k].voname = strdup(vo->voname);

            switch (vo->type) {
            case TYPE_NODATA:
                lcmaps_log_debug(2, "%s(): NO DATA TYPE \n", __func__);
                break;
            case TYPE_CUSTOM:
                lcmaps_log_debug(2, "%s(): CUSTOM TYPE, %s\n", __func__,
                                 vo->datalen - 10, vo->custom);
                break;
            case TYPE_STD:
                lcmaps_log_debug(2, "%s(): TYP_STD\n", __func__);
                if (vo->fqan)
                {
                    for (j = 0, tmp_fqan = NULL; (vo->fqan)[j]; j++)
                    {
                        tmp_fqan         = (lcmaps_fqan_unix_t *) realloc(tmp_fqan,sizeof(lcmaps_fqan_unix_t) * (j+1));
                        tmp_fqan[j].fqan = strdup((vo->fqan)[j]);
                        tmp_fqan[j].uid  = -1;
                        tmp_fqan[j].gid  = -1;
                        /* make a copy for the char** list of fqans  */
                        if(fqans_list == NULL)
                        {
                            fqans_list = (char **) calloc(1,sizeof(char*));
                            *fqans_list = NULL;
                        }
                        if(*fqans_list == NULL)
                        {
                            *fqans_list = strdup((vo->fqan)[j]);
                        }
                        else
                        {
                            fqans_list = (char **) realloc(fqans_list, sizeof(char *) * (fqan_count + 1));
                            fqans_list[fqan_count] = strdup((vo->fqan)[j]);
                        }
                        fqan_count++;
                    }
                    tmp_lcmaps_voms[k].fqan_unix = tmp_fqan;
                    tmp_lcmaps_voms[k].nfqan = j;
                }

                /* read generic voms attributes   */
                tmp_lcmaps_voms[k].attr_list = NULL;
                tmp_lcmaps_voms[k].nattr = 0;


#ifdef DEPRECATED_VOMS_VERSION_CHECK
                /*
                 * The extraction of the Generic Attributes from VOMS
                 * can only be performed when the version of the used
                 * API is > 1.7.0
                 */
                if ( (vomsmajor >= 1) &&
                     (vomsminor >= 7) &&
                     (vomspatch >= 0) )
                {
                    if (lcmaps_get_voms_generic_attributes(vo, vd, &tmp_lcmaps_voms[k].attr_list, &nattrs) == 0)
                    {
                        lcmaps_log_debug(4, "%s(): extracted '%d' generic voms attributes \n", __func__, nattrs);
                        tmp_lcmaps_voms[k].nattr = nattrs;
                    }
                    else
                    {
                        lcmaps_log_debug(3, "%s(): could not extract any generic voms attributes (failure?)\n", __func__);
                    }
                }
#else
                if (lcmaps_get_voms_generic_attributes(vo, vd, &tmp_lcmaps_voms[k].attr_list, &nattrs) == 0)
                {
                    lcmaps_log_debug(4, "%s(): extracted '%d' generic voms attributes \n", __func__, nattrs);
                    tmp_lcmaps_voms[k].nattr = nattrs;
                }
                else
                {
                    lcmaps_log_debug(3, "%s(): could not extract any generic voms attributes (failure?)\n", __func__);
                }
#endif
                break;
            default:
                lcmaps_log_debug(2, "%s(): WARNING - unknown vo->type\n", __func__);
            }
            tmp_lcmaps_vomsdata->voms = tmp_lcmaps_voms;
            tmp_lcmaps_vomsdata->nvoms = k+1;
        } /* while */
        if (vd->workvo && tmp_lcmaps_vomsdata) {
            tmp_lcmaps_vomsdata->workvo = strdup(vd->workvo);
        }
        else {
            tmp_lcmaps_vomsdata->workvo = NULL;
        }
        if (vd->extra_data && tmp_lcmaps_vomsdata) {
            tmp_lcmaps_vomsdata->extra_data = strdup(vd->extra_data);
        }
        else {
            tmp_lcmaps_vomsdata->extra_data = NULL;
        }
        goto success_lcmaps_x509_to_voms_fqans;

    }
    else if (errNo == VERR_NOEXT) {     /* Non voms proxies will fall into this error */
        /* Non voms proxies go here: set the number of fqans to zero. Not an error */
        lcmaps_log_debug(2, "%s(): Debug: VOMS extensions not present in the certificate chain.\n", __func__);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_IDCHECK) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS User data in extension different from the real ones. This could mean that the owner's VOMS credentials got transplanted/stolen from another. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_TIME) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS extensions expired for at least one of the VOs. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_ORDER) {
        lcmaps_log(LOG_ERR, "%s(): Warning: The ordering of the VOMS groups, as required by the client, was not delivered by VOMS. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_NOSOCKET) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS Socket problem. (an error that indicates an ABI change in VOMS). %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_NOIDENT) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS Cannot identify itself (certificate problem). %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_COMM) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS server problem %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_PARAM) {
        lcmaps_log(LOG_ERR, "%s(): Warning: Wrong parameters for VOMS %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_NOINIT) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS initialization error. Please check the VOMS directory (default: /etc/grid-security/vomsdir/) %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_EXTRAINFO) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VO name and URI missing making it an invalid VOMS extension. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_FORMAT) {
        lcmaps_log(LOG_ERR, "%s(): Warning: Wrong VOMS data format making it an invalid VOMS extension. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_NODATA) {
        lcmaps_log(LOG_ERR, "%s(): Warning: Empty VOMS extension making it an invalid VOMS extension. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_PARSE) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS parse error making it an invalid VOMS extension. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_DIR) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS directory error. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_SIGN) {
        lcmaps_log(LOG_ERR, "%s(): Warning: VOMS Signature error making it an unverifyable VOMS extension. Hint: check the VOMS directory (default: /etc/grid-security/vomsdir/) and the .lsc files. Also you could fallback on using installed VOMS host certificates. If the problem persists, contact the user. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_SERVER) {
        lcmaps_log(LOG_ERR, "%s(): Warning: Unidentifiable VOMS server. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_MEM) {
        lcmaps_log(LOG_ERR, "%s(): Warning: Memory problems in VOMS_Retrieve() %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_VERIFY) {
        char *err_buffer = VOMS_ErrorMessage(vd, VERR_VERIFY, NULL, 0);
        lcmaps_log(LOG_ERR, "%s(): Warning: Generic verification error for VOMS: \"%s\". %s.\n", __func__, err_buffer, template_hint);
        free(err_buffer);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else if (errNo == VERR_TYPE) {
        lcmaps_log(LOG_ERR, "%s(): Warning: Returned VOMS data of unknown type. %s.\n", __func__, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }
    else {
        lcmaps_log(LOG_ERR, "%s(): VOMS_Retrieve() error --> %d %s.\n", __func__, errNo, template_hint);
        goto fail_lcmaps_x509_to_voms_fqans;
    }


success_lcmaps_x509_to_voms_fqans:
    lcmaps_log_debug(3, "%s(): Success, VOMS destroy\n", __func__);
    if (vd)
        VOMS_Destroy(vd);
    print_lcmaps_vomsdata(tmp_lcmaps_vomsdata);
    *lcmaps_vomsdata = tmp_lcmaps_vomsdata;
    *pnfqan = fqan_count;
    return fqans_list;

fail_lcmaps_x509_to_voms_fqans:
    lcmaps_log_debug(3, "%s(): Failure, VOMS destroy\n", __func__);
    VOMS_Destroy(vd);
    *pnfqan = fqan_count; /* NOTE: invalid FQAN found only if this is != 0 */
    return NULL;

}

static void set_vomsdir_certdir(void)
{
    /* Retrieve the vomsdir and certdir */
    vomsdir = getenv("LCMAPS_X509_VOMS_DIR");
    vomsdir = (vomsdir ? vomsdir : getenv("X509_VOMS_DIR"));
    vomsdir = (vomsdir ? vomsdir : DEFAULT_VOMS_DIR);

    certdir = getenv("LCMAPS_X509_CERT_DIR");
    certdir = (certdir ? certdir : getenv("X509_CERT_DIR"));
    certdir = (certdir ? certdir : DEFAULT_CERT_DIR);
}


/******************************************************************************
Function:    print_lcmaps_vomsdata
Description:
             print the lcmaps_vomsdata structure
Parameters:
    lcmaps_vomsdata_t * lcmaps_vomsdata
Returns:
    void
******************************************************************************/
static void print_lcmaps_vomsdata(lcmaps_vomsdata_t * lcmaps_vomsdata)
{
    char *logstr = "    print_lcmaps_vomsdata()";
    int i = 0;
    int j = 0;


    if (lcmaps_vomsdata) {
        for (i = 0; i < lcmaps_vomsdata->nvoms; i++) {
            lcmaps_log_debug(3, "%s(): lcmaps_vomsdata->voms[ %d / %d ]\n", __func__,
                       i + 1, lcmaps_vomsdata->nvoms);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].user_dn           : %s\n",
                       logstr, i + 1, lcmaps_vomsdata->voms[i].user_dn);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].user_ca           : %s\n",
                       logstr, i + 1, lcmaps_vomsdata->voms[i].user_ca);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].voms_issuer_dn    : %s\n",
                       logstr, i + 1,
                       lcmaps_vomsdata->voms[i].voms_issuer_dn);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].voms_issuer_ca    : %s\n",
                       logstr, i + 1,
                       lcmaps_vomsdata->voms[i].voms_issuer_ca);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].uri               : %s\n",
                       logstr, i + 1, lcmaps_vomsdata->voms[i].uri);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].date1             : %s\n",
                       logstr, i + 1, lcmaps_vomsdata->voms[i].date1);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].date2             : %s\n",
                       logstr, i + 1, lcmaps_vomsdata->voms[i].date2);
            lcmaps_log_debug(3,
                       "%s: lcmaps_vomsdata->voms[%d].voname            : %s\n",
                       logstr, i + 1, lcmaps_vomsdata->voms[i].voname);

            for (j = 0; j < lcmaps_vomsdata->voms[i].nfqan; j++) {
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].fqan_unix[ %d / %d ]\n",
                           logstr, i + 1, j + 1,
                           lcmaps_vomsdata->voms[i].nfqan);
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].fqan_unix[%d].fqan : %s\n",
                           logstr, i + 1, j + 1,
                           lcmaps_vomsdata->voms[i].fqan_unix[j].fqan);
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].fqan_unix[%d].uid  : %d\n",
                           logstr, i + 1, j + 1,
                           lcmaps_vomsdata->voms[i].fqan_unix[j].uid);
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].fqan_unix[%d].gid  : %d\n",
                           logstr, i + 1, j + 1,
                           lcmaps_vomsdata->voms[i].fqan_unix[j].gid);
            }
            for(j=0; j < lcmaps_vomsdata->voms[i].nattr; j++){
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].attr_list[%d].name : %s\n",
                           logstr, i+1, j+1,
                           lcmaps_vomsdata->voms[i].attr_list[j].name ? lcmaps_vomsdata->voms[i].attr_list[j].name : "(n/a)");
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].attr_list[%d].value : %s\n",
                           logstr, i+1, j+1,
                           lcmaps_vomsdata->voms[i].attr_list[j].value ? lcmaps_vomsdata->voms[i].attr_list[j].value : "(n/a)");
                lcmaps_log_debug(3,
                           "%s: lcmaps_vomsdata->voms[%d].attr_list[%d].qualifier : %s\n",
                           logstr, i+1, j+1,
                           lcmaps_vomsdata->voms[i].attr_list[j].qualifier ? lcmaps_vomsdata->voms[i].attr_list[j].qualifier : "(n/a)");
            }
        }
        if(lcmaps_vomsdata->workvo)
            lcmaps_log_debug(3, "%s: lcmaps_vomsdata->workvo                    : %s\n",
                               logstr, lcmaps_vomsdata->workvo);
        if(lcmaps_vomsdata->extra_data)
            lcmaps_log_debug(3, "%s: lcmaps_vomsdata->extra_data                : %s\n",
                               logstr, lcmaps_vomsdata->extra_data);
    }
    else {
        lcmaps_log_debug(3, "%s(): No LCMAPS vomsdata found\n", __func__);
    }
}




/******************************************************************************
Function:    lcmaps_print_stack_of_x509_to_string
Description:
    Extract from X509 certificate chain (STACK_OF(X509)) and
        write the output to a file.
    Simulair to `openssl x509 -in <file.pem> -text -noout

    Note: only for debugging purposes
Parameters:
    px509_chain:    pointer to a x509 chain
Returns:
    void
******************************************************************************/
void lcmaps_print_stack_of_x509_to_string(STACK_OF(X509) * px509_chain,
                                          char *output_file)
{

    if (px509_chain) {
        STACK_OF(X509) * dupChain = NULL;
        X509 *cert = NULL;

/*        dupChain = (STACK_OF(type) *) sk_X509_dup(px509_chain);*/
        dupChain = sk_X509_dup(px509_chain);
        lcmaps_log_debug(1, "%s()\n", __func__);

        while ((cert = sk_X509_pop(dupChain))) {
            lcmaps_print_x509_to_string(cert, output_file);
        }

        if (dupChain)
            sk_X509_free(dupChain);
    }
    else {
        lcmaps_log_debug(1, "%s(): no input X509 chain!\n", __func__);
    }
}

/******************************************************************************
Function:    lcmaps_print_x509_to_string
Description:
    Extract from X509 certificate and
        write the output to a file.
    Simulair to `openssl x509 -in <file.pem> -text -noout

    Note: only for debugging purposes
Parameters:
    px509:    pointer to a x509 cert
Returns:
    void
******************************************************************************/
void lcmaps_print_x509_to_string(X509 * px509, char *output_file)
{
    if (px509) {
        FILE *tmpFile = NULL;

        lcmaps_log_debug(1, "%s()\n", __func__);

        tmpFile = fopen(output_file, "a");
        X509_print_fp(tmpFile, px509);
        fclose(tmpFile);
    }
    else {
        lcmaps_log_debug(1, "%s(): no input X509 cert!\n", __func__);
    }
}


/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/grid_credential_handling/gsi_handling/lcmaps_voms_attributes.c,v $
    $Date: 2012-11-01 21:04:45 +0100 (Thu, 01 Nov 2012) $
    $Revision: 16744 $
    $Author: okoeroo $
******************************************************************************/
