/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_credential.c
    \brief  This files contains the methods to access a lcmaps credential
    \author Martijn Steenbakkers for EGEE

    This header contains the methods to access a lcmaps credential. The lcmaps
    credential is a generic container for all kinds of credentials:
    DN, GSI (gss_cred_id_t), security context (gss_ctx_id_t), or parsed voms attributes (FQANs)
*/

#define _XOPEN_SOURCE	500

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_config.h"

/* Needed for NULL */
#include <stdio.h>

#include <stdlib.h>
#include <string.h>

#include <sys/types.h>

#ifdef LCMAPS_GSI_MODE
#   include <gssapi.h>
#   include <openssl/x509.h>
#   include "x509_handling/_lcmaps_x509_utils.h"
#   include "gsi_handling/_lcmaps_gsi_utils.h"
#   include "gsi_handling/_lcmaps_voms_attributes.h"
#endif
#include "_lcmaps_credential.h"

/* LCMAPS includes */
#include "lcmaps_log.h"
#include "lcmaps_account.h"

#ifdef LCMAPS_GSI_MODE

#if 0
#define TRUE  1
#define FALSE 0
#endif


/******************************************************************************
Function:       lcmaps_credential_store_gss_cred_id_t_and_sub_elements
Description:    Fill credential with gss_cred_id_t (internally gsi credential)
                and all the derivative credentials are extracted and stored too
Parameters:
                gss_credential:     gss credential (internally gsi credential)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:        Succes
                LCMAPS_CRED_NO_GSS_CRED:    The supplied credential is empty
                LCMAPS_CRED_NO_X509_CHAIN:  No certificate chain to be extracted
                LCMAPS_CRED_NO_X509_CRED:   No final certificate extracted
                LCMAPS_CRED_NO_DN:          Failed to retrieve DN from
                                            gss_credential
                LCMAPS_CRED_NO_FQAN:        failed to retrieve VOMS attributes,
                                            because the proxy does not contain
                                            them (VERR_NOEXT)
                LCMAPS_CRED_ERROR:          error in retrieving VOMS attributes.
                LCMAPS_CRED_INVOCATION_ERROR:   failure, because lcmaps_credential
                                                does not exist
******************************************************************************/
int lcmaps_credential_store_gss_cred_id_t_and_sub_elements(
        gss_cred_id_t user_cred,
        lcmaps_cred_id_t * plcmaps_credential
        )
{
    int                 retval          = LCMAPS_CRED_SUCCESS;
    STACK_OF(X509) *    certchain       = NULL;
    X509 *              cert            = NULL;

    /* Only store the gss_cred_id_t */
    retval = lcmaps_credential_store_gss_cred_id_t(user_cred, plcmaps_credential);
    if (retval != LCMAPS_CRED_SUCCESS) {
        return retval;
    }

    /* Retrieve a newly created X509 chain from gss credential (should be freed) */
    if ((certchain = lcmaps_cred_to_x509_chain(user_cred))) {
        lcmaps_log_debug(LOG_DEBUG, "%s: found X509 chain inside gss credential\n", __func__);
    } else {
        lcmaps_log(LOG_ERR, "%s: could not convert the input GSS Credentials (a gss_cred_id_t object) to a certificate chain in a STACK_OF(X509) structure. (fatal error)\n", __func__);
        return LCMAPS_CRED_NO_GSS_CRED;
    }

    /* Retrieve a newly created X509 cert from gss credential (should be freed) */
    if ((cert = lcmaps_cred_to_x509(user_cred))) {
        lcmaps_log_debug(LOG_DEBUG, "%s: found X509 certificate inside gss credential\n", __func__);
    } else {
        lcmaps_log(LOG_ERR, "%s: could not convert the input GSS Credentials (a gss_cred_id_t object) to a certificate in an X509 structure. (fatal error)\n", __func__);
        return LCMAPS_CRED_NO_GSS_CRED;
    }

    /* Insert the EEC or leaf/last delegation/proxy into the certificate stack
     * at the beginning of the stack effectively the beginning of the chain  */
    lcmaps_log(LOG_DEBUG, "%s: Pushing certificate for the final delegation into the chain...\n", __func__);
    sk_X509_insert(certchain, cert, 0);

    /* Store and extract credential derivatives from the X.509 based credentials */
    retval |= lcmaps_credential_store_x509_and_sub_elements(cert, certchain, plcmaps_credential);

    /* NOTE: Adds cert NOT in the chain */

    return retval;
}


/******************************************************************************
Function:       lcmaps_credential_store_pem_string_and_sub_elements
Description:    Fill credential with PEM string and all the derivative
                credentials are extracted and stored too
Parameters:
                gss_credential:     gss credential (internally gsi credential)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:        Succes
                LCMAPS_CRED_NO_X509_CHAIN:  No certificate chain to be extracted
                LCMAPS_CRED_NO_X509_CRED:   No final certificate extracted
                LCMAPS_CRED_NO_DN:          Failed to retrieve DN from
                                            gss_credential
                LCMAPS_CRED_NO_FQAN:        failed to retrieve VOMS attributes,
                                            because the proxy does not contain
                                            them (VERR_NOEXT)
                LCMAPS_CRED_ERROR:          error in retrieving VOMS attributes.
                LCMAPS_CRED_INVOCATION_ERROR:   failure, because lcmaps_credential
                                                does not exist
******************************************************************************/
int lcmaps_credential_store_pem_string_and_sub_elements(
        char * pem_string,
        lcmaps_cred_id_t * plcmaps_credential
        )
{
    int                 retval          = LCMAPS_CRED_SUCCESS;
    STACK_OF(X509) *    certchain       = NULL;
    X509 *              cert            = NULL;
#if 0
    X509 *              certPrime       = NULL;
#endif

    /* Store the PEM string */
    retval = lcmaps_credential_store_pem_string(pem_string, plcmaps_credential);

    /* Convert from PEM to STACK_OF X.509 */
    if (lcmaps_pem_string_to_x509_chain(&certchain, pem_string) != 0) {
        lcmaps_log(LOG_ERR, "%s: could not convert the input PEM string to a certificate chain in a STACK_OF(X509) structure. (fatal error)\n", __func__);
        return LCMAPS_CRED_NO_X509_CHAIN;
    } else {
        lcmaps_log_debug(LOG_DEBUG, "%s: found X509 chain inside PEM string\n", __func__);
    }

    /* Double check */
    if (!certchain) {
        lcmaps_log(LOG_ERR, "%s: The lcmaps_pem_string_to_x509_chain() reported a positive result, but I don't have a chain... This is most unexpected.\n", __func__);
        return LCMAPS_CRED_NO_X509_CHAIN;
    }

#if 0
    /* Select the leaf certificate of the chain as the individual/last certificate */
    certPrime = sk_X509_value(certchain, 0);
    if (!certPrime) {
        lcmaps_log(LOG_ERR, "%s: could not select an individual X509 structure. (fatal error).\n", __func__);
        return LCMAPS_CRED_NO_X509_CRED;
    }

    cert = X509_dup(certPrime);
    if (!cert) {
        lcmaps_log(LOG_ERR, "%s: could not duplicate an individual X509 structure. (fatal error).\n", __func__);
        return LCMAPS_CRED_NO_X509_CRED;
    }
#endif
    cert=sk_X509_value(certchain, 0);
    if (!cert) {
        lcmaps_log(LOG_ERR, "%s: could not select an individual X509 structure. (fatal error).\n", __func__);
        return LCMAPS_CRED_NO_X509_CRED;
    }

    /* Store and extract credential derivatives from the X.509 based credentials */
    retval |= lcmaps_credential_store_x509_and_sub_elements(cert, certchain, plcmaps_credential);

    /* NOTE: Adds cert also IN the CHAIN */

    return retval;
}


/******************************************************************************
Function:       lcmaps_credential_store_x509_and_sub_elements
Description:    Fill credential with a X.509 certificate and STACK_OF(X509)
                and all the derivative credentials are extracted and stored too

Important note on our assumptions:
                We assume that the X509 *cert is a pointer to the first
                certificate in the stack, which means its effectivly the final
                certificate in the chain. This should typically be the EEC
                certificate or, using proxies, the final delegation in the
                chain.
                The calling function is responsible of providing a chain that
                has all the certificates in it, and optionally a pointer to the
                final certificate in the chain
Parameters:
                X509 *                      final delegation or last certificate
                                            by convention
                STACK_OF(X509) *            the certificate chain
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:        Succes
                LCMAPS_CRED_NO_X509_CHAIN:  No certificate chain to be extracted
                LCMAPS_CRED_NO_X509_CRED:   No final certificate extracted
                LCMAPS_CRED_NO_DN:          Failed to retrieve DN from
                                            gss_credential
                LCMAPS_CRED_NO_FQAN:        failed to retrieve VOMS attributes,
                                            because the proxy does not contain
                                            them (VERR_NOEXT)
                LCMAPS_CRED_ERROR:          error in retrieving VOMS attributes.
                LCMAPS_CRED_INVOCATION_ERROR:   failure, because lcmaps_credential
                                                does not exist
******************************************************************************/
int lcmaps_credential_store_x509_and_sub_elements(
        X509 * cert,
        STACK_OF(X509) * certchain,
        lcmaps_cred_id_t * plcmaps_credential
        )
{
    int                 retval          = LCMAPS_CRED_SUCCESS;
    char **             fqan_list       = NULL;
    int                 nfqan           = -1;
    lcmaps_vomsdata_t * lcmaps_vomsdata = NULL;
    char *              subject_dn      = NULL;
    char *              issuer_dn       = NULL;
    X509 *              the_cert        = NULL;

    /* Store the certificate chain credential */
    if ((LCMAPS_CRED_NO_X509_CHAIN == lcmaps_credential_store_stack_of_x509(certchain, plcmaps_credential))) {
        lcmaps_log(LOG_ERR, "%s: could not store STACK_OF(X509)\n", __func__);
        return LCMAPS_CRED_NO_X509_CHAIN;
    }

    /* Supporting the case where there is a chain, but not an explicit
     * certificate. Defaulting on the final certificate in the chain (aka the
     * last delegation) */
    if (cert == NULL && certchain != NULL) {
        the_cert = cgul_x509_select_final_cert_from_chain(certchain);
    } else {
        the_cert = cert;
    }

    /* Store the supplied certificate credential */
    if ((LCMAPS_CRED_NO_X509_CRED == lcmaps_credential_store_x509(the_cert, plcmaps_credential))) {
        lcmaps_log(LOG_ERR, "%s: could not store STACK_OF(X509)\n", __func__);
        return LCMAPS_CRED_NO_X509_CRED;
    }

    /* Traverse the chain to find the EEC and allocate the space needed to store the entire Subject DN */
    subject_dn = cgul_x509_chain_to_subject_dn(certchain);
    if (subject_dn == NULL) {
        /* Something is really wrong, this certificate doesn't have a Subject DN */
        lcmaps_log(LOG_ERR, "%s: The EEC certificate in the certificate chain does not contain a Subject DN.\n", __func__);
        return LCMAPS_CRED_NO_DN;
    }
    /* Storing the Subject DN string */
    if ((LCMAPS_CRED_NO_DN == lcmaps_credential_store_dn(subject_dn, plcmaps_credential))) {
        lcmaps_log(LOG_ERR, "%s: could not store Subject DN string\n", __func__);
        return LCMAPS_CRED_NO_DN;
    }
    /* Can free the subject_dn as it's dupped into the plcmaps_credential */
    free(subject_dn);

    /* Traverse the chain to find the EEC and allocate the space needed to store the entire Issuer DN */
    /* This information is not stored, but it's only used as a test to check if the EEC is properly created */
    issuer_dn = cgul_x509_chain_to_issuer_dn(certchain);
    if (issuer_dn == NULL) {
        /* Something is really wrong, this certificate doesn't have a Subject DN */
        lcmaps_log(LOG_ERR, "%s: The EEC certificate in the certificate chain does not contain an Issuer DN.\n", __func__);
        return LCMAPS_CRED_NO_DN;
    }
    /* Liberating memory - not stored */
    free(issuer_dn);

    /* Get FQANs */
    fqan_list = lcmaps_x509_to_voms_fqans(the_cert, certchain, &lcmaps_vomsdata, &nfqan);
    if (fqan_list == NULL) {
        if (nfqan == 0) {
            lcmaps_log(LOG_INFO, "%s: No VOMS FQANs found in X509 credentials\n", __func__);
            retval |= LCMAPS_CRED_NO_FQAN;
            plcmaps_credential->voms_data_list = NULL;
        } else {
            lcmaps_log(LOG_WARNING, "%s: Error retrieving VOMS attributes\n", __func__);
            return LCMAPS_CRED_ERROR;
        }
    } else {
        /* Store the FQAN list */
        if ((retval |= lcmaps_credential_store_fqan_list(nfqan, fqan_list, plcmaps_credential)) != LCMAPS_CRED_SUCCESS) {
            lcmaps_clean_list_of_strings(nfqan, fqan_list);
        } else {
            lcmaps_clean_list_of_strings(nfqan, fqan_list);
        }

        /* Adding the LCMAPS voms data structure to the LCMAPS crednetials */
        /* TODO */
        if (lcmaps_vomsdata) {
            lcmaps_log_debug(3, "%s: Found LCMAPS vomsdata structure, adding it to the lcmaps_credentials.\n", __func__);
            plcmaps_credential->voms_data_list = lcmaps_vomsdata;
        } else {
            lcmaps_log_debug(3, "%s: Did not find LCMAPS a VOMS data structure in the certificate chain.\n", __func__);
        }
    }
    /* End Get FQANs */

    return retval;
}



/******************************************************************************
Function:       lcmaps_credential_store_gss_cred_id_t()
Description:    Fill credential with gss_cred_id_t (internally gsi credential)
                If the lcmaps credential does not yet contain the DN,
                try to retrieve it from the gss credential.
                If the lcmaps credential does not yet contain VOMS
                attributes, try to retrieve them from the gss credential.
Parameters:
                gss_credential:     gss credential (internally gsi credential)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_NO_GSS_CRED:  The supplied credential is empty
                LCMAPS_CRED_NO_DN:   failed to retrieve DN from gss_credential
                LCMAPS_CRED_NO_FQAN: failed to retrieve VOMS attributes, because
                                     the proxy does not contain them (VERR_NOEXT)
                LCMAPS_CRED_ERROR:   error in retrieving VOMS attributes.
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential
                                     does not exist
******************************************************************************/
int lcmaps_credential_store_gss_cred_id_t(
        gss_cred_id_t      gss_credential,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
    {
        return LCMAPS_CRED_INVOCATION_ERROR;
    }

    /* Check if the gss credential is empty */
    if (gss_credential == GSS_C_NO_CREDENTIAL)
        return LCMAPS_CRED_NO_GSS_CRED;

    /* Fill credential, only if not filled before */
    if (plcmaps_credential->cred == GSS_C_NO_CREDENTIAL)
        plcmaps_credential->cred = gss_credential;

    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
Function:       lcmaps_credential_get_gss_cred_id_t()
Description:    returns gss credential (internally gsi credential)
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        gss credential
******************************************************************************/
gss_cred_id_t lcmaps_credential_get_gss_cred_id_t(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.cred);
}

/******************************************************************************
Function:       lcmaps_credential_store_gss_ctx_id_t()
Description:    Fill credential with gss_ctx_id_t (gss security context)
Parameters:
                gss_context:        gss security context
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_credential_store_gss_ctx_id_t(
        gss_ctx_id_t       gss_context,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    return LCMAPS_CRED_ERROR;
}

/******************************************************************************
Function:       lcmaps_credential_get_gss_ctx_id_t()
Description:    returns gss security context
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        gss security context
******************************************************************************/
gss_ctx_id_t lcmaps_credential_get_gss_ctx_id_t(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.context);
}

/******************************************************************************
Function:       lcmaps_credential_store_pem_string()
Description:    Fill credential with PEM-encoded string, needs freeing!
                If the lcmaps credential does not yet contain the X509, try
                to retrieve it from the string
Parameters:
                pem_string:         PEM-encoded string
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:  success
                LCMAPS_CRED_NO_PEM_STRING: PEM-encoded string is empty
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
                LCMAPS_CRED_ERROR: other error
******************************************************************************/
int lcmaps_credential_store_pem_string(
        char *             pem_string,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    char *pem_tmp=NULL;

    if (plcmaps_credential == NULL) {
        return LCMAPS_CRED_INVOCATION_ERROR;
    }

    if (pem_string == NULL)
        return LCMAPS_CRED_NO_PEM_STRING;

    if ( (pem_tmp=strdup(pem_string))==NULL)
	return LCMAPS_CRED_ERROR;

    plcmaps_credential->pem_string = pem_tmp;
    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
Function:       lcmaps_credential_get_pem_string()
Description:    returns PEM-encoded string
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        PEM-encoded string
******************************************************************************/
char * lcmaps_credential_get_pem_string(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.pem_string);
}


/******************************************************************************
Function:       lcmaps_credential_store_stack_of_x509()
Description:    Fill credential with X.509 certificate information
Parameters:
                px509_chain:        x509 proxy chain
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_NO_X509_CHAIN: The supplied credential chain is empty
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential
                                     does not exist
******************************************************************************/
int lcmaps_credential_store_stack_of_x509(
        STACK_OF(X509)   * px509_chain,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
        return LCMAPS_CRED_INVOCATION_ERROR;

    if (px509_chain == NULL) {
        return LCMAPS_CRED_NO_X509_CHAIN;
    } else {
        plcmaps_credential->px509_chain = px509_chain;
        return LCMAPS_CRED_SUCCESS;
    }

}

/******************************************************************************
Function:       lcmaps_credential_store_x509()
Description:    Fill credential with X.509 certificate information
Parameters:
                px509:              x509 proxy certificate (final delegation)
                px509_chain:        x509 proxy chain
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_NO_X509_CRED:  The supplied credential is empty
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential
                                     does not exist
                LCMAPS_CRED_ERROR: other error
******************************************************************************/
int lcmaps_credential_store_x509(
        X509 *             px509,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
        return LCMAPS_CRED_INVOCATION_ERROR;

    if (px509 == NULL)
        return LCMAPS_CRED_NO_X509_CRED;
    else {
	/* NOTE: the cert being stored here is the first of the chain and will
	 * be freed as such. Hence DON'T dup it here */
        plcmaps_credential->px509_cred = px509;
        return LCMAPS_CRED_SUCCESS;
    }
}

/******************************************************************************
Function:       lcmaps_credential_get_x509_cred()
Description:    Return the X509 cert.
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        the X509 cert.
******************************************************************************/
X509 * lcmaps_credential_get_x509_cred(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.px509_cred);
}

/******************************************************************************
Function:       lcmaps_credential_get_x509_chain()
Description:    Return the X509 cert chain
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        the X509 cert chain
******************************************************************************/
STACK_OF(X509) * lcmaps_credential_get_x509_chain(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.px509_chain);
}

#endif /* LCMAPS_GSI_MODE */


/******************************************************************************
Function:       lcmaps_credential_store_dn()
Description:    Fill credential with dn
Parameters:
                dn:                 distinguished name
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_NO_DN:   no DN to fill
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
                LCMAPS_CRED_ERROR:  other failure
******************************************************************************/
int lcmaps_credential_store_dn(
        char *             dn,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    char *dn_tmp=NULL;

    if (plcmaps_credential == NULL)
        return LCMAPS_CRED_INVOCATION_ERROR;

    /* Check if the dn is empty */
    if (dn == NULL)
        return LCMAPS_CRED_NO_DN;

    /* fill DN, only if not filled before from gss_cred or X509 */
    if (plcmaps_credential->dn == NULL)	{
	if ( (dn_tmp= strdup(dn)) == NULL)  /* out of mem */
	    return LCMAPS_CRED_ERROR;

	plcmaps_credential->dn = dn_tmp;
    }

    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
Function:       lcmaps_credential_get_dn()
Description:    returns user dn
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        user dn
******************************************************************************/
char * lcmaps_credential_get_dn(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.dn);
}

/******************************************************************************
Function:       lcmaps_credential_store_fqan_list()
Description:    Fill credential with list of FQANs
Parameters:
                nfqan:              number of elements of the FQAN list
                fqan_list:          FQAN list (list of strings)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_NO_FQAN: supplied fqan list is empty
                LCMAPS_CRED_ERROR: failure, because of a malloc error or a malformed fqan list
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_lcmaps_vomsdata (
        lcmaps_vomsdata_t * lcmaps_vomsdata,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    int i = 0;
    int j = 0;
    /* int nvoms = 0; */


    if (plcmaps_credential == NULL)
    {
        lcmaps_log_debug(1,"lcmaps_credential_store_lcmaps_vomsdata(): Create lcmaps_cred_id_t first!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }

    /* Check if the voms_data_list is empty */
    if (lcmaps_vomsdata == NULL)
    {
        lcmaps_log_debug(1,"lcmaps_credential_store_lcmaps_vomsdata(): Create lcmaps_vomsdata_t!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }

    /* Check if the nvoms itself is >0 */
    if (lcmaps_vomsdata->nvoms <= 0)
    {
        lcmaps_log_debug(1,"lcmaps_credential_store_lcmaps_vomsdata(): nvoms <= 0!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }

    if (plcmaps_credential->voms_data_list == NULL)
    {
        plcmaps_credential->voms_data_list = (lcmaps_vomsdata_t *) malloc (sizeof (lcmaps_vomsdata_t));
	/* lcmaps_vomsdata->nvoms >0 by now */
        plcmaps_credential->voms_data_list->voms = (lcmaps_voms_t *) malloc (sizeof (lcmaps_voms_t) * (size_t)(lcmaps_vomsdata->nvoms));

        for (i = 0; i < lcmaps_vomsdata->nvoms; i++)
        {
            plcmaps_credential->voms_data_list->nvoms                  = lcmaps_vomsdata->nvoms;

            plcmaps_credential->voms_data_list->voms[i].user_dn        = strdup(lcmaps_vomsdata->voms[i].user_dn);
            plcmaps_credential->voms_data_list->voms[i].user_ca        = strdup(lcmaps_vomsdata->voms[i].user_ca);
            plcmaps_credential->voms_data_list->voms[i].voms_issuer_dn = strdup(lcmaps_vomsdata->voms[i].voms_issuer_dn);
            plcmaps_credential->voms_data_list->voms[i].voms_issuer_ca = strdup(lcmaps_vomsdata->voms[i].voms_issuer_ca);
            plcmaps_credential->voms_data_list->voms[i].uri            = strdup(lcmaps_vomsdata->voms[i].uri);
            plcmaps_credential->voms_data_list->voms[i].date1          = strdup(lcmaps_vomsdata->voms[i].date1);
            plcmaps_credential->voms_data_list->voms[i].date2          = strdup(lcmaps_vomsdata->voms[i].date2);
            plcmaps_credential->voms_data_list->voms[i].voname         = strdup(lcmaps_vomsdata->voms[i].voname);

	    plcmaps_credential->voms_data_list->voms[i].nfqan          = lcmaps_vomsdata->voms[i].nfqan;
	    if (lcmaps_vomsdata->voms[i].nfqan > 0) {
		plcmaps_credential->voms_data_list->voms[i].fqan_unix      = (lcmaps_fqan_unix_t *) malloc (sizeof(lcmaps_fqan_unix_t) * (size_t)(lcmaps_vomsdata->voms[i].nfqan));

		for (j = 0; j < lcmaps_vomsdata->voms[i].nfqan; j++)
		{
		    plcmaps_credential->voms_data_list->voms[i].fqan_unix[j].fqan = strdup(lcmaps_vomsdata->voms[i].fqan_unix[j].fqan);
		    plcmaps_credential->voms_data_list->voms[i].fqan_unix[j].uid  = lcmaps_vomsdata->voms[i].fqan_unix[j].uid;
		    plcmaps_credential->voms_data_list->voms[i].fqan_unix[j].gid  = lcmaps_vomsdata->voms[i].fqan_unix[j].gid;
		}
	    } else
		plcmaps_credential->voms_data_list->voms[i].fqan_unix      = NULL;

            plcmaps_credential->voms_data_list->voms[i].nattr = lcmaps_vomsdata->voms[i].nattr;
	    if (lcmaps_vomsdata->voms[i].nattr > 0) {
		plcmaps_credential->voms_data_list->voms[i].attr_list = (lcmaps_voms_generic_attr_t *) calloc((size_t)(lcmaps_vomsdata->voms[i].nattr),sizeof(lcmaps_voms_generic_attr_t));

		lcmaps_log_debug (3, "-- total # of generic attributes in VO: (%d) \n", plcmaps_credential->voms_data_list->voms[i].nattr);
		for(j = 0; j < plcmaps_credential->voms_data_list->voms[i].nattr; j++){
		    plcmaps_credential->voms_data_list->voms[i].attr_list[j].name      = strdup(lcmaps_vomsdata->voms[i].attr_list[j].name);
		    plcmaps_credential->voms_data_list->voms[i].attr_list[j].value     = strdup(lcmaps_vomsdata->voms[i].attr_list[j].value);
		    plcmaps_credential->voms_data_list->voms[i].attr_list[j].qualifier = strdup(lcmaps_vomsdata->voms[i].attr_list[j].qualifier);
		}
	    } else
		plcmaps_credential->voms_data_list->voms[i].attr_list = NULL;


            plcmaps_credential->voms_data_list->workvo                 = strdup(lcmaps_vomsdata->workvo);
            plcmaps_credential->voms_data_list->extra_data             = strdup(lcmaps_vomsdata->extra_data);
        }
    }

    return LCMAPS_CRED_SUCCESS;
}


/******************************************************************************
Function:       lcmaps_credential_store_fqan_list()
Description:    Fill credential with list of FQANs
Parameters:
                nfqan:              number of elements of the FQAN list
                fqan_list:          FQAN list (list of strings)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_NO_FQAN: supplied fqan list is empty
                LCMAPS_CRED_ERROR: failure, because of a malloc error or a malformed fqan list
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_fqan_list(
        int                nfqan,
        char **            fqan_list,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    int     i    = 0;
    char *  fqan = NULL;
    int     retval = LCMAPS_CRED_SUCCESS;

    if (plcmaps_credential == NULL)
    {
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_fqan_list(): Create lcmaps_cred_id_t first!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }

    /* Check if the fqan list is empty */
    if ((fqan_list == NULL) || (nfqan < 1))
        return LCMAPS_CRED_NO_FQAN;

    /* Check if a FQAN list exists already. If so, return happily */
    if (plcmaps_credential->fqan == NULL)
    {
        plcmaps_credential->nfqan = nfqan;
	/* nfqan >= 1 */
        plcmaps_credential->fqan = (char **) malloc((size_t)nfqan * sizeof(char *));
        if (plcmaps_credential->fqan == NULL) /* malloc error */
        {
            lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_fqan_list(): Malloc error!\n");
            return LCMAPS_CRED_ERROR;
        }
        for (i = 0; i < nfqan; i++)
        {
            if ((fqan = fqan_list[i]) != NULL)
            {
                if ( ((plcmaps_credential->fqan)[i] = strdup(fqan)) == NULL)
                {
                    lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_fqan_list(): Malloc error!\n");
                    return LCMAPS_CRED_ERROR;
                }
            }
            else
            {
                lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_fqan_list(): malformed fqan list\n");
                return LCMAPS_CRED_ERROR;
            }
        }
    }
    return retval;
}

/******************************************************************************
Function:       lcmaps_credential_get_fqan_list()
Description:    returns the list of FQANs and number of elements
Parameters:
                pnfqan:            pointer to the number of FQANs in the list
                lcmaps_credential: lcmaps_credential
Returns:        list of FQANs
******************************************************************************/
char ** lcmaps_credential_get_fqan_list(
        int *            pnfqan,
        lcmaps_cred_id_t lcmaps_credential
)
{
    *pnfqan = lcmaps_credential.nfqan;
    return (lcmaps_credential.fqan);
}

/******************************************************************************
Function:       lcmaps_credential_store_mapcounter()
Description:    stores mapcounter
Parameters:
                mapcounter:         the mapping counter
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_mapcounter(
        int mapcounter,
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
    {
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_mapcounter(): Create lcmaps_cred_id_t first!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }
    plcmaps_credential->mapcounter = mapcounter;

    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
Function:       lcmaps_credential_get_mapcounter()
Description:    returns mapcounter
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        mapcounter
******************************************************************************/
int lcmaps_credential_get_mapcounter(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.mapcounter);
}

/******************************************************************************
Function:       lcmaps_credential_store_requested_account()
Description:    stores the requested account (the account that should be verified)
                in an lcmaps_account_info_t structure.
Parameters:
                puid:                pointer to the uid of the local account
                ppgid_list:          pointer to the list of primary gids
                pnpgid:              pointer to the number of primary gids found
                psgid_list:          pointer to the list of secondary gids
                pnsgid:              pointer to the number of secondary gids found
                ppoolindex:          pointer to the pool index
                plcmaps_credential:  pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR:   failure, because error filling lcmaps_account_info_t
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_requested_account(
        uid_t *                 puid,
        gid_t **                ppgid_list,
        int *                   pnpgid,
        gid_t **                psgid_list,
        int *                   pnsgid,
        char **                 ppoolindex,
        lcmaps_cred_id_t *      plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
    {
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_requested_account(): Create lcmaps_cred_id_t first!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }
    if (lcmaps_account_info_fill(
        puid,
        ppgid_list,
        pnpgid,
        psgid_list,
        pnsgid,
        ppoolindex,
        &(plcmaps_credential->requested_account))
    )
    {
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_store_requested_account(): Error storing requested account info\n");
        return LCMAPS_CRED_ERROR;
    }

    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
Function:       lcmaps_credential_get_requested_account()
Description:    returns requested account in lcmaps_account_info_t format
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        requested account
******************************************************************************/
lcmaps_account_info_t lcmaps_credential_get_requested_account(
        lcmaps_cred_id_t lcmaps_credential
)
{
    return (lcmaps_credential.requested_account);
}

/******************************************************************************
Function:       lcmaps_clean_list_of_strings()
Description:    clean list of strings
Parameters:
                listlen: number of elements in the list of strings
                string_list: list of strings
Returns:        LCMAPS_CRED_SUCCESS
******************************************************************************/
int lcmaps_clean_list_of_strings(
    int listlen, char ** string_list
)
{
    int i = 0;

    if (listlen > 0)
    {
        for (i = 0; i < listlen; i++)
        {
            if (string_list[i] != NULL) {
                free(string_list[i]);
                string_list[i]=NULL;
            }
        }
    }
    /* this removes the whole array of char* */
    if (string_list != NULL)
    {
        free(string_list);
        string_list=NULL;
    }
    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
Function:       lcmaps_credential_init()
Description:    initialize the lcmaps_cred_id_t: set the member values to
                the default values
Parameters:
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:  success
                LCMAPS_CRED_ERROR:  failed to set the member values
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_init(
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
    {
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_credential_init(): Create lcmaps_cred_id_t first!\n");
        return LCMAPS_CRED_INVOCATION_ERROR;
    }
#ifdef LCMAPS_GSI_MODE
    plcmaps_credential->cred         = GSS_C_NO_CREDENTIAL;
    plcmaps_credential->context      = GSS_C_NO_CONTEXT;
    /* It's in the planning to move this to the gsi-free */
    /* area, once we use the globus-free voms parser (openssl based) */
    plcmaps_credential->pem_string   = (char *) NULL;
    plcmaps_credential->px509_cred   = (X509 *) NULL;
    plcmaps_credential->px509_chain  = (STACK_OF(X509) *) NULL;
#endif
    plcmaps_credential->dn           = (char *) NULL;
    plcmaps_credential->fqan         = (char **) NULL;
    plcmaps_credential->nfqan        = -1;
    plcmaps_credential->mapcounter   = -1;
    if ( lcmaps_account_info_init(&(plcmaps_credential->requested_account)) )
    {
        lcmaps_log_debug(1, "lcmaps.mod-lcmaps_credential_init(): Error initializing \"requested account\"-info\n");
        return LCMAPS_CRED_ERROR;
    }
    return LCMAPS_CRED_SUCCESS;
}


/******************************************************************************
Function:       lcmaps_clean_vomsdata()
Description:    release vomsdata
Parameters:
                vomsdata: pointer to lcmaps_vomsdata_t
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_clean_vomsdata(lcmaps_vomsdata_t * vomsdata){
    int i,j;
    lcmaps_voms_t * ac_voms;

    if(vomsdata == NULL)
        return LCMAPS_CRED_SUCCESS;

    if(vomsdata->workvo) free(vomsdata->workvo);
    if(vomsdata->extra_data) free(vomsdata->extra_data);

    for (i = 0; i < vomsdata->nvoms; i++){

        ac_voms = (vomsdata->voms)+i;

        free(ac_voms->user_dn);		ac_voms->user_dn=NULL;
        free(ac_voms->user_ca);		ac_voms->user_ca=NULL;
        free(ac_voms->voms_issuer_dn);	ac_voms->voms_issuer_dn=NULL;
        free(ac_voms->voms_issuer_ca);	ac_voms->voms_issuer_ca=NULL;
        free(ac_voms->uri);		ac_voms->uri=NULL;
        free(ac_voms->date1);		ac_voms->date1=NULL;
        free(ac_voms->date2);		ac_voms->date2=NULL;
        free(ac_voms->voname);		ac_voms->voname=NULL;

        for (j = 0; j < ac_voms->nfqan; j++)
        {
            free(ac_voms->fqan_unix[j].fqan);
            ac_voms->fqan_unix[j].fqan=NULL;
        }
        free(ac_voms->fqan_unix);
        ac_voms->fqan_unix=NULL;

        for (j = 0; (j < ac_voms->nattr) && (ac_voms->attr_list); j++)
        {
            free(ac_voms->attr_list[j].name);
            ac_voms->attr_list[j].name=NULL;

            free(ac_voms->attr_list[j].value);
            ac_voms->attr_list[j].value=NULL;

            free(ac_voms->attr_list[j].qualifier);
            ac_voms->attr_list[j].qualifier=NULL;
        }
        free(ac_voms->attr_list);
        ac_voms->attr_list=NULL;

        /* XXX the attr_list members are declared const, so they cannot be free()'d */
        /*for(j=0; j < ac_voms->nattr; j++){
                free(ac_voms->attr_list[j].name);
                free(ac_voms->attr_list[j].value);
                free(ac_voms->attr_list[j].qualifier);
        }*/
        /* the following free is invalid, since you can free block compartimentalized when allocated as one chunk */
        /* free(ac_voms); */
    }

    free(vomsdata->voms);
    vomsdata->voms=NULL;
    free(vomsdata);
    vomsdata=NULL;

    return LCMAPS_CRED_SUCCESS;
}



/******************************************************************************
Function:       lcmaps_release_cred()
Description:    release lcmaps credential
Parameters:
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_release_cred(
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
        return LCMAPS_CRED_SUCCESS;

    free(plcmaps_credential->dn);

    lcmaps_clean_list_of_strings(plcmaps_credential->nfqan, plcmaps_credential->fqan);
    lcmaps_account_info_clean(&(plcmaps_credential->requested_account));
    lcmaps_clean_vomsdata(plcmaps_credential->voms_data_list);

#ifdef LCMAPS_GSI_MODE
    free(plcmaps_credential->pem_string);
    /* The px509_cred is since 25 januari a reference into the first element of the px509_chain
     * if (plcmaps_credential->px509_cred) X509_free(plcmaps_credential->px509_cred); */
    if (plcmaps_credential->px509_chain)
        lcmaps_x509_free_chain(&(plcmaps_credential->px509_chain));
#endif

    return LCMAPS_CRED_SUCCESS;
}


/******************************************************************************
Function:       lcmaps_release_cred_leave_STACK_OF_X509()
Description:    release lcmaps credential
Parameters:
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_release_cred_leave_STACK_OF_X509(
        lcmaps_cred_id_t * plcmaps_credential
)
{
    if (plcmaps_credential == NULL)
        return LCMAPS_CRED_SUCCESS;

    if (plcmaps_credential->dn != NULL)
        free(plcmaps_credential->dn);

    lcmaps_clean_list_of_strings(plcmaps_credential->nfqan, plcmaps_credential->fqan);
    lcmaps_account_info_clean(&(plcmaps_credential->requested_account));
    lcmaps_clean_vomsdata(plcmaps_credential->voms_data_list);

#ifdef LCMAPS_GSI_MODE
    /* The px509_cred is since 25 januari a reference into the first element of the px509_chain
     * if (plcmaps_credential->px509_cred) X509_free(plcmaps_credential->px509_cred); */
#if 0
    if (plcmaps_credential->px509_chain)
        lcmaps_x509_free_chain(&(plcmaps_credential->px509_chain));
#endif
#endif

    return LCMAPS_CRED_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/grid_credential_handling/lcmaps_credential.c,v $
    $Date: 2015-02-02 11:46:42 +0100 (Mon, 02 Feb 2015) $
    $Revision: 18216 $
    $Author: msalle $
******************************************************************************/
