/*
 *  Copyright 1994-2016 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include <inttypes.h>
#include "sequence.h"
#include "pictures.h"
#include "colormaps.h"
#include "globals.h"
#include "xmlutils.h"


static gint
Sequence_find_plugin(gconstpointer a, gconstpointer b)
{
  const Layer_t *la = (const Layer_t *)a;
  const Plugin_t *pb = (const Plugin_t *)b;

  assert(la != NULL);
  return (la->plugin == pb) ? 0 : 1;
}


GList *
Sequence_find(const Sequence_t *s, const Plugin_t *p)
{
  return g_list_find_custom(s->layers, (gconstpointer)p, &Sequence_find_plugin);
}


/* -1 if not found, position else */
short
Sequence_find_position(const Sequence_t *s, const Plugin_t *p)
{
  short pos = 0;
  GList *tmp;

  tmp = g_list_first(s->layers);
  while (tmp != NULL) {
    Layer_t *l = (Layer_t *)tmp->data;
    if (l->plugin == p)
      return pos;
    pos++;
    tmp = g_list_next(tmp);
  }

  return -1;
}


Sequence_t *
Sequence_new(const uint32_t id)
{
  Sequence_t *s = xcalloc(1, sizeof(Sequence_t));

  s->id = id;

  return s;
}


void
Sequence_clear(Sequence_t *s, const uint32_t new_id)
{
  GList *tmp;

  if (s == NULL)
    xerror("Attempt to Sequence_clear() a NULL sequence\n");

  s->id = new_id ? new_id : 0;
  
  if (NULL != s->name)
    xfree(s->name);
  
  tmp = g_list_first(s->layers);
  while (tmp != NULL) {
    Layer_t *l = (Layer_t *)tmp->data;
    Layer_delete(l);
    tmp = g_list_next(tmp);
  }
  g_list_free(s->layers);
  s->layers = NULL;
  
  s->lens = NULL;
  
  s->picture_id = 0;
  s->auto_pictures = 0;
  
  s->cmap_id = 0;
  s->auto_colormaps = 0;

  Sequence_changed(s);
}


void
Sequence_changed(Sequence_t *s)
{
  if (NULL != s)
    s->changed = 1;
}


void
Sequence_delete(Sequence_t *s)
{
  if (s != NULL) {
    GList *tmp;

    VERBOSE(printf("[s] Freeing sequence id %"PRIu32"\n", s->id));

    tmp = s->layers;
    while (tmp != NULL) {
      Layer_t *l = (Layer_t *)tmp->data;
      Layer_delete(l);
      tmp = g_list_next(tmp);
    }

    g_list_free(s->layers);
    if (s->name != NULL)
      g_free(s->name);
    xfree(s);
  } else
    xerror("Attempt to free a NULL Sequence\n");
}


/* TODO enhanced display: show id and pretty-print options --oliv3 */
void
Sequence_display(const Sequence_t *s)
{
  GList *tmp;
  const char *cmap_name;
  const char *pict_name;
  int after_lens = 0;

  if (s == NULL)
    xerror("Attempt to display a NULL Sequence\n");
  else {
    VERBOSE(printf("[s] Sequence id: %"PRIu32"\n", s->id));
  }

  VERBOSE(printf("[s] Name: %s\n", (s->name != NULL) ? s->name : "(none)"));

  if (pictures != NULL) {
    pict_name = (s->picture_id != 0) ? Pictures_name(s->picture_id) : "default";
    VERBOSE(printf("[s] Picture: %s\n", pict_name));
  }

  cmap_name = (s->cmap_id != 0) ? Colormaps_name(s->cmap_id) : "default";
  VERBOSE(printf("[s] Colormap: %s\n", cmap_name));

  for (tmp = g_list_first(s->layers); tmp != NULL; tmp = g_list_next(tmp)) {
    Layer_t *layer = (Layer_t *)tmp->data;
    Plugin_t *P = layer->plugin;
	  
    if (P != NULL) {
      if ((s->lens != NULL) && (P == s->lens)) {
	if (!after_lens) {
	  VERBOSE(printf("--- %s\n", P->name));
	  after_lens = 1;
	} else {
	  VERBOSE(printf("   %s\n", P->name));
	}
      } else {
	VERBOSE(printf(" |  %s\n", P->name));
      }
    } else
      xerror("Oops got a NULL plugin\n");
  }
}


void
Sequence_copy(const Sequence_t *from, Sequence_t *to)
{
  GList *tmp;
  
  to->id = from->id;
  
  if (to->name != NULL)
    xfree(to->name);
  
  if (from->name != NULL)
    to->name = strdup(from->name);

  tmp = g_list_first(to->layers);
  while (tmp != NULL) {
    Layer_t *l = (Layer_t *)tmp->data;

    Layer_delete(l);
    tmp = g_list_next(tmp);
  }
  g_list_free(to->layers);
  to->layers = NULL;
  
  tmp = g_list_first(from->layers);
  while (tmp != NULL) {
    Layer_t *lfrom = (Layer_t *)tmp->data;
    Layer_t *lto = Layer_copy(lfrom);

    to->layers = g_list_append(to->layers, (gpointer)lto);
    tmp = g_list_next(tmp);
  }

  to->lens = from->lens;


  to->picture_id = from->picture_id;
  to->auto_pictures = from->auto_pictures;

  to->cmap_id = from->cmap_id;
  to->auto_colormaps = from->auto_colormaps;
}


gint
Sequence_sort_func(gconstpointer a, gconstpointer b)
{
  const Sequence_t *sa = (const Sequence_t *)a;
  const Sequence_t *sb = (const Sequence_t *)b;

  assert(sa != NULL);
  assert(sb != NULL);

  return (sb->id - sa->id);
}


void
Sequence_insert(Sequence_t *s, Plugin_t *p) /* TODO LayerMode ? */
{
  Layer_t *layer = Layer_new(p);
  
  /* set layer mode */
  if (p->mode != NULL)
    layer->mode = (enum LayerMode)*p->mode;

  /* insert plugin in sequence */
  if (*p->options & BEQ_FIRST)
    s->layers = g_list_prepend(s->layers, (gpointer)layer);
  else
    s->layers = g_list_append(s->layers, (gpointer)layer);
  
  /* set as lens if it's a lens, and there is no lens yet */
  if ((*p->options & BE_LENS) && (s->lens == NULL))
    s->lens = (Plugin_t *)p;
  
  Sequence_changed(s);
}


void
Sequence_insert_at_position(Sequence_t *s, const u_short position, Plugin_t *p) /* TODO LayerMode ? */
{
  Layer_t *layer = Layer_new(p);
  
  /* set layer mode */
  if (p->mode != NULL)
    layer->mode = (enum LayerMode)*p->mode;

  /* insert plugin in sequence */
  s->layers = g_list_insert(s->layers, (gpointer)layer, position);
  
  /* set as lens if it's a lens, and there is no lens yet */
  if ((*p->options & BE_LENS) && (s->lens == NULL))
    s->lens = (Plugin_t *)p;
  
  Sequence_changed(s);
}


void
Sequence_remove(Sequence_t *s, const Plugin_t *p)
{
  const GList *p_layer = Sequence_find(s, p);
  Layer_t *layer = (Layer_t *)p_layer->data;

  /* remove plugin from sequence */
  Layer_delete(layer);

  s->layers = g_list_remove(s->layers, (gconstpointer)layer);
  
  /* unset if it is a lens we removed */
  if (s->lens == p)
    s->lens = NULL;
  
  Sequence_changed(s);
}


uint8_t
Sequence_size(const Sequence_t *seq)
{
  return g_list_length(seq->layers);
}

