/*
 * @(#)AnalysisModuleIO.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCode;

/**
 * Knows how to read and write an AnalysisModule object.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 15, 2002
 */
class AnalysisModuleIO
{
    public AnalysisModuleIO()
    {
        // do nothing
    }
    
    
    static class StaleAnalysisModule implements IAnalysisModule
    {
        private String name;
        private String unit;
        private String mime;
        
        public StaleAnalysisModule( String n, String u, String m )
        {
            this.name = n;
            this.unit = u;
            this.mime = m;
        }
        
        public String getMeasureName() { return this.name; }
        public String getMeasureUnit() { return this.unit; }
        public String getMimeEncoding() { return this.mime; }
        public void analyze( IMethodCode method )
        {
            throw new IllegalStateException(
                "this is a stale module for measure '"+this.name+"'" );
        }
    }
    
    public void writeAnalysisModule( IAnalysisModule ams, Writer out )
            throws IOException
    {
        StringBuffer sb = new StringBuffer();
        
        sb.append( ams.getMeasureName().length() ).
            append( ':' ).
            append( ams.getMeasureName() ).
            append( ',' ).
            append( ams.getMeasureUnit().length() ).
            append( ':' ).
            append( ams.getMeasureUnit() ).
            append( ',' ).
            append( ams.getMimeEncoding().length() ).
            append( ':' ).
            append( ams.getMimeEncoding() );
        
        out.write( sb.toString() );
    }
    
    
    /**
     * Does not read in the actual module, but a datastore that
     * resembles the original.
     */
    public IAnalysisModule readAnalysisModule( Reader in )
            throws IOException
    {
        int count = ReadUtil.toInt(
            ReadUtil.readTo( in, ':' ) );
        String measureName = ReadUtil.readCount( in, count );
        ReadUtil.readTo( in, ',' );
        
        count = ReadUtil.toInt(
            ReadUtil.readTo( in, ':' ) );
        String measureUnit = ReadUtil.readCount( in, count );
        ReadUtil.readTo( in, ',' );
        
        count = ReadUtil.toInt(
            ReadUtil.readTo( in, ':' ) );
        String mimeEncoding = ReadUtil.readCount( in, count );
        
        return new StaleAnalysisModule( measureName, measureUnit,
            mimeEncoding );
    }
}

