// Struct or union component list class -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include <string.h>
#include "ComponentList.h"
#include "util.h"
#include "Printer.h"

/** @file ComponentList.C
 * List of struct or union components
 */

/* Copyright  1998-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

ComponentList::ComponentList () :
  mySize (0), myNames (NULL), myTypes (NULL)
{
}

ComponentList::ComponentList (const class ComponentList& old) :
  mySize (old.mySize), myNames (NULL), myTypes (NULL)
{
  if (mySize) {
    myNames = new char*[mySize];
    for (card_t i = 0; i < mySize; i++)
      myNames[i] = newString (old.myNames[i]);

    myTypes = new const class Type*[mySize];
    memcpy (myTypes, old.myTypes, mySize * sizeof *myTypes);
  }
}

ComponentList::~ComponentList ()
{
  for (card_t i = 0; i < mySize; i++)
    delete[] myNames[i];
  delete[] myNames;
  delete[] myTypes;
}

const class Type*
ComponentList::operator[] (const char* name) const
{
  assert (name != NULL);

  // This is inefficient but rarely used
  for (card_t i = 0; i < mySize; i++)
    if (!strcmp (name, myNames[i]))
      return myTypes[i];
  return NULL;
}

card_t
ComponentList::getIndex (const char* name) const
{
  for (card_t i = 0; i < mySize; i++)
    if (!strcmp (name, myNames[i]))
      return i;

  assert (false);
  return 0;
}

bool
ComponentList::isAssignable (const class ComponentList& other) const
{
  if (&other == this)
    return true;

  if (other.getSize () != getSize ())
    return false;

  for (card_t i = 0; i < mySize; i++)
    if (strcmp (myNames[i], other.myNames[i]) ||
	!myTypes[i]->isAssignable (*other.myTypes[i]))
      return false;

  return true;
}

bool
ComponentList::isAlwaysAssignable (const class ComponentList& other) const
{
  if (&other == this)
    return true;

  if (other.getSize () != getSize ())
    return false;

  for (card_t i = 0; i < mySize; i++)
    if (strcmp (myNames[i], other.myNames[i]) ||
	!myTypes[i]->isAlwaysAssignable (*other.myTypes[i]))
      return false;

  return true;
}

bool
ComponentList::isAssignable (const class Type& type) const
{
  for (card_t i = 0; i < mySize; i++)
    if (myTypes[i]->isAssignable (type))
      return true;
  return false;
}

bool
ComponentList::isAssignableFrom (const class Type& type) const
{
  for (card_t i = 0; i < mySize; i++)
    if (type.isAssignable (*myTypes[i]))
      return true;
  return false;
}

bool
ComponentList::isAlwaysAssignableFrom (const class Type& type) const
{
  for (card_t i = 0; i < mySize; i++)
    if (type.isAlwaysAssignable (*myTypes[i]))
      return true;
  return false;
}

void
ComponentList::addComponent (char* name, const class Type& type)
{
  assert (!(*this)[name]);
  if (mySize) {
    char** names = new char*[mySize + 1];
    memcpy (names, myNames, mySize * sizeof *myNames);
    delete[] myNames;
    myNames = names;

    const class Type** types = new const class Type*[mySize + 1];
    memcpy (types, myTypes, mySize * sizeof *myTypes);
    delete[] myTypes;
    myTypes = types;
  }
  else {
    assert (!myNames);
    assert (!myTypes);
    myNames = new char*[1];
    myTypes = new const class Type*[1];
  }

  myNames[mySize] = name;
  myTypes[mySize] = &type;
  mySize++;
}

void
ComponentList::display (const class Printer& printer) const
{
  for (card_t i = 0; i < mySize; i++) {
    if (const char* name = myTypes[i]->getName ())
      printer.printRaw (name);
    else
      myTypes[i]->display (printer);
    printer.delimiter (' ');
    printer.print (myNames[i]);
    printer.delimiter (';');
    printer.linebreak ();
  }
}
