# -*-ruby-*-
#
# Copyright (c) 2010 SASADA Koichi <ko1 at atdot.net>
#
# This program is free software with ABSOLUTELY NO WARRANTY.
# You can re-distribute and/or modify this program under
# the same terms of the Ruby's license.
#

=begin

== Usage with irc client

  trans> hello
    -> translate hello as a English to Japanese.

  trans:ja> hello
    -> ditto.

  trans:en,ja> hello
    -> ditto.

  trans:(([lang_from],)[lang_to])> [phrase]
    -> translate [phrase] as lang_from to lang_to.

  transj> [phrase]
    -> translate to Japanese

  transe> [phrase]
    -> translate to English

== Configuration:

BotConfig = [
{
  :name                => :TranslateBot,
  :ch                  => /.*/,
  :referer             => 'http://rubyforge.org/projects/nadoka/',
  # Register URL at http://code.google.com/intl/ja/apis/ajaxsearch/signup.html
  # and set your URL to :referer and your API key to :api_key if you want.
  :to_lang             => 'ja',
  :to_lang2            => 'en',
  :ch_kcode            => :tojis,
},
]

=end

require 'iconv'
require 'kconv'
require 'shellwords'
require 'cgi'
require 'open-uri'
begin
  require 'json'
rescue LoadError
  require 'rubygems'
  require 'json'
end

class TranslateBot < Nadoka::NDK_Bot
  def bot_initialize
    @available_channel   = @bot_config[:ch] || /.*/
    @search_default_lang = (@bot_config[:search_default_lang] || 'ja').sub(/^lang_/, '')
    @referer = @bot_config[:referer] || 'http://rubyforge.org/projects/nadoka/'
    @ch_kcode = @bot_config.fetch(:ch_kcode, :tojis)
    @to_lang = @bot_config.fetch(:to_lang, 'ja')
    @to_lang2 = @bot_config.fetch(:to_lang2, 'en')
    @bing_appid = @bot_config.fetch(:bing_appid, nil)
  end

  def on_privmsg prefix, ch, msg
    if @available_channel === ch and /^tr/ =~ msg
      if response = dispatch_command(msg)
        response.each{|r|
          send_notice(ch, r) if r
        }
      end
    end
  end

  SHORT_LANG = {'e' => 'en', 'j' => 'ja'}

  def dispatch_command msg
    begin
      case msg
      when /^trans(?:late)?(:(.*?))?>\s*(.+)/o
        translate($3, *parse_trans_option($2, $3))
      when /^trans([ej])>\s*(.+)/o
        translate($2, nil, SHORT_LANG[$1])
      when /^trans(?:late)?r(:(.*?))?>\s*(.+)/o
        translate_r($3, *parse_trans_option($2, $3))
      when /^translang>(.+)/
        lang = $1.strip
        desc = 'Unknown.  See http://code.google.com/intl/ja/apis/ajaxlanguage/documentation/reference.html#LangNameArray'
        r = LANGUAGE_MAP_S2L.fetch(lang.downcase,
                                   LANGUAGE_MAP.fetch(lang.upcase, desc))
        "translang> #{lang} = #{r}"
      end
    rescue Exception => e
      "translate bot: #{e.message}"
    end
  end

  def detect_lang str
    uri = "http://ajax.googleapis.com/ajax/services/language/detect?v=1.0&q="
    uri << CGI.escape(str.toutf8)

    result = open(uri, "Referer" => @referer) do |f|
      JSON.parse(f.read)
    end

    if result["responseData"]
      result["responseData"]["language"]
    end
  end

  def parse_trans_option opt, str
    case opt
    when nil
      from_lang = detect_lang(str)
      to_lang = @to_lang
      if to_lang == from_lang
        to_lang = @to_lang2
      end 
      [from_lang, to_lang]
    when /\A([\w\-]+)[, \>]([\w\-]+)\z/
      [$1, $2]
    when /\A([\w\-]+)\z/
      [nil, $1]
    else
      raise "can't parse translation option: #{opt}"
    end
  end

  def translate phrase, from_lang, to_lang
    r = []

    gr = translate_ggl(phrase, from_lang, to_lang)
    r << "g:translate bot (#{gr[1]}): #{gr[0]}"

    if @bing_appid
      mr = translate_ms(phrase, from_lang, to_lang) if @bing_appid
      r <<  "m:translate bot (#{mr[1]}): #{mr[0]}"
    end
    r
  end

  def translate_r phrase, from_lang, to_lang
    rs = []
    %w(ggl ms).each{|system|
      r = send("translate_#{system}", phrase, from_lang, to_lang)
      from_lang = r[2]
      first = r[0]
      if from_lang
        r = send("translate_#{system}", r[0], to_lang, from_lang)
        rs << "#{system.split(//)[0]}:trans_r (#{from_lang}>#{to_lang}>#{from_lang}): #{r[0]} (#{first})"
      end
    }
    rs
  end


  def translate_ggl(phrase, from_lang, to_lang)
    uri = 'http://ajax.googleapis.com/ajax/services/language/translate?v=1.0&q='
    uri << CGI.escape(phrase.toutf8)
    uri << "&langpair=#{from_lang}%7C#{to_lang}"

    result = open(uri, "Referer" => @referer) do |f|
      JSON.parse(f.read)
    end

    if result["responseData"]
      text = CGI.unescapeHTML(result["responseData"]["translatedText"])
      text = text.send(@ch_kcode) if @ch_kcode
      from_lang ||= result["responseData"]["detectedSourceLanguage"]
      opts = "#{from_lang}>#{to_lang}"
      [text, opts, from_lang]
    else
      opts = "#{from_lang ? "from #{from_lang} to " : ''}#{to_lang}"
      ["#{result["responseDetails"]} (#{uri})", opts]
    end
  end

  ## ms translate

  def get_result_ms result
    # puts result
    doc = REXML::Document.new(result)
    doc.elements.map{|e| e.text}[0]
  end

  def translate_ms phrase, from_lang, to_lang
    api_url = 'http://api.microsofttranslator.com/V2/Http.svc/Translate'
    uri = "#{api_url}?appId=#{@bing_appid}&text=#{CGI.escape(phrase.toutf8)}&to=#{CGI.escape(to_lang)}"
    begin
      text = get_result_ms open(uri, "Referer" => @referer).read
      text = text.send(@ch_kcode) if @ch_kcode
      opts = "#{from_lang}>#{to_lang}"
      [text, opts, from_lang]
    rescue OpenURI::HTTPError => e
      opts = "#{from_lang ? "from #{from_lang} to " : ''}#{to_lang}"
      ["#{e.message} (uri)", opts]
    end
  end

# copy from http://code.google.com/intl/ja/apis/ajaxlanguage/documentation/reference.html
LANGUAGE_MAP = {
  'AFRIKAANS' => 'af',
  'ALBANIAN' => 'sq',
  'AMHARIC' => 'am',
  'ARABIC' => 'ar',
  'ARMENIAN' => 'hy',
  'AZERBAIJANI' => 'az',
  'BASQUE' => 'eu',
  'BELARUSIAN' => 'be',
  'BENGALI' => 'bn',
  'BIHARI' => 'bh',
  'BULGARIAN' => 'bg',
  'BURMESE' => 'my',
  'CATALAN' => 'ca',
  'CHEROKEE' => 'chr',
  'CHINESE' => 'zh',
  'CHINESE_SIMPLIFIED' => 'zh-CN',
  'CHINESE_TRADITIONAL' => 'zh-TW',
  'CROATIAN' => 'hr',
  'CZECH' => 'cs',
  'DANISH' => 'da',
  'DHIVEHI' => 'dv',
  'DUTCH'=> 'nl',  
  'ENGLISH' => 'en',
  'ESPERANTO' => 'eo',
  'ESTONIAN' => 'et',
  'FILIPINO' => 'tl',
  'FINNISH' => 'fi',
  'FRENCH' => 'fr',
  'GALICIAN' => 'gl',
  'GEORGIAN' => 'ka',
  'GERMAN' => 'de',
  'GREEK' => 'el',
  'GUARANI' => 'gn',
  'GUJARATI' => 'gu',
  'HEBREW' => 'iw',
  'HINDI' => 'hi',
  'HUNGARIAN' => 'hu',
  'ICELANDIC' => 'is',
  'INDONESIAN' => 'id',
  'INUKTITUT' => 'iu',
  'ITALIAN' => 'it',
  'JAPANESE' => 'ja',
  'KANNADA' => 'kn',
  'KAZAKH' => 'kk',
  'KHMER' => 'km',
  'KOREAN' => 'ko',
  'KURDISH'=> 'ku',
  'KYRGYZ'=> 'ky',
  'LAOTHIAN'=> 'lo',
  'LATVIAN' => 'lv',
  'LITHUANIAN' => 'lt',
  'MACEDONIAN' => 'mk',
  'MALAY' => 'ms',
  'MALAYALAM' => 'ml',
  'MALTESE' => 'mt',
  'MARATHI' => 'mr',
  'MONGOLIAN' => 'mn',
  'NEPALI' => 'ne',
  'NORWEGIAN' => 'no',
  'ORIYA' => 'or',
  'PASHTO' => 'ps',
  'PERSIAN' => 'fa',
  'POLISH' => 'pl',
  'PORTUGUESE' => 'pt-PT',
  'PUNJABI' => 'pa',
  'ROMANIAN' => 'ro',
  'RUSSIAN' => 'ru',
  'SANSKRIT' => 'sa',
  'SERBIAN' => 'sr',
  'SINDHI' => 'sd',
  'SINHALESE' => 'si',
  'SLOVAK' => 'sk',
  'SLOVENIAN' => 'sl',
  'SPANISH' => 'es',
  'SWAHILI' => 'sw',
  'SWEDISH' => 'sv',
  'TAJIK' => 'tg',
  'TAMIL' => 'ta',
  'TAGALOG' => 'tl',
  'TELUGU' => 'te',
  'THAI' => 'th',
  'TIBETAN' => 'bo',
  'TURKISH' => 'tr',
  'UKRAINIAN' => 'uk',
  'URDU' => 'ur',
  'UZBEK' => 'uz',
  'UIGHUR' => 'ug',
  'VIETNAMESE' => 'vi',
  'UNKNOWN' => ''
}

LANGUAGE_MAP_S2L = LANGUAGE_MAP.invert
end
