/*###############################################################################
# Linux Management Providers (LMP), CPU provider package
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr ,desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2009 Rakhimov Rustam, TUIT <rusyasoft@gmail.com>
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
###############################################################################*/

#include "OpenDRIM_ProcessorAccess.h"

string _SystemName;
vector<_processor_topology> _previous_cpu_info_processors_information;

int CPU_OpenDRIM_Processor_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CPU_getProcessors("/proc/cpuinfo", _previous_cpu_info_processors_information, errorMessage));
	CF_assert(CPU_getLoadAverages(_previous_cpu_info_processors_information, _previous_cpu_info_processors_information, errorMessage));
	
	CF_assert(CF_getSystemName(_SystemName, errorMessage));
	/*
	int errorCode=CF_getMachineIP(_SystemName, errorMessage);
	if (errorCode!=OK)
		return errorCode;
	*/
	_L_;
	return OK;
}

int CPU_OpenDRIM_Processor_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int CPU_OpenDRIM_Processor_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_Processor>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	// get info from SMBIOS
	vector<_processor_information> SMBIOS_processors_information;
	int errorCode = SMBIOS_getProcessorsInformation(SMBIOS_processors_information, errorMessage);
	// if we cannot get info from SMBIOS, we'll just use info from cpuinfo
	bool use_SMBIOS = false;
	if (errorCode == OK)
		use_SMBIOS = true;
	
	// get info from cpuinfo
	vector<_processor_topology> cpu_info_processors_information;
	CF_assert(CPU_getProcessors("/proc/cpuinfo", cpu_info_processors_information, errorMessage));
	CF_assert(CPU_getLoadAverages(cpu_info_processors_information, _previous_cpu_info_processors_information, errorMessage));
	_previous_cpu_info_processors_information = cpu_info_processors_information;
	
	// different processors count...
	if (cpu_info_processors_information.size() != SMBIOS_processors_information.size())
		use_SMBIOS = false;
	
	for (unsigned int i = 0; i < cpu_info_processors_information.size(); i++) {
		OpenDRIM_Processor instance;
		instance.setSystemName(_SystemName);
		instance.setSystemCreationClassName("OpenDRIM_ComputerSystem");
		instance.setCreationClassName("OpenDRIM_Processor");
		instance.setDeviceID(CF_intToStr(i));
		if (discriminant == "ei") {
			_processor_information SMBIOS_processor_information;
			if (use_SMBIOS)
				SMBIOS_processor_information = SMBIOS_processors_information[i];
			CF_assert(CPU_OpenDRIM_Processor_populate(instance, cpu_info_processors_information[i], SMBIOS_processor_information, use_SMBIOS, errorMessage));
		}
		result.push_back(instance);
	}
	_L_;
	return OK;
}

int CPU_OpenDRIM_Processor_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_Processor& instance, const char** properties, string& errorMessage) {
	_E_;
	string SystemName, SystemCreationClassName, CreationClassName, DeviceID;
	instance.getSystemName(SystemName);
	instance.getSystemCreationClassName(SystemCreationClassName);
	instance.getCreationClassName(CreationClassName);
	instance.getDeviceID(DeviceID);
	if (!CF_strCmpNoCase(SystemName, _SystemName) || !CF_strCmpNoCase(SystemCreationClassName, "OpenDRIM_ComputerSystem") || !CF_strCmpNoCase(CreationClassName, "OpenDRIM_Processor"))
		return NOT_FOUND;
	if (!CF_isNumber(DeviceID))
		return NOT_FOUND;
	
	// get info from cpuinfo
	vector<_processor_topology> cpu_info_processors_information;
	CF_assert(CPU_getProcessors("/proc/cpuinfo", cpu_info_processors_information, errorMessage));
	CF_assert(CPU_getLoadAverages(cpu_info_processors_information, _previous_cpu_info_processors_information, errorMessage));
	_previous_cpu_info_processors_information = cpu_info_processors_information;
	
	unsigned int _DeviceID = atoll(DeviceID.c_str());
	if (_DeviceID >= cpu_info_processors_information.size())
		return NOT_FOUND;
	
	// get info from SMBIOS
	vector<_processor_information> SMBIOS_processors_information;
	int errorCode = SMBIOS_getProcessorsInformation(SMBIOS_processors_information, errorMessage);
	// if we cannot get info from SMBIOS, we'll just use info from cpuinfo
	bool use_SMBIOS = false;
	if (errorCode == OK)
		use_SMBIOS = true;
	
	// different processors count...
	if (cpu_info_processors_information.size() != SMBIOS_processors_information.size())
		use_SMBIOS = false;
	
	_processor_information SMBIOS_processor_information;
	if (use_SMBIOS)
		SMBIOS_processor_information = SMBIOS_processors_information[_DeviceID];
	
	CF_assert(CPU_OpenDRIM_Processor_populate(instance, cpu_info_processors_information[_DeviceID], SMBIOS_processor_information, use_SMBIOS, errorMessage));
	_L_;
	return OK;
}

int CPU_OpenDRIM_Processor_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& newInstance, const OpenDRIM_Processor& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, const OpenDRIM_Processor_RequestStateChange_In& in, OpenDRIM_Processor_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_SetPowerState(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, const OpenDRIM_Processor_SetPowerState_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_Reset(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_EnableDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, const OpenDRIM_Processor_EnableDevice_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_OnlineDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, const OpenDRIM_Processor_OnlineDevice_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_QuiesceDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, const OpenDRIM_Processor_QuiesceDevice_In& in, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_SaveProperties(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_RestoreProperties(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Processor& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_Processor_populate(OpenDRIM_Processor& instance, const _processor_topology& cpu_info_processor_information, const _processor_information& SMBIOS_processor_information, bool use_SMBIOS, string& errorMessage) {
	_E_;
	
	/*
	 * Properties to fill from profile
	 * + Mandatory:
	 * [X] SystemCreationClassName
	 * [X] SystemName
	 * [X] CreationClassName
	 * [X] DeviceID
	 * [X] Family
	 * [X] CurrentClockSpeed
	 * [X] MaxClockSpeed
	 * [X] ExternalBusClockSpeed
	 * [X] CPUStatus
	 * [X] EnabledState
	 * [X] RequestedState
	 * [X] OperationalStatus
	 * [X] HealthState
	 * [X] ElementName
	 * + Optional:
	 * [X] LoadPercentage
	 */
	
	// default settings in case SMBIOS didn't give results
	instance.setFamily(2); // unknown
	instance.setExternalBusClockSpeed(0);
	instance.setCPUStatus(0); // unknown
	instance.setUpgradeMethod(2); // unknown
	instance.setNumberOfEnabledCores(1); // default 1 core
	
	// set SMBIOS info first
	if (use_SMBIOS)
	{
		instance.setFamily(SMBIOS_processor_information.processor_family);
		instance.setExternalBusClockSpeed(SMBIOS_processor_information.external_clock);
		instance.setCPUStatus(SMBIOS_processor_information.status);
		instance.setUpgradeMethod(SMBIOS_processor_information.processor_upgrade);
		instance.setCharacteristics(SMBIOS_processor_information.processor_characteristics);
		
		if (SMBIOS_processor_information.core_enabled != 0)
			instance.setNumberOfEnabledCores(SMBIOS_processor_information.core_enabled);
		
		instance.setExternalBusClockSpeed(SMBIOS_processor_information.external_clock);
		instance.setMaxClockSpeed(SMBIOS_processor_information.max_speed);
		instance.setCurrentClockSpeed(SMBIOS_processor_information.current_speed);
	}
	
	// we don't support RequestedStateChange()
	instance.setRequestedState(12);
	
	if (instance.CPUStatus == 0) {
		// Unknown
		instance.setEnabledState(0); // unknown
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(0); // unknown
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(0); // unknown
	} else if (instance.CPUStatus == 1) {
		// CPU Enabled
		instance.setEnabledState(2); // enabled
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(2); // OK
		OperationalStatus.push_back(11); // in service
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(5); // OK
	} else if (instance.CPUStatus == 2) {
		// CPU Disabled by User via BIOS Setup
		instance.setEnabledState(3); // disabled
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(2); // OK
		OperationalStatus.push_back(10); // stopped
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(5); // OK
	} else if (instance.CPUStatus == 3) {
		// CPU Disabled By BIOS (POST Error)
		instance.setEnabledState(3); // disabled
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(6); // error
		OperationalStatus.push_back(10); // stopped
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(25); // critical failure
	} else if (instance.CPUStatus == 4) {
		// CPU is Idle
		instance.setEnabledState(6); // enabled but offline
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(2); // OK
		OperationalStatus.push_back(15); // dormant
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(5); // OK
	} else if (instance.CPUStatus == 7) {
		// Other
		instance.setEnabledState(1); // other
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(0); // unknown
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(0); // unknown
	} else {
		// default
		instance.setEnabledState(0); // unknown
		vector<unsigned short> OperationalStatus;
		OperationalStatus.push_back(0); // unknown
		instance.setOperationalStatus(OperationalStatus);
		instance.setHealthState(0); // unknown
	}
	
	// set /proc/cpuinfo info now...
	instance.setElementName(cpu_info_processor_information.name);
	instance.setDescription(cpu_info_processor_information.vendor + " " + cpu_info_processor_information.architecture + " family " + CF_intToStr(cpu_info_processor_information.family) + " model " + CF_intToStr(cpu_info_processor_information.model) + " stepping " + CF_intToStr(cpu_info_processor_information.stepping));
	
	// load percentage
	instance.setLoadPercentage(cpu_info_processor_information.load_average);
	
	_L_;
	return OK;
}
