/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "LexerBase_p.h"

#include "String.h"
#include "Token_p.h"

using namespace GTLCore;

struct LexerBase::Private {
  std::istream* stream;
  int col;
  int line;
  int followingnewline;
};

LexerBase::LexerBase(std::istream* sstream) : d(new Private)
{
  d->col = 1;
  d->line = 1;
  d->followingnewline = 1;
  d->stream = sstream;
}

LexerBase::~LexerBase()
{
  delete d;
}

int LexerBase::getNextNonSeparatorChar()
{
  int lastChar = ' ';
  while( not eof() and isspace(lastChar = getNextChar() )  )
  { // Ignore space
  }
  return lastChar;
}

int LexerBase::getNextChar()
{
  int nc = d->stream->get();
  if( nc == '\n' )
  {
    ++d->line;
    ++d->followingnewline;
    d->col = 1;
  } else {
    ++d->col;
    d->followingnewline = 0;
  }
  return nc;
}

void LexerBase::unget()
{
  --d->col;
  d->stream->unget();
  if( d->followingnewline > 0 )
  {
    --d->followingnewline;
    --d->line;
  }
}

bool LexerBase::eof() const
{
  return d->stream->eof() or d->stream->fail() or d->stream->bad();
}

int LexerBase::line() const
{
  return d->line;
}
int LexerBase::column() const
{
  return d->col;
}

bool LexerBase::ignoreComment(GTLCore::Token& _token, int _lastChar )
{
  if( _lastChar == '/' )
  {
    int initial_line = line();
    int initial_col = column();
    int nextChar = getNextChar();
    if( nextChar == '/' )
    { // Mono line comment
      while( not eof() and getNextChar() != '\n' )
      {
      }
      _token = nextToken();
      return true;
    } else if( nextChar == '*' )
    { // Multi line comment
      while( not eof() )
      {
        int nextChar = getNextChar();
        if( nextChar == '*' )
        {
          if( getNextChar() == '/' )
          {
            _token = nextToken();
            return true;
          } else {
            unget();
          }
        }
      }
      _token = GTLCore::Token(GTLCore::Token::UNFINISHED_COMMENT, initial_line, initial_col);
      return true;
    } else {
      unget();
    }
  }
  return false;
}

GTLCore::String LexerBase::getIdentifier(int lastChar)
{
  GTLCore::String identifierStr;
  if( lastChar != 0) {
    identifierStr = lastChar;
  }
  while (not eof() )
  {
    lastChar = getNextChar();
    if( isalnum(lastChar) or lastChar == '_')
    {
      identifierStr += lastChar;
    } else {
      unget();
      break;
    }
  }
  return identifierStr;
}

GTLCore::Token LexerBase::getDigit(int lastChar)
{
  int initial_col = column();
  GTLCore::String identifierStr;
  identifierStr= lastChar;
  bool integer = true;
  bool exponent = false;
  while (not eof()
         and ( isdigit((lastChar = getNextChar()))
               or lastChar == '.' or (lastChar == 'e' and not exponent )
               or (exponent and lastChar =='-' ) ) )
  {
    identifierStr += lastChar;
    if( lastChar == '.' )
    {
      integer = false;
    }
    if( lastChar == 'e' )
    {
      integer = false;
      exponent = true;
    }
  }
  unget();
  if(integer)
  {
    return GTLCore::Token( identifierStr.toInt(), line(), initial_col );
  } else {
    return GTLCore::Token( identifierStr.toFloat(), line(), initial_col );
  }
}

GTLCore::Token LexerBase::getString(int lastChar)
{
  int initial_col = column();
  int previousChar = lastChar;
  GTLCore::String identifierStr = "";
  while( not eof() )
  {
    int nextChar = getNextChar();
    if( nextChar == '"' and previousChar != '\\')
    {
      return GTLCore::Token( GTLCore::Token::STRING_CONSTANT, identifierStr, line(), initial_col );
    } else {
      previousChar = nextChar;
      identifierStr += nextChar;
    }
  }
  return Token( Token::UNFINISHED_STRING, line(), initial_col);
}
