/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Library.h"
#include "Library_p.h"

#include <sstream>

#include <llvm/Module.h>
#include <llvm/Support/DynamicLibrary.h>
#include <llvm/LLVMContext.h>

#include "GTLCore/CompilationMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Region.h"
#include "GTLCore/Type.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/Metadata/Group.h"
#include "GTLCore/Metadata/ParameterEntry.h"
#include "Metadata.h"
#include "GTLCore/Metadata/Factory_p.h"
#include <GTLCore/LLVMBackend/ContextManager_p.h>

#include "GTLCore/Debug.h"
#include "Compiler_p.h"

#include "LibraryCompilation_p.h"
#include "Wrapper_p.h"

using namespace GTLFragment;

void Library::Private::metadataToParameters( const GTLCore::Metadata::Group* group)
{
  GTL_ASSERT( group );
  foreach( const GTLCore::Metadata::Entry* entry, group->entries() )
  {
    if( const GTLCore::Metadata::ParameterEntry* pe = entry->asParameterEntry() )
    {
      createParameter(pe->name(), pe->defaultValue());
    } else if( const GTLCore::Metadata::Group* ge = entry->asGroup() )
    {
      metadataToParameters( ge );
    }
  }
}

void Library::Private::initparameters()
{
  const Metadata* metadata = source.metadata();
  parameters_info.clear();
  parameters_name2id.clear();
  if( metadata and metadata->parameters() )
  {
    metadataToParameters( metadata->parameters() );
  }
  GTL_ASSERT(parameters_info.size() == parameters_name2id.size());
}

int Library::Private::createParameter(GTLCore::String _name, GTLCore::Value _value)
{
  int id = parameters_name2id.size();
  parameters_name2id[ _name ] = id;
  parameters_info.resize(id + 1);
  parameters_info[ id ].value = _value;
  parameters_info[ id ].name = _name;
  return id;
}

// -------- Library -------- //

Library::Library( Type _mode , int _channelsNb) : d(new Private)
{
  d->name = "";
  d->compiled = false;
  d->m_llvmModule = 0;
  d->m_moduleData = 0;
  d->count_channels_generic = _channelsNb;
  d->mode = _mode;
}

Library::~Library()
{
  cleanup();
  delete d;
}

void Library::cleanup()
{
  TEST_GUARD_WRITE(d)
  if(d->m_moduleData and d->m_moduleData->llvmLinkedModule())
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->m_moduleData->llvmLinkedModule());
  }
  delete d->m_moduleData;
  d->m_llvmModule = 0;
  d->m_moduleData = 0;
  if( d->libraryCompilation )
  {
    d->libraryCompilation->cleanup();
  }
}


GTLCore::String Library::name() const
{
  return d->name;
}

void Library::setSource(const GTLCore::String& _source )
{
  d->source.setSource(_source);
  d->initparameters();
}

void Library::setSource(const Source& source)
{
    d->source = source;
    d->initparameters();
}

Source Library::source() const
{
    return d->source;
}

void Library::loadFromFile(const GTLCore::String& _fileName)
{
  if( _fileName.endWith( "stdlib.fragment" ) ) {
    d->standardLibrary = FragmentStdLibrary;
  } else if( _fileName.endWith( "rijnstdlib.rijn" ) ) {
    d->standardLibrary = RijnStdLibrary;
  } else if( _fileName.endWith( "rijnwrappers.rijn" ) ) {
    d->standardLibrary = RijnWrappersLibrary;
  } else if( _fileName.endWith( "shivawrappers.shiva" ) ) {
    d->standardLibrary = ShivaWrappersLibrary;
  }
  d->source.loadFromFile(_fileName);
  d->initparameters();
}

void Library::compile()
{
  TEST_GUARD_WRITE(d)
  if( d->libraryCompilation )
  {
    d->libraryCompilation->preCompilation();
  }
  if(not d->source.metadata() )
  {
    d->compilationErrors = d->source.metadataCompilationMessages();
    return;
  }
  if(d->source.source().isEmpty()) return;
  cleanup();
  llvm::LLVMContext& context = *LLVMBackend::ContextManager::context(); // TODO someday (llvm 2.7 ?) use own context
  d->m_llvmModule = new llvm::Module((const std::string&)d->name, context);
  d->m_moduleData = new GTLCore::ModuleData(d->m_llvmModule);
  Compiler c( d->mode,  d->count_channels_generic );
  if( d->libraryCompilation )
  {
    d->libraryCompilation->createWrapper(d->m_moduleData, d->m_llvmModule, d->count_channels_generic, d->standardLibrary);
    GTL_ASSERT(d->libraryCompilation->wrapper());
    d->libraryCompilation->wrapper()->fillTypesManager( d->m_moduleData->typesManager(), c.convertCenter() );
    d->libraryCompilation->addOperatorOverloads(c.operatorOverloadCenter());
  } else {
    Wrapper::createColorType(d->m_moduleData->typesManager(), c.convertCenter());
  }
  GTLCore::String nameSpace;
  bool result = c.compile( d->standardLibrary, d->source.source(), d->name, d->m_moduleData, d->m_llvmModule, nameSpace, d->parameters_info );

  if(result)
  {
    d->m_llvmModule = 0;
    d->compiled = true;
    // Register the compiled module in the virtual machine
    llvm::sys::DynamicLibrary::LoadLibraryPermanently( _GTLFRAGMENT_LIB_, 0 ); // This needed because when OpenShiva is used in a plugins, OpenShiva symbols aren't loaded globally and then llvm can't find them (FIXME move that somewhere else)
    d->m_moduleData->doLink();
    GTLCore::VirtualMachine::instance()->registerModule( d->m_moduleData->llvmLinkedModule() );
    d->name = nameSpace;
    if( d->libraryCompilation )
    {
      d->libraryCompilation->postCompilation();
    }
    GTL_ASSERT( d->standardLibrary != NotAStdLibrary or d->name != "" );
  } else {
    d->compiled = false;
    cleanup();
    d->compilationErrors = c.errorMessages();
  }
  
}

bool Library::isCompiled() const
{
  return d->compiled;
}

const GTLCore::CompilationMessages& Library::compilationMessages() const
{
  return d->compilationErrors;
}

GTLCore::String Library::asmSourceCode() const
{
  TEST_GUARD_READ(d)
  std::ostringstream os;
  os << *d->m_moduleData->llvmModule() << std::endl;
  return os.str();
}

const GTLCore::ModuleData* Library::data() const
{
  TEST_GUARD_READ(d)
  return d->m_moduleData;
}

std::list<GTLCore::Function*> Library::functions()
{
  TEST_GUARD_READ(d)
  return d->m_moduleData->functions();
}

const Metadata* Library::metadata() const
{
  return d->source.metadata();
}
