///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/DataSet.h>
#include <core/data/DataSetManager.h>
#include <core/data/ObjectSaveStream.h>
#include <core/data/ObjectLoadStream.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportConfiguration.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/SelectionSet.h>
#include <core/scene/animation/AnimManager.h>
#include <core/reference/CloneHelper.h>
#include <core/rendering/RenderSettings.h>

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(DataSet, RefTarget)
DEFINE_FLAGS_REFERENCE_FIELD(DataSet, ViewportConfiguration, "ViewportConfiguration", PROPERTY_FIELD_NO_CHANGE_MESSAGE|PROPERTY_FIELD_ALWAYS_DEEP_COPY, _viewportConfig)
DEFINE_FLAGS_REFERENCE_FIELD(DataSet, AnimationSettings, "AnimationSettings", PROPERTY_FIELD_NO_CHANGE_MESSAGE|PROPERTY_FIELD_ALWAYS_DEEP_COPY, _animSettings)
DEFINE_FLAGS_REFERENCE_FIELD(DataSet, SceneRoot, "SceneRoot", PROPERTY_FIELD_NO_CHANGE_MESSAGE|PROPERTY_FIELD_ALWAYS_DEEP_COPY, _sceneRoot)
DEFINE_FLAGS_REFERENCE_FIELD(DataSet, SelectionSet, "CurrentSelection", PROPERTY_FIELD_NO_CHANGE_MESSAGE|PROPERTY_FIELD_ALWAYS_DEEP_COPY, _selection)
DEFINE_FLAGS_REFERENCE_FIELD(DataSet, RenderSettings, "RenderSettings", PROPERTY_FIELD_NO_CHANGE_MESSAGE|PROPERTY_FIELD_ALWAYS_DEEP_COPY, _renderSettings)
SET_PROPERTY_FIELD_LABEL(DataSet, _viewportConfig, "Viewport Configuration")
SET_PROPERTY_FIELD_LABEL(DataSet, _animSettings, "Animation Settings")
SET_PROPERTY_FIELD_LABEL(DataSet, _sceneRoot, "Scene")
SET_PROPERTY_FIELD_LABEL(DataSet, _selection, "Selection")
SET_PROPERTY_FIELD_LABEL(DataSet, _renderSettings, "Render Settings")

DataSet::DataSet(bool isLoading) :
	RefTarget(isLoading), _hasBeenChanged(false)
{
	INIT_PROPERTY_FIELD(DataSet, _viewportConfig);
	INIT_PROPERTY_FIELD(DataSet, _animSettings);
	INIT_PROPERTY_FIELD(DataSet, _sceneRoot);
	INIT_PROPERTY_FIELD(DataSet, _selection);
	INIT_PROPERTY_FIELD(DataSet, _renderSettings);

	if(!isLoading) {
		// Create a new viewport configuration by copying the default template.
		ViewportConfiguration::SmartPtr defaultViewConfig = VIEWPORT_MANAGER.defaultConfiguration();
		_viewportConfig = CloneHelper().cloneObject(defaultViewConfig, true);

		_animSettings = new AnimationSettings();
		_sceneRoot = new SceneRoot();
		_selection = new SelectionSet();
		_renderSettings = new RenderSettings();
	}
}


bool DataSet::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(msg->type() == REFTARGET_CHANGED) {
		// Set dirty flag.
		setDirty();

		// Update all viewports when something has changed in the current data set.
		if(this == DATASET_MANAGER.currentSet()) {
			VIEWPORT_MANAGER.updateViewports();
		}
	}
	else if(msg->type() == CUSTOM_ATTRIBUTE_CHANGED) {
		// Set dirty flag.
		setDirty();
	}
	return RefTarget::onRefTargetMessage(source, msg);
}

void DataSet::saveToStream(ObjectSaveStream& stream)
{
	// Acquire viewport configuration first.
	if(DATASET_MANAGER.currentSet() == this)
		viewportConfig()->saveConfiguration();

	RefTarget::saveToStream(stream);

	stream.beginChunk(0x33672);
	stream.endChunk();
}

void DataSet::loadFromStream(ObjectLoadStream& stream)
{
	RefTarget::loadFromStream(stream);
	OVITO_ASSERT(_sceneRoot);
	OVITO_ASSERT(_animSettings);

	stream.expectChunk(0x33672);
	stream.closeChunk();
}

void DataSet::rescaleTime(const TimeInterval& oldAnimationInterval, const TimeInterval& newAnimationInterval)
{
	// Iterate over all controllers in the scene.
	Q_FOREACH(RefTarget* reftarget, getAllDependencies()) {
		Controller* ctrl = dynamic_object_cast<Controller>(reftarget);
		if(!ctrl) continue;
		ctrl->rescaleTime(oldAnimationInterval, newAnimationInterval);
	}
}

void DataSet::clearScene()
{
	Q_FOREACH(SceneNode* node, sceneRoot()->children()) {
		node->deleteNode();
	}
}

};
