///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/NativePlugin.h>
#include <core/utilities/PathManager.h>
#include <core/reference/NativePropertyFieldDescriptor.h>

namespace Core {

/******************************************************************************
* Constructor for the NativePlugin class.
******************************************************************************/
NativePlugin::NativePlugin(const QString& manifestFile) :
	Plugin(manifestFile), _library(NULL), _isScriptable(false),
	infoBefore(NULL), infoAfter(NULL)
{
}


/******************************************************************************
* Parses a custom toplevel element from the manifest that is specific to the plugin type.
******************************************************************************/
bool NativePlugin::parseToplevelManifestElement(const QDomElement& element)
{
	// Process the <NativePlugin> element that describes the native plugin's characteristics.
	if(element.localName() == "NativePlugin") {

		// Get the name of the shared library file.
		QString libBasename = element.attribute("Library");
		// Resolve the filename by adding the platform specific suffix/extension
		// and make the path absolute.
		QDir baseDir;
		if(isCore())	// The core library is not in the plugins directory.
			baseDir = PATH_MANAGER.libraryDirectory();
		else
			baseDir = QFileInfo(manifestFile()).dir();
		QFileInfo libFile(baseDir.absoluteFilePath(libBasename));
#if defined(Q_CC_MINGW) || defined(Q_WS_X11)
		libFile.setFile(libFile.absolutePath() + "/lib" + libFile.fileName());
#endif
		_libraryFilename = QDir::cleanPath(libFile.absoluteFilePath());

		// Parse the scripting flag.
		if(element.attribute("IsScriptable").compare("true", Qt::CaseInsensitive) == 0 ||
			element.attribute("IsScriptable").compare("on", Qt::CaseInsensitive) == 0 ||
			element.attribute("IsScriptable").compare("yes", Qt::CaseInsensitive) == 0)
			_isScriptable = true;

		return true;
	}
	return false;
}

/******************************************************************************
* Loads a native plugin's library.
******************************************************************************/
void NativePlugin::loadPluginImpl()
{
#ifndef OVITO_MONOLITHIC_BUILD
	if(infoBefore == NULL)
		infoBefore = NativePluginClassInfo::_firstInfo;
	if(isCore()) infoBefore = NULL;

	// Load dynamic library.
	if(_library == NULL || _library->isLoaded() == false) {
		if(libraryFilename().isEmpty())
			throw Exception(QString("The manifest file of the native plugin %1 does not specify the library name.").arg(pluginId()));
		VerboseLogger() << logdate << "Loading native plugin library:" << libraryFilename() << endl;
		_library = new QLibrary(libraryFilename(), this);
		if(!_library->load()) {
			infoBefore = NULL;
			throw Exception(QString("Failed to load native plugin library.\nLibrary file: %1\nError: %2").arg(libraryFilename(), _library->errorString()));
		}
	}
	if(infoAfter == NULL)
		infoAfter = NativePluginClassInfo::_firstInfo;
#else
	infoBefore = NULL;
	infoAfter = NativePluginClassInfo::_firstInfo;
#endif

	// Connect the class infos with the descriptors.
	Q_FOREACH(PluginClassDescriptor* descriptor, classes()) {
		CHECK_POINTER(descriptor);
		NativePluginClassDescriptor* nativeDescriptor = (NativePluginClassDescriptor*)descriptor;

		// Find the class info for this descriptor.
		for(NativePluginClassInfo* info = infoAfter; info != infoBefore; info = info->next) {
			if(descriptor->name() == info->className()) {
				nativeDescriptor->associateWithClassInfo(info);
				break;
			}
		}

		if(nativeDescriptor->classInfo == NULL)
			throw Exception(QString("Plugin class %1 was defined in the manifest of plugin %2 but was not found in the library %3").arg(nativeDescriptor->name(), pluginId(), libraryFilename()));

		// Check abstract status.
		if(nativeDescriptor->isAbstract() != (nativeDescriptor->classInfo->ctor == NULL && nativeDescriptor->classInfo->singletonInstance == NULL))
			throw Exception(QString("Plugin class '%1' has wrong abstract status in plugin library:\n\n%2").arg(nativeDescriptor->name(), libraryFilename()));

		// Check base class.
		NativePluginClassDescriptor* nativeBaseDescriptor = (NativePluginClassDescriptor*)descriptor->baseClass();
		if(nativeBaseDescriptor != NULL && nativeBaseDescriptor->classInfo->classInfo !=
			nativeDescriptor->classInfo->classInfo->superClass()) {
			const char* nativeBaseClassName = nativeDescriptor->classInfo->classInfo->superClass() ? nativeDescriptor->classInfo->classInfo->superClass()->className() : "None";
			throw Exception(QString("Base class of plugin class '%1' does not match base class defined in plugin manifest:\nManifest file: %2\nBase class in manifest file: %3\nActual base class: %4").arg(nativeDescriptor->name(), manifestFile(), nativeBaseDescriptor->name(), QString(nativeBaseClassName)));
		}
	}

#ifndef OVITO_MONOLITHIC_BUILD
	// Make sure that each class in the plugin has been defined in the manifest.
	for(NativePluginClassInfo* info = infoAfter; info != infoBefore; info = info->next) {
		if(info->_descriptor == NULL)
			throw Exception(QString("Plugin class '%1' has not been defined in plugin manifest:\n%2").arg(info->className(), manifestFile()));
	}
#endif

	// Resolve the property field descriptor classes.
	Q_FOREACH(PluginClassDescriptor* descriptor, classes()) {
		for(const PropertyFieldDescriptor* field = descriptor->firstPropertyField(); field; field = field->next()) {
			((NativePropertyFieldDescriptor*)field)->resolveClassReferences();
		}
	}
}

};
