///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include "CylinderObject.h"
#include "CylinderCreationMode.h"

namespace StdObjects {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CylinderObject, SimpleGeometryObject)
DEFINE_REFERENCE_FIELD(CylinderObject, FloatController, "Radius", _radius)
DEFINE_REFERENCE_FIELD(CylinderObject, FloatController, "Height", _height)
DEFINE_PROPERTY_FIELD(CylinderObject, "RadiusSegments", _radiusSegments)
DEFINE_PROPERTY_FIELD(CylinderObject, "HeightSegments", _heightSegments)
DEFINE_PROPERTY_FIELD(CylinderObject, "SmoothFaces", _smoothFaces)
SET_PROPERTY_FIELD_LABEL(CylinderObject, _radius, "Radius")
SET_PROPERTY_FIELD_LABEL(CylinderObject, _height, "Height")
SET_PROPERTY_FIELD_LABEL(CylinderObject, _radiusSegments, "Radius segments")
SET_PROPERTY_FIELD_LABEL(CylinderObject, _heightSegments, "Height segments")
SET_PROPERTY_FIELD_LABEL(CylinderObject, _smoothFaces, "Smooth faces")
SET_PROPERTY_FIELD_UNITS(CylinderObject, _radius, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(CylinderObject, _height, WorldParameterUnit)

IMPLEMENT_PLUGIN_CLASS(CylinderObjectEditor, PropertiesEditor)
IMPLEMENT_PLUGIN_CLASS(CylinderCreationMode, SimpleCreationMode)

/******************************************************************************
* Constructs a new cylinder object.
******************************************************************************/
CylinderObject::CylinderObject(bool isLoading) : SimpleGeometryObject(isLoading),
	_radiusSegments(32), _heightSegments(1), _smoothFaces(true)
{
	INIT_PROPERTY_FIELD(CylinderObject, _radius);
	INIT_PROPERTY_FIELD(CylinderObject, _height);
	INIT_PROPERTY_FIELD(CylinderObject, _radiusSegments);
	INIT_PROPERTY_FIELD(CylinderObject, _heightSegments);
	INIT_PROPERTY_FIELD(CylinderObject, _smoothFaces);
	if(!isLoading) {
		_radius = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_height = CONTROLLER_MANAGER.createDefaultController<FloatController>();
	}
}

/******************************************************************************
* Asks the object for its validity interval at the given time.
******************************************************************************/
TimeInterval CylinderObject::objectValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	_radius->validityInterval(time, interval);
	_height->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Builds the mesh representation of this geometry object.
******************************************************************************/
void CylinderObject::buildMesh(TimeTicks time, TriMesh& mesh, TimeInterval& meshValidity)
{
	// Reset mesh
	meshValidity.setInfinite();

	// Query parameters
	FloatType radius=0, height=0;
	_radius->getValue(time, radius, meshValidity);
	_height->getValue(time, height, meshValidity);
	int radiusSegments = max(3, (int)_radiusSegments);
	int heightSegments = max(1, (int)_heightSegments);
	bool smoothFaces = _smoothFaces;
	quint32 smGroup = smoothFaces ? 1 : 0;
	bool flipFaces = height < 0;

	// Build vertices.
	mesh.setVertexCount(radiusSegments * (heightSegments + 1) + 2);
	QVector<Point3>::iterator vertex = mesh.vertices().begin();
	*vertex++ = ORIGIN;
	*vertex++ = Point3(0, 0, height);
	for(int a = 0; a < radiusSegments; a++) {
		FloatType angle = (FloatType)a * FLOATTYPE_PI * 2.0 / (FloatType)radiusSegments;
		FloatType x = radius * cos(angle);
		FloatType y = radius * sin(angle);
		for(int h = 0; h <= heightSegments; h++, ++vertex) {
			FloatType z = (FloatType)h * height / (FloatType)heightSegments;
			*vertex = Point3(x, y, z);
		}
	}

	// Build faces.
	mesh.setFaceCount(2 * radiusSegments * (heightSegments + 1));
	QVector<TriMeshFace>::iterator face = mesh.faces().begin();
	for(int a = 0; a < radiusSegments; a++) {
		int vertexBase1 = a * (heightSegments + 1) + 2;
		int vertexBase2 = (a < radiusSegments-1) ? (vertexBase1 + (heightSegments + 1)) : 2;

		// Create lower cap face
		if(!flipFaces)
			face->setVertices(vertexBase2, vertexBase1, 0);
		else
			face->setVertices(vertexBase1, vertexBase2, 0);
		face->setEdgeVisibility(true, false, false);
		face->setSmoothingGroup(0);
		face++;

		for(int h = 0; h < heightSegments; h++, vertexBase1++, vertexBase2++) {

			// Create first triangle
			if(!flipFaces)
				face->setVertices(vertexBase1, vertexBase2, vertexBase1 + 1);
			else
				face->setVertices(vertexBase1, vertexBase1 + 1, vertexBase2);
			face->setEdgeVisibility(true, false, true);
			face->setSmoothingGroup(smGroup);
			face++;

			// Create second triangle
			if(!flipFaces)
				face->setVertices(vertexBase1 + 1, vertexBase2, vertexBase2 + 1);
			else
				face->setVertices(vertexBase2, vertexBase1 + 1, vertexBase2 + 1);
			face->setEdgeVisibility(false, true, true);
			face->setSmoothingGroup(smGroup);
			face++;
		}

		// Create upper cap face
		if(!flipFaces)
			face->setVertices(vertexBase1, vertexBase2, 1);
		else
			face->setVertices(vertexBase2, vertexBase1, 1);
		face->setEdgeVisibility(true, false, false);
		face->setSmoothingGroup(0);
		face++;
	}

	mesh.invalidateVertices();
	mesh.invalidateFaces();
}

/******************************************************************************
* Creates the UI controls for the editor.
******************************************************************************/
void CylinderObjectEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Cylinder"), rolloutParams);

	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Radius parameter.
	FloatControllerUI* radiusPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(CylinderObject, _radius));
	layout->addWidget(radiusPUI->label(), 0, 0);
	layout->addWidget(radiusPUI->textBox(), 0, 1);
	layout->addWidget(radiusPUI->spinner(), 0, 2);
	radiusPUI->setMinValue(0);

	// Height parameter.
	FloatControllerUI* heightPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(CylinderObject, _height));
	layout->addWidget(heightPUI->label(), 2, 0);
	layout->addWidget(heightPUI->textBox(), 2, 1);
	layout->addWidget(heightPUI->spinner(), 2, 2);

	// Radius segments parameter.
	IntegerPropertyUI* radiusSegmentsPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CylinderObject, _radiusSegments));
	layout->addWidget(radiusSegmentsPUI->label(), 3, 0);
	layout->addWidget(radiusSegmentsPUI->textBox(), 3, 1);
	layout->addWidget(radiusSegmentsPUI->spinner(), 3, 2);
	radiusSegmentsPUI->setMinValue(3);

	// Height segments parameter.
	IntegerPropertyUI* heightSegmentsPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CylinderObject, _heightSegments));
	layout->addWidget(heightSegmentsPUI->label(), 4, 0);
	layout->addWidget(heightSegmentsPUI->textBox(), 4, 1);
	layout->addWidget(heightSegmentsPUI->spinner(), 4, 2);
	heightSegmentsPUI->setMinValue(1);

	// Smooth faces parameter.
	BooleanPropertyUI* smoothFacesPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CylinderObject, _smoothFaces));
	layout->addWidget(smoothFacesPUI->checkBox(), 5, 0, 1, 3);
}

};
