/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 */

#if !defined(SIMDE_ARM_NEON_QABS_H)
#define SIMDE_ARM_NEON_QABS_H

#include "types.h"

#include "abs.h"

HEDLEY_DIAGNOSTIC_PUSH
SIMDE_DISABLE_UNWANTED_DIAGNOSTICS
SIMDE_BEGIN_DECLS_

SIMDE_FUNCTION_ATTRIBUTES
int8_t simde_vqabsb_s8(int8_t a)
{
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vqabsb_s8(a);
#else
    return a == INT8_MIN ? INT8_MAX : (a < 0 ? -a : a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqabsb_s8
#define vqabsb_s8(a) simde_vqabsb_s8(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
int16_t simde_vqabsh_s16(int16_t a)
{
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vqabsh_s16(a);
#else
    return a == INT16_MIN ? INT16_MAX : (a < 0 ? -a : a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqabsh_s16
#define vqabsh_s16(a) simde_vqabsh_s16(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
int32_t simde_vqabss_s32(int32_t a)
{
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vqabss_s32(a);
#else
    return a == INT32_MIN ? INT32_MAX : (a < 0 ? -a : a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqabss_s32
#define vqabss_s32(a) simde_vqabss_s32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
int64_t simde_vqabsd_s64(int64_t a)
{
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vqabsd_s64(a);
#else
    return a == INT64_MIN ? INT64_MAX : (a < 0 ? -a : a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqabsd_s64
#define vqabsd_s64(a) simde_vqabsd_s64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int8x8_t simde_vqabs_s8(simde_int8x8_t a)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqabs_s8(a);
#else
    simde_int8x8_private r_, a_ = simde_int8x8_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabsb_s8(a_.values[i]);
    }

    return simde_int8x8_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqabs_s8
#define vqabs_s8(a) simde_vqabs_s8(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int16x4_t simde_vqabs_s16(simde_int16x4_t a)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqabs_s16(a);
#else
    simde_int16x4_private r_, a_ = simde_int16x4_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabsh_s16(a_.values[i]);
    }

    return simde_int16x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqabs_s16
#define vqabs_s16(a) simde_vqabs_s16(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x2_t simde_vqabs_s32(simde_int32x2_t a)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqabs_s32(a);
#else
    simde_int32x2_private r_, a_ = simde_int32x2_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabss_s32(a_.values[i]);
    }

    return simde_int32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqabs_s32
#define vqabs_s32(a) simde_vqabs_s32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x1_t simde_vqabs_s64(simde_int64x1_t a)
{
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vqabs_s64(a);
#else
    simde_int64x1_private r_, a_ = simde_int64x1_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabsd_s64(a_.values[i]);
    }

    return simde_int64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqabs_s64
#define vqabs_s64(a) simde_vqabs_s64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int8x16_t simde_vqabsq_s8(simde_int8x16_t a)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqabsq_s8(a);
#else
    simde_int8x16_private r_, a_ = simde_int8x16_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabsb_s8(a_.values[i]);
    }

    return simde_int8x16_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqabsq_s8
#define vqabsq_s8(a) simde_vqabsq_s8(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int16x8_t simde_vqabsq_s16(simde_int16x8_t a)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqabsq_s16(a);
#else
    simde_int16x8_private r_, a_ = simde_int16x8_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabsh_s16(a_.values[i]);
    }

    return simde_int16x8_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqabsq_s16
#define vqabsq_s16(a) simde_vqabsq_s16(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x4_t simde_vqabsq_s32(simde_int32x4_t a)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqabsq_s32(a);
#else
    simde_int32x4_private r_, a_ = simde_int32x4_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabss_s32(a_.values[i]);
    }

    return simde_int32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqabsq_s32
#define vqabsq_s32(a) simde_vqabsq_s32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x2_t simde_vqabsq_s64(simde_int64x2_t a)
{
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vqabsq_s64(a);
#else
    simde_int64x2_private r_, a_ = simde_int64x2_to_private(a);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqabsd_s64(a_.values[i]);
    }

    return simde_int64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqabsq_s64
#define vqabsq_s64(a) simde_vqabsq_s64(a)
#endif

SIMDE_END_DECLS_
HEDLEY_DIAGNOSTIC_POP

#endif /* !defined(SIMDE_ARM_NEON_QABS_H) */
