#!/bin/bash
#
# switch.sh
# Brings up a GUI for switching between
# Nvidia (discrete) and Intel (integrated) GPUs
#
# Copyright (c) 2020 Rui Guilherme da Cunha
# Copyright (c) 2022 Erich Eickmeyer
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

_switchGraphicsFn () {
  _uuid=$(uuidgen);
  touch "/tmp/plasma-optimus-${_uuid}";

  if (pkexec plasma-prime-select "$1" "${_uuid}" |"${_zenityExeStr}" \
    --window-icon="/usr/share/pixmaps/plasma-optimus-${_icon}.png" \
    --title "Plasma Optimus" \
    --progress --text "Please wait while your graphics system is configured..."\
    --pulsate --no-cancel --auto-close); then
    qdbus org.kde.ksmserver /KSMServer logout 0 1 0;
  else
    exit 1;
  fi
}

_logoutDialogFn () {
  _questionTitle="Plasma Optimus"
  _questionText=$(cat <<EOF
<html>
  <style>
    table{width:100%;margin:0}
    td{padding:4px 16px 4px 0;}
  </style>
  <table align="center">
    <tr><td width="200px">
      <p align="center">Switching To</p>
      <h3 style="color:#1d99f3;text-align:center">${_gpuStr}</h3>
      <p>You must reboot to apply the new GPU settings.
      <br>Make sure to save all work before continuing.</p>
      <p>Do you wish to reboot now?</p>
    </td><td width="72px" align="center">
      <img src='/usr/share/pixmaps/plasma-optimus-${_icon}.png'></img>
    </td></tr>
  </table>
</html>
EOF
  );

  # We use kdialog here because it supports HTML and images
  if "${_kdiagExeStr}" \
    --icon "/usr/share/pixmaps/plasma-optimus-${_icon}.png" \
    --title "${_questionTitle}" --yesno "${_questionText}"; then
      _switchGraphicsFn "$1";
  fi
}

## BEGIN MAIN
if [ $# -lt 1 ]; then exit 1; fi
_option="${1}";

_intelStr="Intel GPU (Power Save Mode)";
_nvidiaStr="Nvidia GPU (Performance Mode)";
_hybridStr="Hybrid GPU (On-Demand Mode)<br><br>Not Recommended if Using<br>External Monitors";
_kdiagExeStr="$(command -v kdialog)";
_zenityExeStr="$(command -v zenity)";

# Bail if kdialog is not usable (e.g. bad environment, broken deps)
if [ -z "${_kdiagExeStr}" ] || [ ! -x "${_kdiagExeStr}" ]; then
  1>&2 echo 'kdialog is not executable.
Please ensure it is installed and in your PATH.';
  exit 1;
fi

# Bail if zenity is not usable as above
if [ -z "${_zenityExeStr}" ] || [ ! -x "${_kdiagExeStr}" ]; then
  1>&2 echo 'zenity is not executable.
Please ensure it is installed and in your PATH.';
  exit 2;
fi

if [ "${_option}" = "nvidia" ]; then
  _gpuStr="${_nvidiaStr}"
  _icon="nvidia"
  _logoutDialogFn nvidia
  exit 0;
elif [ "${_option}" = "intel" ]; then
  _gpuStr="${_intelStr}"
  _icon="intel"
  _logoutDialogFn intel
  exit 0;
elif [ "${_option}" = "on-demand" ]; then
  _gpuStr="${_hybridStr}"
  _icon="hybrid"
  _logoutDialogFn on-demand
  exit 0;
else
  exit 1;
fi
## . END MAIN
