#  holidays
#  --------
#  A fast, efficient Python library for generating country, province and state
#  specific sets of holidays on the fly. It aims to make determining whether a
#  specific date is a holiday as fast and flexible as possible.
#
#  Authors: Vacanza Team and individual contributors (see AUTHORS file)
#           dr-prodigy <dr.prodigy.github@gmail.com> (c) 2017-2023
#           ryanss <ryanssdev@icloud.com> (c) 2014-2017
#  Website: https://github.com/vacanza/holidays
#  License: MIT (see LICENSE file)

from unittest import TestCase

from holidays.countries.iran import Iran, IR, IRN
from tests.common import CommonCountryTests


class TestIran(CommonCountryTests, TestCase):
    @classmethod
    def setUpClass(cls):
        super().setUpClass(Iran)

    def test_country_aliases(self):
        self.assertAliases(Iran, IR, IRN)

    def test_no_holidays(self):
        self.assertNoHolidays(Iran(years=1979))
        self.assertNoHolidays(Iran(years=2102))

    def test_2021(self):
        self.assertHolidays(
            ("2021-01-17", "شهادت حضرت فاطمه زهرا سلام الله علیها"),
            ("2021-02-10", "پیروزی انقلاب اسلامی"),
            ("2021-02-25", "ولادت امام علی علیه السلام و روز پدر"),
            ("2021-03-11", "مبعث رسول اکرم (ص)"),
            ("2021-03-19", "روز ملی شدن صنعت نفت ایران"),
            ("2021-03-20", "آخرین روز سال"),
            ("2021-03-21", "جشن نوروز"),
            ("2021-03-22", "عیدنوروز"),
            ("2021-03-23", "عیدنوروز"),
            ("2021-03-24", "عیدنوروز"),
            ("2021-03-29", "ولادت حضرت قائم عجل الله تعالی فرجه و جشن نیمه شعبان"),
            ("2021-04-01", "روز جمهوری اسلامی"),
            ("2021-04-02", "روز طبیعت"),
            ("2021-05-04", "شهادت حضرت علی علیه السلام"),
            ("2021-05-13", "عید سعید فطر"),
            ("2021-05-14", "تعطیل به مناسبت عید سعید فطر"),
            ("2021-06-04", "رحلت حضرت امام خمینی"),
            ("2021-06-05", "قیام 15 خرداد"),
            ("2021-06-06", "شهادت امام جعفر صادق علیه السلام"),
            ("2021-07-21", "عید سعید قربان"),
            ("2021-07-29", "عید سعید غدیر خم"),
            ("2021-08-18", "تاسوعای حسینی"),
            ("2021-08-19", "عاشورای حسینی"),
            ("2021-09-27", "اربعین حسینی"),
            ("2021-10-05", "رحلت رسول اکرم؛شهادت امام حسن مجتبی علیه السلام"),
            ("2021-10-07", "شهادت امام رضا علیه السلام"),
            ("2021-10-15", "شهادت امام حسن عسکری علیه السلام"),
            ("2021-10-24", "میلاد رسول اکرم و امام جعفر صادق علیه السلام"),
        )

    def test_2022(self):
        self.assertHolidays(
            ("2022-01-06", "شهادت حضرت فاطمه زهرا سلام الله علیها"),
            ("2022-02-11", "پیروزی انقلاب اسلامی"),
            ("2022-02-15", "ولادت امام علی علیه السلام و روز پدر"),
            ("2022-03-01", "مبعث رسول اکرم (ص)"),
            ("2022-03-18", "ولادت حضرت قائم عجل الله تعالی فرجه و جشن نیمه شعبان"),
            ("2022-03-20", "روز ملی شدن صنعت نفت ایران"),
            ("2022-03-21", "جشن نوروز"),
            ("2022-03-22", "عیدنوروز"),
            ("2022-03-23", "عیدنوروز"),
            ("2022-03-24", "عیدنوروز"),
            ("2022-04-01", "روز جمهوری اسلامی"),
            ("2022-04-02", "روز طبیعت"),
            ("2022-04-23", "شهادت حضرت علی علیه السلام"),
            ("2022-05-03", "عید سعید فطر"),
            ("2022-05-04", "تعطیل به مناسبت عید سعید فطر"),
            ("2022-05-27", "شهادت امام جعفر صادق علیه السلام"),
            ("2022-06-04", "رحلت حضرت امام خمینی"),
            ("2022-06-05", "قیام 15 خرداد"),
            ("2022-07-10", "عید سعید قربان"),
            ("2022-07-18", "عید سعید غدیر خم"),
            ("2022-08-07", "تاسوعای حسینی"),
            ("2022-08-08", "عاشورای حسینی"),
            ("2022-09-17", "اربعین حسینی"),
            ("2022-09-25", "رحلت رسول اکرم؛شهادت امام حسن مجتبی علیه السلام"),
            ("2022-09-27", "شهادت امام رضا علیه السلام"),
            ("2022-10-05", "شهادت امام حسن عسکری علیه السلام"),
            ("2022-10-14", "میلاد رسول اکرم و امام جعفر صادق علیه السلام"),
            ("2022-12-27", "شهادت حضرت فاطمه زهرا سلام الله علیها"),
        )

    def test_l10n_default(self):
        self.assertLocalizedHolidays(
            ("2023-02-04", "ولادت امام علی علیه السلام و روز پدر"),
            ("2023-02-11", "پیروزی انقلاب اسلامی"),
            ("2023-02-18", "مبعث رسول اکرم (ص)"),
            ("2023-03-08", "ولادت حضرت قائم عجل الله تعالی فرجه و جشن نیمه شعبان"),
            ("2023-03-20", "روز ملی شدن صنعت نفت ایران"),
            ("2023-03-21", "جشن نوروز"),
            ("2023-03-22", "عیدنوروز"),
            ("2023-03-23", "عیدنوروز"),
            ("2023-03-24", "عیدنوروز"),
            ("2023-04-01", "روز جمهوری اسلامی"),
            ("2023-04-02", "روز طبیعت"),
            ("2023-04-12", "شهادت حضرت علی علیه السلام"),
            ("2023-04-22", "عید سعید فطر"),
            ("2023-04-23", "تعطیل به مناسبت عید سعید فطر"),
            ("2023-05-16", "شهادت امام جعفر صادق علیه السلام"),
            ("2023-06-04", "رحلت حضرت امام خمینی"),
            ("2023-06-05", "قیام 15 خرداد"),
            ("2023-06-29", "عید سعید قربان"),
            ("2023-07-07", "عید سعید غدیر خم"),
            ("2023-07-27", "تاسوعای حسینی"),
            ("2023-07-28", "عاشورای حسینی"),
            ("2023-09-06", "اربعین حسینی"),
            ("2023-09-14", "رحلت رسول اکرم؛شهادت امام حسن مجتبی علیه السلام"),
            ("2023-09-16", "شهادت امام رضا علیه السلام"),
            ("2023-09-24", "شهادت امام حسن عسکری علیه السلام"),
            ("2023-10-03", "میلاد رسول اکرم و امام جعفر صادق علیه السلام"),
            ("2023-12-17", "شهادت حضرت فاطمه زهرا سلام الله علیها"),
        )

    def test_l10n_en_us(self):
        self.assertLocalizedHolidays(
            "en_US",
            ("2023-02-04", "Birthday of Imam Ali"),
            ("2023-02-11", "Islamic Revolution Day"),
            ("2023-02-18", "Isra' and Mi'raj"),
            ("2023-03-08", "Birthday of Mahdi"),
            ("2023-03-20", "Iranian Oil Industry Nationalization Day"),
            ("2023-03-21", "Nowruz"),
            ("2023-03-22", "Nowruz Holiday"),
            ("2023-03-23", "Nowruz Holiday"),
            ("2023-03-24", "Nowruz Holiday"),
            ("2023-04-01", "Islamic Republic Day"),
            ("2023-04-02", "Nature's Day"),
            ("2023-04-12", "Martyrdom of Imam Ali"),
            ("2023-04-22", "Eid al-Fitr"),
            ("2023-04-23", "Eid al-Fitr Holiday"),
            ("2023-05-16", "Martyrdom of Imam Ja'far al-Sadiq"),
            ("2023-06-04", "Death of Imam Khomeini"),
            ("2023-06-05", "15 Khordad Uprising"),
            ("2023-06-29", "Eid al-Adha"),
            ("2023-07-07", "Eid al-Ghadeer"),
            ("2023-07-27", "Tasua"),
            ("2023-07-28", "Ashura"),
            ("2023-09-06", "Arbaeen"),
            ("2023-09-14", "Death of Prophet Muhammad and Martyrdom of Hasan ibn Ali"),
            ("2023-09-16", "Martyrdom of Ali al-Rida"),
            ("2023-09-24", "Martyrdom of Hasan al-Askari"),
            ("2023-10-03", "Birthday of Muhammad and Imam Ja'far al-Sadiq"),
            ("2023-12-17", "Martyrdom of Fatima"),
        )
