// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/signin/profile_creation_customize_themes_handler.h"

#include "chrome/browser/search/chrome_colors/chrome_colors_service.h"
#include "chrome/common/search/generated_colors_info.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "ui/base/l10n/l10n_util.h"

ProfileCreationCustomizeThemesHandler::ProfileCreationCustomizeThemesHandler(
    mojo::PendingRemote<customize_themes::mojom::CustomizeThemesClient>
        pending_client,
    mojo::PendingReceiver<customize_themes::mojom::CustomizeThemesHandler>
        pending_handler)
    : remote_client_(std::move(pending_client)),
      receiver_(this, std::move(pending_handler)) {}

ProfileCreationCustomizeThemesHandler::
    ~ProfileCreationCustomizeThemesHandler() = default;

void ProfileCreationCustomizeThemesHandler::ApplyAutogeneratedTheme(
    SkColor frame_color) {
  chrome_colors::ChromeColorsService::RecordColorAppliedHistogram(
      chrome_colors::kOtherColorId);
  auto theme = customize_themes::mojom::Theme::New();
  theme->type = customize_themes::mojom::ThemeType::kAutogenerated;
  auto theme_colors = customize_themes::mojom::ThemeColors::New();
  auto colors = GetAutogeneratedThemeColors(frame_color);
  theme_colors->frame = colors.frame_color;
  theme_colors->active_tab = colors.active_tab_color;
  theme_colors->active_tab_text = colors.active_tab_text_color;
  theme->info = customize_themes::mojom::ThemeInfo::NewAutogeneratedThemeColors(
      std::move(theme_colors));
  remote_client_->SetTheme(std::move(theme));
}

void ProfileCreationCustomizeThemesHandler::ApplyDefaultTheme() {
  chrome_colors::ChromeColorsService::RecordColorAppliedHistogram(
      chrome_colors::kDefaultColorId);
  auto theme = customize_themes::mojom::Theme::New();
  theme->type = customize_themes::mojom::ThemeType::kDefault;
  theme->info = customize_themes::mojom::ThemeInfo::NewChromeThemeId(
      chrome_colors::kDefaultColorId);
  remote_client_->SetTheme(std::move(theme));
}

void ProfileCreationCustomizeThemesHandler::ApplyChromeTheme(int32_t id) {
  chrome_colors::ChromeColorsService::RecordColorAppliedHistogram(id);
  auto theme = customize_themes::mojom::Theme::New();
  theme->type = customize_themes::mojom::ThemeType::kChrome;
  theme->info = customize_themes::mojom::ThemeInfo::NewChromeThemeId(id);
  remote_client_->SetTheme(std::move(theme));
}

void ProfileCreationCustomizeThemesHandler::InitializeTheme() {
  // Do nothing.
  // The profile picker initializes the theme from JavaScript.
}

void ProfileCreationCustomizeThemesHandler::GetChromeThemes(
    GetChromeThemesCallback callback) {
  std::vector<customize_themes::mojom::ChromeThemePtr> themes;
  for (const auto& color_info : chrome_colors::kGeneratedColorsInfo) {
    auto theme_colors = GetAutogeneratedThemeColors(color_info.color);
    auto theme = customize_themes::mojom::ChromeTheme::New();
    theme->id = color_info.id;
    theme->label = l10n_util::GetStringUTF8(color_info.label_id);
    auto colors = customize_themes::mojom::ThemeColors::New();
    colors->frame = theme_colors.frame_color;
    colors->active_tab = theme_colors.active_tab_color;
    colors->active_tab_text = theme_colors.active_tab_text_color;
    theme->colors = std::move(colors);
    themes.push_back(std::move(theme));
  }
  std::move(callback).Run(std::move(themes));
}

void ProfileCreationCustomizeThemesHandler::ConfirmThemeChanges() {}

void ProfileCreationCustomizeThemesHandler::RevertThemeChanges() {}
