/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtAudioSettings.h"

#include "ui_AudioSettings.h"

#include <model/config/ConfigManager.h>
#include <model/config/Config.h>
#include <model/profile/UserProfile.h>
#include <model/phoneline/IPhoneLine.h>

#include <control/CQuteCom.h>
#include <control/profile/CUserProfile.h>
#include <control/profile/CUserProfileHandler.h>

#include <presentation/qt/QtQuteCom.h>

#include <sound/AudioDeviceManager.h>
#include <sound/Sound.h>
#include <qtutil/StringListConvert.h>
#include <qtutil/SafeConnect.h>
#include <util/SafeDelete.h>

#include <QtGui/QtGui>

static AudioDevice getAudioDeviceFromComboBox(QComboBox* comboBox, const std::list<AudioDevice> deviceList) {
	std::string concatString(comboBox->itemData(comboBox->currentIndex()).toString().toUtf8());
	for (std::list<AudioDevice>::const_iterator it = deviceList.begin();
		it != deviceList.end();
		++it) {
		if (it->getData().toString() == concatString) {
			return *it;
		}
	}
	return AudioDevice();
}

QtAudioSettings::QtAudioSettings(CQuteCom & cQuteCom, QWidget * parent)
	: QWidget(parent),
	_cQuteCom(cQuteCom) {

	_ui = new Ui::AudioSettings();
	_ui->setupUi(this);

	SAFE_CONNECT(_ui->makeTestCallButton, SIGNAL(pressed()), SLOT(makeTestCallClicked()));
	SAFE_CONNECT(_ui->testOutputDeviceButton, SIGNAL(pressed()), SLOT(testOutputDevice()));
	SAFE_CONNECT(_ui->testRingingDeviceButton, SIGNAL(pressed()), SLOT(testRingingDevice()));
	SAFE_CONNECT(_ui->AdvButton, SIGNAL(pressed()), SLOT(AdvPressed()));
	SAFE_CONNECT(_ui->UpButton, SIGNAL(pressed()), SLOT(UpPressed()));
	SAFE_CONNECT(_ui->DownButton, SIGNAL(pressed()), SLOT(DownPressed()));
	SAFE_CONNECT(_ui->ResetButton, SIGNAL(pressed()), SLOT(ResetPressed()));
	_ui->AdvGroupBox->setVisible(false);

	readConfig();
}

QtAudioSettings::~QtAudioSettings() {
	OWSAFE_DELETE(_ui);
}

QString QtAudioSettings::getName() const {
	return tr("Audio");
}

QString QtAudioSettings::getTitle() const {
	return tr("Audio Settings");
}

QString QtAudioSettings::getIconName() const {
	return "audio";
}

void QtAudioSettings::saveConfig() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	AudioDevice device = getAudioDeviceFromComboBox(_ui->inputDeviceComboBox, AudioDeviceManager::getInstance().getInputDeviceList());
	config.set(Config::AUDIO_INPUT_DEVICEID_KEY, device.getData());

	device = getAudioDeviceFromComboBox(_ui->outputDeviceComboBox, AudioDeviceManager::getInstance().getOutputDeviceList());
	config.set(Config::AUDIO_OUTPUT_DEVICEID_KEY, device.getData());

	device = getAudioDeviceFromComboBox(_ui->ringingDeviceComboBox, AudioDeviceManager::getInstance().getOutputDeviceList());
	config.set(Config::AUDIO_RINGER_DEVICEID_KEY, device.getData());

	// advanced
	StringList saveCodec;
	int nbCodec = _ui->audioCodecListWidget->count();
	for (int i = 0; i < nbCodec; ++i) {
		std::string audioCodec(_ui->audioCodecListWidget->item(i)->text().toUtf8());
		// HACK: PHAPI_SPEEXWB_REPLACES_G726WB_HACK
		#ifdef PH_SPEEXWB_REPLACES_G726WB  
			if (audioCodec == "SPEEX/16000")
				audioCodec = "G726-64wb/16000";
		#endif
		////
		saveCodec += audioCodec;
	}
	config.set(Config::AUDIO_CODEC_LIST_KEY, saveCodec);
	////

	config.set(Config::AUDIO_HALFDUPLEX_KEY, _ui->halfDuplexCheckBox->isChecked());
}

void QtAudioSettings::readConfig() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	//inputDeviceList
	_ui->inputDeviceComboBox->clear();

#ifdef OS_LINUX
	bool savedSettingsFound = false;
	AudioDevice tmpDev(config.getAudioInputDeviceId());
#endif

	std::list<AudioDevice> inputDeviceList = AudioDeviceManager::getInstance().getInputDeviceList();
	for (std::list<AudioDevice>::const_iterator it = inputDeviceList.begin();
		it != inputDeviceList.end();
		++it) {
		_ui->inputDeviceComboBox->addItem(
#ifdef OS_MACOSX
			QString::fromUtf8((*it).getName().c_str()),
#else
			(*it).getName().c_str(),
#endif
			QString::fromUtf8((*it).getData().toString().c_str())
		);
#ifdef OS_LINUX
		if ((*it).getData() == tmpDev.getData()) {
			savedSettingsFound = true;
		}
#endif
	}

#ifdef OS_LINUX
	if (!savedSettingsFound) {
		_ui->outputDeviceComboBox->addItem(
			//QString::fromUtf8(tmpDev.getName().c_str()),
			QString(tmpDev.getName().c_str()),
			QString::fromUtf8(tmpDev.getData().toString().c_str())
		);
	}
#endif
	QString currentInputDeviceId = 
#ifdef OS_MACOSX
		QString::fromUtf8(config.getAudioInputDeviceId().toString().c_str());
#else
		QString(config.getAudioInputDeviceId().toString().c_str());
#endif
	_ui->inputDeviceComboBox->setCurrentIndex(
		_ui->inputDeviceComboBox->findData(currentInputDeviceId)
	);
	////

	//outputDeviceList
	_ui->outputDeviceComboBox->clear();

#ifdef OS_LINUX
	savedSettingsFound = false;
	tmpDev = AudioDevice(config.getAudioOutputDeviceId());
#endif
	std::list<AudioDevice> outputDeviceList = AudioDeviceManager::getInstance().getOutputDeviceList();
	for (std::list<AudioDevice>::const_iterator it = outputDeviceList.begin();
		it != outputDeviceList.end();
		++it) {
		_ui->outputDeviceComboBox->addItem(
#ifdef OS_MACOSX
			QString::fromUtf8((*it).getName().c_str()),
#else
			(*it).getName().c_str(),
#endif
			QString::fromUtf8((*it).getData().toString().c_str())
		);
#ifdef OS_LINUX
		if ((*it).getData() == tmpDev.getData()) {
			savedSettingsFound = true;
		}
#endif
	}

#ifdef OS_LINUX
	if (!savedSettingsFound) {
		_ui->outputDeviceComboBox->addItem(
			//QString::fromUtf8(tmpDev.getName().c_str()),
			QString(tmpDev.getName().c_str()),
			QString::fromUtf8(tmpDev.getData().toString().c_str())
		);
	}
#endif
	QString currentOutputDeviceId = 
#ifdef OS_MACOSX
		QString::fromUtf8(config.getAudioOutputDeviceId().toString().c_str());
#else
		QString(config.getAudioOutputDeviceId().toString().c_str());
#endif
	_ui->outputDeviceComboBox->setCurrentIndex(
		_ui->outputDeviceComboBox->findData(currentOutputDeviceId)
	);
	////

	//ringingDeviceList = outputDeviceList
	_ui->ringingDeviceComboBox->clear();
#ifdef OS_LINUX
	savedSettingsFound = false;
	tmpDev = AudioDevice(config.getAudioOutputDeviceId());
#endif
	for (std::list<AudioDevice>::const_iterator it = outputDeviceList.begin();
		it != outputDeviceList.end();
		++it) {
		_ui->ringingDeviceComboBox->addItem(
#ifdef OS_MACOSX
			QString::fromUtf8((*it).getName().c_str()),
#else
			(*it).getName().c_str(),
#endif
			QString::fromUtf8((*it).getData().toString().c_str())
		);
#ifdef OS_LINUX
		if ((*it).getData() == tmpDev.getData()) {
			savedSettingsFound = true;
		}
#endif
	}

#ifdef OS_LINUX
	if (!savedSettingsFound) {
		_ui->ringingDeviceComboBox->addItem(
			//QString::fromUtf8(tmpDev.getName().c_str()),
			QString(tmpDev.getName().c_str()),
			QString::fromUtf8(tmpDev.getData().toString().c_str())
		);
	}
#endif
	QString currentRingerDeviceId = 
#ifdef OS_MACOSX
		QString::fromUtf8(config.getAudioRingerDeviceId().toString().c_str());
#else
		QString(config.getAudioRingerDeviceId().toString().c_str());
#endif
	_ui->ringingDeviceComboBox->setCurrentIndex(
		_ui->ringingDeviceComboBox->findData(currentRingerDeviceId)
	);
	////
	
	// advanced
	fillAudioCodecList();

	_ui->halfDuplexCheckBox->setChecked(config.getAudioHalfDuplex());
}

void QtAudioSettings::makeTestCallClicked() {
	CUserProfile * cUserProfile = _cQuteCom.getCUserProfileHandler().getCUserProfile();
	if ((cUserProfile) && (cUserProfile->getUserProfile().getActivePhoneLine())) {
                Config & config = ConfigManager::getInstance().getCurrentConfig();
                std::string testnumber = config.getSipAudioTestCall ();
		cUserProfile->getUserProfile().getActivePhoneLine()->makeCall(testnumber);
		QtQuteCom* qQuteCom = static_cast<QtQuteCom*>(_cQuteCom.getPresentation());
		qQuteCom->getWidget()->raise();
	}
}

void QtAudioSettings::testOutputDevice() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	Sound* testSound = new Sound(config.getAudioCallClosedFile());
	AudioDevice device = getAudioDeviceFromComboBox(_ui->outputDeviceComboBox, AudioDeviceManager::getInstance().getOutputDeviceList());
	testSound->setWaveOutDevice(device);
	//Play the sound 4 times
	testSound->setLoops(4);
	testSound->play();
}

void QtAudioSettings::testRingingDevice() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	Sound* testSound = new Sound(config.getAudioCallClosedFile());
	AudioDevice device = getAudioDeviceFromComboBox(_ui->ringingDeviceComboBox, AudioDeviceManager::getInstance().getOutputDeviceList());
	testSound->setWaveOutDevice(device);
	//Play the sound 4 times
	testSound->setLoops(4);
	testSound->play();
}

void QtAudioSettings::AdvPressed() {
	if (_ui->AdvGroupBox->isVisible()) {
		_ui->AdvGroupBox->setVisible(false);
		_ui->AdvButton->setText(tr("Advanced") + " >>");
	} else {
		_ui->AdvGroupBox->setVisible(true);
		_ui->AdvButton->setText(tr("Advanced") + " <<");
	}
}

void QtAudioSettings::UpPressed() {

	//extracts current Item
	int currentrow = _ui->audioCodecListWidget->currentRow();

	//if an item is selected
	if (currentrow < 0) {
		return;
	}

	QListWidgetItem * mover = _ui->audioCodecListWidget->takeItem(currentrow);

	//inserts it to its new place
	 _ui->audioCodecListWidget->insertItem(currentrow - 1,mover);

	//sets focus on mover
	_ui->audioCodecListWidget->setCurrentItem(mover);
}

void QtAudioSettings::DownPressed() {

	//extracts current Item
	int currentrow = _ui->audioCodecListWidget->currentRow();

	//if an item is selected
	if (currentrow < 0) {
		return;
	}

	QListWidgetItem * mover = _ui->audioCodecListWidget->takeItem(currentrow);

	//inserts it to its new place
	 _ui->audioCodecListWidget->insertItem(currentrow + 1,mover);

	//sets focus on mover
	_ui->audioCodecListWidget->setCurrentItem(mover);
}

void QtAudioSettings::ResetPressed() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	//resets saved audio codec list
	config.resetToDefaultValue(Config::AUDIO_CODEC_LIST_KEY);

	fillAudioCodecList();
}

void QtAudioSettings::fillAudioCodecList() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	//resets displayed audio codec list
	StringList audioCodecList = config.getAudioCodecList();
	_ui->audioCodecListWidget->clear();
	for (StringList::const_iterator it = audioCodecList.begin(); it != audioCodecList.end(); ++it) {
		QString audioCodec = QString(it->c_str());
		// HACK: PHAPI_SPEEXWB_REPLACES_G726WB_HACK
		if (audioCodec == "G726-64wb/16000") {
			audioCodec = "SPEEX/16000";
		}
		////
		_ui->audioCodecListWidget->addItem(audioCodec);
	}
	////
}
