#ifndef RINPUT_H
#define RINPUT_H

/*
 * Copyright (c) 1999-2002 Vojtech Pavlik
 * Copyright (c) 2009 Chase Douglas
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 */

/*
 * This file is derived from linux/input.h from Linux 2.6.28
 */

#include <arpa/inet.h>
#include <stdint.h>

#define RINPUT_PROTOCOL_VERSION 1

/*
 * Event types
 */

#define RINPUT_EV_SYN			0x00
#define RINPUT_EV_KEY			0x01
#define RINPUT_EV_REL			0x02
#define RINPUT_EV_ABS			0x03
#define RINPUT_EV_MSC			0x04
#define RINPUT_EV_SW			0x05
#define RINPUT_EV_LED			0x11
#define RINPUT_EV_SND			0x12
#define RINPUT_EV_UTF8			0x80

typedef enum rinput_type {
    RINPUT_VERSION = 0,
    RINPUT_SET_CAPABILITY = 1,
    RINPUT_SET_ABS_PARAM = 2,
    RINPUT_CREATE = 3,
    RINPUT_DESTROY = 4,
    RINPUT_EVENT = 5,
    RINPUT_ERROR = 6
} rinput_type_t;

typedef enum rinput_abs_param_type {
    RINPUT_ABS_PARAM_MAX = 0,
    RINPUT_ABS_PARAM_MIN = 1,
    RINPUT_ABS_PARAM_FUZZ = 2,
    RINPUT_ABS_PARAM_FLAT = 3
} rinput_abs_param_type_t;

typedef struct rinput_capability {
    uint16_t type;
    uint16_t code;
} rinput_capability_t;

typedef struct rinput_abs_param {
    uint16_t axis;
    uint16_t type;
    int32_t value;
} rinput_abs_param_t;

typedef struct rinput_event {
    uint16_t type;
    uint16_t code;
    int32_t value;
} rinput_event_t;

typedef struct rinput_utf8 {
    uint16_t type;
    int16_t value;
    uint8_t character[4];
} rinput_utf8_t;

typedef enum rinput_error_type {
    RINPUT_INTERNAL_ERROR = 0,
    RINPUT_INPUT_DEVICE_ALLOC_FAILED = 1,
    RINPUT_MESSAGE_TYPE_INVALID = 2,
    RINPUT_CAPABILITY_TYPE_INVALID = 3,
    RINPUT_SET_CAPABILITY_FAILED = 4,
    RINPUT_ABS_AXIS_INVALID = 5,
    RINPUT_ABS_PARAM_TYPE_INVALID = 6,
    RINPUT_INPUT_DEVICE_CREATE_FAILED = 7,
    RINPUT_INPUT_EVENTS_FAILED = 8,
    RINPUT_REVOKED_AUTHORIZATION = 9
} rinput_error_type_t;

typedef struct rinput_error {
    rinput_error_type_t type;
    uint16_t code1;
    uint16_t code2;
} rinput_error_t;

typedef struct rinput_message {
    rinput_type_t type;
    union {
        uint32_t protocol_version;
        rinput_capability_t capability;
        rinput_abs_param_t abs_param;
        rinput_event_t event;
        rinput_utf8_t utf8;
        rinput_error_t error;
    } data;
} rinput_message_t;

static inline void hton_rinput(rinput_message_t *message) {
    switch (message->type) {
        case RINPUT_VERSION:
            message->data.protocol_version = htonl(message->data.protocol_version);
            break;
        case RINPUT_SET_CAPABILITY:
            message->data.capability.type = htons(message->data.capability.type);
            message->data.capability.code = htons(message->data.capability.code);
            break;
        case RINPUT_SET_ABS_PARAM:
            message->data.abs_param.axis = htons(message->data.abs_param.axis);
            message->data.abs_param.type = htons(message->data.abs_param.type);
            message->data.abs_param.value = htonl(message->data.abs_param.value);
            break;
        case RINPUT_EVENT:
            if (message->data.event.type != RINPUT_EV_UTF8) {
                message->data.event.value = htonl(message->data.event.value);
            }
            message->data.event.type = htons(message->data.event.type);
            message->data.event.code = htons(message->data.event.code);
            break;
        case RINPUT_ERROR:
            message->data.error.type = (rinput_error_type_t)htonl(message->data.error.type);
            message->data.error.code1 = htons(message->data.error.code1);
            message->data.error.code2 = htons(message->data.error.code2);
            break;
        case RINPUT_CREATE:
        case RINPUT_DESTROY:
            break;
    }
    message->type = (rinput_type_t)htonl(message->type);
}

static inline void ntoh_rinput(rinput_message_t *message) {
    message->type = (rinput_type_t)ntohl(message->type);
    switch (message->type) {
        case RINPUT_VERSION:
            message->data.protocol_version = ntohl(message->data.protocol_version);
            break;
        case RINPUT_SET_CAPABILITY:
            message->data.capability.type = ntohs(message->data.capability.type);
            message->data.capability.code = ntohs(message->data.capability.code);
            break;
        case RINPUT_SET_ABS_PARAM:
            message->data.abs_param.axis = ntohs(message->data.abs_param.axis);
            message->data.abs_param.type = ntohs(message->data.abs_param.type);
            message->data.abs_param.value = ntohl(message->data.abs_param.value);
            break;
        case RINPUT_EVENT:
            message->data.event.type = ntohs(message->data.event.type);
            message->data.event.code = ntohs(message->data.event.code);
            if (message->data.event.type != RINPUT_EV_UTF8) {
                message->data.event.value = ntohl(message->data.event.value);
            }
            break;
        case RINPUT_ERROR:
            message->data.error.type = (rinput_error_type_t)ntohl(message->data.error.type);
            message->data.error.code1 = ntohs(message->data.error.code1);
            message->data.error.code2 = ntohs(message->data.error.code2);
            break;
        case RINPUT_CREATE:
        case RINPUT_DESTROY:
            break;
    }
}

/*
 * Keys and buttons
 *
 * Most of the keys/buttons are modeled after USB HUT 1.12
 * (see http://www.usb.org/developers/hidpage).
 * Abbreviations in the comments:
 * AC - Application Control
 * AL - Application Launch Button
 * SC - System Control
 */

#define RINPUT_KEY_RESERVED		0
#define RINPUT_KEY_ESC			1
#define RINPUT_KEY_1			2
#define RINPUT_KEY_2			3
#define RINPUT_KEY_3			4
#define RINPUT_KEY_4			5
#define RINPUT_KEY_5			6
#define RINPUT_KEY_6			7
#define RINPUT_KEY_7			8
#define RINPUT_KEY_8			9
#define RINPUT_KEY_9			10
#define RINPUT_KEY_0			11
#define RINPUT_KEY_MINUS		12
#define RINPUT_KEY_EQUAL		13
#define RINPUT_KEY_BACKSPACE		14
#define RINPUT_KEY_TAB			15
#define RINPUT_KEY_Q			16
#define RINPUT_KEY_W			17
#define RINPUT_KEY_E			18
#define RINPUT_KEY_R			19
#define RINPUT_KEY_T			20
#define RINPUT_KEY_Y			21
#define RINPUT_KEY_U			22
#define RINPUT_KEY_I			23
#define RINPUT_KEY_O			24
#define RINPUT_KEY_P			25
#define RINPUT_KEY_LEFTBRACE		26
#define RINPUT_KEY_RIGHTBRACE		27
#define RINPUT_KEY_ENTER		28
#define RINPUT_KEY_LEFTCTRL		29
#define RINPUT_KEY_A			30
#define RINPUT_KEY_S			31
#define RINPUT_KEY_D			32
#define RINPUT_KEY_F			33
#define RINPUT_KEY_G			34
#define RINPUT_KEY_H			35
#define RINPUT_KEY_J			36
#define RINPUT_KEY_K			37
#define RINPUT_KEY_L			38
#define RINPUT_KEY_SEMICOLON		39
#define RINPUT_KEY_APOSTROPHE		40
#define RINPUT_KEY_GRAVE		41
#define RINPUT_KEY_LEFTSHIFT		42
#define RINPUT_KEY_BACKSLASH		43
#define RINPUT_KEY_Z			44
#define RINPUT_KEY_X			45
#define RINPUT_KEY_C			46
#define RINPUT_KEY_V			47
#define RINPUT_KEY_B			48
#define RINPUT_KEY_N			49
#define RINPUT_KEY_M			50
#define RINPUT_KEY_COMMA		51
#define RINPUT_KEY_DOT			52
#define RINPUT_KEY_SLASH		53
#define RINPUT_KEY_RIGHTSHIFT		54
#define RINPUT_KEY_KPASTERISK		55
#define RINPUT_KEY_LEFTALT		56
#define RINPUT_KEY_SPACE		57
#define RINPUT_KEY_CAPSLOCK		58
#define RINPUT_KEY_F1			59
#define RINPUT_KEY_F2			60
#define RINPUT_KEY_F3			61
#define RINPUT_KEY_F4			62
#define RINPUT_KEY_F5			63
#define RINPUT_KEY_F6			64
#define RINPUT_KEY_F7			65
#define RINPUT_KEY_F8			66
#define RINPUT_KEY_F9			67
#define RINPUT_KEY_F10			68
#define RINPUT_KEY_NUMLOCK		69
#define RINPUT_KEY_SCROLLLOCK		70
#define RINPUT_KEY_KP7			71
#define RINPUT_KEY_KP8			72
#define RINPUT_KEY_KP9			73
#define RINPUT_KEY_KPMINUS		74
#define RINPUT_KEY_KP4			75
#define RINPUT_KEY_KP5			76
#define RINPUT_KEY_KP6			77
#define RINPUT_KEY_KPPLUS		78
#define RINPUT_KEY_KP1			79
#define RINPUT_KEY_KP2			80
#define RINPUT_KEY_KP3			81
#define RINPUT_KEY_KP0			82
#define RINPUT_KEY_KPDOT		83

#define RINPUT_KEY_ZENKAKUHANKAKU	85
#define RINPUT_KEY_102ND		86
#define RINPUT_KEY_F11			87
#define RINPUT_KEY_F12			88
#define RINPUT_KEY_RO			89
#define RINPUT_KEY_KATAKANA		90
#define RINPUT_KEY_HIRAGANA		91
#define RINPUT_KEY_HENKAN		92
#define RINPUT_KEY_KATAKANAHIRAGANA	93
#define RINPUT_KEY_MUHENKAN		94
#define RINPUT_KEY_KPJPCOMMA		95
#define RINPUT_KEY_KPENTER		96
#define RINPUT_KEY_RIGHTCTRL		97
#define RINPUT_KEY_KPSLASH		98
#define RINPUT_KEY_SYSRQ		99
#define RINPUT_KEY_RIGHTALT		100
#define RINPUT_KEY_LINEFEED		101
#define RINPUT_KEY_HOME		102
#define RINPUT_KEY_UP			103
#define RINPUT_KEY_PAGEUP		104
#define RINPUT_KEY_LEFT		105
#define RINPUT_KEY_RIGHT		106
#define RINPUT_KEY_END			107
#define RINPUT_KEY_DOWN		108
#define RINPUT_KEY_PAGEDOWN		109
#define RINPUT_KEY_INSERT		110
#define RINPUT_KEY_DELETE		111
#define RINPUT_KEY_MACRO		112
#define RINPUT_KEY_MUTE		113
#define RINPUT_KEY_VOLUMEDOWN		114
#define RINPUT_KEY_VOLUMEUP		115
#define RINPUT_KEY_POWER		116	/* SC System Power Down */
#define RINPUT_KEY_KPEQUAL		117
#define RINPUT_KEY_KPPLUSMINUS		118
#define RINPUT_KEY_PAUSE		119
#define RINPUT_KEY_SCALE		120	/* AL Compiz Scale (Expose) */

#define RINPUT_KEY_KPCOMMA		121
#define RINPUT_KEY_HANGEUL		122
#define RINPUT_KEY_HANGUEL		KEY_HANGEUL
#define RINPUT_KEY_HANJA		123
#define RINPUT_KEY_YEN			124
#define RINPUT_KEY_LEFTMETA		125
#define RINPUT_KEY_RIGHTMETA		126
#define RINPUT_KEY_COMPOSE		127

#define RINPUT_KEY_STOP		128	/* AC Stop */
#define RINPUT_KEY_AGAIN		129
#define RINPUT_KEY_PROPS		130	/* AC Properties */
#define RINPUT_KEY_UNDO		131	/* AC Undo */
#define RINPUT_KEY_FRONT		132
#define RINPUT_KEY_COPY		133	/* AC Copy */
#define RINPUT_KEY_OPEN		134	/* AC Open */
#define RINPUT_KEY_PASTE		135	/* AC Paste */
#define RINPUT_KEY_FIND		136	/* AC Search */
#define RINPUT_KEY_CUT			137	/* AC Cut */
#define RINPUT_KEY_HELP		138	/* AL Integrated Help Center */
#define RINPUT_KEY_MENU		139	/* Menu (show menu) */
#define RINPUT_KEY_CALC		140	/* AL Calculator */
#define RINPUT_KEY_SETUP		141
#define RINPUT_KEY_SLEEP		142	/* SC System Sleep */
#define RINPUT_KEY_WAKEUP		143	/* System Wake Up */
#define RINPUT_KEY_FILE		144	/* AL Local Machine Browser */
#define RINPUT_KEY_SENDFILE		145
#define RINPUT_KEY_DELETEFILE		146
#define RINPUT_KEY_XFER		147
#define RINPUT_KEY_PROG1		148
#define RINPUT_KEY_PROG2		149
#define RINPUT_KEY_WWW			150	/* AL Internet Browser */
#define RINPUT_KEY_MSDOS		151
#define RINPUT_KEY_COFFEE		152	/* AL Terminal Lock/Screensaver */
#define RINPUT_KEY_SCREENLOCK		KEY_COFFEE
#define RINPUT_KEY_DIRECTION		153
#define RINPUT_KEY_CYCLEWINDOWS	154
#define RINPUT_KEY_MAIL		155
#define RINPUT_KEY_BOOKMARKS		156	/* AC Bookmarks */
#define RINPUT_KEY_COMPUTER		157
#define RINPUT_KEY_BACK		158	/* AC Back */
#define RINPUT_KEY_FORWARD		159	/* AC Forward */
#define RINPUT_KEY_CLOSECD		160
#define RINPUT_KEY_EJECTCD		161
#define RINPUT_KEY_EJECTCLOSECD	162
#define RINPUT_KEY_NEXTSONG		163
#define RINPUT_KEY_PLAYPAUSE		164
#define RINPUT_KEY_PREVIOUSSONG	165
#define RINPUT_KEY_STOPCD		166
#define RINPUT_KEY_RECORD		167
#define RINPUT_KEY_REWIND		168
#define RINPUT_KEY_PHONE		169	/* Media Select Telephone */
#define RINPUT_KEY_ISO			170
#define RINPUT_KEY_CONFIG		171	/* AL Consumer Control Configuration */
#define RINPUT_KEY_HOMEPAGE		172	/* AC Home */
#define RINPUT_KEY_REFRESH		173	/* AC Refresh */
#define RINPUT_KEY_EXIT		174	/* AC Exit */
#define RINPUT_KEY_MOVE		175
#define RINPUT_KEY_EDIT		176
#define RINPUT_KEY_SCROLLUP		177
#define RINPUT_KEY_SCROLLDOWN		178
#define RINPUT_KEY_KPLEFTPAREN		179
#define RINPUT_KEY_KPRIGHTPAREN	180
#define RINPUT_KEY_NEW			181	/* AC New */
#define RINPUT_KEY_REDO		182	/* AC Redo/Repeat */

#define RINPUT_KEY_F13			183
#define RINPUT_KEY_F14			184
#define RINPUT_KEY_F15			185
#define RINPUT_KEY_F16			186
#define RINPUT_KEY_F17			187
#define RINPUT_KEY_F18			188
#define RINPUT_KEY_F19			189
#define RINPUT_KEY_F20			190
#define RINPUT_KEY_F21			191
#define RINPUT_KEY_F22			192
#define RINPUT_KEY_F23			193
#define RINPUT_KEY_F24			194

#define RINPUT_KEY_PLAYCD		200
#define RINPUT_KEY_PAUSECD		201
#define RINPUT_KEY_PROG3		202
#define RINPUT_KEY_PROG4		203
#define RINPUT_KEY_DASHBOARD		204	/* AL Dashboard */
#define RINPUT_KEY_SUSPEND		205
#define RINPUT_KEY_CLOSE		206	/* AC Close */
#define RINPUT_KEY_PLAY		207
#define RINPUT_KEY_FASTFORWARD		208
#define RINPUT_KEY_BASSBOOST		209
#define RINPUT_KEY_PRINT		210	/* AC Print */
#define RINPUT_KEY_HP			211
#define RINPUT_KEY_CAMERA		212
#define RINPUT_KEY_SOUND		213
#define RINPUT_KEY_QUESTION		214
#define RINPUT_KEY_EMAIL		215
#define RINPUT_KEY_CHAT		216
#define RINPUT_KEY_SEARCH		217
#define RINPUT_KEY_CONNECT		218
#define RINPUT_KEY_FINANCE		219	/* AL Checkbook/Finance */
#define RINPUT_KEY_SPORT		220
#define RINPUT_KEY_SHOP		221
#define RINPUT_KEY_ALTERASE		222
#define RINPUT_KEY_CANCEL		223	/* AC Cancel */
#define RINPUT_KEY_BRIGHTNESSDOWN	224
#define RINPUT_KEY_BRIGHTNESSUP	225
#define RINPUT_KEY_MEDIA		226

#define RINPUT_KEY_SWITCHVIDEOMODE	227	/* Cycle between available video
outputs (Monitor/LCD/TV-out/etc) */
#define RINPUT_KEY_KBDILLUMTOGGLE	228
#define RINPUT_KEY_KBDILLUMDOWN	229
#define RINPUT_KEY_KBDILLUMUP		230

#define RINPUT_KEY_SEND		231	/* AC Send */
#define RINPUT_KEY_REPLY		232	/* AC Reply */
#define RINPUT_KEY_FORWARDMAIL		233	/* AC Forward Msg */
#define RINPUT_KEY_SAVE		234	/* AC Save */
#define RINPUT_KEY_DOCUMENTS		235

#define RINPUT_KEY_BATTERY		236

#define RINPUT_KEY_BLUETOOTH		237
#define RINPUT_KEY_WLAN		238
#define RINPUT_KEY_UWB			239

#define RINPUT_KEY_UNKNOWN		240

#define RINPUT_KEY_VIDEO_NEXT		241	/* drive next video source */
#define RINPUT_KEY_VIDEO_PREV		242	/* drive previous video source */
#define RINPUT_KEY_BRIGHTNESS_CYCLE	243	/* brightness up, after max is min */
#define RINPUT_KEY_BRIGHTNESS_ZERO	244	/* brightness off, use ambient */
#define RINPUT_KEY_DISPLAY_OFF		245	/* display device to off state */

#define RINPUT_KEY_WIMAX		246

/* Range 248 - 255 is reserved for special needs of AT keyboard driver */

#define RINPUT_BTN_MISC		0x100
#define RINPUT_BTN_0			0x100
#define RINPUT_BTN_1			0x101
#define RINPUT_BTN_2			0x102
#define RINPUT_BTN_3			0x103
#define RINPUT_BTN_4			0x104
#define RINPUT_BTN_5			0x105
#define RINPUT_BTN_6			0x106
#define RINPUT_BTN_7			0x107
#define RINPUT_BTN_8			0x108
#define RINPUT_BTN_9			0x109

#define RINPUT_BTN_MOUSE		0x110
#define RINPUT_BTN_LEFT		0x110
#define RINPUT_BTN_RIGHT		0x111
#define RINPUT_BTN_MIDDLE		0x112
#define RINPUT_BTN_SIDE		0x113
#define RINPUT_BTN_EXTRA		0x114
#define RINPUT_BTN_FORWARD		0x115
#define RINPUT_BTN_BACK		0x116
#define RINPUT_BTN_TASK		0x117

#define RINPUT_BTN_JOYSTICK		0x120
#define RINPUT_BTN_TRIGGER		0x120
#define RINPUT_BTN_THUMB		0x121
#define RINPUT_BTN_THUMB2		0x122
#define RINPUT_BTN_TOP			0x123
#define RINPUT_BTN_TOP2		0x124
#define RINPUT_BTN_PINKIE		0x125
#define RINPUT_BTN_BASE		0x126
#define RINPUT_BTN_BASE2		0x127
#define RINPUT_BTN_BASE3		0x128
#define RINPUT_BTN_BASE4		0x129
#define RINPUT_BTN_BASE5		0x12a
#define RINPUT_BTN_BASE6		0x12b
#define RINPUT_BTN_DEAD		0x12f

#define RINPUT_BTN_GAMEPAD		0x130
#define RINPUT_BTN_A			0x130
#define RINPUT_BTN_B			0x131
#define RINPUT_BTN_C			0x132
#define RINPUT_BTN_X			0x133
#define RINPUT_BTN_Y			0x134
#define RINPUT_BTN_Z			0x135
#define RINPUT_BTN_TL			0x136
#define RINPUT_BTN_TR			0x137
#define RINPUT_BTN_TL2			0x138
#define RINPUT_BTN_TR2			0x139
#define RINPUT_BTN_SELECT		0x13a
#define RINPUT_BTN_START		0x13b
#define RINPUT_BTN_MODE		0x13c
#define RINPUT_BTN_THUMBL		0x13d
#define RINPUT_BTN_THUMBR		0x13e

#define RINPUT_BTN_DIGI		0x140
#define RINPUT_BTN_TOOL_PEN		0x140
#define RINPUT_BTN_TOOL_RUBBER		0x141
#define RINPUT_BTN_TOOL_BRUSH		0x142
#define RINPUT_BTN_TOOL_PENCIL		0x143
#define RINPUT_BTN_TOOL_AIRBRUSH	0x144
#define RINPUT_BTN_TOOL_FINGER		0x145
#define RINPUT_BTN_TOOL_MOUSE		0x146
#define RINPUT_BTN_TOOL_LENS		0x147
#define RINPUT_BTN_TOUCH		0x14a
#define RINPUT_BTN_STYLUS		0x14b
#define RINPUT_BTN_STYLUS2		0x14c
#define RINPUT_BTN_TOOL_DOUBLETAP	0x14d
#define RINPUT_BTN_TOOL_TRIPLETAP	0x14e

#define RINPUT_BTN_WHEEL		0x150
#define RINPUT_BTN_GEAR_DOWN		0x150
#define RINPUT_BTN_GEAR_UP		0x151

#define RINPUT_KEY_OK			0x160
#define RINPUT_KEY_SELECT		0x161
#define RINPUT_KEY_GOTO		0x162
#define RINPUT_KEY_CLEAR		0x163
#define RINPUT_KEY_POWER2		0x164
#define RINPUT_KEY_OPTION		0x165
#define RINPUT_KEY_INFO		0x166	/* AL OEM Features/Tips/Tutorial */
#define RINPUT_KEY_TIME		0x167
#define RINPUT_KEY_VENDOR		0x168
#define RINPUT_KEY_ARCHIVE		0x169
#define RINPUT_KEY_PROGRAM		0x16a	/* Media Select Program Guide */
#define RINPUT_KEY_CHANNEL		0x16b
#define RINPUT_KEY_FAVORITES		0x16c
#define RINPUT_KEY_EPG			0x16d
#define RINPUT_KEY_PVR			0x16e	/* Media Select Home */
#define RINPUT_KEY_MHP			0x16f
#define RINPUT_KEY_LANGUAGE		0x170
#define RINPUT_KEY_TITLE		0x171
#define RINPUT_KEY_SUBTITLE		0x172
#define RINPUT_KEY_ANGLE		0x173
#define RINPUT_KEY_ZOOM		0x174
#define RINPUT_KEY_MODE		0x175
#define RINPUT_KEY_KEYBOARD		0x176
#define RINPUT_KEY_SCREEN		0x177
#define RINPUT_KEY_PC			0x178	/* Media Select Computer */
#define RINPUT_KEY_TV			0x179	/* Media Select TV */
#define RINPUT_KEY_TV2			0x17a	/* Media Select Cable */
#define RINPUT_KEY_VCR			0x17b	/* Media Select VCR */
#define RINPUT_KEY_VCR2		0x17c	/* VCR Plus */
#define RINPUT_KEY_SAT			0x17d	/* Media Select Satellite */
#define RINPUT_KEY_SAT2		0x17e
#define RINPUT_KEY_CD			0x17f	/* Media Select CD */
#define RINPUT_KEY_TAPE		0x180	/* Media Select Tape */
#define RINPUT_KEY_RADIO		0x181
#define RINPUT_KEY_TUNER		0x182	/* Media Select Tuner */
#define RINPUT_KEY_PLAYER		0x183
#define RINPUT_KEY_TEXT		0x184
#define RINPUT_KEY_DVD			0x185	/* Media Select DVD */
#define RINPUT_KEY_AUX			0x186
#define RINPUT_KEY_MP3			0x187
#define RINPUT_KEY_AUDIO		0x188
#define RINPUT_KEY_VIDEO		0x189
#define RINPUT_KEY_DIRECTORY		0x18a
#define RINPUT_KEY_LIST		0x18b
#define RINPUT_KEY_MEMO		0x18c	/* Media Select Messages */
#define RINPUT_KEY_CALENDAR		0x18d
#define RINPUT_KEY_RED			0x18e
#define RINPUT_KEY_GREEN		0x18f
#define RINPUT_KEY_YELLOW		0x190
#define RINPUT_KEY_BLUE		0x191
#define RINPUT_KEY_CHANNELUP		0x192	/* Channel Increment */
#define RINPUT_KEY_CHANNELDOWN		0x193	/* Channel Decrement */
#define RINPUT_KEY_FIRST		0x194
#define RINPUT_KEY_LAST		0x195	/* Recall Last */
#define RINPUT_KEY_AB			0x196
#define RINPUT_KEY_NEXT		0x197
#define RINPUT_KEY_RESTART		0x198
#define RINPUT_KEY_SLOW		0x199
#define RINPUT_KEY_SHUFFLE		0x19a
#define RINPUT_KEY_BREAK		0x19b
#define RINPUT_KEY_PREVIOUS		0x19c
#define RINPUT_KEY_DIGITS		0x19d
#define RINPUT_KEY_TEEN		0x19e
#define RINPUT_KEY_TWEN		0x19f
#define RINPUT_KEY_VIDEOPHONE		0x1a0	/* Media Select Video Phone */
#define RINPUT_KEY_GAMES		0x1a1	/* Media Select Games */
#define RINPUT_KEY_ZOOMIN		0x1a2	/* AC Zoom In */
#define RINPUT_KEY_ZOOMOUT		0x1a3	/* AC Zoom Out */
#define RINPUT_KEY_ZOOMRESET		0x1a4	/* AC Zoom */
#define RINPUT_KEY_WORDPROCESSOR	0x1a5	/* AL Word Processor */
#define RINPUT_KEY_EDITOR		0x1a6	/* AL Text Editor */
#define RINPUT_KEY_SPREADSHEET		0x1a7	/* AL Spreadsheet */
#define RINPUT_KEY_GRAPHICSEDITOR	0x1a8	/* AL Graphics Editor */
#define RINPUT_KEY_PRESENTATION	0x1a9	/* AL Presentation App */
#define RINPUT_KEY_DATABASE		0x1aa	/* AL Database App */
#define RINPUT_KEY_NEWS		0x1ab	/* AL Newsreader */
#define RINPUT_KEY_VOICEMAIL		0x1ac	/* AL Voicemail */
#define RINPUT_KEY_ADDRESSBOOK		0x1ad	/* AL Contacts/Address Book */
#define RINPUT_KEY_MESSENGER		0x1ae	/* AL Instant Messaging */
#define RINPUT_KEY_DISPLAYTOGGLE	0x1af	/* Turn display (LCD) on and off */
#define RINPUT_KEY_SPELLCHECK		0x1b0   /* AL Spell Check */
#define RINPUT_KEY_LOGOFF		0x1b1   /* AL Logoff */

#define RINPUT_KEY_DOLLAR		0x1b2
#define RINPUT_KEY_EURO		0x1b3

#define RINPUT_KEY_FRAMEBACK		0x1b4	/* Consumer - transport controls */
#define RINPUT_KEY_FRAMEFORWARD	0x1b5
#define RINPUT_KEY_CONTEXT_MENU	0x1b6	/* GenDesc - system context menu */
#define RINPUT_KEY_MEDIA_REPEAT	0x1b7	/* Consumer - transport control */

#define RINPUT_KEY_DEL_EOL		0x1c0
#define RINPUT_KEY_DEL_EOS		0x1c1
#define RINPUT_KEY_INS_LINE		0x1c2
#define RINPUT_KEY_DEL_LINE		0x1c3

#define RINPUT_KEY_FN			0x1d0
#define RINPUT_KEY_FN_ESC		0x1d1
#define RINPUT_KEY_FN_F1		0x1d2
#define RINPUT_KEY_FN_F2		0x1d3
#define RINPUT_KEY_FN_F3		0x1d4
#define RINPUT_KEY_FN_F4		0x1d5
#define RINPUT_KEY_FN_F5		0x1d6
#define RINPUT_KEY_FN_F6		0x1d7
#define RINPUT_KEY_FN_F7		0x1d8
#define RINPUT_KEY_FN_F8		0x1d9
#define RINPUT_KEY_FN_F9		0x1da
#define RINPUT_KEY_FN_F10		0x1db
#define RINPUT_KEY_FN_F11		0x1dc
#define RINPUT_KEY_FN_F12		0x1dd
#define RINPUT_KEY_FN_1		0x1de
#define RINPUT_KEY_FN_2		0x1df
#define RINPUT_KEY_FN_D		0x1e0
#define RINPUT_KEY_FN_E		0x1e1
#define RINPUT_KEY_FN_F		0x1e2
#define RINPUT_KEY_FN_S		0x1e3
#define RINPUT_KEY_FN_B		0x1e4

#define RINPUT_KEY_BRL_DOT1		0x1f1
#define RINPUT_KEY_BRL_DOT2		0x1f2
#define RINPUT_KEY_BRL_DOT3		0x1f3
#define RINPUT_KEY_BRL_DOT4		0x1f4
#define RINPUT_KEY_BRL_DOT5		0x1f5
#define RINPUT_KEY_BRL_DOT6		0x1f6
#define RINPUT_KEY_BRL_DOT7		0x1f7
#define RINPUT_KEY_BRL_DOT8		0x1f8
#define RINPUT_KEY_BRL_DOT9		0x1f9
#define RINPUT_KEY_BRL_DOT10		0x1fa

#define RINPUT_KEY_NUMERIC_0		0x200	/* used by phones, remote controls, */
#define RINPUT_KEY_NUMERIC_1		0x201	/* and other keypads */
#define RINPUT_KEY_NUMERIC_2		0x202
#define RINPUT_KEY_NUMERIC_3		0x203
#define RINPUT_KEY_NUMERIC_4		0x204
#define RINPUT_KEY_NUMERIC_5		0x205
#define RINPUT_KEY_NUMERIC_6		0x206
#define RINPUT_KEY_NUMERIC_7		0x207
#define RINPUT_KEY_NUMERIC_8		0x208
#define RINPUT_KEY_NUMERIC_9		0x209
#define RINPUT_KEY_NUMERIC_STAR	0x20a
#define RINPUT_KEY_NUMERIC_POUND	0x20b

/* We avoid low common keys in module aliases so they don't get huge. */
#define RINPUT_KEY_MIN_INTERESTING	KEY_MUTE
#define RINPUT_KEY_MAX			0x2ff
#define RINPUT_KEY_CNT			(RINPUT_KEY_MAX+1)

/*
 * Relative axes
 */

#define RINPUT_REL_X			0x00
#define RINPUT_REL_Y			0x01
#define RINPUT_REL_Z			0x02
#define RINPUT_REL_RX			0x03
#define RINPUT_REL_RY			0x04
#define RINPUT_REL_RZ			0x05
#define RINPUT_REL_HWHEEL		0x06
#define RINPUT_REL_DIAL		0x07
#define RINPUT_REL_WHEEL		0x08
#define RINPUT_REL_MISC		0x09
#define RINPUT_REL_MAX			0x0f
#define RINPUT_REL_CNT			(RINPUT_REL_MAX+1)

/*
 * Absolute axes
 */

#define RINPUT_ABS_X			0x00
#define RINPUT_ABS_Y			0x01
#define RINPUT_ABS_Z			0x02
#define RINPUT_ABS_RX			0x03
#define RINPUT_ABS_RY			0x04
#define RINPUT_ABS_RZ			0x05
#define RINPUT_ABS_THROTTLE		0x06
#define RINPUT_ABS_RUDDER		0x07
#define RINPUT_ABS_WHEEL		0x08
#define RINPUT_ABS_GAS			0x09
#define RINPUT_ABS_BRAKE		0x0a
#define RINPUT_ABS_HAT0X		0x10
#define RINPUT_ABS_HAT0Y		0x11
#define RINPUT_ABS_HAT1X		0x12
#define RINPUT_ABS_HAT1Y		0x13
#define RINPUT_ABS_HAT2X		0x14
#define RINPUT_ABS_HAT2Y		0x15
#define RINPUT_ABS_HAT3X		0x16
#define RINPUT_ABS_HAT3Y		0x17
#define RINPUT_ABS_PRESSURE		0x18
#define RINPUT_ABS_DISTANCE		0x19
#define RINPUT_ABS_TILT_X		0x1a
#define RINPUT_ABS_TILT_Y		0x1b
#define RINPUT_ABS_TOOL_WIDTH		0x1c
#define RINPUT_ABS_VOLUME		0x20
#define RINPUT_ABS_MISC		0x28
#define RINPUT_ABS_MAX			0x3f
#define RINPUT_ABS_CNT			(RINPUT_ABS_MAX+1)

/*
 * Switch events
 */

#define RINPUT_SW_LID			0x00  /* set = lid shut */
#define RINPUT_SW_TABLET_MODE		0x01  /* set = tablet mode */
#define RINPUT_SW_HEADPHONE_INSERT	0x02  /* set = inserted */
#define RINPUT_SW_RFKILL_ALL		0x03  /* rfkill master switch, type "any"
set = radio enabled */
#define RINPUT_SW_RADIO		SW_RFKILL_ALL	/* deprecated */
#define RINPUT_SW_MICROPHONE_INSERT	0x04  /* set = inserted */
#define RINPUT_SW_DOCK			0x05  /* set = plugged into dock */
#define RINPUT_SW_MAX			0x0f
#define RINPUT_SW_CNT			(RINPUT_SW_MAX+1)

/*
 * Misc events
 */

#define RINPUT_MSC_SERIAL		0x00
#define RINPUT_MSC_PULSELED		0x01
#define RINPUT_MSC_GESTURE		0x02
#define RINPUT_MSC_RAW			0x03
#define RINPUT_MSC_SCAN		0x04
#define RINPUT_MSC_MAX			0x07
#define RINPUT_MSC_CNT			(RINPUT_MSC_MAX+1)

/*
 * LEDs
 */

#define RINPUT_LED_NUML		0x00
#define RINPUT_LED_CAPSL		0x01
#define RINPUT_LED_SCROLLL		0x02
#define RINPUT_LED_COMPOSE		0x03
#define RINPUT_LED_KANA		0x04
#define RINPUT_LED_SLEEP		0x05
#define RINPUT_LED_SUSPEND		0x06
#define RINPUT_LED_MUTE		0x07
#define RINPUT_LED_MISC		0x08
#define RINPUT_LED_MAIL		0x09
#define RINPUT_LED_CHARGING		0x0a
#define RINPUT_LED_MAX			0x0f
#define RINPUT_LED_CNT			(RINPUT_LED_MAX+1)

/*
 * Sounds
 */

#define RINPUT_SND_CLICK		0x00
#define RINPUT_SND_BELL		0x01
#define RINPUT_SND_TONE		0x02
#define RINPUT_SND_MAX			0x07
#define RINPUT_SND_CNT			(RINPUT_SND_MAX+1)

#endif // RINPUT_H
