/*****************************************************************************
 * Project: RooFit                                                           *
 * Package: RooFitCore                                                       *
 * @(#)root/roofitcore:$Id$
 * Authors:                                                                  *
 *   WV, Wouter Verkerke, UC Santa Barbara, verkerke@slac.stanford.edu       *
 *   DK, David Kirkby,    UC Irvine,         dkirkby@uci.edu                 *
 *                                                                           *
 * Copyright (c) 2000-2005, Regents of the University of California          *
 *                          and Stanford University. All rights reserved.    *
 *                                                                           *
 * Redistribution and use in source and binary forms,                        *
 * with or without modification, are permitted according to the terms        *
 * listed in LICENSE (http://roofit.sourceforge.net/license.txt)             *
 *****************************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// BEGIN_HTML
// RooAbsDataStore is the abstract base class for data collection that
// use a TTree as internal storage mechanism
// END_HTML
//

#include "RooFit.h"
#include "RooMsgService.h"
#include "RooAbsDataStore.h"

#include "Riostream.h"
#include <iomanip>
#include "TClass.h"

using namespace std ;

ClassImp(RooAbsDataStore)
;


//_____________________________________________________________________________
RooAbsDataStore::RooAbsDataStore() 
{
  // Default constructor
  _iterator = _vars.createIterator() ;
  _cacheIter = _cachedVars.createIterator() ;
  _doDirtyProp = kTRUE ;
}




//_____________________________________________________________________________
RooAbsDataStore::RooAbsDataStore(const char* name, const char* title, const RooArgSet& vars) : 
  TNamed(name,title)
{
  // Default constructor

  // clone the fundamentals of the given data set into internal buffer
  _vars.add(vars) ;

  _iterator = _vars.createIterator() ;
  _cacheIter = _cachedVars.createIterator() ;
  _doDirtyProp = kTRUE ;
}




//_____________________________________________________________________________
RooAbsDataStore::RooAbsDataStore(const RooAbsDataStore& other, const char* newname) : TNamed(other), RooPrintable(other)
{
  if (newname) {
    SetName(newname) ;
  }
  _vars.add(other._vars) ;
  _iterator = _vars.createIterator() ;
  _cacheIter = _cachedVars.createIterator() ;
  _doDirtyProp = other._doDirtyProp ;
}



//_____________________________________________________________________________
RooAbsDataStore::RooAbsDataStore(const RooAbsDataStore& other, const RooArgSet& vars, const char* newname) : TNamed(other), RooPrintable(other)
{
  if (newname) {
    SetName(newname) ;
  }
  _vars.add(vars) ;
  _iterator = _vars.createIterator() ;
  _cacheIter = _cachedVars.createIterator() ;
  _doDirtyProp = other._doDirtyProp ;
}



//_____________________________________________________________________________
RooAbsDataStore::~RooAbsDataStore()
{
  // Destructor
  delete _iterator ;
  delete _cacheIter ;
}



//_____________________________________________________________________________
Bool_t RooAbsDataStore::valid() const 
{
  // Return true if currently loaded coordinate is considered valid within
  // the current range definitions of all observables
  return kTRUE ;
}


//_____________________________________________________________________________
void RooAbsDataStore::printName(ostream& os) const 
{
  // Print name of dataset

  os << GetName() ;
}



//_____________________________________________________________________________
void RooAbsDataStore::printTitle(ostream& os) const 
{
  // Print title of dataset
  os << GetTitle() ;
}



//_____________________________________________________________________________
void RooAbsDataStore::printClassName(ostream& os) const 
{
  // Print class name of dataset
  os << IsA()->GetName() ;
}



//_____________________________________________________________________________
void RooAbsDataStore::printValue(ostream& os) const 
{
  // Print value of the dataset, i.e. the sum of weights contained in the dataset
  os << numEntries() << " entries" ;
}



//_____________________________________________________________________________
void RooAbsDataStore::printArgs(ostream& os) const 
{
  // Print argument of dataset, i.e. the observable names

  os << "[" ;    
  _iterator->Reset() ;
  RooAbsArg* arg ;
  Bool_t first(kTRUE) ;
  while((arg=(RooAbsArg*)_iterator->Next())) {
    if (first) {
      first=kFALSE ;
    } else {
      os << "," ;
    }
    os << arg->GetName() ;
  }
  os << "]" ;
}






//_____________________________________________________________________________
Int_t RooAbsDataStore::defaultPrintContents(Option_t* /*opt*/) const 
{
  // Define default print options, for a given print style

  return kName|kClassName|kArgs|kValue ;
}





//_____________________________________________________________________________
void RooAbsDataStore::printMultiline(ostream& os, Int_t /*content*/, Bool_t verbose, TString indent) const 
{
  // Detailed printing interface

  os << indent << "DataStore " << GetName() << " (" << GetTitle() << ")" << endl ;
  os << indent << "  Contains " << numEntries() << " entries" << endl;

  if (!verbose) {
    os << indent << "  Observables " << _vars << endl ;
  } else {
    os << indent << "  Observables: " << endl ;
    _vars.printStream(os,kName|kValue|kExtras|kTitle,kVerbose,indent+"  ") ;
  }

  if(verbose) {
    if (_cachedVars.getSize()>0) {
      os << indent << "  Caches " << _cachedVars << endl ;
    }
//     if(_truth.getSize() > 0) {
//       os << indent << "  Generated with ";
//       TString deeper(indent) ;
//       deeper += "   " ;
//       _truth.printStream(os,kName|kValue,kStandard,deeper) ;
//     }
  }
}



