use itertools::Itertools;
use malachite_base::num::basic::signeds::PrimitiveSigned;
use malachite_base::num::float::NiceFloat;
use malachite_base::num::random::striped::striped_random_negative_signeds;
use malachite_base::random::EXAMPLE_SEED;
use malachite_base::strings::ToBinaryString;
use malachite_base::test_util::stats::common_values_map::common_values_map;
use malachite_base::test_util::stats::median;
use malachite_base::test_util::stats::moments::{moment_stats, CheckedToF64, MomentStats};
use std::panic::catch_unwind;

fn striped_random_negative_signeds_helper<T: CheckedToF64 + PrimitiveSigned>(
    m_numerator: u64,
    m_denominator: u64,
    expected_values: &[&str],
    expected_common_values: &[(&str, usize)],
    expected_sample_median: (T, Option<T>),
    expected_sample_moment_stats: MomentStats,
) {
    let xs = striped_random_negative_signeds::<T>(EXAMPLE_SEED, m_numerator, m_denominator);
    let actual_values = xs
        .clone()
        .map(|x| x.to_binary_string())
        .take(20)
        .collect_vec();
    let actual_common_values = common_values_map(1000000, 10, xs.clone())
        .iter()
        .map(|(x, frequency)| (x.to_binary_string(), *frequency))
        .collect_vec();
    let actual_sample_median = median(xs.clone().take(1000000));
    let actual_sample_moment_stats = moment_stats(xs.take(1000000));
    assert_eq!(
        (
            actual_values,
            actual_common_values,
            actual_sample_median,
            actual_sample_moment_stats
        ),
        (
            expected_values
                .iter()
                .map(ToString::to_string)
                .collect_vec(),
            expected_common_values
                .iter()
                .map(|(x, frequency)| (x.to_string(), *frequency))
                .collect_vec(),
            expected_sample_median,
            expected_sample_moment_stats
        )
    );
}

#[test]
fn test_striped_random_negative_signeds() {
    // i8, m = 4
    let values = &[
        "10000000", "10101100", "10110000", "11111100", "10001111", "11111110", "10000000",
        "10000111", "10011101", "11100000", "11111111", "11100000", "10000000", "10000010",
        "11000011", "10111111", "10000001", "10000000", "10001111", "10000001",
    ];
    let common_values = &[
        ("10000000", 89042),
        ("11111111", 88624),
        ("10011111", 29871),
        ("11111000", 29848),
        ("11000000", 29802),
        ("11111110", 29796),
        ("11100000", 29664),
        ("11110000", 29649),
        ("10000111", 29644),
        ("10111111", 29621),
    ];
    let sample_median = (-64, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-64.47355199999896),
        standard_deviation: NiceFloat(47.66137677522698),
        skewness: NiceFloat(-0.0011056592983105659),
        excess_kurtosis: NiceFloat(-1.5649370173869896),
    };
    striped_random_negative_signeds_helper::<i8>(
        4,
        1,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // i8, m = 2
    let values = &[
        "10011001", "10011100", "10101010", "11001011", "10101101", "11111100", "10111100",
        "10010100", "10001101", "11111111", "11100011", "11101101", "10001100", "10000100",
        "11100001", "10100011", "10110100", "10110101", "10110100", "10010011",
    ];
    let common_values = &[
        ("10011010", 8131),
        ("10001111", 8059),
        ("11010101", 8004),
        ("11001101", 7998),
        ("10011011", 7993),
        ("11110111", 7978),
        ("11010100", 7959),
        ("10000111", 7958),
        ("11011010", 7953),
        ("11100011", 7947),
    ];
    let sample_median = (-64, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-64.44477500000217),
        standard_deviation: NiceFloat(36.93835958244133),
        skewness: NiceFloat(-0.0007106807748731826),
        excess_kurtosis: NiceFloat(-1.2008391146615451),
    };
    striped_random_negative_signeds_helper::<i8>(
        2,
        1,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // i8, m = 5/4
    let values = &[
        "10101001", "10100101", "10100110", "11101001", "10101101", "11010101", "10111010",
        "10101010", "10010110", "11010101", "11001000", "11001011", "10001000", "10011100",
        "11111010", "10101101", "10110101", "10101010", "10100101", "10100010",
    ];
    let common_values = &[
        ("10101010", 131212),
        ("11010101", 131202),
        ("11001010", 33119),
        ("11011010", 33073),
        ("10010101", 32947),
        ("10100101", 32868),
        ("11010010", 32851),
        ("10101011", 32817),
        ("10101001", 32765),
        ("10110101", 32761),
    ];
    let sample_median = (-63, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-64.48624499999755),
        standard_deviation: NiceFloat(27.103343066698287),
        skewness: NiceFloat(-0.0017292127163029483),
        excess_kurtosis: NiceFloat(-1.1007498380278833),
    };
    striped_random_negative_signeds_helper::<i8>(
        5,
        4,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // i64, m = 32
    let values = &[
        "1000000000000000000000000000000000011111111111111111111111111111",
        "1000000111111111111111111111111111111111111111111111111111111111",
        "1000000000000000000000000000000000001111111111111111111111111111",
        "1111111111111111111111111110000000000000000000000000000000000000",
        "1000000111111111111111111111111111111111111111111111111111111111",
        "1111111111000000000000000000000000000000000000001100000000111111",
        "1000000000000000000000000000000000000000000011111111111111100000",
        "1000000000000000000000000000000000000000000000011111111111111111",
        "1000000000000000000000000000000000000000000000000000111111111111",
        "1111111111111111111111111111111111111111111111111111100000000000",
        "1111111111111100000000000000000000000000000000000000000000000000",
        "1111111111000000000000000000000000000000000000000000000000000000",
        "1000000000000000000001111111111111111111111111111100000000000000",
        "1000000000000000000000000000000000001000000011111111111111111111",
        "1111111000000000000000111111111111111111111111111111111111111111",
        "1000000000000000000000000000000000000000000000000000000000000000",
        "1000000000000011111111111100000000000000000000000000000000000000",
        "1000000000000000111111111111111111111110000000000000000001111111",
        "1001111111111111111111111111111111111111111111111111111111111111",
        "1000000000000000000000000000000000000000000000000000000001111111",
    ];
    let common_values = &[
        (
            "1111111111111111111111111111111111111111111111111111111111111111",
            69948,
        ),
        (
            "1000000000000000000000000000000000000000000000000000000000000000",
            69809,
        ),
        (
            "1111111111111111111111111111111111111111111111111111111111111000",
            2383,
        ),
        (
            "1000011111111111111111111111111111111111111111111111111111111111",
            2362,
        ),
        (
            "1111111111111111100000000000000000000000000000000000000000000000",
            2334,
        ),
        (
            "1000000000000000000000000000000000000000011111111111111111111111",
            2334,
        ),
        (
            "1100000000000000000000000000000000000000000000000000000000000000",
            2328,
        ),
        (
            "1111111111111111111111111111111100000000000000000000000000000000",
            2327,
        ),
        (
            "1000000000000000000000000000000000000000000000000000000111111111",
            2320,
        ),
        (
            "1000000000000000000000000000000000000000000000000000000000000011",
            2318,
        ),
    ];
    let sample_median = (-4611686018427387904, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-4.608396021125303e18),
        standard_deviation: NiceFloat(4.4270184647985106e18),
        skewness: NiceFloat(-0.0014267894129673469),
        excess_kurtosis: NiceFloat(-1.9430528242638716),
    };
    striped_random_negative_signeds_helper::<i64>(
        32,
        1,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // i64, m = 2
    let values = &[
        "1001100110001101010100101101001000001100001110101111001000000001",
        "1000111011011100111110111000010111001101001101010010111011001100",
        "1011110000010100111000110001101100111001001010101110001000100000",
        "1110010001100111101011100111111100001001110001100001001000000011",
        "1000001100110011001011000001001111001010100010110100011010010001",
        "1111000101110010000110100101010100100111001100100001101010011011",
        "1000010000011110001100010001010101101101100100000010011101111010",
        "1001110110110110011010011011010000111001101110110010001101011110",
        "1011010010000001101100111000010011100101110010101101001100110000",
        "1110000010001000110001011100111001101110010001111000110001111001",
        "1100011101111011001001101101011100000110001110100111011011011111",
        "1111101111100000110000001010001001101011110011110110100010110010",
        "1001010001010011101001011011111100101110000001010101000111100001",
        "1011110011011100110101011110010001110100010111001010000100011101",
        "1110001001100111101011111000100111101011110111101110011010100111",
        "1000001111011101101101011111011010011011001011010001101011100101",
        "1011111101110101010010000100011110100110100000110100101000110111",
        "1010101011010100000101011100111011000001101010001000101111111010",
        "1000110000111110001111000001110011101110100001101011111010100110",
        "1001101010111111111000111001111000111110001111110100101000001111",
    ];
    let common_values = &[
        (
            "1111111111111111111101100100101101000010000101001100011100001011",
            1,
        ),
        (
            "1111111111111111111101010010001100000001111011100011001110010101",
            1,
        ),
        (
            "1111111111111111111101000101011100111011001010001100011011001010",
            1,
        ),
        (
            "1111111111111111111100101001000100000101110101000111010111100101",
            1,
        ),
        (
            "1111111111111111111100000101111010110010101111000000001111000010",
            1,
        ),
        (
            "1111111111111111111010001000001010110110111101111011110111000001",
            1,
        ),
        (
            "1111111111111111110111101001011000111000100110110000000111010011",
            1,
        ),
        (
            "1111111111111111110110001001110101000111010110110110001100010111",
            1,
        ),
        (
            "1111111111111111110101100100101011001110101010101111101001011101",
            1,
        ),
        (
            "1111111111111111101110111110101110101010000001001011010001101111",
            1,
        ),
    ];
    let sample_median = (-4607307127481905067, Some(-4607304507984009164));
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-4.6086823879191747e18),
        standard_deviation: NiceFloat(2.66222507998862e18),
        skewness: NiceFloat(-0.0006757801147528804),
        excess_kurtosis: NiceFloat(-1.1997523258037708),
    };
    striped_random_negative_signeds_helper::<i64>(
        2,
        1,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // i64, m = 33/32
    let values = &[
        "1010101010101010101010101101010101010101010101010101010101010101",
        "1010101010101010100010101010010101101010101010101011010101011010",
        "1010101010010101011010101010101010101101010101010101010101010101",
        "1101010101010101010101101010101010101010101010101010101010101010",
        "1010101010101010101010101011010101010101001010010101001010101010",
        "1101010101010101010101010101010101010100101010010101010110101010",
        "1010101010101010101010101010101010101010101010101010110101010101",
        "1010101010101010101010010101010101010101010101010101010101010101",
        "1010101010010101010101010101010101010101010101010101010101010101",
        "1101010101100101010101010101010101010101010101010101010010101010",
        "1101010101010101010101010101010101011001010101010101101010101010",
        "1101101010101010011010101010101010101010101010101010101010101010",
        "1010101001010101010101011010101010101010101010101010101010101010",
        "1010101010101010101011010010010110101010101010101010101010101010",
        "1101010101010101010110101010101010101010101010010101010101010101",
        "1010101010101010101010101010101010101010101010101010101010101010",
        "1010101010101010101010101010101010101010101010101011010101010101",
        "1010101010101010101010101010101010101010101101010101010101010101",
        "1010101010101010101011011010100101010101001010101010101010101010",
        "1010101010101010101010101010101011010101010101010101010101010101",
    ];
    let common_values = &[
        (
            "1101010101010101010101010101010101010101010101010101010101010101",
            74299,
        ),
        (
            "1010101010101010101010101010101010101010101010101010101010101010",
            74072,
        ),
        (
            "1101010101010101010101010010101010101010101010101010101010101010",
            2429,
        ),
        (
            "1101010101011010101010101010101010101010101010101010101010101010",
            2419,
        ),
        (
            "1101010101010101010101010101010101010101010101010101010101011010",
            2419,
        ),
        (
            "1101010101010101010101001010101010101010101010101010101010101010",
            2411,
        ),
        (
            "1101010101010101010101010101010101010101010101010010101010101010",
            2393,
        ),
        (
            "1101010101001010101010101010101010101010101010101010101010101010",
            2390,
        ),
        (
            "1010101010101010101010101010101010101010101010101010101010100101",
            2389,
        ),
        (
            "1010101010101010101010101010101001010101010101010101010101010101",
            2389,
        ),
    ];
    let sample_median = (-3942150873824974166, Some(-3942150873824971435));
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-4.610217758608903e18),
        standard_deviation: NiceFloat(1.5991405421620288e18),
        skewness: NiceFloat(-0.0019511120341977582),
        excess_kurtosis: NiceFloat(-1.7372862317601778),
    };
    striped_random_negative_signeds_helper::<i64>(
        33,
        32,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );
}

fn striped_random_negative_signeds_fail_helper<T: PrimitiveSigned>() {
    assert_panic!(striped_random_negative_signeds::<T>(EXAMPLE_SEED, 1, 0));
    assert_panic!(striped_random_negative_signeds::<T>(EXAMPLE_SEED, 2, 3));
}

#[test]
fn striped_random_negative_signeds_fail() {
    apply_fn_to_signeds!(striped_random_negative_signeds_fail_helper);
}
