/*
  session.cc, copyright (c) 2006,2008 by Vincent Fourmond: 
  The class embedding a whole SCalc session, implementation.
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

#include <calc_internals.hh>

namespace SCalc {

  std::string version()
  {
    return std::string(PACKAGE_VERSION);
  }

  Session::Session()
  {
    FuncDef::register_common_functions(this);
  }
  
  /** Does the actual job of evaluating a statement.
      If the returned expression is not NULL, it should
      be deleted.
  */
  ParserResult * Session::eval(const char * str)
  {
    Scanner s(str, this);
    return s.scan();
  }
  
  void Session::eval_and_free(const char * str)
  {
    ParserResult * res = eval(str);
    if(res->can_delete())
      delete res;
  }


  // Not inline, to avoid missing symbols...
  int Session::register_varname(const char * str)
  {
    return register_varname(std::string(str));
  }

  int Session::register_varname(const std::string &varname)
  {
    std::map<std::string,int>::iterator i = variables_numbers.find(varname);
    if(i == variables_numbers.end())
      { // not found
	int j;
	variables.push_back(varname);
	j = variables.size() - 1;
	variables_numbers[varname] = j;
	return j;
      }
    else
      return i->second;
  }

  const char * Session::varname(int id)
  {
    if(id < (int) variables.size() && id >= 0)
      return variables[id].c_str();
    else return "invalid varname";
  }

  int Session::set_var(int id, double val)
  {
    if(id >= 0 && id < (int) variables.size())
      values[id] = val;
    else
      return 0; /* failed */
    return 1; /* success */
  }

  void Session::fill_default(double * val)
  {
    for(std::map<int, double>::iterator i = values.begin();
	i != values.end(); i++)
      val[i->first] = i->second;
  }
  
  int Session::evaluable(Expression * expr)
  {
    std::set<int> used = expr->used_variables();
    int retval = 1;
    for(std::set<int>::iterator i = used.begin();
	i != used.end(); i++)
      if(values.find(*i) == values.end())
	{
	  retval = 0; // we have found one that isn't there
	  break;
	}
    return retval;
  }

  /// Returns the number of the corresponding slot in functions.
  int Session::get_func(std::string name)
  {
    if(functions_numbers.find(name) == functions_numbers.end())
      return -1; // not found
    return functions_numbers[name];
  }

  int Session::register_func_def(FuncDef * def)
  {
    if(def->name().empty() || 
       get_func(def->name()) != -1)
      return 0; /* fails to register in that case */
    /* now, the actual registering stuff */
    int i = functions.size();
    functions.push_back(def);
    functions_numbers[std::string(def->name())] = i;
    return 1; // success !
  }

  FuncDef * Session::get_func_def(std::string name)
  {
    int i = get_func(name);
    if(i >= 0)
      return functions[i];
    else
      return NULL;
  }

  int Session::replace_func_def(FuncDef * def)
  {
    if(def->name().empty())
      return 0; /* fails to register in that case */
    int i = get_func(def->name());
    if(i < 0)
      return register_func_def(def);
    delete functions[i];
    functions[i] = def;
    return 1; // success !
  }

  /* destruction of the Session */
  Session::~Session()
  {
    // We first need to make sure that the objects listed in functions
    // reference only "named" functions
    for(int i = 0; i < nb_funcs(); i++)
      functions[i]->destroy_anonymous_derivatives();
    // Once this is sure, we can delete them.
    for(int i = 0; i < nb_funcs(); i++)
      delete functions[i];
    functions.clear();
  }

  std::vector<std::string> Session::func_names()
  {
    std::vector<std::string> res;
    for(int i = 0; i < nb_funcs(); i++)
      res.push_back(functions[i]->name());
    return res;
  }

  int Session::nb_args_func(std::string name)
  {
    FuncDef * f = get_func_def(name);
    if(f)
      return f->nb_params();
    return -1; // didn't work out which function...
  }

  int Session::unset_var(int i)
  {
    std::map<int, double>::iterator el = values.find(i);
    if( el == values.end())
      return 0;
    values.erase(el);
    return 1;
  }

  int Session::unset_var(std::string name) 
  {
    std::map<std::string,int>::iterator i = variables_numbers.find(name);
    if(i != variables_numbers.end())
      return unset_var(i->second);
    else
      return 0; 		// there is no such variable.
  }

  Expression * Session::constant(double value)
  {
    return new Const(this, value);
  }
};
