/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Cadstar netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t ecadstar_net;

static int cadstar_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "cadstar") == 0)
		return 100;
	return 0;
}

/* Write a single attribute to f */
static void export_attr(FILE *f, const char *refdes, const char *key, const char *val)
{
	if ((key == NULL) || (val == NULL))
		return;
	fprintf(f, ".ATT_COM %s \"%s\" \"%s\"\n", refdes, key, val);
}


/* Export all component attribute */
static void export_comp_attrs(FILE *f, const char *refdes, csch_ahdr_t *obj)
{
	htsp_entry_t *e;
	for(e = htsp_first(&obj->attr); e != NULL; e = htsp_next(&obj->attr, e)) {
		csch_attrib_t *a = e->value;
		export_attr(f, refdes, a->export_name, a->val);
	}
}

/* Export abstract components with all component attributes; pin attributes
   can not be exported in this format. */
static void cadstar_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val, *desc;
		char desctmp[32];

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);


		if (dev != NULL) desc = dev;
		else if (refdes != NULL) { /* use the first alpha-sequence of refdes */
			const char *si;
			char *so, *soend;
			soend = desctmp + sizeof(desctmp)-1;

			for(si = refdes, so = desctmp; isalpha(*si) && so < soend; si++, so++)
				*so = *si;
			*so = '\0';
			desc = desctmp;
		}

		fprintf(f, "\n.ADD_COM %s \"%s\"\n", refdes, desc);

		export_attr(f, refdes, "Footprint", fp);
		export_attr(f, refdes, "Value", val);
		export_comp_attrs(f, refdes, &comp->hdr);

		/* Can't print terminal attributes - no support in the file format */
	}
}

/* Export abstract nets with their connections; net attributes can not
   be exported in this format */
static void cadstar_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		long lin;

		if (net->hdr.omit) continue;

		lin = 0;
		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "cadstar: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "cadstar: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "cadstar: can't determine port's parent component refdes\n");*/
				continue;
			}


			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					switch(lin) {
						case 0:   fprintf(f, "\n.ADD_TER %s.%s \"%s\"\n", refdes, my_num, netname); break;
						case 1:   fprintf(f, ".TER     %s.%s\n", refdes, my_num); break;
						default:  fprintf(f, "         %s.%s\n", refdes, my_num); break;
					}
					lin++;
				}
			);

		}
	}
}


/* Export netlist from the abstract model */
static int cadstar_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, ".HEA\n");
	fprintf(f, ".TIM 01/02/2025 03:04:05\n"); /* fake, static date for reproducible files */
	fprintf(f, ".APP \"sch-rnd export_cadstar\"\n");


	cadstar_export_comps(f, abs);
	cadstar_export_nets(f, abs);

	fprintf(f, ".END\n");
	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_cadstar(int ver_needed) { return 0; }

void pplg_uninit_export_cadstar(void)
{
	csch_plug_io_unregister(&ecadstar_net);
	rnd_export_remove_opts_by_cookie(cadstar_cookie);
	rnd_hid_remove_hid(&cadstar_hid);
}

int pplg_init_export_cadstar(void)
{
	RND_API_CHK_VER;

	ecadstar_net.name = "export to Cadstar";
	ecadstar_net.export_prio = cadstar_export_prio;
	ecadstar_net.export_project_abst = cadstar_export_project_abst;
	ecadstar_net.ext_export_project = ".net";
	csch_plug_io_register(&ecadstar_net);


	rnd_hid_nogui_init(&cadstar_hid);

	cadstar_hid.struct_size = sizeof(rnd_hid_t);
	cadstar_hid.name = "cadstar";
	cadstar_hid.description = "Exports project's Cadstar netlist";
	cadstar_hid.exporter = 1;

	cadstar_hid.get_export_options = cadstar_get_export_options;
	cadstar_hid.do_export = cadstar_do_export;
	cadstar_hid.parse_arguments = cadstar_parse_arguments;
	cadstar_hid.argument_array = cadstar_values;

	cadstar_hid.usage = cadstar_usage;

	rnd_hid_register_hid(&cadstar_hid);
	rnd_hid_load_defaults(&cadstar_hid, cadstar_options, NUM_OPTIONS);


	return 0;
}

